#!/bin/bash
# compiler test script running in target
#
# Author: Jiajun Xu <jiajun.xu@intel.com>
#
# This file is licensed under the GNU General Public License,
# Version 2.
#

# Prepare test folder for compiler test
COMPILE_FOLDER="/opt/test/compile_test"
TEST_FILE="$COMPILE_FOLDER/compile_test.c"
EXECUTE_FILE="$COMPILE_FOLDER/compile_test"
TEST_MAKEFILE="$COMPILE_FOLDER/makefile"
TEST_LIST="gcc g++ make"

if [ ! -d $COMPILE_FOLDER ]; then
	mkdir -p $COMPILE_FOLDER
fi

Target_Info()
{
	echo -e "\tTARGET: $*"
}

Target_Err()
{
	echo -e "\tTARGET: ##### Error Log #####"
	$@
	echo -e "\tTARGET: #####    End    #####"
}

# Function to generate a c test file for compiler testing
Gen_File()
{
	temp=`mktemp`

	# Generate c/c++ test file for compiler testing
	echo "#include <stdio.h>" >> $temp
	echo "#include <math.h>" >> $temp
	echo "" >> $temp
	echo "double" >> $temp
	echo "convert(long long l)" >> $temp
	echo "{" >> $temp
	echo "    return (double)l;    // or double(l)" >> $temp
	echo "}" >> $temp
	echo "" >> $temp
	echo "int" >> $temp
	echo "main(int argc, char * argv[])" >> $temp
	echo "{" >> $temp
	echo "    long long l = 10;" >> $temp
	echo "    double f;" >> $temp
	echo "" >> $temp
	echo "    f = convert(l);" >> $temp
	echo "    printf(\"convert: %lld => %f\n\", l, f);" >> $temp
	echo "" >> $temp
	echo "    f = 1234.67;" >> $temp
	echo "    printf(\"floorf(%f) = %f\n\", f, floorf(f));" >> $temp
	echo "    return 0;" >> $temp
	echo "}" >> $temp
	echo $temp
}

# Function to generate a makefile for compiler testing
Gen_Makefile()
{
	temp=`mktemp`
	basename=`basename $EXECUTE_FILE`

	echo -e "$basename: $basename.o" >> $temp
	echo -e "\tgcc -o $basename $basename.o -lm" >> $temp
	echo -e "$basename.o: $basename.c" >> $temp
	echo -e "\tgcc -c $basename.c" >> $temp

	echo $temp
}

# Generate a c test file for compiler testing
test_file=`Gen_File`

MOVE=`which mv`
$MOVE $test_file $TEST_FILE

# Begin compiler test in target
for cmd in $TEST_LIST
do
	which $cmd
	if [ $? -ne 0 ]; then
		Target_Info "No $cmd command found"
		exit 1
	fi

	if [ "$cmd" == "make" ]; then
		rm -rf $EXECUTE_FILE

		# For makefile test, we need to generate a makefile and run with a c file
		makefile=`Gen_Makefile`
		$MOVE $makefile $TEST_MAKEFILE

		cd `dirname $TEST_MAKEFILE`
		make

		if [ $? -ne 0 ]; then
			Target_Info "$cmd running with error, Pls. check error in following"
			Target_Err make
			exit 1
		fi
	else
		rm -rf $EXECUTE_FILE

		# For gcc/g++, we compile a c test file and check the output
		$cmd $TEST_FILE -o $EXECUTE_FILE -lm

		if [ $? -ne 0 ]; then
			Target_Info "$cmd running with error, Pls. check error in following"
			Target_Err $cmd $TEST_FILE -o $EXECUTE_FILE -lm
			exit 1
		fi
	fi

		# Check if the binary file generated by $cmd can work without error
		if [ -f $EXECUTE_FILE ]; then
			$EXECUTE_FILE
			if [ $? -ne 0 ]; then
				Target_Info "$EXECUTE_FILE running with error, Pls. check error in following"
				Target_Err $EXECUTE_FILE
				exit 1
			else
				Target_Info "$cmd can work without problem in target"
			fi
		else
			Target_Info "No executalbe file $EXECUTE_FILE found, Pls. check the error log"
			exit 1
		fi
done

exit 0
