/*
 * OProfile User Interface
 *
 * Copyright (C) 2007 Nokia Corporation. All rights reserved.
 *
 * Author: Robert Bradford <rob@openedhand.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


#include <stdlib.h>
#include <glib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <wait.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <netdb.h>
#include <assert.h>

#include <gtk/gtk.h>
#include <glade/glade.h>
#include <glib.h>

#include "response.h"
#include "command.h"

#include "archive.h"
#include "client.h"
#include "main.h"
#include "util.h"
 
static GIOChannel *connection;
static struct response *cur_response;
static struct command *cur_cmd = NULL;

void
client_handle_response_status (struct response *reply)
{
  if (reply->opcode == RESPONSE_OP_STATUS)
    {
      char status = reply->payload[0];

      switch (status)
        {
          case RESPONSE_STATUS_RUNNING:
            break;
          case RESPONSE_STATUS_STOPPED:
            break;
          default:
            g_warning ("Invalid payload value.");
        }

        main_status_response_cb (TRUE, NULL, status);
    } else {
      if (reply->opcode == RESPONSE_OP_ERROR)
        {
          printf ("ERROR: %s\n", reply->payload);
          main_status_response_cb (FALSE, reply->payload, 0);
        }
    }
}

void
client_handle_response_stop (struct response *reply)
{
  if (reply->opcode == RESPONSE_OP_OK)
    {
      main_stop_response_cb (TRUE, NULL);
    }

  if (reply->opcode == RESPONSE_OP_ERROR)
    {
      printf ("ERROR: %s\n", reply->payload);
      main_stop_response_cb (FALSE, reply->payload);
    }

  g_free (reply);
}

void
client_handle_response_reset (struct response *reply)
{
  if (reply->opcode == RESPONSE_OP_OK)
    {
      main_reset_response_cb (TRUE, NULL);
    }

  if (reply->opcode == RESPONSE_OP_ERROR)
    {
      printf ("ERROR: %s\n", reply->payload);
      main_reset_response_cb (FALSE, reply->payload);
    }

  g_free (reply);
}

void
client_handle_response_start (struct response *reply)
{
  if (reply->opcode == RESPONSE_OP_OK)
    {
      main_start_response_cb (TRUE, NULL);
    }

  if (reply->opcode == RESPONSE_OP_ERROR)
    {
      printf ("ERROR: %s\n", reply->payload);
      main_start_response_cb (FALSE, reply->payload);
    }

  g_free (reply);
}

void
client_handle_response_archive (struct response *reply)
{
  if (reply->opcode == RESPONSE_OP_ERROR)
    {
      main_archive_response_cb (FALSE, reply->payload);
    } else {
      main_archive_response_cb (TRUE, NULL);
      archive_handle_response (reply);
    }

  g_free (reply);
}

void
client_handle_response_get (struct response *reply)
{
  archive_file_got ();

  g_free (reply);
}


void
client_handle_connection_lost ()
{
  client_disconnect ();
  main_connection_lost_cb ();
}

/*
 * This callback is fired whenever there is data to be read from the channel.
 * It is used to process data about a response and depending on what that
 * response is it will call the appropriate handler/callback.
 *
 * This is a very scary function. I've thought about ways to split this up,
 * but they all seem pretty fugly.
 *
 */
gboolean 
client_channel_read_cb (GIOChannel *source, GIOCondition cond, gpointer data)
{
  GError *err = NULL;
  struct response *reply = cur_response;

  /*
   * Used for GET file requests to channel the data straight into the target
   * file
   */
  static int file_fd = 0;

  /* 
   * This is the number of bytes that of the response that we have already
   * read in
   */
  static unsigned long already_read = 0;

  /*
   * This is needed to poke the I/O channel into updating it's internal state.
   */
  if (!(g_io_channel_get_flags (source) & G_IO_FLAG_IS_READABLE))
    return TRUE;

  /* If this is a new response or a partially retrieved header of response */
  if (cur_response == NULL || 
        already_read < sizeof (struct response))
    {
      gsize bytes_read = 0;
      GIOStatus status;

      /* 
       * We must set the cur_response to NULL after we have dealt with
       * the response
       */
      if (cur_response == NULL)
        {
          /* Create a new response to be filled */
          reply = g_malloc0 (sizeof (struct response));
          already_read = 0; /* Important: zero this counter */
        }

      /*
       * Read in as much data as we can but no more than we strictly need at
       * this point. This I/O channel is non-blocking. So it is a real
       * possibility that we might get a short read.
       */
      status = g_io_channel_read_chars (source, 
          (gchar *)&reply[already_read], 
          sizeof (struct response) - already_read, 
          &bytes_read, &err);

      /* Update counter */
      already_read += bytes_read; 

      if (status == G_IO_STATUS_ERROR)
        {
          main_generic_error (err->message);
          g_clear_error (&err);
          return TRUE;
        }

      if (status == G_IO_STATUS_EOF)
        {
          /* Improve error handling here */
          client_handle_connection_lost ();
          return FALSE;
        }

      /* If we have read in the complete header of the structure */
      if (already_read == sizeof (struct response))
        {
          /* Prepare to receive a file */
          if (reply->opcode == RESPONSE_OP_FILE)
            {
              gchar *path = cur_cmd->payload;
              gchar *dest;
              gchar *dirname;
              gchar *archive_path;

              archive_path = archive_get_archive_path (); 

              /* Generate the destination directory structure */
              dest = g_strdup_printf("%s/%s",archive_path, path);
              dirname = g_path_get_dirname (dest);
              g_mkdir_with_parents (dirname, 00700);
              
              file_fd = open (dest, O_WRONLY|O_CREAT, 00755);

              if (file_fd < 0)
                {
                  gchar *err_string = 
                    g_strdup_printf ("Error opening file for writing: %s, %s",
                        dest, g_strerror (errno));

                  printf (err_string);
                  g_free (err_string);
                }

              g_free (dirname);
              g_free (dest);
            } else {
              /* 
               * In all other cases we must allocate more memory for the
               * payload if necessary.
               */
              if (reply->length > 0)
                {
                  /* If we have more data to get then allocate some memory */
                  reply = g_realloc (reply, SIZE_OF_RESPONSE(reply));
                }
            }

        }

      /* Update pointer in state */
      cur_response = reply;

    } else {

      /* We get here if we just need to collect the payload */
      if (already_read < SIZE_OF_RESPONSE (reply))

        /* And we need to get some more still */
        {
          if (reply->opcode == RESPONSE_OP_ERROR ||
                reply->opcode == RESPONSE_OP_ARCHIVE ||
                reply->opcode == RESPONSE_OP_FILE ||
                reply->opcode == RESPONSE_OP_STATUS)
            {
              gsize bytes_read = 0;
              GIOStatus status;
              gchar *buf;

              /* Like already_read but for the payload */
              static int payload_pos = 0;

              if (reply->opcode == RESPONSE_OP_FILE)
              {
                buf = g_malloc0 (reply->length);
              } else {
                buf = &(reply->payload[0]);
              }

              /* 
               * Once again read in as much as possible but not more than we
               * need.
               */
              status = g_io_channel_read_chars (source, 
                  &buf[payload_pos],
                  reply->length-payload_pos, 
                  &bytes_read, &err);

              /* On error fire a generic handler */
              if (status == G_IO_STATUS_ERROR)
                {
                  main_generic_error (err->message);
                  g_clear_error (&err);
                  return TRUE;
                }

              /* 
               * If we get an EOF. Then we've lost the connection with the
               * server 
               */
              if (status == G_IO_STATUS_EOF)
                {
                  client_handle_connection_lost ();
                  return FALSE;
                }

              /* 
               * If we are dealing with a file GET then we need to dump out
               * the output to a file
               */
              if (reply->opcode == RESPONSE_OP_FILE)
                {
                  int res = 0;
                  res = util_write_in_full (file_fd, &buf[payload_pos], bytes_read);

                  if (res < 0)
                    {
                      const gchar *err_string = g_strerror (errno);

                      main_generic_error ((gchar *)err_string);
                    }

                  g_free (buf);
                }

              /* Update counters */
              payload_pos += bytes_read;
              already_read += bytes_read;

              /* 
               * When we have read in all the payload then we must reset the
               * payload counter for the next time.
               */
              if (payload_pos == reply->length)
                payload_pos = 0;
            }
        }
  }

  /* Test if we have basically finished doing what we needed to do */
  if (already_read >= sizeof (struct response) && 
        already_read == SIZE_OF_RESPONSE(reply))
    {
      char opcode = cur_cmd->opcode;

      /* Clear the previous command */
      g_free (cur_cmd);
      cur_cmd = NULL;

      switch (opcode)
        {
          case COMMAND_OP_START:
            client_handle_response_start (reply);
            break;
          case COMMAND_OP_ARCHIVE:
            client_handle_response_archive (reply);
            break;
          case COMMAND_OP_STOP:
            client_handle_response_stop (reply);
            break;
          case COMMAND_OP_RESET:
            client_handle_response_reset (reply);
            break;
          case COMMAND_OP_GET:
            /* Close the file handle we opened earlier (for file GET mode) */
            close (file_fd);
            client_handle_response_get (reply);
            break;
          case COMMAND_OP_STATUS:
            client_handle_response_status (reply);
            break;
          default:
            g_warning ("Unknown response type received.");
        }

      /* Must reset current to NULL */
      cur_response = NULL;

    }
  return TRUE;
}

void
client_connect (char *hostname, int port)
{
  GError *err = NULL;

  int fd = 0;
  struct sockaddr_in addr;
  struct hostent *host = gethostbyname (hostname);
  char *ip;
  
  if (host)
    {
      ip = host->h_addr_list[0];
    }
  else
    {
      gchar *err_string = g_strdup_printf ("Could not resolve the hostname for \"%s\"", hostname);

      main_connect_response_cb (FALSE, err_string);

      g_free (err_string);

      return;
    }


  addr.sin_family = AF_INET;
  addr.sin_port = htons (port);

  memcpy (&addr.sin_addr.s_addr, ip, sizeof (ip));

  fd = socket (AF_INET,SOCK_STREAM, 0);

  if (fd < 0)
    {
      gchar *err_string = NULL;
      const gchar *error = g_strerror (errno);

      err_string = g_strdup_printf ("Could not open client socket: %s\n", error);

      main_connect_response_cb (FALSE, err_string);

      g_free (err_string);

      return;
    }

  if (connect (fd, (struct sockaddr*) &addr, sizeof (addr)) == -1)
    {
      gchar *err_string = NULL;
      const gchar *error = g_strerror (errno);

      err_string = g_strdup_printf ("Could not open client socket: %s\n", error);

      main_connect_response_cb (FALSE, err_string);

      g_free (err_string);

      return;
    }

  connection = g_io_channel_unix_new (fd);

  /* 
   * Set the channel to non-blocking. This is necessary due to the possibility
   * of a callback being fired but there actually being no data to be read.
   */
  g_io_channel_set_flags (connection, G_IO_FLAG_NONBLOCK, &err);

  if (err != NULL)
    {
      main_connect_response_cb (FALSE, err->message);
      g_clear_error (&err);
      return;
    }

  /* NULL encoding because we have raw binary data */
  g_io_channel_set_encoding (connection, NULL, &err);

  if (err != NULL)
    {
      main_connect_response_cb (FALSE, err->message);
      g_clear_error (&err);
      return;
    }

  /* Don't buffer the channel */
  g_io_channel_set_buffered (connection, FALSE);

  /* Setup the callback */
  g_io_add_watch (connection, G_IO_IN, client_channel_read_cb, NULL);

  main_connect_response_cb (TRUE, NULL);
} 

void
client_disconnect ()
{
  GError *err = NULL;

  /* Shutdown the channel and flush */
  g_io_channel_shutdown (connection, TRUE, &err);

  /* Unref -> destroy */
  g_io_channel_unref (connection);

  connection = NULL;

  if (err == NULL)
    {
      main_disconnect_response_cb (TRUE, NULL);
    } else {
      main_disconnect_response_cb (FALSE, err->message);
      g_clear_error (&err);
    }
}

struct command *
client_command_new (char opcode, int length, char *payload)
{
  struct command *buf;

  buf = g_malloc0 (sizeof (struct command) + length);

  buf->opcode = opcode;
  buf->length = length;

  if (length > 0)
    {
      assert (payload != NULL);
      memcpy(&(buf->payload[0]), payload, length);
    }

  return buf;
}

void
client_send_command (struct command *cmd, GError **err)
{
  GIOStatus status;

  /* 
   * If this is the case then a command has been issued and the reponse has
   * not been processed. The state management should prevent this from
   * happening.
   */
  g_assert (cur_cmd == NULL);


  /*
   * Set the current command in the state. Needed for demultiplexing the responses later
   */
  cur_cmd = cmd;

  status = g_io_channel_write_chars (connection, (gchar *)cmd, 
      SIZE_OF_COMMAND(cmd), NULL, err);
}

void
client_send_command_archive ()
{
  struct command *cmd;
  GError *err = NULL;

  cmd = client_command_new (COMMAND_OP_ARCHIVE, 0, NULL);

  client_send_command (cmd, &err);

  if (err != NULL)
    {
      main_archive_response_cb (FALSE, err->message);
      g_clear_error (&err);
    }
}

void
client_send_command_get (gchar *path)
{
  struct command *cmd;
  GError *err = NULL;

  if (path != NULL)
    {
      cmd = client_command_new (COMMAND_OP_GET, strlen (path)+1, path);

      client_send_command (cmd, &err);
    }
}

void
client_send_command_start ()
{
  struct command *cmd;
  GError *err = NULL;

  cmd = client_command_new (COMMAND_OP_START, 0, NULL);

  client_send_command (cmd, &err);

  if (err != NULL)
    {
      main_start_response_cb (FALSE, err->message);
      g_clear_error (&err);
    }
}

void
client_send_command_reset ()
{
  struct command *cmd;
  GError *err = NULL;

  cmd = client_command_new (COMMAND_OP_RESET, 0, NULL);

  client_send_command (cmd, &err);

  if (err != NULL)
    {
      main_start_response_cb (FALSE, err->message);
      g_clear_error (&err);
    }
}



void
client_send_command_stop ()
{
  struct command *cmd;
  GError *err = NULL;

  cmd = client_command_new (COMMAND_OP_STOP, 0, NULL);

  client_send_command (cmd, &err);

  if (err != NULL)
    {
      main_stop_response_cb (FALSE, err->message);
      g_clear_error (&err);
    }

}

void
client_send_command_status ()
{
  struct command *cmd;
  GError *err = NULL;

  cmd = client_command_new (COMMAND_OP_STATUS, 0, NULL);

  client_send_command (cmd, &err);

  if (err != NULL)
    {
      main_status_response_cb (FALSE, err->message, 0);
      g_clear_error (&err);
    }

}
