/* mb-desktop-xine - a desktop media playing module.

   Copyright 2004 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "xinebrowser.h"

static int trapped_error_code = 0;
static int (*old_error_handler) (Display *d, XErrorEvent *e);

int
error_handler(Display     *display,
              XErrorEvent *error)
{
  trapped_error_code = error->error_code;
  return 0;
}

void
trap_errors(void)
{
  trapped_error_code = 0;
  old_error_handler = XSetErrorHandler(error_handler);
}
 
int
untrap_errors(void)
{
  XSetErrorHandler(old_error_handler);
  return trapped_error_code;
}

Bool 
file_exists(char *filename)
{
  struct stat st;
  if (stat(filename, &st)) return False;
  return True;
}

int
xinebrowser_cd_get_disc_type(char *device)
{
  int fd;           /* file descriptor for CD-ROM device */
  int status;       /* return status for system calls */
  int i, have_cd = 0, had_cd = 0, cd_type = CD_TYPE_UNKNOWN;


  /* Should check permissions  */
  fd = open(device, O_RDONLY | O_NONBLOCK);
  if (fd < 0) {
    fprintf (stderr, "mbxine: open failed for `%s': %s\n",
	     device, strerror (errno));
    return CD_TYPE_ERROR;
  }

  status = ioctl (fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);
  if (status < 0) 
    {
      perror(" CDROM_DRIVE_STATUS");
      close(fd);
      return CD_TYPE_ERROR;
    } 

  if (status != CDS_DISC_OK) /* CDS_TRAY_OPEN CDS_DRIVE_NOT_READY */
    {
      return CD_TYPE_NO_DISC;
    }

  cd_type = CD_TYPE_UNKNOWN;

  status = ioctl (fd, CDROM_DISC_STATUS);

  switch (status) 
    {
    case CDS_AUDIO:
      cd_type = CD_TYPE_AUDIO;
      break;
    case CDS_DATA_1:
    case CDS_DATA_2:
    case CDS_XA_2_1:
    case CDS_XA_2_2:
      cd_type = CD_TYPE_DATA;
      break;
    case CDS_NO_INFO:
    case CDS_MIXED:
    default:
      cd_type = CD_TYPE_UNKNOWN;
    }

  if (cd_type == CD_TYPE_UNKNOWN || cd_type == CD_TYPE_DATA)
    {
      /* See if its a DVD */
      int foo;
      dvd_struct dvd;
      
      dvd.type = DVD_STRUCT_COPYRIGHT;
      dvd.copyright.layer_num = 0;
      
      status = ioctl( fd, DVD_READ_STRUCT, &dvd );

      /* 
	 dvd.copyright.cpst == 0 if encrypted

      */      
      if (status >= 0)
	{
	  cd_type = CD_TYPE_DVD;
	} 
    }

  close(fd);
  return cd_type;

}


/* Code taken from:
 * transcode Copyright (C) Thomas Oestreich - June 2001
 * enix      enix.berlios.de
 */

void 
yuy2toyv12 (unsigned char *y, 
	    unsigned char *u, 
	    unsigned char *v, 
	    unsigned char *input,
	    int            width, 
	    int            height) 
{
  int i, j, w2;
  
  w2 = width / 2;

  for (i = 0; i < height; i += 2) 
    {
      for (j = 0; j < w2; j++) {
	/* packed YUV 422 is: Y[i] U[i] Y[i+1] V[i] */
	*(y++) = *(input++);
	*(u++) = *(input++);
	*(y++) = *(input++);
	*(v++) = *(input++);
      }
      
      /* down sampling */
      for (j = 0; j < w2; j++) {
	/* skip every second line for U and V */
	*(y++) = *(input++);
	input++;
	*(y++) = *(input++);
	input++;
      }
    }
}

#define clip_8_bit(val)				\
{						\
	if (val < 0)				\
		val = 0;			\
	else					\
		if (val > 255) val = 255;	\
}

unsigned char* 
yv12torgb (unsigned char *src_y, 
	   unsigned char *src_u, 
	   unsigned char *src_v,
	   int width, 
	   int height) 
{
	  int     i, j;

	  int     y, u, v;
	  int     r, g, b;

	  int     sub_i_uv;
	  int     sub_j_uv;

	  int     uv_width, uv_height;

	  unsigned char *rgb;

	  uv_width  = width / 2;
	  uv_height = height / 2;

	  rgb = (unsigned char *) malloc (width * height * 3);
	  if (!rgb)
		  return NULL;

	  for (i = 0; i < height; ++i) {
		  /* calculate u & v rows */
		  sub_i_uv = ((i * uv_height) / height);

		  for (j = 0; j < width; ++j) {
			  /* calculate u & v columns */
			  sub_j_uv = ((j * uv_width) / width);

	  /***************************************************
	   *  Colour conversion from
	   *    http://www.inforamp.net/~poynton/notes/colour_and_gamma/ColorFAQ.html#RTFToC30
	   *
	   *  Thanks to Billy Biggs <vektor@dumbterm.net>
	   *  for the pointer and the following conversion.
	   *
	   *   R' = [ 1.1644         0    1.5960 ]   ([ Y' ]   [  16 ])
	   *   G' = [ 1.1644   -0.3918   -0.8130 ] * ([ Cb ] - [ 128 ])
	   *   B' = [ 1.1644    2.0172         0 ]   ([ Cr ]   [ 128 ])
	   *
	   *  Where in xine the above values are represented as
	   *   Y' == image->y
	   *   Cb == image->u
	   *   Cr == image->v
	   *
	   ***************************************************/

			  y = src_y[(i * width) + j] - 16;
			  u = src_u[(sub_i_uv * uv_width) + sub_j_uv] - 128;
			  v = src_v[(sub_i_uv * uv_width) + sub_j_uv] - 128;

			  r = (1.1644 * y) + (1.5960 * v);
			  g = (1.1644 * y) - (0.3918 * u) - (0.8130 * v);
			  b = (1.1644 * y) + (2.0172 * u);

			  clip_8_bit (r);
			  clip_8_bit (g);
			  clip_8_bit (b);

			  rgb[(i * width + j) * 3 + 0] = r;
			  rgb[(i * width + j) * 3 + 1] = g;
			  rgb[(i * width + j) * 3 + 2] = b;
		  }
	  }

	  return rgb;
}
