/* Implementation of the lobotomized mplayer "iplayer" AI in Xconq.
   Copyright (C) 1987-1989, 1991-1999 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
extern int select_by_weight(int *arr, int numvals);
extern int carryable(int u);
extern int accelerable(int u);
extern int accelerator(int u1, int u2);
#include "kpublic.h"
extern void try_to_draw(Side *side, int flag, char *ainame);
extern void give_up(Side *side, char *ainame);
#include "ai.h"

extern int bhw_max;

/* Needed in iplayer_adjust_plan. */
extern short any_auto_repair;

/* what does the game look like? */
typedef enum a_game_class {
    gc_none,
    gc_standard,
    gc_time,
    gc_advanced
} GameClass;

static GameClass game_class = gc_none;

/* Strategy is what a side uses to make decisions. */

typedef struct a_strategy {
    int type;
} Strategy;

#define can_see_actual_units(side, x, y) \
  ((side)->see_all || cover((side), (x), (y)) > 0)

/* Local function declarations. */

static void iplayer_init(Side *side);
static void iplayer_init_turn(Side *side);
static void create_strategy(Side *side);
static void analyze_the_game(Side *side);
static void determine_subgoals(Side *side);
static void review_goals(Side *side);
static void review_units(Side *side);
static void update_unit_plans(Side *side);
static void estimate_strengths(Side *side);
static void add_goal(Side *side, Goal *goal);
static Goal *has_goal(Side *side, GoalType goaltype);
static Goal *has_unsatisfied_goal(Side *side, GoalType goaltype);
static void iplayer_decide_plan(Side *side, Unit *unit);
static int iplayer_adjust_plan(Side *side, Unit *unit);

/* New action-reaction code. */
static void iplayer_react_to_action_result(Side *side, Unit *unit, int rslt);
static void defensive_reaction(Unit *unit);
static int keep_defensive_goal(Unit *unit, Unit *unit2);
static int maybe_defend_own_transport(Unit *unit);
static Unit *undefended_neighbour(Unit *unit);
static int maybe_defend_other_transport(Unit *unit, Unit *unit2);
static void offensive_reaction(Unit *unit);
static int iplayer_go_after_victim(Unit *unit, int range);
static int iplayer_go_after_captive(Unit *unit, int range);
static int iplayer_fire_at_opportunity(Unit *unit);

static int need_this_type_to_explore(Side *side, int u);
/* static int compare_weights(struct weightelt *w1, struct weightelt *w2); */
static void assign_to_exploration(Side *side, Unit *unit);
static int need_this_type_to_build_explorers(Side *side, int u);
static void assign_to_explorer_construction(Side *side, Unit *unit);
static void assign_to_offense(Side *side, Unit *unit);
static void assign_to_offense_support(Side *side, Unit *unit);
static int type_can_build_attackers(Side *side, int u);
static int preferred_build_type(Side *side, Unit *unit, int plantype);
static int need_more_transportation(Side *side);
static void assign_to_defense(Side *side, Unit *unit);
static void assign_to_defense_support(Side *side, Unit *unit);
static int assign_to_develop_on(Side *side, Unit *unit, int u2);
static int can_develop_on(int u, int u2);
static int needs_develop(Side *side, int u);
static int probably_explorable(Side *side, int x, int y, int u);
static int build_depot_for_self(Side *side, Unit *unit);
static void assign_to_colonize(Side *side, Unit *unit);
static void assign_to_improve(Side *side, Unit *unit);
static void iplayer_react_to_task_result(Side *side, Unit *unit, Task *task, TaskOutcome rslt);
static int desired_direction_impassable(Unit *unit, int x, int y);
static int blocked_by_enemy(Unit *unit, int x, int y, int shortest);
static int attack_blockage(Side *side, Unit *unit, int x, int y, int shortest);
static void iplayer_react_to_new_side(Side *side, Side *side2);
static void iplayer_finish_movement(Side *side);
static Unit *search_for_available_transport(Unit *unit, int purpose);
static void assign_to_defend_unit(Unit *unit, Unit *unit2);
static void assign_to_defend_cell(Unit *unit, int x, int y);
static void assign_to_defend_vicinity(Unit *unit, int x, int y, int w, int h);
static void rethink_plan(Unit *unit);
static int enemy_close_by(Side *side, Unit *unit, int dist, int *xp, int *yp);
static void iplayer_receive_message(Side *side, Side *sender, char *str);
static char *iplayer_at_desig(Side *side, int x, int y);
static int iplayer_theater_at(Side *side, int x, int y);
static int iplayer_read_strengths(Side *side);
static Obj *iplayer_save_state(Side *side);
static int compare_weights(const void *w1, const void *w2);

/* This is the set of operations that generic code will use. */

AI_ops iplayer_ops = {
    "iplayer",			/* name */
    NULL,			/* to_test_compat */		
    iplayer_init,		/* to_init */
    iplayer_init_turn,		/* to_init_turn */
    iplayer_decide_plan,	/* to_decide_plan */
    iplayer_react_to_action_result,	/* to_react_to_action_result */
    iplayer_react_to_task_result,	/* to_react_to_task_result */
    iplayer_react_to_new_side,	/* to_react_to_new_side */
    iplayer_adjust_plan,	/* to_adjust_plan */
    iplayer_finish_movement,	/* to_finish_movement */
    iplayer_receive_message,	/* to_receive_message */
    iplayer_save_state,		/* to_save_state */
    iplayer_theater_at,		/* region_at */
    iplayer_at_desig,		/* at_desig */
    -1				/* dummy */
};

/* Flag to detect when shared iplayer init has been done. */

static int iplayerinited = FALSE;

/* Determine game type from name of included modules. */

static GameClass find_game_class(void);

static GameClass
find_game_class(void)
{
    Module *m;

    for_all_modules(m) {
	if (strcmp(m->name, "time") == 0
	    || (m->origmodulename && strcmp(m->origmodulename, "time") == 0))
	  return gc_time;
    	else if (strcmp(m->name, "advanced") == 0
		 || (m->origmodulename
		     && strcmp(m->origmodulename, "advanced") == 0))
	  return gc_advanced;
    }
    return gc_standard;
}

static void
iplayer_init(side)
Side *side;
{
    Unit *unit;

    if (game_class == gc_none) {
	game_class = find_game_class();
    }

    /* (should do this only when absolutely needed - iplayer might
       never actually be used) */
    if (!iplayerinited) {
	iplayerinited = TRUE;
    }
    /* Delete any old strategy object in case we just switched AI type. */
    if (side->ai != NULL) {
    	free(side->ai);
    	side->ai = NULL;
    }    
      create_strategy(side);
    /* If the side has no units at the moment, it doesn't really need to
       plan. */
    if (!side_has_units(side))
      return;
    /* Reset plans of any units that were not doing anything. */
    for_all_side_units(side, unit) {
    	if (in_play(unit) && unit->plan && unit->plan->aicontrol) {
	    net_force_replan(side, unit, TRUE);
    	}
    }
}

/* At the beginning of each turn, make plans and review the situation. */

static void
iplayer_init_turn(side)
Side *side;
{
    int u, u2;

    /* Cases where we no longer need to run. */
    if (!side->ingame)
      return;
    /* A side without units hasn't got anything to do but wait. */
    /* (should account for possible units on controlled sides) */
    if (!side_has_units(side))
      return;
    /* Mplayers in a hacked game will not play,
       unless they're being debugged. */
    if (compromised && !DebugM)
      return;
    update_all_progress_displays("ai turn init start", side->id);
    DMprintf("%s iplayer init turn\n", side_desig(side));
	/* Check out all of our units. */
	review_units(side);
	/* (should be integrated better) */
	iplayer_finish_movement(side);
	/* Propagate this to individual unit plans. */
	update_unit_plans(side);
    update_all_progress_displays("", side->id);
    DMprintf("%s iplayer init turn done\n", side_desig(side));
}

/* Create and install an entirely new strategy object for the side. */

static void
create_strategy(side)
Side *side;
{
    Strategy *strategy = (Strategy *) xmalloc(sizeof(Strategy));

    /* Put the specific structure into a generic slot. */
    side->ai = (struct a_ai *) strategy;
    strategy->type = iplayertype;
}

/* Go through all our units (and allied ones?). */

static void
review_units(side)
Side *side;
{
    int u, u2, cp, cpmin, any;
    int numoffensive[MAXUTYPES], numdefensive[MAXUTYPES];
    Unit *unit, *occ, *unit2;
    Plan *plan;

    for_all_unit_types(u) {
	numoffensive[u] = numdefensive[u] = 0;
    }
    for_all_side_units(side, unit) {
	if (in_play(unit) && unit->plan && unit->plan->aicontrol) {
	    /* Count plan types. */
	    switch (unit->plan->type) {
	      case PLAN_OFFENSIVE:
	      case PLAN_EXPLORATORY:
		++numoffensive[unit->type];
		break;
	      case PLAN_DEFENSIVE:
		++numdefensive[unit->type];
		break;
	      default:
	        break;
	    }
	}
    }

    for_all_side_units(side, unit) {
	if (in_play(unit) && unit->plan && unit->plan->aicontrol) {
	    plan = unit->plan;
	    /* Goal might have become satisfied. */
	    if (plan->maingoal) {
		if (goal_truth(side, plan->maingoal) == 100) {
		    DMprintf("%s %s satisfied, removing\n",
			     unit_desig(unit), goal_desig(plan->maingoal));
		    net_force_replan(side, unit, FALSE);
		}
	    }
	    /* Don't let defense-only units pile up. */
	    if (plan->type == PLAN_DEFENSIVE
		&& mobile(unit->type)
		&& (numoffensive[unit->type] / 3) < numdefensive[unit->type]
		/* However, don't mess with units that have specific defensive goals. */
		&& (plan->maingoal == NULL
			|| (plan->maingoal->type != GOAL_UNIT_OCCUPIED
			 && plan->maingoal->type != GOAL_CELL_OCCUPIED))
		&& flip_coin()) {
		DMprintf("%s one of too many on defense (%d off, %d def), replanning\n",
			 unit_desig(unit), numoffensive[unit->type], numdefensive[unit->type]);
		net_force_replan(side, unit, FALSE);
	    }
	}
    }
}

/* For each unit, decide what it should be doing (if anything).  This is
   when a side takes the initiative;  a unit can also request info from
   its side when it is working on its individual plan. */

static void
update_unit_plans(side)
Side *side;
{
    Unit *unit;

    for_all_side_units(side, unit) {
	if (is_active(unit) && unit->plan != NULL) {
	    iplayer_decide_plan(side, unit);
	}
    }
}

/* This is for when a unit needs a plan and asks its side for one. */

static void
iplayer_decide_plan(Side *side, Unit *unit)
{
    Plan *plan = unit->plan;
    int u = unit->type;

    /* Nothing to do for units that can't act or are not under AI control. */
    if (plan == NULL || !plan->aicontrol)
      return;
    /* Plans for advanced units are (for now) handled in run.c instead. */
    if (u_advanced(u))
    	return;
    switch (plan->type) {
      case PLAN_PASSIVE:
      case PLAN_NONE:
	if (mobile(u)) {
	    if (u_colonizer_worth(u) > 0) {
		assign_to_colonize(side, unit);
		return;
	    }
	    if (type_can_attack(u) || type_can_fire(u)) {
		/* (A more precise test would be "can attack types
		    known to be or likely to be in the current game".)  */
		/* Assign most units to offense, save some for defense. */
		if (probability(75))
		  assign_to_offense(side, unit);
		else
		  assign_to_defense(side, unit);
	    } else if (type_can_build_attackers(side, u)) {
		assign_to_offense_support(side, unit);
	    } else {
	    }
	} else {
	    if (type_can_build_attackers(side, u)) {
	    	assign_to_offense_support(side, unit);
	    } else if (type_can_attack(u) || type_can_fire(u)) {
		assign_to_defense(side, unit);
	    } else {
		assign_to_defense_support(side, unit);
	    }
	}
	break;
      case PLAN_OFFENSIVE:
	/* leave plan alone */
	break;
      case PLAN_COLONIZING:
	/* leave plan alone */
	break;
      case PLAN_IMPROVING:
	/* leave plan alone */
	break;
      case PLAN_EXPLORATORY:
	/* leave plan alone */
	break;
      case PLAN_DEFENSIVE:
	/* leave plan alone */
	break;
      default:
	break;
    }
}

static int
iplayer_adjust_plan(side, unit)
Side *side;
Unit *unit;
{
    int u3, lowm;
    Task *task;

    /* Plans for advanced units are (for now) handled in run.c instead. */
    if (u_advanced(unit->type))
      return TRUE;

    if ((unit->plan->type == PLAN_OFFENSIVE
	 || unit->plan->type == PLAN_COLONIZING
	 || unit->plan->type == PLAN_IMPROVING
	 || unit->plan->type == PLAN_EXPLORATORY)
	&& !mobile(unit->type)
	&& unit->plan->aicontrol
	&& !unit->plan->asleep
	&& unit->plan->tasks == NULL
	) {
	    u3 = preferred_build_type(side, unit, unit->plan->type);
	    if (is_unit_type(u3)) {
		task = unit->plan->tasks;
		if (task == NULL || task->type != TASK_BUILD) {
		    DMprintf("%s directed to build %s\n",
			     unit_desig(unit), u_type_name(u3));
		    net_set_build_task(unit, u3, 2);
		} else {
		    DMprintf("%s already building, leaving alone\n",
			     unit_desig(unit));
		}
		/* Only do one at a time, wait for next go-around for
                   next unit. */
		return FALSE;
	    }
    }
    if (unit->plan->waitingfortasks
	&& unit->plan->aicontrol
	) {
	net_force_replan(side, unit, FALSE);
    }
    if (!unit->plan->reserve
    	&& g_units_may_go_into_reserve()
	&& unit->plan->execs_this_turn > 10 * max(1, u_acp(unit->type))) {
	net_set_unit_reserve(side, unit, TRUE, FALSE);
    }
    /* Look at more units. */
    return TRUE;
}

/* This hook is triggered when certain actions are executed within 
the tactical range of our unit. */

static void
iplayer_react_to_action_result(side, unit, rslt)
Side *side;
Unit *unit;
int rslt;
{
    int lowm;
    
    /* (Should use flags here for efficienc, but unit->supply_is_low 
    does not work like resupply_is_low.) */ 

    /* Don't mess with units that are heading back for supplies. */    
    lowm = low_on_supplies_one(unit);
    if (lowm != NONMTYPE)
      return;
    /* Don't mess with units that are heading back for ammo. */    
    lowm = low_on_ammo_one(unit);
    if (lowm != NONMTYPE)
      return;
    /* Don't mess with units that are heading back for repair. */    
    if (unit->hp <= (u_hp_max(unit->type) 
    			  * unit_doctrine(unit)->repair_percent) / 100
        && (u_hp_recovery(unit->type) > 0 || any_auto_repair))
      return;

    /* Deal with tactical emergencies. */
    defensive_reaction(unit);
    offensive_reaction(unit);
}

/* Check that advanced units are adequately defended, but release 
any excess defenders. */

static void
defensive_reaction(Unit *unit)
{
    Plan 	*plan = unit->plan;
    Unit 	*unit2;

    /* First check if our unit is assigned to protecting another unit,
       and release it from that duty if it is no longer needed. This is 
       very important or all our units will soon be soaked up in this 
       kind of defense! */
    if (plan->maingoal 
        && plan->maingoal->type == GOAL_UNIT_OCCUPIED) {
	unit2 = find_unit(plan->maingoal->args[0]);
	/* Return if we are still busy with unit2. */
	if (keep_defensive_goal(unit, unit2))
	    return;
	/* We need a new plan and goal. */
	net_force_replan(unit->side, unit, FALSE);
	DMprintf("%s released from defending %s\n", 
		          unit_desig(unit), unit_desig(unit2));
    }
    /* Then check if our own transport needs defense. */
    if (maybe_defend_own_transport(unit))
        return;
    /* If not, check if any neigbour needs defense. */
    unit2 = undefended_neighbour(unit);
    if (unit2) {
	maybe_defend_other_transport(unit, unit2);
    }
}

/* Returns true if unit is busy defending or recapturing unit2, 
false if it needs a new goal. */

static int
keep_defensive_goal(Unit *unit, Unit *unit2)
{
	Plan 	*plan = unit->plan;
	int	garrison = 0;
	Unit 	*unit3;

	/* unit2 has been destroyed. */
	if (!in_play(unit2)) {
		return FALSE;
	/* We can't defend unit2 (should never happen). */
	} else if (!occ_can_defend_transport(unit->type, unit2->type)) {
		return FALSE;
	/* unit2 is held by the enemy. */
	} else if (enemy_side(unit->side, unit2->side)) {
	    /* Don't interfere if we are trying to recapture unit2. */
	    if (plan->tasks
		&& (((plan->tasks->type == TASK_HIT_UNIT
			&& plan->tasks->args[0] == unit2->x
			&& plan->tasks->args[1] == unit2->y
			&& plan->tasks->args[2] == unit2->type)
		    || (plan->tasks->type == TASK_CAPTURE
			&& plan->tasks->args[0] == unit2->x
			&& plan->tasks->args[1] == unit2->y
			&& plan->tasks->args[2] == unit2->type))
		    /* Or if we are on our way to recapture unit2. */
		    || (plan->tasks->type == TASK_MOVE_TO
			&& plan->tasks->args[0] == unit2->x
			&& plan->tasks->args[1] == unit2->y
			&& plan->tasks->next
			&& ((plan->tasks->next->type == TASK_HIT_UNIT
				&& plan->tasks->args[0] == unit2->x
				&& plan->tasks->args[1] == unit2->y
				&& plan->tasks->args[2] == unit2->type)
			     || plan->tasks->next->type == TASK_CAPTURE
				&& plan->tasks->args[0] == unit2->x
				&& plan->tasks->args[1] == unit2->y
				&& plan->tasks->args[2] == unit2->type)))) {
		    return TRUE;
	    }
	    /* Try to recapture unit2 directly if we have a decent chance. */
	    if (real_capture_chance(unit, unit2) > 20) {
		net_set_capture_task(unit, unit2->x, unit2->y);
    		DMprintf("%s tries to recapture %s directly\n",
	     					unit_desig(unit), unit_desig(unit2));
		    return TRUE;
	    /* Otherwise attack unit2 if possible. */
	    } else if (attack_can_damage_or_capture(unit, unit2) 
		       || fire_can_damage_or_capture(unit, unit2)) {
		net_set_hit_unit_task(unit, unit2->x, unit2->y, 
				      unit2->type, unit2->side->id);
    		DMprintf("%s tries to take %s back by attacking it\n",
	     					unit_desig(unit), unit_desig(unit2));
		    return TRUE;
	    }
	/* Check if unit2 is adequately defended by OTHER occupants. */
	} else {
		for_all_occupants(unit2, unit3) {
		    if (is_active(unit3)
			&& unit3 != unit
			&& unit3->plan
			&& unit3->plan->maingoal
			&& unit3->plan->maingoal->type == GOAL_UNIT_OCCUPIED
			&& unit3->plan->maingoal->args[0] == unit2->id
			/* Occs that cannot defend the transport should never
			   have been assigned to occupy it, but check
			   anyway. */
			&& occ_can_defend_transport(unit3->type, unit2->type)) {
			++garrison;
		    }
		}
		/* If so, release our unit from its duty. */
		if (enough_to_garrison(unit2, garrison)) {
			return FALSE;
		/* Else make sure we have the occupy task set if we are not there. */
		} else if (unit->transport != unit2
			&& plan->tasks
		         && plan->tasks->type != TASK_OCCUPY) {
		    	net_set_occupy_task(unit, unit2);
			return TRUE;
		/* Else just keep defending unit2. */
		} else {
			return TRUE;
		}			
	}
}

/* Returns true if we decide to defend it. */

static int
maybe_defend_own_transport(Unit *unit)
{
    	Unit 	*unit2;
	  
    /* First assign our unit to protect its own city if needed. */
    unit2 = unit->transport;
    if (unit2 != NULL
	/* A city is a transport that can build. */
		&& u_can_make(unit2->type)
		/* u_can_make is much faster than can_build. */
		&& occ_can_defend_transport(unit->type, unit2->type)
		/* could_carry, which is called by occ_can_defend_transport,
		is much faster than can_carry_type, which was used before. */
		&& !defended_by_occupants(unit2)) {
	assign_to_defend_unit(unit, unit2);
	DMprintf("%s assigned to defend own transport %s\n",
     				unit_desig(unit), unit_desig(unit2));
		return TRUE;
	}
	return FALSE;
}

/* This code is twice as fast as the old code which used
for_all_side_units. */

static Unit *
undefended_neighbour(Unit *unit)
{
	Unit *unit2;
	int x, y;

	for_all_cells_within_range(unit->x, unit->y, 
			u_ai_tactical_range(unit->type), x, y) {
		if (!inside_area(x, y))
			continue;
		if (!terrain_visible(unit->side, x, y))
			continue;
		for_all_stack(x, y, unit2) {
			if (is_active(unit2)
			  && unit2->side == unit->side
			 /* (u_can_make is much faster than can_build). */
			  && u_can_make(unit2->type)
			 /* (could_carry, which is called by occ_can_defend_transport,
			 is much faster than can_carry_type, which was used before). */
		    	  && occ_can_defend_transport(unit->type, unit2->type)
		    	  && !defended_by_occupants(unit2)) {
				return unit2;
			}
		}
	}
	return NULL;
}

/* Returns true if we decide to defend it. */

static int
maybe_defend_other_transport(Unit *unit, Unit *unit2)
{
	Plan 	*plan = unit->plan;
	Unit 	*unit3;
	int	garrison = 0;

	/* Check if other defenders on their way are closer or same distance. 
	This prevents buzzing of flocks of defenders in and out of a city without 
	adequate defense. */
	for_all_side_units(unit->side, unit3) {

		    if (is_active(unit3)
			/* Should never happen, but check. */
			&& unit3 != unit
			&& unit3->plan
			&& unit3->plan->maingoal
			&& unit3->plan->maingoal->type == GOAL_UNIT_OCCUPIED
			&& unit3->plan->maingoal->args[0] == unit2->id
			&& occ_can_defend_transport(unit3->type, unit2->type)
			/* (should take unit speeds into consideration here). */
			&& distance(unit2->x, unit2->y, unit3->x, unit3->y) <=
			      distance(unit2->x, unit2->y, unit->x, unit->y)) {
				/* This is really the potential garrison as soon as
				all defenders get there. */
				++garrison;
			}							
		}
	        /* If not enough defenders, assign our unit to the city's defense. */
	        if (!enough_to_garrison(unit2, garrison)) { 
    			assign_to_defend_unit(unit, unit2);
	    		DMprintf("%s assigned to defend neighbour transport %s\n",
		     					unit_desig(unit), unit_desig(unit2));
		return TRUE;
	}				
	return FALSE;
}

/* Deal with tactical emergencies such as enemy units that suddenly
   appear nearby. */

static void
offensive_reaction(Unit *unit)
{
    Plan 	*plan = unit->plan;
    int		range;
	
    /* Drop the current task and promptly attack any enemies within
       range.  This is very important or we will loose the tactical
       initiative! */

    /* But first check if we are assigned to defending another unit. */
    if (plan->maingoal
	&& plan->maingoal->type == GOAL_UNIT_OCCUPIED) {

	    Unit *unit2 = find_unit(plan->maingoal->args[0]);
	    /* Limit attack range to 1 if we are defending a unit. This will in 
	    most cases assure that the defender stays inside its protege, but an 
	    overrun action into an adjacent cell may still cause the defender to 
	    stray outside. */
	    if (unit->transport && unit->transport == unit2) {
		  range = 1;
	    } else {
	    	  range = u_ai_tactical_range(unit->type);
	    }
	    /* Fall through: don't interfere if we are on our way to defend a city. */	
	    if (in_play(unit2)
	        && plan->tasks
	        && plan->tasks->type == TASK_OCCUPY
	        && plan->tasks->args[0] == unit2->id) {
		return;
	    }

    /* Also check if we are assigned to defending a specific cell. */
    } else if (plan->maingoal
          && plan->maingoal->type == GOAL_CELL_OCCUPIED) {
          
    	    int x = plan->maingoal->args[0];
	    int y = plan->maingoal->args[1];
	    /* Limit attack range to 1 in that case. */
	    if (unit->x == x && unit->y == y) {
		  range = 1;
	    } else {
	    	  range = u_ai_tactical_range(unit->type);
	    }
	    /* Fall through: don't interfere if we are on our way to defend a cell. */	
	    if (inside_area(x, y)
	        && plan->tasks
	        && plan->tasks->type == TASK_MOVE_TO
	        && plan->tasks->args[0] == x
	        && plan->tasks->args[1] == y) {
	        	return;
	    }

    /* Use default tactical range in all other cases. */
    } else range = u_ai_tactical_range(unit->type);

    /* Fall-through: don't interfere if we already are in a fight. */
    if (plan->tasks
	&& (plan->tasks->type == TASK_HIT_UNIT
	    || plan->tasks->type == TASK_HIT_POSITION
	    || plan->tasks->type == TASK_CAPTURE
	    /* Or if we are chasing a victim. */
	    || (plan->tasks->type == TASK_MOVE_TO
		&& plan->tasks->next
		&& (plan->tasks->next->type == TASK_HIT_UNIT
		    || plan->tasks->next->type == TASK_HIT_POSITION
		    || plan->tasks->next->type == TASK_CAPTURE)))) {
		
    /* Else try to capture enemy within attack range. */
    } else if (can_capture_directly(unit)) {
	         iplayer_go_after_captive(unit, range);
    /* Else fire at enemy within firing range. */
    } else if (can_fire(unit)) {
	         iplayer_fire_at_opportunity(unit);
    /* Else attack enemy within attack range (or move in to attack or fire). */
    } else if (can_fire(unit) || can_attack(unit)) {
	         iplayer_go_after_victim(unit, range);
    } 
}

extern int victimx, victimy, victimrating, victimutype, victimsidenum;

static int
iplayer_go_after_victim(Unit *unit, int range)
{
    int x, y, rslt;

    tmpunit = unit;
    DMprintf("%s (iplayer) seeking victim within %d; found ",
	     unit_desig(unit), range);
    victimrating = -9999;
    rslt = search_around(unit->x, unit->y, range, victim_here, &x, &y, 1);
    if (rslt) {
	DMprintf("s%d %s at %d,%d\n", victimsidenum, u_type_name(victimutype), x, y);
	/* Set up a task to go after the unit found. */
	/* (should be able to set capture task if better) */
	net_set_hit_unit_task(unit, x, y, victimutype, victimsidenum);
	if (unit->transport != NULL
	    && mobile(unit->transport->type)
	    && unit->transport->plan) {
	    net_set_move_to_task(unit->transport, x, y, 1);
	}
    } else if (victimrating > -9999) {
	DMprintf("s%d %s (rated %d) at %d,%d\n",
		 victimsidenum, u_type_name(victimutype), victimrating,
		 victimx, victimy);
	/* Set up a task to go after the unit found. */
	/* (should be able to set capture task if better) */
	net_set_hit_unit_task(unit, victimx, victimy, victimutype,
			      victimsidenum);
	if (unit->transport != NULL
	    && mobile(unit->transport->type)
	    && unit->transport->plan) {
	    net_set_move_to_task(unit->transport, victimx, victimy, 1);
	}
	/* We succeeded after all. */
	rslt = TRUE;
    } else {
	DMprintf("nothing\n");
    }
    return rslt;
}

/* The point of this new function is to limit the search for captives
   to a given range. */

static int
iplayer_go_after_captive(Unit *unit, int range)
{
    int x, y, rslt;

    tmpunit = unit;

    DMprintf("%s (iplayer) searching for useful capture within %d; found ",
	     unit_desig(unit), range);
    rslt = search_around(unit->x, unit->y, range, 
			 useful_captureable_here, &x, &y, 1);
    if (rslt) {
	DMprintf("one at %d,%d\n", x, y);
	/* Set up a task to go after the unit found. */
	net_set_capture_task(unit, x, y);
	if (unit->transport
	    && mobile(unit->transport->type)
	    && unit->transport->plan) {
	    net_set_move_to_task(unit->transport, x, y, 1);
	}
	return TRUE;
    } else {
	DMprintf("nothing\n");
	return FALSE;
    }
}

static int
iplayer_fire_at_opportunity(Unit *unit)
{
    int x, y, range, rslt;
    extern int targetx, targety, targetrating, targetutype, targetsidenum;

    tmpunit = unit;
    range = u_range(unit->type);
    targetrating = -9999;
    DMprintf("%s (iplayer) seeking target within %d; found ",
             unit_desig(unit), range);
    rslt = search_around(unit->x, unit->y, range, target_here, &x, &y, 1);
    if (rslt) {
	DMprintf("s%d %s at %d,%d\n", targetsidenum, u_type_name(targetutype), x, y);
	/* Set up a task to shoot at the unit found. */
	net_set_hit_unit_task(unit, x, y, targetutype, targetsidenum);
    } else if (targetrating > -9999) {
	DMprintf("s%d %s (rated %d) at %d,%d\n",
		 targetsidenum, u_type_name(targetutype), targetrating, x, y);
	/* Set up a task to shoot at the unit found. */
	net_set_hit_unit_task(unit, targetx, targety, targetutype, targetsidenum);
    } else {
	DMprintf("nothing\n");
    }
    return rslt;
}

/* Set the unit up as an explorer and let it go. */

static void
assign_to_exploration(Side *side, Unit *unit)
{
    int numweights = 0, weight, i, dist;

    /* Unit's goal in life will be to see that the world is all known. */
    net_set_unit_plan_type(side, unit, PLAN_EXPLORATORY);
}

/* Unit's goal in life will be to colonize the world. */

static void
assign_to_colonize(Side *side, Unit *unit)
{
   	int	type[MAXUTYPES] = {0};
	int	numtypes = 0, u2;
	Goal	*goal;
	
	net_clear_task_agenda(side, unit);

	/* Pick a unit type to build. */
	for_all_unit_types(u2) {
		if (u_advanced(u2) 
		    && could_create(unit->type, u2)
		    && has_advance_to_build(unit->side, u2)
		    && type_allowed_on_side(u2, unit->side)) {
			type[numtypes] = u2;
			numtypes++;
		}
	}
	if (numtypes) {
		/* (Should make a more intelligent choice here) */
		/* (Should perhaps wait til execution time) */
	    	u2 = type[xrandom(numtypes)];
		goal = create_goal(GOAL_COLONIZE, side, TRUE);
		goal->args[0] = u2;
		net_set_unit_plan_type(side, unit, PLAN_COLONIZING);
		net_set_unit_main_goal(side, unit, goal);
		DMprintf("%s will colonize by building %s\n", 
		     		unit_desig(unit), u_type_name(u2));
	} else	 {
		/* We cannot use this unit to colonize. */
		net_set_unit_plan_type(side, unit, PLAN_PASSIVE);
		DMprintf("%s cannot colonize, got passive plan instead\n", 
				unit_desig(unit));
	}
}

static void
assign_to_improve(side, unit)
Side *side;
Unit *unit;
{
    /* Unit's goal in life will be to improve the world. */
    net_set_unit_plan_type(side, unit, PLAN_IMPROVING);
    net_clear_task_agenda(side, unit);
    net_set_unit_main_goal(side, unit, NULL);
    DMprintf("%s assigned to improve\n", unit_desig(unit));
}

static void
assign_to_offense(side, unit)
Side *side;
Unit *unit;
{
    int numweights = 0, weight;
    Goal *goal;

    net_set_unit_plan_type(side, unit, PLAN_OFFENSIVE);
    net_clear_task_agenda(side, unit);
}

static void
assign_to_offense_support(side, unit)
Side *side;
Unit *unit;
{
    net_set_unit_plan_type(side, unit, PLAN_OFFENSIVE);
    net_clear_task_agenda(side, unit);
}

static int
type_can_build_attackers(side, u)
Side *side;
int u;
{
    int u2;
	
    for_all_unit_types(u2) {
	if (mobile(u2)
	    && (type_can_attack(u2) || type_can_fire(u2))
	    && has_advance_to_build(side, u2)
	    && type_allowed_on_side(u2, side)
	    && uu_acp_to_create(u, u2) > 0)
	  return TRUE;
    }
    return FALSE;
}

/* For the given side and unit and plan, calculate the right type of
   unit to build. */

static int
preferred_build_type(side, unit, plantype)
Side *side;
Unit *unit;
int plantype;
{
    int u = unit->type, u2, u3, t;
    int x, y, dir, x1, y1, blockedallaround, uview, est, rslt;
    int prefs[MAXUTYPES];
    int fringeterrain[MAXTTYPES], sumfringe, totfringe;
    int enemytypes[MAXUTYPES];
    int numtotransport[MAXUTYPES];
    Unit *unit2, *occ;

    if (plantype == PLAN_EXPLORATORY) {
	/* Calculate the amount of each type of terrain at the edges
	   of the known world. */
	for_all_terrain_types(t)
	  fringeterrain[t] = 0;
	for_all_cells(x, y) {
	    if (terrain_view(side, x, y) != UNSEEN) {
		for_all_directions(dir) {
		    if (point_in_dir(x, y, dir, &x1, &y1)
			&& terrain_view(side, x1, y1) == UNSEEN) {
			++(fringeterrain[(int) terrain_at(x, y)]);
			break;
		    }
		}
	    }
	}
    } else {
	/* should use estimated strengths instead? */
    	for_all_unit_types(u2)
    	  enemytypes[u2] = 0;
    	for_all_interior_cells(x, y) {
	    if (can_see_actual_units(side, x, y)) {
		/* Use new recursive macro instead that also sees occs within occs. */
		for_all_stack_with_occs(x, y, unit2) {
		    if (!trusted_side(side, unit2->side))
		      	    ++enemytypes[unit2->type];
		}
	    } else {
		uview = unit_view(side, x, y);
		if (uview != EMPTY) {
		    if (!trusted_side(side, side_n(vside(uview))))
		      ++enemytypes[vtype(uview)];
		}
	    }
    	}
    }
    /* Calculate a basic preference for each possible type. */
    for_all_unit_types(u2) {
	prefs[u2] = 0;
	est = est_completion_time(unit, u2);
	if (could_create(u, u2)
	    /* tmp hack until iplayer can do develop */
	    && (u_tech_to_build(u2) > 0 ? side->tech[u2] >= u_tech_to_build(u2) : TRUE)
	    && est >= 0
	    /* No longer included in type_allowed_on_side. */
	    && has_advance_to_build(side, u2)
	    && type_allowed_on_side(u2, side)) {
		/* Prefer units by overall suitability for general plan. */
		if (plantype == PLAN_EXPLORATORY) {
		    /* Weight unit types by suitability for exploration around
		       the edges of the known area. */
		    sumfringe = totfringe = 0;
		    for_all_terrain_types(t) {
			totfringe += fringeterrain[t];
			if (!terrain_always_impassable(u2, t))
			  sumfringe += fringeterrain[t];
		    }
		    if (totfringe < 1)
		      sumfringe = totfringe = 1;
		    /* Scale - so 5% diffs in amt of crossable terrain
		       don't affect result. */
		    prefs[u2] = (20 * sumfringe) / totfringe;
		    /* Prefer types that are quicker to build. */
		    prefs[u2] /= max(1, est / 8);
		} else if (plantype == PLAN_COLONIZING) {
		    /* First eliminate non-colonizing units. */
		    if (u_colonizer_worth(u2) <= 0) {
			prefs[u2] = 0;
		    	/* Weight unit type by effectiveness against
                           known enemies. */
		    } else {
			for_all_unit_types(u3) {
			    if (enemytypes[u3] > 0) {
				if (uu_zz_bhw(u2, u3) > 0) {
				    prefs[u2] +=
				      ((uu_zz_bhw(u2, u3) * 100) / bhw_max) * enemytypes[u3];
				}
				if (uu_zz_bcw(u2, u3) > 0) {
				    prefs[u2] += uu_zz_bcw(u2, u3) * enemytypes[u3];
				}
			    }
			}
			/* Prefer types that are quicker to build. */
			prefs[u2] /= max(1, est / 8);
		    }			
		} else if (plantype == PLAN_IMPROVING) {
		    /* Eliminate all mobile units. */
		    if (mobile(u2))
		      prefs[u2] = 0;
		} else {
		    /* Weight unit type by effectiveness against known
                       enemies. */
		    for_all_unit_types(u3) {
			if (enemytypes[u3] > 0) {
			    if (uu_zz_bhw(u2, u3) > 0) {
				prefs[u2] +=
				  ((uu_zz_bhw(u2, u3) * 100) / bhw_max) * enemytypes[u3];
			    }
			    if (uu_zz_bcw(u2, u3) > 0) {
				prefs[u2] += uu_zz_bcw(u2, u3) * enemytypes[u3];
			    }
			}
		    }
		    /* Prefer types that are quicker to build. */
		    prefs[u2] /= max(1, est / 8);
		}
		if (prefs[u2] < 1)
		  prefs[u2] = 1;
	}
    }
    /* Units that can't even get out of the builder get their
       preference cut.  This helps prevent the construction of large
       ships in Denver. */
    /* (should allow if units would have some other way to leave) */
    if (1 /* plantype == PLAN_EXPLORATORY */) {
	for_all_unit_types(u2) {
	    if (prefs[u2] > 0) {
		blockedallaround = TRUE;
		for_all_directions(dir) {
		    point_in_dir(unit->x, unit->y, dir, &x1, &y1);
		    if (!terrain_always_impassable(u2, terrain_at(x1, y1))) {
		    	blockedallaround = FALSE;
		    	break;
		    }
#if 0  /* should replace this with a more useful test */
		    if (unit_at(x1, y1) != NULL) {
		    	blockedallaround = FALSE;
		    	break;
		    }
#endif
		}
		if (blockedallaround)
		  prefs[u2] = 0;
	    }
	}
    }
    DMprintf("%s preferred build type is ", unit_desig(unit));
    /* Look for an existing incomplete occupant and prefer to build its type,
       if it is in the choices in the typelist. */
    for_all_occupants(unit, occ) {
	if (in_play(occ) && !completed(occ)) {
	    if (prefs[occ->type] > 0 && flip_coin()) {
		rslt = occ->type;
		DMprintf("%s (incomplete occupant)\n", u_type_name(rslt));
		return rslt;
	    }
	}
    }
    for_all_unit_types(u2)
      if (prefs[u2] < 0)
        prefs[u2] = 0;
    rslt = select_by_weight(prefs, numutypes);
    if (!is_unit_type(rslt))
      rslt = NONUTYPE;
    if (DebugM) {
	if (is_unit_type(rslt)) {
	    DMprintf("%s (choices were", u_type_name(rslt));
	    for_all_unit_types(u2) {
		if (prefs[u2] > 0)
		  DMprintf(" %s,%d", utype_name_n(u2, 1), prefs[u2]);
	    }
	    DMprintf(")");
	} else {
	    DMprintf("nothing (no choices)");
	}
	DMprintf("\n");
    }
    return rslt;
}

static void
assign_to_defense(Side *side, Unit *unit)
{
    net_set_unit_plan_type(side, unit, PLAN_DEFENSIVE);
}

/* Assign unit to defend unit2. Does NOT change the plan type. */

void
assign_to_defend_unit(Unit *unit, Unit *unit2)
{
    Goal *goal;

    /* Clear the task aganda. */
    net_clear_task_agenda(unit->side, unit);
    net_set_unit_plan_type(unit->side, unit, PLAN_DEFENSIVE);
    /* Set main goal to occupy unit2. */
    goal = create_goal(GOAL_UNIT_OCCUPIED, unit->side, TRUE);
    goal->args[0] = unit2->id;
    net_set_unit_main_goal(unit->side, unit, goal);
    DMprintf("%s assigned to occupy %s\n",
	     unit_desig(unit), unit_desig(unit2));
}

/* Assign unit to defend cell at (x, y). */

void
assign_to_defend_cell(Unit *unit, int x, int y)
{
    	Goal 	*goal;

	/* Clear the task aganda. */
	net_clear_task_agenda(unit->side, unit);
	net_set_unit_plan_type(unit->side, unit, PLAN_DEFENSIVE);
	/* Set main goal to occupy (x, y). */
	goal = create_goal(GOAL_CELL_OCCUPIED, unit->side, TRUE);
	goal->args[0] = x;
	goal->args[1] = y;
	net_set_unit_main_goal(unit->side, unit, goal);
	DMprintf("%s assigned to occupy cell at (%d, %d)\n",
	 		unit_desig(unit), x, y);
}

/* Assign unit to defend the vicinity of (x, y). Does NOT change the
   plan type. */

void
assign_to_defend_vicinity(Unit *unit, int x, int y, int w, int h)
{
    	Goal 	*goal;

	/* Clear the task aganda. */
	net_clear_task_agenda(unit->side, unit);
	net_set_unit_plan_type(unit->side, unit, PLAN_DEFENSIVE);
	/* Set main goal to hold the vicinity of (x, y). */
	goal = create_goal(GOAL_VICINITY_HELD, unit->side, TRUE);
	goal->args[0] = x;  
	goal->args[1] = y;
	goal->args[2] = w;
	goal->args[3] = h;
	net_set_unit_main_goal(unit->side, unit, goal);
	DMprintf("%s now assigned to defense the vicinity of (%d, %d)\n",
		 unit_desig(unit), x, y);
}

static void
assign_to_defense_support(side, unit)
Side *side;
Unit *unit;
{
    net_set_unit_plan_type(side, unit, PLAN_DEFENSIVE);
    net_clear_task_agenda(side, unit);
}

static int
build_depot_for_self(Side *side, Unit *unit)
{
    int u = unit->type, u2, cando = FALSE;

    for_all_unit_types(u2) {
	if (uu_acp_to_create(u, u2) > 0
	    && ((uu_creation_cp(u, u2) >= u_cp(u2)
		&& type_allowed_on_side(u2, side)
		&& has_advance_to_build(side, u2))
		|| uu_acp_to_build(u, u2) > 0)
	    /* (should check if any advantage to building) */
	    ) {
	   cando = TRUE;
	   break;
	}
    }
    if (cando) {
	DMprintf("%s building %s as a depot for itself\n",
		 unit_desig(unit), u_type_name(u2));
	net_set_build_task(unit, u2, 1);
	return TRUE;
    }
    return FALSE;
}

/* This is a hook that runs after each task is executed. */

static void
iplayer_react_to_task_result(side, unit, task, rslt)
Side *side;
Unit *unit;
Task *task;
TaskOutcome rslt;
{
    int dx, dy, x1, y1, fact;
    Unit *occ;

    /* React to an apparent blockage. */
    if (rslt == TASK_FAILED
	&& task != NULL
	&& task->type == TASK_MOVE_TO
	&& task->retrynum > 2) {
	if (desired_direction_impassable(unit, task->args[0], task->args[1])) {
	    	if (unit->occupant) {
		    DMprintf("%s blocked while transporting, will sit briefly\n",
			     unit_desig(unit));
		    net_set_unit_reserve(side, unit, TRUE, FALSE);
		    for_all_occupants(unit, occ) {
		    	net_wake_unit(side, occ, FALSE);
		    }
		    return;
	    	}
	    /* Try moving sideways. */
	    if (probability(80)) {
		dx = task->args[0] - unit->x;  dy = task->args[1] - unit->y;
		fact = (flip_coin() ? 50 : -50);
		x1 = unit->x - ((fact * dy) / 100);
		y1 = unit->y + ((fact * dx) / 100);
		if (inside_area(x1, y1))
		  net_push_move_to_task(unit, x1, y1, 1);
	    }
	    return;
	} else if (blocked_by_enemy(unit, task->args[0], task->args[1], TRUE)) {
	    /* (should decide if allowable risk to passengers) */
	    DMprintf("%s blocked by enemy\n", unit_desig(unit));
	    if (!attack_blockage(side, unit, task->args[0], task->args[1], TRUE)) {
		if (blocked_by_enemy(unit, task->args[0], task->args[1], FALSE)) {
		    attack_blockage(side, unit, task->args[0], task->args[1], FALSE);
		} else {
		    /* (should move sideways?) */
		}
	    }
	} else {
	    /* what to do about other failures? */
	}
	return;
    }
    /* React to inability to resupply by trying to build a base. */
    if (rslt == TASK_FAILED
	&& task != NULL
	&& task->type == TASK_RESUPPLY
	&& task->retrynum > 2) {
    	net_set_unit_reserve(side, unit, FALSE, FALSE);
    	build_depot_for_self(side, unit);
    }
}

/* (should account for impassability because of borders, etc) */

static int
desired_direction_impassable(Unit *unit, int x, int y)
{
    int dirs[NUMDIRS], numdirs, i, x1, y1, t, numbaddirs = 0;

    numdirs = choose_move_dirs(unit, x, y, TRUE, NULL, NULL, dirs);
    for (i = 0; i < numdirs; ++i) {
	point_in_dir(unit->x, unit->y, dirs[i], &x1, &y1);
	t = terrain_at(x1, y1);
	if (terrain_always_impassable(unit->type, t))
	  ++numbaddirs;
    }
    return (numbaddirs == numdirs);
}

static int
blocked_by_enemy(Unit *unit, int x, int y, int shortest)
{
    int dirs[NUMDIRS], numdirs, i, x1, y1, numbaddirs = 0;
    Unit *unit2;

    numdirs = choose_move_dirs(unit, x, y, shortest, plausible_move_dir, NULL, dirs);
    /* No plausible dirs anyhow, so presence of enemy irrelevant. */
    if (numdirs == 0)
      return FALSE;
    for (i = 0; i < numdirs; ++i) {
	point_in_dir(unit->x, unit->y, dirs[i], &x1, &y1);
	unit2 = unit_at(x1, y1);
	if (in_play(unit2) && !trusted_side(unit->side, unit2->side))
	  ++numbaddirs;
    }
    return (numbaddirs == numdirs);
}

static int
attack_blockage(Side *side, Unit *unit, int x, int y, int shortest)
{
    int dirs[NUMDIRS], numdirs, i, x1, y1;
    Unit *unit2;

    numdirs = choose_move_dirs(unit, x, y, shortest, plausible_move_dir, NULL, dirs);
    for (i = 0; i < numdirs; ++i) {
	point_in_dir(unit->x, unit->y, dirs[i], &x1, &y1);
	unit2 = unit_at(x1, y1);
	if (in_play(unit2)
	    && !trusted_side(unit->side, unit2->side)
	    && uu_zz_bhw(unit->type, unit2->type) > 0
	    ) {
	    net_push_hit_unit_task(unit, x1, y1, unit2->type, side_number(unit2->side));
	    return TRUE;
	}
    }
    return FALSE;
}

/* This function is called whenever a new side appears in the game.  It
   mainly needs to make that any allocated data is resized appropriately. */

static void
iplayer_react_to_new_side(Side *side, Side *side2)
{
}

/* At the end of a turn, re-evaluate the plans of some units in case
   the situation changed. */

static void
iplayer_finish_movement(side)
Side *side;
{
    int u, scan;
    Unit *unit;

    for_all_side_units(side, unit) {
	if (is_active(unit)
	    && unit->plan
	    && unit->plan->aicontrol) {
	    rethink_plan(unit);
	}
    }
}

/* For units with plans and that are under AI control, consider changing the
   current plan/tasks. */

static void
rethink_plan(unit)
Unit *unit;
{
    int dist, x1, y1;
    Task *toptask = unit->plan->tasks, *nexttask = NULL;
    Plan *plan = unit->plan;
    Unit *transport;

    if (toptask)
      nexttask = toptask->next;
    if (unit->plan->type == PLAN_OFFENSIVE
        && toptask != NULL
        && toptask->type == TASK_MOVE_TO
        && distance(unit->x, unit->y, toptask->args[0], toptask->args[1])
		>= min(2, u_acp(unit->type))
        && enemy_close_by(unit->side, unit, 1 /* 2 would be better? */, &x1, &y1)
        ) {
	net_push_hit_unit_task(unit, x1, y1, NONUTYPE, -1);
	DMprintf("%s sees enemy close by, will attack it\n", unit_desig(unit));
    }
    /* (should also notice fire opportunities) */
    /* If we see somebody that could be captured and help us explore, set up
       to produce capturers. */
    if (!mobile(unit->type)
	&& (unit->plan->type == PLAN_EXPLORATORY || unit->plan->type == PLAN_OFFENSIVE)
	) {
	int range = 4, rslt, x, y;

	DMprintf("%s searching for useful capture within %d in order to choose build; found ",
	     	 unit_desig(unit), range);
	tmpunit = unit;
	rslt = search_around(unit->x, unit->y, range, useful_captureable_here,
			     &x, &y, 1);
	if (rslt && is_unit_type(tmputype)) {
	    DMprintf("%s at %d,%d", u_type_name(tmputype), x, y);
	    if (toptask != NULL
		&& toptask->type == TASK_BUILD
		&& uu_capture(toptask->args[0], tmputype)
		) {
		/* Already doing the right thing. */
		DMprintf(" - already building %s", u_type_name(toptask->args[0]));
	    } else {
		/* (should find best type that can capture quickly,
		    schedule to build it) */
		DMprintf(" - duhhh, what now?");
	    }
	} else {
	    DMprintf("nothing");
	}
	DMprintf("\n");
    }
}

/* Detect an enemy unit in our own or an adjacent cell. */

/* static int victimx, victimy, victimrating; */

static int
enemy_close_by(side, unit, dist, xp, yp)
Side *side;
Unit *unit;
int dist, *xp, *yp;
{
    int x = unit->x, y = unit->y, dir, x1, y1;

    victimrating = -9999;
    tmpunit = unit;
    victim_here(x, y);
    for_all_directions(dir) {
	if (point_in_dir(x, y, dir, &x1, &y1)) {
	    victim_here(x, y);
	}
    }
    if (victimrating > -9999) {
	*xp = victimx;  *yp = victimy;
	return TRUE;
    } else {
	return FALSE;
    }
}

static void
iplayer_receive_message(side, sender, str)
Side *side, *sender;
char *str;
{
    /* First detect standard messages. */
    if (strcmp(str, "Eh?") == 0) {
	/* Don't respond, otherwise we might infinitely recurse. */
    } else if (allied_side(side, sender)) {
	/* (should say something) */
    } else {
	/* Detect insults and respond appropriately. */
	if (strstr(str, "idiot")) {
	    net_send_message(side, add_side_to_set(sender, NOSIDES), "Loser!");
	} else if (strstr(str, "loser")) {
	    net_send_message(side, add_side_to_set(sender, NOSIDES), "Idiot!");
	} else {
	    /* No idea what the message was, be puzzled. */
	    net_send_message(side, add_side_to_set(sender, NOSIDES), "Eh?");
	}
    }
}

/* This is used by interfaces to display the name of the theater in
   use at a given point. */

static char *
iplayer_at_desig(Side *side, int x, int y)
{
   return ("<no theater>");
}

/* This is used by interfaces to display boundaries between theaters,
   by comparing the numbers returned. */

static int
iplayer_theater_at(Side *side, int x, int y)
{
    return 0;
}

/* Collect initial strength information stored in the iplayer's
   private saved data. */

static int
iplayer_read_strengths(Side *side)
{
    int sn1, u, found = FALSE;
    char *propname;
    Obj *props, *bdg, *rest, *sidebdg, *urest;
    Side *side1;

    props = find_at_key(side->aidata, "iplayer");
    for (; props != lispnil; props = cdr(props)) {
	bdg = car(props);
	propname = c_string(car(bdg));
	if (strcmp(propname, "strengths0") == 0) {
	    found = TRUE;
	    rest = cdr(bdg);
	    for_all_sides(side1) {
		sn1 = side1->id;
		sidebdg = car(rest);
		urest = cadr(sidebdg);
		for_all_unit_types(u) {
		    urest = cdr(urest);
		}
		rest = cdr(rest);
	    }
	} else if (strcmp(propname, "alstrengths0") == 0) {
	    found = TRUE;
	    rest = cdr(bdg);
	    for_all_sides(side1) {
		sn1 = side1->id;
		sidebdg = car(rest);
		urest = cadr(sidebdg);
		for_all_unit_types(u) {
		    urest = cdr(urest);
		}
		rest = cdr(rest);
	    }
	} else if (strcmp(propname, "points0") == 0) {
	    found = TRUE;
	    rest = cdr(bdg);
	    for_all_sides(side1) {
		sn1 = side1->id;
		rest = cdr(rest);
	    }
	} else if (strcmp(propname, "alpoints0") == 0) {
	    found = TRUE;
	    rest = cdr(bdg);
	    for_all_sides(side1) {
		sn1 = side1->id;
		rest = cdr(rest);
	    }
	} else {
	}
    }
    return found;
}

/* Write out any state that the iplayer must preserve.  We don't actually write;
   instead we build a Lisp object and pass that back to the writing routines. */

static Obj *
iplayer_save_state(side)
Side *side;
{
    int sn1, u;
    Obj *rslt, *vallist, *uvallist;
    Side *side1;

    rslt = lispnil;
    /* We're pushing bindings onto a list, so do in reverse of desired order. */
    vallist = lispnil;
    for_all_sides(side1) {
	sn1 = side1->id;
	uvallist = lispnil;
	push_binding(&vallist, new_number(sn1), uvallist);
    }
    vallist = reverse(vallist);
    push_cdr_binding(&rslt, intern_symbol("alstrengths0"), vallist);
    vallist = lispnil;
    for_all_sides(side1) {
	sn1 = side1->id;
	uvallist = lispnil;
	push_binding(&vallist, new_number(sn1), uvallist);
    }
    vallist = reverse(vallist);
    push_cdr_binding(&rslt, intern_symbol("strengths0"), vallist);
    vallist = lispnil;
    push_cdr_binding(&rslt, intern_symbol("alpoints0"), vallist);
    vallist = lispnil;
    push_cdr_binding(&rslt, intern_symbol("points0"), vallist);
    return lispnil;
}
