/* Commands for the tcl/tk interface to Xconq.
   Copyright (C) 1998, 1999 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

#include "conq.h"
#include "kpublic.h"
#include "tkconq.h"

void really_do_design(Side *side);

static void aux_add_terrain(Side *side, Map *map, int cancel);
static void aux_add_terrain_2(Side *side, Map *map, int cancel);
static void do_add_terrain_2(Side *side, Map *map, int t);
static void aux_attack(Side *side, Map *map, int cancel);
static void aux_build(Side *side, Map *map, int cancel);
static void impl_build(Side *side, Unit *unit, int u2, int n);
static void aux_distance(Side *side, Map *map, int cancel);
static void aux_fire_at(Side *side, Map *map, int cancel);
static void aux_give_unit(Side *side, Map *map, int cancel);
static void aux_move_look(Side *side, Map *map);
static void aux_move_dir(Side *side, Map *map, Unit *unit);
static void aux_message(Side *side, Map *map, int cancel);
static void aux_move_to(Side *side, Map *map, int cancel);
static void aux_name(Side *side, Map *map, int cancel);
static void aux_others(Side *side, Map *map, int cancel);
static void aux_remove_terrain(Side *side, Map *map, int cancel);
static void aux_remove_terrain_2(Side *side, Map *map, int cancel);
static void do_remove_terrain_2(Side *side, Map *map, int t);
static void aux_resign(Side *side, Map *map, int cancel);
static void aux_resign_2(Side *side, Map *map, int cancel);
static void aux_leave_game(Side *side, Map *map, int cancel);
static void aux_kill_game(Side *side, Map *map, int cancel);
static void aux_resign_b(Side *side, Map *map, int cancel);
static void aux_save_2(Side *side, Map *map, int cancel);
static void aux_save_1(Side *side, Map *map, int cancel);
static void aux_save_1_1(Side *side, Map *map, int cancel);
static void aux_set_formation(Side *side, Map *map, int cancel);
static void save_the_game(Side *side);

static void aux_design(Side *side, Map *map, int cancel);

/* Use this macro in any command if it requires a current unit. */

#define REQUIRE_UNIT(map)  \
  if (!in_play((map)->curunit) || (map)->curunit_id != (map)->curunit->id) {  \
    (map)->curunit = NULL;  \
    (map)->curunit_id = 0;  \
    cmd_error((dside), "No current unit to command!");  \
    return;  \
  }

int designed_on = FALSE;

/* Definitions of all the command functions. */

void
do_add_terrain(Side *side)
{
    Map *map = side->ui->curmap;
    int u, t, numtypes, tfirst = NONTTYPE, possibles[MAXTTYPES];
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    u = unit->type;
    numtypes = 0;
    for_all_terrain_types(t) {
	if (ut_acp_to_add_terrain(u, t) > 0) {
	    possibles[t] = TRUE;
	    ++numtypes;
	    tfirst = t;
	} else {
	    possibles[t] = FALSE;
	}
    }
    if (numtypes == 0) {
	cmd_error(side, "%s cannot add or alter terrain!", unit_handle(side, unit));
    } else if (numtypes == 1) {
	map->argunitid = unit->id;
	do_add_terrain_2(side, map, tfirst);
    } else {
	map->argunitid = unit->id;
	ask_terrain_type(side, map, "Type to add:", possibles, aux_add_terrain);
    }
}

static void
aux_add_terrain(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int t;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Type a key to select terrain type.");
    }
    if (grok_terrain_type(side, map, &t)) {
	if (t != NONTTYPE) {
	    do_add_terrain_2(side, map, t);
	}
    } else {
        if (map->inpch != '?')
	  beep(side);
	/* Stay in this mode until we get it right. */
	map->modalhandler = aux_add_terrain;
    }
}

/* This is like do_add_terrain, but with a terrain type given. */

static void
do_add_terrain_2(side, map, t)
Side *side;
Map *map;
int t;
{
    char abuf[100];

    map->tmpt = t;
    sprintf(abuf, "Add %s where?", t_type_name(t));
    ask_position(side, map, abuf, aux_add_terrain_2);
}

static void
aux_add_terrain_2(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y, dir;
    Unit *unit;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Pick a location to add %s.", t_type_name(map->tmpt));
	map->modalhandler = aux_add_terrain_2;
	return;
    }
    if (grok_position(side, map, &x, &y, NULL)) {
	if (in_area(x, y)) {
	    unit = find_unit(map->argunitid);
	    if (in_play(unit) && side_controls_unit(side, unit)) {
		switch (t_subtype(map->tmpt)) {
		  case cellsubtype:
		    net_prep_alter_cell_action(unit, unit, x, y, map->tmpt);
		    break;
		  case bordersubtype:
		  case connectionsubtype:
		    dir = closest_dir(x - unit->x, y - unit->y);
		    net_prep_add_terrain_action(unit, unit, unit->x, unit->y, dir, map->tmpt);
		    break;
		  case coatingsubtype:
		    net_prep_add_terrain_action(unit, unit, unit->x, unit->y, 1, map->tmpt);
		    break;
		}
	    }
	}
    } else {
	beep(side);     
	map->modalhandler = aux_add_terrain_2;
    }
}

void
do_attack(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;
    
    REQUIRE_UNIT(map);
    map->argunitid = unit->id;
    ask_position(side, map, "Attack where?", aux_attack);
}

static void
aux_attack(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y;
    Unit *unit, *unit2;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Click on a location");
    }
    if (grok_position(side, map, &x, &y, &unit2)) {
	unit = find_unit(map->argunitid);
	if (in_play(unit) && side_controls_unit(side, unit)) {
	    if (unit2 != NULL) {
		net_prep_attack_action(unit, unit, unit2, 100);
	    } else {
	      /* (should attack anything in cell) */
	    }
	} else {
	    cmd_error(side, "unit disappeared!?");
	}
    } else {
        if (map->inpch != '?')
	  beep(side);
	map->modalhandler = aux_attack;
    }
}

void
do_auto(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;
    
    REQUIRE_UNIT(map);
    if (unit->plan) {
	net_set_unit_ai_control(side, unit, !unit->plan->aicontrol, FALSE);
	/* a hack */
	unit->plan->waitingfortasks = !unit->plan->aicontrol;
    }
}

void
do_build(Side *side)
{
    Map *map = side->ui->curmap;
    int u, u2, possibles[MAXUTYPES], numtypes, ufirst = NONUTYPE;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    u = unit->type;
    if (!can_build(unit)) {
	cmd_error(side, "%s can't build anything!", unit_handle(side, unit));
	return;
    }
    numtypes = 0;
    for_all_unit_types(u2) {
	if (uu_acp_to_create(u, u2) > 0 && type_allowed_on_side(u, side)) {
	    possibles[u2] = TRUE;
	    ++numtypes;
	    ufirst = u2;
	} else {
	    possibles[u2] = FALSE;
	}
    }
    if (unit->transport != NULL
	&& !uu_occ_can_build(unit->transport->type, u)
	&& !(numtypes == 1
	     && !completed(unit->transport)
	     && uu_acp_to_build(u, unit->transport->type) > 0)) {
	cmd_error(side, "%s can't build anything while inside another unit!",
		  unit_handle(side, unit));
	return;
    }
    switch (numtypes) {
      case 0:
	cmd_error(side, "Nothing to build!");
	break;
      case 1:
	/* Only one type to build - do it. */
	impl_build(side, unit, ufirst, map->prefixarg);
	break;
      default:
	/* Player has to choose a type to build. */
	map->argunitid = unit->id;
	ask_unit_type(side, map, "Type to build:", possibles, aux_build);
	break;
    }
}

static void
aux_build(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int u2;
    Unit *unit;

    if (cancel) {
	cancel_unit_type(side, map);
	return;
    }
    if (map->inpch == '?') {
	notify(side, "Type a key or click on a unit type to select build.");
    }
    if (grok_unit_type(side, map, &u2)) {
	if (u2 != NONUTYPE) {
	    unit = find_unit(map->argunitid);
	    if (in_play(unit) && side_controls_unit(side, unit)) {
		impl_build(side, unit, u2, map->prefixarg);
	    } else {
		cmd_error(side, "unit disappeared!?");
	    }
	}
    } else {
        if (map->inpch != '?')
	  beep(side);
	map->modalhandler = aux_build;
    }
}

static void
impl_build(side, unit, u2, n)
Side *side;
Unit *unit;
int u2, n;
{
    if (n < 0)
      n = unit_doctrine(unit)->construction_run[u2];
    if (n <= 0)
      n = 99;
    notify(side, "%s will build %d %s.",
	   unit_handle(side, unit), n, u_type_name(u2));
    net_push_build_task(unit, u2, n);
}

void
do_clear_plan(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    net_set_unit_plan_type(side, unit, PLAN_NONE);
}

void
do_collect(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;
    int mtocollect;
    char *arg, *rest;

    mtocollect = NONMTYPE;
    if (nummtypes == 0) {
	cmd_error(side, "No materials to collect");
	return;
    }
    if (!empty_string(cmdargstr)) {
	rest = get_next_arg(cmdargstr, tmpbuf, &arg);
	mtocollect = mtype_from_name(arg);
    } else {
	cmd_error(side, "No material name given");
	return;
    }
    if (!is_material_type(mtocollect)) {
	cmd_error(side, "`%s' is not a recognized material name", arg);
	return;
    }
    if (unit->plan)
      net_set_collect_task(unit, mtocollect, unit->x, unit->y);
}

void
do_copying(Side *side)
{
    popup_help(side, copying_help_node);
}

void
do_delay(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    if (unit != NULL) {
	if (unit->plan)
	  net_delay_unit(unit, TRUE);
    } else {
	unit = find_next_awake_mover(side, map->curunit);
	if (unit != map->curunit) {
	    set_current_unit(map, unit);
	} else {
	    cmd_error(side, "No next awake mover found.");
	}
    }
}

void
do_detach(Side *side)
{
    Map *map = side->ui->curmap;
    int rslt;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    if (!completed(unit)) {
	cmd_error(side, "%s is incomplete; cannot detach",
		  unit_handle(side, unit));
	return;
    }
    rslt = check_transfer_part_action(unit, unit, unit->hp / 2, NULL);
    if (valid(rslt)) {
	net_prep_transfer_part_action(unit, unit, unit->hp / 2, NULL);
    } else {
	notify(side, "can't detach for some reason?");
    }
}

void
do_detonate(Side *side)
{
    Map *map = side->ui->curmap;
    int rslt;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    rslt = check_detonate_action(unit, unit, unit->x, unit->y, unit->z);
    if (valid(rslt)) {
	net_prep_detonate_action(unit, unit, unit->x, unit->y, unit->z);
    } else {
	notify(side, "can't detonate for some reason?");
    }
}

void
do_dir(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    aux_move_dir(side, map, map->curunit);
}

void
do_dir_multiple(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    map->prefixarg = 9999;
    aux_move_dir(side, map, map->curunit);
}

static void
aux_move_dir(side, map, unit)
Side *side;
Map *map;
Unit *unit;
{
    int ndirs, dir, n = map->prefixarg, x, y;
  
    if (!unit->act || !unit->plan) { /* use a more sophisticated test? */
	/* ??? can't act ??? */
	return;
    }
    ndirs = char_to_dir(tmpkey, &dir, NULL, NULL);
    if (ndirs < 1) {
	cmd_error(side, "what direction is that?!?");
	return;
    }
    if (n > 1) {
	DGprintf("Ordering %s to move %d %s\n",
		 unit_desig(unit), n, dirnames[dir]);
	net_set_move_dir_task(unit, dir, n);
    } else {
	if (!point_in_dir(unit->x, unit->y, dir, &x, &y)) {
	    return;
	}
	if (!advance_into_cell(side, unit, x, y, unit_at(x, y))) {
	    beep(side);
	}
    }
}

/* Get rid of a unit. */

void
do_disband(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    net_disband_unit(side, unit);
}

void
do_disembark(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit, *transport;

    REQUIRE_UNIT(map);
    transport = unit->transport;
    if (transport == NULL) {
	cmd_error(side, "Not in a transport");
	return;
    }
    if (!in_play(transport)) {
	cmd_error(side, "Transport is nonsensical?");
	return;
    }
    /* Try moving into the transport's transport, if there is one. */
    if (transport->transport != NULL
        && can_occupy(unit, transport->transport)) {
	net_prep_enter_action(unit, unit, transport->transport);
	/* (should be able to set up task if can't do action immedly) */
	return;
    }
    /* Try moving into the open in the cell. */
    if (can_occupy_cell(unit, unit->x, unit->y)
	|| can_occupy_conn(unit, unit->x, unit->y, unit->z)) {
	net_prep_move_action(unit, unit, unit->x, unit->y, unit->z);
	/* (should be able to set up task if can't do action immedly) */
	return;
    }
    cmd_error(side, "Can't disembark here!");
}

void
do_distance(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    map->argunitid = map->curunit->id;
    ask_position(side, map, "Distance to where?", aux_distance);
}

static void
aux_distance(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y, dist;
    Unit *unit;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Pick a location to which you want the distance.");
	map->modalhandler = aux_distance;
	return;
    }
    if (grok_position(side, map, &x, &y, NULL)) {
	unit = find_unit(map->argunitid);
	if (in_area(x, y) && unit != NULL) {
	    dist = distance(unit->x, unit->y, x, y);
	    notify(side, "Distance from the current unit is %d cells.", dist);
	} else {
	    cmd_error(side, "Measurement failed.");
	}
    } else {
        beep(side);
	map->modalhandler = aux_distance;
    }
}

void
do_embark(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *transport, *occ;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    /* look for the first possible transport */
    for_all_stack(unit->x, unit->y, transport) {
	/* make sure its not the transport we're in and we can enter it */
	if (transport != unit->transport &&
	    valid(check_enter_action(unit, unit, transport))) {
	    net_prep_enter_action(unit, unit, transport);
	    return;
	}

	/* check the occupants too */
	for_all_occupants(transport, occ) {
	    if (occ != unit->transport &&
		valid(check_enter_action(unit, unit, occ))) {
		net_prep_enter_action(unit, unit, occ);
		return;
	    }
	}
    }
    cmd_error(side, "Nothing for this unit to enter!");
}

void
do_escape(Side *side)
{
}

void
do_fire(Side *side)
{
    Map *map = side->ui->curmap;
    int sx, sy, x, y, rslt;
    Unit *other;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);

#if 0
    if (map->frombutton) {
	map->argunitid = unit->id;
	save_cur(side, map);
	ask_position(side, map, "Fire at what unit? [click to set]",
		     aux_fire_at);
	return;
    }
#endif

    sx = map->inpsx;  sy = map->inpsy;

    if (nearest_cell(widget_vp(map), sx, sy, &x, &y, NULL, NULL)) {
	if (x != unit->x || y != unit->y) {
	    if (unit->act && unit->plan) { /* (should be more sophisticated?) */
		/* (should only be able to target unit if covered...) */
		other = unit_at(x, y);
		if (other != NULL) {
		    /* There's a unit to fire at. */
		    if (other->side == unit->side) {
			cmd_error(side, "You can't fire at one of your own units!");
		    } else {
			rslt = check_fire_at_action(unit, unit, other, -1);
			if (valid(rslt)) {
			    net_prep_fire_at_action(unit, unit, other, -1);
			} else {
			    /* (should say which unit was target) */
			    cmd_error(side, "%s fire at unit not valid: %s",
				      unit_handle(side, unit),
				      action_result_desc(rslt));
			}
		    }
		} else {
		    cmd_error(side, "Nothing there to fire at");
		}
	    }
	}
    }
}

static void
aux_fire_at(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y;
    Unit *unit, *other;

    if (cancel)
      return;
    if (grok_position(side, map, &x, &y, NULL)) {
	unit = find_unit(map->argunitid);
	if (x != unit->x || y != unit->y) {
	    if (unit->act && unit->plan) { /* (should be more sophisticated?) */
		other = unit_at(x, y);
		if (other != NULL) {
		    /* There's a unit to fire at. */
		    if (other->side == unit->side) {
			cmd_error(side, "You can't fire at one of your own units!");
		    } else if (valid(check_fire_at_action(unit, unit, other, -1))) {
			net_prep_fire_at_action(unit, unit, other, -1);
		    }
		} else {
		    cmd_error(side, "Nothing there to fire at");
		}
	    }
	}
    } else {
	map->modalhandler = aux_move_to;
    }
}

void
do_fire_into(Side *side)
{
    Map *map = side->ui->curmap;
    int x = map->inpx, y = map->inpy, rslt;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);

    if (inside_area(x, y)) {
	if (x != unit->x || y != unit->y) {
	    rslt = check_fire_into_action(unit, unit, x, y, 0, -1);
	    if (valid(rslt)) {
		net_prep_fire_into_action(unit, unit, x, y, 0, -1);
	    } else {
		cmd_error(side, "%s fire into %d,%d not valid: %s",
			  unit_handle(side, unit), x, y,
			  action_result_desc(rslt));
	    }
	}
    }
}

void
do_flash(Side *side)
{
    cmd_error(side, "Not implemented.");
}

/* Toggle the "follow-action" flag. */

void
do_follow_action(Side *side)
{
    Map *map = side->ui->curmap;
    
    map->follow_action = !map->follow_action;
    if (map->follow_action) {
	notify(side, "Following the action on map.");
    } else {
	notify(side, "Not following the action on map.");
    }
}

void
do_give(Side *side)
{
    Map *map = side->ui->curmap;
    int something = FALSE;
    int n, u, m, r, gift, actual;
    Unit *unit = map->curunit, *main = NULL;
    char buf[BUFSIZE];

    if (nummtypes == 0) {
	cmd_error(side, "No materials in this game!");
	return;
    }
    REQUIRE_UNIT(map);
    u = unit->type;
    main = unit->transport;
    if (main == NULL) {
	cmd_error(side, "Nothing to give to here!");
	return;
    }
    m = main->type;
    n = (map->prefixarg < 0 ? 1 : map->prefixarg);

    buf[0] = '\0';
    for_all_material_types(r) {
	gift = (n < 0 ? (um_storage_x(m, r) - main->supply[r]) : n);
	if (gift > 0) {
	    something = TRUE;
	    /* Be stingy if we're low */
	    if (2 * unit->supply[r] < um_storage_x(u, r))
	      gift = max(1, gift/2);
	    actual = transfer_supply(unit, main, r, gift);
	    tprintf(buf, " %d %s", actual, m_type_name(r));
	}
    }
    if (something) {
	notify(side, "%s gave%s.", unit_handle(side, unit), buf);
    } else {
	notify(side, "%s gave nothing.", unit_handle(side, unit));
    }
}

/* Give a unit to another side or make it independent. */

/* (but giving to indep should be tested, otherwise might kill unit) */

void
do_give_unit(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;
    Side *side2;

    REQUIRE_UNIT(map);
    if (between(0, map->prefixarg, numsides)) {
	side2 = side_n(map->prefixarg);
	really_do_give_unit(map->curunit, side2);
    } else {
	ask_side(side, map, "To whom do you wish to give the unit?", NULL,
		 aux_give_unit);
    }
}

static void
aux_give_unit(Side *side, Map *map, int cancel)
{
    Side *side2;

    if (cancel)
      return;
    if (grok_side(side, map, &side2)) {
	really_do_give_unit(map->curunit, side2);
    } else {
	/* Iterate until grok_side is happy. */
	map->modalhandler = aux_give_unit;
    }
}

really_do_give_unit(Unit *unit, Side *side2)
{
    if (unit->side == side2)
      return;
#ifdef DESIGNERS
    if (is_designer(dside)) {
	net_designer_change_side(unit, side2);
	return;
    }
#endif /* DESIGNERS */
    if (0 /* unit is a type that cannot act */) {
	/* (should add case for nonacting units) */
    } else {
	if (side2 != NULL)
	  notify(dside, "You give %s to %s.", unit_handle(dside, unit),
		 short_side_title(side2));
	else
	  notify(dside, "You give %s its independence.",
		 unit_handle(dside, unit));
	net_prep_change_side_action(unit, unit, side2);
    }
}

void
do_help(Side *side)
{
    popup_help(side, NULL);
}

/* Tell the unit to sit around for a given number of turns. */

void
do_idle(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    if (map->prefixarg < 0)
      map->prefixarg = 9999;
    net_set_sentry_task(unit, map->prefixarg);
}

void
do_map(Side *side)
{
    Map *map = side->ui->curmap;
    VP *vp;

    if (empty_string(cmdargstr))
      return;
    if (map == NULL)
      return;
    vp = widget_vp(map);
    if (strcmp(cmdargstr, "clouds") == 0) {
	vp->draw_clouds = !vp->draw_clouds;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "control") == 0) {
	vp->draw_control = !vp->draw_control;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "cover") == 0) {
	vp->draw_cover = !vp->draw_cover;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "elevations") == 0) {
	vp->draw_elevations = !vp->draw_elevations;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "fboundaries") == 0) {
	vp->draw_feature_boundaries = !vp->draw_feature_boundaries;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "fnames") == 0) {
	vp->draw_feature_names = !vp->draw_feature_names;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "grid") == 0) {
	vp->draw_grid = !vp->draw_grid;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "iso") == 0) {
	vp->isometric = TRUE;
	if (map->prefixarg >= 0)
	  vp->vertscale = map->prefixarg;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "lighting") == 0) {
	vp->draw_lighting = !vp->draw_lighting;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "lines") == 0) {
	toggle_mapw_lines(map);
	redraw_map(map);
    } else if (strcmp(cmdargstr, "meridians") == 0) {
	vp->draw_latlong = !vp->draw_latlong;
	if (map->prefixarg > 0)
	  vp->latlong_interval = map->prefixarg;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "over") == 0) {
	vp->isometric = FALSE;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "people") == 0) {
	vp->draw_people = !vp->draw_people;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "plan-none") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_NONE);
    } else if (strcmp(cmdargstr, "plan-passive") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_PASSIVE);
    } else if (strcmp(cmdargstr, "plan-defensive") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_DEFENSIVE);
    } else if (strcmp(cmdargstr, "plan-exploratory") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_EXPLORATORY);
    } else if (strcmp(cmdargstr, "plan-offensive") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_OFFENSIVE);
    } else if (strcmp(cmdargstr, "plan-random") == 0) {
	if (map->curunit)
	  net_set_unit_plan_type(dside, map->curunit, PLAN_RANDOM);
    } else if (strcmp(cmdargstr, "polygons") == 0) {
	toggle_mapw_polygons(map);
	redraw_map(map);
    } else if (strcmp(cmdargstr, "rotl") == 0) {
	vp->isodir = left_dir(vp->isodir);
	printf("Looking %s\n", dirnames[vp->isodir]);
	redraw_map(map);
    } else if (strcmp(cmdargstr, "rotr") == 0) {
	vp->isodir = right_dir(vp->isodir);
	printf("Looking %s\n", dirnames[vp->isodir]);
	redraw_map(map);
    } else if (strcmp(cmdargstr, "seeall") == 0) {
	if (!side->may_set_see_all) {
	    cmd_error(side, "May not set to see all!");
	    return;
	}
	map->see_all = !map->see_all;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "temperature") == 0) {
	vp->draw_temperature = !vp->draw_temperature;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "transitions") == 0) {
	toggle_mapw_transitions(map);
	redraw_map(map);
    } else if (strcmp(cmdargstr, "tstyle-solid") == 0) {
	map->terrain_style = 0;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "tstyle-pattern") == 0) {
	map->terrain_style = 1;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "tstyle-image") == 0) {
	map->terrain_style = 2;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "unames") == 0) {
	vp->draw_names = !vp->draw_names;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "unit-color") == 0) {
	int u;
	map->unit_style = 0;
	for_all_unit_types(u)
	  side->ui->uimages[u]->preferred_style = 0;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "unit-colorize") == 0) {
	map->colorize_units = !map->colorize_units;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "unit-silhouette") == 0) {
	int u;
	map->unit_style = 1;
	for_all_unit_types(u)
	  side->ui->uimages[u]->preferred_style = 1;
	redraw_map(map);
    } else if (strcmp(cmdargstr, "winds") == 0) {
	vp->draw_winds = !vp->draw_winds;
	redraw_map(map);
    } else if (*cmdargstr == '!') {
	eval_tcl_cmd("%s", cmdargstr + 1);
    } else {
	cmd_error(side, "Map command \"%s\" not recognized!", cmdargstr);
    }
}

/* Send a short (1 line) message to another player.  Some messages are
   recognized specially, causing various actions. */

void
do_message(Side *side)
{
    Map *map = side->ui->curmap;
    char prompt[BUFSIZE];
    Side *side2;

    side2 = side_n(map->prefixarg);
    if (side == side2) {
	sprintf(prompt, "Message to yourself: ");
    } else if (side2) {
	sprintf(prompt, "Message to %s: ", short_side_title(side2));
    } else {
	sprintf(prompt, "Broadcast to all: ");
    }
    map->argside = side2;
    ask_string(side, map, prompt, NULL, aux_message);
}

static void
aux_message(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    char *msg;
    SideMask sidemask;

    if (cancel)
      return;
    if (grok_string(side, map, &msg)) {
	if (empty_string(msg)) {
	    notify(side, "You keep your mouth shut.");
	    sidemask = NOSIDES;
	} else if (map->argside == NULL) {
	    notify(side, "You broadcast to everybody.", msg);
	    sidemask = ALLSIDES;
	} else {
	    notify(side, "You send the message.");
	    sidemask = add_side_to_set(map->argside, NOSIDES);
	}
	if (!empty_string(msg) && sidemask != NOSIDES)
	  net_send_message(side, sidemask, msg);
    } else {
	map->modalhandler = aux_message;
    }
}

/* Set unit to move to a given location.  */

/* The command proper. */

void
do_move_to(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    map->argunitid = unit->id;
    ask_position(side, map, "Move to where?", aux_move_to);
}

static void
aux_move_to(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y;
    Unit *unit;

    if (cancel)
      return;
    if (grok_position(side, map, &x, &y, NULL)) {
	unit = find_unit(map->argunitid);
#ifdef DESIGNERS
	if (side->designer) {
	    net_designer_teleport(unit, x, y, NULL);
	} else
#endif /* DESIGNERS */
	if (in_play(unit)) {
	    net_set_move_to_task(unit, x, y, 0);
	} else {
	    beep(side);
	}
    } else {
	map->modalhandler = aux_move_to;
    }
}

/* Name/rename the current unit. */

void
do_name(Side *side)
{
    Map *map = side->ui->curmap;
    char tmpbuf[BUFSIZE];
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    map->argunitid = unit->id;
    sprintf(tmpbuf, "New name for %s:", unit_handle(side, unit));
    ask_string(side, map, tmpbuf, unit->name, aux_name);
}

static void
aux_name(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    char *name;
    Unit *unit;

    if (cancel)
      return;
    if (grok_string(side, map, &name)) {
	unit = find_unit(map->argunitid);
	if (in_play(unit) && side_controls_unit(side, unit)) {
	    net_set_unit_name(side, unit, name);
	} else {
	    cmd_error(side, "Nothing here that could be named!");
	}
    } else {
	map->modalhandler = aux_name;
    }
}

/* Create a new map, of standard size and zoom. */
/* (should clone last map in list perhaps?) */

void
do_new_map(Side *side)
{
    cmd_error(side, "Not implemented.");
}

/* This is a command to examine all occupants and suboccupants, in an
   inorder fashion. */

/* Should have an option to open up a list window that shows everything
   all at once. */

void
do_occupant(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *nextocc;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    nextocc = find_next_occupant(unit);
    if (nextocc != unit) {
	set_current_unit(map, nextocc);
    }
}

void
do_orders_popup(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_other(Side *side)
{
    Map *map = side->ui->curmap;

    ask_string(side, map, "Command:", "", aux_others);
}

static void
aux_others(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    char *cmd;

    if (cancel)
      return;
    if (grok_string(side, map, &cmd)) {
	if (empty_string(cmd)) {
	    notify(side, "No command.");
	} else if (strcmp(cmd, "?") == 0) {
#if 0
	    do_help(side);
	    /* (should do with special jump routine) */
	    side->ui->curhelpnode = long_commands_help_node;
	    update_help(side);
#endif
	} else {
	    execute_long_command(side, cmd);
	}
    } else {
	map->modalhandler = aux_others;
    }
}

void
do_print_view(Side *side)
{  
    cmd_error(side, "Not implemented.");
}

void
do_produce(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_quit(Side *side)
{
    Map *map = side->ui->curmap;

    if (side->ingame) {
	if (all_others_willing_to_quit(side)) {
	    /* Everbody else is willing to get out, but confirm us anyway. */
	    ask_bool(side, map, "Do you really want to quit?", FALSE,
		     aux_kill_game);
	} else {
	    if (1 /* outcome needs resolution */) {
		/* if not everybody willing to draw, then we have to resign */
		ask_bool(side, map, "Do you really want to give up?", FALSE,
			 aux_resign);
	    } else {
		/* Everybody is just participating. */
		ask_bool(side, map, "Do you want to leave this game?", FALSE,
			 aux_leave_game);
	    }
	}
    } else {
	/* We're already out of the game, not really anything to confirm. */
	/* (is this common to all interfaces?) */
	if (all_others_willing_to_quit(side) || num_active_displays() == 1)
	  exit_xconq(side);
    }
}

static void
aux_resign(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	if (numsides > 2) {
	    /* (should suggest resigning to a trusted side) */
	    ask_side(side, map, "To whom do you wish to surrender?", NULL,
		     aux_resign_2);
	} else {
	    net_resign_game(side, NULL);
	}
    }
}

static void
aux_resign_2(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    Side *side2;

    if (cancel)
      return;
    if (grok_side(side, map, &side2)) {
	net_resign_game(side, side2);
    } else {
	/* Iterate until grok_side is happy. */
	map->modalhandler = aux_resign_2;
    }
}

/* Do the act of leaving the game. */

static void
aux_leave_game(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	remove_side_from_game(side);
	/* Now go back and see what happens if we're not in the game. */ 
	do_quit(side);
    } else {
	/* nothing to do if we said not to exit */
    }
}

/* (Have an extra confirm for designers not to lose unsaved work?) */

static void
aux_kill_game(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	exit_xconq(side);
	/* Nothing to do if we said not to exit. */
    }
}

/* Center the screen on the current location. */

void
do_recenter(Side *side)
{
    Map *map = dside->ui->curmap;

    REQUIRE_UNIT(map);
    recenter(map, map->curunit->x, map->curunit->y);
}

/* Redraw everything using the same code as when windows need a redraw. */

void
do_refresh(Side *side)
{
    Map *map;

    for_all_maps(map) {
	redraw_map(map);
    }
}

void
do_remove_terrain(Side *side)
{
    Map *map = side->ui->curmap;
    int u, t, numtypes, tfirst = NONTTYPE, possibles[MAXTTYPES];
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    u = unit->type;
    numtypes = 0;
    for_all_terrain_types(t) {
	if (ut_acp_to_remove_terrain(u, t) > 0) {
	    possibles[t] = TRUE;
	    ++numtypes;
	    tfirst = t;
	} else {
	    possibles[t] = FALSE;
	}
    }
    if (numtypes == 0) {
	cmd_error(side, "%s cannot remove terrain!", unit_handle(side, unit));
    } else if (numtypes == 1) {
	map->argunitid = unit->id;
	do_remove_terrain_2(side, map, tfirst);
    } else {
	map->argunitid = unit->id;
	ask_terrain_type(side, map, "Type to remove:",
			 possibles, aux_remove_terrain);
    }
}

static void
aux_remove_terrain(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int t;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Type a key to select terrain type to remove.");
    }
    if (grok_terrain_type(side, map, &t)) {
	if (t != NONTTYPE) {
	    do_remove_terrain_2(side, map, t);
	}
    } else {
        if (map->inpch != '?')
	  beep(side);
	/* Stay in this mode until we get it right. */
	map->modalhandler = aux_remove_terrain;
    }
}

/* This is like do_remove_terrain, but with a terrain type given. */

static void
do_remove_terrain_2(side, map, t)
Side *side;
Map *map;
int t;
{
    map->tmpt = t;
    ask_position(side, map, "Remove where?", aux_remove_terrain_2);
}

static void
aux_remove_terrain_2(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y, dir;
    Unit *unit;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Pick a location to remove %s.", t_type_name(map->tmpt));
	map->modalhandler = aux_remove_terrain_2;
	return;
    }
    if (grok_position(side, map, &x, &y, NULL)) {
	if (in_area(x, y)) {
	    unit = find_unit(map->argunitid);
	    if (in_play(unit) && side_controls_unit(side, unit)) {
		switch (t_subtype(map->tmpt)) {
		  case cellsubtype:
		    cmd_error(side, "can't remove cell terrain!");
		    break;
		  case bordersubtype:
		  case connectionsubtype:
		    dir = closest_dir(x - unit->x, y - unit->y);
		    net_prep_remove_terrain_action(unit, unit, unit->x, unit->y, dir, map->tmpt);
		    break;
		  case coatingsubtype:
		    net_prep_remove_terrain_action(unit, unit, unit->x, unit->y, 1, map->tmpt);
		    break;
		}
	    }
	}
    } else {
	beep(side);     
	map->modalhandler = aux_remove_terrain_2;
    }
}

void
do_repair(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit, *other;

    REQUIRE_UNIT(map);
    unit = map->curunit;
    if (valid(check_repair_action(unit, unit, unit))) {
	net_prep_repair_action(unit, unit, unit);
	return;
    }
    other = unit->transport;
    if (other != NULL
	&& valid(check_repair_action(other, other, unit))) {
	net_prep_repair_action(other, other, unit);
	return;
    }
    cmd_error(side, "No repair possible right now");
}

void
do_reserve(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    net_set_unit_reserve(side, map->curunit, TRUE, TRUE /* is this right??? */);
}

void
do_resign(Side *side)
{
    Map *map = side->ui->curmap;

    if (endofgame) {
	notify(side, "Game is already over.");
	beep(side);
    } else if (!side->ingame) {
	notify(side, "You are already out of the game.");
	beep(side);
    } else {
	ask_bool(side, map, "You really want to resign?", FALSE, aux_resign_b);
    }
}

/* This is semi-redundant with aux_resign. */

static void
aux_resign_b(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	net_resign_game(side, NULL);
    }
}

/* Set up a task to resupply the unit. */

/* (should warn if task is likely to fail) */

void
do_return(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    /* (should doublecheck range and error out if no chance) */
    if (in_play(unit)) {
	net_set_resupply_task(unit, NONMTYPE);
    }
}

/* Stuff game state into a file.  By default, it goes into the current
   directory.  If designing a game, we can specify exactly which parts
   of the game state are to be written. */

/* The command proper just sets up different modal handlers, depending on
   whether we're building (and therefore saving a scenario/fragment), or
   saving as much game state as possible, for resumption later. */

void
do_save(Side *side)
{
    Map *map = side->ui->curmap;

#ifdef DESIGNERS
    if (side->designer) {
	eval_tcl_cmd("popup_designer_save");
	return;
    }
#endif /* DESIGNERS */
    if (0 /* (should be "savemustquit") */) {
	ask_bool(side, map, "You really want to save?", FALSE, aux_save_2);
    } else {
	save_the_game(side);
    }
}

/* Make a module appropriate to a save file, write the file, and leave. */

static void
aux_save_2(Side *side, Map *map, int cancel)
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	save_the_game(side);
    }
}

static void
save_the_game(Side *side)
{
    char *savename = saved_game_filename();

    /* (should use name in save dialog) */
    notify_all("Game will be saved to \"%s\" ...", savename);
    if (write_entire_game_state(savename)) {
	close_displays();
	/* this should be conditional? */
	exit(0);
    } else {
	cmd_error(side, "Can't open file \"%s\"!", savename);
    }
}

#ifdef DESIGNERS

static Module *tmpmodule;

static void
aux_save_1_1(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    char *filenamespec;

    if (cancel)
      return;
    if (grok_string(side, map, &filenamespec)) {
	tmpmodule->filename = filenamespec;
	notify(side, "File will be written to \"%s\" ...",
	       tmpmodule->filename);
	if (write_game_module(tmpmodule)) {
	    notify(side, "Done writing to \"%s\".", tmpmodule->filename);
	} else {
	    cmd_error(side, "Can't open file \"%s\"!", tmpmodule->filename);
	}
    } else {
	map->modalhandler = aux_save_1_1;
    }
}

static void
aux_save_1(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int save;
    char *contentspec;

    if (cancel)
      return;
    if (grok_string(side, map, &contentspec)) {
	tmpmodule = create_game_module("game-data");
	/* Use the spec string to decide which pieces to save. */
	save = FALSE;
	if (strcmp(contentspec, "all") == 0) {
	    tmpmodule->def_all = TRUE;
	    tmpmodule->compress_layers = TRUE;
	    save = TRUE;
	} else if (strcmp(contentspec, "world") == 0) {
	    tmpmodule->def_world = TRUE;
	    tmpmodule->def_areas = TRUE;
	    tmpmodule->def_area_terrain = TRUE;
	    tmpmodule->def_area_misc = TRUE;
	    tmpmodule->def_area_weather = TRUE;
	    tmpmodule->def_area_material = TRUE;
	    save = TRUE;
	} else {
	    cmd_error(side, "Don't understand content spec \"%s\"!",
		      contentspec);
	    return;
	}
	if (save) {
	    ask_string(side, map, "Save data to where?", "game-data.g",
		       aux_save_1_1);
	} else {
	    notify(side, "Nothing requested to be saved.");
	}
    } else {
	map->modalhandler = aux_save_1;
    }
}

#endif /* DESIGNERS */

void
do_set_formation(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    map->argunitid = map->curunit->id;
    ask_position(side, map, "Form up on who?", aux_set_formation);
}

static void
aux_set_formation(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    int x, y;
    Unit *unit, *leader;

    if (cancel)
      return;
    if (map->inpch == '?') {
	notify(side, "Click on a unit that you want to follow.");
	map->modalhandler = aux_set_formation;
	return;
    }
    if (grok_position(side, map, &x, &y, NULL)) {
	unit = find_unit(map->argunitid);
	if (in_play(unit) && in_area(x, y)) {
	    leader = unit_at(x, y);
	    if (in_play(leader)
		&& leader != unit
		&& trusted_side(unit->side, leader->side)) {
		net_set_formation(unit, leader,
				  unit->x - leader->x, unit->y - leader->y,
				  1, 1);
	    } else {
		cmd_error(side, "Nobody here to form on!");
	    }
	}
    } else {
	beep(side);
	map->modalhandler = aux_set_formation;
    }
}

void
do_set_view_angle(Side *side)
{
    Map *map = dside->ui->curmap;
    VP *vp = widget_vp(map);

    vp->isometric = !vp->isometric;
    if (map->prefixarg >= 0)
      vp->vertscale = map->prefixarg;
    redraw_map(map);
}

void
do_side_closeup(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_sleep(Side *side)
{
    Map *map = side->ui->curmap;
    Unit *unit = map->curunit;

    REQUIRE_UNIT(map);
    net_set_unit_asleep(side, unit, TRUE, TRUE /* is this right??? */);
}

void
do_standing_orders(Side *side)
{
    Map *map = side->ui->curmap;
    int rslt;

    if (cmdargstr) {
	rslt = parse_standing_order(side, cmdargstr);
	if (rslt < 0)
	  cmd_error(side, "Parse error");
    } else
      cmd_error(side, "No arguments given.");
}

void
do_survey(Side *side)
{
    Map *map = side->ui->curmap;

    if (map->mode == survey_mode) {
	map->mode = move_mode;
	if (side->designer)
	  map->mode = survey_mode; /* should be last design mode */
    } else if (map->mode == move_mode) {
	map->mode = survey_mode;
    } else if (side->designer) {
	map->mode = survey_mode;
    } else {
	beep(side);
	return;
    }
    if (!side->designer) {
	map->autoselect = !map->autoselect;
	map->move_on_click = !map->move_on_click;
    }
    set_tool_cursor(side, map);
    if (!map->autoselect && map->curunit != NULL)
      update_at_unit(map, map->curunit);
    if (map->mode == move_mode)
      eval_tcl_cmd("update_mode move");
    else if (map->mode == survey_mode)
      eval_tcl_cmd("update_mode survey");
}

/* Take supplies from transport. */

void
do_take(Side *side)
{
    Map *map = side->ui->curmap;
    short m, amts[MAXMTYPES], rslts[MAXMTYPES], needed;
    Unit *unit = map->curunit;

    if (nummtypes == 0) {
	cmd_error(side, "No materials in this game!");
	return;
    }
    REQUIRE_UNIT(map);
    for_all_material_types(m) {
	amts[m] = map->prefixarg;
    }
    needed = take_supplies(unit, amts, rslts);
    report_take(side, unit, needed, rslts);
}

void
do_unit_closeup(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_up(Side *side)
{
    cmd_error(side, "Not implemented.");
}

/* Command to display the program version. */

void
do_version(Side *side)
{
    notify(side, "Xconq version %s", version_string());
    notify(side, "(c) %s", copyright_string());
}

void
do_wake(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    /* Do the curunit explicitly, might only be an occupant. */
    if (in_play(map->curunit)) {
	net_wake_unit(side, map->curunit, FALSE);
    }
    /* If an argument was given, apply to all "top-level" units within
       the radius specified by the argument. */
    if (map->prefixarg >= 0)
      net_wake_area(side, map->curunit->x, map->curunit->y, map->prefixarg,
		    FALSE);
}

void
do_wake_all(Side *side)
{
    Map *map = side->ui->curmap;

    REQUIRE_UNIT(map);
    /* Do the curunit explicitly, might only be an occupant. */
    if (in_play(map->curunit)) {
	net_wake_unit(side, map->curunit, TRUE);
    }
    /* If an argument was given, apply to all "top-level" units within
       the radius specified by the argument. */
    if (map->prefixarg >= 0)
      net_wake_area(side, map->curunit->x, map->curunit->y, map->prefixarg,
		    TRUE);
}

void
do_warranty(Side *side)
{
    popup_help(side, warranty_help_node);
}

/* Create a new world map (a regular map zoomed to display the whole
   world at once). */

void
do_world_map(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_zoom_in(Side *side)
{
    zoom_in_out(side, side->ui->curmap, 1);
}

void
do_zoom_out(Side *side)
{
    zoom_in_out(side, side->ui->curmap, -1);
}

#ifdef DESIGNERS

void
do_design(Side *side)
{
    Map *map = side->ui->curmap;

    int u;
    Map *map2;

    if (!side->designer) {
	if (!designed_on) {
	    ask_bool(side, map, "Do you really want to start designing?",
		     FALSE, aux_design);
	} else {
	    really_do_design(side);
	}
    } else {
	net_become_nondesigner(side);
	for_all_maps(map2) {
	    map2->see_all = (all_see_all || dside->may_set_see_all);
	}
	eval_tcl_cmd("dismiss_design_palette");
    }
}

static void
aux_design(side, map, cancel)
Side *side;
Map *map;
int cancel;
{
    if (cancel)
      return;
    if (grok_bool(side, map)) {
	really_do_design(side);
    } else {
	/* nothing to do if we said not to design */
    }
}

void
really_do_design(Side *side)
{
    int u;
    Map *map;

    net_become_designer(side);
    for_all_maps(map) {
	map->see_all = (all_see_all || dside->may_set_see_all);
	map->autoselect = FALSE;
	map->mode = survey_mode;
    }
    eval_tcl_cmd("update_mode survey");
    eval_tcl_cmd("popup_design_palette");
}

#endif /* DESIGNERS */

#ifdef DEBUGGING

void
do_profile(Side *side)
{
    cmd_error(side, "Not implemented.");
}

void
do_trace(Side *side)
{
    cmd_error(side, "Not implemented.");
}

#endif /* DEBUGGING */

/* Generic command error feedback. */

void
cmd_error(Side *side, char *fmt, ...)
{
    va_list ap;

    if (!empty_string(fmt)) {
	va_start(ap, fmt);
	vnotify(side, fmt, ap);
	va_end(ap);
    }
    /* Only beep once, even if a command generates multiple error messages. */
    if (side->ui->beepcount++ < 1)
      beep(side);
}
