/* Imf sample list code for the tcl/tk interface to Xconq.
   Copyright (C) 1998 Stanley T. Shebs.

Xconq is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.  See the file COPYING.  */

/* This file implements the "imfsample" widget, which is used for
   displaying a list of given image families in a gridded pattern.
   The case of one image family is useful for displaying images
   embedded in other windows. */

/* Note that this widget depends only on image machinery, and has
   no knowledge of Xconq constructs such as units and sides. */

#include "config.h"
#include "misc.h"
#include "lisp.h"
#include "imf.h"

#include <tcl.h>
#include <tk.h>

#include <math.h>

#include "tkimf.h"

int imfsample_cmd(ClientData cldata, Tcl_Interp *interp,
		  int argc, char *argv[]);

extern Tcl_Interp *interp;

/* MAXROWS, MAXCOLS, MINROWS, and MINCOLS can be changed safely */
#define MAXROWS 50
#define MAXCOLS 16
#define MINROWS 2
#define MINCOLS 8

#define MAXFAMS (MAXROWS*MAXCOLS)

extern int tmp_valid;

extern Display *tmp_display;

static XColor *black_color;
static XColor *white_color;

/* Image family list display widget. */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the imfsample.  NULL
				 * means window has been deleted but
				 * widget record hasn't been cleaned up yet. */
    Display *display;		/* X's token for the window's display. */
    Tcl_Interp *interp;		/* Interpreter associated with widget. */
    Tcl_Command widgetCmd;	/* Token for imfsample's widget command. */
    int border_width;		/* Width of 3-D border around whole widget. */
    Tk_3DBorder bg_border;	/* Used for drawing background. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */
    GC gc;			/* Graphics context for copying from
				 * off-screen pixmap onto screen. */
    int double_buffer;		/* Non-zero means double-buffer redisplay
				 * with pixmap;  zero means draw straight
				 * onto the display. */
    int update_pending;		/* Non-zero means a call to imfsample_display
				 * has already been scheduled. */
 
    int pad;			/* Extra space between images. */
    int iwidth, iheight;
    int show_color;
    int show_names;
    int show_masks;

    int width, height;

    XColor *fill_color;

    char *main_imf_name;
    int numimages;
    ImageFamily **imf_list;

    ImageFamily *fill_tile;
    ImageFamily *all_emblem;

    int selected;

    int eltw, elth;
    int cols;

} Imfsample;

static Tk_ConfigSpec config_specs[] = {
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"white", Tk_Offset(Imfsample, bg_border), TK_CONFIG_COLOR_ONLY},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	"white", Tk_Offset(Imfsample, bg_border), TK_CONFIG_MONO_ONLY},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	"0", Tk_Offset(Imfsample, border_width), 0},
    {TK_CONFIG_INT, "-dbl", "doubleBuffer", "DoubleBuffer",
	"1", Tk_Offset(Imfsample, double_buffer), 0},
    {TK_CONFIG_PIXELS, "-height", "height", "Height",
	"32", Tk_Offset(Imfsample, height), 0},
#if 0 /* disabled until we debug */
    {TK_CONFIG_STRING, "-imf", "imf", "Imf",
	"0", Tk_Offset(Imfsample, main_imf_name), 0},
#endif
    {TK_CONFIG_INT, "-iheight", "iheight", "IHeight",
	"0", Tk_Offset(Imfsample, iheight), 0},
    {TK_CONFIG_INT, "-iwidth", "iwidth", "IWidth",
	"0", Tk_Offset(Imfsample, iwidth), 0},
    {TK_CONFIG_INT, "-pad", "pad", "Pad",
	"0", Tk_Offset(Imfsample, pad), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	"raised", Tk_Offset(Imfsample, relief), 0},
    {TK_CONFIG_INT, "-showcolor", "showColor", "ShowColor",
	"1", Tk_Offset(Imfsample, show_color), 0},
    {TK_CONFIG_INT, "-shownames", "showNames", "ShowNames",
	"1", Tk_Offset(Imfsample, show_names), 0},
    {TK_CONFIG_INT, "-showmasks", "showMasks", "ShowMasks",
	"1", Tk_Offset(Imfsample, show_masks), 0},
    {TK_CONFIG_COLOR, "-fillcolor", "fillColor", "FillColor",
	"gray", Tk_Offset(Imfsample, fill_color), 0},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	"32", Tk_Offset(Imfsample, width), 0},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};

static void imfsample_cmd_deleted_proc PARAMS ((ClientData cldata));
static int imfsample_configure PARAMS ((Tcl_Interp *interp, Imfsample *imfsample,
				      int argc, char **argv, int flags));
static void imfsample_destroy PARAMS ((char *memPtr));
static void imfsample_display PARAMS ((ClientData cldata));
static void imfsample_event_proc PARAMS ((ClientData cldata,
					XEvent *eventPtr));
static int imfsample_widget_cmd PARAMS ((ClientData cldata,
				     Tcl_Interp *interp,
				     int argc, char **argv));

int
imfsample_cmd(cldata, interp, argc, argv)
ClientData cldata;
Tcl_Interp *interp;
int argc;
char **argv;
{
    Tk_Window main = (Tk_Window) cldata;
    Imfsample *imfsample;
    Tk_Window tkwin;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    tkwin = Tk_CreateWindowFromPath(interp, main, argv[1], (char *) NULL);
    if (tkwin == NULL)
      return TCL_ERROR;

    Tk_SetClass(tkwin, "Imfsample");

    /* Allocate and initialize the widget record.  */

    imfsample = (Imfsample *) ckalloc(sizeof(Imfsample));
    imfsample->tkwin = tkwin;
    imfsample->display = Tk_Display(tkwin);
    imfsample->interp = interp;
    imfsample->widgetCmd =
      Tcl_CreateCommand(interp,
			Tk_PathName(imfsample->tkwin), imfsample_widget_cmd,
			(ClientData) imfsample, imfsample_cmd_deleted_proc);
    imfsample->border_width = 0;
    imfsample->bg_border = NULL;
    imfsample->relief = TK_RELIEF_FLAT;
    imfsample->gc = None;
    imfsample->double_buffer = 1;
    imfsample->update_pending = 0;
    imfsample->show_color = 1;
    imfsample->show_names = 0;
    imfsample->show_masks = 0;
    imfsample->fill_color = NULL;
    imfsample->selected = -1;

    imfsample->main_imf_name = "";
    imfsample->numimages = 0;
    imfsample->imf_list =
      (ImageFamily **) xmalloc(MAXFAMS * sizeof(ImageFamily *));

    Tk_CreateEventHandler(imfsample->tkwin, ExposureMask|StructureNotifyMask,
			  imfsample_event_proc, (ClientData) imfsample);
    if (imfsample_configure(interp, imfsample, argc-2, argv+2, 0) != TCL_OK) {
	Tk_DestroyWindow(imfsample->tkwin);
	return TCL_ERROR;
    }

    interp->result = Tk_PathName(imfsample->tkwin);
    return TCL_OK;
}

static int
imfsample_widget_cmd(cldata, interp, argc, argv)
ClientData cldata;
Tcl_Interp *interp;
int argc;
char **argv;
{
    Imfsample *imfsample = (Imfsample *) cldata;
    int result = TCL_OK;
    size_t length;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) imfsample);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
	&& (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " cget option\"",
		    (char *) NULL);
	    goto error;
	}
	result = Tk_ConfigureValue(interp, imfsample->tkwin, config_specs,
		(char *) imfsample, argv[2], 0);
    } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	       && (length >= 2)) {
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, imfsample->tkwin, config_specs,
		    (char *) imfsample, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, imfsample->tkwin, config_specs,
		    (char *) imfsample, argv[2], 0);
	} else {
	    result = imfsample_configure(interp, imfsample, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    } else if ((c == 'c') && (strncmp(argv[1], "curselection", length) == 0)
	       && (length >= 2)) {
	sprintf(interp->result, "%d", imfsample->selected);
    } else if ((c == 'a') && (strncmp(argv[1], "add", length) == 0)
	       && (length >= 2)) {
	if (strcmp(argv[2], "imf") == 0) {
	    imfsample_add_imf(imfsample, argv[3]);
	}
    } else if ((c == 'r') && (strncmp(argv[1], "replace", length) == 0)
	       && (length >= 2)) {
	if (strcmp(argv[2], "imf") == 0) {
	    imfsample_replace_imf(imfsample, argv[3]);
	}
    } else if ((c == 'r') && (strncmp(argv[1], "remove", length) == 0)
	       && (length >= 2)) {
	if (strcmp(argv[2], "imf") == 0) {
	    imfsample_remove_imf(imfsample, argv[3]);
	}
    } else if ((c == 's') && (strncmp(argv[1], "select", length) == 0)
	       && (length >= 2)) {
	int x, y, col, row, n;
	int firstvisrow = 0;

	x = strtol(argv[2], NULL, 10);
	y = strtol(argv[3], NULL, 10);
	col = x / imfsample->eltw;
	row = y / imfsample->elth + firstvisrow;
	n = row * imfsample->cols + col;
	if (n < 0 || n >= imfsample->numimages)
	  n = -1;
	imfsample->selected = n;
    } else if ((c == 'x') && (strncmp(argv[1], "xview", length) == 0)) {
    } else if ((c == 'y') && (strncmp(argv[1], "yview", length) == 0)) {
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be cget, configure, position, or size",
		(char *) NULL);
	goto error;
    }
    if (!imfsample->update_pending) {
	Tcl_DoWhenIdle(imfsample_display, (ClientData) imfsample);
	imfsample->update_pending = 1;
    }
    Tcl_Release((ClientData) imfsample);
    return result;

error:
    Tcl_Release((ClientData) imfsample);
    return TCL_ERROR;
}

imfsample_add_imf(imfsample, imfname)
Imfsample *imfsample;
char *imfname;
{
    int i;
    ImageFamily *imf = NULL;

    imf = hack_find_imf(imfname);
    if (imf == NULL) {
	fprintf(stderr, "Missing imf %s\n", imfname);
	return;
    }
    imfsample->imf_list[imfsample->numimages++] = imf;
}

imfsample_replace_imf(imfsample, imfname)
     Imfsample *imfsample;
     char *imfname;
{
    int i;
    ImageFamily *imf = NULL;

    if (imfsample->numimages == 0) {
	imfsample_add_imf(imfsample, imfname);
	return;
    }
    imf = hack_find_imf(imfname);
    if (imf == NULL) {
	fprintf(stderr, "Missing imf %s\n", imfname);
	return;
    }
    imfsample->imf_list[0] = imf;
}

imfsample_remove_imf(imfsample, imfname)
     Imfsample *imfsample;
     char *imfname;
{
    /* (should look for specific imf) */
    imfsample->numimages = 0;
}

static int
imfsample_configure(interp, imfsample, argc, argv, flags)
Tcl_Interp *interp;
Imfsample *imfsample;
int argc;
char **argv;
int flags;
{
    /* Interpret the configuration arguments according to the specs. */
    if (Tk_ConfigureWidget(interp, imfsample->tkwin, config_specs,
			   argc, argv, (char *) imfsample, flags) != TCL_OK)
      return TCL_ERROR;

    /* Set the background for the window and create a graphics context
       for use during redisplay.  */
    Tk_SetWindowBackground(imfsample->tkwin,
			   Tk_3DBorderColor(imfsample->bg_border)->pixel);
    if ((imfsample->gc == None) && imfsample->double_buffer) {
	XGCValues gcValues;

	gcValues.function = GXcopy;
	gcValues.graphics_exposures = False;
	imfsample->gc = Tk_GetGC(imfsample->tkwin,
				  GCFunction|GCGraphicsExposures, &gcValues);
    }

    /* Register the desired geometry for the window, then arrange for
       the window to be redisplayed.  */
    Tk_GeometryRequest(imfsample->tkwin, imfsample->width, imfsample->height);
    Tk_SetInternalBorder(imfsample->tkwin, imfsample->border_width);
    if (!imfsample->update_pending) {
	Tcl_DoWhenIdle(imfsample_display, (ClientData) imfsample);
	imfsample->update_pending = 1;
    }
    return TCL_OK;
}

static void
imfsample_event_proc(cldata, eventPtr)
ClientData cldata;
XEvent *eventPtr;
{
    Imfsample *imfsample = (Imfsample *) cldata;

    if (eventPtr->type == Expose) {
	if (!imfsample->update_pending) {
	    Tcl_DoWhenIdle(imfsample_display, cldata);
	    imfsample->update_pending = 1;
	}
    } else if (eventPtr->type == ConfigureNotify) {
	if (!imfsample->update_pending) {
	    Tcl_DoWhenIdle(imfsample_display, cldata);
	    imfsample->update_pending = 1;
	}
    } else if (eventPtr->type == DestroyNotify) {
	if (imfsample->tkwin != NULL) {
	    imfsample->tkwin = NULL;
	    Tcl_DeleteCommand(imfsample->interp,
			      Tcl_GetCommandName(imfsample->interp,
						 imfsample->widgetCmd));
	}
	if (imfsample->update_pending) {
	    Tcl_CancelIdleCall(imfsample_display, cldata);
	}
	Tcl_EventuallyFree(cldata, imfsample_destroy);
    }
}

static void
imfsample_cmd_deleted_proc(cldata)
ClientData cldata;
{
    Imfsample *imfsample = (Imfsample *) cldata;
    Tk_Window tkwin = imfsample->tkwin;

    if (tkwin != NULL) {
	imfsample->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

static void
imfsample_display(ClientData cldata)
{
    char *str;
    int rows, row, col, namex, namey, i, n, sx, sy;
    int firstvisrow = 0, numvisrows;
    Imfsample *imfsample = (Imfsample *) cldata;
    Display *dpy = imfsample->display;
    Tk_Window tkwin = imfsample->tkwin;
    GC gc;
    Pixmap pm = None;
    Drawable d;
    Tk_Font tkfont;
    int winwidth = Tk_Width(tkwin), winheight = Tk_Height(tkwin);

    imfsample->update_pending = 0;
    if (!Tk_IsMapped(tkwin))
      return;
    /* Create a pixmap for double-buffering if necessary. */
    if (imfsample->double_buffer) {
	pm = Tk_GetPixmap(imfsample->display, Tk_WindowId(tkwin),
			  Tk_Width(tkwin), Tk_Height(tkwin),
			  DefaultDepthOfScreen(Tk_Screen(tkwin)));
	d = pm;
    } else {
	d = Tk_WindowId(tkwin);
    }
    if (black_color == NULL)
      black_color = Tk_GetColor(interp, tkwin, Tk_GetUid("black"));
    if (white_color == NULL)
      white_color = Tk_GetColor(interp, tkwin, Tk_GetUid("white"));
    /* Collect GC and font for subsequent work. */
    gc = Tk_GetGC(imfsample->tkwin, None, NULL);
    XSetClipMask(dpy, gc, None);
    if (imfsample->show_names)
      tkfont = Tk_GetFont(interp, tkwin, "system");

    /* Redraw the widget's background and border. */
    Tk_Fill3DRectangle(tkwin, d, imfsample->bg_border, 0, 0, Tk_Width(tkwin),
		       Tk_Height(tkwin), imfsample->border_width, imfsample->relief);

#if 0
    for (i = 0; i < imfsample->numimages; i++) {
	if (imf_interp_hook)
	  imfsample->imf_list[i] =
	    (*imf_interp_hook)(imfsample->imf_list[i], NULL, TRUE);
    }
#endif
    /* Tweak the default item width/height to something better. */
    if (imfsample->iheight == 0) {
	imfsample->iheight = 32;
	if (imfsample->numimages == 1)
	  imfsample->iheight = imfsample->height;
    }
    if (imfsample->iwidth == 0) {
	imfsample->iwidth = 32;
	if (imfsample->numimages == 1)
	  imfsample->iwidth = imfsample->width;
    }
    imfsample->eltw = imfsample->iwidth + 2 * imfsample->pad;
    if (imfsample->show_names)
      imfsample->eltw += 80;
    imfsample->elth = imfsample->iheight + 2 * imfsample->pad;
    /* Fix a lower bound on the vertical spacing. */
    /* (should be determined by choice of app font) */
    if (imfsample->elth < 10)
      imfsample->elth = 10;
    /* Compute and save the number of columns to use. */
    imfsample->cols = winwidth / imfsample->eltw;
    if (imfsample->cols <= 0)
      imfsample->cols = 1;
    /* We can get a little wider spacing by recalculating the element
       width. */
    if (imfsample->show_names)
      imfsample->eltw = (winwidth - 10) / imfsample->cols;
    rows = imfsample->numimages / imfsample->cols;
    /* Account for a last partial row. */
    if (rows * imfsample->cols < imfsample->numimages)
      ++rows;
    /* Compute the number of visible rows.  It would be time-consuming
       to render all the images, so try to do only the visible ones. */
    numvisrows = winheight / imfsample->elth;
    if (numvisrows > rows)
      numvisrows = rows;
    if (numvisrows < 1)
      numvisrows = min(1, rows);
    if (firstvisrow + numvisrows > rows)
      firstvisrow = rows - numvisrows;
    /* Now iterate through all the images we want to draw. */
    for (row = firstvisrow; row < (firstvisrow + numvisrows); ++row) {
	for (col = 0; col < imfsample->cols; ++col) {
	    n = row * imfsample->cols + col;
	    if (n >= imfsample->numimages)
	      break;
	    sx = col * imfsample->eltw + imfsample->pad;
	    sy = (row - firstvisrow) * imfsample->elth + imfsample->pad;
	    draw_one_image(imfsample, d, gc, imfsample->imf_list[n], sx, sy,
			   imfsample->iwidth, imfsample->iheight);
	    if (imfsample->show_names) {
		namex = col * imfsample->eltw
		  + imfsample->iwidth + imfsample->pad + 2;
		namey = (row - firstvisrow) * imfsample->elth
		  + (imfsample->elth / 2) + 5;
		XSetClipMask(dpy, gc, None);
		XSetFillStyle(dpy, gc, FillSolid);
		str = images[n]->name;
		/* Draw the text in black only. */
		/* (should invent a way to make show up when black bg; an
		   offset version in white is semi-OK, but hard on eyes) */
		XSetForeground(dpy, gc, black_color->pixel);
		Tk_DrawChars(dpy, d, gc, tkfont, str, strlen(str),
			     namex, namey);
	    }
	    /* Indicate the selected imf. */
	    if (n == imfsample->selected) {
		XSetClipMask(dpy, gc, None);
		XSetForeground(dpy, gc, black_color->pixel);
		XDrawRectangle(dpy, d, gc, col * imfsample->eltw, (row - firstvisrow) * imfsample->elth, imfsample->eltw, imfsample->elth);
	    }
	}
    }
    /* If double-buffered, copy to the screen and release the pixmap.  */
    if (imfsample->double_buffer) {
	XCopyArea(imfsample->display, pm, Tk_WindowId(tkwin), imfsample->gc,
		  0, 0, (unsigned) winwidth, (unsigned) winheight,
		  0, 0);
	Tk_FreePixmap(imfsample->display, pm);
    }
    Tk_FreeGC(imfsample->display, gc);
    if (imfsample->show_names)
      Tk_FreeFont(tkfont);
}

static void
imfsample_destroy(char *ptr)
{
    Imfsample *imfsample = (Imfsample *) ptr;

    Tk_FreeOptions(config_specs, (char *) imfsample, imfsample->display, 0);
    if (imfsample->gc != None)
      Tk_FreeGC(imfsample->display, imfsample->gc);
    ckfree((char *) imfsample);
}

/* Draw a single image at a given location. */

int
draw_one_image(imfsample, d, gc, imf, sx, sy, sw, sh)
Imfsample *imfsample;
Drawable d;
GC gc;
ImageFamily *imf;
int sx, sy;
{
    int sx2, sy2;
    Image *uimg;
    TkImage *tkimg;
    Tk_Window tkwin = imfsample->tkwin;
    Display *dpy = Tk_Display(tkwin);
    XColor *imagecolor = black_color;
    XColor *maskcolor = white_color;

    uimg = best_image(imf, sw, sh);
    if (uimg == NULL)
      return;
    if (!uimg->istile) {
	/* Offset the image to draw in the middle of its area,
	   whether larger or smaller than the given area. */
	sx2 = sx + (sw - uimg->w) / 2;  sy2 = sy + (sh - uimg->h) / 2;
	/* Only change the size of the rectangle being drawn if it's
	   smaller than what was passed in. */
	if (uimg->w < sw) {
	    sx = sx2;
	    sw = uimg->w;
	}
	if (uimg->h < sh) {
	    sy = sy2;
	    sh = uimg->h;
	}
    }
    tkimg = (TkImage *) uimg->hook;
    if (tkimg == NULL)
      return;
    XSetClipMask(dpy, gc, None);
    XSetClipOrigin(dpy, gc, sx2, sy2);
    XSetFillStyle(dpy, gc, FillSolid);
    if (tkimg->colr != None && imfsample->show_color) {
	if (uimg->istile) {
	    XSetFillStyle(dpy, gc, FillTiled);
	    XSetTile(dpy, gc, tkimg->colr);
	    XFillRectangle(dpy, d, gc, sx, sy, sw, sh);
	} else {
	    if (tkimg->mask != None && imfsample->show_masks) {
		/* Use the image's mask to do clipped drawing. */
		XSetClipMask(dpy, gc, tkimg->mask);
	    }
	    /* Draw the color image. */
	    XCopyArea(dpy, tkimg->colr, d, gc, 0, 0, sw, sh, sx, sy);
	}
    } else if (tkimg->solid != NULL && imfsample->show_color) {
	XSetForeground(dpy, gc, tkimg->solid->pixel);
	XFillRectangle(dpy, d, gc, sx, sy, sw, sh);
    } else if (tkimg->mono != None || tkimg->mask != None) {
	/* Set the color we're going to use for the mask; use the
	   imagecolor if we'll be using the mask as the only image. */
	XSetForeground(dpy, gc,
		       (tkimg->mono == None ? imagecolor : maskcolor)->pixel);
	/* Set the clip mask to be explicit mask or unit's image. */
	if (tkimg->mask && imfsample->show_masks)
	  XSetClipMask(dpy, gc, tkimg->mask);
	else
	  XSetClipMask(dpy, gc, tkimg->mono);
	/* Draw the mask. */
	XFillRectangle(dpy, d, gc, sx, sy, sw, sh);
	/* Draw the image proper. */
	if (tkimg->mono != None) {
	    XSetForeground(dpy, gc, imagecolor->pixel);
	    XSetClipMask(dpy, gc, tkimg->mono);
	    XFillRectangle(dpy, d, gc, sx, sy, sw, sh);
	}
    }
}

