
/*--------------------------------------------------------------------*/
/*--- Management of the translation table and cache.               ---*/
/*---                                                vg_transtab.c ---*/
/*--------------------------------------------------------------------*/

/*
   This file is part of Valgrind, an extensible x86 protected-mode
   emulator for monitoring program execution on x86-Unixes.

   Copyright (C) 2000-2005 Julian Seward 
      jseward@acm.org

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "core.h"
#include "memcheck/memcheck.h"

/* #define DEBUG_TRANSTAB */


/*-------------------------------------------------------------*/
/*--- Management of the FIFO-based translation table+cache. ---*/
/*-------------------------------------------------------------*/

/*------------------ CONSTANTS ------------------*/

/* Number of sectors the TC is divided into. */
#define VG_TC_N_SECTORS 8

/* Calculated once at startup and never changed. */
static /* const */ Int vg_tc_sector_szB = 0;

/* Number of entries in the translation table.  This must be a prime
   number in order to make the hashing work properly. */
#define VG_TT_SIZE /*5281*/ /*100129*/ /*200191*/ /*250829*/ 300007

/* Do an LRU pass when the translation table becomes this full. */
#define VG_TT_LIMIT_PERCENT /*67*/ 80

#define VG_TT_LIMIT ((VG_TT_SIZE * VG_TT_LIMIT_PERCENT) / 100)


/*------------------ TYPES ------------------*/

#define CODE_ALIGNMENT	sizeof(void*)     // alignment of TCEntries
#define CODE_ALIGN(a)	(((a)+CODE_ALIGNMENT-1) & ~(CODE_ALIGNMENT-1))
#define IS_ALIGNED(a)	(((a) & (CODE_ALIGNMENT-1)) == 0)

/* An entry in TT. */
typedef
   struct {
      Addr     orig_addr;
      TCEntry* tcentry;
   }
   TTEntry;

/* Denotes an empty TT slot, when TTEntry.orig_addr holds this
   value. */
#define VG_TTE_EMPTY ((Addr)1)

/* Denotes an empty TT slot, when TTEntry.orig_addr holds this
   value. */
#define VG_TTE_DELETED ((Addr)3)

/* A bogus TCEntry which hopefully does not match code from any valid
   address.  This is what all VG_(tt_fast) entries are made to point
   at when we want to invalidate it. */
static const TCEntry vg_tc_bogus_TCEntry = { ((Addr)5), 0, 0 };


/*------------------ DECLS ------------------*/

/* The translation cache sectors.  These are NULL until allocated
   dynamically. */
static UChar* vg_tc[VG_TC_N_SECTORS];

/* Count of bytes used in each sector of the TC. */
static Int vg_tc_used[VG_TC_N_SECTORS];

/* The age of each sector, so we can find the oldest.  We just use the
   global count of translations made when the sector was brought into
   use.  Doesn't matter if this mechanism gets confused (wraps around
   4G) once in a while. */
static Int vg_tc_age[VG_TC_N_SECTORS];

/* The number of the sector currently being allocated in. */
static Int vg_tc_current;

/* Count of number of translations, orig and new bytes in each sector.
   For stats purposes only. */
static Int vg_tc_stats_count[VG_TC_N_SECTORS];
static Int vg_tc_stats_osize[VG_TC_N_SECTORS];
static Int vg_tc_stats_tsize[VG_TC_N_SECTORS];

static UInt n_tt_fast_misses = 0;   // number of lookups missing fast TT helper
static UInt n_tc_discards = 0;      // number of TT/TC discards

// Number and total original/translated size of translations overall.
static UInt overall_in_count  = 0;
static UInt overall_in_osize  = 0;
static UInt overall_in_tsize  = 0;
// Number and total original/t size of discards overall.
static UInt overall_out_count = 0;
static UInt overall_out_osize = 0;
static UInt overall_out_tsize = 0;



/*------------------ TRANSLATION TABLE ------------------*/

/* The translation table.  An array of VG_TT_SIZE TTEntrys. */
static TTEntry* vg_tt = NULL;

/* Count of non-empty TT entries.  This includes deleted ones. */
static Int vg_tt_used = 0;

/* Fast helper for the TT.  A direct-mapped cache which holds a
   pointer to a TT entry which may or may not be the correct one, but
   which we hope usually is.  This array is referred to directly from
   vg_dispatch.S. */
Addr /* TCEntry*, really */ VG_(tt_fast)[VG_TT_FAST_SIZE];

static void for_each_tc(Int sector, void (*fn)(TCEntry *));


/*------------------ T-CHAINING HELPERS ------------------*/

static
void for_each_jumpsite(TCEntry *tce, void (*fn)(Addr))
{
   Int i;
   for(i = 0; i < VG_MAX_JUMPS; i++) {
      Addr a;
      UShort idx = tce->jump_sites[i];

      if (idx == (UShort)-1)
	 continue;
      
      a = (Addr)&tce->payload[idx];

      (*fn)(a);
   }
}

#ifdef __i386__
static inline
void unchain_tce(TCEntry *tce)
{
   for_each_jumpsite(tce, VG_(unchain_jumpsite));
}
#endif

/* Unchain any jumps pointing to a sector we're about to free */

static
void unlink_jumpsite(Addr a)
{
   Addr target = VG_(get_jmp_dest)(a);
   Addr *pp, p;

   if (target == 0) {
      if (*(UInt *)a != 0x4e800021)
	 VG_(printf)("unlink_jumpsite didn't find a jump at %x\n", a);
      return;
   }
   a += VG_PATCHME_JMPSZ;
   for (pp = (Addr *)(target - 4); (p = *pp) != 0; pp = (Addr *)p) {
      vg_assert(p != (Addr)pp);
      if (p == a) {
	 *pp = *(Addr *)a;
	 return;
      }
   }
   VG_(core_panic)("didn't find jumpsite to unlink");
}

static
void unchain_refs_to_tce(TCEntry *tce)
{
   Addr *p, a;

   for (p = &tce->chain_back; (a = *p) != 0; p = (Addr *)a) {
      if ((*(UInt *)(a - VG_PATCHME_JMPSZ) & 0xfc000003) != 0x48000000)
	 VG_(printf)("unchain_refs_to_tce: bad p=%p a=%x tce=%p\n", p, a, tce);
      *p = 0;
      VG_(unchain_jumpsite)(a - VG_PATCHME_JMPSZ);
   }
}

static
void unchain_refs_and_links_to_tce(TCEntry *tce)
{
   unchain_refs_to_tce(tce);
   for_each_jumpsite(tce, unlink_jumpsite);
}

static
void unchain_refs_to_sector(Int s)
{
   for_each_tc(s, unchain_refs_and_links_to_tce);
}


/*------------------ TT HELPERS ------------------*/

static
void pp_tt_tc_status ( Char* submsg )
{
   Int tc_used, s;
   if (VG_(clo_verbosity) <= 2)
      return;
   tc_used = 0;
   for (s = 0; s < VG_TC_N_SECTORS; s++)
      tc_used += vg_tc_used[s];

   VG_(message)(Vg_DebugMsg, 
       "%lluk bbs: tt %d, tc %d, %s",
       VG_(bbs_done) / 1000,
       vg_tt_used, tc_used, submsg );
}

/* Invalidate the tt_fast cache, for whatever reason, by pointing all
   entries at vg_tc_bogus_TCEntry.  */
static
void vg_invalidate_tt_fast( void )
{
   Int j;
   for (j = 0; j < VG_TT_FAST_SIZE; j++)
      VG_(tt_fast)[j] = (Addr)&vg_tc_bogus_TCEntry;
}


static
void add_tt_entry ( TCEntry* tce )
{
   UInt i;
   /* VG_(printf)("add_TT_entry orig_addr %p\n", tce->orig_addr); */
   /* Hash to get initial probe point. */
   i = VG_TT_HASH(tce->orig_addr);
   while (True) {
      if (vg_tt[i].orig_addr == tce->orig_addr)
         VG_(core_panic)("add_TT_entry: duplicate");
      if (vg_tt[i].orig_addr == VG_TTE_EMPTY)
         break;
      i++;
      if (i == VG_TT_SIZE) 
         i = 0;
   }

   vg_tt[i].orig_addr = tce->orig_addr;
   vg_tt[i].tcentry = tce;
   vg_tt_used++;
   /* sanity ... */
   vg_assert(vg_tt_used < VG_TT_SIZE-1000);
}


/* Search TT to find the translated address of the supplied original,
   or NULL if not found.  This routine is used when we miss in
   VG_(tt_fast). 
*/
static __inline__
TTEntry* search_tt ( Addr orig_addr )
{
   Int i;
   /* Hash to get initial probe point. */
   i = VG_TT_HASH(orig_addr);
   while (True) {
      if (vg_tt[i].orig_addr == orig_addr)
         return &vg_tt[i];
      if (vg_tt[i].orig_addr == VG_TTE_EMPTY)
         return NULL;
      i++;
      if (i == VG_TT_SIZE) i = 0;
   }
}


static
void initialise_tt ( void )
{
   Int i;
   vg_tt_used = 0;
   for (i = 0; i < VG_TT_SIZE; i++) {
      vg_tt[i].orig_addr = VG_TTE_EMPTY;
   }
   vg_invalidate_tt_fast();
}


static 
void rebuild_TT ( void )
{
   Int      s;

   /* Throw away TT. */
   initialise_tt();
   
   /* Rebuild TT from the remaining quarters. */
   for (s = 0; s < VG_TC_N_SECTORS; s++) {
      for_each_tc(s, add_tt_entry);
   }
   pp_tt_tc_status ( "after  rebuild of TC" );
}


/*------------------ TC HELPERS ------------------*/

static
void for_each_tc(Int s, void (*fn)(TCEntry *))
{
   UChar *pc;
   UChar *pc_lim;
   TCEntry *tce;

   pc     = &(vg_tc[s][0]);
   pc_lim = &(vg_tc[s][vg_tc_used[s]]);
   while (True) {
      if (pc >= pc_lim) break;
      tce = (TCEntry*)pc;
      pc += sizeof(TCEntry) + tce->trans_size;
      if (tce->orig_addr != VG_TTE_DELETED)
	 (*fn)(tce);
   }
}

/* Find the oldest non-NULL, non-empty sector, or -1 if none such. */
static 
Int find_oldest_sector ( void ) 
{
   Int oldest_age, oldest, i;
   oldest_age = 1000 * 1000 * 1000;
   oldest = -1;
   for (i = 0; i < VG_TC_N_SECTORS; i++) {
      if (vg_tc[i] == NULL) 
         continue;
      if (vg_tc_used[i] == 0)
         continue;
      if (vg_tc_age[i] < oldest_age) {
         oldest = i;
         oldest_age = vg_tc_age[i];
      }
   }
   return oldest;
}


/* Discard the oldest sector, if any such exists. */
static
void discard_oldest_sector ( void )
{
   Char msg[100];
   Int s = find_oldest_sector();
   if (s != -1) {
      //Int i;

      vg_assert(s >= 0 && s < VG_TC_N_SECTORS);
      VG_(sprintf)(msg, "before discard of sector %d (%d bytes)", 
                        s, vg_tc_used[s]);

#if 0
      VALGRIND_DISCARD_TRANSLATIONS(vg_tc[s], vg_tc_used[s]);
      for(i = 0; i < VG_TC_N_SECTORS; i++) {
	 if (i != s && vg_tc[i] != NULL)
	    unchain_sector(i, (Addr)vg_tc[s], vg_tc_used[s]);
      }
#else
      unchain_refs_to_sector(s);
#endif
      pp_tt_tc_status ( msg );
      overall_out_count += vg_tc_stats_count[s];
      overall_out_osize += vg_tc_stats_osize[s];
      overall_out_tsize += vg_tc_stats_tsize[s]; 
      vg_tc_used[s] = 0;
      vg_tc_stats_count[s] = 0;
      vg_tc_stats_osize[s] = 0;
      vg_tc_stats_tsize[s] = 0;
      n_tc_discards++;
   }
}


/* Find an empty sector and bring it into use.  If there isn't one,
   try and allocate one.  If that fails, return -1. */
static
Int maybe_commission_sector ( void )
{
   Char msg[100];
   Int  s;
   for (s = 0; s < VG_TC_N_SECTORS; s++) {
      if (vg_tc[s] != NULL && vg_tc_used[s] == 0) {
         vg_tc_age[s] = overall_in_count;
         VG_(sprintf)(msg, "after  commission of sector %d "
                           "at time %d", 
                           s, vg_tc_age[s]);
         pp_tt_tc_status ( msg );
#        ifdef DEBUG_TRANSTAB
         VG_(sanity_check_tt_tc)();
#        endif
         return s;
      }
   }
   for (s = 0; s < VG_TC_N_SECTORS; s++) {
      if (vg_tc[s] == NULL) {
	 if (1)
	    vg_tc[s] = VG_(get_memory_from_mmap) 
	       ( vg_tc_sector_szB, "trans-cache(sector)" );
	 else {
	    // Alternative: put translations in an mmap'd file.  The main
	    // reason is to help OProfile -- OProfile can assign time spent in
	    // translations to a particular file.  The file format doesn't
	    // really matter, which is good because it's not really readable,
	    // being generated code but not a proper ELF file.
	    Char buf[20];
	    static Int count = 0;
	    Int fd;
	 
	    VG_(sprintf)(buf, ".transtab.%d", count++);

	    fd = VG_(open)(buf, VKI_O_RDWR|VKI_O_CREAT|VKI_O_TRUNC, 0700);
	    //VG_(unlink)(buf);
	    VG_(do_syscall)(__NR_ftruncate, fd, PGROUNDUP(vg_tc_sector_szB));
	    vg_tc[s] = VG_(mmap)(0, PGROUNDUP(vg_tc_sector_szB), VKI_PROT_READ|VKI_PROT_WRITE|VKI_PROT_EXEC, VKI_MAP_SHARED, 0, fd, 0);
	    VG_(close)(fd);
	 }

	 VALGRIND_MAKE_WRITABLE(vg_tc[s], vg_tc_sector_szB);
         vg_tc_used[s] = 0;
         VG_(sprintf)(msg, "after  allocation of sector %d (size %d)", 
                           s, vg_tc_sector_szB );
         pp_tt_tc_status ( msg );
         return maybe_commission_sector();
      }
   }
   return -1;
}


static
UChar* allocate ( Int nBytes )
{
   Int i;

   vg_assert(IS_ALIGNED(nBytes));

   /* Ensure the TT is still OK. */
   while (vg_tt_used >= VG_TT_LIMIT) {
      (void)discard_oldest_sector();
      rebuild_TT();
      vg_assert(vg_tt_used < VG_TT_LIMIT);
   }

   /* Can we get it into the current sector? */
   if (vg_tc_current >= 0 
       && vg_tc_current < VG_TC_N_SECTORS
       && vg_tc[vg_tc_current] != NULL
       && vg_tc_used[vg_tc_current] + nBytes <= vg_tc_sector_szB) {
      /* Yes. */
      UChar* p = &(vg_tc[vg_tc_current][ vg_tc_used[vg_tc_current] ]);
      vg_tc_used[vg_tc_current] += nBytes;
      return p;
   }

   /* Perhaps we can bring a new sector into use, for the first
      time. */
   vg_tc_current = maybe_commission_sector();
   if (vg_tc_current >= 0 && vg_tc_current < VG_TC_N_SECTORS)
      return allocate(nBytes);

   /* That didn't work.  We'll have to dump the oldest.  We take the
      opportunity to dump the N oldest at once. */
   for (i = 0; i < 1; i++)
      (void)discard_oldest_sector();

   rebuild_TT();
   vg_tc_current = maybe_commission_sector();
   vg_assert(vg_tc_current >= 0 && vg_tc_current < VG_TC_N_SECTORS);
#  ifdef DEBUG_TRANSTAB
   VG_(sanity_check_tt_tc)();
#  endif

   return allocate(nBytes);
}


/* Just so these counts can be queried without making them globally
   visible. */
void VG_(get_tt_tc_used) ( UInt* tt_used, UInt* tc_used )
{
   Int s;
   *tt_used = vg_tt_used;
   *tc_used = 0;
   for (s = 0; s < VG_TC_N_SECTORS; s++)
      *tc_used += vg_tc_used[s];
}


/* Do a sanity check on TT/TC.
*/
void VG_(sanity_check_tt_tc) ( void )
{
   Int i, s;
   TTEntry* tte;
   TCEntry* tce;
   /* Checks: 
      - Each TT entry points to a valid and corresponding TC entry.
   */
   for (i = 0; i < VG_TT_SIZE; i++) {
      tte = &vg_tt[i];
      /* empty slots are harmless. */
      if (tte->orig_addr == VG_TTE_EMPTY) continue;
      /* all others should agree with the TC entry. */
      tce = tte->tcentry;
      vg_assert(IS_ALIGNED4_ADDR(tce));
      /* does this point into a valid TC sector? */
      for (s = 0; s < VG_TC_N_SECTORS; s++)
	if (vg_tc[s] != NULL
            && ((Addr)tce) >= (Addr)&vg_tc[s][0]
            && ((Addr)tce) <  (Addr)&vg_tc[s][ vg_tc_used[s] ])
	  break; 
      vg_assert(s < VG_TC_N_SECTORS);
      /* It should agree with the TC entry on the orig_addr.  This may
         be VG_TTE_DELETED, or a real orig addr. */
      vg_assert(tte->orig_addr == tce->orig_addr);
#ifdef __powerpc__
      {
	 Addr *p, a, loc, dst;
	 Int j;
	 dst = (Addr) &tce->payload;
	 for (p = &tce->chain_back; (a = *p) != 0; p = (Addr *)a) {
	    loc = a - VG_PATCHME_JMPSZ;
	    if (*(UInt *)loc != 0x48000000 + ((dst - loc) & 0x03fffffc))
	       VG_(printf)("back chain wrong for tce=%p a=%x\n", tce, a);
	    vg_assert(*(UInt *)loc == 0x48000000 + ((dst - loc) & 0x03fffffc));
	 }
	 for (j = 0; j < VG_MAX_JUMPS; ++j) {
	    UShort idx;
	    idx = tce->jump_sites[j];
	    if (idx != (UShort) -1) {
	       p = (Addr *) (tce->payload + idx);
	       if ((*p & 0xfc000003) != 0x48000000 && *p != 0x4e800021)
		  VG_(printf)("bad jump_site index [%d] %x (%x %x %x)\n",
			      j, idx, p[-1], p[0], p[1]);
	       vg_assert((*p & 0xfc000003) == 0x48000000 || *p == 0x4e800021);
	    }
	 }
      }
#endif
   }
}

#ifdef __powerpc__
/*
 * On PowerPC, the icache doesn't automatically maintain coherence
 * with memory or the dcache, so we have to do an explicit flush.
 */
static void flush_cache(void *ptr, int nbytes)
{
   unsigned long startaddr = (unsigned long) ptr;
   unsigned long endaddr = startaddr + nbytes;
   unsigned long addr;
   unsigned long cls = VG_(cache_line_size);

   startaddr &= ~(cls - 1);
   for (addr = startaddr; addr < endaddr; addr += cls)
      asm volatile("dcbst 0,%0" : : "r" (addr));
   asm volatile("sync");
   for (addr = startaddr; addr < endaddr; addr += cls)
      asm volatile("icbi 0,%0" : : "r" (addr));
   asm volatile("sync; isync");
}
#endif
#ifdef __i386__
#define flush_cache(ptr, nb)	do { } while (0)
#endif

static __inline__ Int safe_idiv(Int a, Int b)
{
   return (b == 0 ? 0 : a / b);
}

void VG_(print_tt_tc_stats)(void)
{
   VG_(message)(Vg_DebugMsg,
                "    TT/TC: %d tc sectors discarded.",
                n_tc_discards );
   VG_(message)(Vg_DebugMsg,
                "           %d tt_fast misses.",
                n_tt_fast_misses);
   VG_(message)(Vg_DebugMsg,
                "translate: new     %d (%d -> %d; ratio %d:10)",
                overall_in_count, overall_in_osize, overall_in_tsize,
                safe_idiv(10*overall_in_tsize, overall_in_osize));
   VG_(message)(Vg_DebugMsg,
                "           discard %d (%d -> %d; ratio %d:10).",
                overall_out_count, overall_out_osize, overall_out_tsize,
                safe_idiv(10*overall_out_tsize, overall_out_osize));
}

Int VG_(get_bbs_translated) ( void )
{
   return overall_in_count;
}

/* Add this already-filled-in entry to the TT.  Assumes that the
   relevant code chunk has been placed in TC, along with a dummy back
   pointer, which is inserted here.
*/
void VG_(add_to_trans_tab) ( Addr orig_addr,  Int orig_size,
                             Addr trans_addr, Int trans_size,
			     UCodeBlock *cb )
{
   Int i, nBytes, trans_size_aligned;
   TCEntry* tce;
   unsigned int *src, *dst, trans_words;
   UShort prev;

   /*
   VG_(printf)("add_to_trans_tab(%d) %x %d %x %d\n",
               vg_tt_used, tte->orig_addr, tte->orig_size, 
               tte->trans_addr, tte->trans_size);
   */

   vg_assert(offsetof(TCEntry, payload) == VG_CODE_OFFSET);

   /* figure out how many bytes we require. */
   nBytes = CODE_ALIGN(trans_size + sizeof(TCEntry));
   trans_size_aligned = nBytes-sizeof(TCEntry);
   vg_assert(IS_ALIGNED(nBytes));

   tce = (TCEntry*)allocate(nBytes);
   /*
   VG_(printf)("allocate returned %p (code start %p)\n", 
               tce, &tce->payload[0]);
   */
   vg_assert(vg_tc_current >= 0 && vg_tc_current < VG_TC_N_SECTORS);

   tce->orig_addr  = orig_addr;
   tce->orig_size  = (UShort)orig_size;  /* what's the point of storing this? */
   tce->trans_size = (UShort)trans_size_aligned;
   for (i = 0; i < VG_MAX_JUMPS; i++) {
      tce->jump_sites[i] = cb->jumps[i];
   }
   tce->chain_back = 0;
   src = (unsigned int *) trans_addr;
   dst = (unsigned int *) tce->payload;
   trans_words = trans_size_aligned / sizeof(unsigned int);
   for (i = 0; i < trans_words; ++i)
      dst[i] = src[i];
   flush_cache(dst, trans_size_aligned);

   if (VG_(clo_trace_codegen) & 1)
      VG_(message)(Vg_DebugMsg, "add_to_trans_tab: inserting code for %p into transtab at %p\n",
		   orig_addr, tce->payload);

   /* relocate absolute jumps */
   prev = 0;
   for(i = 0; i < cb->nrelocs; i++) {
      UShort off = cb->relocs[i];

      vg_assert(off < trans_size);
      vg_assert(i == 0 || off > prev);

      VG_(reloc_abs_jump)(&tce->payload[off]);

      prev = off;
   }
   add_tt_entry(tce);

   /* Update stats. */
   overall_in_count ++;
   overall_in_osize += orig_size;
   overall_in_tsize += trans_size;

   vg_tc_stats_count[vg_tc_current] ++;
   vg_tc_stats_osize[vg_tc_current] += orig_size;
   vg_tc_stats_tsize[vg_tc_current] += trans_size;
}


/* Find the translation address for a given (original) code address.
   If found, update VG_(tt_fast) so subsequent lookups are fast.  If
   no translation can be found, return zero.  This routine is (the
   only one) called from vg_run_innerloop.  */
Addr VG_(search_transtab) ( Addr original_addr )
{
   TTEntry* tte;

   if (vg_tt == NULL)
      return 0;

   VGP_PUSHCC(VgpSlowFindT);
   tte = search_tt ( original_addr );
   if (tte == NULL) {
      /* We didn't find it.  vg_run_innerloop will have to request a
         translation. */
      VGP_POPCC(VgpSlowFindT);
      return (Addr)0;
   } else {
      /* Found it.  Put the search result into the fast cache now. */
      UInt cno = VG_TT_FAST_INDEX(original_addr);
      VG_(tt_fast)[cno] = (Addr)(tte->tcentry);
      n_tt_fast_misses++;
      VGP_POPCC(VgpSlowFindT);
      return (Addr)&(tte->tcentry->payload[0]);
   }
}


/* Invalidate translations of original code [start .. start + range - 1].
   This is slow, so you *really* don't want to call it very often.
   Set 'unchain_blocks' if the translation being invalidated may be chained
   to by other local blocks (which are NOT being discarded).
*/
void VG_(invalidate_translations) ( Addr start, UInt range, Bool unchain_blocks )
{
   Addr     i_start, i_end, o_start, o_end;
   UInt     out_count, out_osize, out_tsize;
   Int      i;
   TCEntry* tce;

   if (vg_tt == NULL)
      return;

#  ifdef DEBUG_TRANSTAB
   VG_(sanity_check_tt_tc)();
#  endif
   i_start = start;
   i_end   = start + range - 1;
   out_count = out_osize = out_tsize = 0;

   for (i = 0; i < VG_TT_SIZE; i++) {
      if (vg_tt[i].orig_addr == VG_TTE_EMPTY
          || vg_tt[i].orig_addr == VG_TTE_DELETED) continue;
      tce = vg_tt[i].tcentry;
      o_start = tce->orig_addr;
      o_end   = o_start + tce->trans_size - 1;
      if (o_end < i_start || o_start > i_end)
         continue;

      if (VG_(needs).basic_block_discards)
         SK_(discard_basic_block_info)( tce->orig_addr, 
                                        tce->orig_size );

      vg_tt[i].orig_addr = VG_TTE_DELETED;
      tce->orig_addr = VG_TTE_DELETED;

#if 0
      if (VG_(clo_chain_bb) && unchain_blocks) {
         /* make sure no other blocks chain to the one we just discarded */
	 Int j;
         for(j = 0; j < VG_TC_N_SECTORS; j++) {
            if (vg_tc[j] != NULL)
               unchain_sector(j, (Addr)tce->payload, tce->trans_size);
         }
      }
#else
      if (VG_(clo_chain_bb))
	 unchain_refs_and_links_to_tce(tce);
#endif

      overall_out_count ++;
      overall_out_osize += tce->orig_size;
      overall_out_tsize += tce->trans_size;
      out_count ++;
      out_osize += tce->orig_size;
      out_tsize += tce->trans_size;
   }

   if (out_count > 0) {
      vg_invalidate_tt_fast();
      VG_(sanity_check_tt_tc)();
#     ifdef DEBUG_TRANSTAB
      { Addr aa;
        for (aa = i_start; aa <= i_end; aa++)
           vg_assert(search_tt ( aa ) == NULL);
      }
#     endif
   }

   if (VG_(clo_verbosity) > 2)
      VG_(message)(Vg_UserMsg,   
         "discard %d (%d -> %d) translations in range %p .. %p",
         out_count, out_osize, out_tsize, i_start, i_end );
}

#ifdef __powerpc__
/* Helper called from generated code to invalidate one cache line
   from the translation cache when the synthetic cpu executes the
   icbi (instruction cache block invalidate) instruction.
   We assume that the program will do an icbi on each cache line
   covering any BBs that we need to invalidate, so we only need to
   invalidate BBs that start in this cache line.
*/
extern void VG_(helper_cache_inval)(Addr start)
{
   Addr     i_start, i_end, o_start, o_end;
   UInt     out_count, out_osize, out_tsize;
   Int      i, i_last;
   Bool     can_stop = False;
   TCEntry* tce;
   UInt cls = VG_(cache_line_size);

   out_count = out_osize = out_tsize = 0;

   start &= ~(cls - 1);
   i = VG_TT_HASH(start);
   i_last = VG_TT_HASH(start + cls);
   i_start = start;
   i_end   = start + cls - 1;

   for (;; ++i) {
      if (i == VG_TT_SIZE)
	 i = 0;
      if (i == i_last)
	 can_stop = True;
      if (vg_tt[i].orig_addr == VG_TTE_EMPTY) {
	 if (can_stop)
	    break;
	 continue;
      }
      if (vg_tt[i].orig_addr == VG_TTE_DELETED)
	 continue;
      tce = vg_tt[i].tcentry;
      o_start = tce->orig_addr;
      o_end   = o_start + tce->trans_size - 1;
      if (o_end < i_start || o_start > i_end)
         continue;

      if (VG_(needs).basic_block_discards)
         SK_(discard_basic_block_info)( tce->orig_addr, 
                                        tce->orig_size );

      vg_tt[i].orig_addr = VG_TTE_DELETED;
      tce->orig_addr = VG_TTE_DELETED;

      VG_(tt_fast)[VG_TT_FAST_INDEX(tce->orig_addr)]
	 = (Addr)&vg_tc_bogus_TCEntry;

      if (VG_(clo_chain_bb))
	 unchain_refs_and_links_to_tce(tce);

      overall_out_count ++;
      overall_out_osize += tce->orig_size;
      overall_out_tsize += tce->trans_size;
      out_count ++;
      out_osize += tce->orig_size;
      out_tsize += tce->trans_size;
   }

#  ifdef DEBUG_TRANSTAB
   VG_(sanity_check_tt_tc)();
#  endif

   if (VG_(clo_verbosity) > 2)
      VG_(message)(Vg_UserMsg,   
         "discard %d (%d -> %d) translations in range %p .. %p",
         out_count, out_osize, out_tsize, i_start, i_end );
}
#endif


/*------------------------------------------------------------*/
/*--- Initialisation.                                      ---*/
/*------------------------------------------------------------*/

void VG_(init_tt_tc) ( void )
{
   Int s;

   /* Otherwise we wind up with non-32-bit-aligned code in
      TCEntries. */
   vg_assert((VG_MAX_JUMPS % 2) == 0);

   // Otherwise lots of things go wrong... 
   vg_assert(VG_CODE_OFFSET == sizeof(TCEntry));
   
   /* Figure out how big each sector should be.  */
   vg_tc_sector_szB 
      = (VG_TT_LIMIT /* max TT entries we expect */
         * (VG_(details).avg_translation_sizeB 
            + sizeof(TCEntry)
            + (CODE_ALIGNMENT/2) /* avg alignment loss */)
        )
        / VG_TC_N_SECTORS;
   /* Ensure the calculated value is not way crazy. */
   vg_assert(vg_tc_sector_szB >= 200000);
   vg_assert(vg_tc_sector_szB <= 8000000);

   for (s = 0; s < VG_TC_N_SECTORS; s++) {
      vg_tc[s] = NULL;
      vg_tc_used[s] = 0;
      vg_tc_age[s] = 0;
      vg_tc_stats_count[s] = 0;
      vg_tc_stats_osize[s] = 0;
      vg_tc_stats_tsize[s] = 0;
   }
   vg_tc_current = 0;

   vg_tt = VG_(get_memory_from_mmap) ( VG_TT_SIZE * sizeof(TTEntry),
                                       "trans-table" );
   /* The main translation table is empty. */
   initialise_tt();

   if (VG_(clo_verbosity) > 2) {
      VG_(message)(Vg_DebugMsg,
         "Translation Cache: using %d sectors of %d bytes each", 
          VG_TC_N_SECTORS, vg_tc_sector_szB );
      VG_(message)(Vg_DebugMsg,
         "Translation Table: %d total entries, max occupancy %d (%d%%)",
         VG_TT_SIZE, VG_TT_LIMIT, VG_TT_LIMIT_PERCENT );
   }

#  ifdef DEBUG_TRANSTAB
   VG_(sanity_check_tt_tc)();
#  endif
}

/*--------------------------------------------------------------------*/
/*--- end                                            vg_transtab.c ---*/
/*--------------------------------------------------------------------*/
