/*
   Copyright (C) 2004 Paul Mackerras <paulus@samba.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

#include "core.h"
#include "ppc_private.h"

#define uFlagsR(cb, r)		uFlagsRWU((cb), (r), FlagsEmpty, FlagsEmpty)
#define uFlagsW(cb, w)		uFlagsRWU((cb), FlagsEmpty, (w), FlagsEmpty)
#define uFlagsRW(cb, rw)	uFlagsRWU((cb), (rw), (rw), FlagsEmpty)

#define UIMM(instr)	((unsigned short)(instr))
#define SIMM(instr)	((signed short)(instr))
#define IMMS(instr)	((instr) << 16)

#define MASKG(mb, me)	((0xffffffffUL >> (mb)) - (0x7fffffffUL >> (me)) \
			 - ((mb > (me))))

/* Used for constant propagation */
static Bool reg_is_const[32];
static UInt reg_value[32];

static const char *regnames[] = {
   "r0",  "r1",  "r2",  "r3",  "r4",  "r5",  "r6",  "r7",
   "r8",  "r9",  "r10", "r11", "r12", "r13", "r14", "r15",
   "r16", "r17", "r18", "r19", "r20", "r21", "r22", "r23",
   "r24", "r25", "r26", "r27", "r28", "r29", "r30", "r31",
   "cr",  "lr",  "ctr", "xer", "ip"
};

const Char *VG_(name_of_int_reg)(Int sz, Int reg)
{
    if (0 <= reg && reg < sizeof(regnames) / sizeof(regnames[0]))
	return regnames[reg];
    return "??";
}

const Char *VG_(name_of_seg_reg)(Int reg)
{
    return "we have sregs?";
}

Char VG_(name_of_int_size)(Int size)
{
    return "bw?l"[size-1];
}

/* Allocate a new temp reg number. */
static __inline__ int new_temp(UCodeBlock *cb)
{
    int t = cb->nextTemp;

    cb->nextTemp += 2;
    return t;
}

int VG_(get_new_temp)(UCodeBlock *cb)
{
    return new_temp(cb);
}

int VG_(get_new_shadow)(UCodeBlock *cb)
{
    int t = new_temp(cb);
    return SHADOW(t);
}

static __inline__ void set_cond(UCodeBlock *cb, Condcode c)
{
    LAST_UINSTR(cb).cond = c;
}

static void do_record(UCodeBlock *cb, int tmp)
{
    int t2;

    t2 = new_temp(cb);
    uInstr2(cb, CMP0,  4, TempReg, tmp, TempReg, t2);
    uFlagsR(cb, FlagSO);
    uInstr3(cb, ICRF,  4, TempReg, t2, Lit16, 0, ArchReg, R_CR);
}

static int calc_ea_disp(UCodeBlock *cb, int ra, int imm, unsigned int update)
{
    int t1;

    t1 = new_temp(cb);
    if (ra == 0) {
	uInstr2(cb, MOV,   4, Literal, 0, TempReg, t1);
	uLiteral(cb, imm);
    } else {
	if (reg_is_const[ra]) {
	    uInstr2(cb, MOV,   4, Literal, 0, TempReg, t1);
	    uLiteral(cb, reg_value[ra] + imm);
	} else {
	    uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	    if (imm) {
		uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
		uLiteral(cb, imm);
	    }
	}
    }
    if (update && ra) {
	if (reg_is_const[ra])
	    reg_value[ra] += imm;
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, ra);
    }
    return t1;
}

static int calc_ea_index(UCodeBlock *cb, int ra, int rb, unsigned int update)
{
    int t1, t2;

    t1 = new_temp(cb);
    uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
    if (ra != 0) {
	if (reg_is_const[ra]) {
	    uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, reg_value[ra]);
	} else {
	    t2 = new_temp(cb);
	    uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t2);
	    uInstr2(cb, ADD,   4, TempReg, t2, TempReg, t1);
	}
    }
    if (update && ra) {
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, ra);
	reg_is_const[ra] = False;
    }
    return t1;
}

static void set_lr(UCodeBlock *cb, unsigned int val)
{
    int t1 = new_temp(cb);

    uInstr2(cb, MOV,   4, Literal,  0, TempReg, t1);
    uLiteral(cb, val);
    uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, R_LR);
}

/* Disassemble a conditional branch */
static void dis_condition(UCodeBlock *cb, unsigned int instr, unsigned int nip)
{
    int t1;
    unsigned int bo = (instr >> 21) & 0x1f;

    if ((bo & 4) == 0) {
	/* decrement CTR */
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, R_CTR, TempReg, t1);
	uInstr2(cb, ADD,   4, Literal,     0, TempReg, t1);
	uLiteral(cb, -1);
	uInstr2(cb, PUT,   4, TempReg,    t1, ArchReg, R_CTR);
	if (bo & 2)
	    uInstr1(cb, SETZ,  4, TempReg,    t1);
	uInstr2(cb, JIFZ,  4, TempReg,    t1, Literal, 0);
	uLiteral(cb, nip);
    }
}

static void dis_crop(UCodeBlock *cb, int dis, unsigned int instr)
{
    unsigned int minor = (instr >> 6) & 0xf;
    int rd = (instr >> 21) & 0x1f;
    int ra = (instr >> 16) & 0x1f;
    int rb = (instr >> 11) & 0x1f;
    int tcr, ta, tb;
    static const char *names[] = {
	NULL,	"nor",	NULL,	NULL,	"andc",	NULL,	"xor",	"nand",
	"and",	"eqv",	NULL,	NULL,	NULL,	"orc",	"or",	NULL
    };

    if (dis)
	VG_(printf)("cr%s %d,%d,%d\n", names[minor], rd, ra, rb);
    tcr = new_temp(cb);
    ta = new_temp(cb);
    tb = new_temp(cb);
    uInstr2(cb, GET,  4, ArchReg, R_CR, TempReg, tcr);
    uInstr3(cb, XBIT, 1, TempReg, tcr, Literal, 0, TempReg, ta);
    uLiteral(cb, ra);
    uInstr3(cb, XBIT, 1, TempReg, tcr, Literal, 0, TempReg, tb);
    uLiteral(cb, rb);
    switch (minor) {
    case 1:		/* crnor */
	uInstr1(cb, NOT, 1, TempReg, ta);
    case 4:		/* crandc */
	uInstr1(cb, NOT, 1, TempReg, tb);
    case 8:		/* crand */
	uInstr2(cb, AND, 1, TempReg, tb, TempReg, ta);
	break;
    case 6:		/* crxor */
	uInstr2(cb, XOR, 1, TempReg, tb, TempReg, ta);
	break;
    case 9:		/* creqv */
	uInstr2(cb, XOR, 1, TempReg, tb, TempReg, ta);
	uInstr1(cb, NOT, 1, TempReg, ta);
	break;
    case 7:		/* crnand */
	uInstr1(cb, NOT, 1, TempReg, ta);
    case 13:		/* crorc */
	uInstr1(cb, NOT, 1, TempReg, tb);
    case 14:		/* cror */
	uInstr2(cb, OR,  1, TempReg, tb, TempReg, ta);
	break;
    }
    uInstr3(cb, IBIT, 4, TempReg, ta, Literal, 0, TempReg, tcr);
    uLiteral(cb, rd);
    uInstr2(cb, PUT,  4, TempReg, tcr, ArchReg, R_CR);
}

static int dis_opc_31(UCodeBlock *cb, int dis, unsigned int instr)
{
    unsigned int minor = (instr >> 1) & 0x3ff;
    int rd = (instr >> 21) & 0x1f;
    int ra = (instr >> 16) & 0x1f;
    int rb = (instr >> 11) & 0x1f;
    int record = instr & 1;
    int t1, t2, t3;
    int tcc = INVALID_TEMPREG;
    int i, reg, sz, op;
    const char *rname;
    static char sznames[4] = "bh?w";

#define B(x, str)		((minor & (x))? str: "")
#define B2(x, str, y, s2, s3)	((minor & (x))? str: (minor & (y))? s2: s3)

#define OBIT	B(0x200, "o")
#define RC	(record? ".": "")

    switch (minor) {
    case 0:		/* cmp */
    case 0x20:		/* cmpl */
	rd >>= 2;	/* should check L bit */
	if (dis)
	    VG_(printf)("cmp%s cr%d,r%d,r%d\n", B(0x20, "l"), rd, ra, rb);
	t1 = new_temp(cb);
	t2 = new_temp(cb);
	t3 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t2);
	op = (minor & 0x20)? CMPU: CMP;
	uInstr3(cb, op,    4, TempReg, t1, TempReg, t2, TempReg, t3);
	uFlagsR(cb, FlagSO);
	uInstr3(cb, ICRF,  4, TempReg, t3, Lit16, rd, ArchReg, R_CR);
	break;

    case 0x200:		/* mcrxr */
	t2 = new_temp(cb);
	t3 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, R_XER, TempReg, t2);
	uInstr3(cb, ICRF,  4, TempReg, t2, Lit16, rd >> 2, ArchReg, R_CR);
	uInstr2(cb, AND,   4, Literal,  0, TempReg, t2);
	uLiteral(cb, 0x0fffffffU);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, R_XER);
	break;

    case 4:		/* tw */
	goto bad;

    case 0x6:		/* lvsl */
    case 0x26:		/* lvsr */
	if (dis)
	    VG_(printf)("lvs%c vr%d,r%d,r%d\n", (minor & 0x20? 'r': 'l'),
			rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	uInstr2(cb, AND, 4, Literal, 0, TempReg, t1);
	uLiteral(cb, 0xf);
	op = minor & 0x20;
	uInstr2(cb, VEC_FROMREG, 16, TempReg, t1, Lit16, (op << 3) + rd);
	break;

    case 0x7:		/* lvebx */
    case 0x27:		/* lvehx */
    case 0x47:		/* lvewx */
	sz = 1 << ((minor >> 5) & 3);
	if (dis)
	    VG_(printf)("lve%cx vr%d,r%d,r%d\n", sznames[sz], rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	if (sz > 1) {
	    uInstr2(cb, AND,  4, Literal, 0, TempReg, t1);
	    uLiteral(cb, -sz);
	}
	uInstr2(cb, VEC_R, sz, Lit16, rd, TempReg, t1);
	break;

    case 0x67:		/* lvx */
    case 0x167:		/* lvxl */
	if (dis)
	    VG_(printf)("lvx%s vr%d,r%d,r%d\n", B(0x100, "l"), rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	uInstr2(cb, AND,  4, Literal, 0, TempReg, t1);
	uLiteral(cb, -16);
	uInstr2(cb, VEC_R, 16, Lit16, rd, TempReg, t1);
	LAST_UINSTR(cb).extra4b = minor >> 8;
	break;

    case 0x87:		/* stvebx */
    case 0xa7:		/* stvehx */
    case 0xc7:		/* stvewx */
	sz = 1 << ((minor >> 5) & 3);
	if (dis)
	    VG_(printf)("stve%cx vr%d,r%d,r%d\n", sznames[sz], rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	if (sz > 1) {
	    uInstr2(cb, AND,  4, Literal, 0, TempReg, t1);
	    uLiteral(cb, -sz);
	}
	uInstr2(cb, VEC_W, sz, Lit16, rd, TempReg, t1);
	break;

    case 0xe7:		/* stvx */
    case 0x1e7:		/* stvxl */
	if (dis)
	    VG_(printf)("stvx%s vr%d,r%d,r%d\n", B(0x100, "l"), rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	uInstr2(cb, AND,  4, Literal, 0, TempReg, t1);
	uLiteral(cb, -16);
	uInstr2(cb, VEC_W, 16, Lit16, rd, TempReg, t1);
	LAST_UINSTR(cb).extra4b = minor >> 8;
	break;

    case 8:		/* subfc[.] */
    case 0x28:		/* subf[.] */
    case 0x88:		/* subfe[.] */
    case 0x208:		/* subfco[.] */
    case 0x228:		/* subfo[.] */
    case 0x288:		/* subfeo[.] */
	if (dis)
	    VG_(printf)("subf%s%s%s r%d,r%d,r%d\n",
			B2(0x80, "e", 0x20, "", "c"), OBIT, RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	op = (minor & 0x20)? SUB: SBB;
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, op,    4, TempReg, t1, TempReg, tcc);
	if (minor & 0x80)
	    uFlagsRW(cb, FlagCA);
	else if ((minor & 0x20) == 0)
	    uFlagsW(cb, FlagCA);
	if (minor & 0x200)	/* record overflow */
	    LAST_UINSTR(cb).flags_w |= FlagSO | FlagOV;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x68:		/* neg[.] */
    case 0x268:		/* nego[.] */
	if (dis)
	    VG_(printf)("neg%s%s r%d,r%d\n", OBIT, RC, rd, ra);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, tcc);
	uInstr1(cb, NEG,   4, TempReg, tcc);
	if (minor & 0x200)	/* record overflow */
	    LAST_UINSTR(cb).flags_w |= FlagSO | FlagOV;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0xc8:		/* subfze[.] */
    case 0xe8:		/* subfme[.] */
    case 0x2c8:		/* subfzeo[.] */
    case 0x2e8:		/* subfmeo[.] */
    case 0xca:		/* addze[.] */
    case 0xea:		/* addme[.] */
    case 0x2ca:		/* addzeo[.] */
    case 0x2ea:		/* addmeo[.] */
	if (dis)
	    VG_(printf)("%s%se%s%s r%d,r%d\n", (minor & 2)? "add": "subf",
			(minor & 0x20)? "m": "z", OBIT, RC, rd, ra);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, tcc);
	if ((minor & 2) == 0)
	    uInstr1(cb, NOT,   4, TempReg, tcc);
	uInstr2(cb, ADC,   4, Literal,  0, TempReg, tcc);
	uLiteral(cb, (minor & 0x20)? -1: 0);
	uFlagsRW(cb, FlagCA);
	if (minor & 0x200)	/* record overflow */
	    LAST_UINSTR(cb).flags_w |= FlagSO | FlagOV;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0xa:		/* addc[.] */
    case 0x8a:		/* adde[.] */
    case 0x10a:		/* add[.] */
    case 0x20a:		/* addco[.] */
    case 0x28a:		/* addeo[.] */
    case 0x30a:		/* addo[.] */
	if (dis)
	    VG_(printf)("add%s%s%s r%d,r%d,r%d\n",
			B2(0x80, "e", 0x100, "", "c"), OBIT, RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	op = (minor & 0x100)? ADD: ADC;
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, op,    4, TempReg, t1, TempReg, tcc);
	if (minor & 0x80)
	    uFlagsRW(cb, FlagCA);
	else if ((minor & 0x100) == 0)
	    uFlagsW(cb, FlagCA);
	if (minor & 0x200)	/* record overflow */
	    LAST_UINSTR(cb).flags_w |= FlagSO | FlagOV;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0xb:		/* mulhwu[.] */
	if (dis)
	    VG_(printf)("mulhwu%s r%d,r%d,r%d\n", RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, UMULH, 4, TempReg, t1, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x4b:		/* mulhw[.] */
	if (dis)
	    VG_(printf)("mulhw%s r%d,r%d,r%d\n", RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, MULH,  4, TempReg, t1, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0xeb:		/* mullw[.] */
	if (dis)
	    VG_(printf)("mullw%s r%d,r%d,r%d\n", RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, MUL,   4, TempReg, t1, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x1cb:		/* divwu[.] */
    case 0x1eb:		/* divw[.] */
    case 0x3cb:		/* divwuo[.] */
    case 0x3eb:		/* divwo[.] */
	if (dis)
	    VG_(printf)("divw%s%s%s r%d, r%d, r%d\n",
			(minor & 0x20)? "": "u",
			OBIT, RC, rd, ra, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	op = (minor & 0x20)? DIV: UDIV;
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, tcc);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
	uInstr2(cb, op,    4, TempReg, t1, TempReg, tcc);
	if (minor & 0x200)	/* record overflow */
	    LAST_UINSTR(cb).flags_w |= FlagSO | FlagOV;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x90:		/* mtcrf */
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	ra = (instr >> 12) & 0xff;
	if (ra == 0xff) {
	    if (dis)
		VG_(printf)("mtcr r%d\n", rd);
	    uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, R_CR);
	} else {
	    if (dis)
		VG_(printf)("mtcrf 0x%x,r%d\n", ra, rd);
	    for (i = 0; i < 8; ++i, ra <<= 1) {
		if ((ra & 0x80) == 0)
		    continue;
		uInstr3(cb, ICRF, 4, TempReg, t1, Lit16, i, ArchReg, R_CR);
		LAST_UINSTR(cb).extra4b = i;
	    }
	}
	break;

    case 0x13:		/* mfcr */
	if (dis)
	    VG_(printf)("mfcr r%d\n", rd);
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, R_CR, TempReg, t1);
	uInstr2(cb, PUT,   4, TempReg, t1,   ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x173:		/* mftb/mftbu */
	reg = (rb << 5) + ra;
	if (reg != 268 && reg != 269)
	    goto bad;
	t1 = new_temp(cb);
	uInstr2(cb, RDTB, 4, Lit16, reg,  TempReg, t1);
	uInstr2(cb, PUT,  4, TempReg, t1, ArchReg, rd);
	break;

    case 0x153:		/* mfspr */
    case 0x1d3:		/* mtspr */
	switch ((rb << 5) + ra) {
	case 1:
	    reg = R_XER;
	    rname = "xer";
	    break;
	case 8:
	    reg = R_LR;
	    rname = "lr";
	    break;
	case 9:
	    reg = R_CTR;
	    rname = "ctr";
	    break;
	case 256:
	    /* this one is a bit different, being part of the vector state */
	    t1 = new_temp(cb);
	    if (minor & 0x80) {
		/* mtspr */
		uInstr2(cb, GET, 4, ArchReg, rd, TempReg, t1);
		uInstr2(cb, VEC_FROMREG, 4, TempReg, t1, Lit16, 0x200);
	    } else {
		/* mfspr */
		uInstr2(cb, VEC_TOREG, 4, Lit16, 0x200, TempReg, t1);
		uInstr2(cb, PUT, 4, TempReg, t1, ArchReg, rd);
	    }
	    return 0;
	default:
	    goto bad;
	}
	if (dis)
	    VG_(printf)("m%c%s r%d\n", (minor & 0x80)? 't': 'f', rname, rd);
	t1 = new_temp(cb);
	if (minor & 0x80) {
	    uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	    uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, reg);
	} else {
	    uInstr2(cb, GET,   4, ArchReg, reg, TempReg, t1);
	    uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, rd);
	    reg_is_const[rd] = False;
	}
	break;

    case 0x14:		/* lwarx */
	if (dis)
	    VG_(printf)("lwarx r%d,r%d,r%d\n", rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	t2 = new_temp(cb);
	uInstr0(cb, LOCK,  0);
	uInstr2(cb, LOAD,  4, TempReg, t1, TempReg, t2);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x215:		/* lswx, nasty */
    case 0x295:		/* stswx, nasty */
	goto bad;

    case 0x255:		/* lswi */
	if (dis)
	    VG_(printf)("lswi r%d,r%d,%d\n", rd, ra, rb);
	t1 = calc_ea_disp(cb, ra, 0, 0);
	if (rb == 0)
	    rb = 32;			/* rb is really NB */
	while (rb >= 4) {
	    t2 = new_temp(cb);
	    uInstr2(cb, LOAD, 4, TempReg, t1, TempReg, t2);
	    uInstr2(cb, PUT,  4, TempReg, t2, ArchReg, rd);
	    uInstr2(cb, ADD,  4, Literal,  0, TempReg, t1);
	    uLiteral(cb, 4);
	    rd = (rd + 1) & 0x1f;
	    rb -= 4;
	}
	if (rb > 0) {
	    t2 = new_temp(cb);
	    uInstr2(cb, LOAD, rb, TempReg, t1, TempReg, t2);
	    uInstr2(cb, SHL,   4, Lit16, (4 - rb) * 8, TempReg, t2);
	    uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	}
	break;

    case 0x2d5:		/* stswi */
	if (dis)
	    VG_(printf)("stswi r%d,r%d,%d\n", rd, ra, rb);
	t1 = calc_ea_disp(cb, ra, 0, 0);
	if (rb == 0)
	    rb = 32;			/* rb is really NB */
	while (rb >= 4) {
	    t2 = new_temp(cb);
	    uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	    uInstr2(cb, STORE, 4, TempReg, t2, TempReg, t1);
	    uInstr2(cb, ADD,  4, Literal,  0, TempReg, t1);
	    uLiteral(cb, 4);
	    rd = (rd + 1) & 0x1f;
	    rb -= 4;
	}
	if (rb > 0) {
	    t2 = new_temp(cb);
	    uInstr2(cb, GET,    4, ArchReg, rd, TempReg, t2);
	    uInstr2(cb, SHR,    4, Lit16, (4 - rb) * 8, TempReg, t2);
	    uInstr2(cb, STORE, rb, TempReg, t2, TempReg, t1);
	}
	break;

    case 0x36:		/* dcbst */
    case 0x56:		/* dcbf */
	if (dis)
	    VG_(printf)("dcb%s r%d,r%d\n", (minor & 0x20)? "st": "f", ra, rb);
	break;

    case 0xf6:		/* dcbtst */
    case 0x116:		/* dcbt */
	if (dis)
	    VG_(printf)("dcbt%s r%d,r%d\n", B(0x20, "st"), ra, rb);
	break;

    case 0x256:		/* sync */
	if (dis)
	    VG_(printf)("sync\n");
	break;

    case 0x356:		/* eieio */
	if (dis)
	    VG_(printf)("eieio\n");
	break;

    case 0x3d6:		/* icbi */
	if (dis)
	    VG_(printf)("icbi r%d,r%d\n", ra, rb);
	/* emit a CALLM to a helper to invalidate the translation cache
	   for the cache line at the effective address */
	t1 = calc_ea_index(cb, ra, rb, 0);
	uInstr0(cb, CALLM_S, 0);
	uInstr1(cb, PUSH, 4, TempReg, t1);
	uInstr1(cb, CALLM, 0, Literal, 0);
	uLiteral(cb, (Addr) &VG_(helper_cache_inval));
	uInstr0(cb, CALLM_E, 0);
	break;

    case 0x96:		/* stwcx. */
	if (!record)
	    goto bad;
	record = 0;
	if (dis)
	    VG_(printf)("stwcx. r%d,r%d,r%d\n", rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	t2 = new_temp(cb);
	uInstr2(cb, GET,    4, ArchReg, rd, TempReg, t2);
	uInstr0(cb, LOCK,   0);
	uInstr2(cb, STORE,  4, TempReg, t2, TempReg, t1);
	uFlagsR(cb, FlagSO);
	uInstr3(cb, ICRF,   4, RealReg, R_CR, Lit16, 0, ArchReg, R_CR);
	break;

    case 0x216:		/* lwbrx */
    case 0x316:		/* lhbrx */
	sz = (minor & 0x100)? 2: 4;
	if (dis)
	    VG_(printf)("l%cbrx r%d,r%d,r%d\n", sznames[sz-1], rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD,  sz, TempReg, t1, TempReg, t2);
	uInstr1(cb, BSWAP, sz, TempReg, t2);
	uInstr2(cb, PUT,    4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x296:		/* stwbrx */
    case 0x396:		/* sthbrx */
	sz = (minor & 0x100)? 2: 4;
	if (dis)
	    VG_(printf)("st%cbrx r%d,r%d,r%d\n", sznames[sz-1], rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	t2 = new_temp(cb);
	uInstr2(cb, GET,    4, ArchReg, rd, TempReg, t2);
	uInstr1(cb, BSWAP, sz, TempReg, t2);
	uInstr2(cb, STORE, sz, TempReg, t2, TempReg, t1);
	break;

    case 0x3f6:		/* dcbz */
	if (dis)
	    VG_(printf)("dcbz r%d,r%d\n", ra, rb);
	t1 = calc_ea_index(cb, ra, rb, 0);
	uInstr2(cb, AND,   4, Literal,  0, TempReg, t1);
	uLiteral(cb, -VG_(cache_line_size));
	t2 = new_temp(cb);
	uInstr2(cb, MOV,   4, Literal,  0, TempReg, t2);
	uLiteral(cb, 0);
	uInstr2(cb, STORE, 4, TempReg, t2, TempReg, t1);
	for (i = VG_(cache_line_size) / 4 - 1; i > 0; --i) {
	    uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, 4);
	    uInstr2(cb, STORE, 4, TempReg, t2, TempReg, t1);
	}
	break;

    case 0x156:		/* dst[t] */
    case 0x176:		/* dstst[t] */
	if (dis)
	    VG_(printf)("dst%s%s r%d,r%d,%d\n", B(0x20, "st"),
			(rd & 0x10)? "t": "",
			ra, rb, rd & 3);
	break;

    case 0x336:		/* dss */
	if (dis) {
	    if (rd & 0x10)
		VG_(printf)("dssall\n");
	    else
		VG_(printf)("dss %d\n", rd);
	}
	break;

    case 0x17:		/* lwzx */
    case 0x37:		/* lwzux */
    case 0x57:		/* lbzx */
    case 0x77:		/* lbzux */
    case 0x117:		/* lhzx */
    case 0x137:		/* lhzux */
    case 0x157:		/* lhax */
    case 0x177:		/* lhaux */
	sz = (minor & 0x100)? 2: (minor & 0x40)? 1: 4;
	if (dis)
	    VG_(printf)("l%cz%sx r%d,r%d,r%d\n", sznames[sz-1],
			B(0x20, "u"), rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, minor & 0x20);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD, sz, TempReg, t1, TempReg, t2);
	if (minor > 0x140) {
	    uInstr1(cb, WIDEN, 4, TempReg, t2);
	    LAST_UINSTR(cb).extra4b = 2;
	    LAST_UINSTR(cb).signed_widen = True;
	}
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 0x97:		/* stwx */
    case 0xb7:		/* stwux */
    case 0xd7:		/* stbx */
    case 0xf7:		/* stbux */
    case 0x197:		/* sthx */
    case 0x1b7:		/* sthux */
	sz = (minor & 0x100)? 2: (minor & 0x40)? 1: 4;
	if (dis)
	    VG_(printf)("st%c%sx r%d,r%d,r%d\n", "bh?w"[sz-1],
			B(0x20, "u"), rd, ra, rb);
	t2 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	/* must get reg before updating it, for update forms with rd==ra */
	t1 = calc_ea_index(cb, ra, rb, minor & 0x20);
	uInstr2(cb, STORE, sz, TempReg, t2, TempReg, t1);
	break;

    case 0x217:		/* lfsx */
    case 0x237:		/* lfsux */
    case 0x257:		/* lfdx */
    case 0x277:		/* lfdux */
    case 0x297:		/* stfsx */
    case 0x2b7:		/* stfsux */
    case 0x2d7:		/* stfdx */
    case 0x2f7:		/* stfdux */
	if (dis)
	    VG_(printf)("%s%s%sx f%d,r%d,r%d\n",
			(minor & 0x80)? "stf": "lf",
			(minor & 0x40)? "d": "s", B(0x20, "u"), rd, ra, rb);
	t1 = calc_ea_index(cb, ra, rb, minor & 0x20);
	sz = (minor & 0x40)? 8: 4;
	op = (minor & 0x80)? FPU_W: FPU_R;
	uInstr2(cb, op, sz, Lit16, rd, TempReg, t1);
	break;
	
    case 0x3d7:		/* stfiwx */
	goto bad;

    case 0x18:		/* slw[.] */
    case 0x218:		/* srw[.] */
	if (dis)
	    VG_(printf)("s%sw%s r%d,r%d,r%d\n", (minor & 0x200)? "r": "l",
			RC, ra, rd, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	op = (minor & 0x200)? SHR: SHL;
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, tcc);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
	uInstr2(cb, op,    4, TempReg, t1, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x318:		/* sraw[.] */
	if (dis)
	    VG_(printf)("sraw%s r%d,r%d,r%d\n", RC, ra, rd, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, tcc);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
	uInstr2(cb, SAR,   4, TempReg, t1, TempReg, tcc);
	uFlagsW(cb, FlagCA);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x338:		/* srawi[.] */
	if (dis)
	    VG_(printf)("srawi%s r%d,r%d,%d\n", RC, ra, rd, rb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, tcc);
	uInstr2(cb, SAR,   4, Lit16,   rb, TempReg, tcc);
	uFlagsW(cb, FlagCA);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x1a:		/* cntlzw[.] */
	if (dis)
	    VG_(printf)("cntlzw%s r%d,r%d\n", RC, ra, rd);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, tcc);
	uInstr1(cb, CNTLZ, 4, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x39a:		/* extsh[.] */
	if (dis)
	    VG_(printf)("extsh%s r%d,r%d\n", RC, ra, rd);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   2, ArchReg, rd, TempReg, tcc);
	uInstr1(cb, WIDEN, 4, TempReg, tcc);
	LAST_UINSTR(cb).extra4b = 2;
	LAST_UINSTR(cb).signed_widen = True;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x3ba:		/* extsb[.] */
	if (dis)
	    VG_(printf)("extsb%s r%d,r%d\n", RC, ra, rd);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   1, ArchReg, rd, TempReg, tcc);
	uInstr1(cb, WIDEN, 4, TempReg, tcc);
	LAST_UINSTR(cb).extra4b = 1;
	LAST_UINSTR(cb).signed_widen = True;
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x1c:		/* and[.] */
    case 0x1dc:		/* nand[.] */
	if (dis)
	    VG_(printf)("%sand%s r%d,r%d,r%d\n", B(0x100, "n"),
			RC, ra, rd, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, AND,   4, TempReg, t1, TempReg, tcc);
	if (minor & 0x100)
	    uInstr1(cb, NOT,   4, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x3c:		/* andc[.] */
    case 0x19c:		/* orc[.] */
	if (dis)
	    VG_(printf)("%sc%s r%d,r%d,r%d\n", (minor & 0x100)? "or": "and",
			RC, ra, rd, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	op = (minor & 0x100)? OR: AND;
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr1(cb, NOT,   4, TempReg, tcc);
	uInstr2(cb, op,    4, TempReg, t1, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x7c:		/* nor[.] */
    case 0x1bc:		/* or[.] */
	if (dis)
	    VG_(printf)("%sor%s r%d,r%d,r%d\n", (minor & 0x100)? "": "n",
			RC, ra, rd, rb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, tcc);
	if (rb != rd) {
	    t1 = new_temp(cb);
	    uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
	    uInstr2(cb, OR,    4, TempReg, t1, TempReg, tcc);
	}
	if ((minor & 0x100) == 0)
	    uInstr1(cb, NOT,   4, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    case 0x11c:		/* eqv[.] */
    case 0x13c:		/* xor[.] */
	if (dis)
	    VG_(printf)("%s%s r%d,r%d,r%d\n", (minor & 0x20)? "xor": "eqv",
			RC, ra, rd, rb);
	t1 = new_temp(cb);
	tcc = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, tcc);
	uInstr2(cb, XOR,   4, TempReg, t1, TempReg, tcc);
	if ((minor & 0x20) == 0)
	    uInstr1(cb, NOT,   4, TempReg, tcc);
	uInstr2(cb, PUT,   4, TempReg, tcc, ArchReg, ra);
	reg_is_const[ra] = False;
	break;

    default:
    bad:
	return -1;
    }

    if (record && tcc != INVALID_TEMPREG)
	do_record(cb, tcc);

    return 0;

#undef B
#undef B2
}

static void dis_instr(UCodeBlock *cb, Addr nip, int dis, Bool *isend)
{
    unsigned int instr;
    int ra, rb, rd, imm;
    int t1, t2, t3;
    int sh, mb, me, cr;
    int ui = cb->used;
    int op, sz, opcode;
    int rc;
    unsigned int mask;
    char loc_buf[M_VG_ERRTXT];

#define B(x, str)	((instr & (x))? str: "")

    instr = *(unsigned int *)nip;
    if (dis)
	VG_(printf)("\t0x%x:  %08x  ", nip, instr);

    /* extract some fields */
    rd = (instr >> 21) & 0x1f;
    ra = (instr >> 16) & 0x1f;

    *isend = 0;

    /* switch on major opcode */
    opcode = instr >> 26;
    switch (opcode) {
    case 3:		/* twi */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("twi %d,r%d,%d\n", rd, ra, imm);
	/* hmmm */
	break;

    case 4:		/* various altivec instructions */
	rc = 0;
	if ((instr & 0x3f) == 6)
	    rc = instr & 0x400;		/* vcmp... instructions */
	if (dis)
	    VG_(printf)("vecop%s %x\n", (rc? ".": ""), instr);
	uInstr1(cb, VEC, 16, Literal, 0);
	uLiteral(cb, instr);
	if (rc) {
	    /* some instructions set cr6 */
	    uInstr3(cb, ICRF,  4, RealReg, R_CR, Lit16, 6, ArchReg, R_CR);
	    LAST_UINSTR(cb).extra4b = 6;
	}
	break;

    case 7:		/* mulli */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("mulli r%d,r%d,%d\n", rd, ra, imm);
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, MUL,   4, Literal,  0, TempReg, t1);
	uLiteral(cb, imm);
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, rd);
	reg_is_const[rd] = reg_is_const[ra];
	reg_value[rd] = reg_value[ra] * imm;
	break;

    case 8:		/* subfic */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("subfic r%d,r%d,%d\n", rd, ra, imm);
	t1 = new_temp(cb);
	t2 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, MOV,   4, Literal, 0,  TempReg, t2);
	uLiteral(cb, imm);
	uInstr2(cb, SBB,   4, TempReg, t1, TempReg, t2);
	uFlagsW(cb, FlagCA);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = reg_is_const[ra];
	reg_value[rd] = imm - reg_value[ra];
	break;

    case 10:		/* cmpli */
    case 11:		/* cmpi */
	imm = (opcode & 1)? SIMM(instr): UIMM(instr);
	rd >>= 2;	/* should check L bit */
	if (dis)
	    VG_(printf)("cmp%si cr%d,r%d,%d\n", (opcode & 1)? "": "l",
			rd, ra, imm);
	t1 = new_temp(cb);
	t3 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	if (opcode == 11 && imm == 0) {
	    uInstr2(cb, CMP0,   4, TempReg, t1, TempReg, t3);
	} else {
	    t2 = new_temp(cb);
	    uInstr2(cb, MOV,   4, Literal, 0,  TempReg, t2);
	    uLiteral(cb, imm);
	    op = (opcode & 1)? CMP: CMPU;
	    uInstr3(cb, op,    4, TempReg, t1, TempReg, t2, TempReg, t3);
	}
	uFlagsR(cb, FlagSO);
	uInstr3(cb, ICRF,  4, TempReg, t3, Lit16, rd, ArchReg, R_CR);
	break;

    case 12:		/* addic */
    case 13:		/* addic. */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("addic%s r%d,r%d,%d\n",
			(opcode & 1)? ".": "", rd, ra, imm);
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, ADC,   4, Literal, 0,  TempReg, t1);
	uLiteral(cb, imm);
	uFlagsW(cb, FlagCA);
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, rd);
	reg_is_const[rd] = reg_is_const[ra];
	reg_value[rd] = reg_value[ra] + imm;
	if (opcode & 1)
	    do_record(cb, t1);
	break;

    case 14:		/* addi */
	imm = SIMM(instr);
	t1 = new_temp(cb);
	if (ra) {
	    if (dis)
		VG_(printf)("addi r%d,r%d,%d\n", rd, ra, imm);
	    reg_is_const[rd] = reg_is_const[ra];
	    if (reg_is_const[ra]) {
		reg_value[rd] = reg_value[ra] + imm;
		uInstr2(cb, MOV,   4, Literal,  0, TempReg, t1);
		uLiteral(cb, reg_value[rd]);
	    } else {
		uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
		uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
		uLiteral(cb, imm);
	    }
	} else {
	    if (dis)
		VG_(printf)("li r%d,%d\n", rd, imm);
	    uInstr2(cb, MOV,   4, Literal, 0,  TempReg, t1);
	    uLiteral(cb, imm);
	    reg_is_const[rd] = True;
	    reg_value[rd] = imm;
	}
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, rd);
	break;

    case 15:		/* addis */
	imm = IMMS(instr);
	t1 = new_temp(cb);
	if (ra) {
	    if (dis)
		VG_(printf)("addis r%d,r%d,%d\n", rd, ra, SIMM(instr));
	    reg_is_const[rd] = reg_is_const[ra];
	    if (reg_is_const[ra]) {
		reg_value[rd] = reg_value[ra] + imm;
		uInstr2(cb, MOV,   4, Literal,  0, TempReg, t1);
		uLiteral(cb, reg_value[rd]);
	    } else {
		uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
		uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
		uLiteral(cb, imm);
	    }
	} else {
	    if (dis)
		VG_(printf)("lis r%d,%d\n", rd, SIMM(instr));
	    uInstr2(cb, MOV,   4, Literal, 0,  TempReg, t1);
	    uLiteral(cb, imm);
	    reg_is_const[rd] = True;
	    reg_value[rd] = imm;
	}
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, rd);
	break;

    case 16:		/* bc */
	imm = (signed short)(instr & 0xfffc);
	if ((instr & 2) == 0)
	    imm += nip;
	if (dis)
	    VG_(printf)("bc%s%s %d,%d,0x%x\n",
			(instr & 1)? "l": "",
			(instr & 2)? "a": "", rd, ra, imm);
	if (instr & 1)
	    set_lr(cb, nip + 4);
	if (instr == 0x429f0005)
	    break;		/* special form for getting NIP into LR */
	dis_condition(cb, instr, nip + 4);
	uInstr1(cb, JMP,   0, Literal, 0);
	uLiteral(cb, imm);
	if (rd & 0x10) {
	    /* unconditional or bd[n]z[l] */
	    set_cond(cb, CondAlways);
	} else {
	    set_cond(cb, ((rd & 8)? CondIfSet: CondIfClear) + ra);
	}
	if (instr & 1)
	    LAST_UINSTR(cb).jmpkind = JmpCall;
	*isend = 1;
	break;

    case 17:		/* sc */
	if (dis)
	    VG_(printf)("sc\n");
	uInstr1(cb, JMP, 0, Literal, 0);
	uLiteral(cb, nip + 4);
	set_cond(cb, CondAlways);
	LAST_UINSTR(cb).jmpkind = JmpSyscall;
	*isend = 1;
	break;

    case 18:		/* b */
	imm = instr & 0x03fffffc;
	if (imm & 0x02000000)
	    imm -= 0x04000000;
	if ((instr & 2) == 0)
	    imm += nip;
	if (dis)
	    VG_(printf)("b%s%s 0x%x\n",
			(instr & 1)? "l": "",
			(instr & 2)? "a": "", imm);
	if (instr & 1)
	    set_lr(cb, nip + 4);
	if (imm != nip + 4) {
	    uInstr1(cb, JMP,   0, Literal,  0);
	    uLiteral(cb, imm);
	    set_cond(cb, CondAlways);
	    if (instr & 1)
		LAST_UINSTR(cb).jmpkind = JmpCall;
	    *isend = 1;
	}
	break;

    case 19:		/* various */
	switch ((instr >> 1) & 0x3ff) {
	case 0:		/* mcrf */
	    if (dis)
		VG_(printf)("mcrf cr%d,cr%d\n", rd >> 2, ra >> 2);
	    uInstr3(cb, ICRF, 4, ArchReg, R_CR, Lit16, rd >> 2, ArchReg, R_CR);
	    LAST_UINSTR(cb).extra4b = ra >> 2;
	    break;

	case 16:	/* bclr */
	    if (dis) {
		if ((rd & 0x14) == 0x14)
		    VG_(printf)("blr%s\n", B(1, "l"));
		else
		    VG_(printf)("bclr%s %d,%d\n", B(1, "l"), rd, ra);
	    }
	    t1 = new_temp(cb);
	    uInstr2(cb, GET,   4, ArchReg, R_LR, TempReg, t1);
	    if (instr & 1)
		set_lr(cb, nip + 4);
	    dis_condition(cb, instr, nip + 4);
	    /* bclr always aligns the dest PC, even if LR is not aligned */
	    uInstr2(cb, AND,   4, Literal, 0, TempReg, t1);
	    uLiteral(cb, ~0x3UL);
	    uInstr1(cb, JMP,   0, TempReg, t1);
	    if (rd & 0x10) {
		/* unconditional blr[l] or bd[n]zlr[l] */
		set_cond(cb, CondAlways);
	    } else {
		set_cond(cb, ((rd & 8)? CondIfSet: CondIfClear) + ra);
	    }
	    LAST_UINSTR(cb).jmpkind = JmpRet;
	    *isend = 1;
	    break;

	case 33:	/* crnor, 0x21 */
	case 129:	/* crandc, 0x81 */
	case 193:	/* crxor, 0xc1 */
	case 225:	/* crnand, 0xe1 */
	case 257:	/* crand, 0x101 */
	case 289:	/* creqv, 0x121 */
	case 417:	/* crorc, 0x1a1 */
	case 449:	/* cror, 0x1c1 */
	    dis_crop(cb, dis, instr);
	    break;

	case 150:	/* isync */
	    break;

	case 528:	/* bctr */
	    if (dis) {
		if ((rd & 0x14) == 0x14)
		    VG_(printf)("bctr%s\n", B(1, "l"));
		else
		    VG_(printf)("bcctr%s %d,%d\n", B(1, "l"), rd, ra);
	    }
	    if (instr & 1)
		set_lr(cb, nip + 4);
	    t1 = new_temp(cb);
	    uInstr2(cb, GET,   4, ArchReg, R_CTR, TempReg, t1);
	    dis_condition(cb, instr, nip + 4);
	    /* bctr always aligns the dest PC, even if CTR is not aligned */
	    uInstr2(cb, AND,   4, Literal, 0, TempReg, t1);
	    uLiteral(cb, ~0x3UL);
	    uInstr1(cb, JMP,   0, TempReg, t1);
	    if (rd & 0x10) {
		/* unconditional bctr[l] or bd[n]zctr[l] */
		set_cond(cb, CondAlways);
	    } else {
		set_cond(cb, ((rd & 8)? CondIfSet: CondIfClear) + ra);
	    }
	    if (instr & 1)
		LAST_UINSTR(cb).jmpkind = JmpCall;
	    *isend = 1;
	    break;

	default:
	    goto bad;
	}
	break;

    case 20:		/* rlwimi[.] */
	sh = (instr >> 11) & 0x1f;
	mb = (instr >> 6) & 0x1f;
	me = (instr >> 1) & 0x1f;
	if (dis)
	    VG_(printf)("rlwimi%s r%d,r%d,%d,%d,%d\n",
			(instr & 1? ".": ""), ra, rd, sh, mb, me);
	t1 = new_temp(cb);
	t2 = new_temp(cb);
	mask = MASKG(mb, me);
	uInstr2(cb, GET,   4, ArchReg, ra, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	if (sh != 0)
	    uInstr2(cb, ROL,   4, Lit16, sh, TempReg, t2);
	if (mask != ~0U) {
	    uInstr2(cb, AND,   4, Literal,  0, TempReg, t2);
	    uLiteral(cb, mask);
	    uInstr2(cb, AND,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, ~mask);
	    uInstr2(cb, OR,    4, TempReg, t1, TempReg, t2);
	}
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, ra);
	reg_is_const[ra] = False;
	if (instr & 1)
	    do_record(cb, t2);
	break;

    case 21:		/* rlwinm[.] */
	sh = (instr >> 11) & 0x1f;
	mb = (instr >> 6) & 0x1f;
	me = (instr >> 1) & 0x1f;
	if (dis)
	    VG_(printf)("rlwinm%s r%d,r%d,%d,%d,%d\n",
			(instr & 1? ".": ""), ra, rd, sh, mb, me);
	t2 = new_temp(cb);
	mask = MASKG(mb, me);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	if (sh != 0) {
	    if (mb == 0 && sh + me == 31) {
		uInstr2(cb, SHL,   4, Lit16, sh, TempReg, t2);
		mask = ~0U;
	    } else if (mb + sh == 32 && me == 31) {
		uInstr2(cb, SHR,   4, Lit16, mb, TempReg, t2);
		mask = ~0U;
	    } else {
		uInstr2(cb, ROL,   4, Lit16, sh, TempReg, t2);
	    }
	}
	if (mask != ~0U) {
	    uInstr2(cb, AND,   4, Literal,  0, TempReg, t2);
	    uLiteral(cb, mask);
	}
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, ra);
	reg_is_const[ra] = False;
	if (instr & 1)
	    do_record(cb, t2);
	break;

    case 23:		/* rlwnm[.] */
	rb = (instr >> 11) & 0x1f;
	mb = (instr >> 6) & 0x1f;
	me = (instr >> 1) & 0x1f;
	if (dis)
	    VG_(printf)("rlwnm%s r%d,r%d,%d,%d,%d\n",
			(instr & 1? ".": ""), ra, rd, rb, mb, me);
	t1 = new_temp(cb);
	t2 = new_temp(cb);
	mask = MASKG(mb, me);
	uInstr2(cb, GET,   4, ArchReg, rb, TempReg, t1);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	uInstr2(cb, ROL,   4, TempReg, t1, TempReg, t2);
	if (mask != ~0U) {
	    uInstr2(cb, AND,   4, Literal,  0, TempReg, t2);
	    uLiteral(cb, mask);
	}
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, ra);
	reg_is_const[ra] = False;
	if (instr & 1)
	    do_record(cb, t2);
	break;

    case 24:		/* ori */
	if (instr == 0x60000000) {
	    if (dis)
		VG_(printf)("nop\n");
	    uInstr0(cb, NOP, 0);
	    break;
	}
	/* fall through */

    case 25:		/* oris */
    case 26:		/* xori */
    case 27:		/* xoris */
	imm = (opcode & 1)? IMMS(instr): UIMM(instr);
	if (dis)
	    VG_(printf)("%sori%s r%d,r%d,0x%x\n",
			(opcode & 2)? "x": "",
			(opcode & 1)? "s": "",
			ra, rd, instr & 0xffff);
	t1 = new_temp(cb);
	reg_is_const[ra] = reg_is_const[rd];
	if (reg_is_const[rd]) {
	    if (opcode & 2)
		imm ^= reg_value[rd];
	    else
		imm |= reg_value[rd];
	    reg_value[ra] = imm;
	    uInstr2(cb, MOV,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, imm);
	} else {
	    uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	    if (imm != 0) {
		op = (opcode & 2)? XOR: OR;
		uInstr2(cb, op,    4, Literal,  0, TempReg, t1);
		uLiteral(cb, imm);
	    }
	}
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, ra);
	break;

    case 28:		/* andi. */
    case 29:		/* andis. */
	imm = (opcode & 1)? IMMS(instr): UIMM(instr);
	if (dis)
	    VG_(printf)("andi%s. r%d,r%d,0x%x\n",
			(opcode & 1)? "s": "",
			ra, rd, instr & 0xffff);
	t1 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t1);
	uInstr2(cb, AND,   4, Literal,  0, TempReg, t1);
	uLiteral(cb, imm);
	uInstr2(cb, PUT,   4, TempReg, t1, ArchReg, ra);
	reg_is_const[ra] = False;
	do_record(cb, t1);
	break;

    case 31:		/* various */
	if (instr == 0x7c03d808) {
	    /* magic client request sequence */
	    if (dis)
		VG_(printf)("client_request\n");
	    uInstr1(cb, JMP,  0, Literal, 0);
	    uLiteral(cb, nip + 4);
	    set_cond(cb, CondAlways);
	    LAST_UINSTR(cb).jmpkind = JmpClientReq;
	    *isend = 1;
	    break;
	}
	if (dis_opc_31(cb, dis, instr))
	    goto bad;
	break;

    case 32:		/* lwz */
    case 33:		/* lwzu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("lwz%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD,  4, TempReg, t1, TempReg, t2);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 34:		/* lbz */
    case 35:		/* lbzu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("lbz%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD,  1, TempReg, t1, TempReg, t2);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 36:		/* stw */
    case 37:		/* stwu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("stw%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t2 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	uInstr2(cb, STORE, 4, TempReg, t2, TempReg, t1);
	break;

    case 38:		/* stb */
    case 39:		/* stbu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("stb%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t2 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	uInstr2(cb, STORE, 1, TempReg, t2, TempReg, t1);
	break;

    case 40:		/* lhz */
    case 41:		/* lhzu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("lhz%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD,  2, TempReg, t1, TempReg, t2);
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 42:		/* lha */
    case 43:		/* lhau */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("lha%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	t2 = new_temp(cb);
	uInstr2(cb, LOAD,  2, TempReg, t1, TempReg, t2);
	uInstr1(cb, WIDEN, 4, TempReg, t2);
	LAST_UINSTR(cb).extra4b = 2;
	LAST_UINSTR(cb).signed_widen = True;
	uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	reg_is_const[rd] = False;
	break;

    case 44:		/* sth */
    case 45:		/* sthu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("sth%s r%d,%d(r%d)\n",
			(opcode & 1)? "u": "", rd, imm, ra);
	t2 = new_temp(cb);
	uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	uInstr2(cb, STORE, 2, TempReg, t2, TempReg, t1);
	break;

    case 46:		/* lmw */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("lmw r%d,%d(r%d)\n", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, 0);
	t2 = new_temp(cb);
	do {
	    /* XXX should we skip if rd == ra && ra != 0 ? */
	    uInstr2(cb, LOAD,  4, TempReg, t1, TempReg, t2);
	    uInstr2(cb, PUT,   4, TempReg, t2, ArchReg, rd);
	    reg_is_const[rd] = False;
	    uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, 4);
	} while (++rd < 32);
	break;

    case 47:		/* stmw */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("stmw r%d,%d(r%d)\n", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, 0);
	t2 = new_temp(cb);
	do {
	    uInstr2(cb, GET,   4, ArchReg, rd, TempReg, t2);
	    uInstr2(cb, STORE, 4, TempReg, t2, TempReg, t1);
	    uInstr2(cb, ADD,   4, Literal,  0, TempReg, t1);
	    uLiteral(cb, 4);
	} while (++rd < 32);
	break;

    case 48:		/* lfs */
    case 49:		/* lfsu */
    case 50:		/* lfd */
    case 51:		/* lfdu */
    case 52:		/* stfs */
    case 53:		/* stfsu */
    case 54:		/* stfd */
    case 55:		/* stfdu */
	imm = SIMM(instr);
	if (dis)
	    VG_(printf)("%s%s%s f%d,%d(r%d)\n",
			(opcode & 4)? "stf": "lf",
			(opcode & 2)? "d": "s",
			(opcode & 1)? "u": "", rd, imm, ra);
	t1 = calc_ea_disp(cb, ra, imm, opcode & 1);
	sz = (opcode & 2)? 8: 4;
	op = (opcode & 4)? FPU_W: FPU_R;
	uInstr2(cb, op, sz, Lit16, rd, TempReg, t1);
	break;

    case 59:		/* single-precision FP ops */
	if (dis)
	    VG_(printf)("spfpuop%s %x\n", (instr & 1)? ".": "", instr);
	uInstr1(cb, FPU, 4, Literal, 0);
	uLiteral(cb, instr);
	if (instr & 1) {
	    uInstr3(cb, ICRF,  4, RealReg, R_CR, Lit16, 1, ArchReg, R_CR);
	    LAST_UINSTR(cb).extra4b = 1;
	}
	break;

    case 63:		/* double-precision and special FP ops */
	if (dis)
	    VG_(printf)("fpuop%s %x\n", (instr & 1)? ".": "", instr);
	cr = (instr & 1)? 1: -1;
	switch ((instr >> 1) & 0x3ff) {
	case 0:		/* fcmpu */
	case 32:	/* fcmps */
	case 63:	/* mcrfs */
	    cr = rd >> 2;	/* destination CR field */
	    /* force the crfD field to 1 */
	    instr = (instr & ~0x03e00000) | 0x00800000;
	    break;
	}
	uInstr1(cb, FPU, 8, Literal, 0);
	uLiteral(cb, instr);
	if (cr >= 0) {
	    uInstr3(cb, ICRF,  4, RealReg, R_CR, Lit16, cr, ArchReg, R_CR);
	    LAST_UINSTR(cb).extra4b = 1;
	}
	break;

    default:
	goto bad;
    }

    if (!*isend)
	uInstr1(cb, INCEIP, 4, Lit16, 4);
    else
	LAST_UINSTR(cb).extra4b = 4;	/* instruction size */

    for (; ui < cb->used; ++ui) {
	VG_(sanity_check_UInstr)( ui, &cb->instrs[ui] );
	if (dis)
	    VG_(pp_UInstr)(ui, &cb->instrs[ui]);
    }
    if (dis)
	VG_(printf)("\n");
    return;

 bad:
    if (dis)
	VG_(printf)("<unrecognized>\n");
    VG_(printf)("unrecognized PowerPC instruction: %x\n", instr);
    VG_(describe_eip)(nip, loc_buf, M_VG_ERRTXT);
    VG_(printf)("          at %s\n", loc_buf);

    uInstr0(cb, CALLM_S, 0);
    uInstr1(cb, CALLM, 0, Literal, 0);
    uLiteral(cb, (Addr) &VG_(helper_undefined_instruction));
    uInstr0(cb, CALLM_E, 0);

    uInstr1(cb, JMP, 0, Literal, 0);
    set_cond(cb, CondAlways);
    uLiteral(cb, nip);
    *isend = 1;
}

int VG_(disBB)(UCodeBlock *cb, Addr nip)
{
    Bool isend = 0;
    Addr nip0 = nip;
    int i, n;

    /* Put a limit of 1024 instructions on the length of the basic
       block so that the translation doesn't get too big. */
    n = 1024;
    if (VG_(clo_single_step))
	n = 1;

    for (i = 0; i < 32; ++i)
	reg_is_const[i] = False;

    do {
	dis_instr(cb, nip, VG_(print_codegen), &isend);
	nip += 4;
    } while (--n > 0 && !isend);

    if (cb->used == 0 || LAST_UINSTR(cb).opcode != JMP
	|| LAST_UINSTR(cb).cond != CondAlways) {
	uInstr1(cb, JMP, 0, Literal, 0);
	uLiteral(cb, nip);
	set_cond(cb, CondAlways);
	LAST_UINSTR(cb).extra4b = 4;
	if (VG_(print_codegen))
	    VG_(pp_UInstr)(cb->used - 1, &cb->instrs[cb->used - 1]);
    }
    if (VG_(print_codegen))
	VG_(printf)("\n");

    return nip - nip0;
}

/* -*- c-basic-offset: 4 -*- */
