/*
 * test of the interrupts
 */

#include "../lib/oea_pal.h"

/* spr functions */
void move_to_spr(unsigned spr, unsigned value);
unsigned move_from_spr(unsigned spr);


/* interrupt table addresses */

typedef void trap_function(void);
extern trap_function system_error_interrupt;
extern trap_function machine_check_interrupt;
extern trap_function data_storage_interrupt;
extern trap_function instruction_storage_interrupt;
extern trap_function external_interrupt;
extern trap_function alignment_interrupt;
extern trap_function program_interrupt;
extern trap_function floatingpoint_unavailable_interrupt;
extern trap_function decrementer_interrupt;
extern trap_function reserved_00a00_interrupt;
extern trap_function reserved_00b00_interrupt;
extern trap_function system_call_interrupt;
extern trap_function trace_interrupt;
extern trap_function floatingpoint_assist_interrupt;
extern trap_function reserved_0_interrupt;

/* msr bit names */
typedef enum {
  ILE = 0x00010000, /*15*/
  EE =  0x00008000, /*16*/
  PR =  0x00004000, /*17*/
  FP =  0x00002000, /*18*/
  ME =  0x00001000, /*19*/
  FE0 = 0x00000800, /*20*/
  SE =  0x00000400, /*21*/
  BE =  0x00000200, /*22*/
  FE1 = 0x00000100, /*23*/
  IP =  0x00000040, /*25*/
  IR =  0x00000020, /*26*/
  DR =  0x00000010, /*27*/
  RI =  0x00000002, /*30*/
  LE =  0x00000001  /*31*/
} msr;

#ifdef _LITTLE_ENDIAN
#define _ILE ILE
#define _LE LE
#else
#define _ILE 0
#define _LE 0
#endif

/* test routines */
typedef void test_function(unsigned a0, unsigned a1, unsigned a2, unsigned a3);

extern test_function tst_sc1;
extern trap_function trap_sc1;

extern test_function tst_trap1;
extern trap_function trap_trap1;

extern test_function tst_fp1;
extern trap_function trap_fp1_load;
extern trap_function trap_fp1_assist;
extern trap_function trap_fp1_trap;

extern test_function tst_ill1;
extern trap_function trap_ill1;

extern test_function tst_priv1;
extern trap_function trap_priv1_ok;
extern trap_function trap_priv1_priv;

extern test_function tst_store;
extern trap_function trap_store;
extern trap_function trap_store_ok;

extern test_function tst_load;
extern trap_function trap_load;
extern trap_function trap_load_ok;

extern test_function tst_dec;
extern trap_function trap_dec_a;
extern trap_function trap_dec_nop_2;
extern trap_function trap_dec_trap;

extern test_function tst_smpee;
extern trap_function tst_smpee_idle;
extern trap_function tst_smpee_msr;
extern trap_function trap_smpee_idle;

enum bat_constants {
  nr_bats = 16,
  first_bat_spr = 528
};


enum batu_constants {
  batu_Vs = 0x2,
  batu_Vp = 0x1,
  batu_bl_128k = 0x000 << 2,
  batu_bl_256k = 0x001 << 2,
  batu_bl_512k = 0x003 << 2,
  batu_bl_1m = 0x00007 << 2,
  batu_bl_2m = 0x0000f << 2,
  batu_bl_4m = 0x0001f << 2,
  batu_bl_8m = 0x0003f << 2,
  batu_bl_16m = 0x007f << 2,
  batu_bl_32m = 0x00ff << 2,
  batu_bl_64m = 0x01ff << 2,
  batu_bl_128m = 0x3ff << 2,
  batu_bl_256m = 0x7ff << 2,
  batu_bepi_mask = 0xfffe0000
} batu_constants;

enum batl_constants {
  batl_bprn_mask = 0xfffe0000,
  batl_pp_no_access = 0x0,
  batl_pp_read_only = 0x1,
  batl_pp_read_write = 0x2,
  batl_wimg_write_through = 0x40,
  batl_wimg_caching_inhibited = 0x20,
  batl_wimg_memory_coherence = 0x10,
  batl_wimg_guarded = 0x08
};

typedef bat_table[nr_bats];

void
set_bats(bat_table bats)
{
  int bat;
  for (bat = 0; bat < nr_bats; bat++) {
    move_to_spr(first_bat_spr+bat, bats[bat]);
  }
}

void
tst_bats(unsigned a0, unsigned a1, unsigned a2, unsigned a3)
{
  unsigned msr = a0;
  unsigned *bats = (unsigned*)a1;
  unsigned dar = a2;
  test_function *test = (test_function*)a3;

  set_bats(bats);
  test(a0, a1, a2, a3);
}


bat_table bat_s_ro_128k = {
  /* ibat0u */ 0 | batu_Vs | batu_bl_128k,
  /* ibat0l */ 0 | batl_pp_read_only,
  /* ibat1[ul] */ 0, 0,
  /* ibat2[ul] */ 0, 0,
  /* ibat3[ul] */ 0, 0,
  /* dbat0u */ 0 | batu_Vs | batu_bl_128k,
  /* dbat0l */ 0 | batl_pp_read_only,
  /* dbat1[ul] */ 0, 0,
  /* dbat2[ul] */ 0, 0,
  /* dbat3[ul] */ 0, 0,
};


/* page hash table */

unsigned htab = 0x20000;
unsigned sizeof_htab = 0x20000;

typedef struct _htab_entries {
  unsigned va;
  unsigned ra;
  unsigned writeable;
} htab_entries;

typedef struct _tab_init {
  htab_entries *htab_entry;
  unsigned sr[16];
} htab_init;


void
load_page(unsigned va, unsigned ra, unsigned writeable)
{
}


void
tst_htab(unsigned a0, unsigned a1, unsigned a2, unsigned a3)
{
  unsigned msr = a0;
  unsigned dar = a1;
  htab_init *init = (htab_init*)a2;
  test_function *test = (test_function*)a3;
  int i;

  /* set up the htab address */
  move_to_spr(25, htab | ((sizeof_htab - 1) >> (10+6)));

  /* load the sr's */
  for (i = 0; i < 16; i++) {
    move_to_sr(i, init->sr[i]);
  }

  /* load any relevant ptes */
  bzero(htab, sizeof_htab);
  for (i = 0; init->htab_entry[i].ra != 0; i++) {
    load_page(init->htab_entry[i].va,
	      init->htab_entry[i].ra,
	      init->htab_entry[i].writeable);
  }
  
  test(msr, dar, 0, 0);
}



/* interrupt tests */


enum interrupt_checks {
  skip_srr0 = 0x1,
  skip_srr1 = 0x2,
  skip_trap = 0x4,
  skip_msr = 0x8,
  skip_dar = 0x10
};

typedef struct _interrupt_test {
  char *name;
  enum interrupt_checks check;

  test_function *function;
  unsigned args[4];

  trap_function *srr0; /*pc*/
  unsigned srr1; /*msr*/

  trap_function *trap;
  unsigned msr;

  unsigned dar;
} interrupt_test;

/* GLOBALS that keep info across each interrupt */

volatile int failures = 0;
interrupt_test *volatile current_test;
interrupt_test *volatile next_test; /* NULL -> no next */

/* Test table */

interrupt_test tests[] = {
  { "sc1 - supervisor mode", skip_dar,
      tst_sc1, { _ILE|RI|_LE },
      trap_sc1, RI|_LE,
      system_call_interrupt, _ILE|_LE },
  { "sc2 - user mode", skip_dar,
      tst_sc1, { _ILE|RI|_LE|PR },
      trap_sc1, RI|_LE|PR,
      system_call_interrupt, _ILE|_LE },
  { "trap1 - supervisor mode", skip_dar,
      tst_trap1, { _ILE|RI|_LE },
      trap_trap1, RI|_LE|0x20000, /* trap bit */
      program_interrupt, _ILE|_LE },
  { "trap2 - user mode", skip_dar,
      tst_trap1, { _ILE|RI|_LE|PR },
      trap_trap1, RI|_LE|PR|0x20000, /* trap bit */
      program_interrupt, _ILE|_LE },
  { "fp1 - no fp, supervisor mode", skip_dar,
      tst_fp1, { _ILE|RI|_LE },
      trap_fp1_load, RI|_LE,
      floatingpoint_unavailable_interrupt, _ILE|_LE },
  { "fp2 - no fp, user mode", skip_dar,
      tst_fp1, { _ILE|RI|_LE|PR },
      trap_fp1_load, RI|_LE|PR,
      floatingpoint_unavailable_interrupt, _ILE|_LE },
  { "fp3 - fp assist, supervisor mode", skip_dar,
      tst_fp1, { _ILE|RI|_LE|FP },
      trap_fp1_assist, RI|_LE|FP,
      floatingpoint_assist_interrupt, _ILE|_LE },
  { "fp4 - fp assist, user mode", skip_dar,
      tst_fp1, { _ILE|RI|_LE|FP|PR },
      trap_fp1_assist, RI|_LE|FP|PR,
      floatingpoint_assist_interrupt, _ILE|_LE },
  { "ill1 - illegal, supervisor", skip_dar,
      tst_ill1, { _ILE|RI|_LE },
      trap_ill1, RI|_LE|0x80000, /* illegal */
      program_interrupt, _ILE|_LE },
  { "ill2 - illegal, user", skip_dar,
      tst_ill1, { _ILE|RI|_LE|PR },
      trap_ill1, RI|_LE|PR|0x80000, /* illegal */
      program_interrupt, _ILE|_LE },
  { "priv1 - privlidged, supervisor", skip_dar,
      tst_priv1, { _ILE|RI|_LE },
      trap_priv1_ok, RI|_LE|0x20000, /* trap */
      program_interrupt, _ILE|_LE },
  { "priv2 - privlidged, user", skip_dar,
      tst_priv1, { _ILE|RI|_LE|PR },
      trap_priv1_priv, RI|_LE|PR|0x40000, /* privlidged */
      program_interrupt, _ILE|_LE },
#if defined(_LITTLE_ENDIAN)
  { "al1 - alignment", 0,
      tst_fp1, { _ILE|RI|_LE|FP, 1 },
      trap_fp1_load, RI|_LE|FP,
      alignment_interrupt, _ILE|_LE,
      1 },
  { "al2 - alignment", 0,
      tst_fp1, { _ILE|RI|_LE|FP, 2 },
      trap_fp1_load, RI|_LE|FP,
      alignment_interrupt, _ILE|_LE,
      2 },
  { "al3 - alignment", 0,
      tst_fp1, { _ILE|RI|_LE|FP, 3 },
      trap_fp1_load, RI|_LE|FP,
      alignment_interrupt, _ILE|_LE,
      3 },
  { "al4(7) - alignment", 0,
      tst_fp1, { _ILE|RI|_LE|FP, 7 },
      trap_fp1_load, RI|_LE|FP,
      alignment_interrupt, _ILE|_LE,
      7 },
#endif
  { "bat1 - store", 0,
      tst_bats,
 { IR|DR|_ILE|RI|_LE, (unsigned)bat_s_ro_128k, 0, (unsigned)tst_store },
      trap_store, IR|DR|RI|_LE,
      data_storage_interrupt, _ILE|_LE,
      0 },
  { "ee0 - interrupt self", 0,
      tst_smpee, { RI|_LE|EE|_ILE,
#if defined(_LITTLE_ENDIAN)
		     0x0100, /*level=1, cpu=0*/
#else
		     0x0001, /*cpu=0, level=1*/
#endif
		     OEA_DEV + PAL_INT, 0, },
      tst_smpee_idle, RI|_LE|EE,
      external_interrupt, _ILE|_LE,
      0 },
  { "ee1 - re-enable interrupts", 0,
      tst_smpee, { RI|_LE|EE|_ILE, 0, OEA_DEV + PAL_INT, 0, },
      tst_smpee_msr, RI|_LE|EE,
      external_interrupt, _ILE|_LE,
      0 },
  { "ee2 - disable interrupt source", 0,
      tst_store, { RI|_LE|_ILE, 0x00, OEA_DEV + PAL_INT, 0, },
      trap_store_ok, RI|_LE|0x20000, /* trap bit */
      program_interrupt, _ILE|_LE,
      0 },
  { "ee3 - re-enable interrupts", 0,
      tst_store, { RI|_LE|_ILE|EE, 0x00, OEA_DEV + PAL_INT, 0, },
      trap_store_ok, RI|_LE|EE|0x20000, /* trap bit */
      program_interrupt, _ILE|_LE,
      0 },
  { "dec1 - count down from 10", 0,
      tst_dec,
      { _ILE|RI|_LE|EE, 9, 0, 0},
      trap_dec_a, RI|_LE|EE,
      decrementer_interrupt, _ILE|_LE,
      0 },
  { "dec2 - time out when EE disabled - trap at re-enable", 0,
      tst_dec,
      { _ILE|RI|_LE, 9, _ILE|RI|_LE|EE, 0},
      trap_dec_nop_2, RI|_LE|EE,
      decrementer_interrupt, _ILE|_LE,
      0 },
  { "dec3 - trap on 0-1 transition of DEC", 0,
      tst_dec,
      { _ILE|RI|_LE|EE, 0x0fffffff, _ILE|RI|_LE|EE, 0x80000000},
      trap_dec_trap, RI|_LE|EE,
      decrementer_interrupt, _ILE|_LE,
      0 },
  { "smp0 - interrupt cpu 1", 0,
      tst_smpee, { RI|_LE|EE|_ILE,
#if defined(_LITTLE_ENDIAN)
		     0x0101, /*level=1, cpu=1*/
#else
		     0x0101, /*cpu=1, level=1*/
#endif
		     OEA_DEV + PAL_INT, 0, },
      trap_smpee_idle, RI|_LE|EE|FP,
      external_interrupt, _LE,
      0 },
  { "smp1 - interrupt cpu 0", 0,
      tst_smpee, { RI|_LE, /* don't enable interrupts */
#if defined(_LITTLE_ENDIAN)
		     0x0100, /*level=1, cpu=0*/
#else
		     0x0001, /*cpu=0, level=1*/
#endif
		     OEA_DEV + PAL_INT, 0, },
      tst_smpee_idle, RI|_LE|EE,
      external_interrupt, _LE|_ILE,
      0 },
  { 0 }, /* sentinal */
};



void
throw(interrupt_test *test)
{
  if (test->name == 0) {
    write_string("Internal error: null test"); write_line();
    exit(1);
  }
  write_string(test->name);
  write_string(":"); write_line();
  test->function(test->args[0],
		 test->args[1],
		 test->args[2],
		 test->args[3]);
}

void
verify(char *name,
       unsigned result,
       unsigned answer)
{
  write_string("\t");
  write_string(name);
  write_string("=0x");
  write_hex(result);
  if (result == answer) {
    write_string(" (ok)"); write_line();
  }
  else {
    write_string(" != (answer) 0x");
    write_hex(answer);
    write_line();
    failures ++;
  }
}


void
catch(trap_function *srr0,
      unsigned srr1,
      trap_function *trap,
      unsigned msr,
      unsigned dar)
{
  interrupt_test *test = current_test;

  /* check this test ran ok */
  if (!(test->check & skip_srr0))
    verify("srr0", (unsigned)srr0, (unsigned)test->srr0);
  if (!(test->check & skip_srr1))
    verify("srr1", srr1, test->srr1);
  if (!(test->check & skip_trap))
    verify("trap", (unsigned)trap, (unsigned)test->trap);
  if (!(test->check & skip_msr))
    verify("msr", msr, test->msr);
  if (!(test->check & skip_dar))
    verify("dar", dar, test->dar);

  /* go onto the next */
  if (next_test != 0 && next_test->name != 0) {
    interrupt_test *test = next_test;
    current_test = test;
    next_test = test+1;
    throw(test);
  }
  else {
    if (failures > 0) {
      write_string("There were errors");
      write_line();
      exit(1);
    }
    else {
      write_string("done.");
      write_line();
      exit(0);
    }
  }
}


int
main(int argc, char **argv)
{

  if (nr_cpu() < 2) {
    write_string("Error: must enable more than one processor");
    write_line();
    exit(1);
  }

  if (argc > 2) {
    interrupt_test *test;
    write_string("Usage:"); write_line();
    write_string("\tinterrupt [ <test-name> ]"); write_line();
    write_string("where possible tests are"); write_line();
    for (test = tests; test->name != 0; test++) {
      write_string("\t");
      write_string(test->name);
      write_line();
    }
  }
  else if (argc == 2) {
    interrupt_test *test;
    for (test = tests; test->name != 0; test++) {
      if (strcmp(argv[1], test->name) == 0) {
	current_test = test;
	next_test = 0;
	throw(test);
      }
    }
    write_string("Error: test `");
    write_string(argv[1]);
    write_string("' not found.");
    write_line();
    exit(failures + 1);
  }
  else {
    next_test = tests+1;
    current_test = tests;
    throw(tests);
  }

  exit(1); /* should never reach this line */
}
