/*
 * Copyright (C) 2001-2004 Sistina Software, Inc. All rights reserved.  
 * Copyright (C) 2004 Red Hat, Inc. All rights reserved.
 *
 * This file is part of LVM2.
 *
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU General Public License v.2.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "tools.h"

static char *_expand_filename(const char *template, const char *vg_name,
			      char **last_filename)
{
	char *filename;

	if (security_level())
		return dbg_strdup(template);

	filename = dbg_malloc(PATH_MAX);
	if (snprintf(filename, PATH_MAX, template, vg_name) < 0) {
		log_error("Error processing filename template %s",
			   template);
		dbg_free(filename);	
		return NULL;
	}
	if (*last_filename && !strncmp(*last_filename, filename,
				      strlen(template))) {
		log_error("VGs must be backed up into different files. "
			  "Use %%s in filename for VG name.");
		dbg_free(filename);
		return NULL;
	}

	dbg_free(*last_filename);
	*last_filename = filename;

	return filename;
}

static int vg_backup_single(struct cmd_context *cmd, const char *vg_name,
			    struct volume_group *vg, int consistent,
			    void *handle)
{
	char **last_filename = (char **)handle;
	char *filename;

	if (!vg) {
		log_error("Volume group \"%s\" not found", vg_name);
		return ECMD_FAILED;
	}

	if (!consistent)
		log_error("Warning: Volume group \"%s\" inconsistent", vg_name);

	if (arg_count(cmd, file_ARG)) {
		if (!(filename = _expand_filename(arg_value(cmd, file_ARG),
						  vg->name, last_filename))) {
			stack;
			return ECMD_FAILED;
		}

		backup_to_file(filename, vg->cmd->cmd_line, vg);
	} else {
		if (!consistent) {
			log_error("No backup taken: specify filename with -f "
				  "to backup an inconsistent VG");
			stack;
			return ECMD_FAILED;
		}

		/* just use the normal backup code */
		backup_enable(cmd, 1);	/* force a backup */
		if (!backup(vg)) {
			stack;
			return ECMD_FAILED;
		}
	}

	log_print("Volume group \"%s\" successfully backed up.", vg_name);
	return ECMD_PROCESSED;
}

int vgcfgbackup(struct cmd_context *cmd, int argc, char **argv)
{
	int ret;
	char *last_filename = NULL;

	if (partial_mode())
		init_pvmove(1);

	ret = process_each_vg(cmd, argc, argv, LCK_VG_READ, 0, &last_filename,
			      &vg_backup_single);

	dbg_free(last_filename);

	init_pvmove(0);

	return ret;
}
