/*
 * Author: Heinz Mauelshagen, Germany
 *
 * April-May 1997
 * May,June 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/16/1997 - change lv_get_index to lv_get_index_by_name
 *    11/09/1997 - added lvmtab handling
 *    05/16/1998 - added lvmtab checking
 *    06/12/1998 - enhanced checking numbers in option arguments
 *    06/27/1998 - changed lvm_tab_* calling convention
 *
 */

/*
 * TODO
 *
 * allow striped volume extensiom on multiple stripe count volumes
 *
 */

#include <lvm_user.h>

char *cmd = NULL;

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int c = 0;
   int i = 0;
   int l = 0;
   int p = 0;
   int pa = 0;
   int opt_A = 1;
   int opt_l = 0;
   int opt_L = 0;
   int opt_v = 0;
   int ret = 0;
   int sign = 0;
   int size_rest = 0;
   uint new_size = 0;
#ifdef DEBUG
   char *options = "A:dh?l:L:v";
#else
   char *options = "A:h?l:L:v";
#endif
   char *lv_name = NULL;
   char *vg_name = NULL;
   char **pv_allowed = NULL;
   vg_t *vg = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Logical Volume Extend\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t{-l [+]LogicalExtentsNumber |\n"
                     "\t -L [+]LogicalVolumeSize}\n"
                     "\t[-v]\n"
                     "\tLogicalVolumePath [ PhysicalVolumePath... ]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'l':
            if ( opt_L > 0) {
               fprintf ( stderr, "%s -- L option yet given\n\n", cmd);
               return 1;
            }
            if ( opt_l > 0) {
               fprintf ( stderr, "%s -- l option yet given\n\n", cmd);
               return 1;
            }
            sign = 0;
            if ( *optarg == '+') { sign = 1; optarg++;}
            if ( lvm_check_number ( optarg) == TRUE) {
               new_size = atoi ( optarg);
               opt_l++;
            } else {
               fprintf ( stderr, "%s -- ERROR option l argument \"%s\"\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

         case 'L':
            if ( opt_l > 0) {
               fprintf ( stderr, "%s -- l option yet given\n\n", cmd);
               return 1;
            }
            if ( opt_L > 0) {
               fprintf ( stderr, "%s -- L option yet given\n\n", cmd);
               return 1;
            }
            sign = 0;
            if ( *optarg == '+') { sign = 1; optarg++;}
            if ( lvm_check_number ( optarg) == TRUE) {
               new_size = atoi ( optarg);
               opt_L++;
            } else {
               fprintf ( stderr, "%s -- ERROR option L argument \"%s\"\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }


   if ( optind < argc && *argv[optind] == '-') {
      fprintf ( stderr, "%s -- invalid command line\n\n", cmd);
      return 1;
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a logical volume path\n\n", cmd);
      return 1;
   }
   lv_name = argv[optind];
   optind++;

   /* physical volumes in command line */
   if ( optind < argc) {
      if ( opt_v > 0) printf ( "%s -- checking for physical volume paths "
                               "on command line\n", cmd);
      for ( i = optind; i < argc; i++) {
         if ( pv_check_name ( argv[i]) < 0) {
            fprintf ( stderr, "%s -- %s is an invalid physical "
                              "volume name\n\n", cmd, argv[i]);
            return 1;
         }
      }
      pv_allowed = &argv[optind];
   }

   if ( opt_v > 0) printf ( "%s -- checking for valid logical volume path\n",
                            cmd);
   if ( lv_check_name ( lv_name) < 0) {
      fprintf ( stderr, "%s -- no valid logical volume name %s\n\n",
                cmd, lv_name);
      return 1;
   }

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   if ( opt_v > 0) printf ( "%s -- checking existence of logical volume %s\n",
                            cmd, lv_name);
   if ( lvm_tab_lv_check_exist ( lv_name) == FALSE) {
      fprintf ( stderr, "%s -- logical volume %s doesn't exist\n\n",
                cmd, lv_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking for active logical volume %s\n",
                            cmd, lv_name);
   if ( lv_check_active ( vg_name, lv_name) == FALSE) {
      fprintf ( stderr, "%s -- logical volume %s isn't active\n\n",
                cmd, lv_name);
      return 1;
   }

   vg_name = vg_name_of_lv ( lv_name);

   if ( opt_v > 0) printf ( "%s -- checking for active volume group %s\n",
                            cmd, vg_name);
   if ( vg_check_active ( vg_name) == FALSE) {
      fprintf ( stderr, "%s -- can't extend logical volume in inactive "
                        " volume group %s\n\n", cmd, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- getting volume group status from "
                            "VGDA in kernel\n", cmd);
   if ( vg_status ( vg_name, &vg) != 0) {
      fprintf ( stderr, "%s -- can't extend logical volume without "
                        "status of %s\n\n", cmd, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s "
                            "from disk(s)\n",
                            cmd, vg_name);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) != 0) {
      fprintf ( stderr, "%s -- can't extend logical volume:"
                        " couldn't read data of %s\n\n", cmd, vg_name);
      return 1;
   }

   if ( pv_allowed != NULL) {
      for ( pa = 0; pv_allowed[pa] != NULL; pa++) {
         if ( opt_v > 0) printf ( "%s -- checking physical volume %s against "
                                  "volume group\n", pv_allowed[pa], cmd);
         if ( pv_check_in_vg ( vg, pv_allowed[pa]) == FALSE) {
            fprintf ( stderr, "%s -- %s doesn't belong to volume group %s\n\n",
                              cmd, pv_allowed[pa], vg_name);
            return 1;
         }
      }
   }

   if ( opt_v > 0) printf ( "%s -- getting index of logical volume %s "
                            "in volume group %s\n",
                            cmd, lv_name, vg_name);
   if ( ( l = lv_get_index_by_name ( vg, lv_name)) < 0) {
      fprintf ( stderr, "%s -- can't extend logical volume without "
                        " the index of %s\n\n", cmd, lv_name);
      return 1;
   }

   if ( opt_L > 0 || opt_l > 0) {
      if ( opt_L > 0) new_size *= 1024 * 2;
      else            new_size *= vg->pe_size;
      if ( new_size % vg->pe_size > 0) {
         new_size += vg->pe_size;
         printf ( "%s -- rounding size to physical extend boundary\n", cmd);
      }
      new_size = new_size / vg->pe_size * vg->pe_size;
      if ( sign != 0) new_size += vg->lv[l]->lv_size;

      size_rest = new_size % ( vg->lv[l]->lv_stripes * vg->pe_size);
      if ( size_rest != 0) {
         printf ( "%s -- rounding size %d KB to stripe boundary size ",
                  cmd, new_size / 2);
         new_size = new_size - size_rest +
                    vg->lv[l]->lv_stripes * vg->pe_size;
         printf ( "%d KB\n", new_size / 2);
      }

      if ( new_size <= vg->lv[l]->lv_size) {
         fprintf ( stderr, "%s -- new size is smaller than old size\n\n",
                   cmd);
         return 1;
      }
   } else {
      fprintf ( stderr, "%s -- please give an l or L option\n\n", cmd);
      return 1;
   }

   if ( new_size - vg->lv[l]->lv_size >
        ( vg->pe_total - vg->pe_allocated) * vg->pe_size) {
      if ( vg->pe_total - vg->pe_allocated == 0) {
         fprintf ( stderr, "%s -- no free physical extends in %s\n\n",
                           cmd, vg_name);
      } else {
         fprintf ( stderr, "%s -- only %u free physical extends in %s\n\n",
                           cmd, vg->pe_total - vg->pe_allocated, vg_name);
      }
      return 1;
   }

   if ( new_size > LVM_LV_SIZE_MAX( vg)) {
      fprintf ( stderr, "%s -- new size %d is to large for VGDA in kernel\n\n",
                cmd, new_size / 2);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- setting up logical volume %s "
                            "for extend\n", cmd, lv_name);
   if ( ( ret = lv_setup_for_extend ( vg_name, vg, lv_name,
                                      new_size, pv_allowed)) != 0) {
      if ( ret == -LVM_ESIZE) {
         if ( vg->lv[l]->lv_allocation & LV_CONTIGUOUS) {
            if ( vg->lv[l]->lv_stripes > 1)
               fprintf ( stderr, "%s -- not enough free/allocatable "
                                 "contiguous extends on one or more of:\n",
                                 cmd);
            for ( p = 0; vg->pv[p] != NULL; p++) {
               if ( pv_allowed != NULL) {
                  for ( pa = 0; pv_allowed[pa] != NULL; pa++)
                     if ( strcmp ( vg->pv[p]->pv_name, pv_allowed[pa]) == 0)
                        break;
                  if ( pv_allowed[pa] == NULL) continue;
               }
               if ( lv_check_on_pv ( vg->pv[p], l + 1) == TRUE) {
                  fprintf ( stderr, "%s\n", vg->pv[p]->pv_name);
               } else {
                  if ( opt_v > 0) printf ( "%s -- %s is not on %s\n",
                                           cmd, lv_name, vg->pv[p]->pv_name);
               }
            }
         } else {
            fprintf ( stderr, "%s -- not enough free/allocatable physical "
                              "extends to extend %s\n", cmd, lv_name);
         }
      } else {
         fprintf ( stderr, "%s -- ERROR %d setting %s up for extend\n",
                           cmd, ret, lv_name);
      }
      fprintf ( stderr, "\n");
      return 1;
   }

   lvm_dont_interrupt ( 0);

   /* create it in kernel */
   if ( opt_v > 0) printf ( "%s -- extending logical volume %s in "
                            "VGDA of kernel\n", cmd, lv_name);
   if ( ( ret = lv_extend ( vg, vg->lv[l], lv_name)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d extending %s in kernel's VGDA\n\n", 
                cmd, ret, lv_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- storing volume group data for %s "
                            "on disk(s)\n", cmd, vg_name);
   if ( ( ret = vg_write_with_pv_and_lv ( vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d storing volume group data of %s"
                        " on disk(s)\n\n", 
                cmd, ret, vg_name);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
   if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, opt_v, vg) == 0 &&
        opt_A > 0) {
      printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
      vg_cfgbackup ( vg_name, VG_BACKUP_DIR, opt_v, vg);
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "%s -- logical volume %s successfully extended\n\n",
             cmd, lv_name);

   return 0;
}
