/*
 * Author: Heinz Mauelshagen, Germany
 *
 * Oktober-November 1997
 * May,June 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * structure of the lvmtab file (all names have NAME_LEN)
 *
 *  name of first VG\n
 *  name of second VG\n
 *     ....
 *
 */

/*
 * Changelog
 *
 *    02/12/1998 - extended parameter list of lvm_tab_vg_check_exist
 *                 with vg_t** type
 *               - implemented lvm_tab_get_free_vg_number()
 *    05/01/1998 - VG number bug fix in lvm_tab_get_free_vg_number()
 *    05/03/1998 - fixed bug with lvmtab creation return code in
 *                 lvm_tab_vg_insert()
 *               - fixed lvmtab remove bug in lvm_tab_vg_remove()
 *    06/08/1998 - added lvm_tab_create() to create an empty
 *                 lvmtab file and directory
 *               - checked for empty lvmtab file in
 *                 lvm_tab_vg_check_exist_all_vg() and lvm_tab_vg_check_exist()
 *               - added handling of empty ( only one 0 Byte) to
 *                 lvm_tab_vg_insert() and lvm_tab_vg_remove()
 *
 */



#include <liblvm.h>

/* internal prototypes */
int lvm_tab_read ( char **, int *);
int lvm_tab_write ( char *, int);


int lvm_tab_vg_insert ( char *vg_name) {
   int i = 0, j = 0, k = 0;
   int ret = 0;
   int size = 0;
   int this_ret = 0;
   int nv = 0;
   char *data = NULL;
   char *vg_names = NULL;
   char *vg_name_ptr_tmp = NULL;
   char **vg_name_ptr = NULL;
   char **vg_name_ptr_sav = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_insert -- CALLED with %s\n", vg_name);
#endif

   if ( vg_name == NULL ||
        vg_check_name ( vg_name) < 0 ||
        vg_name[0] == 0) return -LVM_EPARAM;

   if ( ( ret = lvm_tab_read ( &data, &size)) < 0) {
      if ( ret == -LVM_ELVM_TAB_READ_OPEN) ret = 0;
      size = strlen ( vg_name) + 1;
      if ( ( this_ret = lvm_tab_write ( vg_name, size)) < 0) ret = this_ret;
   } else {
      while ( i < size) {
         if ( strcmp ( &data[i], vg_name) == 0) break;
         vg_name_ptr_sav = vg_name_ptr;
         if ( ( vg_name_ptr = realloc ( vg_name_ptr,
                                       ( nv + 2) * sizeof ( char*))) == NULL) {
            fprintf ( stderr, "realloc error in %s [line %d]\n",
                              __FILE__, __LINE__);
            if ( vg_name_ptr_sav != NULL) free ( vg_name_ptr);
            free ( data);
            ret = -LVM_ELVM_TAB_VG_INSERT_REALLOC;
            goto lvm_tab_vg_insert_end;
         }
         vg_name_ptr[nv] = &data[i];
         if ( *vg_name_ptr[nv] != 0) nv++;
         i += strlen ( &data[i]) + 1;
      }

      if ( i >= size) {
         vg_name_ptr[nv++] = vg_name;
         for ( k = 0; k < nv; k++) {
            for ( i = 0; i < nv - 1; i++) {
               if ( strcmp ( vg_name_ptr[i], vg_name_ptr[i+1]) > 0) {
                  vg_name_ptr_tmp = vg_name_ptr[i];
                  vg_name_ptr[i] = vg_name_ptr[i+1];
                  vg_name_ptr[i+1] = vg_name_ptr_tmp;
               }
            }
         }

         /* Special size handling for lvmtab file with 1 zero byte */
         if ( size < 2) size = 0;
         size += strlen ( vg_name) + 1;
         if ( ( vg_names = malloc ( size)) == NULL) {
            free ( data);
            free ( vg_name_ptr);
            ret = -LVM_ELVM_TAB_VG_INSERT_REALLOC;
            goto lvm_tab_vg_insert_end;
         }

         i = 0;
         for ( j = 0; j < nv; i += strlen ( vg_name_ptr[j]) + 1, j++)
            strcpy ( &vg_names[i], vg_name_ptr[j]);

         free ( data);
         free ( vg_name_ptr);
         ret = lvm_tab_write ( vg_names, size);
         free ( vg_names);
      } else ret = -LVM_ELVM_TAB_VG_INSERT_VG_EXISTS;
   }

lvm_tab_vg_insert_end:

#ifdef DEBUG
   debug ( "lvm_tab_vg_insert -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lvm_tab_vg_remove ( char *vg_name) {
   int i = 0;
   int len = 0;
   int ret = 0;
   int size = 0;
   char command[NAME_LEN] = { 0, };
   char *data = NULL;
   char *dst = NULL;
   char *src = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_remove -- CALLED  vg_name: \"%s\"\n", vg_name);
#endif

   if ( vg_name == NULL || vg_check_name ( vg_name) < 0) return -LVM_EPARAM;

   if ( ( ret = lvm_tab_read ( &data, &size)) == 0) {
#ifdef DEBUG
   debug ( "lvm_tab_vg_remove -- lvm_tab_read o.k.\n");
#endif
      while ( i < size) {
         if ( strcmp ( &data[i], vg_name) == 0) break;
         i += strlen ( &data[i]) + 1;
      }
      if ( i >= size) ret = -LVM_ELVM_TAB_VG_REMOVE_NOT_EXISTS;
      else {
         len = strlen ( &data[i]) + 1;
         dst = &data[i];
         src = dst + len;
         if ( len >= size) {
            data[0] = 0;
            size = 1;
         } else {
            while ( src < data + size) *dst++ = *src++;
            size -= len;
         }

         if ( ( ret = lvm_tab_write ( data, size)) == 0) {
            sprintf ( command, "rm -f %s/%s%c", LVMTAB_DIR, vg_name, 0);
            ret = system ( command);
         }
      }
   }

#ifdef DEBUG
   debug ( "lvm_tab_vg_remove -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lvm_tab_vg_check_exist ( char *vg_name, vg_t **vg_ptr) {
   int i = 0;
   int p = 0;
   int pv_count = 0;
   int ret = 0;
   int size = 0;
   char *data = NULL;
   char lvmtab_path[NAME_LEN] = { 0, };
   static vg_t vg;

#ifdef DEBUG
   debug ( "lvm_tab_vg_check_exist -- CALLED with vg_name: \"%s\"\n", vg_name);
#endif

   if ( vg_name == NULL || vg_check_name ( vg_name) < 0) return -LVM_EPARAM;

   if ( ( ret = lvm_tab_read ( &data, &size)) == 0) {
      if ( size > 1) while ( i < size) {
         if ( strcmp ( &data[i], vg_name) == 0) {
            ret = TRUE;
            break;
         }
         i += strlen ( &data[i]) + 1;
      }
   }
   if ( i >= size || size < 2) ret = FALSE;

   free ( data);

   if ( ret == TRUE) {
      sprintf ( lvmtab_path, "%s/%s%c", LVMTAB_DIR, vg_name, 0);
      if ( ( ret = vg_cfgrestore ( vg_name, lvmtab_path, 0, &vg)) == 0) {
         ret = TRUE;
         for ( p = 0; p < vg.pv_cur; p++) {
            if ( strcmp ( vg_name, vg.pv[p]->vg_name) == 0) {
               pv_count++;
               if ( vg.pv[p]->version != LVM_STRUCT_VERSION)
                  return -LVM_EVG_READ_LVM_STRUCT_VERSION;
               if ( strncmp ( vg.pv[p]->system_id, EXPORTED,
                              strlen ( EXPORTED)) == 0)
                  return -LVM_EPV_READ_PV_EXPORTED;
            }
         }
   
#ifdef DEBUG
      debug ( "lvm_tab_vg_check_exist -- before vg.pv_cur check with "
               "vg.pv_cur: %lu  pv_count: %d\n", vg.pv_cur, pv_count);
#endif
         if ( vg.pv_cur != pv_count) return -LVM_EVG_CHECK_EXIST_PV_COUNT;
         if ( vg_ptr != NULL) *vg_ptr = &vg;
      }
   }

#ifdef DEBUG
   debug ( "lvm_tab_vg_check_exist -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


char **lvm_tab_vg_check_exist_all_vg ( void) {
   int i = 0, nv = 0;
   int ret = 0;
   int size = 0;
   char *data = NULL;
   char **vg_name_ptr = NULL;
   char **vg_name_ptr_sav = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_check_exist_all_vg -- CALLED\n");
#endif

   if ( ( ret = lvm_tab_read ( &data, &size)) == 0) {
      if ( size > 1) {
         while ( i < size) {
            vg_name_ptr_sav = vg_name_ptr;
            if ( ( vg_name_ptr = realloc ( vg_name_ptr,
                                           ( nv + 2) *
                                           sizeof ( char*))) == NULL) {
               fprintf ( stderr, "realloc error in %s [line %d]\n",
                                 __FILE__, __LINE__);
               if ( vg_name_ptr_sav != NULL) free ( vg_name_ptr_sav);
               ret = -LVM_ELVM_TAB_VG_CHECK_EXIST_ALL_VG_REALLOC;
               break;
            }
            vg_name_ptr[nv] = &data[i];
            i += strlen ( &data[i]) + 1;
            nv++;
         }
         vg_name_ptr[nv] = NULL;
      }
   }

#ifdef DEBUG
   debug ( "lvm_tab_vg_check_exist_all_vg -- LEAVING with ret: %d\n", ret);
#endif
   if ( ret < 0 || size < 2) {
      free ( data);
      if ( vg_name_ptr != NULL) {
         free ( vg_name_ptr);
         vg_name_ptr = NULL;
      }
   }

   return vg_name_ptr;
}


int lvm_tab_vg_read ( char *vg_name, vg_t **vg) {
   int ret = 0;
   vg_t *vg_this = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_read -- CALLED\n");
#endif

   if ( vg_name == NULL || vg_check_name ( vg_name) < 0 ||
        vg == NULL) return -LVM_EPARAM;

   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg_this)) == 0)
      *vg = vg_this;
   else
      *vg = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_read -- LEAVING\n");
#endif
   return ret;
}


int lvm_tab_vg_read_with_pv_and_lv ( char *vg_name, vg_t **vg) {
   int ret = 0;
   char lvmtab_path[NAME_LEN] = { 0, };
   static vg_t vg_this;
   
#ifdef DEBUG
   debug ( "lvm_tab_vg_read_with_pv_and_lv -- CALLED vg_name: %s\n", vg_name);
#endif

   if ( vg_name == NULL || vg_check_name ( vg_name) < 0 ||
        vg == NULL) return -LVM_EPARAM;

   sprintf ( lvmtab_path, "%s/%s%c", LVMTAB_DIR, vg_name, 0);
   if ( ( ret = vg_cfgrestore ( vg_name, lvmtab_path, 0, &vg_this)) == 0)
      *vg = &vg_this;
   else
      *vg = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_vg_read_with_pv_and_lv -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lvm_tab_lv_check_exist ( char *lv_name) {
   int l = 0;
   int ret = 0;
   vg_t *vg = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_lv_check_exit -- CALLED\n");
#endif

   if ( lv_name == NULL || lv_check_name ( lv_name) < 0) return -LVM_EPARAM;

   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_of_lv ( lv_name),
                                                 &vg)) == 0) {
      ret = FALSE;
      for ( l = 0; l < vg->lv_max; l++) {
         if ( vg->lv[l] != NULL && strcmp ( lv_name, vg->lv[l]->lv_name) == 0) {
            ret = TRUE;
            break;
         }
      }
   }

#ifdef DEBUG
   debug ( "lvm_tab_lv_check_exit -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


/* internal functions */
int lvm_tab_read ( char **data, int *size) {
   int in = -1;
   int ret = 0;
   char *buffer = NULL;
   struct stat stat_b;

#ifdef DEBUG
   debug ( "lvm_tab_read -- CALLED\n");
#endif

   if ( data == NULL || size == NULL) return -LVM_EPARAM;

   *data = NULL;
   *size = 0;

   if ( ( in = open ( LVMTAB, O_RDONLY)) == -1)
      ret = -LVM_ELVM_TAB_READ_OPEN;
   else if ( fstat ( in, &stat_b) == -1)
      ret =  -LVM_ELVM_TAB_READ_FSTAT;
   else if ( stat_b.st_size == 0)
      ret =  -LVM_ELVM_TAB_READ_SIZE;
   else if ( ( buffer = malloc ( stat_b.st_size)) == NULL)
      ret =  -LVM_ELVM_TAB_READ_MALLOC;
   else if ( read ( in, buffer, stat_b.st_size) != stat_b.st_size)
      ret =  -LVM_ELVM_TAB_READ_READ;

   if ( ret == 0) {
      *data = buffer;
      *size = stat_b.st_size; 
   } else free ( buffer);

   if ( in != -1) close ( in);

#ifdef DEBUG
   debug ( "lvm_tab_read -- LEAVING with ret: %d  data: %X  size: %d\n",
           ret, *data, *size);
#endif
   return ret;
}


int lvm_tab_create ( void) {
   int ret = 0;
   char c = 0;

   if ( ( ret = lvm_tab_write ( &c, 1)) == 0) {
      if ( mkdir ( LVMTAB_DIR, 0755) == -1) {
         unlink ( LVMTAB);
         ret = -LVM_ELVM_TAB_CREATE_LVMTAB_DIR;
      }
   } else ret = -LVM_ELVM_TAB_CREATE_LVMTAB;

   return ret;
}


int lvm_tab_write ( char *data, int size) {
   int out = -1;
   int ret = 0;

#ifdef DEBUG
   debug ( "lvm_tab_write -- CALLED\n");
#endif

   if ( data == NULL || size == 0) return -LVM_EPARAM;

   if ( ( out = open ( LVMTAB, O_WRONLY | O_CREAT | O_TRUNC, 0640)) == -1)
      ret = -LVM_ELVM_TAB_STORE_OPEN;
   else if ( write ( out, data, size) != size)
      ret =  -LVM_ELVM_TAB_STORE_WRITE;
   else if ( fchmod ( out, 0640) == -1)
      ret = -LVM_ELVM_TAB_STORE_FCHMOD;

   if ( out != -1) {
      fsync ( out);
      close ( out);
   }

#ifdef DEBUG
   debug ( "lvm_tab_write -- LEAVING\n");
#endif
   return ret;
}



int lvm_tab_get_free_vg_number ( void) {
   int i = 0;
   int vg_number = 0;
   int *vg_number_stack = NULL;
   char **vg_name_ptr = NULL;
   vg_t *vg = NULL;

#ifdef DEBUG
   debug ( "lvm_tab_get_free_vg_number -- CALLED\n");
#endif

   if ( ( vg_number_stack = malloc ( MAX_VG * sizeof ( int))) == NULL)
      return LVM_ELVM_TAB_GET_FREE_VG_NUMBER_MALLOC;

   for ( i = 0; i < MAX_VG; i++) vg_number_stack[i] = -1;

   if ( ( vg_name_ptr = lvm_tab_vg_check_exist_all_vg ()) != NULL) {
      for ( i = 0; vg_name_ptr[i] != NULL; i++) {
         if ( lvm_tab_vg_check_exist ( vg_name_ptr[i], &vg) < 0) {
            fprintf ( stderr, "Error lvm_tab_get_free_vg_number\n");
            continue;
         }
         vg_number_stack[vg->vg_number] = vg->vg_number;
      }
      for ( i = 0; i < MAX_VG; i++) {
         if ( vg_number_stack[i] == -1) {
            vg_number = i;
            break;
         }
      }
      if ( i == MAX_VG) vg_number = -1;
   } else {
      vg_number = 0;
   }

   free ( vg_number_stack);

#ifdef DEBUG
   debug ( "lvm_tab_get_free_vg_number -- LEAVING\n");
#endif
   return vg_number;
}
