/*
 * Author: Heinz Mauelshagen, Germany
 *
 * March 1997
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

#include <liblvm.h>

int lv_write ( char *pv_name, vg_t *vg, lv_t *lv, int l) {
   int pvol = -1;
   int ret  = 0;

#ifdef DEBUG
   debug ( "lv_write -- CALLED with pv_name: %s  vg->vg_name: %s  "
            "lv->lv_name: %s  index: %d\n",
            pv_name, vg->vg_name, lv->lv_name, l);
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        vg == NULL ||
        lv == NULL ||
        vg->lv_max == 0) return -LVM_EPARAM;

   if ( ( pvol = open ( pv_name, O_WRONLY)) == -1)
      ret = -LVM_ELV_STORE_OPEN;
   else if ( lseek ( pvol,
                     LVM_LV_DISK_OFFSET (
                        vg->pv[pv_get_index_by_name ( vg, pv_name)], l),
                     SEEK_SET) != LVM_LV_DISK_OFFSET ( vg->pv[0], l))
      ret = -LVM_ELV_STORE_LSEEK;
   else if ( write ( pvol, lv, sizeof ( lv_t)) != sizeof ( lv_t))
      ret = -LVM_ELV_STORE_WRITE;

   if ( pvol != -1) {
      fsync ( pvol);
      close ( pvol);
   }

#ifdef DEBUG
   debug ( "lv_write -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lv_write_all_pv ( vg_t *vg, int l) {
   int p = 0;
   int ret = 0;

#ifdef DEBUG
   debug ( "lv_write_all_pv -- CALLED with l: %d\n", l);
#endif

   if ( vg == NULL) return -LVM_EPARAM;

   for ( p = 0; p < vg->pv_cur; p++)
      if ( ( ret = lv_write ( vg->pv[p]->pv_name, vg, vg->lv[l], l)) != 0)
         break;

#ifdef DEBUG
   debug ( "lv_write_all_pv -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}


int lv_write_all_lv_of_vg ( char *pv_name, vg_t *vg) {
   int ret = 0;
   int l = 0;
   int pvol = -1;
   uint size = 0;
   lv_t *lv_this;

#ifdef DEBUG
   debug ( "lv_write_all_lv_of_vg -- CALLED  pv_name: %s\n", pv_name);
#endif

   if ( pv_name == NULL ||
        pv_check_name ( pv_name) < 0 ||
        vg == NULL ||
        vg_check_name ( vg->vg_name) < 0 ||
        vg->lv_max == 0 ||
        vg->lv_max > MAX_LV) return -LVM_EPARAM;

   size = vg->lv_max * sizeof ( lv_t);
   if ( ( lv_this = malloc ( size)) == NULL) {
      fprintf ( stderr, "malloc error in %s [line %d]\n",
                        __FILE__, __LINE__);
      return -LVM_ELV_STORE_ALL_LV_OF_VG_MALLOC;
   }
   for ( l = 0; l < vg->lv_max; l++) {
      if ( vg->lv[l] != NULL) {
#ifdef DEBUG
         debug ( "lv_write_all_lv_of_vg -- copying vg->lv[%d] \"%s\"\n",
                  l, vg->lv[l]->lv_name);
#endif
         memcpy ( &lv_this[l], vg->lv[l], sizeof ( lv_t));
      } else {
#ifdef DEBUG
         debug ( "lv_write_all_lv_of_vg -- copying EMPTY LV\n");
#endif
         memset ( &lv_this[l], 0, sizeof ( lv_t));
      }
   }

#ifdef DEBUG
   debug ( "lv_write_all_lv_of_vg -- storing %d byte of %d LVs on %s\n",
            size, size / sizeof ( lv_t), pv_name);
#endif

   if ( ( pvol = open ( pv_name, O_WRONLY)) == -1)
      ret = -LVM_ELV_STORE_ALL_LV_OF_VG_OPEN;
   else if ( lseek ( pvol, vg->pv[0]->lv_on_disk.base, SEEK_SET) != \
             vg->pv[0]->lv_on_disk.base)
      ret = -LVM_ELV_STORE_ALL_LV_OF_VG_LSEEK;
   else if ( write ( pvol, lv_this, size) != size)
      ret = -LVM_ELV_STORE_ALL_LV_OF_VG_WRITE;
   free ( lv_this);

   if ( pvol != -1) {
      fsync ( pvol);
      close ( pvol);
   }

#ifdef DEBUG
   debug ( "lv_write_all_lv_of_vg -- LEAVING with ret: %d\n", ret);
#endif
   return ret;
}
