/*
 * Author: Heinz Mauelshagen, Germany (mge@ez-darmstadt.telekom.de)
 *
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

#include <lvm_user.h>


#ifdef DEBUG
   int opt_d = 0;
#endif


int main ( int argc, char** argv) {
   int c = 0;
   int l = 0;
   int p = 0;
   int opt_A = 1;
   int opt_v = 0;
   int ret = 0;
   char *cmd = NULL;
   char *lv_name = NULL;
   char *vg_name_old = NULL;
   char *vg_name_new = NULL;
   char  vg_name_old_buf[NAME_LEN] = { 0, };
   char  vg_name_new_buf[NAME_LEN] = { 0, };
   char lv_name_buf[NAME_LEN] = { 0, };
   vg_t *vg = NULL;
   struct stat stat_buf;

#ifdef DEBUG
   char *options = "A:dh?v";
#else
   char *options = "A:h?v";
#endif

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif

         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Logical volume rename\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/?]\n"
                     "\t[-v]\n"
                     "\tOldVolumeGroupPath NewVolumeGroupPath /\n"
                     "\tOldVolumeGroupName NewVolumeGroupName\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;

         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( argc - optind != 2) {
      fprintf ( stderr, "%s -- invalid command line\n\n", cmd);
      return 1;
   }

   vg_name_old = argv[optind];
   vg_name_new = argv[optind+1];

   if ( strncmp ( vg_name_old, "/dev/", 5) != 0) {
      sprintf ( vg_name_old_buf, "/dev/%s%c", vg_name_old, 0);
      vg_name_old = vg_name_old_buf;
   }
   if ( strncmp ( vg_name_new, "/dev/", 5) != 0) {
      sprintf ( vg_name_new_buf, "/dev/%s%c", vg_name_new, 0);
      vg_name_new = vg_name_new_buf;
   }


   if ( opt_v > 0) printf ( "%s -- checking new volume group name\n", cmd);
   if ( vg_check_name ( vg_name_new) < 0) {
      fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                        cmd, vg_name_new);
      return 1;
   }

   if ( lvm_tab_vg_check_exist ( vg_name_old, &vg) != TRUE) {
      fprintf ( stderr, "%s -- volume group %s doesn't exist\n\n",
                        cmd, vg_name_old);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking for inactivity of volume group\n",
                            cmd);
   if ( vg_check_active ( vg_name_old) == TRUE) {
      fprintf ( stderr, "%s -- volume group %s must be inactive\n\n",
                        cmd, vg_name_old);
      return 1;
   }

   if ( strcmp ( vg_name_old, vg_name_new) == 0) {
      fprintf ( stderr, "%s -- volume group names must be different\n\n", cmd);
      return 1;
   }

   if ( lstat ( vg_name_new, &stat_buf) != -1) {
      lvm_filetype_show ( stat_buf.st_mode, cmd, vg_name_new);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking old volume group name\n", cmd);
   if ( vg_check_name ( vg_name_old) < 0) {
      fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                        cmd, vg_name_old);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking for existence of old "
                            "volume group\n",
                            cmd);
   if ( lvm_tab_vg_check_exist ( vg_name_old, &vg) != TRUE) {
      fprintf ( stderr, "%s -- volume group %s doesn't exist\n\n",
                        cmd, vg_name_old);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- checking for nonexistence of new "
                            "volume group\n", cmd);
   if ( lvm_tab_vg_check_exist ( vg_name_new, &vg) == TRUE) {
      fprintf ( stderr, "%s -- volume group %s already exists\n\n",
                        cmd, vg_name_new);
      return 1;
   }



   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s\n",
                            cmd, vg_name_old);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name_old, &vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d: couldn't get volume "
                        "group data of %s\n\n",
                        cmd, ret, vg_name_old);
      return 1;
   }

   /* change the volume name in all structures */
   strcpy ( vg->vg_name, vg_name_new);

   for ( p = 0; p < vg->pv_max; p++)
      if ( vg->pv[p] != NULL) strcpy ( vg->pv[p]->vg_name, vg_name_new);

   for ( l = 0; l < vg->lv_max; l++) {
      if ( vg->lv[l] != NULL) {
         strcpy ( vg->lv[l]->vg_name, vg_name_new);
         lv_name = strrchr ( vg->lv[l]->lv_name, '/');
         lv_name++;
         sprintf ( lv_name_buf, "/dev/%s/%s%c", vg_name_new, lv_name, 0);
         strcpy ( vg->lv[l]->lv_name, lv_name_buf);
      }
   }


   /* store it on disks */
   if ( opt_v > 0) printf ( "%s -- storing VGDA on disk(s)\n",
                            cmd);
   if ( ( ret = vg_write_with_pv_and_lv ( vg)) != 0) {
      fprintf ( stderr, "%s -- ERROR %d: storing volume group data of %s"
                        " on disks\n\n", 
                cmd, ret, vg_name_new);
      fprintf ( stderr, "%s -- removing new VGDA in kernel\n",
                cmd);
      if ( ( ret = vg_remove ( vg_name_new)) != 0)
         fprintf ( stderr, "%s -- ERROR %d removing VGDA for %s "
                           "from kernel\n\n", cmd, ret, vg_name_new);
      return 1;
   }


   if ( opt_v > 0) printf ( "%s -- removing lvmtab of \n"
                            "volume group %s",
                            cmd, vg_name_old);
   if ( ( ret = lvm_tab_vg_remove ( vg_name_old)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d removing %s from %s\n\n",
                        cmd, ret, vg_name_old, LVMTAB);
      return 1;
   }

   if ( opt_v) printf ( "%s -- removing any invalid special files "
                        "for %s\n", cmd, vg_name_old);
   if ( vg_remove_dir_and_group_and_nodes ( vg_name_old) < 0) {
      fprintf ( stderr, "%s -- ERROR removing volume group nodes and "
                        "directory of %s\n",
                        cmd, vg_name_old);
      return 1;
   }

   if ( opt_v > 0) printf ( "%s -- creating volume group directory /dev/%s\n",
                            cmd, vg_name_new);
   if ( ( ret = vg_create_dir_and_group_and_nodes ( vg, cmd, opt_v)) < 0) {
      if ( ret  == -LVM_EVG_CREATE_DIR_AND_GROUP_MKDIR) {
         fprintf ( stderr, "%s -- problem creating volume group "
                           "directory /dev/%s\n",
                   cmd, vg_name_new);
      } else if ( ret == -LVM_EVG_CREATE_DIR_AND_GROUP_CHMOD_DIR) {
         fprintf ( stderr, "%s -- problem changing permission "
                           "for volume group directory /dev/%s\n",
                   cmd, vg_name_new);
      } else if ( ret == -LVM_EVG_CREATE_DIR_AND_GROUP_CHMOD_GROUP) {
         fprintf ( stderr, "%s -- problem changing permission "
                           "for volume group file /dev/%s/group\n",
                   cmd, vg_name_new);
      } else {
         fprintf ( stderr, "%s -- ERROR %d creating volume group "
                           "directory /dev/%s\n",
                   cmd, ret, vg_name_new);
      }
   }


   if ( opt_v > 0) printf ( "%s -- inserting %s into %s\n",
                            cmd, vg_name_new, LVMTAB);
   if ( ( ret = lvm_tab_vg_insert ( vg_name_new)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d inserting %s into %s\n\n",
                        cmd, ret, vg_name_new, LVMTAB);
      return 1;
   }

   if ( ( ret = vg_cfgbackup ( vg_name_new, LVMTAB_DIR, cmd,
                               opt_v, vg)) == 0) {
      if ( opt_A > 0) {
         printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name_new);
         if ( ( ret = vg_cfgbackup ( vg_name_new, VG_BACKUP_DIR, cmd,
                                     opt_v, vg)) < 0) {
            fprintf ( stderr, "%s -- ERROR %d writing VG backup\n\n", cmd, ret);
            return 1;
         }
      }
   } else {
      fprintf ( stderr, "%s -- ERROR %d writing lvmtab\n\n", cmd, ret);
      return 1;
   }


   lvm_interrupt ();
   LVM_UNLOCK ( 0);


   printf ( "%s -- volume group %s successfully renamed to %s\n\n",
            cmd, vg_name_old, vg_name_new);
   return 0;
}
