/*
 * Author: Heinz Mauelshagen, Germany
 *
 * April,October 1997
 * May 1998
 *
 * LVM is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * LVM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU CC; see the file COPYING.  If not, write to
 * the Free Software Foundation, 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA. 
 *
 */

/*
 * Changelog
 *
 *    10/08/1997 - added option -a
 *    11/09/1997 - added lvmtab handling
 *    05/09/1998 - check for volume group beeing reducable
 *    05/16/1998 - added lvmtab checking
 *
 */

#include <lvm_user.h>

#ifdef DEBUG
int opt_d = 0;
#endif

int main ( int argc, char **argv) {
   int error = 0;
   int p = 0;
   int np = 0;
   int np_reduced = 0;
   int opt_a = 0;
   int opt_A = 1;
   int opt_v = 0;
   int ret = 0;
   int c = 0;
   char **pv_names = NULL;
   char *cmd = NULL;
   char *pv_name = NULL;
   char *vg_name = NULL;
   vg_t *vg = NULL;
#ifdef DEBUG
   char *options = "aA:dh?v";
#else
   char *options = "aA:h?v";
#endif
   pv_t **pv_this = NULL;

   cmd = basename ( argv[0]);

   SUSER_CHECK;
   LVMTAB_CHECK;

   while ( ( c = getopt ( argc, argv, options)) != EOF) {
      switch ( c) {
         case 'a':
            if ( opt_a > 0) {
               fprintf ( stderr, "%s -- a option yet given\n\n", cmd);
               return 1;
            }
            opt_a++;
            break;

         case 'A':
            if ( opt_A > 1) {
               fprintf ( stderr, "%s -- A option yet given\n\n", cmd);
               return 1;
            }
            if ( strcmp ( optarg, "y") == 0);
            else if ( strcmp ( optarg, "n") == 0) opt_A = 0;
            else {
               fprintf ( stderr, "%s -- invalid option argument %s\n\n",
                                 cmd, optarg);
               return 1;
            }
            break;

#ifdef DEBUG
         case 'd':
            if ( opt_d > 0) {
               fprintf ( stderr, "%s -- d option yet given\n\n", cmd);
               return 1;
            }
            opt_d++;
            break;
#endif
 
         case 'h':
         case '?':
            printf ( "\n%s\n\n%s -- Volume group reduce\n\n"
                     "Synopsis:\n"
                     "---------\n\n"
                     "%s\n"
                     "\t[-a]\n"
                     "\t[-A y/n]\n"
#ifdef DEBUG
                     "\t[-d]\n"
#endif
                     "\t[-h/-?]\n"
                     "\t[-v]\n"
                     "\tVolumeGroupName\n"
                     "\t[PhysicalVolumePath...]\n\n",
                     lvm_version, cmd, cmd);
            return 0;
            break;

         case 'v':
            if ( opt_v > 0) {
               fprintf ( stderr, "%s -- v option yet given\n\n", cmd);
               return 1;
            }
            opt_v++;
            break;
 
         default:
            fprintf ( stderr, "%s -- invalid command line option \"%c\"\n",
                      cmd, c);
            return 1;
      }
   }

   if ( optind == argc) {
      fprintf ( stderr, "%s -- please enter a volume group name"
                         " and physical volume paths\n\n", cmd);
      return 1;
   }

   vg_name = argv[optind++];

   /* valid VG name? */
   if ( opt_v > 0) printf ( "%s -- checking volume group name %s\n",
                            cmd, vg_name);
   if ( vg_check_name ( vg_name) < 0) {
	  fprintf ( stderr, "%s -- invalid volume group name %s\n\n",
                cmd, vg_name);
	  return 1;
   } 

   LVM_LOCK ( 0);
   LVM_CHECK_IOP;

   /* does VG exist? */
   if ( opt_v > 0) printf ( "%s -- checking volume group name %s\n",
                            cmd, vg_name);
   if ( lvm_tab_vg_check_exist ( vg_name, NULL) != TRUE) {
	  fprintf ( stderr, "%s -- volume group %s doesn't exist\n\n",
                cmd, vg_name);
	  return 1;
   } 

   /* is VG exist? */
   if ( opt_v > 0) printf ( "%s -- checking volume group %s activity\n",
                            cmd, vg_name);
   if ( vg_check_active ( vg_name) != TRUE) {
	  fprintf ( stderr, "%s -- volume group %s is not active\n\n",
                cmd, vg_name);
	  return 1;
   } 

   /* read complete VG */
   if ( opt_v > 0) printf ( "%s -- reading volume group data of %s "
                            "from disk(s)\n", cmd, vg_name);
   if ( ( ret = lvm_tab_vg_read_with_pv_and_lv ( vg_name, &vg)) < 0) {
      fprintf ( stderr, "%s -- ERROR %d: can't reduce;"
                        " couldn't get volume group data of %s\n\n",
                        cmd, ret, vg_name);
      return 1;
   }

   if ( ! ( vg->vg_status & VG_EXTENDABLE)) {
      fprintf ( stderr, "%s -- volume group %s can't be reduced "
                        "(see vgchange(8))\n\n",
                        cmd, vg_name);
      return 1;
   }

   if ( optind == argc) {
      if ( opt_a == 0) {
         fprintf ( stderr, "%s -- please enter physical volume paths "
                           "or option -a\n\n", cmd);
         return 1;
      } else {
         np = 0;
         pv_names = NULL;
         for ( p = 0; p < vg->pv_max; p++) {
            if ( vg->pv[p] != NULL && vg->pv[p]->pe_allocated == 0) {
               pv_names = realloc ( pv_names, sizeof ( char*) * ( np + 2));
               if ( pv_names == NULL) {
                  fprintf ( stderr, "%s -- realloc error in file %s "
                                    "[line %d]\n\n", cmd, __FILE__, __LINE__);
                  return 1;
               }
               pv_names[np] = malloc ( NAME_LEN);
               if ( pv_names[np] == NULL) {
                  fprintf ( stderr, "%s -- malloc error in file %s "
                                    "[line %d]\n\n", cmd, __FILE__, __LINE__);
                  return 1;
               }
               strcpy ( pv_names[np], vg->pv[p]->pv_name);
               np++;
               pv_names[np] = NULL;
            }
         }
         if ( np == 0) {
            fprintf ( stderr, "%s -- no empty physical volumes found\n\n", cmd);
            return 1;
         }
         if ( np == vg->pv_cur) {
            np--;
            free ( pv_names[np]);
            pv_names[np] = NULL;
         }
         if ( opt_v > 0) {
            for ( p = 0; p < np; p++) {
               printf ( "%s -- reducing %s by %s\n", cmd, vg_name, pv_names[p]);
            }
         }
         optind = 0; argc = np;
      }
   } else if ( opt_a != 0) {
      fprintf ( stderr, "%s -- you can't give option -a AND enter "
                        "physical volume paths\n\n", cmd);
      return 1;
   } else pv_names = argv;


   /* check, if PVs are all defined and in this volume group
      and reduce them in VG structures */
   if ( opt_v > 0) printf ( "%s -- checking physical volumes "
                            "against volume group %s\n",
                            cmd, vg_name);
   np_reduced = 0;
   for ( ; optind < argc; optind++) {
      pv_name = pv_names[optind];
      if ( pv_check_name ( pv_name) < 0) {
	     fprintf ( stderr, "%s -- invalid physical volume name %s\n\n",
                   cmd, pv_name);
         return 1;
      }
      for ( p = 0; p < vg->pv_max; p++) {
         if ( vg->pv[p] != NULL &&
              strcmp ( pv_name, vg->pv[p]->pv_name) == 0) {
            if ( opt_v > 0) printf ( "%s -- reducing %s in %s\n",
                                     cmd, vg->pv[p]->pv_name, vg_name);
            /* correct VG */
            if ( vg->pv[p]->lv_cur > 0) {
	           fprintf ( stderr, "%s -- can't reduce %s by "
                                     "%s with %lu logical volume",
                         cmd, vg_name, vg->pv[p]->pv_name, vg->pv[p]->lv_cur);
               if ( vg->pv[p]->lv_cur > 1) fprintf ( stderr, "s");
               fprintf ( stderr, "\n\n");
               return 1;
            }
            vg->pv_act--;
            vg->pv_cur--;

            if ( opt_v > 0) printf ( "%s -- checking if last physical volume "
                                     "in volume group %s\n", cmd, vg_name);
            if ( vg->pv_cur == 0) {
               fprintf ( stderr, "%s -- can't reduce volume group %s "
                                 "to zero physical volumes\n\n", cmd, vg_name);
               return 1;
            }

            vg->pe_total -= vg->pv[p]->pe_total;
            if ( ( pv_this = realloc ( pv_this,
                                       ( np_reduced + 2) * \
                                       sizeof ( pv_t*))) == NULL) {
               fprintf ( stderr, "%s -- malloc error in file %s [line %d]\n\n",
                                 cmd, __FILE__, __LINE__);
               return 1;
            }
            pv_this[np_reduced] = vg->pv[p];
            vg->pv[p] = NULL;
            np_reduced++;
            pv_this[np_reduced] = NULL;
            break;
         }
      }
      if ( p == vg->pv_max ) fprintf ( stderr,
                                       "%s -- %s doesn't belong to %s\n",
                                       cmd, pv_name, vg_name);
   }

   if ( np_reduced == 0) {
      fprintf ( stderr, "%s -- no physical volumes found to reduce %s by\n\n",
                        cmd, vg_name);
      return 1;
   }

   /* make PV-pointer in VG-array contiguous */
   np = 0;
   for ( p = 0; p < vg->pv_max; p++) {
      if ( vg->pv[p] == NULL) continue;
      vg->pv[np] = vg->pv[p];
      vg->pv[np]->pv_number = np + 1;
      np++;
   }
   for ( ; np < vg->pv_max; np++) vg->pv[np] = NULL;

   if ( opt_v > 0) {
      printf ( "%s -- volume group %s will be reduced by %d physical volume",
               cmd, vg_name, np_reduced);
      if ( np_reduced > 1) putchar ( 's');
      printf ( "\n");
   }

   lvm_dont_interrupt ( 0);

   /* reduce vg */
   if ( opt_v > 0) printf ( "%s -- reducing physical volumes in VGDA "
                            "in kernel\n", cmd);
   for ( p = 0; pv_this[p] != NULL; p++) {
      if ( ( ret = vg_reduce ( vg_name, pv_this[p], vg)) < 0) {
         fprintf ( stderr,
                   "%s -- ERROR %d reducing %s by %s in kernel\n",
                   cmd, ret, vg_name, pv_this[p]->pv_name);
         return 1;
      }
   }

   /* store vg on disk(s) */
   if ( opt_v > 0) printf ( "%s -- storing volume group data of %s "
                            "on disk(s)\n", cmd, vg_name);
   if ( ( ret = vg_write_with_pv_and_lv ( vg)) < 0) {
      fprintf ( stderr,
                "%s -- ERROR %d storing volume group data "
                "for %s on disk(s)\n\n",
                cmd, ret, vg_name);
      return 1;
   }

   /* setup freed PVs and store them */
   error = 0;
   for ( p = 0; pv_this[p] != NULL; p++) {
      pv_setup_for_create ( pv_this[p]->pv_name, pv_this[p],
                            pv_this[p]->pv_size);
      if ( ( ret = pv_write ( pv_this[p]->pv_name, pv_this[p])) < 0) {
         fprintf ( stderr,
                   "%s -- ERROR %d storing physical volume %s\n",
                   cmd, ret, pv_this[p]->pv_name);
         error++;
      }
   }

   if ( error > 0) fprintf ( stderr, "%s -- %d errors storing physical "
                                     "volume data for %s\n",
                                     cmd, error, vg_name);

   free ( pv_this);
   pv_this = NULL;

   if ( opt_v > 0) printf ( "%s -- changing lvmtab\n", cmd);
   if ( vg_cfgbackup ( vg_name, LVMTAB_DIR, cmd, opt_v, vg) == 0 &&
        opt_A > 0) {
      printf ( "%s -- doing automatic backup of %s\n", cmd, vg_name);
      vg_cfgbackup ( vg_name, VG_BACKUP_DIR, cmd, opt_v, vg);
   }

   lvm_interrupt ();
   LVM_UNLOCK ( 0);

   printf ( "%s -- %s ", cmd, vg_name);
   if ( error > 0) printf ( "NOT ");
   printf ( "successfully reduced\n\n");

   return 0;
}
