#!/usr/bin/env python
# -*- Mode: Python; py-indent-offset: 4 -*-

from string import *
import sys
import os
import re
import exceptions
import output
from definitions import *
import parser
import message
import string

## Some regular expressions
wrap_method = re.compile ("method ([a-zA-Z_0-9]+)")
wrap_enum = re.compile ("enum ([A-Za-z_0-9]+) *([A-Za-z_0-9]+)?")
wrap_signal = re.compile ("signal ([-A-Za-z_0-9]+) \(([-A-Za-z_0-9/: &*]+)\) ([A-Za-z_0-9]+)")
make_slot = re.compile ("slot ([0-9]+)")
make_bind = re.compile ("bind ([0-9]+)")
make_gtksignal = re.compile ("gtksignal ([0-9]+)")
make_nativesignal = re.compile ("nativesignal ([0-9]+)")
make_class = re.compile ("class (concrete|abstract) ([-A-Za-z_0-9]+) ([-A-Za-z_0-9]+) ([-A-Za-z_0-9]+)")

def process_code_directive (outfile, code_repo, indent, directive, params, line_list, namespace_stack):
    if directive == 'make':
        process_make_directive (outfile, indent, params, line_list, namespace_stack)
    elif directive == 'wrap':
        process_wrap_directive (outfile, code_repo, indent, params, line_list, namespace_stack)
    else:
        line_list.die ("unknown code generation directive: " + directive)

def process_make_directive (outfile, indent, params, line_list, namespace_stack):
    match = make_slot.search (params)
    if match:
        process_make_slot (outfile, indent, match.group (1))
        return
    match = make_bind.search (params)
    if match:
        process_make_bind (outfile, indent, match.group (1))
        return
    match = make_gtksignal.search (params)
    if match:
        process_make_gtksignal (outfile, indent, match.group (1))
        return
    match = make_nativesignal.search (params)
    if match:
        process_make_nativesignal (outfile, indent, match.group (1))
        return
    match = make_class.search (params)
    if match:
        process_make_class (outfile, indent,
                            match.group (1) == 'concrete',
                            match.group (2),
                            match.group (3),
                            match.group (4),
                            namespace_stack)
        return

    line_list.die ("Unknown make directive")
    
def argNtype (count):
    return "Arg%dType" % count

def arg_list (preceding_arg, count_int):
    retval = ""
    count = "%d" % count_int
    i = 1
    while i <= count_int:
        if preceding_arg:
            comma = ", "
        else:
            if i == 1:
                comma = ""
            else:
                comma = ", "
        retval = retval + comma + ("arg%d" % i)
        i = i + 1;

    return retval

def param_list (preceding_arg, count_int):
    retval = ""
    count = "%d" % count_int
    i = 1
    while i <= count_int:
        if preceding_arg:
            comma = ", "
        else:
            if i == 1:
                comma = ""
            else:
                comma = ", "
        retval = retval + comma + argNtype (i) + (" arg%d" % i)
        i = i + 1;

    return retval


def type_list (preceding_arg, count_int):
    retval = ""
    count = "%d" % count_int
    i = 1
    while i <= count_int:
        if preceding_arg:
            comma = ", "
        else:
            if i == 1:
                comma = ""
            else:
                comma = ", "
        retval = retval + comma + argNtype (i)
        i = i + 1;

    return retval

def template_list (preceding_arg, count_int):
    retval = ""
    count = "%d" % count_int
    i = 1
    while i <= count_int:
        if preceding_arg:
            comma = ", "
        else:
            if i == 1:
                comma = ""
            else:
                comma = ", "
        retval = retval + comma + "class " + argNtype (i)
        i = i + 1;

    return retval

def write_abstract_slot (outfile, indent, count, count_int):
    
    ### Write the SlotN base class
    
    outfile.write_to_header (("\n" + indent + "// Abstract base class for slots with %d argument(s)\n") % count_int)

    outfile.write_to_header (indent + "template <class ReturnType")
    outfile.write_to_header (template_list (1, count_int))

    outfile.write_to_header (">\n");

    outfile.write_to_header (indent + "class Slot" + count + " : public Slot\n")

    outfile.write_to_header (indent + "{\n" + indent + "protected:\n")

    outfile.write_to_header (indent + "  Slot" + count + " ()\n")

    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "  }\n")

    outfile.write_to_header (indent + "public:\n")

    outfile.write_to_header (indent + "  virtual ReturnType invoke (")

    outfile.write_to_header (param_list(0, count_int))

    outfile.write_to_header (") const = 0;\n\n")

    ## operator ()

    outfile.write_to_header (indent + "  ReturnType operator() (")

    outfile.write_to_header (param_list(0, count_int))

    outfile.write_to_header (") const\n")

    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "    return invoke (" + arg_list (0, count_int) + \
                             ");\n")
    outfile.write_to_header (indent + "  }\n")

    outfile.write_to_header (indent + "};\n");

def write_method_slot (outfile, indent, count, count_int):

    ### Write the MethodSlotN concrete class
    
    outfile.write_to_header (("\n" + indent + "// Concrete class for slots created from methods with %d argument(s)\n") % count_int)

    outfile.write_to_header (indent + "template <class MethodClassType, class ReturnType")
    outfile.write_to_header (template_list (1, count_int))

    outfile.write_to_header (">\n");

    outfile.write_to_header (indent + "class MethodSlot" + count + " : public Slot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))

    outfile.write_to_header (">\n");

    outfile.write_to_header (indent + "{\n" + indent + "private:\n")

    outfile.write_to_header (indent +
                             "  typedef ReturnType (MethodClassType::* MethodType) (")
    outfile.write_to_header (type_list(0, count_int))

    outfile.write_to_header (");\n")

    outfile.write_to_header (indent + "public:\n")
    outfile.write_to_header (indent + "  MethodSlot" + count + " (MethodClassType *object,\n")
    outfile.write_to_header (indent + "               MethodType method)\n")
    outfile.write_to_header (indent + "    : obj_(object), method_(method)\n")
    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "  }\n")

    outfile.write_to_header (indent + "  virtual ReturnType invoke (")

    outfile.write_to_header (param_list(0, count_int))

    outfile.write_to_header (") const\n")
    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "    return (obj_->*method_) (")
    outfile.write_to_header (arg_list(0, count_int))
    outfile.write_to_header (");\n");
    outfile.write_to_header (indent + "  }\n")

    # destructor
    outfile.write_to_header (indent + "protected:\n")
    outfile.write_to_header (indent + "  virtual ~MethodSlot" + count + " ()\n")
    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "  }\n")

    # members
    outfile.write_to_header (indent + "private:\n")
    outfile.write_to_header (indent + "  MethodClassType *obj_;\n")
    outfile.write_to_header (indent + "  MethodType method_;\n")

    # close MethodSlot class
    outfile.write_to_header (indent + "}; // class MethodSlot" + count + "\n");
    
def write_function_slot (outfile, indent, count, count_int):    

    ### Write the MethodSlotN concrete class
    
    outfile.write_to_header (("\n" + indent + "// Concrete class for slots created from static functions with %d argument(s)\n") % count_int)

    outfile.write_to_header (indent + "template <class ReturnType")
    outfile.write_to_header (template_list (1, count_int))

    outfile.write_to_header (">\n");

    outfile.write_to_header (indent + "class FunctionSlot" + count + " : public Slot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))

    outfile.write_to_header (">\n");

    outfile.write_to_header (indent + "{\n" + indent + "private:\n")

    outfile.write_to_header (indent +
                             "  typedef ReturnType (* FunctionType) (")
    outfile.write_to_header (type_list(0, count_int))

    outfile.write_to_header (");\n")

    outfile.write_to_header (indent + "public:\n")
    outfile.write_to_header (indent + "  FunctionSlot" + count + " (FunctionType function)\n")
    outfile.write_to_header (indent + "    : function_(function)\n")
    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "  }\n")

    outfile.write_to_header (indent + "  virtual ReturnType invoke (")

    outfile.write_to_header (param_list(0, count_int))

    outfile.write_to_header (") const\n")
    outfile.write_to_header (indent + "  {\n")
    
    outfile.write_to_header (indent + "    return (* function_) (")
    outfile.write_to_header (arg_list(0, count_int))
    outfile.write_to_header (");\n");
    outfile.write_to_header (indent + "  }\n")

    # destructor
    outfile.write_to_header (indent + "protected:\n")
    outfile.write_to_header (indent + "  virtual ~FunctionSlot" + count + " ()\n")
    outfile.write_to_header (indent + "  {\n")
    outfile.write_to_header (indent + "  }\n")

    # member variable
    outfile.write_to_header (indent + "private:\n")
    outfile.write_to_header (indent + "  FunctionType function_;\n")
    
    # close FunctionSlot class
    outfile.write_to_header (indent + "}; // class FunctionSlot" + count + "\n");

def process_make_slot (outfile, indent, count):
    count_int = atoi (count)

    write_abstract_slot (outfile, indent, count, count_int)
    write_method_slot (outfile, indent, count, count_int)
    write_function_slot (outfile, indent, count, count_int)

    ## slot() convenience function for MethodSlot
    outfile.write_to_header ("\n" + indent + "// convenience function that creates a MethodSlot\n");
    outfile.write_to_header (indent + "template <class MethodClassType, class ReturnType")
    outfile.write_to_header (template_list(1, count_int))
    outfile.write_to_header (">\n")
    outfile.write_to_header (indent + "inline Slot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))
    outfile.write_to_header ("> *\n")
    outfile.write_to_header (indent + "slot (MethodClassType *obj, ReturnType (MethodClassType::* method) (")
    outfile.write_to_header (type_list(0, count_int))
    outfile.write_to_header ("))\n");
    outfile.write_to_header (indent + "{\n")
    outfile.write_to_header (indent + "  return new MethodSlot" + count + "<MethodClassType, ReturnType")
    outfile.write_to_header (type_list(1, count_int))
    outfile.write_to_header ("> (obj, method);\n")
    outfile.write_to_header (indent + "}\n")

    ## slot() convenience function for FunctionSlot
    outfile.write_to_header ("\n" + indent + "// convenience function that creates a FunctionSlot\n");
    outfile.write_to_header (indent + "template <class ReturnType")
    outfile.write_to_header (template_list(1, count_int))
    outfile.write_to_header (">\n")
    outfile.write_to_header (indent + "inline Slot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))
    outfile.write_to_header ("> *\n")
    outfile.write_to_header (indent + "slot (ReturnType (* function) (")
    outfile.write_to_header (type_list(0, count_int))
    outfile.write_to_header ("))\n");
    outfile.write_to_header (indent + "{\n")
    outfile.write_to_header (indent + "  return new FunctionSlot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))
    outfile.write_to_header ("> (function);\n")
    outfile.write_to_header (indent + "}\n")


def process_make_bind (outfile, indent, count):
    count_int = atoi (count)
    plusone = ("%d" % (count_int+1))
    last_argtype = "Arg" + plusone + "Type"
    last_argname = "arg" + plusone
    boundslotname = "BoundSlot" + count + "_" + plusone
    targetslottype = "Slot" + count + "<ReturnType" + type_list(1, count_int) + ">"
    sourceslottype = "Slot" + plusone + \
                     "<ReturnType" + type_list(1, count_int+1) + ">"

    ## The class itself

    outfile.write_to_header("\n")
    outfile.write_to_header(indent + "// slot with " + count + " argument(s) created from a slot with " + plusone + " argument(s)\n")
    outfile.write_to_header(indent + "template <class ReturnType")
    outfile.write_to_header(template_list(1, count_int + 1))
    outfile.write_to_header(">\n")
    outfile.write_to_header(indent + "class " + boundslotname + ": public " + targetslottype + "\n")

    outfile.write_to_header(indent + "{\n")
    

    outfile.write_to_header(indent + "public:\n")
    # constructor
    outfile.write_to_header(indent + "  " + boundslotname + " (" + sourceslottype + \
                            " * slot, " + last_argtype + " " + last_argname + \
                            ")\n")
    outfile.write_to_header(indent + "    : original_slot_(slot), " + \
                            last_argname + "_(" + last_argname + ")\n")
    outfile.write_to_header(indent + "  {\n")
    outfile.write_to_header(indent + "    original_slot_->ref ();\n")
    outfile.write_to_header(indent + "    original_slot_->sink ();\n")
    outfile.write_to_header(indent + "  }\n\n")
    
    # invoke
    outfile.write_to_header(indent + "  virtual ReturnType invoke (")
    outfile.write_to_header(param_list(0, count_int))
    outfile.write_to_header(") const\n")

    outfile.write_to_header(indent + "  {\n")
    comma = None
    if count_int > 0:
        comma = ", "
    else:
        comma = ""
    outfile.write_to_header(indent + "    return original_slot_->invoke (" + \
                            arg_list(0, count_int) + \
                            comma + \
                            last_argname + "_" + ");\n")
    outfile.write_to_header(indent + "  }\n")


    outfile.write_to_header(indent + "protected:\n")
    # destructor
    outfile.write_to_header(indent + "  virtual ~" + boundslotname + " ()\n")
    outfile.write_to_header(indent + "  {\n")
    outfile.write_to_header(indent + "    original_slot_->unref ();\n")
    outfile.write_to_header(indent + "  }\n")

    # members
    outfile.write_to_header(indent + "private:\n")
    outfile.write_to_header(indent + "  " + sourceslottype + " * original_slot_;\n")
    outfile.write_to_header(indent + "  " + last_argtype + " " + last_argname + "_;\n")

    # close class
    outfile.write_to_header(indent + "}; // class " + boundslotname + "\n")
    
    ## bind() convenience function
    outfile.write_to_header ("\n" + indent + "// convenience function that creates a BoundSlot\n");
    outfile.write_to_header (indent + "template <class ReturnType")
    outfile.write_to_header (template_list(1, count_int + 1))
    outfile.write_to_header (">\n")
    outfile.write_to_header (indent + "inline Slot" + count + "<ReturnType")
    outfile.write_to_header (type_list(1, count_int))
    outfile.write_to_header ("> *\n")
    outfile.write_to_header (indent + "bind (" + sourceslottype + \
                             " * s, " + last_argtype + " " + last_argname + \
                             ")\n");
    outfile.write_to_header (indent + "{\n")
    outfile.write_to_header (indent + "  return new " + boundslotname + \
                             "<ReturnType")
    outfile.write_to_header (type_list(1, count_int+1))
    outfile.write_to_header ("> (s, " + last_argname + ");\n")
    outfile.write_to_header (indent + "}\n")

    
def process_make_gtksignal (outfile, indent, count):
    count_int = atoi (count)
    signame = "WrapSignal" + count

    ## slot() convenience function for MethodSlot
    outfile.write_to_header ("\n" + indent + "// Signal wrapping a GTK signal with " + count + " argument(s)\n");
    outfile.write_to_header (indent + "template <class ReturnType")
    outfile.write_to_header (template_list(1, count_int))
    outfile.write_to_header (">\n")

    outfile.write_to_header(indent + "class " + signame + " : public WrapSignal\n")

    outfile.write_to_header(indent + "{\n")

    # slot typedef
    outfile.write_to_header(indent + "public:\n")
    outfile.write_to_header(indent + "  typedef Slot" + count + \
                            "<ReturnType" + type_list(1, count_int) + \
                            "> SlotType;\n\n")

    # constructor
    outfile.write_to_header(indent + "  " + signame + " (const char * name, Private::SignalFunc callback)\n")
    outfile.write_to_header(indent + "  : WrapSignal (name, callback)\n")
    outfile.write_to_header(indent + "  {\n")
    outfile.write_to_header(indent + "  }\n\n")

    # destructor
    outfile.write_to_header(indent + "  ~" + signame + " ()\n")
    outfile.write_to_header(indent + "  {\n")
    outfile.write_to_header(indent + "  }\n")

    # connect
    connect = \
"""
Connection connect (Object * object,
                    SlotType * slot,
                    DestroyNotifier * alive_object = 0) const
{
  return real_connect (object, slot, alive_object);
}
"""
    
    for line in split (connect, '\n'):
        outfile.write_to_header (indent + "  " + line + "\n")

    # Private copy/assignment
    outfile.write_to_header (indent + "private:\n")
    outfile.write_to_header (indent + "  " + signame + "(const " + signame + " &);\n")
    outfile.write_to_header (indent + "  " + signame + " & operator=(const " + signame + " &);\n")
    outfile.write_to_header (indent + "}; // class " + signame + "\n")


nativesignal_template = """
  template <class ReturnType%(template_list_1)s, typename Marshal = class DefaultMarshal<ReturnType> >
  class Signal%(count)d : public NativeSignal
  {
  public:
    typedef Slot%(count)d<ReturnType%(type_list_1)s> SlotType;
    
    ReturnType emit (SignalEmitterBase * emitter_object%(param_list_1)s) const;

    Connection connect (SignalEmitterBase * emitter_object,
                        SlotType* slot,
                        DestroyNotifier * alive_object = 0) const
    {
      return real_connect (emitter_object, slot, alive_object);
    }
    
  }; // class Signal%(count)d

  // We don't want to inline emit()
  template <class ReturnType%(template_list_1)s, class Marshal>
  ReturnType Signal%(count)d<ReturnType%(type_list_1)s, Marshal>::emit (SignalEmitterBase * emitter_object%(param_list_1)s) const
  {
    Marshal m;
    Emission e (this, emitter_object);
    
    Slot * s = e.next ();
    while (s != 0)
      {
        SlotType * slot = (SlotType*)s;
        
        if (m.marshal (slot->invoke (%(arg_list_0)s)))
          break;
        
        s = e.next ();
      }
    
    return m.value ();
  }
  
  // partially specialize for void return value
  template <%(template_list_0)s%(comma_if_nonzero)sclass IgnoredMarshal>
  class Signal%(count)d<void%(comma_if_0)s%(type_list_1)s%(comma_if_nonzero)sIgnoredMarshal> : public NativeSignal
  {
  public:
    typedef Slot%(count)d<void%(type_list_1)s> SlotType;
    
    void emit (SignalEmitterBase * emitter_object%(param_list_1)s) const;

    Connection connect (SignalEmitterBase * emitter_object,
                        SlotType* slot,
                        DestroyNotifier * alive_object = 0) const
    {
      return real_connect (emitter_object, slot, alive_object);
    }
    
  }; // class Signal%(count)d (void specialization)

  // don't inline emit()
  template <%(template_list_0)s%(comma_if_nonzero)sclass IgnoredMarshal>
  void Signal%(count)d<void%(comma_if_0)s%(type_list_1)s%(comma_if_nonzero)sIgnoredMarshal>::emit (SignalEmitterBase * emitter_object%(param_list_1)s) const
  {
    Emission e (this, emitter_object);
    
    Slot * s = e.next ();
    while (s != 0)
      {
        SlotType * slot = (SlotType*)s;
        
        slot->invoke (%(arg_list_0)s);
        
        s = e.next ();
      }
  }
"""

def process_make_nativesignal (outfile, indent, count):
    count_int = atoi (count)
    comma_if_nonzero = None
    comma_if_zero = None
    if (count_int > 0):
        comma_if_nonzero = ', '
        comma_if_zero = ''
    else:
        comma_if_nonzero = ''
        comma_if_zero = ', '
    vals = {
        'count' : count_int,
        'template_list_0' : template_list (0, count_int),
        'template_list_1' : template_list (1, count_int),
        'type_list_1' : type_list (1, count_int),
        'arg_list_0' : arg_list (0, count_int),
        'arg_list_1' : arg_list (1, count_int),
        'param_list_0' : param_list (0, count_int),
        'param_list_1' : param_list (1, count_int),
        'comma_if_nonzero' : comma_if_nonzero,
        'comma_if_0' : comma_if_zero
        }

    str = nativesignal_template % vals

    outfile.write_to_header (str)

def process_make_class (outfile, indent, is_concrete, prefix, objname, funcprefix, namespace_stack):
    indent__ = indent + "  "
    indent____ = indent + "    "
    classname = objname + "Class"
    c_obj = prefix + objname
    c_class = prefix + classname
    
    outfile.write_to_header (indent + "class " + classname + "\n")
    outfile.write_to_header (indent + "{\n")
    outfile.write_to_header (indent + "public:\n")

    outfile.write_to_header (indent__ + "static GType type ()\n")
    outfile.write_to_header (indent__ + "{\n")
    outfile.write_to_header (indent____ + "return " + funcprefix + "_get_type ();\n")
    outfile.write_to_header (indent__ + "}\n")

    outfile.write_to_header (indent__ + "static GType proxy_type ();\n\n")


    outfile.write_to_header (indent__ + "static Inti::Object * " + \
                             "get_wrapper (GObject * obj);\n\n")

    outfile.write_to_header (indent__ + "static GObject * make_proxy ()\n")
    outfile.write_to_header (indent__ + "{\n")
    outfile.write_to_header (indent____ + "return (GObject*) g_type_create_instance (proxy_type ());\n")
    outfile.write_to_header (indent__ + "}\n")

    outfile.write_to_header (indent__ + "static GObjectClass * original_class ();\n")

    outfile.write_to_header (indent__ + "static void class_init (" + c_class + " * klass);\n")


    outfile.write_to_header (indent + "private:\n")
    outfile.write_to_header (indent__ + "static GType type_;\n")
    outfile.write_to_header (indent__ + "static " + c_class + " * original_class_;\n")

    outfile.write_to_header (indent__ + "// This class can't be instantiated.\n")
    outfile.write_to_header (indent__ + "friend class LookMrCompilerICanCallThesePrivateConstructors;\n")
    outfile.write_to_header (indent__ + classname + " (const " + classname + \
                             " &);\n")
    outfile.write_to_header (indent__ + classname + " & operator=(const " + \
                             classname + " & );\n")
    outfile.write_to_header (indent__ + classname + " ();\n")

    outfile.write_to_header (indent + "}; // class " + classname + "\n")


    ### Now write implementation
    qualifier = namespace_stack.qualifier () + "::" + classname + "::"

    # type_ static variable
    outfile.write_to_impl ("GType\n")
    outfile.write_to_impl (qualifier + "type_ = 0;\n")
    # original_class_ static
    outfile.write_to_impl (c_class + " *\n")
    outfile.write_to_impl (qualifier + "original_class_ = 0;\n\n")

    # write the proxy_type function
    outfile.write_to_impl ("GType\n")
    outfile.write_to_impl (qualifier + "proxy_type ()\n")
    outfile.write_to_impl ("{\n")
    outfile.write_to_impl ("  if (type_ == 0)\n")
    outfile.write_to_impl ("    {\n")
    outfile.write_to_impl ("       static const GTypeInfo info = \n")
    outfile.write_to_impl ("       {\n")
    outfile.write_to_impl ("          sizeof (" + c_class + "),\n")
    outfile.write_to_impl ("          (GBaseInitFunc) NULL,\n")
    outfile.write_to_impl ("          (GBaseFinalizeFunc) NULL,\n")
    outfile.write_to_impl ("          (GClassInitFunc) class_init,\n")
    outfile.write_to_impl ("          NULL, NULL,\n")
    outfile.write_to_impl ("          sizeof (" + c_obj + "),\n")
    outfile.write_to_impl ("          0, /* preallocs */\n")
    outfile.write_to_impl ("          (GInstanceInitFunc) NULL\n");
    outfile.write_to_impl ("       };\n\n")
    outfile.write_to_impl ("       type_ = g_type_register_static (type (),\n")
    outfile.write_to_impl ("                                      \"Inti_" + prefix + "_" + objname + "\",\n")
    outfile.write_to_impl ("                                      &info, GTypeFlags(0));\n")
    outfile.write_to_impl ("    }\n")
    outfile.write_to_impl ("  return type_;\n")
    outfile.write_to_impl ("}\n\n")

    # if we are abstract, write a Concrete class to use for get_wrapper
    if not is_concrete:
        outfile.write_to_impl ("namespace Inti\n")
        outfile.write_to_impl ("{\n")
        outfile.write_to_impl ("  namespace " + prefix + "\n")
        outfile.write_to_impl ("  {\n")
        outfile.write_to_impl ("    namespace Private\n")
        outfile.write_to_impl ("    {\n")

        outfile.write_to_impl ("      class Concrete" + objname + " : public " + objname + "\n")
        outfile.write_to_impl ("      {\n")
        outfile.write_to_impl ("      public:\n");
        outfile.write_to_impl ("        Concrete" + objname + " (" + c_obj + "* obj," + \
                               c_class + " * klass)\n")
        outfile.write_to_impl ("          : " + objname + " (obj, klass)\n")
        outfile.write_to_impl ("        {\n")
        outfile.write_to_impl ("        }\n")
        outfile.write_to_impl ("      }; // class Concrete" + objname + "\n")
        
        outfile.write_to_impl ("    } // namespace Private\n")
        outfile.write_to_impl ("  } // namespace " + prefix + "\n")
        outfile.write_to_impl ("} // namespace Inti\n")
        
    # write the get_wrapper function

    concrete_obj = None
    if is_concrete:
        concrete_obj = objname
    else:
        concrete_obj = "Private::Concrete" + objname

    outfile.write_to_impl ("Inti::Object *\n")
    outfile.write_to_impl (qualifier + "get_wrapper (GObject * obj)\n")
    outfile.write_to_impl ("{\n")
    outfile.write_to_impl ("  Inti::Object *o;\n")
    outfile.write_to_impl ("  o = new Inti::" + prefix + "::" + concrete_obj + \
                           "( (" + c_obj + " *) obj, (" + \
                           c_class + " *) G_OBJECT_GET_CLASS (obj));\n\n")
    outfile.write_to_impl ("  return o;\n")
    
    outfile.write_to_impl ("}\n\n")

    # write the original_class function
    outfile.write_to_impl ("GObjectClass *\n")
    outfile.write_to_impl (qualifier + "original_class ()\n")
    outfile.write_to_impl ("{\n")
    outfile.write_to_impl ("  if (original_class_ == 0)\n")
    outfile.write_to_impl ("    original_class_ = (" + c_class + " *) g_type_class_ref (type ());\n")
    outfile.write_to_impl ("  return (GObjectClass*) original_class_;\n")
    outfile.write_to_impl ("}\n")

###########################################################
################## Wrap
###########################################################

def process_wrap_directive (outfile, code_repo, indent, params, line_list, namespace_stack):    
    match = wrap_method.search (params)
    if match:
        process_wrap_method (outfile, code_repo, indent, match.group (1), namespace_stack)
        return
    match = wrap_enum.search (params)
    if match:
        process_wrap_enum (outfile, code_repo, indent, match.group (1), match.group (2))
        return
    match = wrap_signal.search (params)
    if match:
        process_wrap_signal (outfile, code_repo, namespace_stack,
                             indent, match.group (1),
                             split (match.group(2), '/'),
                             match.group (3))
        return

    line_list.die ("Unknown wrap directive")
    
def remove_blank_lines (lines):
    return filter (lambda s: strip(s) != "", lines)

def process_wrap_method (outfile, code_repo, indent, method_name, namespace_stack):
    tup = code_repo.get (method_name)
    if not tup:
        outfile.write_to_header ('#error "failed to wrap method %s"' % method_name)
        message.die ("Don't have code in .code files for method %s" % method_name)
        return

    decl = tup[0]
    for line in split(decl, '\n'):
        if strip(line) != '':
            outfile.write_to_header (indent + line + '\n')

    impl = tup[1]
    impl = re.sub ('<QUALIFIER>', namespace_stack.qualifier () + '::', impl)
    if tup[1]:
        for line in split(impl, '\n'):
            if strip(line) != '':
                outfile.write_to_impl (line  + '\n')

    
def process_wrap_enum (outfile, code_repo, indent, orig_name, cpp_name):
    tup = code_repo.get (orig_name)
    if not tup or not tup[1]:
        message.warn ("Don't have code in .code files for enum %s" % orig_name)
        return

    lines = split (tup[1], '\n')

    # nuke blank lines
    lines = remove_blank_lines (lines)

    if cpp_name:
        # override the default C++ name for the enum
        lines[-1] = "} " + cpp_name + ";"

    for l in lines:
        outfile.write_to_header (indent + l + '\n')

def unix_to_studly_caps (str):
    dest = ""
    saw_uscore = 1
    for c in str:
        if c == '_':
            saw_uscore = 1
        else:
            if saw_uscore:
                c = upper (c)
                saw_uscore = 0
            dest = dest + c

    return dest
            
def process_wrap_signal (outfile, code_repo, namespace_stack,
                         indent, signame, argtypes, callback):

    studly = unix_to_studly_caps (signame)
    sigtype = studly + "SignalType"
    proxytype = studly + "ProxyType"
    sigvar = signame + "_signal"
    argtypes = remove_blank_lines (argtypes)
    rettype = argtypes[0]
    argtypes = argtypes[1:]
    lessindent = indent[:-2]

    # Typedef for the signal type
    outfile.write_to_header (lessindent + "protected:\n")
    outfile.write_to_header (indent + "typedef WrapSignal" + \
                             ("%d" % (len(argtypes))) + \
                             "<" + rettype)
    for a in argtypes:
        outfile.write_to_header ("," + a)

    outfile.write_to_header ("> " + sigtype + ";\n")

    # declare signal static variable
    outfile.write_to_header (indent + "static const " + sigtype + " " + sigvar + ";\n\n")

    # define static variable
    outfile.write_to_impl ('const ' + namespace_stack.qualifier() + "::" + sigtype + "\n")
    outfile.write_to_impl (namespace_stack.qualifier() + "::" + sigvar + \
                           " (\"" + signame + "\", (Private::SignalFunc)Private::" + callback + \
                           ");\n\n")
                           
    # Connection proxy typedef and accessor
    outfile.write_to_header (lessindent + "public:\n")
    outfile.write_to_header (indent + "typedef SignalProxy<Object," +
                             sigtype + "> " + \
                             proxytype + ";\n")
    outfile.write_to_header (indent + 'const ' + proxytype + " sig_" + signame + \
                             " ()\n")
    outfile.write_to_header (indent + "{\n")
    outfile.write_to_header (indent + "  return " + proxytype + " (this, &" + \
                             sigvar + ");\n")
    outfile.write_to_header (indent + "}\n")                    


