/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include "_scm.h"



#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#include <sys/stat.h>


#include <pwd.h>

#include <sys/types.h>
#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include <signal.h>

#ifdef FD_SET

#define SELECT_TYPE fd_set
#define SELECT_SET_SIZE FD_SETSIZE

#else /* no FD_SET */

/* Define the macros to access a single-int bitmap of descriptors.  */
#define SELECT_SET_SIZE 32
#define SELECT_TYPE int
#define FD_SET(n, p) (*(p) |= (1 << (n)))
#define FD_CLR(n, p) (*(p) &= ~(1 << (n)))
#define FD_ISSET(n, p) (*(p) & (1 << (n)))
#define FD_ZERO(p) (*(p) = 0)

#endif /* no FD_SET */


extern char *ttyname ();
extern FILE *popen ();

#include <grp.h>
#include <sys/utsname.h>


 /* Only the superuser can successfully execute this call */
PROC (s_sys_chown, "%chown", 3, 0, 0, scm_sys_chown);
#ifdef __STDC__
SCM 
scm_sys_chown (SCM path, SCM owner, SCM group)
#else
SCM 
scm_sys_chown (path, owner, group)
     SCM path;
     SCM owner;
     SCM group;
#endif
{
  int val;
  ASSERT (NIMP (path) && STRINGP (path), path, ARG1, s_sys_chown);
  ASSERT (INUMP (owner), owner, ARG2, s_sys_chown);
  ASSERT (INUMP (group), group, ARG3, s_sys_chown);
  SYSCALL (val = chown (CHARS (path), INUM (owner), INUM (group)));
  return val ? BOOL_F : BOOL_T;
}


PROC (s_sys_link, "%link", 2, 0, 0, scm_sys_link);
#ifdef __STDC__
SCM 
scm_sys_link (SCM oldpath, SCM newpath)
#else
SCM 
scm_sys_link (oldpath, newpath)
     SCM oldpath;
     SCM newpath;
#endif
{
  int val;
  ASSERT (NIMP (oldpath) && STRINGP (oldpath), oldpath, ARG1, s_sys_link);
  ASSERT (NIMP (newpath) && STRINGP (newpath), newpath, ARG2, s_sys_link);
  SYSCALL (val = link (CHARS (oldpath), CHARS (newpath)));
  return val ? BOOL_F : BOOL_T;
}


PROC (s_sys_pipe, "%pipe", 0, 0, 0, scm_sys_pipe);
#ifdef __STDC__
SCM 
scm_sys_pipe (void)
#else
SCM 
scm_sys_pipe ()
#endif
{
  int fd[2], rv;
  FILE *f_rd, *f_wt;
  SCM p_rd, p_wt;
  NEWCELL (p_rd);
  NEWCELL (p_wt);
  rv = pipe (fd);
  if (rv)
    {
      ALLOW_INTS;
      return BOOL_F;
    }
  f_rd = fdopen (fd[0], "r");
  if (!f_rd)
    {
      SYSCALL (close (fd[0]));
      SYSCALL (close (fd[1]));
      ALLOW_INTS;
      return BOOL_F;
    }
  f_wt = fdopen (fd[1], "w");
  if (!f_wt)
    {
      fclose (f_rd);
      SYSCALL (close (fd[1]));
      ALLOW_INTS;
      return BOOL_F;
    }
  CAR (p_rd) = tc16_fport | scm_mode_bits ("r");
  CAR (p_wt) = tc16_fport | scm_mode_bits ("w");
  SETSTREAM (p_rd, f_rd);
  SETSTREAM (p_wt, f_wt);
  scm_add_to_port_table (p_rd);
  scm_add_to_port_table (p_wt);
  ALLOW_INTS;
  return scm_cons (p_rd, p_wt);
}


/* FIXME: pipe streams are not currently added to the scm_list of ports.
 * If pipe streams are to be kept then some things need to be changed.
 * open-pipe should also be given a exception wrapper.
 */
PROC (s_open_pipe, "open-pipe", 2, 0, 0, scm_open_pipe);
#ifdef __STDC__
SCM 
scm_open_pipe (SCM pipestr, SCM modes)
#else
SCM 
scm_open_pipe (pipestr, modes)
     SCM pipestr;
     SCM modes;
#endif
{
  FILE *f;
  register SCM z;
  ASSERT (NIMP (pipestr) && STRINGP (pipestr), pipestr, ARG1, s_open_pipe);
  ASSERT (NIMP (modes) && STRINGP (modes), modes, ARG2, s_open_pipe);
  NEWCELL (z);
  /* DEFER_INTS, SYSCALL, and ALLOW_INTS are probably paranoid here*/
  DEFER_INTS;
  scm_ignore_signals ();
  SYSCALL (f = popen (CHARS (pipestr), CHARS (modes)));
  scm_unignore_signals ();
  if (!f)
    z = BOOL_F;
  else
    {
      CAR (z) = tc16_pipe | OPN | (strchr (CHARS (modes), 'r') ? RDNG : WRTNG);
      SETSTREAM (z, f);
    }
  ALLOW_INTS;
  return z;
}


PROC (s_open_input_pipe, "open-input-pipe", 1, 0, 0, scm_open_input_pipe);
#ifdef __STDC__
SCM
scm_open_input_pipe(SCM pipestr)
#else
SCM
scm_open_input_pipe(pipestr)
     SCM pipestr;
#endif
{
  return scm_open_pipe(pipestr, scm_makfromstr("r", (sizeof "r")-1, 0));
}

PROC (s_open_output_pipe, "open-output-pipe", 1, 0, 0, scm_open_output_pipe);
#ifdef __STDC__
SCM
scm_open_output_pipe(SCM pipestr)
#else
SCM
scm_open_output_pipe(pipestr)
     SCM pipestr;
#endif
{
  return scm_open_pipe(pipestr, scm_makfromstr("w", (sizeof "w")-1, 0));
}


#ifdef __STDC__
static int
prinpipe(SCM exp, SCM port, int writing)
#else
static int
prinpipe(exp, port, writing)
     SCM exp;
     SCM port;
     int writing;
#endif
{
  scm_prinport(exp, port, s_open_output_pipe);
  return !0;
}



PROC (s_sys_getgroups, "%getgroups", 0, 0, 0, scm_sys_getgroups);
#ifdef __STDC__
SCM
scm_sys_getgroups(void)
#else
SCM
scm_sys_getgroups()
#endif
{
  SCM grps, ans;
  int ngroups = getgroups (0, NULL);
  if (!ngroups) return BOOL_F;
  NEWCELL(grps);
  DEFER_INTS;
  {
    GETGROUPS_T *groups = (gid_t *)scm_must_malloc(ngroups * sizeof(GETGROUPS_T),
					     s_sys_getgroups);
    int val = getgroups(ngroups, groups);
    if (val < 0) {
      scm_must_free((char *)groups);
      ALLOW_INTS;
      return BOOL_F;
    }
    SETCHARS(grps, groups);	/* set up grps as a GC protect */
    SETLENGTH(grps, 0L + ngroups * sizeof(GETGROUPS_T), tc7_string);
    ALLOW_INTS;
    ans = scm_make_vector(MAKINUM(ngroups), SCM_UNDEFINED);
    while (--ngroups >= 0) VELTS(ans)[ngroups] = MAKINUM(groups[ngroups]);
    SETCHARS(grps, groups);	/* to make sure grps stays around. */
    return ans;
  }
}  

/* These 2 routines are not protected against `entry' being reused
 * before access to that structure is completed
 */

PROC (s_sys_getpwuid, "%getpwuid", 0, 1, 0, scm_sys_getpwuid);
#ifdef __STDC__
SCM 
scm_sys_getpwuid (SCM user)
#else
SCM 
scm_sys_getpwuid (user)
     SCM user;
#endif
{
  SCM result;
  struct passwd *entry;
  SCM *ve;

  result = scm_make_vector (MAKINUM (7), UNSPECIFIED);
  ve = VELTS (result);
  if (UNBNDP (user) || FALSEP (user))
    SYSCALL (entry = getpwent ());
  else if (INUMP (user))
    entry = getpwuid (INUM (user));
  else
    {
      ASSERT (NIMP (user) && STRINGP (user), user, ARG1, s_sys_getpwuid);
      entry = getpwnam (CHARS (user));
    }
  if (!entry)
    return BOOL_F;
  ve[0] = makfrom0str (entry->pw_name);
  ve[1] = makfrom0str (entry->pw_passwd);
  ve[2] = scm_ulong2num ((unsigned long) entry->pw_uid);
  ve[3] = scm_ulong2num ((unsigned long) entry->pw_gid);
  ve[4] = makfrom0str (entry->pw_gecos);
  if (!entry->pw_dir)
    ve[5] = makfrom0str ("");
  else
    ve[5] = makfrom0str (entry->pw_dir);
  if (!entry->pw_shell)
    ve[6] = makfrom0str ("");
  else
    ve[6] = makfrom0str (entry->pw_shell);
  return result;
}


PROC (s_setpwent, "setpwent", 0, 1, 0, scm_setpwent);
#ifdef __STDC__
SCM 
scm_setpwent (SCM arg)
#else
SCM 
scm_setpwent (arg)
     SCM arg;
#endif
{
  if (UNBNDP (arg) || FALSEP (arg))
    endpwent ();
  else
    setpwent ();
  return UNSPECIFIED;
}


/* Combines getgrgid and getgrnam.  */
PROC (s_sys_getgrgid, "%getgrgid", 0, 1, 0, scm_sys_getgrgid);
#ifdef __STDC__
SCM 
scm_sys_getgrgid (SCM name)
#else
SCM 
scm_sys_getgrgid (name)
     SCM name;
#endif
{
  SCM result;
  struct group *entry;
  SCM *ve;
  result = scm_make_vector (MAKINUM (4), UNSPECIFIED);
  ve = VELTS (result);
  DEFER_INTS;
  if (UNBNDP (name) || (name == BOOL_F))
    SYSCALL (entry = getgrent ());
  else if (INUMP (name))
    SYSCALL (entry = getgrgid (INUM (name)));
  else
    {
      ASSERT (NIMP (name) && STRINGP (name), name, ARG1, s_sys_getgrgid);
      SYSCALL (entry = getgrnam (CHARS (name)));
    }
  ALLOW_INTS;
  if (!entry)
    return BOOL_F;
  ve[0] = makfrom0str (entry->gr_name);
  ve[1] = makfrom0str (entry->gr_passwd);
  ve[2] = scm_ulong2num ((unsigned long) entry->gr_gid);
  ve[3] = scm_makfromstrs (-1, entry->gr_mem);
  return result;
}


PROC (s_setgrent, "setgrent", 0, 1, 0, scm_setgrent);
#ifdef __STDC__
SCM 
scm_setgrent (SCM arg)
#else
SCM 
scm_setgrent (arg)
     SCM arg;
#endif
{
  if (UNBNDP (arg) || FALSEP (arg))
    endgrent ();
  else
    setgrent ();
  return UNSPECIFIED;
}

PROC (s_sys_kill, "%kill", 2, 0, 0, scm_sys_kill);
#ifdef __STDC__
SCM 
scm_sys_kill (SCM pid, SCM sig)
#else
SCM 
scm_sys_kill (pid, sig)
     SCM pid;
     SCM sig;
#endif
{
  int i;
  ASSERT (INUMP (pid), pid, ARG1, s_sys_kill);
  ASSERT (INUMP (sig), sig, ARG2, s_sys_kill);
  /* Signal values are interned in scm_init_posix().  */
  SYSCALL (i = kill ((int) INUM (pid), (int) INUM (sig)));
  return i ? BOOL_F : BOOL_T;
}


PROC (s_sys_waitpid, "%waitpid", 1, 1, 0, scm_sys_waitpid);
#ifdef __STDC__
SCM 
scm_sys_waitpid (SCM pid, SCM options)
#else
SCM 
scm_sys_waitpid (pid, options)
     SCM pid;
     SCM options;
#endif
{
  int i;
  int status;
  int ioptions;
  ASSERT (INUMP (pid), pid, ARG1, s_sys_waitpid);
  if (UNBNDP (options))
    ioptions = 0;
  else
    {
      ASSERT (INUMP (options), options, ARG2, s_sys_waitpid);
      /* Flags are interned in scm_init_posix.  */
      ioptions = INUM (options);
    }
  SYSCALL (i = waitpid (INUM (pid), &status, ioptions));
  return ((i == -1)
	  ? BOOL_F
	  : scm_cons (MAKINUM (0L + i), MAKINUM (0L + status)));
}


PROC (s_getppid, "getppid", 0, 0, 0, scm_getppid);
#ifdef __STDC__
SCM 
scm_getppid (void)
#else
SCM 
scm_getppid ()
#endif
{
  return MAKINUM (0L + getppid ());
}

PROC (s_getuid, "getuid", 0, 0, 0, scm_getuid);
#ifdef __STDC__
SCM 
scm_getuid (void)
#else
SCM 
scm_getuid ()
#endif
{
  return MAKINUM (0L + getuid ());
}

PROC (s_getgid, "getgid", 0, 0, 0, scm_getgid);
#ifdef __STDC__
SCM 
scm_getgid (void)
#else
SCM 
scm_getgid ()
#endif
{
  return MAKINUM (0L + getgid ());
}

#ifndef LACK_E_IDs
PROC (s_geteuid, "geteuid", 0, 0, 0, scm_geteuid);
#ifdef __STDC__
SCM 
scm_geteuid (void)
#else
SCM 
scm_geteuid ()
#endif
{
  return MAKINUM (0L + geteuid ());
}

PROC (s_getegid, "getegid", 0, 0, 0, scm_getegid);
#ifdef __STDC__
SCM 
scm_getegid (void)
#else
SCM 
scm_getegid ()
#endif
{
  return MAKINUM (0L + getegid ());
}
#endif


PROC (s_sys_setuid, "%setuid", 1, 0, 0, scm_sys_setuid);
#ifdef __STDC__
SCM 
scm_sys_setuid (SCM id)
#else
SCM 
scm_sys_setuid (id)
     SCM id;
#endif
{
  ASSERT (INUMP (id), id, ARG1, s_sys_setuid);
  return setuid (INUM (id)) ? BOOL_F : BOOL_T;
}

PROC (s_sys_setgid, "%setgid", 1, 0, 0, scm_sys_setgid);
#ifdef __STDC__
SCM 
scm_sys_setgid (SCM id)
#else
SCM 
scm_sys_setgid (id)
     SCM id;
#endif
{
  ASSERT (INUMP (id), id, ARG1, s_sys_setgid);
  return setgid (INUM (id)) ? BOOL_F : BOOL_T;
}

#ifndef LACK_E_IDs
PROC (s_sys_seteuid, "%seteuid", 1, 0, 0, scm_sys_seteuid);
#ifdef __STDC__
SCM 
scm_sys_seteuid (SCM id)
#else
SCM 
scm_sys_seteuid (id)
     SCM id;
#endif
{
  ASSERT (INUMP (id), id, ARG1, s_sys_seteuid);
  return seteuid (INUM (id)) ? BOOL_F : BOOL_T;
}

PROC (s_sys_setegid, "%setegid", 1, 0, 0, scm_sys_setegid);
#ifdef __STDC__
SCM 
scm_sys_setegid (SCM id)
#else
SCM 
scm_sys_setegid (id)
     SCM id;
#endif
{
  ASSERT (INUMP (id), id, ARG1, s_sys_setegid);
  return setegid (INUM (id)) ? BOOL_F : BOOL_T;
}
#endif

#ifndef ttyname
extern char * ttyname();
#endif

PROC (s_ttyname, "ttyname", 1, 0, 0, scm_ttyname);
#ifdef __STDC__
SCM 
scm_ttyname (SCM port)
#else
SCM 
scm_ttyname (port)
     SCM port;
#endif
{
  char *ans;
  int fd;
  ASSERT (NIMP (port) && OPPORTP (port), port, ARG1, s_ttyname);
  if (tc16_fport != TYP16 (port))
    return BOOL_F;
  fd = fileno (STREAM (port));
  if (fd != -1)
    SYSCALL (ans = ttyname (fd));
  /* ans could be overwritten by another call to ttyname */
  return (((fd != -1) && ans)
	  ? makfrom0str (ans)
	  : BOOL_F);
}


/* Copy exec args from an SCM vector into a new C array.  */
#ifdef __STDC__
static char **
scm_convert_exec_args (SCM args)
#else
static char **
scm_convert_exec_args (args)
     SCM args;
#endif
{
  char **execargv;
  int num_args;
  int i;
  DEFER_INTS;
  num_args = scm_ilength (args);
  execargv = (char **) 
    scm_must_malloc ((num_args + 1) * sizeof (char *), s_ttyname);
  for (i = 0; NNULLP (args); args = CDR (args), ++i)
    {
      sizet len;
      char *dst;
      char *src;
      ASSERT (NIMP (CAR (args)) && STRINGP (CAR (args)), CAR (args),
	      "wrong type in ARG", "exec arg");
      len = 1 + LENGTH (CAR (args));
      dst = (char *) scm_must_malloc ((long) len, s_ttyname);
      src = CHARS (CAR (args));
      while (len--) 
	dst[len] = src[len];
      execargv[i] = dst;
    }
  execargv[i] = 0;
  ALLOW_INTS;
  return execargv;
}

PROC (s_sys_execl, "%execl", 0, 0, 1, scm_sys_execl);
#ifdef __STDC__
SCM
scm_sys_execl (SCM args)
#else
SCM
scm_sys_execl (args)
     SCM args;
#endif
{
  char **execargv;
  SCM filename = CAR (args);
  ASSERT (NIMP (filename) && STRINGP (filename), filename, ARG1, s_sys_execl);
  args = CDR (args);
  execargv = scm_convert_exec_args (args);
  execv (CHARS (filename), execargv);
  return BOOL_F;
}

PROC (s_sys_execlp, "%execlp", 0, 0, 1, scm_sys_execlp);
#ifdef __STDC__
SCM
scm_sys_execlp (SCM args)
#else
SCM
scm_sys_execlp (args)
     SCM args;
#endif
{
  char **execargv;
  SCM filename = CAR (args);
  ASSERT (NIMP (filename) && STRINGP (filename), filename, ARG1, s_sys_execlp);
  args = CDR (args);
  execargv = scm_convert_exec_args (args);
  execvp (CHARS (filename), execargv);
  return BOOL_F;
}

/* Flushing streams etc., is not done here.  */
PROC (s_sys_fork, "%fork", 0, 0, 0, scm_sys_fork);
#ifdef __STDC__
SCM
scm_sys_fork(void)
#else
SCM
scm_sys_fork()
#endif
{
  pid_t pid;
  pid = fork ();
  if (pid == -1)
    return BOOL_F;
  else
    return MAKINUM (0L+pid);
}




#ifdef __STDC__
void
fill_select_type (SELECT_TYPE * set, SCM list)
#else
void
fill_select_type (set, list)
     SELECT_TYPE * set;
     SCM list;
#endif
{
  while (list != EOL)
    {
      if (   NIMP (CAR (list))
	  && (tc16_fport == TYP16 (CAR (list)))
	  && OPPORTP (CAR (list)))
	FD_SET (fileno (STREAM (CAR (list))), set);
      else if (INUMP (CAR (list)))
	FD_SET (INUM (CAR (list)), set);
      list = CDR (list);
    }
}

#ifdef __STDC__
SCM 
retrieve_select_type (SELECT_TYPE * set, SCM list)
#else
SCM 
retrieve_select_type (set, list)
     SELECT_TYPE * set;
     SCM list;
#endif
{
  SCM answer;
  answer = EOL;
  while (list != EOL)
    {
      if (   NIMP (CAR (list))
	  && (tc16_fport == TYP16 (CAR (list)))
	  && OPPORTP (CAR (list)))
	{
	  if (FD_ISSET (fileno (STREAM (CAR (list))), set))
	    answer = scm_cons (CAR (list), answer);
	}
      else if (INUMP (CAR (list)))
	{
	  if (FD_ISSET (INUM (CAR (list)), set))
	    answer = scm_cons (CAR (list), answer);
	}
      list = CDR (list);
    }
  return answer;
}


PROC (s_sys_select, "%select", 5, 0, 0, scm_sys_select);
#ifdef __STDC__
SCM
scm_sys_select (SCM reads, SCM writes, SCM excepts, SCM secs, SCM msecs)
#else
SCM
scm_sys_select (reads, writes, excepts, secs, msecs)
     SCM reads;
     SCM writes;
     SCM excepts;
     SCM secs;
     SCM msecs;
#endif
{
#ifdef HAVE_SELECT
  int ret;
  struct timeval timeout;
  struct timeval * time_p;
  SELECT_TYPE read_set;
  SELECT_TYPE write_set;
  SELECT_TYPE except_set;
  SCM answer;
  int sreturn;

  ASSERT (-1 < scm_ilength (reads), reads, ARG1, s_sys_select);
  ASSERT (-1 < scm_ilength (writes), reads, ARG1, s_sys_select);
  ASSERT (-1 < scm_ilength (excepts), reads, ARG1, s_sys_select);
  ASSERT (INUMP (secs), secs, ARG4, s_sys_select);
  ASSERT (INUMP (msecs), msecs, ARG5, s_sys_select);

  FD_ZERO (&read_set);
  FD_ZERO (&write_set);
  FD_ZERO (&except_set);

  fill_select_type (&read_set, reads);
  fill_select_type (&write_set, writes);
  fill_select_type (&except_set, excepts);

  if (INUM (secs) || INUM (msecs))
    {
      timeout.tv_sec = INUM (secs);
      timeout.tv_usec = 1000 * INUM (msecs);
      time_p = &timeout;
    }
  else
    time_p = 0;

  DEFER_INTS;
  sreturn = select (SELECT_SET_SIZE,
		    &read_set, &write_set, &except_set, time_p);
  ALLOW_INTS;
  if (sreturn < 0)
    return MAKINUM (sreturn);
  else
    return scm_listify (retrieve_select_type (&read_set, reads),
			retrieve_select_type (&write_set, writes),
			retrieve_select_type (&except_set, excepts),
			SCM_UNDEFINED);
#else
  return BOOL_F;
#endif
}


PROC (s_sys_uname, "%uname", 0, 0, 0, scm_sys_uname);
#ifdef __STDC__
SCM 
scm_sys_uname (void)
#else
SCM 
scm_sys_uname ()
#endif
{
#ifdef HAVE_UNAME
  struct utsname buf;
  SCM ans = scm_make_vector(MAKINUM(5), UNSPECIFIED);
  SCM *ve = VELTS (ans);
  if (uname (&buf))
    return BOOL_F;
  ve[0] = makfrom0str (buf.sysname);
  ve[1] = makfrom0str (buf.nodename);
  ve[2] = makfrom0str (buf.release);
  ve[3] = makfrom0str (buf.version);
  ve[4] = makfrom0str (buf.machine);
/* 
  FIXME
  ve[5] = makfrom0str (buf.domainname);
*/
  return ans;
#else
  return BOOL_F;
#endif
}

extern char ** environ;
PROC (s_environ, "environ", 0, 1, 0, scm_environ);
#ifdef __STDC__
SCM
scm_environ (SCM env)
#else
SCM
scm_environ (env)
     SCM env;
#endif
{
  if (UNBNDP (env))
    return scm_makfromstrs (-1, environ);
  else
    {
      int num_strings;
      char **new_environ;
      int i = 0;
      ASSERT (NIMP (env) && CONSP (env), env, ARG1, s_environ);
      num_strings = scm_ilength (env);
      new_environ = (char **) scm_must_malloc ((num_strings + 1)
					       * sizeof (char *),
					       s_environ);
      while (NNULLP (env))
	{
	  int len;
	  char *src;
	  ASSERT (NIMP (CAR (env)) && STRINGP (CAR (env)), env, ARG1,
		  s_environ);
	  len = 1 + LENGTH (CAR (env));
	  new_environ[i] = scm_must_malloc ((long) len, s_environ);
	  src = CHARS (CAR (env));
	  while (len--) 
	    new_environ[i][len] = src[len];
	  env = CDR (env);
	  i++;
	}
      new_environ[i] = 0;
      /* Free the old environment, except when called for the first
       * time.
       */
      {
	char **ep;
	static int first = 1;
	if (!first)
	  {
	    for (ep = environ; *ep != NULL; ep++)
	      scm_must_free (*ep);
	    scm_must_free ((char *) environ);
	  }
	first = 0;
      }
      environ = new_environ;
      return UNSPECIFIED;
    }
}


#ifdef __STDC__
void 
scm_init_posix (void)
#else
void 
scm_init_posix ()
#endif
{
  scm_add_feature ("posix");
#ifdef WAIT_ANY
  scm_sysintern ("WAIT_ANY", MAKINUM (WAIT_ANY));
#endif
#ifdef WAIT_MYPGRP
  scm_sysintern ("WAIT_MYPGRP", MAKINUM (WAIT_MYPGRP));
#endif
#ifdef WNOHANG
  scm_sysintern ("WNOHANG", MAKINUM (WNOHANG));
#endif
#ifdef WUNTRACED
  scm_sysintern ("WUNTRACED", MAKINUM (WUNTRACED));
#endif
#ifdef SIGHUP
  scm_sysintern ("SIGHUP", MAKINUM (SIGHUP));
#endif
#ifdef SIGINT
  scm_sysintern ("SIGINT", MAKINUM (SIGINT));
#endif
#ifdef SIGQUIT
  scm_sysintern ("SIGQUIT", MAKINUM (SIGQUIT));
#endif
#ifdef SIGILL
  scm_sysintern ("SIGILL", MAKINUM (SIGILL));
#endif
#ifdef SIGTRAP
  scm_sysintern ("SIGTRAP", MAKINUM (SIGTRAP));
#endif
#ifdef SIGABRT
  scm_sysintern ("SIGABRT", MAKINUM (SIGABRT));
#endif
#ifdef SIGIOT
  scm_sysintern ("SIGIOT", MAKINUM (SIGIOT));
#endif
#ifdef SIGBUS
  scm_sysintern ("SIGBUS", MAKINUM (SIGBUS));
#endif
#ifdef SIGFPE
  scm_sysintern ("SIGFPE", MAKINUM (SIGFPE));
#endif
#ifdef SIGKILL
  scm_sysintern ("SIGKILL", MAKINUM (SIGKILL));
#endif
#ifdef SIGUSR1
  scm_sysintern ("SIGUSR1", MAKINUM (SIGUSR1));
#endif
#ifdef SIGSEGV
  scm_sysintern ("SIGSEGV", MAKINUM (SIGSEGV));
#endif
#ifdef SIGUSR2
  scm_sysintern ("SIGUSR2", MAKINUM (SIGUSR2));
#endif
#ifdef SIGPIPE
  scm_sysintern ("SIGPIPE", MAKINUM (SIGPIPE));
#endif
#ifdef SIGALRM
  scm_sysintern ("SIGALRM", MAKINUM (SIGALRM));
#endif
#ifdef SIGTERM
  scm_sysintern ("SIGTERM", MAKINUM (SIGTERM));
#endif
#ifdef SIGSTKFLT
  scm_sysintern ("SIGSTKFLT", MAKINUM (SIGSTKFLT));
#endif
#ifdef SIGCHLD
  scm_sysintern ("SIGCHLD", MAKINUM (SIGCHLD));
#endif
#ifdef SIGCONT
  scm_sysintern ("SIGCONT", MAKINUM (SIGCONT));
#endif
#ifdef SIGSTOP
  scm_sysintern ("SIGSTOP", MAKINUM (SIGSTOP));
#endif
#ifdef SIGTSTP
  scm_sysintern ("SIGTSTP", MAKINUM (SIGTSTP));
#endif
#ifdef SIGTTIN
  scm_sysintern ("SIGTTIN", MAKINUM (SIGTTIN));
#endif
#ifdef SIGTTOU
  scm_sysintern ("SIGTTOU", MAKINUM (SIGTTOU));
#endif
#ifdef SIGIO
  scm_sysintern ("SIGIO", MAKINUM (SIGIO));
#endif
#ifdef SIGPOLL
  scm_sysintern ("SIGPOLL", MAKINUM (SIGPOLL));
#endif
#ifdef SIGURG
  scm_sysintern ("SIGURG", MAKINUM (SIGURG));
#endif
#ifdef SIGXCPU
  scm_sysintern ("SIGXCPU", MAKINUM (SIGXCPU));
#endif
#ifdef SIGXFSZ
  scm_sysintern ("SIGXFSZ", MAKINUM (SIGXFSZ));
#endif
#ifdef SIGVTALRM
  scm_sysintern ("SIGVTALRM", MAKINUM (SIGVTALRM));
#endif
#ifdef SIGPROF
  scm_sysintern ("SIGPROF", MAKINUM (SIGPROF));
#endif
#ifdef SIGWINCH
  scm_sysintern ("SIGWINCH", MAKINUM (SIGWINCH));
#endif
#ifdef SIGLOST
  scm_sysintern ("SIGLOST", MAKINUM (SIGLOST));
#endif
#ifdef SIGPWR
  scm_sysintern ("SIGPWR", MAKINUM (SIGPWR));
#endif
#include "posix.x"
}
