/* Generate the Closed-Date field for an existing GNATS database.
   Copyright (C) 1993, 1995 Free Software Foundation, Inc.
   Contributed by Rick Macdonald (rickm@vsl.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnats-dirs.h"

#include <utime.h>

/* The name this program was run with.  */
char *program_name;

/* If 1, we're running in test mode and PR files are _not_ actually updated.  */
int test_mode = 0;

/* If 1, we're running the daemon.  */
int is_daemon = 0;

/* If 1, we're running a network client.  */
int is_client = 0;

/* Where the list of categories should come from.  */
char *catfile = (char *)NULL;

/* Where the results of query-pr should go.  */
FILE *outfile = stdout;

struct option long_options[] =
{
  {"directory", 1, NULL, 'd'},
  {"catfile", 1, NULL, 'c'},
  {"outfile", 1, NULL, 'o'},
  {"help", 0, NULL, 'h'},
  {"version", 0, NULL, 'V'},
  {"test", 0, NULL, 't'},
  {NULL, 0, NULL, 0}
};

typedef struct category_list
{
  Category *c;
  struct category_list *next;
} Categories;

typedef struct entry
{
  unsigned int number;
  char *string;
} Entry;

void usage (), version ();

static int
get_closed ()
{
     char *final1, *final2, str[133], *from_start, *to_start, from[32], to[32];
     char *p, *c, *when_start, when[133], *new_audit, *copy_ptr;
     int  len, from_len, to_len, closed_date_set = 0, changed_separator;

     p = pr[AUDIT_TRAIL].value;

     if (strlen (p) == 0) {
          /* No Audit-Trail at all; initialize Closed-Date */
          xfree (pr[CLOSED_DATE].value);
          pr[CLOSED_DATE].value = strdup ("");
          return (1);
     }

     new_audit = xmalloc (strlen (p) * 2);
     new_audit[0] = '\0';
     copy_ptr = p;

     /* Skip ahead to the last time it was closed.  */
     final2 = p;
     do {
          final1 = strstr (final2, "State-Changed-From-To:");
          if (final1) {
               final1 += 22;
               while (isspace (final1[0]))
                    final1++;
               from_start = final1;
               while (final1[0] != '-')
                    final1++;
               from_len = final1 - from_start;
               changed_separator = 0;
               if (final1[1] == '>') {
                    final1 += 2;
               } else {
                    /* Change - to -> here */
                    final1++;
                    strncat (new_audit, copy_ptr, final1 - copy_ptr);
                    strcat  (new_audit, ">");
                    copy_ptr = final1;
                    changed_separator = 1;
               }
               to_start = final1;
               while (!isspace (final1[0]))
                    final1++;
               to_len = final1 - to_start;
               strncpy (from, from_start, from_len);
               from[from_len] = '\0';
               strncpy (to, to_start, to_len);
               to[to_len] = '\0';
               final2 = final1;

               if (test_mode && changed_separator) {
                    printf ("\t%s->%s\n", from, to);
               }
               
               if (strcmp (get_state_type (from),
                           get_state_type (to  )) != 0) {
                    /* The state type has changed */
                    if (check_state_type (from, "closed")) {
                         /* No longer closed; clear the Closed-Date field */
                         xfree (pr[CLOSED_DATE].value);
                         pr[CLOSED_DATE].value = strdup ("");
                    } else if (check_state_type (to, "closed")) {
                         /* Now closed; set the Closed-Date field */
                         /* Grab the When date str */
                         final1 = strstr (final2, "State-Changed-When:");
                         if (final1) {
                              final1 += 19;
                              while (isspace (final1[0]))
                                   final1++;
                              when_start = final1;
                              c = strchr (final1, '\n');
                              final2 = c;
                              while (isspace (c[0]))
                                   c--;
                              c++;
                              len = c - when_start;
                              (void) strncpy (when, when_start, len);
                              when[len] = '\0';
                              xfree (pr[CLOSED_DATE].value);
                              pr[CLOSED_DATE].value = strdup (when);
                         }
                         final1 = final2; /* Just to keep the while loop going */
                    }
                    if (test_mode) {
                         printf("\t%s/%s Closed-Date: %s\n", pr[NUMBER].value,
                                pr[CATEGORY].value, pr[CLOSED_DATE].value);
                    }
                    closed_date_set = 1;
               }
          }
     } while (final1);

     strcat (new_audit, copy_ptr);
     xfree  (pr[AUDIT_TRAIL].value);
     pr[AUDIT_TRAIL].value = new_audit;
     
     if (!closed_date_set) {
          /* No state changes at all; initialize Closed-Date */
          xfree (pr[CLOSED_DATE].value);
          pr[CLOSED_DATE].value = strdup ("");
     }
     
     return (1);
}
/*  Write a file out to FILENAME with all of the GNATS info in it.  If
    FORCE is false, then make sure the file doesn't already exist.  */
static void
create_report (filename, force)
     char *filename;
     int force;
{
  FILE *outfile;
  struct stat buf;

  /* Quick check - if file is stat-able, then this is not a good sign.  */
  if (force == 0 && stat (filename, &buf) == 0)
    punt (1, "The file for a new PR already exists, couldn't write it.\n\
The filename for it is %s.", filename);

  block_signals ();

  /* Now build the file.  */
  outfile = fopen (filename, "w+");
  if (outfile == NULL)
    punt (1, "Can't write the PR to %s.", filename);

  write_header (outfile, NUM_HEADER_ITEMS);
  fprintf (outfile, "\n");
  write_pr (outfile, NUM_PR_ITEMS);

  fclose (outfile);
  unblock_signals ();
}

/* For a given category C, go into its directory and either emit an
   index entry for each file in it, or swallow its index entry so we
   can sort it later.  */
void
do_category (c)
     char *c;
{
  register DIR *d;
  register struct dirent *next;
  struct stat    stbuf;
  struct utimbuf utbuf;
  FILE *fp;
  int len = strlen (gnats_root) + 1 + strlen (c) + 2;
  /* Allocate for a PR with 8 digits.  */
  char *path = (char *) xmalloc (len + 9);
  char *p;

  errno = 0;
  if (chdir (gnats_root) < 0)
    {
      fprintf (stderr, "%s: can't read the database directory %s: %s\n",
	       program_name, gnats_root, strerror (errno));
      exit (3);
    }

  if (c == NULL || *c == '\0')
    {
      xfree (path);
      return;
    }

  errno = 0;
  d = opendir (c);
  if (! d)
    {
      fprintf (stderr, "%s: can't open the category directory %s/%s: %s\n",
	       program_name, gnats_root, c, strerror (errno));
      xfree (path);
      return;
    }

  sprintf (path, "%s/%s/", gnats_root, c);

  /* Process each file in the directory; ignore files that have periods
     in their names; either they're the . and .. dirs, or they're a
     lock file (1234.lock).  */
  while ((next = readdir (d)))
    if (strchr (next->d_name, '.') == NULL)
      {
	p = path + len - 1;
	strcat (p, next->d_name);

	fp = fopen (path, "r");
	if (fp == (FILE *) NULL)
	  {
	    fprintf (stderr, "%s: couldn't read pr %s: %s\n",
		     program_name, path, strerror (errno));

	    *p = '\0';
	    continue;
	  }

	if (read_header (fp) < 0)
	  {
	    fprintf (stderr, "%s: error reading pr %s\n", program_name, path);
	    *p = '\0';
	    continue;
	  }

	read_pr (fp, 0);
	fclose (fp);

        get_closed ();

        printf("%s/%s Closed-Date: %s\n", pr[NUMBER].value, pr[CATEGORY].value,
               pr[CLOSED_DATE].value);

        if (!test_mode) {
             /* Reset the file mtime after writing it out */
             stat  (path, &stbuf);
             create_report (path, 1);
             utbuf.actime = utbuf.modtime = stbuf.st_mtime;
             utime (path, &utbuf);
        }
        
        *p = '\0';
      }

  closedir (d);
  xfree (path);
}

/* Get the next entry in the categories file.  */
Categories *
next_category (fp)
     FILE *fp;
{
  char *start, *end, *b;
  char *buf;
  Categories *p;
  Category *c;
  static int lineno = 0;

  if (feof (fp))
    return (Categories *) NULL;

  buf = (char *) xmalloc (STR_MAX);
  p = (Categories *) xmalloc (sizeof (Categories));
  c = p->c = (Category *) xmalloc (sizeof (Category));
  p->next = NULL;

  b = buf;

  while (1)
    {
      do
	{
	  if (fgets (buf, STR_MAX, fp) == NULL)
	    goto no_entry;
	  lineno++;
	}
      while (buf[0] == '#' || buf[0] == '\n' || buf[0] == ' ');

      start = b;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->key = start;

      start = end + 1;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->fullname = start;

      start = end + 1;
      end = strchr (start, ':');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->person = start;

      start = end + 1;
      end = strchr (start, '\n');
      if (end == NULL)
	goto retry;
      *end = '\0';
      c->notify = start;

      return p;
retry:
      fprintf (stderr, "%s: malformed line %d\n", program_name, lineno);
    }

no_entry:
  xfree ((char *)p->c);
  xfree ((char *)p);
  xfree (buf);
  return (Categories *) NULL;
}

/* Return a linked list of categories.  */
Categories *
get_categories ()
{
  char *path = (char *) xmalloc (PATH_MAX);
  FILE *fp;
  Categories *cat_list = (Categories *) NULL;
  Categories *cat_list_end = (Categories *) NULL;
  Categories *c;

  if (! catfile)
    sprintf (path, "%s/gnats-adm/%s", gnats_root, CATEGORIES);
  else
    path = catfile;

  errno = 0;
  fp = fopen (path, "r");
  if (fp == (FILE *) NULL)
    {
      fprintf (stderr, "%s: couldn't read the categories file %s: %s\n",
	       program_name, path, strerror (errno));
      exit (3);
    }

  while ((c = next_category (fp)))
    {
      if (cat_list == NULL)
	cat_list = cat_list_end = c;
      else
	{
	  cat_list_end->next = c;
	  cat_list_end = c;
	}

      c->next = NULL;
    }

  fclose (fp);
  xfree (path);

  return cat_list;
}
  
int
main (argc, argv)
     int argc;
     char **argv;
{
  int optc, i;
  Categories *clist, *c;

  program_name = (char *) basename (argv[0]);

  while ((optc = getopt_long (argc, argv, "o:c:htd:V",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'o':
	  outfile = fopen (optarg, "w+");
	  if (outfile == (FILE *) NULL)
	    {
	      fprintf (stderr, "%s: can't write to %s: %s\n", program_name,
		       optarg, strerror (errno));
	      exit (3);
	    }
	  break;

	case 'c':
	  catfile = (char *) strdup (optarg);
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;

	case 'h':
	  usage (0);
	  break;

	case 't':
	  test_mode = 1;
	  break;

	default:
	  usage (1);
	}
    }

  init_gnats (program_name);
  umask (022);

  /* lock the whole thing. */
  lock_gnats ();

  clist = get_categories ();

  for (c = clist ; c ; c = c->next)
    do_category (c->c->key);

  fclose (outfile);
  
  /* unlock the whole thing. */
  unlock_gnats ();

  exit (0);
}

void
usage (s)
     int s;
{
  fprintf (stderr, "\
Usage: %s [-htV] [-d directory] [-o outfile] [-c filename ]\n\
	  [--directory=directory] [--catfile=filename]\n\
          [--outfile=filename] [--version] [--help] [--test]\n\
          The --test option reports the changes that it would make\n\
          but doesn't actually change the files.\n",
	   program_name);
  exit (s);
}

void
version ()
{
  printf ("gen-closed-date %s\n", version_string);
}
