/* Miscellaneous tools common to all types of queries.
   Copyright (C) 1994, 95, 96, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "query.h"
#include "pcodes.h"

char *
disbar (str)
     char *str;
{
  char *q;
  
  while ((q = strchr (str, '|')))
    *q = '!';

  return str;
}

/* Return the numeric equivalent of this state; 0 if not available. */
int
enum_numeric (text, field)
     char *text;
     int   field;
{
  char *values;
  char *idx, *last_idx;
  int count, slen, llen;

  /* Init everybody and perform trivial checks. */
  count = 0;
  if ((text == NULL) || (text[0] == '\0'))
    return 0;
  /* else */
  values = pr[field].enum_values; /* gets us a '|'-separated list. */
  if ((values == NULL) || (values[0] == '\0'))
    return 0; /* don't enforce format of values here */
  /* else */
  slen = strlen (text);
  llen = strlen (values);

  for (last_idx = idx = values; idx != NULL; idx = strchr (idx, '|'))
    {
      /* Inch past the bar and its trailing space; but don't do this
         if we're still on the first value. */
      if (count > 0)
        idx += 2;

      /* Return 0 if match impossible. */
      llen -= (idx - last_idx);
      if (slen > llen)
        return 0;

      if ((strncmp (idx, text, slen) == 0)
          && ((idx[slen] == ' ') || (idx[slen] == '\0')))
        return (count + 1); /* no o-b-o-e playing here! */
      else
        count++;

      last_idx = idx;
    }

  return 0;
}

int
sql_types (p, type)
     char *p;
     Sql_Types type;
{
  switch (type)
    {
    case Severity:
      return enum_numeric (p, SEVERITY);
      break;
    case Priority:
      return enum_numeric (p, PRIORITY);
      break;
    case State:
      return enum_numeric (p, STATE);
      break;
    case Class:
      return enum_numeric (p, CLASS);
      break;
    }

  return 0;
}

char *
sql_time (s)
     char *s;
{
  time_t t;
  struct tm *ar_time;
  /* Note: we deliberately use 18 here, since that is what the width of
     this time string will end up looking like.  In the case where we
     use things relative to timezone and such, it'd grow---but not here.  */
  char *buf = (char *) xmalloc (18);

  t = get_date (s, NULL);
  ar_time = (struct tm *) localtime (&t);
  strftime (buf, 18, "%Y-%m-%d %H:%M", ar_time);

  return buf;
}

char *
make_path (c, n)
     char *c, *n;
{
  char *path = (char *) xmalloc (PATH_MAX);

  sprintf (path, "%s/%s/%s", gnats_root, c, n);
  return path;
}

int
numeric (p)
     char *p;
{
  int i;
  int l = strlen(p);
  for (i=0; i<l; i++)
    {
      if (!isdigit (p[i]))
	return 0;
    }
  return 1;
}

static char case_fold[256];

/* Return 0 if matched, 1 otherwise. */
int
regcmp (pat, match)
     char *pat, *match;
{
  struct re_pattern_buffer buf;
  union {
    const char *c;
    int i;
  } r;

  memset ((void *) &buf, 0, sizeof (buf));

  if (case_fold[1] == 0)
    {
      int i;
      for (i=0; i<256; i++)
	case_fold[i] = tolower(i);
    }
  buf.translate = case_fold;
#ifdef USE_RX
  buf.syntax_parens = (char *)1;
#endif
  
  r.c = re_compile_pattern (pat, strlen (pat), &buf);
  if (r.c)
    {
      fprintf (stderr, "%s: re_compile_pattern: %s\n", program_name, r.c);
      exit (2);
    }
  r.i = re_match (&buf, match, strlen (match), 0, 0);
  buf.translate = NULL;
  regfree (&buf);
  switch (r.i)
    {
    case -2:
      fprintf (stderr, "%s: warning: re_match died on %s\n",
	       program_name, pr[NUMBER].value);
      /*FALLTHRU*/
    case -1:
      return 1;
    default:
      return 0;
    }
}
      
/* Return 0 if found, 1 otherwise. */
int
regfind (pat, match)
     char *pat, *match;
{
  struct re_pattern_buffer buf;
  union {
    const char *c;
    int i;
  } r;

  memset ((void *) &buf, 0, sizeof (buf));

  if (case_fold[1] == 0)
    {
      int i;
      for (i=0; i<256; i++)
	case_fold[i] = tolower(i);
    }
  buf.translate = case_fold;
  buf.fastmap = xmalloc (256);
#ifdef USE_RX
  buf.syntax_parens = (char *)1;
#endif
  
  r.c = re_compile_pattern (pat, strlen (pat), &buf);
  if (r.c)
    {
      fprintf (stderr, "%s: re_compile_pattern: %s\n", program_name, r.c);
      exit (2);
    }
  r.i = strlen (match);
  r.i = re_search (&buf, match, r.i, 0, r.i, 0);
  buf.translate = NULL;
  regfree (&buf);
  switch (r.i)
    {
    case -2:
      fprintf (stderr, "%s: warning: re_search died on %s\n",
	       program_name, pr[NUMBER].value);
      /*FALLTHRU*/
    case -1:
      return 1;
    default:
      return 0;
    }
}

char *
get_category (p)
     char *p;
{
  FILE *fp;
  char *path, *category = NULL;

  if (index_chain)
    {
      Index *j;
      for (j = index_chain ; j ; j = j->next)
	if (strcmp (p, j->number) == 0)
	  {
	    category = j->category;
	    break;
	  }
    }
  else
    {
      fp = open_index ();
      if (fp == (FILE *)NULL)
	return NULL;
      category = find_pr_category (fp, p);
      close_index (fp);
    }

  if (category == NULL)
    return NULL;

  path = (char *) xmalloc (PATH_MAX);
  sprintf (path, "%s/%s/%s", gnats_root, category, p);

  return path;
}

/* Should we try to convert the date given to us?  */
#define NOCONVERT	0
#define DOCONVERT	1

int
date_compare (tst, sdate, type, convert_p)
     time_t tst;
     char *sdate;
     int type;
     int convert_p;
{
  time_t val;

  /* If it's still a flat date string, we need to use get_date instead of just
     doing an atol on the pre-converted value.  We assume that get_date can
     handle the value, since we should be getting one that's already made
     it through the system.  */
  if (convert_p)
    val = get_date (sdate, NULL);
  else
    val = atol (sdate);

  /* 0 means the date is unset, so nothing should match it */
  if (! val)
    return 1;

  if (type == 0)
    return val > tst; /* ! val < tst */
  else if (type == 1)
    return val < tst; /* ! val > tst */
  else
    abort (); /* Should never happen.  */
}

int
pr_matches (s, i)
     Index *s;
     Index *i;
{
  if (skip_closed && check_state_type (i->state, "closed"))
    return 0;
  if (!s || !searching) return 1;
  return (!s->category || (regcmp (s->category, i->category) == 0))
    && (!s->submitter || (regcmp (s->submitter, i->submitter) == 0))
    && (!s->responsible || (regcmp (s->responsible, i->responsible) == 0))
    && (!s->state || (regcmp (s->state, i->state) == 0))
    && (!s->confidential || (regcmp (s->confidential, i->confidential) == 0))
#ifdef GNATS_RELEASE_BASED
    /* We can check this here cuz the date required is included in the index.  */
    && (!required_before || (date_compare (required_before, i->date_required, 0, NOCONVERT) == 0))
    && (!required_after || (date_compare (required_after, i->date_required, 1, NOCONVERT) == 0))
#endif
    && (!s->severity || (regcmp (s->severity, i->severity) == 0))
    && (!s->priority || (regcmp (s->priority, i->priority) == 0));
}

int
check_text ()
{
  PR_Name i;

  if (text_search)
    {
      for (i = NUMBER; i < NUM_PR_ITEMS; i++)
	{
	  if (pr[i].datatype != Text)
	    continue;
	  
	  if (pr[i].value && regfind (text_search, pr[i].value) == 0)
	    goto part2;
	}
      return 0;
    }
 part2:
  if (m_text_search)
    {
      for (i = NUMBER; i < NUM_PR_ITEMS; i++)
	{
	  if (pr[i].datatype != MultiText)
	    continue;
	  
	  if (pr[i].value && regfind (m_text_search, pr[i].value) == 0)
	    return 1;
	}
      return 0;
    }
  return 1;
}

/* Get the PR at PATH.  If EXISTP is non-nil, only say if it existed
   or not.  If it's nil, read in the PR.  If SILENT is non-nil, we
   don't emit an error message.  */
int
get_pr (path, p, silent)
     char *path, *p;
     int silent;
{
  FILE *fp = fopen (path, "r");
  if (fp == (FILE *)NULL)
    {
      if (! silent)
	fprintf (stderr, "%s: couldn't read PR %s\n", program_name, p);
      return 0;
    }

  if (read_header (fp) < 0)
    {
      if (! silent)
	fprintf (stderr, "%s: error reading PR %s\n", program_name, p);
      return 0;
    }

  read_pr (fp, (query_format & ~FORMAT_FULL) && !m_text_search);

  fclose (fp);

  return 1;
}

/* Possibly print out a PR, checking if we want to look at unindexed things
   first.  */
int
do_pr_internal (path, prnum)
     char *path, *prnum;
{
  if (originator || text_search || m_text_search
      || synopsis || release || arrived_before || arrived_after
      || modified_before || modified_after || closed_before || closed_after
#ifdef GNATS_RELEASE_BASED
      || keywords || quarter
#endif
      || class)
    {
      if (! get_pr (path, prnum, quiet))
	return 0;

      if ((originator && (pr[ORIGINATOR].value == NULL
			  || regfind (originator, pr[ORIGINATOR].value) != 0))
	  || (class && (pr[CLASS].value == NULL
			|| regfind (class, pr[CLASS].value) != 0))
	  || (synopsis && (pr[SYNOPSIS].value == NULL
			|| regfind (synopsis, pr[SYNOPSIS].value) != 0))
	  || (release && (pr[RELEASE].value == NULL
			|| regfind (release, pr[RELEASE].value) != 0))
	  /* We have to check these here cuz the arrival date isn't in the
	     index, though it probably SHOULD be.  FIXME  */
	  || (arrived_before  && (pr[ARRIVAL_DATE].value == NULL
              || *pr[ARRIVAL_DATE].value == 0
	      || date_compare (arrived_before, pr[ARRIVAL_DATE].value,
				1, DOCONVERT) == 0))
	  || (arrived_after   && (pr[ARRIVAL_DATE].value == NULL
              || *pr[ARRIVAL_DATE].value == 0
	      || date_compare (arrived_after, pr[ARRIVAL_DATE].value,
				0, DOCONVERT) == 0))
	  || (modified_before && (pr[LAST_MODIFIED].value == NULL
              || *pr[LAST_MODIFIED].value == 0
	      || date_compare (modified_before, pr[LAST_MODIFIED].value,
				1, DOCONVERT) == 0))
	  || (modified_after  && (pr[LAST_MODIFIED].value == NULL
              || *pr[LAST_MODIFIED].value == 0
	      || date_compare (modified_after, pr[LAST_MODIFIED].value,
				0, DOCONVERT) == 0))
	  || (closed_before   && (pr[CLOSED_DATE].value == NULL
              || *pr[CLOSED_DATE].value == 0
	      || date_compare (closed_before, pr[CLOSED_DATE].value,
				1, DOCONVERT) == 0))
	  || (closed_after    && (pr[CLOSED_DATE].value == NULL
              || *pr[CLOSED_DATE].value == 0
	      || date_compare (closed_after, pr[CLOSED_DATE].value,
				0, DOCONVERT) == 0))
#ifdef GNATS_RELEASE_BASED
	  || (quarter && (pr[QUARTER].value == NULL
			|| regfind (quarter, pr[QUARTER].value) != 0))
	  || (keywords && (pr[KEYWORDS].value == NULL
			|| regfind (keywords, pr[KEYWORDS].value) != 0))
#endif
	  || !check_text())
	return 0;
      return 1;
    }
  else
    return get_pr (path, prnum, quiet);
}

int
do_pr (i)
     Index *i;
{
  int val = 0;
  char *path = make_path (i->category, i->number);

  if ((val = do_pr_internal (path, i->number)))
    print_pr (path, i->number, 1);
  
  xfree (path);

  return val;
}

/* Only say `PRs follow' once.  */
static int started = 0;

void
reset_started ()
{
  started = 0;
}

void
print_pr (path, p, opened)
     char *path, *p;
     int opened;
{
  /* Note: don't use P in any real way, cuz it could be "1234" or
     "category/1234".  */
  if (! opened && ! get_pr (path, p, quiet))
    return;

  if (is_daemon && !started)
    {
      printf ("%d PRs follow.\r\n", CODE_PR_READY);
      started = 1;
    }

  if (query_format & FORMAT_FULL)
    {
      write_header (outfile, NUM_HEADER_ITEMS);
      fprintf (outfile, "\n");
      write_pr (outfile, NUM_PR_ITEMS);
    }
  else if (query_format & FORMAT_SQL)
    {
      /* Trim `foo (Full Foo)' to just `foo'.  */
      char *t, *q = (char *) strchr (field_value (RESPONSIBLE), ' ');
      if (q != NULL)
	*q = '\0';

      fprintf (outfile, "%-8.8s|%-16.16s|%-128.128s|%-3.3s",
	       field_value (NUMBER), field_value (CATEGORY),
	       disbar(field_value (SYNOPSIS)),
	       field_value (CONFIDENTIAL));
      fprintf (outfile, "|%1.1d|%1.1d|%-16.16s|%1.1d|%1.1d|%-16.16s",
	       sql_types (field_value (SEVERITY), Severity),
	       sql_types (field_value (PRIORITY), Priority),
	       field_value (RESPONSIBLE),
	       sql_types (field_value (STATE), State),
	       sql_types (field_value (CLASS), Class),
	       field_value (SUBMITTER));
      t = sql_time (field_value (ARRIVAL_DATE));
      fprintf (outfile, "|%-18.18s|%-64.64s|%-64.64s|", t,
	       disbar (field_value (ORIGINATOR)),
	       field_value (RELEASE));
      xfree (t);
      if (strlen (field_value (LAST_MODIFIED)) == 0) {
           fprintf (outfile, "%-18.18s|", "");
      } else {
           t = sql_time (field_value (LAST_MODIFIED));
           fprintf (outfile, "%-18.18s|", t);
           xfree (t);
      }
      if (strlen (field_value (CLOSED_DATE)) == 0) {
           fprintf (outfile, "%-18.18s|", "");
      } else {
           t = sql_time (field_value (CLOSED_DATE));
           fprintf (outfile, "%-18.18s|", t);
           xfree (t);
      }
    }
  else if (query_format & FORMAT_SUMM)
    {
      char *s;

      s = (char *) strchr (pr[RESPONSIBLE].value, ' ');
      if (s)
	*s = '\0';
      fprintf (outfile,
	       "%8s %-8.8s %-8.8s %-9.9s %-9.9s %-8.8s %-10.10s %s",
	       pr[NUMBER].value, pr[RESPONSIBLE].value,
	       pr[CATEGORY].value, pr[STATE].value, pr[SEVERITY].value,
	       pr[PRIORITY].value, pr[SUBMITTER].value,
	       pr[SYNOPSIS].value);
    }
  else
    {
      /* Print this out for emacs when people do `M-x query-pr' and want
	 to be able to use the next-error function on the buffer.  */
      if (print_path)
	fprintf (outfile, "%s:0:\n", path);

      write_pr (outfile, NUMBER);
      write_pr (outfile, CATEGORY);
      write_pr (outfile, SYNOPSIS);
      write_pr (outfile, CONFIDENTIAL);
      write_pr (outfile, SEVERITY);
      write_pr (outfile, PRIORITY);
      write_pr (outfile, RESPONSIBLE);
      write_pr (outfile, STATE);
      write_pr (outfile, CLASS);
#ifdef GNATS_RELEASE_BASED
      write_pr (outfile, QUARTER);
      write_pr (outfile, KEYWORDS);
      write_pr (outfile, DATE_REQUIRED);
#endif
      write_pr (outfile, SUBMITTER);
      write_pr (outfile, ORIGINATOR);
      write_pr (outfile, RELEASE);
      write_pr (outfile, ARRIVAL_DATE);
      write_pr (outfile, LAST_MODIFIED);
      write_pr (outfile, CLOSED_DATE);
    }
}
