/* Remote GNATS server.
   Copyright (C) 1994, 95, 96, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"
#include "gnatsd.h"
#include "query.h"

char myname[MAXHOSTNAMELEN];

char *program_name;

/* If 1, we're running the daemon.  */
int is_daemon = 1;

/* The name of the host talking with us.  */
char *current_host;

/* Whether or not we've made sure they are who they say they are.  */
int host_verified = 0;

struct option long_options[] =
{
  {"directory", 1, NULL, 'd'},
  {"not-inetd", 0, NULL, 'n'},
  {"version", 0, NULL, 'V'},
  {NULL, 0, NULL, 0}
};

/* The search state we use.  */
Index *search;

/* Whether we should emit a linefeed and a newline.  */
extern int doret;

void usage (), version ();

#ifdef HAVE_KERBEROS
/* Non-zero if the client has used Kerberos authentication.  */
int client_authorized;
#endif

/* Non-zero if the host has to use Kerberos.  */
int kerberos_host;

#define FLAG_NEED_AUTHORIZATION		1
#define FLAG_ONE_ARG			2

typedef struct _command
{
  char *s;
  void (*fn)();
  int flags;
} Command;

static Command cmds[] =
{
  { "HELO", GNATS_helo, 0 },
  { "QUIT", GNATS_quit, 0 },

  /* Query a PR.  */
  { "QURY", GNATS_qury, FLAG_NEED_AUTHORIZATION },
  { "SUMM", GNATS_summ, FLAG_NEED_AUTHORIZATION },
  { "FULL", GNATS_full, FLAG_NEED_AUTHORIZATION },
  { "SQLF", GNATS_sqlf, FLAG_NEED_AUTHORIZATION },

  /* Edit a PR.  */
  { "LOCK", GNATS_lock, FLAG_NEED_AUTHORIZATION },
  { "EDIT", GNATS_edit, FLAG_NEED_AUTHORIZATION },
  { "UNLK", GNATS_unlk, FLAG_NEED_AUTHORIZATION },

  /* Return the people that should be mailed about a PR.  */
  { "MLPR", GNATS_mlpr, FLAG_NEED_AUTHORIZATION },
  /* Return the people that should be mailed about a given category.  */
  { "MLCT", GNATS_mlct, FLAG_NEED_AUTHORIZATION },
  /* Return the people that should be mailed about a given submitter.  */
  { "MLSU", GNATS_mlsu, FLAG_NEED_AUTHORIZATION },

  /* Return the type of submitter of a given name.  */
  { "TYPE", GNATS_type, FLAG_NEED_AUTHORIZATION },

  { "RESP", GNATS_resp, FLAG_NEED_AUTHORIZATION },
  { "CATG", GNATS_catg, FLAG_NEED_AUTHORIZATION },
  { "SYNP", GNATS_synp, FLAG_NEED_AUTHORIZATION },
  { "STAT", GNATS_stat, FLAG_NEED_AUTHORIZATION },
  { "CONF", GNATS_conf, 0 },
  { "SVTY", GNATS_svty, 0 },
  { "ORIG", GNATS_orig, FLAG_NEED_AUTHORIZATION|FLAG_ONE_ARG },
  { "RLSE", GNATS_rlse, FLAG_NEED_AUTHORIZATION },
  { "CLSS", GNATS_clss, FLAG_ONE_ARG },
  { "PRIO", GNATS_prio, 0 },
  { "SUBM", GNATS_subm, FLAG_NEED_AUTHORIZATION|FLAG_ONE_ARG },
  { "TEXT", GNATS_text, FLAG_ONE_ARG },
  { "MTXT", GNATS_mtxt, FLAG_ONE_ARG },

#ifdef GNATS_RELEASE_BASED
  { "BFOR", GNATS_bfor, FLAG_ONE_ARG },
  { "AFTR", GNATS_aftr, FLAG_ONE_ARG },
#endif
  { "ABFR", GNATS_abfr, FLAG_ONE_ARG },
  { "ARAF", GNATS_araf, FLAG_ONE_ARG },
  { "CBFR", GNATS_cbfr, FLAG_ONE_ARG },
  { "CAFT", GNATS_caft, FLAG_ONE_ARG },
  { "MBFR", GNATS_mbfr, FLAG_ONE_ARG },
  { "MAFT", GNATS_maft, FLAG_ONE_ARG },
  { "VDAT", GNATS_vdat, FLAG_ONE_ARG },

  { "LCAT", GNATS_lcat, FLAG_NEED_AUTHORIZATION },
  { "LCFG", GNATS_lcfg, FLAG_NEED_AUTHORIZATION },
  { "LCLA", GNATS_lcla, FLAG_NEED_AUTHORIZATION },
  { "LRES", GNATS_lres, FLAG_NEED_AUTHORIZATION },
  { "LSUB", GNATS_lsub, FLAG_NEED_AUTHORIZATION },
  { "LSTA", GNATS_lsta, FLAG_NEED_AUTHORIZATION },

  /* Stop and start the database.  */
  { "LKDB", GNATS_lkdb, FLAG_NEED_AUTHORIZATION },
  { "UNDB", GNATS_undb, FLAG_NEED_AUTHORIZATION },

  { "AUTH", GNATS_auth, 0 },
  { "RSET", GNATS_rset, 0 },
  { "NOCL", GNATS_nocl, 0 },
  { "HELP", GNATS_help, 0 },

#ifdef GNATS_RELEASE_BASED
  { "QRTR", GNATS_qrtr, FLAG_ONE_ARG },
  { "KYWD", GNATS_kywd, FLAG_ONE_ARG },
#endif

  { NULL, NULL },
};

int
match (line, p)
     char *line;
     char *p;
{
  int matched;

  for ( ; *p; line++, p++)
    {
      if (*line == '\0' && *p != '*')
	return 0;
      switch (*p)
	{
	case '\\':
	  /* Literal match with following character. */
	  p++;
	default:
	  if (tolower (*line) != tolower (*p))
	    return 0;
	  continue;
	case '?':
	  /* Match anything. */
	  continue;
	case '*':
	  while (*++p == '*')
	    /* Consecutive stars act just like one. */
	    continue;
	  if (*p == '\0')
	    /* Trailing star matches everything. */
	    return 1;
	  while (*line)
	    if ((matched = match (line++, p)) != 0)
	      return matched;
	  return 0;
	}
    }

  return 1;
}

char *
get_name (host)
     struct in_addr *host;
{
  char *buf;
  int i;
  struct hostent *hp;
#ifdef h_addr
  char **pp;
#endif

  hp = gethostbyaddr ((char *) host, sizeof (host), AF_INET);
  if (hp == NULL)
    return NULL;
  i = strlen (hp->h_name);
  buf = (char *) xmalloc (i + 1);
  strcpy (buf, hp->h_name);
  hp = gethostbyname (buf);
  if (hp == NULL)
    return NULL;

#ifdef h_addr
  for (pp = hp->h_addr_list; *pp; pp++)
    if (memcmp ((char *) &host->s_addr, (char *) *pp, hp->h_length) == 0)
      break;
  if (*pp == NULL)
    return NULL;
#else
  if (memcmp ((char *) &host->s_addr, (char *) hp->h_addr, hp->h_length) != 0)
    return NULL;
#endif

  return buf;
}

/* See if we know about this host.  If we don't, then run with it.  */
int
allowed (host)
     char *host;
{
  FILE *acc;
  char buffer[BUFSIZ], *x;
  char *fields[3];
  int i;
  int found = 0;
  char *filename = (char *) alloca (PATH_MAX);

  /* The format of the file is:
        sitename:all:
     where `sitename' is the hostname allowed; `all' is just a fill-in
     for now, but may be the field to control what PRs, categories, or
     submitter-id PRs are allowed; and the final field is blank, but
     may hold something in the future.  */
  sprintf (filename, "%s/gnats-adm/%s", gnats_root, ACCESS_FILE);
  acc = fopen (filename, "r");
  if (acc == (FILE *)NULL)
    {
      syslog (LOG_ERR, "%s for %s cannot open %m", host, filename);
      return 0;
    }

  while (fgets (buffer, sizeof (buffer), acc) != NULL)
    {
      x = strchr (buffer, '\n');
      if (x != NULL)
	*x = '\0';
      x = strchr (buffer, '#');
      if (x != NULL)
	*x = '\0';
      if (buffer[0] == '\0')
	continue;

      for (fields[0] = buffer, i = 0, x = buffer; *x && i < 3; x++)
	if (*x == ':')
	  {
	    *x = '\0';
	    fields[++i] = x + 1;
	  }
      /* Is there a mistake on the setup of this line?  */
      if (i != 2)
	continue;

      /* XXX check hostname and IP */
      if (! match (host, fields[0]))
	continue;
      found = 1;
    }

  fclose (acc);

#ifdef HAVE_KERBEROS
  if (! found)
    found = kerberos_host = 1;
#endif

  return found;
}

void
start_connection ()
{
  char *host;
  struct sockaddr_in s;
  int len;

  len = sizeof (s);
  if (getpeername (0, (struct sockaddr *)&s, &len) < 0)
    {
      if (! isatty (0))
	{
	  syslog (LOG_ERR, "%s: can't get peername %m", "?");
	  printf ("%d Why can't I figure out your name?  Later.\r\n",
		  CODE_NO_ACCESS);
	  exit (1);
	}
      host = "stdin";
    }
  else
    {
      if (s.sin_family != AF_INET)
	{
	  syslog (LOG_ERR, "%s: bad address family %ld",
		  "?", (long) s.sin_family);
	  printf ("%d You're a member of a bad address family.  Later.\r\n",
		  CODE_NO_ACCESS);
	  exit (1);
	}

      host = get_name (&s.sin_addr);
      if (! host)
	host = (char *) inet_ntoa (s.sin_addr);
    }

  if (!allowed (host))
    {
      syslog (LOG_NOTICE, "%s not allowed access", host);
      printf ("%d You are not on the access list.\r\n", CODE_NO_ACCESS);
      exit (1);
    }

  current_host = host;
  syslog (LOG_INFO, "%s connect", host);
}


#include <setjmp.h>
jmp_buf env;

void /* FIXME */
read_timeout(sig)
     int sig;
{
     longjmp(env,1);
}

MsgType
get_line (start, size, timeout)
     char *start;
     int size;
     int timeout;
{
  static int count;
  static char buffer[BUFSIZ];
  static char *bp;
  register char *p;
  register char *end;
  struct timeval t;
  fd_set rmask;
  int i;
  char c;

  for (p = start, end = &start[size - 1]; ; ) {
    if (count == 0) {
      /* Fill the buffer. */
    Again:
      FD_ZERO(&rmask);
      FD_SET(0, &rmask);
      t.tv_sec = timeout;
      t.tv_usec = 0;
      i = select (0 + 1, &rmask, (fd_set *)NULL, (fd_set *)NULL, &t);
      if (i < 0) {
	if (errno == EINTR)
	  goto Again;
	return PR_timeout;
      }
      if (i == 0 || !FD_ISSET(0, &rmask))
	return PR_timeout;
      count = read (0, buffer, sizeof buffer);
      if (count < 0) {
	return PR_timeout;
      }
      if (count == 0)
	return PR_eof;
      bp = buffer;
    }

    /* Process next character. */
    count--;
    c = *bp++;
    if (c == '\n')
      break;
    if (p < end)
      *p++ = c;
  }

  /* If last two characters are \r\n, kill the \r as well as the \n. */
  if (p > start && p < end && p[-1] == '\r')
    p--;
  *p = '\0';
  return p == end ? PR_long : PR_ok;
}

int
Argify(line, argvp, cp)
    char		*line;
    char		***argvp;
    Command		**cp;
{
    register char	**argv;
    register char	*p;
    register int	i;

    if (*argvp != NULL) {
      xfree (*argvp[0]);
      xfree ((char *)*argvp);
    }

    /*  Copy the line, which we will split up. */
    while (*line == ' ' || *line == '\n')
	line++;
    i = strlen(line);
    p = (char *) xmalloc (sizeof (char) * (i + 1));
    strcpy(p, line);

    /* Allocate worst-case amount of space. */
    *argvp = argv = (char **) xmalloc (sizeof (char *) * (i + 2));
    argv[0] = p;
    if ((p = strpbrk(p, " \n")))
      *p++ = '\0';
    for (*cp = cmds; (*cp)->s; (*cp)++) {
      if (strncasecmp ((*cp)->s, argv[0], strlen ((*cp)->s)) == 0)
	break;
    }
    if (! (*cp)->s)
      *cp = NULL;

    if (! p) {
      argv[1] = NULL;
      return 1;
    }

    if (*cp && (*cp)->flags & FLAG_ONE_ARG) {
      while (*p && (*p == ' ' || *p == '\n'))
	p++;
      argv[1] = p;
      argv[2] = NULL;
      return 2;
    }
      
    for (argv++; *p; ) {
	/* Mark start of this word, find its end. */
	for (*argv++ = p; *p && *p != ' ' && *p != '\n'; )
	    p++;
	if (*p == '\0')
	    break;

	/* Nip off word, skip whitespace. */
	for (*p++ = '\0'; *p == ' ' || *p == '\n'; )
	    p++;
    }
    *argv = NULL;
    return argv - *argvp;
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  char buff[512];
  struct hostent *hp;
  char **av;
  int ac;
  Command *cp;
  MsgType r;
  int optc;
  int not_inetd = 0;

  program_name = basename (argv[0]);

  search = (Index *) xmalloc (sizeof (Index));
  memset (search, 0, sizeof (Index));

  while ((optc = getopt_long (argc, argv, "d:nV",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'd':
	  gnats_root = optarg;
	  break;

	case 'n':
	  not_inetd = 1;
	  break;

	case 'V':
	  version ();
	  exit (0);
	  break;

	default:
	  usage ();
	}
    }

  configure ();
  init_gnats ();
  umask (022);
  re_set_syntax ((RE_SYNTAX_POSIX_EXTENDED | RE_BK_PLUS_QM) & ~RE_DOT_NEWLINE);

  /* Don't emit any errors from get_pr.  */
  quiet = 1;
#ifdef HAVE_KERBEROS
  kerberos_host = client_authorized = 0;
#endif
  if (not_inetd)
    {
      host_verified = 1;
#ifdef HAVE_KERBEROS
      client_authorized = 1;
#endif
    }

  /* chdir */

  fflush (stderr);
  fflush (stdout);
  /* close file descriptors */

  closelog ();
#ifdef LOG_DAEMON
  openlog ("gnatsd", (LOG_PID|LOG_CONS), LOG_DAEMON);
#else
  openlog ("gnatsd", LOG_PID);
#endif

  if (gethostname (myname, MAXHOSTNAMELEN) != 0)
    {
      fprintf (stderr, "%s: cannot find out the name of this host\n",
	       program_name);
      exit (1);
    }

  /* Now see if we can get a FQDN for this host.  */
  hp = (struct hostent *) gethostbyname (myname);
  if (hp && (strlen (hp->h_name) > strlen (myname)))
    strncpy (myname, hp->h_name, MAXHOSTNAMELEN);

  /* Don't pay attention to SIGPIPE; read will give us the problem
     if it happens.  */
  signal (SIGPIPE, SIG_IGN);

  /* We want to emit both a linefeed and a newline for each line.  */
  doret = 1;

  if (! not_inetd)
    start_connection ();
  else
    current_host = myname;

  index_chain = get_index ();
  if (index_chain == NULL)
    {
      printf ("%d GNATS server cannot read the index.\r\n", CODE_FILE_ERROR);
      exit (1);
    }

  printf ("%d%c%s GNATS server %s ready.\r\n", CODE_GREETING,
	  kerberos_host ? '-' : ' ', myname, version_string);

  if (kerberos_host)
    printf ("%d Hi %s; you must use Kerberos authentication.\r\n",
	    CODE_GREETING, current_host);

  for (av = NULL; ; )
    {
      fflush (stdout);
      r = get_line (buff, (int) sizeof (buff), 2 * 60 * 60);
      switch (r)
	{
	case PR_timeout:
	  printf ("%d Read timeout.\r\n", CODE_TIMEOUT);
	  exit (1);
	  break;
	case PR_long:
	  printf ("%d Line too long\r\n", CODE_ERROR);
	  continue;
	case PR_ok:
	  if (buff[0] == '\0' || (buff[4] != ' ' && buff[4] != '\0'))
	    {
	      printf ("%d Unrecognized command.\r\n", CODE_ERROR);
	      continue;
	    }
	  ac = Argify (buff, &av, &cp);
	  if (ac == 0)
	    continue;
	  break;
	case PR_eof:
	  /* Handled below.  */
	  break;
	}

      if (r == PR_eof || strcasecmp (av[0], "quit") == 0)
	break;

      if (! cp)
	{
	  printf ("%d Unrecognized command.\r\n", CODE_ERROR);
	  continue;
	}
      /* Check usage */

      /* Did they say hi? */
#ifdef HAVE_KERBEROS
      if (kerberos_host && (cp->flags & FLAG_NEED_AUTHORIZATION)
	  && !client_authorized)
	{
	  syslog (LOG_ERR, "GNATS command `%s' without authentication for %s",
		  cp->s, current_host);
	  printf ("%d You are not authorized to perform this operation (%s).\r\n",
		  CODE_NO_ACCESS, cp->s);
	  continue;
	}
      else
#endif
	/* Call the function.  We skip the command itself in the arg list.  */
	(*cp->fn) (ac - 1, av + 1);
    }

  printf ("%d Later.\r\n", CODE_CLOSING);
  exit (0);
}

void
usage ()
{
  fprintf (stderr, "\
Usage: %s [-n] [-d directory] [--directory=directory] [--not-inetd]\n",
	   program_name);
  exit (1);
}

void
version ()
{
  printf ("gnatsd %s\n", version_string);
}
