/* Miscellaneous utility routines for GNATS.
   Copyright (C) 1993, 1994, 1995 Free Software Foundation, Inc.
   Contributed by Tim Wicinski (wicinski@barn.com)
   and Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

#include "config.h"
#include "gnats.h"

/* Debugging level.  */
int debug_level = LOG_ERR; 

/* File to log all messages to.  */
FILE *gnats_logfile = NULL;

/* Current logging method.  */
Logging_Methods log_method = STDERR;

/* Table to tell if a char c is whitespace.  */
unsigned char is_space[256];

/* Set to either \n or \r\n for writing things out.  */
char *ret = "\n";

void
init_gnats ()
{
  init_space ();
  init_pr ();
  init_header ();
  init_network ();
}

/* VARARGS */
void 
#if defined(__STDC__) || defined(_AIX)
log_msg (int severity, int has_arg, ...)
#else
log_msg (severity, has_arg, va_alist)
     int severity, has_arg;
     va_dcl
#endif
{
  va_list args;
  char buf[STR_MAX], *message, *message2;

  VA_START (args, has_arg);
  message = va_arg (args, char *);
  if (has_arg)
    message2 = va_arg (args, char *);

  if (has_arg)
    sprintf (buf, "%s: %s %s\n", program_name, message, message2);
  else
    sprintf (buf, "%s: %s\n", program_name, message);

  if (debug_level >= severity)
    {
#ifdef HAVE_SYSLOG_H
      if (log_method == SYSLOG)
	syslog (severity, buf);
      else
#endif
      if (gnats_logfile != (FILE *) NULL)
	fprintf (gnats_logfile, "%s", buf);
      else
	fprintf (stderr, "%s", buf);
    }

  va_end (args);

  if (severity == LOG_ERR)
    exit (2);
}

/* Open the file (actually the pipe) to which the mail message will
   be written.  */
FILE *
open_mail_file ()
{
  FILE *fp;

  /* Can't use log_msg here, since gnats_logfile is being set by this first
     thing.  */

  fp = popen (mail_agent, "w");

  if (debug_mode)
    {
      fprintf (fp, "From: %s (GNATS Management)\n", gnats_user);
      fprintf (fp, "To: %s\n", gnats_admin);
      fprintf (fp, "Subject: mail output from %s\n", program_name);
      fprintf (fp, "\n\n");
    }

  return fp;
}

void
close_mail_file (fp)
     FILE *fp;
{

  if (fp)
    {
      fflush (fp);
      pclose (fp);
    }
}

/* Initialize the `is_space' array.  */
void
init_space ()
{
  is_space[' '] = 1;
  is_space['\t'] = 1;
  is_space['\v'] = 1;
  is_space['\f'] = 1;
  is_space['\n'] = 1;
  is_space['\r'] = 1;
}

/* Initialize network-related possibilities.  */
void
init_network ()
{
  extern int doret;

  if (doret)
    ret = "\r\n";
}

/* Scan down LINE, returning the next token.  We can't use strtok, since
   we often deal with constant strings like "foo | bar" for the default
   values for a PR.

   Stores the returned token in token and returns the address in line
   at which to begin parsing for the next token, or returns null if
   the token being returned is the last one. */
char *
get_next_field (line, token, delim)
     char *line, *token;
     int delim;
{
  char *t = token;

  while (*line != delim && *line != '\0')
    *t++ = *line++;
  *t = '\0';

  /* erase whitespace at the end of the token */
  while ((t > token) && (is_space[*(t-1)]))
    *--t = '\0';

  if (! *line)
    return NULL;
  
  /* skip over any white space after delim */
  while (*line != '\0' && is_space[*++line])
    /* Nothing */ ;

  t = strchr (line, '\n');
  if (t)
    *t = '\0';

  return line;
}

/* Read in the next line using fgets, and return its length.  */
int
read_string (line, fp)
     char *line;
     FILE *fp;
{
  if (fgets (line, STR_MAX, fp) == NULL)
    return -1;

  if (*line == '\n')
    return 1;

  return strlen (line);
}

/* Grab a token, and returning both the token and the shortened string. */
char *
get_token (line, token)
     char *line;
     char *token;
{
  char *t = token;

  if (*line == ' ')
    return line;

  /* Check for quote. if we see if, read everything up to the next quote
     as the token.  */
  if (*line == '"')
    {
      line++; /* ignore the first quote */
      while (*line != '"' && *line != '\0')
	*t++ = *line++;

      line++; /*  ignore the last quote */
      SKIP_WHITE_SPACE (line);
    }
  else
    {
      /* If not quote, then just regular strings for now.  */
      /* up to the ':' character, which is the delimiter for tokens */
      while (*line != ' ' && *line != '\t' && *line != '\n' && *line != '\0' 
          && *line != ':' )
        *t++ = *line++;
      if (*line == ':') /* include the ':' since this is part of the token */
        *t++ = *line++;

    }

  /* I want to make sure this happens.  */
  *t = '\0';

  SKIP_WHITE_SPACE (line);
  return line;
}

#ifndef HAVE_MKDIR
/* mkdir and rmdir adapted from GNU tar.  */

/* Make directory DPATH, with permission mode DMODE.

   Written by Robert Rother, Mariah Corporation, August 1985
   (sdcsvax!rmr or rmr@@uscd).  If you want it, it's yours.

   Severely hacked over by John Gilmore to make a 4.2BSD compatible
   subroutine.	11Mar86; hoptoad!gnu

   Modified by rmtodd@@uokmax 6-28-87 -- when making an already existing dir,
   subroutine didn't return EEXIST.  It does now.  */

int
mkdir (dpath, dmode)
     char *dpath;
     int dmode;
{
  int cpid, status;
  struct stat statbuf;

  if (stat (dpath, &statbuf) == 0)
    {
      errno = EEXIST;		/* stat worked, so it already exists.  */
      return -1;
    }

  /* If stat fails for a reason other than non-existence, return error.  */
  if (errno != ENOENT)
    return -1;

  cpid = fork ();
  switch (cpid)
    {
    case -1:			/* Cannot fork.  */
      return -1;		/* errno is set already.  */

    case 0:			/* Child process.  */
      /* Cheap hack to set mode of new directory.  Since this child
	 process is going away anyway, we zap its umask.
	 This won't suffice to set SUID, SGID, etc. on this
	 directory, so the parent process calls chmod afterward.  */
      status = umask (0);	/* Get current umask.  */
      umask (status | (0777 & ~dmode));	/* Set for mkdir.  */
      execl ("/bin/mkdir", "mkdir", dpath, (char *) 0);
      _exit (1);

    default:			/* Parent process.  */
      while (wait (&status) != cpid) /* Wait for kid to finish.  */
	/* Do nothing.  */ ;

      if (status & 0xFFFF)
	{
	  errno = EIO;		/* /bin/mkdir failed.  */
	  return -1;
	}
      return chmod (dpath, dmode);
    }
}
#endif /* HAVE_MKDIR */
