set TkGnats(lib) ./; ##TKGNATSLIB##
source $TkGnats(lib)/tkgnats.config

proc file_get_text {file} {
    if {[catch {open $file r} fin]} {
	return ""
    }
    set    text [read $fin]
    close  $fin
    return [string trimright $text "\n"]
}

if {[info exists env(GNATS_ROOT)]} {
    set config [file_get_text $env(GNATS_ROOT)/gnats-adm/config]
    if {"$config" != ""} {
        set TkGnats(GNATS_ADDR) \
                [string trim [lindex [split [lindex $config [lsearch -regexp $config GNATS_ADDR]] =] 1] \"]
        
        set TkGnats(GNATS_USER) \
                [string trim [lindex [split [lindex $config [lsearch -regexp $config GNATS_USER]] =] 1] \"]
        
        set TkGnats(GNATS_SITE) \
                [string trim [lindex [split [lindex $config [lsearch -regexp $config GNATS_SITE]] =] 1] \"]
        
        set TkGnats(Submitter-Id) \
                [string trim [lindex [split [lindex $config [lsearch -regexp $config SUBMITTER]] =] 1] \"]

        set GNATS_ROOT $env(GNATS_ROOT)
        set GNATS_BASE [file dirname $env(GNATS_ROOT)]

        set TkGnats(CategoriesFile)	$GNATS_BASE/$TkGnats(GNATS_SITE)
        set TkGnats(SubmittersFile)	$GNATS_ROOT/gnats-adm/submitters
        set TkGnats(ResponsibleFile)	$GNATS_ROOT/gnats-adm/responsible
    }
}

set system [exec uname -s]
if {[file readable $TkGnats(lib)/tkgnatsrc]} {
    source $TkGnats(lib)/tkgnatsrc
}
if {[file readable $TkGnats(lib)/tkgnatsrc.$system]} {
    source $TkGnats(lib)/tkgnatsrc.$system
}
set TkGnats(tkgnats_version) [file_get_text $TkGnats(lib)/VERSION]

proc check_tkgnats_userdir {} {
    global TkGnats
    if {![file isdirectory $TkGnats(UserDir)]} {
        if {[file isdirectory ~/TkGnats]} {
            file rename [glob ~/TkGnats] $TkGnats(UserDir)
	    file rename $TkGnats(UserDir)/default_sort $TkGnats(UserDir)/default_sort.old
        } {
            file mkdir $TkGnats(UserDir)
        }
    }
    if {![file isdirectory $TkGnats(UserDir)/query]} {
        file mkdir $TkGnats(UserDir)/query
    }
    if {![file isdirectory $TkGnats(UserDir)/sort]} {
        file mkdir $TkGnats(UserDir)/sort
    }
}

if {![info exists TkGnats(UserDir)]} {
    set TkGnats(UserDir) [glob ~]/.tkgnats
}
set TkGnats(UserDir) [string trimright $TkGnats(UserDir) /]
check_tkgnats_userdir

if {[file readable $TkGnats(UserDir)/tkgnatsrc]} {
    source $TkGnats(UserDir)/tkgnatsrc
}
if {[file readable $TkGnats(UserDir)/tkgnatsrc.$system]} {
    source $TkGnats(UserDir)/tkgnatsrc.$system
}

# Determine if this user is authorized to edit problem reports
set glist {}
set ulist {}
if {[info exists TkGnats(edit_authorized_groups)]} {
    set glist $TkGnats(edit_authorized_groups)
}
if {[info exists TkGnats(edit_authorized_users)]} {
    set ulist $TkGnats(edit_authorized_users)
}
set TkGnats(edit_authorized) 0
if {$ulist == {} && $glist == {}} {
    set TkGnats(edit_authorized) 1
}
if {$ulist != {} && [lsearch -exact $ulist $TkGnats(LogName)] > -1} {
    set TkGnats(edit_authorized) 1
}
if {$glist != {} && [lsearch -exact $glist $TkGnats(GroupName)] > -1} {
    set TkGnats(edit_authorized) 1
}

# Determine if this user is authorized to delete problem reports
set glist {}
set ulist {}
if {[info exists TkGnats(delete_authorized_groups)]} {
    set glist $TkGnats(delete_authorized_groups)
}
if {[info exists TkGnats(delete_authorized_users)]} {
    set ulist $TkGnats(delete_authorized_users)
}
set TkGnats(delete_authorized) 0
if {$ulist != {} && [lsearch -exact $ulist $TkGnats(LogName)] > -1} {
    set TkGnats(delete_authorized) 1
}
if {$glist != {} && [lsearch -exact $glist $TkGnats(GroupName)] > -1} {
    set TkGnats(delete_authorized) 1
}

proc Msg {args} {
    set msg ""
    set nargs [expr [llength $args] - 1]
    for {set i 0} {$i < $nargs} {incr i} {
        append msg "[lindex $args $i]\n"
    }
    append msg "[lindex $args $nargs]"
    bell
    tk_dialog .tkerr "TkGnats Error" $msg "error" 0 "OK"
}

# reap any zombied exec's
set TkGnats(reap_scheduled) 0
proc do_reap {} {
    global TkGnats
    exec true
    set TkGnats(reap_scheduled) 0
}
proc schedule_reap {} {
    global TkGnats
    if {!$TkGnats(reap_scheduled)} {
	set TkGnats(reap_scheduled) 1
	after 5000 do_reap
    }
}

proc Exit {x} {
    #destroy .; # use if before tk3.3
    exit $x
}

proc my_pr_addr {name} {
    global TkGnats
    # strip off whitespace and then take out the first word.
    # we assume the rest is a (Full Name) type comment
    set name [string trim $name "\t\n "]
    set name [lindex [split $name "\t\n "] 0]
    set tmp [exec $TkGnats(pr-addr) $name]
    if {"$tmp" != ""} {
	set name $tmp
    }
    return $name
}

#
# trim pr field data whitespace
#
proc ftrim {s} {
    return [string trim $s "\t\n "]
}

#
# get a list of valid gnats categories
#
proc get_categories {{pat "*"} {file ""}} {
    global TkGnats
    if {"$file" == ""} {
	set file $TkGnats(CategoriesFile)
    }
    set catlist {}
    if {[file readable $file]} {
	foreach c [split [file_get_text $file] "\n"] {
	    # ignore lines with leading hash or underscore
	    case $c "#*" {
	    } "_*" {
	    } $pat {
		lappend catlist [lindex [string trim $c "\n\t "] 0]
	    }
	}
    } else {
	Msg "Cannot read category file '$file'"
	return ""
    }
    return [lsort $catlist]
}

#
# get a list of valid gnats submitters
#
proc get_submitters {{pat "*"} {file ""}} {
    global TkGnats
    if {"$file" == ""} {
	set file $TkGnats(SubmittersFile)
    }
    set sublist {}
    if {[file readable $file]} {
	foreach s [split [file_get_text $file] "\n"] {
	    # ignore lines with leading hash or underscore
	    case $s "#*" {
	    } "_*" {
	    } $pat {
		lappend sublist [lindex [split $s ":"] 0]
	    }
	}
    } else {
	Msg "Cannot read submitters file '$file'"
	return ""
    }
    return [lsort $sublist]
}

#
# get a list of valid gnats responsibles
#
proc get_responsibles {{pat "*"} {file ""}} {
    global TkGnats
    if {"$file" == ""} {
	set file $TkGnats(ResponsibleFile)
    }
    set reslist {}
    if {[file readable $file]} {
	foreach r [split [file_get_text $file] "\n"] {
	    # ignore lines with leading hash or underscore
	    case $r "#*" {
	    } "_*" {
	    } $pat {
		lappend reslist [lindex [split $r ":"] 0]
	    }
	}
    } else {
	Msg "Cannot read responsible file '$file'"
	return ""
    }
    return [lsort $reslist]
}

#
# get the users name from the passwd file given their logname
# try NIS first, then try the regular passwd file
#
proc fullname_from_logname {{lname ""}} {
    global TkGnats
    if {"$lname" == ""} {
	set lname $TkGnats(LogName)
    }
    set fullname ""
    if {[catch {set fullname [exec ypcat passwd | grep ^${lname}: | cut -f5 -d:]}]} {
	set fullname [lindex [split [file_search_string /etc/passwd ^${lname}:] :] 4]
    }
    return $fullname
}

proc get_passwd_entry {lname} {
    # first try NIS, then try the regular passwd file...
    set fullname ""
    if {[catch {set fullname [exec ypcat passwd | grep ^${lname}:]}]} {
	set fullname [file_search_string /etc/passwd ^${lname}:]
    }
    return $fullname
}

proc radiobar_frame {parent frname} {
    frame $frname
    frame $frname.labels
    frame $frname.values
    frame $frname.bars
    pack  $frname.labels -side left  -anchor w
    pack  $frname.values -side left  -anchor center -padx 2
    pack  $frname.bars   -side right -expand true   -fill x
    pack  $frname -anchor w
}

# text field related procs

proc textset {l t {p ""}} {
    # a label
    if {[winfo exists ${p}._${l}.textlabel]} {
	${p}._${l}.textlabel configure -text $t
	return
    }
    # text widget
    if {[winfo exists ${p}._${l}_fr.text]} {
	${p}._${l}_fr.text delete 1.0 end
	${p}._${l}_fr.text insert 1.0 $t
	return
    }
    # entry widget
    if {[winfo exists ${p}._${l}.text]} { 
	${p}._${l}.text delete 0 end
	${p}._${l}.text insert 0 $t
	return
    }
    error "no such window for $l"
}

proc textget {l {p ""}} {
    if {[catch  {set x [lindex [${p}._${l}.textlabel configure -text] 4]}  ]} {
	if {[catch {set x [${p}._${l}_fr.text get 1.0 end]}]} {
	    return [string trim [${p}._${l}.text get] "\n"]
	}
    }
    return "\n$x"
}

proc readonly_singletext {l {t ""} {labwid 12} {valwid 0}} {
    if {$valwid == 0} {
        set valwid [string length "$t"]
    }
    set f  [frame ._${l}]
    set lw [button $f.label    -anchor w -width $labwid -text "$l: " -command "helpMsg $l" \
            -relief flat -padx 0 -pady 0 -borderwidth 0 -highlightthickness 1]
    set ew [label $f.textlabel -anchor w -width $valwid -text "$t" -relief groove \
            -background green -padx 2 -pady 0]
    pack $lw -side left -anchor w -pady 0 -padx 1
    pack $ew -side left -anchor w -pady 0
    pack $f  -side top  -anchor w -pady 0 -fill x
    return $ew
}

bind Entry <KeyPress-Return> " "

proc singletext {parent l w {t ""} {labwid 12}} {
    set f [frame $parent._${l}]
    # trim off any leading >'s for the label text
    set lw [button $f.label -anchor w -width $labwid -text "[string trimleft $l >]: " \
            -command "helpMsg [string trimleft $l >]" -relief flat -padx 0 -pady 0 -borderwidth 0]
    set ew [entry $f.text -width $w \
	-insertwidth 1	-insertofftime 400  \
	-relief sunken -borderwidth 2 -background grey95]
    bind $ew <Enter> "+focus $ew"
    $ew insert end $t
    pack $lw -side left -anchor w
    pack $ew -side left -anchor w -fill x -expand true
    pack $f  -side top  -anchor w -fill x -pady 2
    return $ew
}

proc bagged_singletext {l w bagname {prefix ""} {t ""}} {
    upvar #0 $bagname bag
    set f [frame ._${l}]
    set lw [label $f.label -anchor w -text "$l: "]
    set ew [entry $f.text -width 80 \
	-insertwidth 1	-insertofftime 400  \
	-relief sunken -borderwidth 2 -background grey95]
    bind $ew <Enter> "+focus $ew"
    $ew insert end $t
    pack $lw -side left  -anchor w -fill x -expand true
    pack $ew -side right -anchor e
    pack $f  -side top   -anchor w -fill x -pady 2
    set bag($prefix$l) [format {[string trim [%s get]]} $lw]
    return $ew
}

proc multitextdeck {w blist h} {
    global TkGnats
    frame $w 
    menubutton $w.mb
    text $f.text \
	-wrap $TkGnats(TextWrap) \
	-yscrollcommand "$f.sb set" \
	-height $h -width 80 -relief sunken -padx 4 -insertwidth 1 \
	-insertofftime 400 -borderwidth 2 -background grey95
    foreach lbl $blist {
	set l _$lbl
    }
}

proc multitext {lbl h} {
    global TkGnats
    set l _$lbl
    set f [frame .${l}_fr]
    button $f.label -anchor w -text "[string trimleft $lbl >]: " \
            -command "helpMsg [string trimleft $lbl >]" \
            -relief flat -padx 0 -pady 0 -borderwidth 0
    text $f.text \
	-wrap $TkGnats(TextWrap) \
	-yscrollcommand "$f.sb set" \
	-height $h -width 80 -relief sunken -padx 4 -insertwidth 1 \
	-insertofftime 400 -borderwidth 2 -background grey95
    bind $f.text <Enter> "+focus $f.text"
    scrollbar $f.sb -command "$f.text yview" -relief sunken
    pack $f.label -side top   -anchor w
    pack $f.sb    -side left  -fill y
    pack $f.text  -side right -fill both -expand true
    pack $f       -side top   -fill both -expand true -padx 16 -pady 2

    return $f.text
}

proc set_text_traversal {tlist} {
    set ll [llength $tlist]
    if {$ll < 2} {
	return
    }
    for {set x 1} {$x<$ll} {incr x} {
	set w [lindex $tlist $x]
	set prevw [lindex $tlist [expr $x-1]]
	bind $prevw <Tab>       "focus $w
        break"
	bind $prevw <Control-n> "focus $w"
    }
    bind [lindex $tlist [expr $ll-1]] <Tab>       "focus [lindex $tlist 0]
    break"
    bind [lindex $tlist [expr $ll-1]] <Control-n> "focus [lindex $tlist 0]"
    
    bind [lindex $tlist 0] <Shift-Tab> "focus [lindex $tlist [expr $ll-1]]
        break"
    bind [lindex $tlist 0] <Control-p> "focus [lindex $tlist [expr $ll-1]]"
    for {set x 0} {$x < [expr $ll-1]} {incr x} {
	set w [lindex $tlist $x]
	set nextw [lindex $tlist [expr $x+1]]
	bind $nextw <Shift-Tab> "focus $w
        break"
	bind $nextw <Control-p> "focus $w"
    }
}

proc bagged_radiobar {fr n labeltext blist offLabel dstbag {valwid 0}} {
    radiobar $fr $n $labeltext $blist $offLabel > $dstbag $valwid
}

# make one in a list a radiobutton bar
proc radiobar {fr n labeltext blist offLabel {varprefix ""} {aname ""} {valwid 0}} {
    global flds
    if {"$aname" != ""} {
	set vname [set aname]($varprefix$labeltext)
    } else {
	set vname $varprefix$labeltext
    }
    global $vname
    
    set $vname ""
    button $fr.labels.$n -text "${labeltext}: " -command "helpMsg $labeltext" \
            -relief flat -padx 0 -pady 0 -borderwidth 0 -width 14 -anchor w -highlightthickness 1
    pack   $fr.labels.$n -side top -pady 0 -padx 0 -anchor w
    
    if {$valwid != 0} {
        label $fr.values.$n -text "[string trim $flds($varprefix$labeltext) " \n\t"]" \
                -relief groove -anchor w -width $valwid -background green -padx 2 -pady 0
        pack  $fr.values.$n -side top -expand true -fill x -pady 0 -padx 0 -anchor w
    }

    frame $fr.bars.$n
    foreach b $blist {
	radiobutton $fr.bars.$n._$b \
		-text $b -relief flat -variable $vname -pady 0 -highlightthickness 0
	# Buttons that say None should set variable to the empty
	# string...
	if {"$b" == "$offLabel"} {
	    $fr.bars.$n._$b configure -value ""
	} else {
	    $fr.bars.$n._$b configure -value $b
	}
	pack $fr.bars.$n._$b -side left -padx 8 -pady 0 -anchor w -ipadx 0
    }
    pack $fr.bars.$n -side top -expand true -fill x -padx 0 -pady 0 -anchor w
}

proc radiobar_set {fr n b} {
    $fr.bars.$n._$b invoke
}

# make one in a list a radiobutton bar
proc checkbar {fr n labeltext blist offLabel} {
    upvar #0 gbag ${labeltext}
    button $fr.labels.$n -text "${labeltext}: " -command "helpMsg $labeltext" \
            -relief flat -width 14 -padx 0 -pady 0 -borderwidth 0 -anchor w -highlightthickness 1
    pack   $fr.labels.$n -side top -pady 0 -anchor w
    frame  $fr.bars.$n
    foreach b $blist {
	checkbutton $fr.bars.$n._$b \
		-offvalue "" \
		-text $b -relief flat -highlightthickness 0 \
		-variable [format "%s(%s)" ${labeltext} ${b}] -pady 0
	# Buttons that say None should set variable to the empty
	# string...
	if {"$b" == "$offLabel"} {
	    $fr.bars.$n._$b configure -onvalue "_ALL_" -offvalue ""
	} else {
	    $fr.bars.$n._$b configure -onvalue $b -offvalue ""
	}
	set gbag($b) ""
	pack $fr.bars.$n._$b -side left -padx 8 -pady 0 -anchor w
    }
    # set active [lindex $blist 0]
    # $fr.bars.$n.$active select
    pack $fr.bars.$n -side top -expand true -fill x -pady 0 -anchor w
}

#
# convert some numeric fields in a 'query-pr --sql'
# record named 'f' in the caller to mnemonic strings
#
proc convertsqlflds {f} {
    upvar 1 $f flds
    foreach a [array names flds] {
	set n $flds($a)
	case $a Severity {
	    case $n 1 { } 2 { } 3 { }
	} Priority {
	    case $n 1 { } 2 { } 3 { }
	}
    }
}

#
# split a pr stream into a tcl array named v
#
# A special array index called _prefix_ contains  all the text prior to
# to the first gnats field
#
proc parsepr {fin varname} {
    upvar 1 $varname fields
    set gnats_tag_exp {^(>[^:]+):(.*)}
    set mail_tag_exp {^([A-Z][^:]+):[ 	]+(.*)}
    set no_gnats_tags_yet 1
    set fields(_prefix_) ""
    set fldtags {_prefix_}

    set leftoverln ""
    while {1} {

	if {"$leftoverln" == ""} {
	    set x [gets $fin ln]
	    if {$x < 0} {
		break
	    }
	} else {
	    set ln $leftoverln
	    set leftoverln ""
	}

	set tag ""
	set val ""

	regexp $gnats_tag_exp $ln matched tag val
	if {"$tag" != ""} {
	    set no_gnats_tags_yet 0
	    # a gnats tag
	    # gnats tags can bu multiline so now
	    # get all the lines 'till the next gnats tag
	    lappend fldtags $tag
	    set fields($tag) "$val\n"
	    while {[gets $fin ln]>=0} {
		set tag2 ""
		regexp $gnats_tag_exp $ln matched tag2 val
		if {"$tag2" != ""} {
		    #  a new gnats tag so we have hit the end of the 
		    # current one.. leave the line we just read in
		    # leftoverln and continue on in the loop
		    set leftoverln $ln
		    break;
		}
		append fields($tag) "$ln\n"
	    }
	    continue
	}
	if {$no_gnats_tags_yet} {
	    append fields(_prefix_) "$ln\n"
	}

	# If we get here the current line is not part of a gnats tag
	# value pair
	set tag ""
	set val ""
	# Here is where we split out regular mail headers if needed.
	regexp $mail_tag_exp $ln matched tag val
	if {"$tag" != ""} {
	    lappend fldtags $tag
	    set fields($tag) "$val\n"
	}
    }

    #
    # Do a little post processing before on the fields we leave
    #
    # For the Reply-To: field make sure the (descriptive name) part of
    #	logname (descriptive name)
    # for the email address is stripped out
    #
    if {![info exists fields(Reply-To)]} {
        lappend fldtags Reply-To
	if {[info exists fields(Sender)]} {
	    set fields(Reply-To) $fields(Sender)
	} else {
	    if {[info exists fields(From)]} {
		set fields(Reply-To) $fields(From)
	    } else {
		set fields(Reply-To) "_unknown_"
	    }
	}
    }
    set fields(Reply-To) [string trim $fields(Reply-To) " \n\t"]

    # If Last-Modified is missing (old GNATS v3.2 problem reports),
    # set it to the file's "mtime".
    # It's never missing, actually, because query-pr adds it if it is.
    # Also, maybe it's not a good idea to load the mtime, since new pr's
    # start out with a blank Last-Modified. We don't want to fill it in.
    
#    if {![info exists fields(>Last-Modified)]} {
#        lappend fldtags >Last-Modified
#        set fields(>Last-Modified) "\n"
#    }
#    if {[string trim $fields(>Last-Modified)] == ""} {
#        global GNATS_ROOT
#        set filename $GNATS_ROOT/[string trim $fields(>Category)]/[string trim $fields(>Number)]
#        set fields(>Last-Modified) "  [clock format [file mtime $filename]]\n"
#    }

    return $fldtags
}

proc write_listbox {lbname fname} {
    set fout [open $fname w]
    set sz [$lbname size]
    for {set x 0} {$x < $sz} {incr x 1} {
	puts $fout [$lbname get $x]
    }
    close $fout
}

proc write_listbox_selection {lbname fname} {
    global Query
    set    fout [open $fname w]
    puts  $fout [$lbname get [$lbname curselection]]
    close $fout
}

proc foreach_listbox {lbname procname} {
    set sz [$lbname size]
    for {set x 0} {$x < $sz} {incr x 1} {
	if {[$procname [$lbname get $x]] != 0} {
	    return
	}
    }
}

proc get_max_strlen { l } {
    set maxlen 0
    foreach e $l {
        if {[string length $e] > $maxlen} {
            set maxlen [string length $e]
        }
    }
    return $maxlen
}

proc build_sort_cmd {fieldnames fieldflgs sortfields} {
    set sz [llength $sortfields]
    set rval "sort -fb -t|"
    for {set x 0} {$x < $sz} {incr x 1} {
	set fname [lindex $sortfields $x]
	set idx [lsearch $fieldnames $fname]
	if {[llength $fieldflgs] > $idx} {
	    set flgs [lindex $fieldflgs $idx]
	} else {
	    set flgs ""
	}
	append rval [format " +%d%s -%d" $idx $flgs [expr $idx+1]]
    }
    return "$rval"
}

proc sort_listbox {lb} {
    set vals [lsort [$lb get 0 end]]
    $lb delete 0 end
    eval $lb insert end $vals
}

proc make_txt_mb {multitextflds} {
    global TkGnats
    set   f [frame .multiline -relief groove  -borderwidth 2]
    
    frame  .mb
    button .mb.lab -text "Text Fields =>" -width 14 -anchor w -command "helpMsg Text-Fields" \
            -relief flat -padx 0 -pady 0 -borderwidth 0
    pack   .mb.lab -side left -anchor w
    set num 0
    foreach x $multitextflds {
	set lbl [string trimleft $x >]
	button .mb.b$num -text "${lbl}:" -command "switch_txt $x {$multitextflds}"
        pack   .mb.b$num -side left -anchor w
        incr num
    }
    button .mb.insert -text "Insert File..." -command "insert_file .multiline.text"
    pack   .mb.insert -side right -anchor e

    pack   .mb -side top -anchor w -fill x -in $f

    text $f.text -height 12 -width 80 -relief sunken -padx 4 -insertwidth 1 \
            -wrap $TkGnats(TextWrap) -yscrollcommand "$f.sby set" -xscrollcommand "$f.sbx set" \
            -insertofftime 400 -borderwidth 2
    set TkGnats(mtextbackround) [$f.text cget -background]
    $f.text configure -background grey95
    
    bind $f.text <Enter> "+focus $f.text"
    bind $f.text <Tab>   ""
    scrollbar $f.sby -command "$f.text yview" -relief sunken

    # Create padding based on the y scrollbar width and border
    frame $f.bottom
    scrollbar $f.sbx -command "$f.text xview" -borderwidth 2 -orient horizontal
    set pad [expr [$f.sby cget -width] + 2 * \
            ([$f.sby cget -bd] + [$f.sby cget -highlightthickness])]
    frame $f.pad -width $pad -height $pad

    pack  $f.pad    -in $f.bottom -side left        
    pack  $f.sbx    -in $f.bottom -side bottom -fill x
    pack  $f.bottom -side bottom -fill x 

    pack  $f.sby    -side left  -fill y
    pack  $f.text   -side right -fill both -expand true
    pack  $f        -side top   -fill both -expand true -pady 4
    
    return "$f.text"
}

proc flush_multitext {} {
    global current_multi_text frm
    set f .multiline
    if {"$current_multi_text" != ""} {
	set frm($current_multi_text) "[string trimright [$f.text get 1.0 end] "\t\n "]\n"
    }
}

proc flush_singletext {lst} {
    global frm
    foreach tag $lst {
	set frm($tag) [string trim [textget $tag] "\t\n "]
    }
}

proc switch_txt {name multitextflds} {
    global current_multi_text frm
    set f .multiline

    # write the current text out back into the frm bag
    flush_multitext

    # reset the currently selected button relief and command
    set num [lsearch $multitextflds $current_multi_text]
    if {$num >= 0} {
        .mb.b$num configure -relief raised \
                -command "switch_txt $current_multi_text {$multitextflds}"
    }

    # load the text widget with the new text
    if {[set state [$f.text cget -state]] == "disabled"} {
        $f.text configure -state normal
    }
    $f.text delete 1.0 end
    $f.text insert 1.0 $frm($name)
    if {$state == "disabled"} {
        $f.text configure -state disabled
    }

    # set the newly selected button relief and command
    set current_multi_text $name
    set num [lsearch $multitextflds $current_multi_text]
    .mb.b$num configure -relief sunken -command "helpMsg [string trimleft $current_multi_text >]"
}

proc save_sort_fields {sortfile sortfields} {
    if {$sortfile != ""} {
        file delete $sortfile
        set fout [open $sortfile "w"]
        puts  $fout "set Query(user_sort_flds) \{$sortfields\}"
        close $fout
    }
}

proc insert_file {w} {
    global TkGnats
    set initialdir ""
    if {[info exists TkGnats(InsertFileDir)]} {
        if {[file isdirectory $TkGnats(InsertFileDir)]} {
            set initialdir $TkGnats(InsertFileDir)
        }
    }
    if {$initialdir == ""} {
        set initialdir [glob ~]
    }
    set file [tk_getOpenFile -initialdir $initialdir -title "Enter filename to insert"]
    if {$file != ""} {
        set fin [open $file r]
        $w insert insert [read $fin]
        close $fin
    }
}

proc file_search_string {file string} {
    if {[catch {open $file r} fin]} {
	return ""
    }
    set    text [split [read $fin] "\n"]
    close  $fin
    return [lindex $text [lsearch -regexp $text $string]]
}

proc file_put_text {file text} {
    if {[catch {open $file w} fout]} {
	return ""
    }
    puts   $fout $text
    close  $fout
    return $file
}

proc email_send {prid textw} {
    global TkGnats bugsval respval origval mail_cc replvals mail_sj
    set addrs ""
    foreach a "bugsval respval origval mail_cc" {
        if {"[string trim [subst $$a]]" != ""} {
            if {"$addrs" != ""} {
                set addrs "$addrs, "
            }
            set addrs "$addrs[string trim [subst $$a]]"
        }
    }
    for {set i 1} {$i <= $replvals(nreps)} {incr i} {
        set a [string trim $replvals($i)]
        if {"$a" != ""} {
            if {"$addrs" != ""} {
                set addrs "$addrs, "
            }
            set addrs "$addrs$a"
        }
    }
    
    set   stamp  $TkGnats(LogName)[clock format [clock seconds] -format "%j:%T"]
    set   tmpfilename /tmp/tksendpr.$stamp
    set   fout [open $tmpfilename "w"]

    puts  $fout "To: $addrs"
    puts  $fout "From: $TkGnats(EmailAddr)"
    puts  $fout "Reply-To: $TkGnats(EmailAddr)"
    puts  $fout "Subject: Re: $prid: $mail_sj"
    puts  $fout "\n"
    puts  $fout "[$textw get 1.0 end]"
    close $fout

    # sendmail errors are mailed back to sender
    catch {eval exec $TkGnats(Mailer) < $tmpfilename} result
    file delete $tmpfilename
}

proc email_originator {to_rep to_res to_org prid synopsis} {
    global TkGnats mail_cc mail_sj bugsval respval origval replvals env

    # expand the Responsible address, which could be an alias in the responsibles file

    set tmp_res [lindex [extract_email_address $to_res] 0]
    set tmp_org [lindex [extract_email_address $to_org] 0]
    set tmp_usr [lindex [extract_email_address $TkGnats(EmailAddr)] 0]

    set responsible_code [catch {my_pr_addr $tmp_res} to_res_addr]
    if {$responsible_code} {
	set to_res_addr ""
	Msg "Could not verify E-mail address of responsible party.\n" "Address used: $tmp_res.\n" \
                "Resulting error(s):\n" "$to_res_addr"
	return -1
    }
    if {"$tmp_res" != "$to_res_addr"} {
        set restmp "$tmp_res: $to_res_addr"
    } {
        set restmp "$tmp_res"
    }
    set to_res "$to_res_addr"
    
    set tlist {}
    
    set t [toplevel .tkgnats_email]
    wm minsize    $t 100 100
    wm title      $t "TkGnats - Send Email to Problem Report: $prid"
    wm iconbitmap $t @$TkGnats(lib)/tkeditpr.xbm
    wm iconname   $t "$TkGnats(LogName)'s tkemailpr [file tail $prid]"
    
    set    f [frame $t.f]
    pack  $f           -side top  -fill both -expand true

    frame $f.ad
    pack  $f.ad        -side top  -anchor n -fill x
    frame $f.ad.lab
    pack  $f.ad.lab    -side left -anchor w -fill y
    frame $f.ad.ent
    pack  $f.ad.ent    -side left -anchor w -fill x -expand true
    
    label $f.ad.lab.to   -relief flat -anchor e -text "To:" -width 8
    pack  $f.ad.lab.to   -side top    -anchor n

    label $f.ad.lab.sj   -relief flat -anchor e -text "Subject:"
    pack  $f.ad.lab.sj   -side bottom -anchor s -pady 2
    
    label $f.ad.lab.cc   -relief flat -anchor e -text "Cc:" -width 8
    pack  $f.ad.lab.cc   -side bottom -anchor s -pady 2
    
    checkbutton $f.ad.ent.to -relief flat -pady 0 -highlightthickness 0 -anchor w \
            -variable bugsval -text "$TkGnats(GNATS_ADDR)" \
            -offvalue "" -onvalue "$TkGnats(GNATS_ADDR)"
    pack        $f.ad.ent.to -side top -anchor w
    
    checkbutton $f.ad.ent.resp -relief flat -pady 0 -highlightthickness 0 -anchor w \
            -variable respval  -text "Responsible   ($restmp)" -offvalue "" -onvalue "$to_res"
    pack        $f.ad.ent.resp -side top -anchor w
    checkbutton $f.ad.ent.orig -relief flat -pady 0 -highlightthickness 0 -anchor w \
            -variable origval  -text "Originator    ($to_org)" -offvalue "" -onvalue "$to_org"
    pack        $f.ad.ent.orig -side top -anchor w

    set reps  [split $to_rep ,]
    set nreps [llength $reps]
    set nrep  0
    catch {unset replvals}
    for {set i 0} {$i < $nreps} {incr i} {
        set rep     [string trim [lindex $reps $i]]
        set tmp_rep [lindex [extract_email_address $rep] 0]
        if {"$tmp_rep" == "$tmp_org"} {
            continue
        }
        incr nrep
#        set replvals($rep)  $nrep
        set replvals($nrep) $rep
        checkbutton $f.ad.ent.repl_$nrep -relief flat -pady 0 -highlightthickness 0 -anchor w \
                -variable replvals($nrep)  -text "Reply-To      ($rep)" -offvalue "" -onvalue "$rep"
        pack        $f.ad.ent.repl_$nrep -side top -anchor w
    }
    set replvals(nreps) $nrep

    lappend tlist [entry $f.ad.ent.cc -relief sunken -borderwidth 2 -background grey95 \
            -textvariable mail_cc]
    pack  $f.ad.ent.cc -side top -anchor w -fill x -expand true
    bind  $f.ad.ent.cc <Enter> "+focus $f.ad.ent.cc"
    
    frame $f.ad.ent.sj
    pack  $f.ad.ent.sj -anchor w -fill x -expand true
    label $f.ad.ent.sj.def -relief sunken -anchor w -text "Re: $prid:"
    pack  $f.ad.ent.sj.def -side left
    lappend tlist [entry $f.ad.ent.sj.ent -relief sunken -borderwidth 2 -background grey95 \
            -textvariable mail_sj]
    pack  $f.ad.ent.sj.ent -side left -anchor w -fill x -expand true
    bind  $f.ad.ent.sj.ent <Enter> "+focus $f.ad.ent.sj.ent"

    scrollbar $f.sb -command "$f.text yview" -relief sunken
    lappend tlist [text $f.text \
	-wrap $TkGnats(TextWrap) \
	-yscrollcommand "$f.sb set" \
	-height 30 -width 80 -relief sunken -padx 4 -insertwidth 1 \
	-insertofftime 400 -borderwidth 2 -background grey95]
    bind  $f.text <Enter> "+focus $f.text"

    set_text_traversal $tlist
    
    pack $f.sb   -side left  -fill y
    pack $f.text -side right -expand true -fill both

    set     b [frame $t.b]
    pack   $b -side bottom -anchor s
    button $b.cancel -text Cancel -command "destroy $t"
    button $b.send   -text Send   -command "email_send $prid $f.text ; destroy $t"
    pack   $b.send   -side left -padx 60 -pady 10
    pack   $b.cancel -side left -padx 60 -pady 10

    focus $f.text

    set bugsval "$TkGnats(GNATS_ADDR)"

    if {$tmp_res != $tmp_usr} {
        set respval $to_res
    } {
        set respval ""
        $f.ad.ent.resp configure -state disabled
    }
    if {$tmp_org != $tmp_usr && $tmp_org != $tmp_res} {
        set origval $to_org
    } {
        set origval ""
        $f.ad.ent.orig configure -state disabled
    }

    # Disable any duplicate recipients
    for {set i 1} {$i <= $nrep} {incr i} {
        set tmp_rep [lindex [extract_email_address $replvals($i)] 0]
        if {$tmp_rep == $tmp_usr || "$tmp_rep" == "$tmp_res"} {
            set replvals($i) ""
            $f.ad.ent.repl_$i configure -state disabled
        }
    }
    
    set mail_cc ""
    set mail_sj "$synopsis"
}

proc entryDialog {msg {no_cancel ""} {initial_value ""}} {
    global  entryDialog_Value
    set entryDialog_Value ""
    catch  {destroy  .entryDialog}
    set t  [toplevel .entryDialog -borderwidth 2 -relief raised]
    message $t.msg -text $msg -aspect 99999
    entry   $t.e   -width 50 -borderwidth 2 -relief sunken
    frame   $t.bar 
    button  $t.bar.ok     -text "OK"     -command "set entryDialog_Value \[$t.e get\]"
    pack    $t.bar.ok -side left -padx 8 -pady 8
    if {$no_cancel == ""} {
        button  $t.bar.cancel -text "Cancel" -command "set entryDialog_Value {}"
        pack    $t.bar.cancel -side left -padx 8 -pady 8
    }
    pack    $t.msg -side top -fill x
    pack    $t.e   -side top -padx 8 -pady 8
    pack    $t.bar -side bottom -anchor center
    bind    $t.e <KeyPress-Return> "set entryDialog_Value \[$t.e get\]"
    bind    $t.e <Enter> "+focus $t.e"
    focus   $t.e
    $t.e insert 0 $initial_value
    set done 0
    while {$done == 0} {
        grab    $t
        tkwait  variable entryDialog_Value
        grab    release $t
        if {$no_cancel != "" && $entryDialog_Value == ""} {
            bell
            Msg "Blank entered!" "You must enter a value."
            set done 0
        } {
            set done 1
        }
    }
    destroy $t
    update idletasks
    return  $entryDialog_Value
}

proc quickfill_entry_from_listbox {ab ew lw vlist} {
    upvar 1 $ab tvar
    if {$tvar == ""} {
        return
    }
    set eidx   [$ew index insert]
    set tmpval [string range [$ew get] 0 [expr $eidx - 1]]
    if {$tmpval != ""} {
        set tvar $tmpval
    }
    set lidx [lsearch -regexp $vlist ^$tvar]
    if {$lidx < 0} {
        set tvar [string range [$ew get] 0 [expr $eidx - 2]]
        set lidx [lsearch -regexp $vlist ^$tvar]
        incr eidx -1
        bell
    }
    $lw selection clear 0 end
    if {$tvar != ""} {
        if {$lidx >= 0} {
            $lw selection set $lidx
            $lw see $lidx
            set tvar [$lw get [$lw curselection]]
        }
    }
    set tmpval   $tvar
    $ew delete 0 end
    $ew insert 0 $tmpval
    $ew icursor  $eidx
    if {$eidx > 0} {
        $ew selection range $eidx [string length $tmpval]
    }
}

proc show_help {title help} {
    set w .help_$title
    catch {destroy $w}
    
    regsub -all "_" $title " " Title

    toplevel  $w
    wm title  $w "$Title Help"

    frame     $w.opts
    pack      $w.opts -side top
    button    $w.opts.quit -text "OK" -command "destroy $w"
    pack      $w.opts.quit -side left -pady 2

    # Create a scrollbar and a text box in the main window.
    scrollbar $w.scrollx -orient horiz -command "$w.text xview"
    pack      $w.scrollx -side bottom -fill x
    scrollbar $w.scrolly -command "$w.text yview"
    pack      $w.scrolly -side right  -fill y
    text      $w.text -relief sunken -bd 2 -yscrollcommand "$w.scrolly set" -font 10x20 \
            -xscrollcommand "$w.scrollx set" -setgrid 1 -height 24 -width 82
    pack      $w.text -side left -fill both -expand yes
    $w.text   insert end $help
    set nlines [lindex [split [$w.text index end] "."] 0]
    set height 24
    if {$height > $nlines} {
        set height $nlines
    }
    $w.text   configure -state disabled -height $height
}

proc extract_email_address {addr} {
    # Supported formats:
    # 01: "Rick Macdonald" <rickm@vsl.com>
    #     Rick Macdonald <rickm@vsl.com>
    # 02: rickm@vsl.com (Rick Macdonald)
    # 03: rickm@vsl.com
    #
    #     01 and 02 return: {rickm@vsl.com} {Rick Macdonald}
    #     03 returns: {rickm@vsl.com} {}
    set fmt01_exp {^(.*[^<]+)(<.*>)}
    set fmt02_exp {^(.*[^\(]+)(\(.*\))}
    set fmt03_exp {(.*)}
    set name    ""
    set address ""
    if {![regexp $fmt01_exp $addr matched name address]} {
        if {![regexp $fmt02_exp $addr matched address name]} {
            set address $addr
        }
    }
    return [list [string trim $address "()<>\"\' 	\n"] \
            [string trim $name "()<>\"\' 	\n"]]
}

proc lock_pr {prid} {
    global TkGnats env GNATS_ROOT

    set full_id [get_pr_full_id $prid]
    if {"$full_id" == ""} {
	wm withdraw .
        Msg "Problem report '$prid' not found!"
        return ""
    }
    
    set pr $GNATS_ROOT/$full_id

    if {![file exists $pr]} {
	wm withdraw .
        Msg "Problem report '$full_id' not found!"
        return ""
    } {
        if {![file readable $pr]} {
            wm withdraw .
	    Msg "Cannot read Problem report '$full_id'"
            return ""
        }
    }

    ### find a username@hostname

    set me $TkGnats(LogName)

    set HOSTNAME ""
    if {[info exists env(HOSTNAME)]} {
        set HOSTNAME $env(HOSTNAME)
    } elseif {[file executable /bin/hostname]} {
        set HOSTNAME [exec /bin/hostname]
    } elseif {[file executable /usr/bin/hostname]} {
        set HOSTNAME [exec /usr/bin/hostname]
    } elseif {[file executable /usr/ucb/hostname]} {
        set HOSTNAME [exec /usr/ucb/hostname]
    } elseif {[file executable /usr/bsd/hostname]} {
        set HOSTNAME [exec /usr/bsd/hostname]
    }

    if {"$HOSTNAME" != ""} {
        set me "$me@$HOSTNAME"
    }

    ### lock the PR and bail out if the lock fails

    catch {exec $TkGnats(pr-edit) --lock $me $full_id} text
    if {$text != ""} {
	wm withdraw .
        if {[lsearch $text exists] < 0} {
	    set who [lindex $text [expr [llength $text] - 1]]
            Msg "Problem report '$full_id' is locked by '$who'"
        } { 
            Msg "GNATS is presently locked.\n" "Please try again in a moment."
        }
        return ""
    }
    
    return $full_id
}

proc unlock_pr {prid} {
    global TkGnats
    catch {exec $TkGnats(pr-edit) --unlock [get_pr_full_id $prid]}
}

proc get_pr_full_id {prid} {
    switch -glob -- "$prid" {
        */* { set full_id $prid }
        *   {
            set fin [open "|query-pr --full [file tail $prid]" r]
            parsepr $fin flds
            close $fin
            set full_id [string trim $flds(>Category)]/$prid
        }
    }
    
    return $full_id
}


# This function couldn't handle categories longer than 16 characters because of the
# format returned by query-pr -i.
proc XXXget_pr_full_id {prid} {
    switch -glob -- "$prid" {
        */* { set full_id $prid }
        *   {
            if {[catch {exec query-pr -i $prid} q]} {
                return ""
            }
            set l [split $q "|"]
            set llen [llength $l]
            if {$llen != 14} {
                Msg [concat "tkgnats: warning: PR $prid \"$q\" has $llen fields." \
                        "It should have 14 fields. Have the gnats " \
                        "administrator check the index file for bogus entries." \
                        "(Especially for |'s in the Synopsis fields)"]
                return ""
            }
            set full_id [string trimright [lindex $l 1] " "]/$prid
        }
    }
    
    return $full_id
}

proc get_pr_state {prid} {
    if {[catch {exec query-pr -i $prid} q]} {
        return ""
    }
    set l [split $q "|"]
    set llen [llength $l]
    if {$llen != 14} {
        puts stderr "tkgnats: warning: PR $prid \"$q\" has $llen fields."
        puts stderr "It should have 14 fields. Have the gnats "
        puts stderr "administrator check the index file for bogus entries."
        puts stderr "(Especially for |'s in the Synopsis fields)"
        return ""
    }
    set state [string trimright [lindex $l 7] " "]
    
    return $state
}

proc delete_pr {prid} {
    global TkGnats GNATS_ROOT

    set INDEX $GNATS_ROOT/gnats-adm/index

    set full_id [get_pr_full_id $prid]
    if {"$full_id" == ""} {
	#####wm withdraw .
        Msg "Error accessing problem report '$prid'!"
        return ""
    }

    set pr $GNATS_ROOT/$full_id

    if {![file writable $INDEX] || ![file writable $pr]} {
        Msg "You don't have proper file permissions to delete $full_id."
        return ""
    }

    set  state [get_pr_state $full_id]
    if {"$state" != "5"} {
        Msg "Problem reports must be closed before deleting."
        unlock_pr $full_id
        return ""
    }
    
    bell
    if {[tk_dialog .tkquerypr_delete "Confirm_Delete" \
            "This will PERMANENTLY delete this bug report.\n\nAre you sure?" \
            "warning" -1 "Delete Report" "Cancel"] != 0} {
        return ""
    }
    
    if {[lock_pr $full_id] == ""} {
        return ""
    }
    
    # take the relevant line out of the index - it should only be there once

    set stamp tkdeletepr.$TkGnats(LogName).[clock format [clock seconds] -format "%j:%T"]
    # catch {exec egrep -v ^$full_id: $INDEX > $INDEX.$stamp}
    set index [split [file_get_text $INDEX] \n]
    if {"$index" == ""} {
        Msg "Unable to read GNATS index."
        return ""
    }
    set idx [lsearch -regexp $index ^$full_id:]
    if {$idx < 0} {
        Msg "Problem report '$prid' not found in GNATS index!"
        return ""
    }
    set index [lreplace $index $idx $idx]
    set idx [lsearch -regexp $index ^$full_id:]
    if {$idx >= 0} {
        Msg "Problem report '$prid' seems to exist more than once in the GNATS index!"
        return ""
    }
    if {[file_put_text $INDEX.$stamp [join $index \n]] == ""} {
        Msg "Unable to write temporary GNATS index file."
        return ""
    }
    
    # here's where we actually delete the file.
    file rename -force $INDEX $INDEX.bak
    file delete $pr
    file rename -force $INDEX.$stamp $INDEX

    #####set MAIL_AGENT $TkGnats(Mailer)
    #####mail_to="gnats-admin"
    #####if [ -n "$HOSTNAME" -a ! -z "$SHOW_HOST" ]; then
    #####  me="$me@$HOSTNAME"
    #####  tmp=""
    #####  for i in $mail_to
    #####  do
    #####    if [ 'echo $i | grep -s "@"' -eq "0" ]; then
    #####      if [ "$tmp" != "" ]; then
    #####	tmp="$tmp $i@$HOSTNAME"
    #####      else
    #####	tmp="$i@$HOSTNAME"
    #####      fi
    #####    fi
    #####  done
    #####  if [ "$tmp" != "" ]; then
    #####    mail_to=$tmp
    #####  fi
    #####fi
    #####$MAIL_AGENT << __EOF__
    #####To: $mail_to
    #####From: $me
    #####Subject: Deleted PR $full_id
    #####
    #####__EOF__
    
    # call PR_EDIT on the new file and clean up
    unlock_pr $full_id
    
    return "All references to $full_id now deleted."
}

proc chk_fld {fldname val {flag_if_missing 1}} {
    global mlist
    upvar 1 $fldname fld
    if {![info exists fld]} {
	if {$flag_if_missing} {
	    append mlist $fldname
	}
	set fld $val
    }
}

proc load_field_defaults {field_array} {
    global TkGnats mlist
    
    upvar 1 $field_array field

    set mlist {}
    
    chk_fld field(>State)	  open
    chk_fld field(>Confidential)  no
    chk_fld field(>Severity)	  serious
    chk_fld field(>Priority)	  medium
    chk_fld field(>Class)	  sw-bug
    chk_fld field(>Arrival-Date)  [clock format [clock seconds]]
    chk_fld field(>Last-Modified) None
    chk_fld field(>Originator)	  Unknown
    chk_fld field(>Responsible)	  gnats
    chk_fld field(>Category)	  pending
    chk_fld field(>Synopsis)	  None
    chk_fld field(>Release)	  Unknown
    chk_fld field(>Description)	  None
    chk_fld field(>Environment)	  "\n"
    chk_fld field(>Audit-Trail)	  "\n"
    chk_fld field(>How-To-Repeat) "\n"

    chk_fld field(Reply-To)       nobody

    # It's ok if these are missing
    chk_fld field(>Unformatted)	 "\n" 0
    chk_fld field(>Fix)		 "\n" 0

    if {$TkGnats(ReleaseBased)} {
	chk_fld field(>Keywords) "\n"
    }

    return $mlist
}
