#ifndef RTC_H
#define RTC_H

/* srikanth, 991001, stuff common to infrtc.c (inferior code) and
   gdbrtc.c (superior code)
*/

/* For each allocated block in the heap, we gather some information
   into a chunk_info structure as described below. These chunk_infos
   are allocated separately (not as a part of the block itself) for
   a number of reasons :

   (1) We don't want buggy programs to corrupt the chunk_info : Well,
       buggy programs can corrupt most anything kept anywhere :-) but
       the odds are better for us this way.

   (2) This separation allows us to shutdown RTC at will and free up
       the resources. This is not done as of now, but could be taken
       up in the future.

   (3) Third and most important, it allows us to treat all allocation
       primitives in a uniform manner : For example, the allocation
       functions valloc() and mmap() are supposed to return a block
       aligned at a page boundary. There is no easy way we could do
       that if we prepend the chunk_info to the block itself. It is
       impossible to append the chunk_info the block.

   (4) Certain operations should have better paging and caching
       behavior this way. For example, after the mark step, when
       we sweep, we will "touch" fewer pages.
*/

typedef struct rtc_chunk_info {

    /* kilroy was here ?  */
    unsigned scanned     : 1;

    /* lest we flog a dead horse ... */
    unsigned old_leak    : 1;

    /* Is this a heap block that the application could free ? For the
       sake of simplicity, we treat mmapped regions and sbroken regions
       as though they are mallocated blocks. However if the user
       attempts to free any of these, we don't want to be caught
       napping.
    */
    unsigned heap_block  : 1;

    /* Is this block padded with guard bytes at both ends ? Only
       heap blocks allocated via malloc, calloc, or realloc are paddable.
       For the sake of simplicity, we treat mmapped regions, shmat blocks
       and sbroken regions as though they are mallocated blocks. However
       we don't want to write cookie bytes in the "red zone" in these cases.
       Likewise for valloc blocks. Attempting to allocate pad bytes in all
       these cases will lead to alignment problems.
    */
    unsigned padded_block  : 1;
    
    /* JAGae73849: Keeps track of allocation that were done even before 
       librtc can be initialized. These were extracted from the kernel using 
       pstat_getprocvm in function find_mmaps_in_startup. These allocations
       are explicitly marked during mark phase.
    */ 
    unsigned preinit_mmap  : 1;

    unsigned new  : 1;

    /* If this identifies a leak, where are its kith and kin ? While
       this may appear to be wasteful, it is necessitated by a need to
       avoid mallocation while collecting garbage. There is no telling
       where the other threads are at that time and if any of them
       is stopped inside a thread-safe mallocator, the garbage
       collector may never halt and that would be a very bad thing.
    */ 
    struct rtc_chunk_info * next_leak;

    /* Where is the chunk_info for the next block in the page ? */
    struct rtc_chunk_info * next;

    /* Pointer to the first byte of mallocated block of space. We
       can safely assume that it is aligned suitably to hold a
       pointer. 
    */
    char * base;        

    /* Pointer *past* the last byte. We can't assume anything
       about its alignment. *end is certainly, absolutely, most
       definitely not a part of this block.
    */
    char * end;         

    /* The allocating stack trace. */
    void ** pc;

} rtc_chunk_info;

/* Event codes for internal as well as external errors. When the
   RTC module finds something amiss, it calls the function rtc_event()
   and passes one of the event codes below. The debugger should have
   a breakpoint in this routine to detect errors.
*/

enum rtc_event {
    RTC_FCLOSE_FAILED = -5,
    RTC_FOPEN_FAILED = -4,
    RTC_UNSAFE_NOW = -3,
    RTC_MUTEX_LOCK_FAILED = -2,
    RTC_NOT_RUNNING = -1,
    RTC_NO_ERROR = 0,
    RTC_NO_MEMORY,
    RTC_BAD_FREE,
    RTC_BAD_REALLOC,
    RTC_ALLOCATED_WATCHED_BLOCK,
    RTC_DEALLOCATED_WATCHED_BLOCK,
    RTC_SBRK_NEGATIVE,
    RTC_STACK_MISSING,  /* not missing so much as we can't locate it */
    RTC_MALLOC_MISSING,
    RTC_BAD_HEADER,
    RTC_BAD_FOOTER,
    RTC_HUGE_BLOCK,
    RTC_HEAP_GROWTH,
    RTC_NOMEM, /* if null-check is enabled and malloc returns null */
    RTC_MEM_NULL, /* JAGaf48255 - gdb should report if malloc returns null pointer */
};

#define DEFAULT_MIN_LEAK_SIZE 0
#define DEFAULT_FRAME_COUNT 4
#define RTC_MAX_THREADS (8 * 1024)
#define DEFAULT_REPEAT_CNT  100


/* From gdbrtc.c */
extern void prepare_for_rtc ( int );
extern boolean threaded_program ( void );
extern void snoop_on_the_heap ( void );
extern void decode_nomem_event ( void );
extern void print_context_for_overrun ( char *, CORE_ADDR , size_t , int );
extern void decode_rtc_event ( void );
extern void set_rtc_catch_nomem ( void );
extern void set_command_line_call_in_inf ( int );
extern void _initialize_gdbrtc ( void );

#endif

