/* Srikanth, May 17th 2004: hprof.c : routines to gather PC samples
   and record the call stack.

   The hprof functionality is implemented in three components :

	(a) Profile engine : This is based on gdb. The profile engine is
            responsible for gathering PC samples and call stacks. It
            simply records these data in ASCII text form in the output
            file specified by the driver.

	(b) Driver/Command : This is the user's interface to this
            capability. The driver (hprof command) is responsible
            for command line option processing, invoking the engine
            with suitable options and when raw profile collection is
	    complete, for processing the raw samples and computing the
            hierarchical profile.

	(c) Helper library : This is a small library with a single
            function and a set of variables using which profile could
            be triggered and controlled.

	From a release stand point, (b) & (c) will be released along
        with prospect and not with gdb.

	Note : The wrapper would invoke gdb as :

	gdb -nx -profile --quiet [profile options] a.out [pid] --commands=file
	where file would contain

	run [args] || continue
	quit

	based on whether we are starting the process or attaching to it.
*/

#include "defs.h"
#include "gdb_string.h"
#include <ctype.h>
#include "symtab.h"
#include "frame.h"
#include "inferior.h"
#include "breakpoint.h"
#include "gdb_wait.h"
#include "gdbcore.h"
#include "gdbcmd.h"
#include "target.h"
#include "gdbthread.h"
#include "annotate.h"
#include "symfile.h"		/* for overlay functions */
#include "top.h"
#include <signal.h>
#include "inf-loop.h"
#include "objfiles.h"
#include "environ.h"

/*******************  Hierarchical profile : Basics **************

    Basically by executing a setitimer call in the context of the
    target process, we can request the kernel to allow us to sample
    the PC values in the program at periodic intervals. The interval
    itself is specified in the call to setitimer. Anytime the specified
    interval expires, the kernel delivers a signal to the process.
    GDB can now intercept the signal, unwind the stack to discover
    the program context and record it.

    By aggregating samples over the run of the program, we can build
    a "hierarchical profile" that shows the samples attributed
    to various functions in the program and their callers.

*********************************************************************/

/******************* External References ****************************/

	/* From breakpoint.c */
extern struct breakpoint *create_rantomain_event_breakpoint ( int );
extern void remove_rantomain_event_breakpoint ( int );

	/* From thread.c */
extern void foreach_live_thread ( void (*function )(void *), void *);

/******************* End External References ************************/

/******************* Profile parameters **************************/

int profile_on = 0;   /* Are we to profile the application ? */

/* We have three choices for the profiling method : sample in real
   time, sample in process virtual time (CPU time) and process
   total time (CPU + SYS). Real time is not very useful for us
   as the timer runs even when the target process is stopped or
   context switched out.

   We allow the user to choose between the other two via
   command line options.
*/
int profiler_signal = TARGET_SIGNAL_PROF;

/* Sampling interval is set to 10 ms by default. User can change
   this via command line option. Less the interval, more the samples,
   and so greater the accuracy, but slower the program runs.
*/
static int sampling_interval = 10;

static char * profile_output_file = "hprof.out";
static FILE * profile_fp;

#define IPF32_PROFILE_HELPER_LIBRARY "/opt/prospect/lib/hpux32/libhprof.so"
#define IPF64_PROFILE_HELPER_LIBRARY "/opt/prospect/lib/hpux64/libhprof.so"
#define PA32_PROFILE_HELPER_LIBRARY  "/opt/prospect/lib/libhprof.sl"
#define PA64_PROFILE_HELPER_LIBRARY  "/opt/prospect/lib/pa20_64/libhprof.sl"

static struct objfile * profile_helper_objfile;
static char * profile_helper_library;

extern int profile_test_on;  /* Are we running under a testsuite ? */

/* Command line control of profile parameters : There is very little
   validation done here. This is because, gdb is supposed to be invoked
   by the wrapper *after* validating its input. In any case, we default
   to sane values if something is wrong.
*/
void set_sampling_interval(char * optarg)
{
    sampling_interval = atoi (optarg);

    /* hpux allows no less than 10ms period for sampling. */
    if (sampling_interval < 10)
      sampling_interval = 10;
}

void set_profile_output_file(char * optarg)
{
    if (optarg && optarg[0])
      profile_output_file = strdup (optarg);
}

void set_sampling_timer_type(char * optarg)
{
    if (optarg)
      {
        if (!strcmp (optarg, "virtual"))
          profiler_signal = TARGET_SIGNAL_VTALRM;
        else
        if (!strcmp (optarg, "profile"))
          profiler_signal = TARGET_SIGNAL_PROF;
       }
}

static void handle_signals_discretly ()
{
  int i;

  /* While profiling, silently and immediately pass the
     signal to the program. In the case of the profiling
     signal itself, we should not pass it to the target.

     This silent and instant delivery is needed for hiding
     the underlying implementation.
  */
  for (i = 0; i < TARGET_SIGNAL_LAST; i++)
    {
      signal_stop_update (i, 0);
      signal_print_update(i, 0);
      if (i == profiler_signal || i == TARGET_SIGNAL_TRAP)
        signal_pass_update(i, 0); /* not for program's consumption */
      else   
        signal_pass_update(i, 1); /* Pass to program */
    }

  /* If we adopted a running process, alert the kernel to the
     changed signal disposition.
  */
  if (target_has_execution)
    require_notification_of_events (inferior_pid);
}

static char *
identify_helper_library ()
{
  char * override_library;

#ifdef HP_IA64
  if (is_swizzled)
    profile_helper_library = IPF32_PROFILE_HELPER_LIBRARY;
  else
    profile_helper_library = IPF64_PROFILE_HELPER_LIBRARY;
#else
#ifndef GDB_TARGET_IS_HPPA_20W
  profile_helper_library = PA32_PROFILE_HELPER_LIBRARY;
#else 
  profile_helper_library = PA64_PROFILE_HELPER_LIBRARY;
#endif
#endif

  /* Check for override. */
  if ((override_library = getenv("LIBHPROF_SERVER")) != NULL)
    profile_helper_library = strdup (override_library);

  return profile_helper_library;
}

void
prepare_for_profiling ()
{
  set_in_environ (inferior_environ, "LD_PRELOAD", identify_helper_library());

  /* Now plant a breakpoint in main and when it triggers,
     we can initiate profiling.
  */
  create_rantomain_event_breakpoint (0);

  handle_signals_discretly ();
}


static struct objfile * profile_helper_available (void)
{
    struct objfile * objfile;

    ALL_OBJFILES (objfile)
        if (!strcmp (objfile->name, profile_helper_library))
          return objfile;
        else
        if (strstr (objfile->name, "libhprof.s"))
          return objfile;

    return 0;
}

void initiate_profiling (int adopted)
{
    value_ptr funcval;
    struct minimal_symbol * m;
    CORE_ADDR anaddr;
    char buf[4];

    if ((profile_fp = fopen (profile_output_file, "w")) == NULL)
      {
        printf_unfiltered ("hprof: error creating output file %s\n",
                                        profile_output_file);
        gdb_flush (gdb_stdout);
        exit (1);
      }

    /* First get rid of the internal breakpoint. This is not needed
       when the process is adopted
    */
    if (!adopted)
      remove_rantomain_event_breakpoint (0); 
    else
      {
        handle_signals_discretly ();
        identify_helper_library ();
      }

    /* If we are running under the testsuite, then there is
       nothing we have to do to initate the profile.
       The program itself generates alarm signals explicitly
       to avoid nondeterminism that would be associated with
       interval timers.
    */
    if (profile_test_on)
      return;

    if ((profile_helper_objfile = profile_helper_available ()) == 0)
      {
        printf_unfiltered ("hprof: profiler helper library not loaded, profiling disabled.\n");
        printf_unfiltered ("You may need to explicitly link with %s\n",
                                     profile_helper_library);
        gdb_flush (gdb_stdout);
        exit(1);
      }

    /* Change the profile parameters if needed. */
    if (profiler_signal != TARGET_SIGNAL_PROF)
      {
        m = lookup_minimal_symbol ("__use_virtual_timer", NULL,
                        profile_helper_objfile);
        if (!m)
          error ("Internal error : Helper library is of no help!");

        anaddr = SYMBOL_VALUE_ADDRESS (m);
        store_unsigned_integer (buf, 4, 1); /* use ITIMER_VIRTUAL */
        target_write_memory (anaddr, buf, 4);
      }   

    if (sampling_interval != 10)
      {
        m = lookup_minimal_symbol ("__sampling_interval", NULL,
                        profile_helper_objfile);
        if (!m)
          error ("Internal error : Helper library is of no help!");

        anaddr = SYMBOL_VALUE_ADDRESS (m);
        store_unsigned_integer (buf, 4, sampling_interval);
        target_write_memory (anaddr, buf, 4);
      }

    /* If we attached to this process, then single step the
       process once, before issuing the command line call.
       This is needed for two reasons (a) Command line calls
       fail immediately upon attach because we restore the
       stale space register after the kernel has updated the
       code space register for copy on debug. (b) If the
       thread is stopped in a system call, then we have to
       step the process out of the syscall in order to be able
       to call.
    */
    if (adopted)
      {
        struct target_waitstatus w;

        target_resume (-1, 1, 0);
        registers_changed ();
        target_wait (inferior_pid, &w);
      }
    /* Now cause setitimer to be called */
    funcval = find_function_in_inferior ("__kickoff_interval_timer");
    call_function_by_hand (funcval, 0, 0);
}

void terminate_profiling ()
{
    if (fclose (profile_fp) == EOF)
      {
        perror ("hprof:");
        exit (1);
      }

    if (!profile_test_on)  /* This exit confuses dejagnu so thoroghly */
      exit (0);
}


static int record_call_stack (PTR unused)
{
  struct frame_info *fi;
  struct minimal_symbol * m;
  static int sample_count;
  char pc_string[32];
  char entry_pc_string[32];
  struct symtab_and_line sal;
  asection *section = 0;
  char * filename = 0;
  int line = -1;
  int spos_available;
  unsigned long long offset;

  fprintf (profile_fp, "Sample # : %d\n\n", ++sample_count);

  for (fi = get_current_frame(); fi; fi = get_prev_frame(fi))
    {
      m = lookup_minimal_symbol_by_pc (fi->pc);

      sprintf (pc_string, sizeof (CORE_ADDR) == 4 ? "0x%08x" : "0x%016llx", 
                             fi->pc);

      offset = fi->pc - SYMBOL_VALUE_ADDRESS(m);
      spos_available = 0;
      sal = find_pc_sect_line (fi->pc, section, 0);

      if (sal.symtab)
        {
          filename = sal.symtab->filename;
          line = sal.line;
          spos_available = 1;
        }

      fprintf (profile_fp, "\tin %s + %d",
               pc_string, (int) offset);

      if (spos_available)
        fprintf (profile_fp, " [%s:%d]", getbasename (filename), line);

      fprintf (profile_fp, " %s\n", SYMBOL_SOURCE_NAME (m));
     
      /* FIXME: PA debugger goes into tailspin when unwinding
         from sigprocmask. No idea why. (CR to be raised)
      */
      if (m && !strcmp (SYMBOL_SOURCE_NAME(m), "sigprocmask"))
          break;
    }
  fprintf (profile_fp, "\n");
  fflush (profile_fp);

  return 0;
}

static void captured_record_call_stack (void * unused)
{
  extern int donot_print_errors;

  donot_print_errors = 1;
  catch_errors (record_call_stack, 0, "", RETURN_MASK_ALL);
  donot_print_errors = 0;
}

void
gather_pc_sample ()
{
  foreach_live_thread (captured_record_call_stack, 0);
}


#ifdef JUST_FOR_REFERENCE

/* Srikanth, This is included here, just for reference and not meant
   to be compiled into gdb. From here to EOF is the implementation of
   the helper library.
*/

#include <signal.h>
#include <sys/time.h>

/* Hierarchical Profile Helper Library : This library will be 
   LD_PRELOADED when the program is started under the debugger (hprof)
   and the user has to link with it or otherwise preload it in
   the attach case.

   All symbols are hidden from the dynamic linker to avoid any
   namespace pollution and related problems. However the static
   linker symbol table should not be stripped, as GDB has to
   update the variables here and call functions here.

   -- Srikanth, Jun 30th 04.
*/

/* GDB will update the following variables per user wishes. */
int __sampling_interval = 10;  /* Milli seconds. */
int __use_virtual_timer = 0;   /* ITIMER_PROF is the default */

static void handler()
{
    /* This is just a dummy stub. GDB will never deliver the profiling
       signal to the target process.
    */
    return;
}

/* GDB helper function to start the interval timer */
__kickoff_interval_timer ()
{

   struct itimerval rttimer;
   struct sigaction act;

   act.sa_handler = handler;

   /* We don't want this signal to interrupt system calls and
      make them fail with an EINTR : we don't know how to
      restart the interrupted system calls in gdb, so best
      not to get into this situation in the first place.
   */
   act.sa_flags = SA_RESTART;
   sigemptyset(&act.sa_mask);

   sigaction (__use_virtual_timer ? SIGVTALRM : SIGPROF, &act, 0);

   /* After a gap of the sampling period ... */
   rttimer.it_value.tv_sec     = 0;
   rttimer.it_value.tv_usec    = __sampling_interval * 1000;

   /* Then on for every sampling period ... */
   rttimer.it_interval.tv_sec  = 0;
   rttimer.it_interval.tv_usec = __sampling_interval * 1000;

   /* Let the timer trigger ... */
   if (setitimer (__use_virtual_timer ? ITIMER_VIRTUAL : ITIMER_PROF, 
                                      &rttimer, 0) == -1)
           perror ("setitimer");
}
#endif
