// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// RefCountedBlockPtr test code.
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Utilities/RefCountedBlockPtr.h"
#include "Utilities/ElementProperties.h"
#include "Utilities/PAssert.h"
#include "Utilities/Tester.h"

#include <iostream>

// This specialized version of ElementProperties makes a deep copy
// of a RefCountedBlockPtr.

template <class T, bool b>
struct ElementProperties< RefCountedBlockPtr<T,b> >
  : MakeOwnCopyProperties< RefCountedBlockPtr<T,b> >
{ };

typedef RefCountedBlockPtr<double,true> RCBlock_t;
typedef RefCountedBlockPtr<double,false> RCFBlock_t;
typedef RefCountedBlockPtr<RCBlock_t,true> RCBlock2D_t;

void print(const RCBlock_t &b, Pooma::Tester &);

int main(int argc, char* argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);
  int test_number = 0;

#if POOMA_EXCEPTIONS
  try {
#endif
    tester.out() << "\nTest that Block<Block<T> > works" 
		 << std::endl;

    RCBlock_t p(10);
    for (int i = 0; i < 10; i++)
      p[i] = (i-5)*(i-5);

    print(p,tester);

    tester.out() << std::endl;

    PAssert(!p.isShared());

    test_number = 1;

    RCBlock2D_t a(5,p);

    test_number = 2;

    PAssert(!p.isShared());

    test_number = 3;

    for (int i = 0; i < 5; i++)
      { PAssert(!a[i].isShared()); }

    test_number = 4;

    for (int i = 0; i < 5; i++)
      print(a[i],tester);

    test_number = 5;

    for (int i = 0; i < 5; i++)
      a[i][3] = -1;

    test_number = 6;

    for (int i = 0; i < 5; i++)
      print(a[i],tester);

    {
      RCBlock2D_t b = a;

      PAssert(b.isShared());
      PAssert(a.isShared());
      
      // Hmmm. The subarrays themselves will NOT show up
      // as being shared. This is because they are NOT. Only
      // the outer block controller was copied. Each copy of
      // the outer controller has a pointer to a single
      // inner block controller for each row. This is
      // a bit confusing, but can it actually cause problems?

      for (int i = 0; i < 5; i++)
        { 
	  PAssert(!b[i].isShared());
	}

      for (int i = 0; i < 5; i++)
	{ 
	  PAssert(!a[i].isShared()); 
	}
      
      b.makeOwnCopy();
      
      PAssert(!b.isShared());
      PAssert(!a.isShared());
      
      b[0][0] = 0;
      b[0][1] = 0;
      b[1][1] = 0;
      b[1][2] = 0;
      b[2][2] = 0;
      b[2][3] = 0;
      b[3][3] = 0;
      b[3][4] = 0;
      b[4][4] = 0;
      b[4][5] = 0;
      
      for (int i = 0; i < 5; i++)
        for (int j = 0; j < 10; j++)
          if (j > i) b[i][j] = 0;
#if POOMA_EXCEPTIONS      
      try {
        b[5][5] = 0;
        throw "Bounds checking failed!";
      } 
      catch(const Pooma::Assertion &) {
	tester.out() << "Bounds check worked." << std::endl; 
      }
#endif
      for (int i = 0; i < 5; i++)
        print(a[i],tester);
      
      for (int i = 0; i < 5; i++)
        print(b[i],tester);
        
      RCBlock2D_t c = a;
      
      PAssert(a.isShared());
      PAssert(c.isShared());   
      
    }

    PAssert(!a.isShared());
     
    for (int i = 0; i < 5; i++)
      { 
	PAssert(!a[i].isShared()); 
      }
#if POOMA_EXCEPTIONS      
  }
  catch(const char *err) 
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
  catch(const Pooma::Assertion &err)
    { 
      tester.exceptionHandler( err );
      tester.set( false );
    }
#endif    
  tester.out() << "All Done!" << std::endl;
  int res = tester.results("rcblock_test3");
  Pooma::finalize();
  return res;
}

void print(const RCBlock_t &b, Pooma::Tester &tester)
{
  for (int i = 0; i < 10; i++)
    tester.out() << b[i] << " ";

  tester.out() << std::endl;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: rcblock_test3.cpp,v $   $Author: swhaney $
// $Revision: 1.10 $   $Date: 2000/03/07 13:18:33 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
