// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes:
// MPDynamicUniform, MPDynamicSpatial,
// MPRemoteDynamicUniform, MPRemoteDynamicSpatial
//-----------------------------------------------------------------------------

#ifndef POOMA_PARTICLES_COMMON_PARTICLE_TRAITS_H
#define POOMA_PARTICLES_COMMON_PARTICLE_TRAITS_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// These are some structs which can be used as a particle traits class
// for defining the Particles' attribute engine tag type and layout
// strategy type.  These are provided for the user's convenience, to
// save their having to define it themselves.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include files:
//-----------------------------------------------------------------------------

#include "Geometry/DiscreteGeometry.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

struct DynamicTag;
struct Dynamic;
template <class Tag> struct Remote;
template <class LayoutTag, class PatchTag> struct MultiPatch;
class UniformLayout;
template <class G, class L> class SpatialLayout;


//-----------------------------------------------------------------------------
// MPDynamicUniform:  MP Dynamic engine for the particle attributes
// and a UniformLayout strategy.  Use MP Dynamic for single context only.
//-----------------------------------------------------------------------------

struct MPDynamicUniform
{
  typedef MultiPatch<DynamicTag,Dynamic> AttributeEngineTag_t;
  typedef UniformLayout                  ParticleLayout_t;
};


//-----------------------------------------------------------------------------
// MPDynamicSpatial:  MP Dynamic engine for the particle attributes
// and a SpatialLayout strategy.  Use MP Dynamic for single context only.
// The template parameters are the field element centering on the mesh,
// the mesh type, and the field layout type.
//-----------------------------------------------------------------------------

template <class Centering, class MeshType, class FL>
struct MPDynamicSpatial
{
  typedef MultiPatch<DynamicTag,Dynamic>       AttributeEngineTag_t;
  typedef DiscreteGeometry<Centering,MeshType> Geometry_t;
  typedef SpatialLayout<Geometry_t,FL>         ParticleLayout_t;
};


//-----------------------------------------------------------------------------
// MPRemoteDynamicUniform:  MP Remote Dynamic engine for particle attributes
// and a UniformLayout strategy.  Use MP Remote Dynamic for multiple contexts.
//-----------------------------------------------------------------------------

struct MPRemoteDynamicUniform
{
  typedef MultiPatch< DynamicTag, Remote<Dynamic> > AttributeEngineTag_t;
  typedef UniformLayout                             ParticleLayout_t;
};


//-----------------------------------------------------------------------------
// MPRemoteDynamicSpatial:  MP Remote Dynamic engine for particle attributes
// and a SpatialLayout strategy.  Use MP Remote Dynamic for multiple contexts.
// The template parameters are the field element centering on the mesh,
// the mesh type, and the field layout type.
//-----------------------------------------------------------------------------

template <class Centering, class MeshType, class FL>
struct MPRemoteDynamicSpatial
{
  typedef MultiPatch< DynamicTag, Remote<Dynamic> > AttributeEngineTag_t;
  typedef DiscreteGeometry<Centering,MeshType>      Geometry_t;
  typedef SpatialLayout<Geometry_t,FL>              ParticleLayout_t;
};



//////////////////////////////////////////////////////////////////////

#endif     // POOMA_PARTICLES_COMMON_PARTICLE_TRAITS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CommonParticleTraits.h,v $   $Author: julianc $
// $Revision: 1.6 $   $Date: 2000/04/27 17:36:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
