// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_PARTICLES_ATTRIBUTE_WRAPPER_H
#define POOMA_PARTICLES_ATTRIBUTE_WRAPPER_H

//-----------------------------------------------------------------------------
// Classes:
//   AttributeWrapper<T>
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview:
//
// AttributeWrapper<T> is a subclass of Attribute that implements
// the basic Attribute interface by passing on the operations in the
// interface to an object of type T that AttributeWrapper wraps.  This is
// basically a standard external polymorphism mechanism for objects of
// various types, for example for wrapping DynamicArray objects.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Include Files
//-----------------------------------------------------------------------------

#include "Particles/Attribute.h"
#include "Utilities/Inform.h"
#include "Utilities/PAssert.h"

#if POOMA_CHEETAH
#include "MatchingHandler/Serialize.h"
#endif

#include <iostream>
#include <fstream>

//-----------------------------------------------------------------------------
// Forward References
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description of AttributeWrapper:
//
// AttributeWrapper<T> is a subclass of the abstract base class Attribute.
// It is templated on a class T, where T should be some form of Array or
// DynamicArray that supports a dynamic data structure interface.
//
// AttributeWrapper is meant to be used as an external polymorphism derived
// class.  You create an AttributeWrapper and give it an object to wrap around;
// the abstract base class is used to provide an abstract interface to a
// heterogenous collection of AttributeWrappers from some other user, for
// example a Particles class.  Particles actually uses an AttributeList object
// to hold a collection of Attributes.
//
//-----------------------------------------------------------------------------

///////////////////////////////////////////////////////////////////////////////
// namespace POOMA {

template<class T>
class AttributeWrapper : public Attribute
{
public:
  //============================================================
  // Typedefs and enumerations
  //============================================================

  typedef Attribute            Base_t;
  typedef AttributeWrapper<T>  This_t;
  typedef T                    Wrapped_t;


  //============================================================
  // Constructors
  //============================================================

  // AttributeWrapper should be initialized with an object of type T
  // that it will wrap.  It will store a reference to this object.

  AttributeWrapper(Wrapped_t &array)
    : wrapped_m(array)
    {
    }


  //============================================================
  // Destructor
  //============================================================

  // AttributeWrapper does not need to do anything in the destructor,
  // since it just wraps a reference to an object.
  virtual ~AttributeWrapper()
    {
    }


  //============================================================
  // Accessors
  //============================================================

  // Return a reference to our wrapped object.

  Wrapped_t &array()
    {
      return wrapped_m;
    }

  const Wrapped_t &array() const
    {
      return wrapped_m;
    }


  //============================================================
  // Public virtual interface
  //============================================================

  // Print the contents to the given Inform stream.

  virtual void print(std::ostream &o) const
    {
      o << array() << std::endl;
    }

  /* Omit this until we have serialize/deserialize functions for DynamicArray.

  // serialize/deserialize the Array using the given stream.

  virtual int serialize(std::ostream &o) const
  {
    return serialize(o,array());
  }

  virtual int serialize(std::fstream &f) const
  {
    return serialize(f,array());
  }

  virtual int deserialize(std::ostream &o)
  {
    return deserialize(array(),o);
  }

  virtual int deserialize(std::fstream &f)
  {
    return deserialize(array(),f);
  }

  */

#if POOMA_CHEETAH

  // packSize, pack and unpack functions for particle swapping

  virtual int packSize(int elems) const
  {
    typedef typename Wrapped_t::Element_t Element_t;
    return ( elems * Cheetah::Serialize<Cheetah::CHEETAH, Element_t>::
	     size(Element_t()) );
  }

  virtual int pack(int pid, const IndirectionList<int> &list,
                   char *buffer) const
  {
    return array().engine().localPatch(pid).pack(list,buffer);
  }

  virtual int unpack(int pid, const Interval<1>& dom, char *buffer)
  {
    return array().engine().localPatch(pid).unpack(dom,buffer);
  }

#endif // POOMA_CHEETAH

private:
  // The object that we're wrapping

  Wrapped_t &wrapped_m;

  // Make copy consructor, default constructor, and operator= private
  // and undefined since they should not be used

  AttributeWrapper()
    {
      PInsist(false, "Called AttributeWrapper<T> default constructor.");
    }

  AttributeWrapper(const This_t &)
    {
      PInsist(false, "Called AttributeWrapper<T> copy constructor.");
    }

  const This_t &operator=(const This_t &)
    {
      PInsist(false, "Called AttributeWrapper<T> operator=.");
      return *this;
    }
};


//-----------------------------------------------------------------------------
//
// A specialization of the Inform traits used to say that AttributeWrapper has
// a print method.
//
//-----------------------------------------------------------------------------

template<class T>
std::ostream &operator<<(std::ostream &o, const AttributeWrapper<T> &attrib)
{
  attrib.print(o);
  return o;
}


// } // namespace POOMA

//////////////////////////////////////////////////////////////////////

#endif // POOMA_PARTICLES_ATTRIBUTE_WRAPPER_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: AttributeWrapper.h,v $   $Author: sa_smith $
// $Revision: 1.11 $   $Date: 2000/05/04 03:17:11 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
