// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef OBJECT_SETS_H
#define OBJECT_SETS_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include <set>
#include <map>

#include <stdio.h>

#include "Utilities/PAssert.h"
//#include "Utilities/Inform.h"
#include "IO/PoomaIO.h"

//-----------------------------------------------------------------------------
// A stub for the pwarn stream.
// Inform.h and this stub are mutually exclusive.
#define pwarn std::cerr

//-----------------------------------------------------------------------------
// Enums
//-----------------------------------------------------------------------------
// This type is used to specify how an object set is opened on construction.
enum ObjectSetOpenMode{Create, Restore};

// This type specifies whether an object set is open or closed.
enum ObjectSetOpenState{Open, Closed};

//-----------------------------------------------------------------------------
// Typedefs
//-----------------------------------------------------------------------------
typedef long ObjectID_t;
typedef long FileLoc_t;
typedef std::set<ObjectID_t> ObjectSubset;
//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------
class ObjectStorageSet;
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// ObjectLoc is a structure containing information 
// that binds object IDs to physical locations.
//-----------------------------------------------------------------------------
struct ObjectLoc{
  ObjectID_t objectID;
  FileLoc_t loc;
};
//-----------------------------------------------------------------------------
// These serializers are used to read and write an ObjectLoc instances
//-----------------------------------------------------------------------------
template <class Stream>
int serialize(Stream& s, const ObjectLoc& object);

template <class Stream>
int deserialize(ObjectLoc& object, Stream& s);

inline int serialSizeof(ObjectLoc&){
  return 2*sizeof(long);
}
//-----------------------------------------------------------------------------
// Storage descriptors are structures containing information about
// files used by object sets and the store/retrieve functions.
//-----------------------------------------------------------------------------
struct StorageDescriptor{
  std::string name;
  std::fstream* file;
};
//-----------------------------------------------------------------------------
// Global store()/retrieve routines are templated on type and are recursively
// called as a result of an initiating call by the object set.
// Signatures:
// 	FileLoc_t store(ObjectStorageSet& set, const T& t) -- will
//   write the object t at the end of the file and return the location.
// 	void retrieve(T& t, const FileLoc& loc, ObjectStorageSet& set) -- will
//   read the object t from the given location.
// These are the generic functions. Specializations must be written
//   if more complex behavior is needed for specific types.
//-----------------------------------------------------------------------------
template <class T>
FileLoc_t store(ObjectStorageSet& set, const T& t);

template <class T>
void retrieve(T& t, const FileLoc_t& loc, ObjectStorageSet& set);
//-----------------------------------------------------------------------------
// AttributeMap is a templated class that contains a multimap from a set of
// attribute keys to a unique object ID. The inverse map is also contained.
// AttributeMap is descended from an untemplated base class AttributeMapBase
// which allows pointers to different maps containing different types of
// keys to be kept in a common container. AttributeDict is the intended
// container.
//-----------------------------------------------------------------------------
// The abstract base class for attribute maps.
//-----------------------------------------------------------------------------
class AttributeMapBase{
 public:
  inline std::string attributeName() const{
    return attributeName_m;}
  
  inline std::string attributeType() const{
    return attributeType_m;}
  
  virtual int size() const=0;
  
 protected:
  std::string attributeName_m;
  std::string attributeType_m;
};

//-----------------------------------------------------------------------------
// The templated attribute map class.
//-----------------------------------------------------------------------------
template <class T>
class AttributeMap : public AttributeMapBase{
 public:
  typedef std::multimap<T,ObjectID_t> Map_t;
  typedef std::pair<const T, ObjectID_t> Value_t;
  typedef typename Map_t::iterator iterator;
  typedef std::pair<iterator,iterator> Range_t;
  
  // Constructors.
  AttributeMap();
  
  AttributeMap(const std::string& attributeName,
	       const std::string& attributeType);
  
  AttributeMap(const AttributeMap& amap);
  
  // Assignment.
  AttributeMap& operator=(const AttributeMap& amap);
  
  // Destructor.
  ~AttributeMap();
  
  // Initializer.
  void init(const std::string& attributeName,
	    const std::string& attributeType);
  
  // Returns the number of entries in the multimap.
  inline virtual int size() const{
    return map_m.size();}
  
  // Inserts and entry in the multimap.
  void insert(const T& t, const ObjectID_t id);
  
  // Returns an iterator over the multimap positioned
  // at the beginning.
  inline iterator begin(){
    return map_m.begin();}
  
  // Returns an iterator over the multimap positioned
  // at the end.
  inline iterator end(){
    return map_m.end();}
  
  // Returns the number of entries associated with the
  // given key.
  inline int count(const T& t){
    return map_m.count(t);
  }
  
  // Clears both the multimap and its inverse map.
  inline void clear(){
    map_m.clear();
    mapInv_m.clear();
  }
  
  // Returns an equal range pair of iterators over
  // the multimap.
  inline Range_t equalRange(const T& t){
    return map_m.equal_range(t);}
  
  // Returns the object ID value associated with the
  // FIRST instance of the given attribute.
  // That is it returns the value of the map
  // attribute ---> ID.
  inline ObjectID_t findAttribute(const T& t){
    Range_t range= equalRange(t);
    return (*range.first).second;}
  
  // Returns the attribute associated with a given object
  // ID. That is, it returns the value of the map
  // ID ---> attribute..
  inline T findID(const ObjectID_t id){
    return mapInv_m[id];}
  
  // Returns a set of object IDs associated with the given
  // attribute value.
  ObjectSubset selectAttribute(const T& t);
  
 private:
  // The multimap from attribute value to object ID.
  std::multimap<T,ObjectID_t> map_m;
  
  // The inverse map from object ID to attribute.
  std::map<ObjectID_t,T> mapInv_m;
  
  // Copy utility for assignment and copy construction.
  void copy(const AttributeMap& amap);
};

//-----------------------------------------------------------------------------
// The AttributeDict class stores pointers to the attribute map base class
// associating them with a given attribute name.
//-----------------------------------------------------------------------------
class AttributeDict{
 public:
  typedef std::map<std::string,AttributeMapBase*> Map_t;
  typedef Map_t::iterator iterator;
  
  // Constructor.	
  AttributeDict();
  
  // Destructor.
  ~AttributeDict();
  
  // Add a new attribute map to the dictionary. This version
  // creates a new map from the newMap() factory method.
  void addAttribute(const std::string& attributeName,
		    const std::string& attributeType);
  
  // Add an attribute map to the dictionary. This version adds
  // a map to the dictionary that has been constructed elsewhere.
  void addAttribute(AttributeMapBase* amap);
  
  // Return the size of the dictionary.
  inline int size() const{return map_m.size();}
  
  // Return an iterator to the beginning of the dictionary.
  inline iterator begin(){return map_m.begin();}
  
  // Return an iterator at the end of the dictionary.
  inline iterator end(){return map_m.end();}
  
  // Returns the number of entries with the given name. Since this
  // is a map the result will be zero or one.
  inline int count(const std::string& attributeName) const{
    return map_m.count(attributeName);
  }
  
  // Returns a pointer to the map associated with the given name.
  // Returns NULL if not found.
  AttributeMapBase* findMap(const std::string& attributeName);
  
  // Returns the number of entries in the named attribute map.
  int size(const std::string& attributeName);
  
  // Empties the dictionary and destroys all existing maps.
  void clear();
  
  // Factory method creates a new map based on type and returns a
  // base class pointer.
  AttributeMapBase* newMap(const std::string& attributeName,
			   const std::string& attributeType);
 private:
  // The dictionary map of base class pointers.
  Map_t map_m;
  
  
};

//-----------------------------------------------------------------------------
// Serializer functions for AttributeMap
//-----------------------------------------------------------------------------
template <class Stream, class T>
int serialize(Stream& s, const AttributeMap<T>& amap);

template <class Stream, class T>
int deserialize(AttributeMap<T>& amap, Stream& s);

template <class T>
int serialSizeof(AttributeMap<T>& amap);
//-----------------------------------------------------------------------------
// ObjectStorageSet is a class that stores objects to a single file resource.
// Multi-file object sets and distributed object sets are built from this. 
//-----------------------------------------------------------------------------
class ObjectStorageSet{
 public:
   
  // Constructor.
  ObjectStorageSet(const std::string& setName,
		   ObjectSetOpenMode mode);
  
  // Destructor.    
  ~ObjectStorageSet();
  
  // Returns the number of objects in the set.
  inline int numObjects() const{
    // Return the size of the object map.
    return objectMap_m.size();
  }
  
  // Returns the name of the object set.
  inline std::string name() const{
    // Return the set name.
    return setName_m;
  }
  
  // Used by store() and retrieve() clients to gain direct access
  // to the data file. Not very safe.
  inline std::fstream* dataFile(){
    // Return a pointer to the data file.
    return dataFile_m.file;
  }
  
  // Returns the type attribute of an object.
  std::string objectType(const ObjectID_t& id);
  
  // Returns the label attribute of an object.
  std::string objectLabel(const ObjectID_t& id);
  
  // Main storage function stores an object instance with no additional
  // attributes and returns the object ID.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    ObjectID_t store(const T& t){
    
    // Generate a type name from the PoomaCTTI functions.
    std::string type= PoomaCTTI(t);
    
    // Obtain a new object ID.
    ObjectID_t id= newID();
    
    // Invoke the global store() function with the object and object set
    //   instances.
    FileLoc_t newLoc= ::store(*this, t);
    
    // Construct a new ObjectLoc instance.
    ObjectLoc object;
    object.objectID= id;
    object.loc= newLoc;
    
    // Enter the new object descriptor in the object map.
    objectMap_m[id]= object;
    
    // Enter the type in the type map.
    AttributeMap<std::string>* typeMap=
      (AttributeMap<std::string>*)
      attributeDict_m.findMap("Type");
    typeMap->insert(type,id);
    
    // Write the location to the recovery file.
    writeLoc(object);
    
    // Write the type attribute to the recovery file.
    writeAttribute(std::string("Type"),
		   std::string("std::string"), id, type);
    
    // Perform a flush check on the transaction.
    flushCheck();
    
    // return the ID.
    return id;
  }
  
  // Augmented store() function accepts a label attribute.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    ObjectID_t store(const T& t, const std::string& objectLabel){
    
    // Store the object using the store() member function
    ObjectID_t newID= store(t);
    
    // Enter the label in the label map.
    AttributeMap<std::string>* labelMap=
      (AttributeMap<std::string>*)
      attributeDict_m.findMap("Label");
    labelMap->insert(objectLabel, newID);
    
    // Write the label attribute to the recovery file.
    writeAttribute(std::string("Label"),
		   std::string("std::string"), newID, objectLabel);
    
    // Perform a flush check on the transaction.
    flushCheck();
    
    // Return the ID.
    return newID;
  }
  
  // Assigns a label to an object that has already been stored.
  void objectLabel(ObjectID_t id, const std::string& objectLabel);
  
  // Return the ID of the FIRST object matching the given label.
  ObjectID_t findLabel(const std::string& objectLabel);
  
  // Main retrieval function instantiates an object given its ID. Returns
  // true or false with success or failure respectively.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    bool retrieve(T& t, ObjectID_t id){
    
    // Check that the id is valid.
    if(objectExists(id)){
      
      // Look up the object descriptor by id.
      ObjectLoc object= objectMap_m[id];
      
      // Generate a type from the target instance.
      std::string targetType= PoomaCTTI(t);
      std::string storedType= objectType(id);
      
      // Check that the target type and stored type are compatible.
      if(storedType!=targetType){
	pwarn<<"Warning: Object with ID "<<id<<" not retrieved. "
	     <<"Target object of type "
	     <<targetType<<" is incompatible with stored type "
	     <<storedType<<"."<<std::endl;
	return false;
      }
      
      // Invoke the global retrieve() function with the given object,
      // location, and object set instances.
      ::retrieve(t, object.loc, *this);
      return true;
    }
    else{
      pwarn<<"Warning: Object not retrieved. No object with ID "
	   <<id<<" exists."<<std::endl;
      return false;
    }
  }
  
  // Augmented retrieval function allows retrieval based on an object
  // label instead of ID.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    ObjectID_t retrieve(T& t, const std::string& objectLabel){
    
    // Look up the object ID by label.
    ObjectID_t id= findLabel(objectLabel);
    if(id!=0){
      
      // Retrieve the object using the ID.
      retrieve(t,id);
      return id;
    }
    else{
      pwarn<<"Warning: Object not retrieved. No object labeled "
	   <<objectLabel<<" exists."<<std::endl;
      return 0;
    }
  }
  
  // Returns the number of objects matching the given label attribute.
  int count(const std::string& objectLabel);	
  
  // Returns a set of IDs of objects matching the given label.
  ObjectSubset selectLabel(const std::string& objectLabel);
  
  // Close the object set.
  void close();
  
  // Flush the object set.
  void flush(){
    dataFile_m.file->flush();
    dictFile_m.file->flush();
    flushCount_m++;
  }
  
  // Set the number of times flushCheck() is called before data and
  // recovery files are actually flushed.
  void setFlushFrequency(int freq){
    flushFreq_m= freq;
  }
  
  // Check to see in an object with the given ID is in the set.
  bool objectExists(ObjectID_t id) const;
  
 private:
  // The name of the object set.
  std::string setName_m;
  
  // Descriptor for the data file.
  StorageDescriptor dataFile_m;
  
  // Descriptor for the dictionary recovery file.
  StorageDescriptor dictFile_m;
  
  // The location in the file header of the pointer to the dictionary
  // location.
  FileLoc_t dictRef_m;
  
  // The location of the dictionary in the file.
  FileLoc_t dictLoc_m;
  
  // The object map.
  std::map<ObjectID_t, ObjectLoc> objectMap_m;
  
  // The attribute dictionary.
  AttributeDict attributeDict_m;
  
  // Indicates whether the object set is open or closed.
  ObjectSetOpenState openState_m;
  
  // The flush frequency and counter parameters.
  int flushFreq_m;
  int flushCount_m;
  
  // Generates new object IDs on demand.
  ObjectID_t newID();
  
  // Clears the state of the object set.
  void clear();
  
  // Creates a new object set.
  void create();
  
  // Restores an existing object set.
  void restore();
  
  // Read and write the data file header.
  void readHeader();	
  void writeHeader();
  
  // Read and write the data dictionary.	
  void readDict();	
  void writeDict();
  
  // Recover the dictionary information when a program
  // failure prevents the dictionary from being written
  // to the file.
  void recoverDict();
  
  // Check the flush counter and flush if the frequency
  // matches.
  inline void flushCheck(){
    if(!(flushCount_m%flushFreq_m)){
      flush();
    }
  }
  
  // Write an object location to the recovery file.
  void writeLoc(ObjectLoc& loc);
  
  // Write an object attribute to the recovery file.
  // This templated member function compiles more dependably when placed
  // in the class declaration block.
  template <class T>
    void writeAttribute(const std::string& attributeName,
			const std::string& attributeType, ObjectID_t id,
			const T& value){
    // Write an attribute record to the recovery file.
    // Write a header to identify as an attribute, then write
    // the attribute information.
    serialize(*dictFile_m.file, std::string("ObjectAttribute"));
    serialize(*dictFile_m.file, attributeName);
    serialize(*dictFile_m.file, attributeType);
    serialize(*dictFile_m.file, id);
    serialize(*dictFile_m.file, value);
  }
  
};
//-----------------------------------------------------------------------------
#include "IO/ObjectSets.cpp"

#endif // OBJECT_SETS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ObjectSets.h,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/07/20 20:47:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo

