// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef BAOPTIONS_H
#define BAOPTIONS_H

#include <iostream>
#include <string>
#include <stdio.h>

// BAOptions - options parsing class.
template <int D>
class BAOptions
{
public:

  double epsilon;           // Purge threshold
  double dt;                // Timestep
  double v[D];              // Propagation velocity
  double dx[D];             // Mesh spacings
  int initialCondition;     // 1=sphere, 2=blimp
  double aspectRatio;       // Ellipsoid aspect ratio;
  bool purge;               // To purge or not to purge
  bool doTextOut;           // Text output of u, every outputIncrement steps
  bool doLuxOut;            // Lux output of u, every outputIncrement steps
  bool doSumOut;            // Output sum(u), every outputIncrement steps
  bool doCompressedFractionOut;// Text output of compressedFraction(u), every "
  int lastTimeStep;         // Last time step
  int outputIncrement;      // Frequency of optional output
  int purgeIncrement;       // How often to purge
  int smoothing;            // Number of smoothing iterations on initial values
  int dim;                  // Dimensionality
  int nPatches[D];          // Number of patches
  int nCells[D];            // Number of cells
  int nVerts[D];            // Number of verts (calculated from nCells)
  int pulseHalfWidthCells;  // Half-width (in cells) of (symmetrical) loaded
                            // pulse
  int finHalfWidthCells;    // Half-width (in cells) of fins.
  int finLengthCells;       // Length (in cells) of fins.
  int finOffsetCellsX;      // Offset of fin back from centroid (in cells)
  int finOffsetCellsYZ;     // Offset of fin away from central axis (in cells)
  std::string progname;     // Name of program.
  
  // Default constructor, sets the default options.
  BAOptions();
  
  // Set options from argc, argv.
  BAOptions(int argc, char *argv[]);
  
  // Prints a summary of the options.
  template <class OStr>
  void print(OStr &out) { ::print(*this, out); }
  
  // Prints a usage message.
  template <class OStr>
  void usage(OStr &out) { ::usage(progname, out); }

private:

  // Helper functions used in parsing options with int and double arguments:
  static bool intArgument(int argc, char **argv, int pos, int &val);
  static bool doubleArgument(int argc, char **argv, int pos, double &val);
  
  // Report bad option value
  // (These are forwarded to globals as I don't want to put the
  // bodies inline here. Fix when CW is fixed.)
  
  template <class OStr>
  static void 
  badValue(OStr &out, const std::string &option) { ::badValue(out,option); }
  
  template <class OStr>
  static void 
  badOption(OStr &out, const char *str, const std::string &option) 
  { ::badOption(out,str,option); }
  
};

template <class OStr>
void usage(const std::string &name, OStr &out);

template <int D, class OStr>
void print(const BAOptions<D> &opts, OStr &out);

// Non-inline function definitions for BAOptions.
  
template <int D>
BAOptions<D>::
BAOptions()
  : epsilon(1.0e-6),
    dt(0.05),
    aspectRatio(6.0),
    purge(false),
    doTextOut(false),
    doCompressedFractionOut(false),
    doLuxOut(true),
    doSumOut(false),
    lastTimeStep(400),
    outputIncrement(1),
    purgeIncrement(1),
    smoothing(7),
    initialCondition(2),
    dim(D)
{
  nCells[0] = 62;
  v[0] = 0.2;
  for (int d = 1; d < D; ++d) {
    v[d] = 0.0;
    nCells[d] = 32;
  }
  for (int d = 0; d < D; ++d) {
      nVerts[d] = nCells[d] + 1;
      dx[d] = 0.05;
      nPatches[d] = 1;
  }
  pulseHalfWidthCells = 2;
  finHalfWidthCells = 1;
  finLengthCells = 2;
  finOffsetCellsX = 8;
  finOffsetCellsYZ = 4;
}
  
template <int D>
BAOptions<D>::
BAOptions(int argc, char *argv[])
{
  // Set the defaults (default copy constructor OK)
  *this = BAOptions();
  progname = argv[0];
    
  // Parse the argument list...
  int i = 1;
  while (i < argc) {
      
    using std::string;
    
    bool hasarg = false;
        
    string arg(argv[i]);
        
    if (arg == "-help") {
      usage(std::cerr);
      exit(0);
    } else if (arg == "-purge") {
      purge = true;
      // Check for optional argument:
      hasarg = intArgument(argc, argv, i+1, purgeIncrement);
      if (hasarg) ++i;	    
    } else if (arg == "-text") {
      doTextOut = true;
    } else if (arg == "-compressedFraction") {
      doTextOut = true;
    } else if (arg == "-lux") {
      doLuxOut = true;
    } else if (arg == "-sum") {
      doSumOut = true;
    } else if (arg == "-steps") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, lastTimeStep);
      if (!hasarg) badOption(std::cerr, 
                             "No value specified for: ", arg);
      ++i;
    } else if (arg == "-smoothing") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, smoothing);
      if (!hasarg) badOption(std::cerr, 
                             "No value specified for: ", arg);
      ++i;
    } else if (arg == "-dim") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, dim);
      if (!hasarg) badOption(std::cerr, 
                             "No value specified for: ", arg);
      ++i;
    } else if (arg == "-out") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, outputIncrement);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-initialCondition") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, initialCondition);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-finHalfWidthCells") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, finHalfWidthCells);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-finLengthCells") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, finLengthCells);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-finOffsetCellsX") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, finOffsetCellsX);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-finOffsetCellsYZ") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, finOffsetCellsYZ);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-cells") {
      // This can be followed by either 1 int or D ints.
      bool hasarg = intArgument(argc, argv, i+1, nCells[0]);
      if (hasarg) {
        ++i;
        if (D > 1) {
          bool moreArgs = intArgument(argc, argv, i+1, nCells[1]);
          if (moreArgs) {
            for (int d = 1; d < D; ++d) {
              hasarg = intArgument(argc, argv, i+1, nCells[d]);
              if (!hasarg) 
                badOption(std::cerr, 
                          "Not enough arguments for: ", arg);
              ++i;
            }
          } else {
            for (int d = 1; d < D; ++d) {
              nCells[d] = nCells[0];
            }
          }
        }
      } else {
        badOption(std::cerr, "No argument specified for: ", arg);
      } 
      pulseHalfWidthCells = nCells[0]/8.0;
    } else if (arg == "-patches") {
      // This can be followed by either 1 int or D ints.
      bool hasarg = intArgument(argc, argv, i+1, nPatches[0]);
      if (hasarg) {
        ++i;
        if (D > 1) {
          bool moreArgs = intArgument(argc, argv, i+1, nPatches[1]);
          if (moreArgs) {
            for (int d = 1; d < D; ++d) {
              hasarg = intArgument(argc, argv, i+1, nPatches[d]);
              if (!hasarg) 
                badOption(std::cerr, 
                          "Not enough arguments for: ", arg);
              ++i;
            }
          } else {
            for (int d = 1; d < D; ++d) {
              nPatches[d] = nPatches[0];
            }
          }
        }
      } else {
        badOption(std::cerr, "No argument specified for: ", arg);
      }            
    } else if (arg == "-epsilon") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = doubleArgument(argc, argv, i+1, epsilon);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-aspectRatio") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = doubleArgument(argc, argv, i+1, aspectRatio);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-dt") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = doubleArgument(argc, argv, i+1, dt);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else if (arg == "-v") {
      // This can be followed by either 1 int or D doubles.
      bool hasarg = doubleArgument(argc, argv, i+1, v[0]);
      if (hasarg) {
        ++i;
        if (D > 1) {
          bool moreArgs = doubleArgument(argc, argv, i+1, v[1]);
          if (moreArgs) {
            for (int d = 1; d < D; ++d) {
              hasarg = doubleArgument(argc, argv, i+1, v[d]);
              if (!hasarg) 
                badOption(std::cerr, 
                          "Not enough arguments for: ", arg);
              ++i;
            }
          } else {
            for (int d = 1; d < D; ++d) {
              v[d] = v[0];
            }
          }
        }
      } else {
        badOption(std::cerr, "No argument specified for: ", arg);
      } 
      pulseHalfWidthCells = nCells[0]/8.0;
    } else if (arg == "-dx") {
      // This can be followed by either 1 int or D doubles.
      bool hasarg = doubleArgument(argc, argv, i+1, dx[0]);
      if (hasarg) {
        ++i;
        if (D > 1) {
          bool moreArgs = doubleArgument(argc, argv, i+1, dx[1]);
          if (moreArgs) {
            for (int d = 1; d < D; ++d) {
              hasarg = doubleArgument(argc, argv, i+1, dx[d]);
              if (!hasarg) 
                badOption(std::cerr, 
                          "Not enough arguments for: ", arg);
              ++i;
            }
          } else {
            for (int d = 1; d < D; ++d) {
              dx[d] = dx[0];
            }
          }
        }
      } else {
        badOption(std::cerr, "No argument specified for: ", arg);
      } 
    } else if (arg == "-pulseHalfWidthCells") {
      if (i+1 == argc) badOption(std::cerr, 
                                 "No value specified for: ", arg);
      hasarg = intArgument(argc, argv, i+1, pulseHalfWidthCells);
      if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
      ++i;
    } else {
      std::cerr << "No such flag: " << arg << std::endl;
      usage(std::cerr);
      exit(0);
    }
                    
    ++i; // next arg
  }
      
  // Do some sanity checks:
  if (lastTimeStep < 1)   
    badValue(std::cerr, "-steps");
  if (outputIncrement < 1 || outputIncrement > lastTimeStep) 
    badValue(std::cerr, "-out");
  if (purgeIncrement < 1 || purgeIncrement > lastTimeStep) 
    badValue(std::cerr, "-purge");

  // Finally, initialize nVerts.    
  for (int d = 0; d < D; ++d) nVerts[d] = nCells[d] + 1;              
}

template <int D>
bool BAOptions<D>::
intArgument(int argc, char **argv, int pos, int &val)
{
  // Make sure there is an argument available
  if (pos >= argc) return false;

  // Make sure the 'pos' argument is a number.  If it starts with a number
  // or -number, it is OK.
  char firstchar = argv[pos][0];
  if (firstchar < '0' || firstchar > '9') {

    // first char is not a number.  Is the second, with the first a '-/+'?
    if ((firstchar != '-' && firstchar != '+') || argv[pos][1] == 0 ||
        (argv[pos][1] < '0' || argv[pos][1] > '9'))
      return false;
  }

  // Get the value and return it in the last argument
  val = atoi(argv[pos]);
  return true;
}

template <int D>
bool BAOptions<D>::
doubleArgument(int argc, char **argv, int pos, double &val)
{
  // Make sure there is an argument available
  if (pos >= argc) return false;

  // Make sure the 'pos' argument is a number.  If it starts with a number
  // or -number, it is OK.
  char firstchar = argv[pos][0];
  if (firstchar < '0' || firstchar > '9') {

    // first char is not a number.  Is the second, with the first a '-/+'?
    if ((firstchar != '-' && firstchar != '+') || argv[pos][1] == 0 ||
        (argv[pos][1] < '0' || argv[pos][1] > '9'))
      return false;
  }

  // Get the value and return it in the last argument
  val = atof(argv[pos]);
  return true;
}

//
// Helper functions: print, usage, badOption, badValue
// 
// To avoid having to put these functions in the class body, I've written 
// them as global template functions. The corresponding member functions
// simply call these.
//

template <int D, class OStr>
void print(const BAOptions<D> &opts, OStr &pout)
{
    
  int d;
    
  pout << "Program name: " << opts.progname << std::endl;
  pout << "Option values: " << std::endl;
  pout << "=====================================================" << std::endl;
    
  pout << "text                = " 
       << (opts.doTextOut ? "true ; " : "false ; ") << std::endl;
  pout << "sum                 = " 
       << (opts.doSumOut ? "true ; " : "false ; ") << std::endl;
  pout << "compressedFraction  = " 
       << (opts.doCompressedFractionOut ? "true ; " : "false ; ") << std::endl;
  pout << "lux                 = " 
       << (opts.doLuxOut ? "true ; " : "false ; ") << std::endl;
  pout << "purge               = " 
       << (opts.purge ? "true ; " : "false ; ") << std::endl;

  pout << "steps               = " << opts.lastTimeStep << std::endl;
  pout << "outSteps            = " << opts.outputIncrement << std::endl;
  pout << "purgeSteps          = " << opts.purgeIncrement << std::endl;
  pout << "smoothing           = " << opts.smoothing << std::endl;
  pout << "dim                 = " << opts.dim << std::endl;
    
  pout << "nCells              = " << opts.nCells[0];
  for (d = 1; d < D; ++d) 
    pout << ", " << opts.nCells[d];
  pout << std::endl;
    
  pout << "nVerts              = " << opts.nVerts[0];
  for (d = 1; d < D; ++d) 
    pout << ", " << opts.nVerts[d];
  pout << std::endl;

  pout << "nPatches            = " << opts.nPatches[0];
  for (d = 1; d < D; ++d) 
    pout << ", " << opts.nPatches[d];
  pout << std::endl;
    
  pout << "epsilon             = " << opts.epsilon << std::endl;

  pout << "v                   = " << opts.v[0];
  for (d = 1; d < D; ++d) 
    pout << ", " << opts.v[d];
  pout << std::endl;

  pout << "dx                  = " << opts.dx[0];
  for (d = 1; d < D; ++d) 
    pout << ", " << opts.dx[d];
  pout << std::endl;

  pout << "dt                  = " << opts.dt << std::endl;
  pout << "initialCondition    = " << opts.initialCondition << std::endl;
  pout << "aspectRatio         = " << opts.aspectRatio << std::endl;
  pout << "pulseHalfWidthCells = " << opts.pulseHalfWidthCells << std::endl;
  pout << "finHalfWidthCells   = " << opts.finHalfWidthCells << std::endl;
  pout << "finLengthCells      = " << opts.finLengthCells << std::endl;
  pout << "finOffsetCellsX     = " << opts.finOffsetCellsX << std::endl;
  pout << "finOffsetCellsYZ    = " << opts.finOffsetCellsYZ << std::endl;
  pout << "=====================================================" << std::endl 
       << std::endl;
}
  
template <class OStr>
void usage(const std::string &name, OStr &out)
{
  out << "Usage: " << name << std::endl
      << " [-cells <nCellsX> [<nCellsY> <nCellsZ>]]" 
      << std::endl
      << " [-patches <nPatchesX> [<nPatchesY> <nPatchesZ>]]" 
      << std::endl
      << " [-steps <lastTimeStep>]"
      << " [-out <outputIncrement>]"
      << " [-text]" 
      << " [-lux]"
      << std::endl
      << " [-purge [<purgeIncrement>]]"
      << " [-smoothing <iterations>]"
      << " [-dim <Dim>]"
      << " [-compressedFraction]" 
      << " [-sum]" 
      << " [-block]" 
      << std::endl
      << " [-epsilon <epsilon>]" 
      << " [-initialCondition <1=sphere, 2=otherShape>]" 
      << " [-aspectRatio <aspectRatio>]" 
      << " [-dt <dt>]" 
      << " [-v <vX> [<vY> <vZ>]]" 
      << " [-dx <dx> [<dy> <dz>]]" 
      << std::endl
      << " [-pulseHalfWidthCells <pulseHalfWidthCells>]" 
      << " [-finHalfWidthCells <finHalfWidthCells>]" 
      << " [-finLengthCells <finLengthCells>]" 
      << " [-finOffsetCellsX <finOffsetCellsX>]" 
      << " [-finOffsetCellsYZ <finOffsetCellsYZ>]" 
      << std::endl;
}

template <class OStr>
void badOption(OStr &out, const char *str, const std::string &option) 
{ 
  out << "Bad option: " << str << option << std::endl;
  exit(1);
}

template <class OStr>
void badValue(OStr &out, const std::string &option) 
{ 
  out << "Bad input value for option: " << option << std::endl;
  exit(1);
}

#endif // BAOPTIONS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BAOptions.h,v $   $Author: swhaney $
// $Revision: 1.4 $   $Date: 2000/03/07 13:17:14 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
