// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   ApplyWeightedFieldStencil - Tag class for defining an engine capable of
//                               applying a Field-based stencil.
//   WeightedFieldStencil      - A wrapper class for a user-defined stencil.
//   Engine                    - Specialization for ApplyWeightedFieldStencil.
//   NewEngine                 - Specializations for ApplyFieldStencil
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_WEIGHTEDFIELDSTENCIL_H
#define POOMA_FIELD_WEIGHTEDFIELDSTENCIL_H

//-----------------------------------------------------------------------------
// Overview:
//
// This file contains the equipment required to write differential operators
// that take the form of stencil objects.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Interval.h"
#include "Engine/Engine.h"
#include "Geometry/DiscreteGeometry.h"
#include "Layout/INode.h"
#include "Layout/Node.h"
#include "PETE/ErrorType.h"

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

template <int Dim>
class DomainLayout;

template<class Functor> class WeightedFieldStencil;

//-----------------------------------------------------------------------------
// Full Description:
// 
// ApplyWeightedFieldStencil is just a tag class for the weighted-
// field-stencil-application engine.
//-----------------------------------------------------------------------------

template <class Functor, class Expression1, class Expression2>
struct ApplyWeightedFieldStencil;


//-----------------------------------------------------------------------------
// Full Description:
// 
// Engine<Dim, T, ApplyFieldStencil<Functor, Expression> > is a specialization
// of Engine for ApplyWeightedFieldStencil<Functor>. It uses the supplied stencil 
// object to apply an arbitrary operation to the input field.
//-----------------------------------------------------------------------------

template<int Dim, class T, class Functor, class Expression1, class Expression2>
class Engine<Dim, T, 
  ApplyWeightedFieldStencil<Functor, Expression1, Expression2> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef ApplyWeightedFieldStencil<Functor, Expression1, Expression2>   
                                                   Tag_t;
  typedef Functor                                  Functor_t;
  typedef Expression1                              Expression1_t;
  typedef Expression2                              Expression2_t;
  typedef Engine<Dim, T, Tag_t>                    This_t;
  typedef This_t                                   Engine_t;
  typedef Interval<Dim>                            Domain_t;
  typedef T                                        Element_t;
  typedef ErrorType                                ElementRef_t;
  typedef typename Expression1_t::Engine_t         ExprEngine_t;
  typedef DomainLayout<Dim>                        Layout_t;

  enum { dimensions = Dim };
  enum { hasDataObject = ExprEngine_t::hasDataObject };
  enum { dynamic = false };
  enum { zeroBased = ExprEngine_t::zeroBased };
  enum { multiPatch = ExprEngine_t::multiPatch };

  //---------------------------------------------------------------------------
  // Construct from a stencil, an input field, a weight field, and a new 
  // total domain.
  // Note: the way domains work with FieldStencils is a little screwy.
  // When originally constructing a FieldStencil, the domain of the engine
  // must match the total domain of the stenciled field in order for the
  // indexing to work. This is why the zeroBased trait above is false.

  Engine(const Functor_t &functor, const Expression1_t &f, 
    const Expression2_t &w, const Interval<Dim> &domain)
    : functor_m(functor), field_m(f), weight_m(w),
      offset_m(0),
      domain_m(domain)
    {
    }

  //---------------------------------------------------------------------------
  // Construct from another ApplyWeightedFieldStencil and an Interval. This is 
  // simpler than with other domains since we just need to bump the offset.

  Engine(const This_t &e, const Interval<Dim> &domain)
    : functor_m(e.functor()), field_m(e.field()), weight_m(e.weight()), 
      offset_m(e.offset()), domain_m(Pooma::NoInit())
    {
      offset_m += domain.firsts();
      for (int d = 0; d < Dim; ++d)
        {
          domain_m[d] = Interval<1>(domain[d].length());
        }   
    }    

  //---------------------------------------------------------------------------
  // Element access via ints for speed.

  inline Element_t read(int i) const 
    {
      return functor_m(field(), weight(),
		       i + offset()[0].first()
		       );
    }
  inline Element_t read(int i, int j) const 
    {
      return functor_m(field(), weight(),
		       i + offset()[0].first(),
		       j + offset()[1].first()
		       );
    }
  inline Element_t read(int i, int j, int k) const 
    {
      return functor_m(field(), weight(),
		       i + offset()[0].first(),
		       j + offset()[1].first(),
		       k + offset()[2].first()
		       );
    }

  inline Element_t read(const Loc<1> &loc) const 
    {
      return functor_m(field(), weight(),
		       loc[0].first() + offset()[0].first()
		       );
    }
  inline Element_t read(const Loc<2> &loc) const 
    {
      return functor_m(field(), weight(),
		       loc[0].first() + offset()[0].first(),
		       loc[1].first() + offset()[1].first()
		       );
    }
  inline Element_t read(const Loc<3> &loc) const 
    {
      return functor_m(field(), weight(),
		       loc[0].first() + offset()[0].first(),
		       loc[1].first() + offset()[1].first(),
		       loc[2].first() + offset()[2].first()
		       );
    }

  inline Element_t operator()(int i) const 
    {
      return read(i);
    }
  inline Element_t operator()(int i, int j) const 
    {
      return read(i, j);
    }
  inline Element_t operator()(int i, int j, int k) const 
    {
      return read(i, j, k);
    }
  inline Element_t operator()(const Loc<Dim> &loc) const 
    {
      return read(loc);
    }

  //---------------------------------------------------------------------------
  // Return the domain.

  inline const Domain_t &domain() const { return domain_m; }
  
  //---------------------------------------------------------------------------
  // Accessors.

  inline const Expression1_t &field() const { return field_m; }
  inline const Expression2_t &weight() const { return weight_m; }
  inline const Loc<Dim> &offset() const { return offset_m; }
  inline const Functor_t &functor() const { return functor_m; }

  //---------------------------------------------------------------------------
  // Need to pass lock requests to the contained engine.

  template<class RequestType>
  inline
  typename DataObjectRequest<RequestType>::Type_t
  dataObjectRequest(const DataObjectRequest<RequestType> &req) const
    {
      return field().engine().dataObjectRequest(req);
    }

private:

  Interval<Dim> domain_m;
  Expression1_t field_m;
  Expression2_t weight_m;
  Loc<Dim> offset_m;
  Functor_t functor_m;
};

//-----------------------------------------------------------------------------
// View types for stencil objects.  Stencils define operator() to return a
// stencil engine object.  If you wanted to store that object, you could write:
//
// A a; B b;
// Laplace laplace;
// typename View2<WeightedFieldStencil<Laplace>,A,B>::Type_t b = laplace(a, b);
//-----------------------------------------------------------------------------

template<class Functor, class G, class T, class E, class TW, class EW>
struct View2<WeightedFieldStencil<Functor>, ConstField<G,T,E>, 
  ConstField<G,TW,EW> >
{
  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef typename Functor::OutputCentering_t OutputCentering_t;
  typedef typename Functor::OutputElement_t OutputElement_t;
  typedef ConstField<G,T,E> InputField_t;  
  typedef ConstField<G,TW,EW> WeightField_t;  

  typedef G InputGeometry_t;
  typedef typename InputGeometry_t::Mesh_t InputMesh_t;
  typedef typename InputGeometry_t::Centering_t InputCentering_t;

  typedef DiscreteGeometry<OutputCentering_t, InputMesh_t> OutputGeometry_t;
  typedef ApplyWeightedFieldStencil<Functor, InputField_t, WeightField_t> 
    OutputEngineTag_t;
  typedef ConstField<OutputGeometry_t, OutputElement_t, OutputEngineTag_t>
    Type_t;
    
  enum { outputDim = InputGeometry_t::dimensions };
  typedef Engine<outputDim, OutputElement_t, OutputEngineTag_t>
    OutputEngine_t;

  //---------------------------------------------------------------------------
  // This function actually coughs up the output ConstField. The output
  // field has the same type as the input field, except that the input
  // centering is replaced by the output centering. The physical domain (PD)
  // of the output field comes from the PD of the output geometry. This 
  // implies that a stencil cannot shave off layers of the PD. The guard
  // layers of the resulting field are a little tricky to compute. There are
  // two effects: (1) the extent of the stencil and (2) the change in centering.
  // The extent of the stencil is pretty simple to analyze: you lose a layer
  // of guard elements for every point the stencil extends. The change in
  // centering follows the rules: Vert -> Vert or Cell -> Cell - no change,
  // Vert -> Cell - add 1 upper layer, Cell -> Vert - subtract 1 upper layer.
  
  // Note: assumes that the output centering can be constructed with a default
  // constructor.

  static inline
  Type_t make(const WeightedFieldStencil<Functor> &s, const InputField_t &f,
    const WeightField_t &w) 
    {
      GuardLayers<outputDim> og(f.geometry().guardLayers());
      for (int d = 0; d < outputDim; d++)
        {
          og.lower(d) -= s.functor().lowerExtent(d);
          og.upper(d) -= s.functor().upperExtent(d);
          if (f.geometry().centering().centeringTag(d) == vert &&
              OutputCentering_t().centeringTag(d) == cell)
            og.upper(d)++;
          if (f.geometry().centering().centeringTag(d) == cell &&
              OutputCentering_t().centeringTag(d) == vert)
            og.upper(d)--;             
        }
      
      OutputGeometry_t g(f.geometry().mesh(), og);
      return Type_t(g, OutputEngine_t(s.functor(), f, w, g.totalDomain()), 
        f.bconds());
    }  
};

//-----------------------------------------------------------------------------
// Full Description:
// NewEngine<Engine,SubDomain>
//
// Specializations of NewEngine for subsetting a constant-function-engine with
// an arbitrary domain. 
//-----------------------------------------------------------------------------

template <int Dim, class T, class F, class E1, class E2>
struct NewEngine<Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> >, 
  Interval<Dim> >
{
  typedef Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> > Type_t;
};

template <int Dim, class T, class F, class E1, class E2>
struct NewEngine<Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> >, 
  INode<Dim> >
{
  typedef Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> > Type_t;
};

template <int Dim, class T, class F, class E1, class E2>
struct NewEngineDomain<Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> >, 
  INode<Dim> >
{
  typedef Interval<Dim> Type_t;
  static inline
  const Type_t &apply(
    const Engine<Dim, T, ApplyWeightedFieldStencil<F,E1,E2> > &,
    const INode<Dim> &i)
  {
    return i.domain();
  }
};

//-----------------------------------------------------------------------------
// Full Description:
//
// WeightedFieldStencil is used to wrap a user-defined field-based stencil
// class.  The idea is to encapsulate the majority of the crazy type
// manipulations required to generate the output ConstField and the calculation
// of the new number of guard layers.
//
// To create a stencil, users must create a class similar to the one below,
// which computes a central difference divergence of a weight-field-multiplied
// Vert-centered Field and maps it to a Cell-centered Field:
//
// template<class OutputCentering, class Subject>
// class Div { };
//  
// template<int Dim, class T1, class T2, class EngineTag,
//          class TW, class EngineTagW>
// class Div<Cell, 
//           ConstField<DiscreteGeometry<Vert, 
//                               UniformRectilinearMesh<Dim, 
//                                                      Cartesian<Dim>, T1 > >,
//                      Vector<Dim, T2>, EngineTag>,
//           ConstField<DiscreteGeometry<Vert, 
//                               UniformRectilinearMesh<Dim, 
//                                                      Cartesian<Dim>, T1 > >,
//                      TW, EngineTagW> >
// {
// public:
// 
//   typedef Cell OutputCentering_t;
//   typedef ConstField<
//           DiscreteGeometry<Vert, 
//           UniformRectilinearMesh<Dim, Cartesian<Dim>, T1 > >,
//           Vector<Dim, T2>,
//           EngineTag> InputField_t;          
//   typedef ConstField<
//           DiscreteGeometry<Vert, 
//           UniformRectilinearMesh<Dim, Cartesian<Dim>, T1 > >,
//           TW,
//           EngineTagW> WeightField_t;          
//   typedef typename BinaryReturn<T2,TW,OpMultiply> OutputElement_t;
// 
//   int lowerExtent(int) const
//     {
//       return 1;
//     }
// 
//   int upperExtent(int) const
//     {
//       return 1;
//     }
//         
//   template<class F, class W>
//   inline OutputElement_t
//   operator()(const F &f, const W &w, int i1) const
//     {
//       return (w(i1 + 1)*f(i1 + 1)(0) - w(i1 + 1)*f(i1 - 1)(0)) / 
//         f.geometry().mesh().meshSpacing(0);
//     }
// 
//   template<class F>
//   inline OutputElement_t
//   operator()(const F &f, int i1, int i2) const
//     {
//       return (w(i1 + 1, i2)*f(i1 + 1, i2)(0) - 
//               w(i1 - 1, i2)*f(i1 - 1, i2)(0)) / 
//         f.geometry().mesh().meshSpacing()(0) +
//         (w(i1, i2 + 1)*f(i1, i2 + 1)(1) - 
//          w(i1, i2 - 1)*f(i1, i2 - 1)(1)) / 
//         f.geometry().mesh().meshSpacing()(1);
//     }
// };
//
// There are 2 required typedefs: OutputCentering_t and OutputElement_t. 
// These export the type of the output centering and the type resulting 
// from applying the stencil at a point. 
// This last type is derived from the result of
// multiplying an element of type T2 from InputField_t and an element of type
// TW from WeightField_t. Then, there are two accessors: lowerExtent(int dir)
// and upperExtent(int dir). These return the extent of the stencil as a
// function of direction. As another example, a forward difference would have a
// lower extent of 0 and an upper extent of 1. Finally, a series of inline
// operator() functions, which take a Field of some sort, a weighting Field of
// some sort, and a set indices, must be supplied. This is what actually
// computes the stencil.
//
// A ConstField that contains an ApplyWeightedFieldStencil-engine that operates
// on Fields f and w, is constructed by using operator()() for 
// WeightedFieldStencil:
//
// View2<WeightedFieldStencil<
//   WeightedAverage<OutputCentering, Geometry, T, TW> >, 
//   ConstField<Geometry, T, EngineTag>,
//   ConstField<Geometry, TW, EngineTagW> >::make(
//   WeightedAverage<OutputCentering, Geometry, T>(), f, w);
// 
//-----------------------------------------------------------------------------

template<class Functor>
class WeightedFieldStencil
{
public:

  WeightedFieldStencil()
  { }

  WeightedFieldStencil(const Functor &functor)
    : functor_m(functor)
  { }

  template<class Init>
  WeightedFieldStencil(const Init &init)
    : functor_m(init)
  { }

  ~WeightedFieldStencil() { }

  template<class G, class T, class E, class TW, class EW>
  typename View2<WeightedFieldStencil<Functor>, 
    ConstField<G,T,E>, ConstField<G,TW,EW> >::Type_t
  operator()(const ConstField<G,T,E> &expr, 
    const ConstField<G,T,E> &weight) const
  {
    typedef View2<WeightedFieldStencil<Functor>, 
      ConstField<G,T,E>, ConstField<G,TW,EW> > Ret_t;
    return Ret_t::make(*this, expr, weight);
  }
  
  inline const Functor &functor() const { return functor_m; }

private:

  // Store the functor here.

  Functor functor_m;
};


//-----------------------------------------------------------------------------
// Specializations for selecting the appropriate evaluator for the Stencil
// engine.  We just get the appropriate types from the Expression's engine.
//-----------------------------------------------------------------------------

template<class Functor, class Expression1, class Expression2>
struct EvaluatorEngineTraits<
  ApplyWeightedFieldStencil<Functor, Expression1, Expression2> >
{
  typedef typename CreateLeaf<Expression1>::Leaf_t Expr_t;
  typedef typename
    ForEach<Expr_t, EvaluatorTypeTag, EvaluatorCombineTag>::Type_t
      Evaluator_t;
};

//-----------------------------------------------------------------------------
// FieldStencilIntersector is a special intersector that gets used when we come
// across a stencil object in an expression.
//-----------------------------------------------------------------------------

template<int Dim, class Intersect>
class WeightedFieldStencilIntersector
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef typename Intersect::IntersectorData_t         IntersectorData_t;
  typedef WeightedFieldStencilIntersector<Dim, Intersect>       This_t;
  typedef typename IntersectorData_t::const_iterator    const_iterator;
  typedef RefCountedPtr<IntersectorData_t>              DataPtr_t;
  typedef Interval<Dim>                                 Domain_t;
  
  enum { dimensions = Intersect::dimensions };
  
  //---------------------------------------------------------------------------
  // Constructors

  WeightedFieldStencilIntersector(const This_t &model)
    : domain_m(model.domain_m), intersector_m(model.intersector_m)
  { }

  WeightedFieldStencilIntersector(const Domain_t &dom, 
    const Intersect &intersect)
    : domain_m(dom), intersector_m(intersect)
  { }

  This_t &operator=(const This_t &model)
  {
    if (this != &model)
    {
      domain_m = model.domain_m;
      intersector_m = model.intersector_m;
    }
  }

  ~WeightedFieldStencilIntersector() { }

  inline DataPtr_t &data() { return intersector_m.data(); }
  inline const DataPtr_t &data() const { return intersector_m.data(); }

  //---------------------------------------------------------------------------
  // Accessors

  // STL iterator support.
  
  inline const_iterator begin() const { return data()->inodes_m.begin(); }
  inline const_iterator end() const { return data()->inodes_m.end(); }

  //---------------------------------------------------------------------------
  // Intersect routines

  // All domains.
  
  template<class Engine>
  inline void intersect(const Engine &engine) 
  {
    typedef typename NewEngine<Engine, Interval<Dim> >::Type_t NewEngine_t;

    NewEngine_t newEngine(engine, domain_m);

    intersector_m.intersect(newEngine);

    data()->shared(engine.layout().ID(), newEngine.layout().ID());
  }

  template<class Engine, int Dim2>
  inline bool intersect(const Engine &engine, const GuardLayers<Dim2> &) 
  {
    intersect(engine);
    return true;
  }

private:

  Interval<Dim> domain_m;
  Intersect     intersector_m;
};


//-----------------------------------------------------------------------------
// IntersectEngine specialization
//-----------------------------------------------------------------------------

template <int Dim, class T, class Functor, class Expression1, class Expression2,
  class Intersect>
struct LeafFunctor<Engine<Dim, T, 
  ApplyWeightedFieldStencil<Functor,Expression1,Expression2> >,
  EngineApply<IntersectorTag<Intersect> > >
{
  typedef int Type_t;

  static
  Type_t apply(const Engine<Dim, T, 
	       ApplyWeightedFieldStencil<Functor,Expression1,Expression2> > 
	       &engine, const EngineApply<IntersectorTag<Intersect> > &tag)
  {
    typedef WeightedFieldStencilIntersector<Dim, Intersect> NewIntersector_t;
    NewIntersector_t newIntersector(engine.domain(), tag.intersector_m);
    EngineApply<IntersectorTag<NewIntersector_t> > newTag(newIntersector); 

    forEach(engine.field(), newTag, NullCombine());
    return 0;
  }
};

#endif // POOMA_FIELD_WEIGHTEDFIELDSTENCIL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: WeightedFieldStencil.h,v $   $Author: julianc $
// $Revision: 1.19 $   $Date: 2000/07/20 22:25:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
