// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Classes:
// ErrorType
// CreateLeaf<Expr>
// MakeUnaryReturn<T>
// MakeBinaryReturn<T1,T2>
// MakeTrinaryReturn<T1,T2,T3>
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_CREATELEAF_H
#define POOMA_FIELD_CREATELEAF_H

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// These are the external traits classes that are used to build trees.
// CreateLeaf is used to convert arbitrary classes into expression objects.
// Make...Return are used to combine expressions together with operators.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Traits classes
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// CreateLeaf is an external functor class used to convert objects into the
// leaves of the expression tree.
//
// CreateLeaf<T> converts objects of type T to leaf objects and requires
// the following interface:
//
// typedef ... Leaf_t;        // The leaf object
// typedef ... Return_t;      // Type returned by make()
// Return_t make(const T&);   // make the leaf object from the T object
//
// Return_t should equivalent to Leaf_t. (Leaf_t needs to be able
// be constructed with a Return_t.)  We avoid making extra copies by building
// expression trees from references, so define Return_t to be a const ref to
// an Leaf_t.  (Returning by value would be bad, since we would create
// a temporary copy that won't survive until the whole expression is put
// together.)
//
// CreateLeaf is used to construct expression trees.  It should also be
// used when performing operations on the expression tree, such as forEach,
// in order to extract the expression.  For example:
// template<class G, class T, class E>
// void func(const Field<G,T,E>& f)
// {
//   forEach(CreateLeaf<Field<G,T,E >::make(f),...,...);
// }
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Fields are leaf objects, we just pass them through unless they have
// Expression engines. Then, we remove the expresion from the field and form
// leaves with tree-nodes. 'FieldCreateLeaf' exists in order to
// work around a CodeWarrior partial specialization bug.

template<class Geom, class T, class EngineTag>
struct FieldCreateLeaf
{
  typedef Field<Geom, T, EngineTag> Input_t;
  typedef ConstField<Geom, T, EngineTag> FieldLeaf_t;
  typedef Reference<FieldLeaf_t> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &f)
  {
    return Leaf_t(f);
  }
};

template<class Geom, class T, class Expr>
struct FieldCreateLeaf<Geom, T, ExpressionTag<Expr> >
{
  typedef Field<Geom, T, ExpressionTag<Expr> > Input_t;
  typedef Expr Leaf_t;
  typedef const Leaf_t &Return_t;
  inline static
  Return_t make(const Input_t &f)
  {
    return f.engine().expression();
  }
};
  
template<class Geom, class T, class EngineTag>
struct CreateLeaf<Field<Geom, T, EngineTag> >
{
  typedef Field<Geom, T, EngineTag> Input_t;
  typedef typename FieldCreateLeaf<Geom, T, EngineTag>::Leaf_t Leaf_t;
  typedef typename FieldCreateLeaf<Geom, T, EngineTag>::Return_t Return_t;
  inline static
  Return_t make(const Input_t &a)
  {
    return FieldCreateLeaf<Geom, T, EngineTag>::make(a);
  }
};

//-----------------------------------------------------------------------------
// Special case for Scalar<Field<>> returns ErrorType to avoid
// hairy type computations.

template<class Geom, class T, class EngineTag>
struct CreateLeaf<Scalar<Field<Geom, T, EngineTag> > >
{
  typedef Scalar<Field<Geom, T, EngineTag> > Input_t;
  typedef Scalar<ErrorType> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &)
  {
    return ErrorType();
  }
};

//-----------------------------------------------------------------------------
// ConstFields are leaf objects, we just pass them through unless they have
// Expression engines. Then, we remove the expresion from the field and form
// leaves with tree-nodes. 'ConstFieldCreateLeaf' exists in order to
// work around a CodeWarrior partial specialization bug.

template<class Geom, class T, class EngineTag>
struct ConstFieldCreateLeaf
{
  typedef ConstField<Geom, T, EngineTag> Input_t;
  typedef ConstField<Geom, T, EngineTag> ConstFieldLeaf_t;
  typedef Reference<ConstFieldLeaf_t> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &f)
  {
    return Leaf_t(f);
  }
};

template<class Geom, class T, class Expr>
struct ConstFieldCreateLeaf<Geom, T, ExpressionTag<Expr> >
{
  typedef ConstField<Geom, T, ExpressionTag<Expr> > Input_t;
  typedef Expr Leaf_t;
  typedef const Leaf_t &Return_t;
  inline static
  Return_t make(const Input_t &f)
  {
    return f.engine().expression();
  }
};
  
template<class Geom, class T, class EngineTag>
struct CreateLeaf<ConstField<Geom, T, EngineTag> >
{
  typedef ConstField<Geom, T, EngineTag> Input_t;
  typedef typename ConstFieldCreateLeaf<Geom, T, EngineTag>::Leaf_t Leaf_t;
  typedef typename ConstFieldCreateLeaf<Geom, T, EngineTag>::Return_t Return_t;
  inline static
  Return_t make(const Input_t &a)
  {
    return ConstFieldCreateLeaf<Geom, T, EngineTag>::make(a);
  }
};

//-----------------------------------------------------------------------------
// Special case for Scalar<ConstField<>> returns ErrorType to avoid
// hairy type computations.

template<class Geom, class T, class EngineTag>
struct CreateLeaf<Scalar<ConstField<Geom, T, EngineTag> > >
{
  typedef Scalar<ConstField<Geom, T, EngineTag> > Input_t;
  typedef Scalar<ErrorType> Leaf_t;
  typedef Leaf_t Return_t;
  inline static
  Return_t make(const Input_t &)
  {
    return ErrorType();
  }
};

//-----------------------------------------------------------------------------
// MakeFieldReturn is a tool used by operator functions to construct the
// expression tree representing that function.  Each function needs to define
// a corresponding operator functor Op which is used to compute the return type.
// The required interface for MakeFieldReturn is:
//
// typedef ... Expression_t;    // type of the expression (UnaryNode<...>)
// Expression_t make(const T&); // construct the tree
//
// These versions are a little more complicated than those for Array because we
// want to preserve Geometry information to the largest extent possible.
//-----------------------------------------------------------------------------

template<class Expr> 
struct MakeFieldReturn { };

// op(ConstField)

template<class Op, class Geom, class T, class EngineTag>
struct MakeFieldReturn<UnaryNode<Op, Reference<ConstField<Geom, T, EngineTag> > > >
{
  // The node type.
  
  typedef UnaryNode<Op, Reference<ConstField<Geom, T, EngineTag> > > Tree_t;

  // The Field type.
    
  typedef ConstField<Geom, T, EngineTag> Field_t;

  // Deduce the template parameters of the expression engine we're building.
  
  enum { dim = Field_t::dimensions };
  typedef typename UnaryReturn<T, Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're outputing.

  typedef ConstField<Geom, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        tree.child().geometry(),
        Engine_t(tree));
    }
};

// op(Expression)

template<class Op, class Leaf>
struct MakeFieldReturn<UnaryNode<Op, Leaf> >
{
  // The node type.
  
  typedef UnaryNode<Op, Leaf> Tree_t;

  // Deduce the template parameters of the expression engine we're building.
  
  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename UnaryReturn<
    typename ForEach<Leaf, EvalLeaf<dim>, OpCombine>::Type_t, 
    Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<NoGeometry<dim>, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        NoGeometry<dim>(forEach(tree, DomainFunctorTag(), DomainFunctorTag())),
        Engine_t(tree));
    }
};

// Scalar op ConstField

template<class Op, class T1, class Geom, class T2, class EngineTag>
struct MakeFieldReturn<BinaryNode<Op, 
  Scalar<T1>, 
  Reference<ConstField<Geom, T2, EngineTag> > 
  > >
{
  // The node type.
  
  typedef Scalar<T1> Left_t;
  typedef Reference<ConstField<Geom, T2, EngineTag> > Right_t;
  typedef BinaryNode<Op, Left_t, Right_t> Tree_t;

  // The Field type.
  
  typedef typename Right_t::Type_t Field_t;
  
  // Deduce the template parameters of the expression engine we're building.

  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Field_t::dimensions };
  typedef typename BinaryReturn<T1, T2, Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<Geom, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        tree.right().geometry(),
        Engine_t(tree));
    }
};

// ConstField op Scalar

template<class Op, class Geom, class T1, class EngineTag, class T2>
struct MakeFieldReturn<BinaryNode<Op, 
  Reference<ConstField<Geom, T1, EngineTag> >,
  Scalar<T2>
  > >
{
  // The node type.
  
  typedef Reference<ConstField<Geom, T1, EngineTag> > Left_t;
  typedef Scalar<T2> Right_t;
  typedef BinaryNode<Op, Left_t, Right_t> Tree_t;

  // The Field type.
  
  typedef typename Left_t::Type_t Field_t;
  
  // Deduce the template parameters of the expression engine we're building.

  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Field_t::dimensions };
  typedef typename BinaryReturn<T1, T2, Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<Geom, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        tree.left().geometry(),
        Engine_t(tree));
    }
};

// Array op ConstField

template<class Op, int Dim, class T1, class EngineTag1, 
  class Geom, class T2, class EngineTag2>
struct MakeFieldReturn<BinaryNode<Op, 
  Reference<Array<Dim, T1, EngineTag1> >, 
  Reference<ConstField<Geom, T2, EngineTag2> > 
  > >
{
  // The node type.
  
  typedef Reference<Array<Dim, T1, EngineTag1> > Left_t;
  typedef Reference<ConstField<Geom, T2, EngineTag2> > Right_t;
  typedef BinaryNode<Op, Left_t, Right_t> Tree_t;

  // The Field type.
  
  typedef typename Right_t::Type_t Field_t;
  
  // Deduce the template parameters of the expression engine we're building.

  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Field_t::dimensions };
  typedef typename BinaryReturn<T1, T2, Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<Geom, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        tree.right().geometry(),
        Engine_t(tree));
    }
};

// ConstField op Array

template<class Op, class Geom, class T1, class EngineTag1, 
  int Dim, class T2, class EngineTag2>
struct MakeFieldReturn<BinaryNode<Op, 
  Reference<ConstField<Geom, T1, EngineTag1> >,
  Reference<Array<Dim, T2, EngineTag2> >
  > >
{
  // The node type.
  
  typedef Reference<ConstField<Geom, T1, EngineTag1> > Left_t;
  typedef Reference<Array<Dim, T2, EngineTag2> > Right_t;
  typedef BinaryNode<Op, Left_t, Right_t> Tree_t;

  // The Field type.
  
  typedef typename Left_t::Type_t Field_t;
  
  // Deduce the template parameters of the expression engine we're building.

  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Field_t::dimensions };
  typedef typename BinaryReturn<T1, T2, Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<Geom, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.
  
  inline static
  Expression_t make(const Tree_t &tree)
    {
      return Expression_t(
        tree.left().geometry(),
        Engine_t(tree));
    }
};

// Expression op Expression

template<class Op, class Left, class Right>
struct MakeFieldReturn<BinaryNode<Op, Left, Right> >
{
  // The node type.
  
  typedef BinaryNode<Op, Left, Right> Tree_t;

  // Deduce the template parameters of the expression engine we're building.
  
  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename BinaryReturn<
    typename ForEach<Left, EvalLeaf<dim>, OpCombine>::Type_t,
    typename ForEach<Right, EvalLeaf<dim>, OpCombine>::Type_t,
    Op>::Type_t T_t;
  typedef Engine<dim, T_t, ExpressionTag<Tree_t> > Engine_t;

  // Construct the type of the ConstField we're making.

  typedef ConstField<NoGeometry<dim>, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  // This function turns the tree node into a ConstField. Note that we don't
  // bother copying the boundary conditions since we will go right to the
  // leaves in order to evaluate these.

  inline static
  Expression_t make(const Tree_t& tree)
    {
      return Expression_t(
        NoGeometry<dim>(forEach(tree, DomainFunctorTag(), DomainFunctorTag())),
        Engine_t(tree));
    }
};

template<class Op, class Cl, class Tr, class Fl>
struct MakeFieldReturn<TrinaryNode<Op, Cl, Tr, Fl> >
{
  typedef TrinaryNode<Op,Cl,Tr,Fl> Tree_t;

  typedef typename 
    ForEach<Tree_t, DomainFunctorTag, DomainFunctorTag>::Type_t Domain_t;
  enum { dim = Domain_t::dimensions };
  typedef typename TrinaryReturn<
    typename ForEach<Cl, EvalLeaf<dim>, OpCombine>::Type_t,
    typename ForEach<Tr, EvalLeaf<dim>, OpCombine>::Type_t,
    typename ForEach<Fl, EvalLeaf<dim>, OpCombine>::Type_t,
    Op>::Type_t T_t;
  typedef Engine<dim,T_t,ExpressionTag<Tree_t> > Engine_t;
  typedef ConstField<NoGeometry<dim>, T_t, ExpressionTag<Tree_t> > 
    Expression_t;

  inline static
  Expression_t make(const Tree_t& tree)
    {
      return Expression_t(
        NoGeometry<dim>(forEach(tree, DomainFunctorTag(), DomainFunctorTag())),
        Engine_t(tree));
    }
};

#endif // POOMA_FIELD_CREATELEAF_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: CreateLeaf.h,v $   $Author: swhaney $
// $Revision: 1.6 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
