// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_EVALUATOR_REDUCTIONKERNEL_H
#define POOMA_EVALUATOR_REDUCTIONKERNEL_H

//-----------------------------------------------------------------------------
// Class:
//   ReductionKernel
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Overview: 
//
// A ReductionKernel encapsulates reducing an expression on a domain.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Engine/Engine.h"
#include "Engine/EngineFunctor.h"
#include "Evaluator/EvaluatorTags.h"
#include "Evaluator/ReductionEvaluator.h"
#include "Evaluator/RequestLocks.h"
#include "Threads/PoomaCSem.h"
#include "Threads/PoomaSmarts.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Full Description:
//
// A ReductionKernel is a specific kind of iterate which
// reduces a particular expression with over a given domain.
//
// A ReductionKernel IS-AN Interate. That means that it
// has three primary functions:
//
//   1. Construct given an expression and a domain. This will acquire
//      locks on the data referenced by the expression.
//
//   2. Destruct. This releases the locks on the data.
//
//   3. Run the kernel by calling the member function run.
//
//-----------------------------------------------------------------------------

template<class T, class Op, class Expr, class KernelTag>
class ReductionKernel : public Pooma::Iterate_t
{
public:

  typedef ReductionKernel<T, Op, Expr, KernelTag> This_t;

  //---------------------------------------------------------------------------
  // Construct from an Expr.
  // Build the kernel that will reduce the expression on the given domain.
  // Acquire locks on the data referred to by the expression.

  ReductionKernel(T &ret, const Op &op, const Expr &e,
		  Pooma::CountingSemaphore &csem);

  //---------------------------------------------------------------------------
  // Virtual Destructor.
  // Release locks on the data referred to by the expression.

  virtual ~ReductionKernel();

  //---------------------------------------------------------------------------
  // Do the reduction.

  virtual void run();

private:

  // The expression we will reduce.

  T &ret_m;
  Op op_m;
  Expr expr_m;
  Pooma::CountingSemaphore &csem_m;
};

//-----------------------------------------------------------------------------
//
// Implementation of the functions for ReductionKernel.
// Since ReductionKernel is templated on an expression type,
// there is little hope of instantiating by hand, so we put
// the definition in the header file.
//
//-----------------------------------------------------------------------------

//---------------------------------------------------------------------------
// Constructor
// Input an expression and record it for later use.

template<class T, class Op, class Expr, class KernelTag>
ReductionKernel<T, Op, Expr, KernelTag>::
ReductionKernel(T &ret, const Op &op, const Expr &e,
		Pooma::CountingSemaphore &csem)
  : Pooma::Iterate_t(Pooma::scheduler()),
    ret_m(ret), op_m(op), expr_m(e), csem_m(csem)
{
  // Request read lock.

  DataObjectRequest<ReadRequest> readReq(*this);
  engineFunctor(expr_m, readReq);
}

//---------------------------------------------------------------------------
// Destroy the kernel.
// Just let the expression destroy itself.
// Release locks and increment the semaphore.

template<class T, class Op, class Expr, class KernelTag>
ReductionKernel<T, Op, Expr, KernelTag>::~ReductionKernel()
{
  // Release read lock.

  DataObjectRequest<ReadRelease> readRelease;
  engineFunctor(expr_m, readRelease);
  
  // Increment semaphore.
  
  csem_m.incr();
}

//---------------------------------------------------------------------------
// Evaluate the kernel

template<class T, class Op, class Expr, class KernelTag>
void ReductionKernel<T, Op, Expr, KernelTag>::run()
{
  // Just evaluate the expression.
  
  ReductionEvaluator<KernelTag>::evaluate(ret_m, op_m, expr_m);
}

#endif     // POOMA_EVALUATOR_REDUCTIONKERNEL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ReductionKernel.h,v $   $Author: sa_smith $
// $Revision: 1.3 $   $Date: 2000/06/08 22:16:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
