// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_INTERSECT_H
#define POOMA_DOMAIN_INTERSECT_H

//-----------------------------------------------------------------------------
// Function:
//   domain intersect(domain, domain);
// Class:
//   IntersectDomain<T1,T2,T3,Dim>
//   IntersectDomainSingle<T1,T2,T3,bool>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// domain3 intersect(domain1,domain2) is a global function which determines if
// two domains intersect with each other, and if so, what their
// intersection is.  By 'intersect', we mean the set of points which are
// found in BOTH domains, expressed as a new domain.  If they have no
// points in common, this returns an empty domain.  The type of domain
// returned is the most general domain type which can store the information
// from domain1 and domain2.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Domain/DomainCalculus.h"
#include "Domain/NewDomain.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// Full Description of IntersectDomainSingle:
//
// IntersectDomainSingle<T1,T2,T3,int Dim,bool strided>::intersect(a,b,c)
// finds the intersection of two domains a and b of type T1 and T2, and
// puts the domain intersection in c[Dim-1]. If there are no points which these
// domains have in common, this returns an empty domain.  The domains a and b
// are assumed to be 1D domains.
// The final boolean template parameter is used to specialize the calculation
// to the following two cases:
//   strided == false: both of the domains have unit stride.  In
//                     this case, the computation is quite simple: check if
//                     the endpoints overlap in any way, and return it.
//   strided == true: one or both domains have non-unit stride.
//                    This is more complicated
//                    since it is possible that even if the endpoints overlap,
//                    they will not have any points in common due to the
//                    striding.  We only do this long calculation when
//                    absolutely necessary.
//
//-----------------------------------------------------------------------------

//
// The default (unit-stride) version of IntersectDomainSingle, which assumes
// that both arguments to 'intersect' are 1D domains with unit stride
//

template<class T1, class T2, class T3, int Dim, bool strided>
struct IntersectDomainSingle {
  static void intersect(const T1 &a, const T2 &b, T3 &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T1>::Element_t E1_t;
    typedef typename DomainTraits<T2>::Element_t E2_t;

    // find min and max of the domains
    E1_t a0 = a.min();
    E1_t a1 = a.max();
    E2_t b0 = b.min();
    E2_t b1 = b.max();

    // make sure they touch in some way
    if (a1 < b0 || a0 > b1)
      return;

    // they overlap; find where
    if (b0 > a0)
      a0 = b0;
    if (b1 < a1)
      a1 = b1;

    // fill in the domain with the correct value
    typedef typename DomainTraits<T3>::OneDomain_t Dom_t;
    c[Dim-1] = Dom_t(a0, a1);
  }
};

//
// The non-unit-stride version of IntersectDomainSingle, which does extra
// work for the case where a or b do not have unit stride.
//

template<class T1, class T2, class T3, int Dim>
struct IntersectDomainSingle<T1,T2,T3,Dim,true> {
  static void intersect(const T1 &a, const T2 &b, T3 &c) {
    // types for elements in these domains
    typedef typename DomainTraits<T1>::Element_t E1_t;
    typedef typename DomainTraits<T2>::Element_t E2_t;

    // if both strides are +1, we can do the quick operation
    E1_t s = a.stride();
    E2_t t = b.stride();
    if (s == 1 && t == 1) {
      IntersectDomainSingle<T1,T2,T3,Dim,false>::intersect(a,b,c);
      return;
    }

    // make sure they touch in some way
    E1_t a0 = a.min();
    E1_t a1 = a.max();
    E2_t b0 = b.min();
    E2_t b1 = b.max();
    if (a1 < b0 || a0 > b1)
      return;

    // OK, the endpoints overlap in some way, and we must find out if there
    // are any points in both a and b.  If so, the result will have a
    // stride which is the least-common-multiple of the strides of a and b,
    // and endpoints which match this stride and the unit-stride intersection
    // endpoints.  We use the general routine 'findIntersectionEndpoints'
    // to do this; if it returns true, then the endpoints and stride
    // parameters are set to new values, otherwise there are no points
    // in common.
    int i1, i2, is;
    if (findIntersectionEndpoints(a0, a1, s, b0, b1, t, i1, i2, is)) {
      // there was an intersection; so, adjust i1, i2, and is to have
      // the right direction.   When i1, i2, and s come back from this
      // routine, it is always the case that i1 <= i2, and is > 0.  This
      // might need to be reversed.
      typedef typename DomainTraits<T3>::OneDomain_t Dom_t;
      if (s < 0)
        c[Dim-1] = Dom_t(i2, i1, -is);
      else
        c[Dim-1] = Dom_t(i1, i2, is);
    }
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of IntersectDomain:
//
// IntersectDomain implements a basic template meta-program to
// intersect each dimension separately of the multidimensional domains.
// It uses IntersectDomainSingle to do the single-domain intersection,
// telling that struct whether the domains have unit stride
// or not.  A general version of IntersectDomain is defined, to intersect the
// domains in the 'Dim' dimension, and then a specialization is provided
// for Dim==1 that stops the metaprogram recursion.
//
//-----------------------------------------------------------------------------

template<class T1, class T2, class T3, int Dim>
struct IntersectDomain {
  static void intersect(const T1 &a, const T2 &b, T3 &c) {
    // the types for the two domains used in IntersectDomainSingle may be a
    // little different than T1 and T2, since IntersectDomainSingle works
    // with 1D domains for a and b, not N-D domains.
    typedef typename DomainTraits<T1>::OneDomain_t Dom1_t;
    typedef typename DomainTraits<T2>::OneDomain_t Dom2_t;

    // the number of domains here which have unit stride
    const int USN = DomainTraits<T1>::unitStride +
                    DomainTraits<T2>::unitStride;

    // which dimension we're working with
    const int n = Dim - 1;

    // intersect the 'Dim' dimension, and then the lower dims
    IntersectDomainSingle<Dom1_t,Dom2_t,T3,Dim,(USN != 2)>::intersect(
      DomainTraits<T1>::getDomain(a,n), DomainTraits<T2>::getDomain(b,n), c);
    IntersectDomain<T1,T2,T3,n>::intersect(a,b,c);
  }
};

template<class T1, class T2, class T3>
struct IntersectDomain<T1,T2,T3,1> {
  static void intersect(const T1 &a, const T2 &b, T3 &c) {
    // the types for the two domains used in IntersectDomainSingle may be a
    // little different than T1 and T2, since IntersectDomainSingle works
    // with 1D domains, not N-D domains.
    typedef typename DomainTraits<T1>::OneDomain_t Dom1_t;
    typedef typename DomainTraits<T2>::OneDomain_t Dom2_t;

    // the number of domains here which have unit stride
    const int USN = DomainTraits<T1>::unitStride +
                    DomainTraits<T2>::unitStride;

    // intersect the lowest dimension
    IntersectDomainSingle<Dom1_t,Dom2_t,T3,1,(USN != 2)>::intersect(
      DomainTraits<T1>::getDomain(a,0), DomainTraits<T2>::getDomain(b,0), c);
  }
};


//-----------------------------------------------------------------------------
//
// Full Description of intersect:
//
// domain3 intersect(domain1,domain2) is a global function which determines if
// two domains intersect with each other, and if so, what their
// intersection is.  By 'intersect', we mean the set of points which are
// found in BOTH domains, expressed as a new domain.  If they have no
// points in common, this returns an empty domain.  The type of domain
// returned is the most general domain type which can store the information
// from domain1 and domain2.  For example, if you intersect an Interval
// with a Range, the most general type is Range.  The same rules that are
// used to combine domain's together are used to determine the return type.
//
// The implementation of intersect is deferred to the IntersectDomain
// struct, which performs the intersection for each dimension and and's
// the results together.
//
//-----------------------------------------------------------------------------

// first, a simple struct used to figure out the return type when intersecting
// types T1 and T2.  It defines a typedef 'Type_t' for what the return type is.
// Note that we use the 'DomainChangeDim' mechanism after we find out the
// type when combining T1 and T2, since the combined type will have a
// dimension of dim(T1) + dim(T2), and we want the dim to be the same as T1.
template<class T1, class T2>
struct IntersectReturnType {
  typedef typename NewDomain2<T1,T2>::Type_t Combine_t;
  enum { cdim = DomainTraits<T1>::dimensions };
  typedef typename DomainChangeDim<Combine_t,cdim>::NewType_t Type_t;
};

// now, finally, the intersect method
template<class T1, class T2>
inline typename IntersectReturnType<T1,T2>::Type_t
intersect(const T1 &a, const T2 &b)
{
  typedef typename IntersectReturnType<T1,T2>::Type_t T3;
  CTAssert(int(DomainTraits<T1>::dimensions)
	   == int(DomainTraits<T2>::dimensions));
  T3 c;
  IntersectDomain<T1,T2,T3,DomainTraits<T1>::dimensions>::intersect(a, b, c);
  return c;
}


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_INTERSECT_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: Intersect.h,v $   $Author: swhaney $
// $Revision: 1.7 $   $Date: 2000/03/07 13:16:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
