// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#ifndef POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_INTERVAL_H
#define POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_INTERVAL_H

//-----------------------------------------------------------------------------
// Class:
// DomainTraits<SliceInterval<Dim,SliceDim>>
//-----------------------------------------------------------------------------

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview:
// DomainTraits<SliceInterval<Dim,SliceDim>> is a specialization of the
// general DomainTraits class, for the case of SliceInterval domain objects.
// It defines the general behavior of SliceInterval, including its typedefed
// and enumerated characteristics, how to store data for a SliceInterval, etc.
// It is used by the SliceDomain base class of SliceInterval to implement most
// of the public interface.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/DomainTraits.h"
#include "Domain/Interval.h"


//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<int TotalDim, int SliceDim> class SliceInterval;


//-----------------------------------------------------------------------------
//
// Full Description:
//
// DomainTraits<SliceInterval<Dim,SliceDim>> stores the characteristics and
// much of the implementation details for SliceInterval domain objects.
// A SliceInterval represents a set of two domain objects, one a "total"
// domain, and the other a "slice" domain which is a subset of the total.
// SliceInterval stores these two domains as Interval<> objects.
//
// The DomainTraits for slice domains is quite a bit simpler than the
// DomainTraits for regular domains.  This is because SliceDomains have
// a much simpler interface than regular domains, and are not intended for
// direct user manipulation.  The DomainTraits for SliceDomain subclasses
// like SliceInterval includes the following interface:
//
// enum { dimensions } = # of total dimensions
// enum { sliceDimensions } = # of slice dimensions
//
// typedef SliceInterval<TotalDim,SliceDim> Domain_t;
// typedef Interval<SliceDim>               SliceDomain_t;
// typedef Interval<TotalDim>               TotalDomain_t;
// typedef Interval<1>                      OneDomain_t;
//
// static OneDomain_t &getDomain(Domain_t &d, int n);
// static OneDomain_t &getSliceDomain(Domain_t &d, int n);
//
//-----------------------------------------------------------------------------

template<int TotalDim, int SliceDim>
struct DomainTraits< SliceInterval<TotalDim,SliceDim> >
{
  // necessary enums
  enum { domain          = 1 };
  enum { dimensions      = TotalDim };
  enum { sliceDimensions = SliceDim };
  enum { unitStride      = 1 };
  enum { singleValued    = 0 };
  enum { wildcard        = 0 };

  // necessary typedefs
  typedef SliceInterval<TotalDim,SliceDim> Domain_t;
  typedef SliceInterval<TotalDim,SliceDim> NewDomain1_t;
  typedef Interval<SliceDim>               SliceDomain_t;
  typedef Interval<TotalDim>               TotalDomain_t;
  typedef Interval<1>                      OneDomain_t;
  typedef Interval<1>                      PointDomain_t;

  // get the Nth element of the total domain, and return a OneDomain_t
  // object with it.
  static OneDomain_t &getDomain(Domain_t &d, int n) {
    return d.totalDomain()[n];
  }
  static const OneDomain_t &getDomain(const Domain_t &d,int n) {
    return d.totalDomain()[n];
  }

  // get the Nth element of the sliced domain, and return a OneDomain_t
  // object with it
  static OneDomain_t &getSliceDomain(Domain_t &d, int n) {
    return d.sliceDomain()[n];
  }
  static const OneDomain_t &getSliceDomain(const Domain_t &d, int n) {
    return d.sliceDomain()[n];
  }
  
  // convert from the Nth element of the domain to a single point, if
  // possible, and return a PointDomain_t.  Here, we just return a OneDomain_t,
  // since this is not a single-valued domain.
  static PointDomain_t &getPointDomain(Domain_t &d, int n) {
    return getDomain(d, n);
  }
  static const PointDomain_t &getPointDomain(const Domain_t &d, int n) {
    return getDomain(d, n);
  }

  // set the given dimension as ignorable
  static void cantIgnoreDomain(Domain_t &d, int n) {
    d.cantIgnoreDomain(n);
  }

  // get the ignore status of the given dimension
  static bool getIgnorable(const Domain_t &d, int n) {
    return d.ignorable(n);
  }

  // set the ignore status of the given dimension
  static void setIgnorable(Domain_t &d, int n, bool i) {
    d.ignorable(n) = i;
  }
  
};


//////////////////////////////////////////////////////////////////////

#endif     // POOMA_DOMAIN_DOMAIN_TRAITS_SLICE_INTERVAL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: DomainTraits.SliceInterval.h,v $   $Author: swhaney $
// $Revision: 1.11 $   $Date: 2000/03/07 13:16:36 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
