// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Class:
// 
// PosReflectFaceBC
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_POSREFLECTFACEBC_H
#define POOMA_BCONDS_POSREFLECTFACEBC_H

//-----------------------------------------------------------------------------
// Overview: 
// 
// PosReflectFaceBC: BCondCategory class setting all guard layers beyond a
//                   specified (logically) rectilinear mesh face to a the value
//                   from the non-guard element symmetrically across the face
//                   (the face is defined at the last vertex).
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "BConds/FieldBCondBase.h"
#include "Field/Field.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Full Description:
// 
// PosReflectFaceBC is a specialization of BCondCategory.
// 
// It represents a symmetric boundary condition on a logically rectilinear
// domain where the value on that face is typically assumed to be zero. A
// constructor switch allows the BC to enforce that the boundary value is zero;
// this affects only vertex-centered Field values/components because the
// boundary is defined to be the last vertex.
//-----------------------------------------------------------------------------

class PosReflectFaceBC : public BCondCategory<PosReflectFaceBC>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  PosReflectFaceBC(int face, bool enforceZeroBoundary = false) : 
    face_m(face), enforceZeroBoundary_m(enforceZeroBoundary) { }
  PosReflectFaceBC(const PosReflectFaceBC &model) : 
    face_m(model.face()), enforceZeroBoundary_m(model.enforceZeroBoundary())
  { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  PosReflectFaceBC &operator=(const PosReflectFaceBC &rhs)
  {
    face_m = rhs.face();

    enforceZeroBoundary_m = rhs.enforceZeroBoundary();

    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }
  bool enforceZeroBoundary() const { return enforceZeroBoundary_m; }

private:

  int face_m;
  bool enforceZeroBoundary_m;
};


// ----------------------------------------------------------------------------
// An actual boundary condition class for PosReflectFaceBC (partial
// specialization of the BCond class)
// ----------------------------------------------------------------------------

template<class Geom, class T, class EngineTag>
class BCond<Field<Geom, T, EngineTag>, PosReflectFaceBC> :
  public FieldBCondBase<Field<Geom, T, EngineTag> >
{
public:

  typedef BCond<Field<Geom, T, EngineTag>, PosReflectFaceBC> This_t;
  typedef FieldBCondBase<Field<Geom, T, EngineTag> > Base_t;
  typedef typename Base_t::Domain_t Domain_t;
  
  //---------------------------------------------------------------------------
  // Constructors.

  BCond(const Field<Geom, T, EngineTag> &f, const PosReflectFaceBC &bc);
  

  //---------------------------------------------------------------------------
  // Destructor.

  ~BCond() {};


  //---------------------------------------------------------------------------
  // Methods.

  // Apply the boundary conditions.
  
  void applyBoundaryCondition();
  
  // Clone and retarget this boundary condition.
  
  Base_t *retarget(const Field<Geom, T, EngineTag> &f) const;

private:

  // Store a copy of the BCondCategory object used to construct this.
  PosReflectFaceBC bc_m;

  // Extra domain, used when PosReflectFaceBC::enforceZeroBoundary_m is true
  Domain_t vertFaceDomain_m;

  // More extra domains, specified as Ranges. Need to set negative strides for
  // tidy single data-parallel assignment in applyBoundaryCondition(). Because
  // the Domain_t deduced from the Array may be Interval<Dim>, which doesn't
  // admit negative strides, have to copy source domain contents from the
  // BCondBase data member into this Ranges (arrange things so only the source
  // domain needs to have negative strides, not the dest domain):
  Range<Geom::dimensions> srcRange_m;
  
};

//-----------------------------------------------------------------------------
// Full Description:
// 
// AllPosReflectFaceBC is a functor class that sets all of a Field's BCs to be 
// PosReflectFaceBC.
//-----------------------------------------------------------------------------

class AllPosReflectFaceBC
{
public:

  AllPosReflectFaceBC(bool enforceZeroBoundary = false)
  : ezb_m(enforceZeroBoundary) { }
  
  template<class Geom, class T, class EngineTag>
  void operator()(Field<Geom, T, EngineTag> &f) const
    {
      for (int i = 0; i < 2 * Geom::dimensions; i++)
        f.addBoundaryCondition(PosReflectFaceBC(i, ezb_m));
    }

private:

  bool ezb_m;
};

#include "BConds/PosReflectFaceBC.cpp"

#endif // POOMA_BCONDS_POSREFLECTFACEBC_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: PosReflectFaceBC.h,v $   $Author: swhaney $
// $Revision: 1.10 $   $Date: 2000/03/07 13:16:13 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
