// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// A tour of the new Field class.
//-----------------------------------------------------------------------------

#include "Pooma/NewFields.h"

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  // To declare a field, you first need to set up a layout. This requires
  // knowing the physical vertex-domain and the number of external guard
  // cell layers. Vertex domains contain enough points to hold all of the
  // rectilinear centerings that POOMA is likely to support for quite
  // awhile. Also, it means that the same layout can be used for all
  // fields, regardless of centering.
  
  Interval<2> physicalVertexDomain(4, 4);
  
  Loc<2> blocks(1, 2);
  UniformGridPartition<2> partition(blocks, GuardLayers<2>(1));   
  UniformGridLayout<2> layout(physicalVertexDomain, partition, ReplicatedTag());
  
  std::cout << layout << std::endl;
  std::cout << layout.domain() << std::endl;
  
  // Now, we can declare a field.

  typedef 
    Field<UniformRectilinear<2>, double, MultiPatch<UniformTag, Brick> > Field_t;
  Field_t f(AllFace(), layout, Vector<2>(0.0), Vector<2>(1.0, 2.0));

  // Ask for the field's physical cell domain.
  
  std::cout << f.physicalCellDomain() << std::endl;

  // If we ask for the physical domain, we should get the physical cell
  // domain back because of the all-face centering. We can get the
  // face-domains by specifying the sub-fields.

  std::cout << f.physicalDomain() << std::endl;
  std::cout << f.physicalDomain(0) << std::endl;
  std::cout << f.physicalDomain(1) << std::endl;

  // Total domains work similarly.

  std::cout << f.totalDomain() << std::endl;
  std::cout << f.totalDomain(0) << std::endl;
  std::cout << f.totalDomain(1) << std::endl;

  // We can do a similar sort of thing by taking sub-field views.

  std::cout << f[0].physicalDomain() << std::endl;
  std::cout << f[1].physicalDomain() << std::endl;

  // Total domains work similarly. Note: taking a sub-field view doesn't
  // remove the guard layers.

  std::cout << f[0].totalDomain() << std::endl;
  std::cout << f[1].totalDomain() << std::endl;
    
  // We can actually index fields after taking a sub-field view. The
  // indices refer to the actual domain.
  
  f[0](1, 2) = 3.0;
  f[1](1, 2) = f[0](1, 2) + 1.2;
  
  std::cout << f[0](1, 2) << std::endl;
  std::cout << f[1](1, 2) << std::endl;

  // Same thing after taking domain & sub-field views.
  
  Interval<1> I(1,2);
  std::cout << f[0](I, I)(0, 1) << std::endl;
  std::cout << f(I, I)[1](0, 1) << std::endl;
  
  // The guard layers are removed when you take a domain view.

  std::cout << f(I, I).physicalDomain() << std::endl;
  std::cout << f(I, I).totalDomain() << std::endl;
  std::cout << f(I, I).physicalDomain(0) << std::endl;
  std::cout << f(I, I).totalDomain(0) << std::endl;
  
  // Check assignment of a scalar.
  
  f = -1.0;
  f(I, I) = -2.0;
  std::cout << f << std::endl;

  // Declare another field. Note how we can reuse the layout for a field
  // with a different centering.
  
  Field_t g(Face(1), layout, Vector<2>(0.0), Vector<2>(1.0, 2.0));

  g = -3.0;
  g(I, I) = -4.0;
  f[1] = g;

  std::cout << f.all() << std::endl;
  std::cout << g.all() << std::endl;

  typedef 
    Field<Lagrangian<2>, double, MultiPatch<UniformTag, Brick> > 
     LagrField_t;
   LagrField_t h(AllFace(), layout, Vector<2>(0.0), Vector<2>(1.0, 2.0));
  
  std::cout << h.fieldEngine().vertexPositions() << std::endl;
  
  // Try assigning to a field with a lagranigan mesh.
  
  h.all() = 3.0;
  h = -6.0;
  std::cout << h.all() << std::endl;
  
  // Check out the patch function.
  
  f.all() = 1.0;
  f = 2.0;

  std::cout << f[0].patchLocal(0) << std::endl;
  std::cout << f[0].patchLocal(1) << std::endl;
  
  // Play with updaters.
  
  f.addUpdaters(AllConstantFaceBC<double>(-1.0));
  f.update(true);
  std::cout << f.all() << std::endl;
  
  // Try to create a vector field.

  typedef 
    Field<UniformRectilinear<2>, Vector<2>, MultiPatch<UniformTag, Brick> > 
     VectorField_t;
  VectorField_t l(AllFace(), layout, Vector<2>(0.0), Vector<2>(1.0, 2.0));
   
  l.all() = Vector<2>(-1.0, 2.0);
  l = Vector<2>(4.0, 6.0);
  
  std::cout << l.all().comp(0) << std::endl;
   
  Pooma::finalize();
  return 0; 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldTour1.cpp,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/20 20:38:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
