// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// BrickBase and BrickViewBase non-inline template definitions.
//-----------------------------------------------------------------------------

#include "Engine/BrickBase.h"
#include "Domain/SliceInterval.h"
#include "Domain/SliceRange.h"
#include "Utilities/NoInit.h"
#include "Utilities/PAssert.h"

#if !defined(NOPassert)
#include <algorithm>
#endif

///////////////////////////////////////////////////////////////////////////////

namespace Pooma {

///////////////////////////////////////////////////////////////////////////////
//
// BrickBase Member Functions
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// BrickBase()
//
// Constructs a BrickBase for an empty Brick.
// 
//-----------------------------------------------------------------------------

template <int Dim> 
BrickBase<Dim>::BrickBase(bool compressible)
  : compressibleBase_m(compressible)
{ }

//-----------------------------------------------------------------------------
//
// BrickBase(const Interval<Dim> &domain)
//
// Constructs a BrickBase for a multidimensional domain described by 
// the Interval<Dim> domain.
//
//-----------------------------------------------------------------------------

template <int Dim> 
BrickBase<Dim>::BrickBase(const Domain_t &dom, bool compressible)
  : domain_m(dom), compressibleBase_m(compressible)
{
  // Compute the strides and offset.

  strides_m[0] = 1;
  firsts_m[0]  = domain()[0].first();
  off_m        = -firsts_m[0];

  for (int d = 1; d < Dim; ++d)
    {
      firsts_m[d]  = domain()[d].first();
      strides_m[d] = strides_m[d-1]*domain()[d-1].length();
      off_m       -= domain()[d].first()*strides_m[d];
    }
  
  for (int d = 0; d < Dim; ++d) ostrides_m[d] = strides_m[d];
}

//-----------------------------------------------------------------------------
//
// ~BrickBase()
//
// Out-of-line destructor to shorten compile times.
//
//-----------------------------------------------------------------------------

template <int Dim> BrickBase<Dim>::~BrickBase() { }


///////////////////////////////////////////////////////////////////////////////
//
// BrickViewBase Member Functions
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// BrickViewBase Constructors
//
// This is where the real complexity lives. There are two categories
// of views:
// 
// Slices (Dim < BaseDim):
//   These can be created either by subsetting a BrickBase with a 
//   SliceInterval or a SliceRange, by subsetting a BrickViewBase
//   with any rectangular domain (Interval, Range, SliceInterval,
//   SliceRange).
//
// Non-slices (Dim == BaseDim):
//   These are created by subsetting a BrickBase with an Interval
//   or Range, or by subsetting another non-sliced view with an
//   Interval or Range. These are handled by partial specializations
//   that fully specialize on the stride since that allows us to
//   only define the useful constructors.
// 
// We do the more general cases (the slices) first.
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//
// BrickViewBase constructors (for sliced BrickViewBase objects)
//
// These just call the various sliceInit functions defined below.
// (Interval types are converted to Range types. This costs a few
// extra multiplies since the Intervals had unit stride in each
// direction, but that savings is not worth the code duplication);
// 
// Note that slice-of-slice constructors are inline member templates. They
// delegate their work to the DoubleSliceHelper functions defined below.
//
//-----------------------------------------------------------------------------

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase()
{ }

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase(const BrickBase<Dim> &bbase,
	      const Interval<Dim> &dom)
: domain_m(Pooma::NoInit()),
  baseOffset_m(bbase.offset()), compressibleBase_m(bbase.compressibleBase())
{    
  viewInit(bbase, Range<Dim>(dom));
}

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase(const BrickBase<Dim> &bbase,
	      const Range<Dim> &dom)
: domain_m(Pooma::NoInit()),
  baseOffset_m(bbase.offset()), compressibleBase_m(bbase.compressibleBase())
{    
  viewInit(bbase, dom);
}

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase(const This_t &bvbase, const Range<Dim> &domain)
: domain_m(Pooma::NoInit()),  
  baseOffset_m(bvbase.baseOffset()), 
  compressibleBase_m(bvbase.compressibleBase())
{
  sliceInit(bvbase, domain);
}

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase(const This_t &bvbase, const Interval<Dim> &domain)
: domain_m(Pooma::NoInit()),  
  baseOffset_m(bvbase.baseOffset()), 
  compressibleBase_m(bvbase.compressibleBase())
{
  sliceInit(bvbase, Range<Dim>(domain));
}

template <int Dim>
BrickViewBase<Dim>::
BrickViewBase(const This_t &bvbase, bool compressible)
{
  *this = bvbase;
  compressibleBase_m = compressible;
  if (!compressible) restoreStrides();
}   

//-----------------------------------------------------------------------------
//
// sliceInit(const BrickViewBase<Dim,Dim,UStrd> &, 
//           const Range<Dim> &)
//
// Helper function used in taking a non-sliced view of a sliced view.
//
//-----------------------------------------------------------------------------

template <int Dim>
void
BrickViewBase<Dim>::
sliceInit(const This_t &bvbase, const Range<Dim> &domain)
{
  // Compute the strides and domain.
    
  for (int d = 0; d < Dim; ++d)
    {
      domain_m[d]   = Interval<1>(domain[d].length());
      strides_m[d]  = bvbase.ostrides_m[d] * domain[d].stride();
      baseOffset_m += domain[d].first() * bvbase.ostrides_m[d];
    }

  for (int d = 0; d < Dim; ++d) ostrides_m[d] = strides_m[d];
}

//-----------------------------------------------------------------------------
//
// DoubleSliceHelper::init
//
// Helper functions used in initializing a slice of a slice.
//
//-----------------------------------------------------------------------------

template <int Dim, int Dim2>
void DoubleSliceHelper<Dim,Dim2>::
init(Interval<Dim> &domain, 
     int *strides,
     int &baseOffset,
     const BrickViewBase<Dim2> &bvbase,
     const SliceInterval<Dim2,Dim> &dom)
{  
  SliceRange<Dim2,Dim> tmp = dom;
  init(domain, strides, baseOffset, bvbase, tmp);
}

template <int Dim, int Dim2>
void DoubleSliceHelper<Dim,Dim2>::
init(Interval<Dim> &domain, 
     int *strides,
     int &baseOffset, 
     const BrickViewBase<Dim2> &bvbase,
     const SliceRange<Dim2,Dim> &dom)
{  
  // Compute the domain and strides.
  // The domain is an Interval with the length of each component
  // equal to the length of the corrsponding domain in the SliceRange.
  // The strides are calculated by multiplying the strides in 
  // the non-ignorable directions of the engine being viewed, by the
  // strides in the SliceRange that is doing the viewing. Since we must
  // skip over the ignorable dimensions in the engine being viewed,
  // we write this as a loop over the viewed engine's dimensions and
  // just do nothing for the ignorable ones.
    
  typedef typename SliceRange<Dim2,Dim>::TotalDomain_t TotalDomain_t;
  const TotalDomain_t &totDomain = dom.totalDomain();
    
  int d, dt;
  for (dt = 0, d = 0; dt < Dim2; ++dt)
    {
      if (!dom.ignorable(dt))
        {
          PAssert(d < Dim);
          domain[d]  = Interval<1>(totDomain[dt].length());
          strides[d] = bvbase.originalStrides()[dt] * totDomain[dt].stride();
          ++d;
        }
      baseOffset += totDomain[dt].first() * bvbase.originalStrides()[dt];
    }
  PAssert(d == Dim);
   
}

//-----------------------------------------------------------------------------
//
// ~BrickViewBase()
//
// Out-of-line destructor to shorten compile times.
//
//-----------------------------------------------------------------------------

template <int Dim>
BrickViewBase<Dim>::~BrickViewBase() { }

} // namespace Pooma

///////////////////////////////////////////////////////////////////////////////

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BrickBase.cpp,v $   $Author: sa_smith $
// $Revision: 1.8 $   $Date: 2000/07/11 23:06:40 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
