// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

#include "Pooma/Arrays.h"
#include "BConds/BCond.h"
#include "BConds/BCondBase.h"
#include "BConds/BCondList.h"
#include "Utilities/Tester.h"

//-----------------------------------------------------------------------------
// Here is an example of a boundary condition category class.
//-----------------------------------------------------------------------------

class ZeroFaceBC : public BCondCategory<ZeroFaceBC>
{
public:

  //---------------------------------------------------------------------------
  // Constructors. 
  
  ZeroFaceBC(int face) : face_m(face) { }
  ZeroFaceBC(const ZeroFaceBC &model) : face_m(model.face()) { }

  //---------------------------------------------------------------------------
  // Assignment operator. Does deep assignment.
  
  ZeroFaceBC &operator=(const ZeroFaceBC &rhs)
  {
    face_m = rhs.face();
    
    return *this;
  }

  //---------------------------------------------------------------------------
  // Accessors for data members.
  
  int face() const { return face_m; }
  
private:

  int face_m;
};

//-----------------------------------------------------------------------------
// Here is an example of a boundary condition class.
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag>
class BCond<Array<Dim, T, EngineTag>, ZeroFaceBC> :
  public BCondBase<Interval<Dim>, Array<Dim, T, EngineTag> >
{
public:

  typedef BCondBase<Interval<Dim>, Array<Dim, T, EngineTag> > Base_t;
  typedef typename Base_t::Domain_t Domain_t;
  
  BCond(const Array<Dim, T, EngineTag> &a, const ZeroFaceBC &bc)
  : Base_t(a, a.domain())
  {
    Domain_t &dest = destDomain();
    
    int d = bc.face() / 2;
    if (bc.face() & 1)
      {
        // High face.
        
        dest[d] = Interval<1>(dest[d].max(), dest[d].max());
      }
    else
      {
        // Low face.
        
        dest[d] = Interval<1>(dest[d].min(), dest[d].min());
      }
  }
  
  void applyBoundaryCondition()
  {
    subject()(destDomain()) = 0;
  }
};

//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);
  Pooma::Tester tester(argc,argv);
  
  Array<2> a(6, 6);
  a = 3.0;
  tester.out() << a << std::endl;
  
  ZeroFaceBC zf1(1), zf3(3);
  
  BCondList bc;
  bc.addBoundaryCondition(a, zf1);
  bc.addBoundaryCondition(a, zf3);
  
  Range<1> r(0,4), s(0,5);
  Range<2> rs(r, s), ss(s, s);
  
  bc.notifyPreRead(rs);
  tester.out() << a << std::endl;
  
  bc.notifyPreRead(ss);
  tester.out() << a << std::endl;
  
  Array<2, Vector<3> > b(6, 6);
  b = Vector<3>(1.0, 2.0, 3.0);
  tester.out() << b << std::endl;
  
  ComponentBC<1, ZeroFaceBC> cf(1, zf1);
  BCondItem *bc2 = cf.create(b);
  
  bc2->notifyPreRead(rs);
  tester.out() << b << std::endl;
  
  bc2->notifyPreRead(ss);
  tester.out() << b << std::endl;
   
  int ret = tester.results("bc_test1");
  Pooma::finalize();
  return ret;
 
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: bc_test1.cpp,v $   $Author: swhaney $
// $Revision: 1.8 $   $Date: 2000/03/07 13:16:15 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
