//==========================================================================
//
//      io/serial/arm/evmcan_serial.c
//
//      ARM EASYCAN Serial I/O Interface Module (interrupt driven)
// (Interupts not implemented yet)
//
//==========================================================================
//####COPYRIGHTBEGIN####
//                                                                          
// -------------------------------------------                              
// The contents of this file are subject to the Red Hat eCos Public License 
// Version 1.1 (the "License"); you may not use this file except in         
// compliance with the License.  You may obtain a copy of the License at    
// http://www.redhat.com/                                                   
//                                                                          
// Software distributed under the License is distributed on an "AS IS"      
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the 
// License for the specific language governing rights and limitations under 
// the License.                                                             
//                                                                          
// The Original Code is eCos - Embedded Configurable Operating System,      
// released September 30, 1998.                                             
//                                                                          
// The Initial Developer of the Original Code is Red Hat.                   
// Portions created by Red Hat are                                          
// Copyright (C) 1998, 1999, 2000 Red Hat, Inc.                             
// All Rights Reserved.                                                     
// -------------------------------------------                              
//                                                                          
//####COPYRIGHTEND####
//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):   Carl van Schaik
//		based on serial driver by gthomas
//		based on Atmel usart library
// Contributors:  Carl van Schaik
// Date:        2000-03-21
// Purpose:     EASYCAN Serial I/O module (interrupt driven version)
// Description: 
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/system.h>
#include <pkgconf/io_serial.h>
#include <pkgconf/io.h>
#include <cyg/io/io.h>
#include <cyg/hal/hal_intr.h>
#include <cyg/io/devtab.h>
#include <cyg/io/serial.h>
#include <cyg/infra/diag.h>

#ifdef CYGPKG_IO_SERIAL_ARM_EVMCAN

#include "easycan_serial.h"

typedef struct easycan_serial_info {
    CYG_ADDRWORD   base;
    CYG_WORD       int_num;
    cyg_interrupt  serial_interrupt;
    cyg_handle_t   serial_interrupt_handle;
} easycan_serial_info;

static bool easycan_serial_init(struct cyg_devtab_entry *tab);
static bool easycan_serial_putc(serial_channel *chan, unsigned char c);
static Cyg_ErrNo easycan_serial_lookup(struct cyg_devtab_entry **tab,
				   struct cyg_devtab_entry *sub_tab,
				   const char *name);
static unsigned char easycan_serial_getc(serial_channel *chan);
static Cyg_ErrNo easycan_serial_set_config(serial_channel *chan, 
				cyg_uint32 key,
				const void *xbuf, cyg_uint32 *len);
static void easycan_serial_start_xmit(serial_channel *chan);
static void easycan_serial_stop_xmit(serial_channel *chan);

static cyg_uint32 easycan_serial_ISR(cyg_vector_t vector, cyg_addrword_t data);
static void       easycan_serial_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data);

static SERIAL_FUNS(easycan_serial_funs,
		   easycan_serial_putc,
		   easycan_serial_getc,
		   easycan_serial_set_config,
		   easycan_serial_start_xmit,
		   easycan_serial_stop_xmit
    );

#ifdef CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL0
static easycan_serial_info easycan_serial_info0 = {EASYCAN_USART0_BASE,
					   CYGNUM_HAL_INTERRUPT_UART0};
#if CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL0_BUFSIZE > 0
static unsigned char easycan_serial_out_buf0[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL0_BUFSIZE];
static unsigned char easycan_serial_in_buf0[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL0_BUFSIZE];

static SERIAL_CHANNEL_USING_INTERRUPTS(easycan_serial_channel0,
				       easycan_serial_funs,
				       easycan_serial_info0,
				       CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL0_BAUD),
				       CYG_SERIAL_STOP_DEFAULT,
				       CYG_SERIAL_PARITY_DEFAULT,
				       CYG_SERIAL_WORD_LENGTH_DEFAULT,
				       CYG_SERIAL_FLAGS_DEFAULT,
				       &easycan_serial_out_buf0[0],
				       sizeof(easycan_serial_out_buf0),
				       &easycan_serial_in_buf0[0],
				       sizeof(easycan_serial_in_buf0)
    );
#else
static SERIAL_CHANNEL(easycan_serial_channel0,
		      easycan_serial_funs,
		      easycan_serial_info0,
		      CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL0_BAUD),
		      CYG_SERIAL_STOP_DEFAULT,
		      CYG_SERIAL_PARITY_DEFAULT,
		      CYG_SERIAL_WORD_LENGTH_DEFAULT,
		      CYG_SERIAL_FLAGS_DEFAULT
    );
#endif

DEVTAB_ENTRY(easycan_serial_io0,
	     CYGDAT_IO_SERIAL_ARM_EVMCAN_SERIAL0_NAME,
	     0,                    // Does not depend on a lower level interface
	     &serial_devio,
	     easycan_serial_init,
	     easycan_serial_lookup,     // Serial driver may need initializing
	     &easycan_serial_channel0
    );
#endif //  CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL0

#ifdef CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL1
static easycan_serial_info easycan_serial_info1 = {EASYCAN_USART1_BASE,
					   CYGNUM_HAL_INTERRUPT_UART1};
#if CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL1_BUFSIZE > 0
static unsigned char easycan_serial_out_buf1[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL1_BUFSIZE];
static unsigned char easycan_serial_in_buf1[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL1_BUFSIZE];

static SERIAL_CHANNEL_USING_INTERRUPTS(easycan_serial_channel1,
				       easycan_serial_funs,
				       easycan_serial_info1,
				       CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL1_BAUD),
				       CYG_SERIAL_STOP_DEFAULT,
				       CYG_SERIAL_PARITY_DEFAULT,
				       CYG_SERIAL_WORD_LENGTH_DEFAULT,
				       CYG_SERIAL_FLAGS_DEFAULT,
				       &easycan_serial_out_buf1[0],
				       sizeof(easycan_serial_out_buf1),
				       &easycan_serial_in_buf1[0],
				       sizeof(easycan_serial_in_buf1)
    );
#else
static SERIAL_CHANNEL(easycan_serial_channel1,
		      easycan_serial_funs,
		      easycan_serial_info1,
		      CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL1_BAUD),
		      CYG_SERIAL_STOP_DEFAULT,
		      CYG_SERIAL_PARITY_DEFAULT,
		      CYG_SERIAL_WORD_LENGTH_DEFAULT,
		      CYG_SERIAL_FLAGS_DEFAULT
    );
#endif

DEVTAB_ENTRY(easycan_serial_io1,
	     CYGDAT_IO_SERIAL_ARM_EVMCAN_SERIAL1_NAME,
	     0,                    // Does not depend on a lower level interface
	     &serial_devio,
	     easycan_serial_init,
	     easycan_serial_lookup,     // Serial driver may need initializing
	     &easycan_serial_channel1
    );
#endif //  CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL1

#ifdef CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL2
static easycan_serial_info easycan_serial_info2 = {EASYCAN_USART2_BASE,
        CYGNUM_HAL_INTERRUPT_UART2};
#if CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL2_BUFSIZE > 0
static unsigned char easycan_serial_out_buf2[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL2_BUFSIZE];
static unsigned char easycan_serial_in_buf2[CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL2_BUFSIZE];

static SERIAL_CHANNEL_USING_INTERRUPTS(easycan_serial_channel2,
           easycan_serial_funs,
           easycan_serial_info2,
           CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL2_BAUD),
           CYG_SERIAL_STOP_DEFAULT,
           CYG_SERIAL_PARITY_DEFAULT,
           CYG_SERIAL_WORD_LENGTH_DEFAULT,
           CYG_SERIAL_FLAGS_DEFAULT,
           &easycan_serial_out_buf2[0],
           sizeof(easycan_serial_out_buf2),
           &easycan_serial_in_buf2[0],
           sizeof(easycan_serial_in_buf2)
    );
#else
static SERIAL_CHANNEL(easycan_serial_channel2,
        easycan_serial_funs,
        easycan_serial_info2,
        CYG_SERIAL_BAUD_RATE(CYGNUM_IO_SERIAL_ARM_EVMCAN_SERIAL2_BAUD),
        CYG_SERIAL_STOP_DEFAULT,
        CYG_SERIAL_PARITY_DEFAULT,
        CYG_SERIAL_WORD_LENGTH_DEFAULT,
        CYG_SERIAL_FLAGS_DEFAULT
    );
#endif

DEVTAB_ENTRY(easycan_serial_io2,
      CYGDAT_IO_SERIAL_ARM_EVMCAN_SERIAL2_NAME,
      0,                    // Does not depend on a lower level interface
      &serial_devio,
      easycan_serial_init,
      easycan_serial_lookup,     // Serial driver may need initializing
      &easycan_serial_channel2
    );
#endif //  CYGPKG_IO_SERIAL_ARM_EVMCAN_SERIAL2




// Internal function to actually configure the hardware to desired baud rate, etc.
static bool
easycan_serial_config_port(serial_channel *chan, cyg_serial_info_t *new_config, bool init)
{
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
    unsigned short baud_divisor = select_baud[new_config->baud];
    int loop_count;

    if (baud_divisor == 0) return false; // Invalid baud rate selected

    /* disable Tx and Rx and clear errors */
    usart->CR = TXDIS |RXDIS | RSTTX | RSTRX;
    for (loop_count = 1000 ; loop_count > 0 ; loop_count--);

 
    if (init)
    {
     
   /* Switch On USART */
  usart->ECR = (USART | PIO);

  /* Reset Error Flags Rx & Tx State Machine */
  usart->CR = (RSTRX | RSTTX);
  usart->CR =  RSTSTA;   
     
	/* Initialize the channel */
	    /* Set Rx & Tx Line as Peripheral Lines */
    usart->PDR = (TXD | RXD);

    }

    // Set databits, stopbits and parity.
    usart->MR = select_word_length[(new_config->word_length -
				    CYGNUM_SERIAL_WORD_LENGTH_5)]   |
			select_stop_bits[new_config->stop]	    |
			select_parity[new_config->parity]	    |
			USCLKS_MCKI | CHMODE_NORMAL;

    // Set baud rate.
    usart->BRGR = baud_divisor;

    // Enable interupts here
    usart->IER = ENDTX|ENDRX;

    /* enable Tx and Rx */
    usart->CR = TXEN|RXEN;
    for (loop_count = 1000 ; loop_count > 0 ; loop_count--);

    if (new_config != &chan->config) {
	chan->config = *new_config;
    }
    return true;
}

// Function to initialize the device.  Called at bootstrap time.
static bool
easycan_serial_init(struct cyg_devtab_entry *tab)
{
    serial_channel *chan = (serial_channel *)tab->priv;
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
#ifdef CYGDBG_IO_INIT
    diag_printf("EASYCAN SERIAL init - dev: %x.%d\n", easycan_chan->base, easycan_chan->int_num);
#endif
    (chan->callbacks->serial_init)(chan);  // Really only required for interrupt driven devices

    if (chan->out_cbuf.len != 0) {
        cyg_drv_interrupt_create(easycan_chan->int_num,
                                 99,                     // Priority - unused
                                 (cyg_addrword_t)chan,   //  Data item passed to interrupt handler
                                 easycan_serial_ISR,
                                 easycan_serial_DSR,
                                 &easycan_chan->serial_interrupt_handle,
                                 &easycan_chan->serial_interrupt);
        cyg_drv_interrupt_attach(easycan_chan->serial_interrupt_handle);
	hal_interrupt_configure(easycan_chan->int_num, 0, 1);
        cyg_drv_interrupt_unmask(easycan_chan->int_num);
    }

    easycan_serial_config_port(chan, &chan->config, true);
    return true;
}

// This routine is called when the device is "looked" up (i.e. attached)
static Cyg_ErrNo
easycan_serial_lookup(struct cyg_devtab_entry **tab,
                  struct cyg_devtab_entry *sub_tab,
                  const char *name)
{
    serial_channel *chan = (serial_channel *)(*tab)->priv;
    (chan->callbacks->serial_init)(chan);  // Really only required for interrupt driven devices
    return ENOERR;
}

// Send a character to the device output buffer.
// Return 'true' if character is sent to device
static bool
easycan_serial_putc(serial_channel *chan, unsigned char c)
{
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
    if (usart->SR & TXRDY) {
    // Transmit buffer is empty
        usart->THR = c;
        return true;
    } else {
    // No space
        return false;
    }
}

// Fetch a character from the device input buffer, waiting if necessary
static unsigned char
easycan_serial_getc(serial_channel *chan)
{
    unsigned char c;
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
    while ((usart->SR & RXRDY) == 0) ;   // Wait for char
    c = usart->RHR;
    return c;
}

// Set up the device characteristics; baud rate, etc.
static bool
easycan_serial_set_config(serial_channel *chan, cyg_serial_info_t *config)
{
    return easycan_serial_config_port(chan, config, false);
}

// Enable the transmitter on the device
static void
easycan_serial_start_xmit(serial_channel *chan)
{

    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
//    port->ier |= IER_XMT;  // Enable xmit interrupt
}

// Disable the transmitter on the device
static void
easycan_serial_stop_xmit(serial_channel *chan)
{
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
//    port->ier &= ~IER_XMT;  // Disable xmit interrupt
}

// Serial I/O - low level interrupt handler (ISR)
static cyg_uint32
easycan_serial_ISR(cyg_vector_t vector, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    cyg_drv_interrupt_mask(easycan_chan->int_num);
    cyg_drv_interrupt_acknowledge(easycan_chan->int_num);
    return CYG_ISR_CALL_DSR;  // Cause DSR to be run
}

// Serial I/O - high level interrupt handler (DSR)
static void
easycan_serial_DSR(cyg_vector_t vector, cyg_ucount32 count, cyg_addrword_t data)
{
    serial_channel *chan = (serial_channel *)data;
    easycan_serial_info *easycan_chan = (easycan_serial_info *)chan->dev_priv;
    volatile struct EASYCAN_USART_REGS *usart =
			    (volatile struct EASYCAN_USART_REGS*)easycan_chan->base;
    if (usart->SR & TXRDY) {
        (chan->callbacks->xmt_char)(chan);
    }
    if (usart->SR & RXRDY) {
        (chan->callbacks->rcv_char)(chan, usart->RHR);
    }
    cyg_drv_interrupt_unmask(easycan_chan->int_num);
}

#endif
