/****************************************************************************
** $Id: qinputcontextplugin.cpp,v 1.2 2004/06/20 18:43:11 daisuke Exp $
**
** Implementation of QInputContextPlugin class
**
** Created : 010920
**
** Copyright (C) 2001 Trolltech AS.  All rights reserved.
**
** This file is part of the widgets module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#include "qinputcontextplugin.h"

#ifndef QT_NO_IM
#ifndef QT_NO_COMPONENT

#include "qinputcontextinterface_p.h"

/*!
    \class QInputContextPlugin qinputcontextplugin.h
    \brief The QInputContextPlugin class provides an abstract base for custom QInputContext plugins.
    \reentrant
    \ingroup plugins

    The input context plugin is a simple plugin interface that makes it
    easy to create custom input contexts that can be loaded dynamically
    into applications.

    Writing a input context plugin is achieved by subclassing this
    base class, reimplementing the pure virtual functions keys(),
    create(), languages(), displayName() description() and exporting
    the class with the \c Q_EXPORT_PLUGIN macro.  See the \link
    plugins-howto.html Qt Plugins documentation \endlink for details.

    \sa QInputContext
*/

/*!
    \fn QStringList QInputContextPlugin::keys() const

    Returns the list of QInputContext keys this plugin provides.

    These keys are usually the class names of the custom input context
    that are implemented in the plugin.

    Return value is the names to identify and specify input methods
    for the input method switching mechanism and so on. The names have
    to be consistent with QInputContext::identifierName(). The names
    have to consist of ASCII characters only. See also
    QInputContext::identifierName() for further information.

    \sa create(), displayName(), QInputContext::identifierName()
*/

/*!
    \fn QInputContext* QInputContextPlugin::create( const QString& key )

    Creates and returns a QInputContext instance for the input context key \a key.
    The input context key is usually the class name of the required input method.

    \sa keys()
*/

/*!
    \fn QStringList languages( const QString &key )

    Returns what languages are supported by the QInputContext instance
    specified by \a key.

    The languages are expressed as language code (e.g. "zh_CN",
    "zh_TW", "zh_HK", "ja", "ko", ...). An input context that suports
    multiple languages can return all supported languages as
    QStringList. The name has to be consistent with
    QInputContextPlugin::language().

    This information may be used to optimize user interface.

    \sa QInputContext::language()
*/

/*!
    \fn QString displayName( const QString &key )

    Returns a user friendly i18n-ized name of the QInputContext
    instance specified by \a key. This string may be appeared in a
    menu and so on for users.

    There are two different names with different responsibility in the
    input method domain. This function returns one of them. Another
    name is called 'identifier name' to identify and specify input
    methods for the input method switching mechanism and so on.

    Although tr( identifierName ) can provide user friendly i18n-ized
    name without this function, the message catalog have to be managed
    by Qt in the case. However, some sophisticated input method
    framework manages their own message catalogs to provide this
    i18n-ized name string. So we need this function rather than just
    call tr() for identifier name.

    \sa keys(), QInputContext::identifierName()
*/

/*!
    \fn QString description( const QString &key )

    Returns a i18n-ized brief description of the QInputContext
    instance specified by \a key. This string may be appeared in some
    user interfaces.
*/



class QInputContextPluginPrivate : public QInputContextFactoryInterface
{
public:
    QInputContextPluginPrivate( QInputContextPlugin *p )
	: plugin( p )
    {
    }

    virtual ~QInputContextPluginPrivate();

    QRESULT queryInterface( const QUuid &iid, QUnknownInterface **iface );
    Q_REFCOUNT;

    QStringList featureList() const;
    QInputContext *create( const QString &key );
    QStringList languages( const QString &key );
    QString displayName( const QString &key );
    QString description( const QString &key );

private:
    QInputContextPlugin *plugin;
};

QRESULT QInputContextPluginPrivate::queryInterface( const QUuid &iid, QUnknownInterface **iface )
{
    *iface = 0;

    if ( iid == IID_QUnknown )
	*iface = this;
    else if ( iid == IID_QFeatureList )
	*iface = this;
    else if ( iid == IID_QInputContextFactory )
	*iface = this;
    else
	return QE_NOINTERFACE;

    (*iface)->addRef();
    return QS_OK;
}

QInputContextPluginPrivate::~QInputContextPluginPrivate()
{
    delete plugin;
}

QStringList QInputContextPluginPrivate::featureList() const
{
    return plugin->keys();
}

QInputContext *QInputContextPluginPrivate::create( const QString &key )
{
    return plugin->create( key );
}

QStringList QInputContextPluginPrivate::languages( const QString &key )
{
    return plugin->languages( key );
}

QString QInputContextPluginPrivate::displayName( const QString &key )
{
    return plugin->displayName( key );
}

QString QInputContextPluginPrivate::description( const QString &key )
{
    return plugin->description( key );
}


/*!
    Constructs a input context plugin. This is invoked automatically by the
    \c Q_EXPORT_PLUGIN macro.
*/
QInputContextPlugin::QInputContextPlugin()
    : QGPlugin( d = new QInputContextPluginPrivate( this ) )
{
}

/*!
    Destroys the input context plugin.

    You never have to call this explicitly. Qt destroys a plugin
    automatically when it is no longer used.
*/
QInputContextPlugin::~QInputContextPlugin()
{
    // don't delete d, as this is deleted by d
}

#endif // QT_NO_COMPONENT
#endif // QT_NO_IM
