#! /bin/sh
# check script for Lziprecover - Data recovery tool for the lzip format
# Copyright (C) 2009-2017 Antonio Diaz Diaz.
#
# This script is free software: you have unlimited permission
# to copy, distribute and modify it.

LC_ALL=C
export LC_ALL
objdir=`pwd`
testdir=`cd "$1" ; pwd`
LZIP="${objdir}"/lziprecover
LZIPRECOVER="${LZIP}"
framework_failure() { echo "failure in testing framework" ; exit 1 ; }

if [ ! -f "${LZIP}" ] || [ ! -x "${LZIP}" ] ; then
	echo "${LZIP}: cannot execute"
	exit 1
fi

[ -e "${LZIP}" ] 2> /dev/null ||
	{
	echo "$0: a POSIX shell is required to run the tests"
	echo "Try bash -c \"$0 $1 $2\""
	exit 1
	}

if [ -d tmp ] ; then rm -rf tmp ; fi
mkdir tmp
cd "${objdir}"/tmp || framework_failure

cat "${testdir}"/test.txt > in || framework_failure
in_lz="${testdir}"/test.txt.lz
in_lzma="${testdir}"/test.txt.lzma
inD="${testdir}"/test21723.txt
fox6_lz="${testdir}"/fox6.lz
f6b1="${testdir}"/fox6_bad1.txt
f6b1_lz="${testdir}"/fox6_bad1.lz
f6b2_lz="${testdir}"/fox6_bad2.lz
f6b3_lz="${testdir}"/fox6_bad3.lz
f6b4_lz="${testdir}"/fox6_bad4.lz
f6b5_lz="${testdir}"/fox6_bad5.lz
bad1_lz="${testdir}"/test_bad1.lz
bad2_lz="${testdir}"/test_bad2.lz
bad3_lz="${testdir}"/test_bad3.lz
bad4_lz="${testdir}"/test_bad4.lz
bad5_lz="${testdir}"/test_bad5.lz
fail=0
test_failed() { fail=1 ; printf " $1" ; [ -z "$2" ] || printf "($2)" ; }

# Description of test files for lziprecover:
# fox6_bad1.lz: byte at offset   5 changed from 0x0C to 0x00 (DS)
#               byte at offset 142 changed from 0x50 to 0x70 (CRC)
#               byte at offset 224 changed from 0x2D to 0x2E (data_size)
#               byte at offset 268 changed from 0x34 to 0x33 (mid stream)
#               byte at offset 327 changed from 0x2A to 0x2B (byte 7)
#               byte at offset 458 changed from 0xA0 to 0x20 (EOS marker)
# fox6_bad2.lz: [110-129] --> zeroed
# fox6_bad3.lz: [180-379] --> zeroed
# fox6_bad4.lz: [330-429] --> zeroed
# fox6_bad5.lz: [380-479] --> zeroed
# test_bad1.lz: byte at offset 66 changed from 0xA6 to 0x46
# test_bad2.lz: [  34-  65] --> copy of bytes [  68-  99]
# test_bad3.lz: [ 512-1535] --> zeroed          [2560-3583] --> zeroed
# test_bad4.lz: [3072-4095] --> random errors   [4608-5631] --> zeroed
# test_bad5.lz: [1024-2047] --> random errors   [5120-6143] --> random data

printf "testing lziprecover-%s..." "$2"

"${LZIP}" -lq in
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -tq in
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -tq < in
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -cdq in
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -cdq < in
[ $? = 2 ] || test_failed $LINENO
# these are for code coverage
"${LZIP}" -lt "${in_lz}" 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -cdl "${in_lz}" > out 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -cdt "${in_lz}" > out 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -t -- nx_file 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --help > /dev/null || test_failed $LINENO
"${LZIP}" -n1 -V > /dev/null || test_failed $LINENO
"${LZIP}" -m 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -z 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --bad_option 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --t 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --test=2 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --output= 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" --output 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
printf "LZIP\001-.............................." | "${LZIP}" -t 2> /dev/null
printf "LZIP\002-.............................." | "${LZIP}" -t 2> /dev/null
printf "LZIP\001+.............................." | "${LZIP}" -t 2> /dev/null
"${LZIPRECOVER}" -mq "${bad1_lz}"
[ $? = 1 ] || test_failed $LINENO
"${LZIPRECOVER}" -Rq
[ $? = 1 ] || test_failed $LINENO
"${LZIPRECOVER}" -sq
[ $? = 1 ] || test_failed $LINENO

"${LZIPRECOVER}" -Aq in
[ $? = 2 ] || test_failed $LINENO
"${LZIPRECOVER}" -Aq < in > copy.lz	# /dev/null returns 1 on OS/2
[ $? = 2 ] || test_failed $LINENO
"${LZIPRECOVER}" -Aq < "${in_lz}" > copy.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIPRECOVER}" -Aq "${in_lz}"
[ $? = 1 ] || test_failed $LINENO
"${LZIPRECOVER}" -Akq "${in_lzma}"
[ $? = 1 ] || test_failed $LINENO
"${LZIPRECOVER}" -Ac "${in_lzma}" > copy.lz
{ [ $? = 0 ] && cmp "${in_lz}" copy.lz ; } || test_failed $LINENO
"${LZIPRECOVER}" -A < "${in_lzma}" > copy.lz
{ [ $? = 0 ] && cmp "${in_lz}" copy.lz ; } || test_failed $LINENO
rm -f copy.lz
cat "${in_lzma}" > copy.lzma || framework_failure
"${LZIPRECOVER}" -Ak copy.lzma
{ [ $? = 0 ] && cmp "${in_lz}" copy.lz ; } || test_failed $LINENO
printf "to be overwritten" > copy.lz || framework_failure
"${LZIPRECOVER}" -Af copy.lzma
{ [ $? = 0 ] && cmp "${in_lz}" copy.lz ; } || test_failed $LINENO
rm -f copy.lz
cat "${in_lzma}" > copy.tlz || framework_failure
"${LZIPRECOVER}" -Ak copy.tlz
{ [ $? = 0 ] && cmp "${in_lz}" copy.tar.lz ; } || test_failed $LINENO
printf "to be overwritten" > copy.tar.lz || framework_failure
"${LZIPRECOVER}" -Af copy.tlz
{ [ $? = 0 ] && cmp "${in_lz}" copy.tar.lz ; } || test_failed $LINENO
rm -f copy.tar.lz
cat "${in_lzma}" > anyothername || framework_failure
"${LZIPRECOVER}" -A -o copy - anyothername - < "${in_lzma}"
{ [ $? = 0 ] && cmp "${in_lz}" copy.lz && cmp "${in_lz}" anyothername.lz ; } ||
	test_failed $LINENO
rm -f copy.lz anyothername.lz

printf "\ntesting decompression..."

"${LZIP}" -lq "${in_lz}" || test_failed $LINENO
"${LZIP}" -t "${in_lz}" || test_failed $LINENO
"${LZIP}" -cd "${in_lz}" > copy || test_failed $LINENO
cmp in copy || test_failed $LINENO

rm -f copy
cat "${in_lz}" > copy.lz || framework_failure
"${LZIP}" -dk copy.lz || test_failed $LINENO
cmp in copy || test_failed $LINENO
printf "to be overwritten" > copy || framework_failure
"${LZIP}" -d copy.lz 2> /dev/null
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -df copy.lz
{ [ $? = 0 ] && [ ! -e copy.lz ] && cmp in copy ; } || test_failed $LINENO

printf "to be overwritten" > copy || framework_failure
"${LZIP}" -df -o copy < "${in_lz}" || test_failed $LINENO
cmp in copy || test_failed $LINENO

rm -f copy
cat "${in_lz}" > anyothername || framework_failure
"${LZIP}" -dv --output copy - anyothername - < "${in_lz}" 2> /dev/null
{ [ $? = 0 ] && cmp in copy && cmp in anyothername.out ; } ||
	test_failed $LINENO
rm -f copy anyothername.out

"${LZIP}" -lq in "${in_lz}"
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -lq nx_file.lz "${in_lz}"
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -tq in "${in_lz}"
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -tq nx_file.lz "${in_lz}"
[ $? = 1 ] || test_failed $LINENO
"${LZIP}" -cdq in "${in_lz}" > copy
{ [ $? = 2 ] && cat copy in | cmp in - ; } || test_failed $LINENO
"${LZIP}" -cdq nx_file.lz "${in_lz}" > copy
{ [ $? = 1 ] && cmp in copy ; } || test_failed $LINENO
rm -f copy
cat "${in_lz}" > copy.lz || framework_failure
for i in 1 2 3 4 5 6 7 ; do
	printf "g" >> copy.lz || framework_failure
	"${LZIP}" -alvv copy.lz "${in_lz}" > /dev/null 2>&1
	[ $? = 2 ] || test_failed $LINENO $i
	"${LZIP}" -atvvvv copy.lz "${in_lz}" 2> /dev/null
	[ $? = 2 ] || test_failed $LINENO $i
done
"${LZIP}" -dq in copy.lz
{ [ $? = 2 ] && [ -e copy.lz ] && [ ! -e copy ] && [ ! -e in.out ] ; } ||
	test_failed $LINENO
"${LZIP}" -dq nx_file.lz copy.lz
{ [ $? = 1 ] && [ ! -e copy.lz ] && [ ! -e nx_file ] && cmp in copy ; } ||
	test_failed $LINENO

cat in in > in2 || framework_failure
cat "${in_lz}" "${in_lz}" > in2.lz || framework_failure
"${LZIP}" -lq in2.lz || test_failed $LINENO
"${LZIP}" -t in2.lz || test_failed $LINENO
"${LZIP}" -cd in2.lz > copy2 || test_failed $LINENO
cmp in2 copy2 || test_failed $LINENO

cat in2.lz > copy2.lz || framework_failure
printf "\ngarbage" >> copy2.lz || framework_failure
"${LZIP}" -tvvvv copy2.lz 2> /dev/null || test_failed $LINENO
rm -f copy2
"${LZIP}" -aD0 -q copy2.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -alq copy2.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -atq copy2.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -atq < copy2.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -adkq copy2.lz
{ [ $? = 2 ] && [ ! -e copy2 ] ; } || test_failed $LINENO
"${LZIP}" -adkq -o copy2 < copy2.lz
{ [ $? = 2 ] && [ ! -e copy2 ] ; } || test_failed $LINENO
printf "to be overwritten" > copy2 || framework_failure
"${LZIP}" -df copy2.lz || test_failed $LINENO
cmp in2 copy2 || test_failed $LINENO

"${LZIPRECOVER}" -D ,18000 "${in_lz}" > copy || test_failed $LINENO
"${LZIPRECOVER}" -D 18000 "${in_lz}" >> copy || test_failed $LINENO
cmp in copy || test_failed $LINENO
"${LZIPRECOVER}" -D 21723-22120 -fo copy "${in_lz}" || test_failed $LINENO
cmp "${inD}" copy || test_failed $LINENO
"${LZIPRECOVER}" -D 21723,397 "${in_lz}" > copy || test_failed $LINENO
cmp "${inD}" copy || test_failed $LINENO
"${LZIPRECOVER}" -D0 -iq "${f6b1_lz}" -fo copy
{ [ $? = 2 ] && cmp "${f6b1}" copy ; } || test_failed $LINENO
"${LZIPRECOVER}" -D0 -iq "${f6b1_lz}" > copy
{ [ $? = 2 ] && cmp "${f6b1}" copy ; } || test_failed $LINENO

printf "\ntesting bad input..."

cat "${in_lz}" "${in_lz}" "${in_lz}" > in3.lz || framework_failure
if dd if=in3.lz of=trunc.lz bs=14752 count=1 2> /dev/null &&
   [ -e trunc.lz ] && cmp in2.lz trunc.lz > /dev/null 2>&1 ; then
	for i in 6 20 14734 14753 14754 14755 14756 14757 14758 ; do
		dd if=in3.lz of=trunc.lz bs=$i count=1 2> /dev/null
		"${LZIP}" -lq trunc.lz
		[ $? = 2 ] || test_failed $LINENO $i
		"${LZIP}" -t trunc.lz 2> /dev/null
		[ $? = 2 ] || test_failed $LINENO $i
		"${LZIP}" -tq < trunc.lz
		[ $? = 2 ] || test_failed $LINENO $i
		"${LZIP}" -cdq trunc.lz > out
		[ $? = 2 ] || test_failed $LINENO $i
		"${LZIP}" -dq < trunc.lz > out
		[ $? = 2 ] || test_failed $LINENO $i
	done
else
	printf "\nwarning: skipping truncation test: 'dd' does not work on your system."
fi

cat "${in_lz}" > ingin.lz || framework_failure
printf "g" >> ingin.lz || framework_failure
cat "${in_lz}" >> ingin.lz || framework_failure
"${LZIP}" -lq ingin.lz
[ $? = 2 ] || test_failed $LINENO
"${LZIP}" -t ingin.lz || test_failed $LINENO
"${LZIP}" -cd ingin.lz > copy || test_failed $LINENO
cmp in copy || test_failed $LINENO
"${LZIP}" -t < ingin.lz || test_failed $LINENO
"${LZIP}" -d < ingin.lz > copy || test_failed $LINENO
cmp in copy || test_failed $LINENO

printf "\ntesting --merge..."

rm -f copy.lz
"${LZIPRECOVER}" -m -o copy.lz "${fox6_lz}" "${f6b1_lz}"
{ [ $? = 0 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -m -o copy.lz "${f6b1_lz}" "${fox6_lz}"
{ [ $? = 0 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -m -o copy.lz "${bad1_lz}" "${bad2_lz}" "${bad1_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -m -o copy.lz "${bad1_lz}" "${bad2_lz}" "${bad2_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
cat "${bad2_lz}" > bad2.lz || framework_failure
"${LZIPRECOVER}" -m -o copy.lz "${bad1_lz}" "${bad2_lz}" bad2.lz -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
rm -f bad2.lz
"${LZIPRECOVER}" -m -o copy.lz "${f6b1_lz}" "${f6b5_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -m -o copy.lz "${f6b3_lz}" "${f6b5_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -m -o copy.lz "${bad3_lz}" "${bad4_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO

"${LZIPRECOVER}" -mf -o copy.lz "${f6b1_lz}" "${f6b4_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${f6b4_lz}" "${f6b1_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO

for i in "${f6b1_lz}" "${f6b3_lz}" "${f6b4_lz}" "${f6b5_lz}" ; do
	"${LZIPRECOVER}" -mf -o copy.lz "${f6b2_lz}" "$i" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "$i" "${f6b2_lz}" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
done

for i in "${f6b3_lz}" "${f6b4_lz}" "${f6b5_lz}" ; do
	"${LZIPRECOVER}" -mf -o copy.lz "${f6b1_lz}" "${f6b2_lz}" "$i" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "${f6b1_lz}" "$i" "${f6b2_lz}" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "${f6b2_lz}" "${f6b1_lz}" "$i" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "${f6b2_lz}" "$i" "${f6b1_lz}" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "$i" "${f6b1_lz}" "${f6b2_lz}" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
	"${LZIPRECOVER}" -mf -o copy.lz "$i" "${f6b2_lz}" "${f6b1_lz}" ||
		test_failed $LINENO "$i"
	cmp "${fox6_lz}" copy.lz || test_failed $LINENO "$i"
done

"${LZIPRECOVER}" -mf -o copy.lz "${f6b3_lz}" "${f6b4_lz}" "${f6b5_lz}" ||
	test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${f6b1_lz}" "${f6b3_lz}" "${f6b4_lz}" \
"${f6b5_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${f6b2_lz}" "${f6b3_lz}" "${f6b4_lz}" \
"${f6b5_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${f6b1_lz}" "${f6b2_lz}" "${f6b3_lz}" \
"${f6b4_lz}" "${f6b5_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO

"${LZIPRECOVER}" -mf -o copy.lz "${bad1_lz}" "${bad2_lz}" || test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad2_lz}" "${bad1_lz}" || test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO

cat "${bad1_lz}" "${in_lz}" "${bad1_lz}" "${bad1_lz}" > bad11.lz || framework_failure
cat "${bad1_lz}" "${in_lz}" "${bad2_lz}" "${in_lz}" > bad12.lz || framework_failure
cat "${bad2_lz}" "${in_lz}" "${bad2_lz}" "${bad2_lz}" > bad22.lz || framework_failure
cat "${in_lz}" "${in_lz}" "${in_lz}" "${in_lz}" > copy4.lz || framework_failure
"${LZIPRECOVER}" -mf -o out4.lz bad11.lz bad12.lz bad22.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad11.lz bad22.lz bad12.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad12.lz bad11.lz bad22.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad12.lz bad22.lz bad11.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad22.lz bad11.lz bad12.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad22.lz bad12.lz bad11.lz || test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO

for i in "${bad1_lz}" "${bad2_lz}" ; do
	for j in "${bad3_lz}" "${bad4_lz}" "${bad5_lz}" ; do
		"${LZIPRECOVER}" -mf -o copy.lz "$i" "$j" ||
			test_failed $LINENO "$i $j"
		cmp "${in_lz}" copy.lz || test_failed $LINENO "$i $j"
		"${LZIPRECOVER}" -mf -o copy.lz "$j" "$i" ||
			test_failed $LINENO "$i $j"
		cmp "${in_lz}" copy.lz || test_failed $LINENO "$i $j"
	done
done

"${LZIPRECOVER}" -mf -o copy.lz "${bad3_lz}" "${bad4_lz}" "${bad5_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad3_lz}" "${bad5_lz}" "${bad4_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad4_lz}" "${bad3_lz}" "${bad5_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad4_lz}" "${bad5_lz}" "${bad3_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad5_lz}" "${bad3_lz}" "${bad4_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o copy.lz "${bad5_lz}" "${bad4_lz}" "${bad3_lz}" ||
	test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO

cat "${bad3_lz}" "${bad4_lz}" "${bad5_lz}" "${in_lz}" > bad345.lz || framework_failure
cat "${bad4_lz}" "${bad5_lz}" "${bad3_lz}" "${in_lz}" > bad453.lz || framework_failure
cat "${bad5_lz}" "${bad3_lz}" "${bad4_lz}" "${in_lz}" > bad534.lz || framework_failure
cat "${in_lz}" "${in_lz}" "${in_lz}" "${in_lz}" > copy4.lz || framework_failure
"${LZIPRECOVER}" -mf -o out4.lz bad345.lz bad453.lz bad534.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad345.lz bad534.lz bad453.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad453.lz bad345.lz bad534.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad453.lz bad534.lz bad345.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad534.lz bad345.lz bad453.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO
"${LZIPRECOVER}" -mf -o out4.lz bad534.lz bad453.lz bad345.lz ||
	test_failed $LINENO
cmp out4.lz copy4.lz || test_failed $LINENO

printf "\ntesting --repair..."

rm -f copy.lz
"${LZIPRECOVER}" -R -o copy.lz "${fox6_lz}" || test_failed $LINENO
{ [ $? = 0 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -R -o copy.lz "${bad2_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -R -o copy.lz "${bad3_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -R -o copy.lz "${bad4_lz}" -q
{ [ $? = 2 ] && [ ! -e copy.lz ] ; } || test_failed $LINENO
"${LZIPRECOVER}" -Rf -o copy.lz "${f6b1_lz}" || test_failed $LINENO
cmp "${fox6_lz}" copy.lz || test_failed $LINENO
"${LZIPRECOVER}" -Rf -o copy.lz "${bad1_lz}" || test_failed $LINENO
cmp "${in_lz}" copy.lz || test_failed $LINENO

cat "${f6b1_lz}" > copy.tar.lz || framework_failure
"${LZIPRECOVER}" -R copy.tar.lz || test_failed $LINENO
{ [ $? = 0 ] && [ -e copy_fixed.tar.lz ] ; } || test_failed $LINENO
mv copy.tar.lz copy.lz || framework_failure
"${LZIPRECOVER}" -R copy.lz || test_failed $LINENO
{ [ $? = 0 ] && [ -e copy_fixed.lz ] ; } || test_failed $LINENO
mv copy.lz copy.tlz || framework_failure
"${LZIPRECOVER}" -R copy.tlz || test_failed $LINENO
{ [ $? = 0 ] && [ -e copy_fixed.tlz ] ; } || test_failed $LINENO

printf "\ntesting --split..."

cat "${in_lz}" "${in_lz}" "${in_lz}" > copy || framework_failure
printf "garbage" >> copy || framework_failure
"${LZIPRECOVER}" -s -o copy.lz copy || test_failed $LINENO
for i in 1 2 3 ; do
	"${LZIPRECOVER}" -cd rec${i}copy.lz > copy || test_failed $LINENO $i
	cmp in copy || test_failed $LINENO $i
done

echo
if [ ${fail} = 0 ] ; then
	echo "tests completed successfully."
	cd "${objdir}" && rm -r tmp
else
	echo "tests failed."
fi
exit ${fail}
