/* This file is part of the KDE project
   Copyright (C) 2005-2008 Jarosław Staniek <staniek@kde.org>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "SmallToolButton.h"
#include "utils.h"

#include <QDebug>
#include <QStyle>
#include <QStyleOption>
#include <QPainter>
#include <QPointer>
#include <QAction>
#include <QIcon>

//! @internal
class KexiSmallToolButton::Private
{
public:
    Private()
            : enableSlotButtonToggled(true)
            , enableSlotActionToggled(true) {
    }

    QPointer<QAction> action;
    bool enableSlotButtonToggled;
    bool enableSlotActionToggled;
};

//--------------------------------

KexiSmallToolButton::KexiSmallToolButton(QWidget* parent)
        : QToolButton(parent)
        , d(new Private)
{
    init();
    update(QString(), QIcon());
}

KexiSmallToolButton::KexiSmallToolButton(const QString& text, QWidget* parent)
        : QToolButton(parent)
        , d(new Private)
{
    init();
    update(text, QIcon());
}

KexiSmallToolButton::KexiSmallToolButton(const QIcon& icon, const QString& text,
        QWidget* parent)
        : QToolButton(parent)
        , d(new Private)
{
    init();
    update(text, icon);
}

KexiSmallToolButton::KexiSmallToolButton(const QIcon& icon, QWidget* parent)
        : QToolButton(parent)
        , d(new Private)
{
    init();
    update(QString(), icon);
    QToolButton::setToolButtonStyle(Qt::ToolButtonIconOnly);
}

KexiSmallToolButton::KexiSmallToolButton(QAction* action, QWidget* parent)
        : QToolButton(parent)
        , d(new Private)
{
    d->action = action;
    init();
    if (d->action) {
        connect(d->action, SIGNAL(changed()), this, SLOT(slotActionChanged()));
        update(d->action->text(), d->action->icon(), false);
        setEnabled(d->action->isEnabled());
        setToolTip(d->action->toolTip());
        setWhatsThis(d->action->whatsThis());
        setCheckable(d->action->isCheckable());
        if (d->action->menu()) {
            setPopupMode(QToolButton::MenuButtonPopup);
            setMenu(d->action->menu());
        } else {
            connect(this, SIGNAL(toggled(bool)), this, SLOT(slotButtonToggled(bool)));
            connect(d->action, SIGNAL(toggled(bool)), this, SLOT(slotActionToggled(bool)));
        }
    }
    connect(this, SIGNAL(clicked()), action, SLOT(trigger()));
    updateAction();
}

KexiSmallToolButton::~KexiSmallToolButton()
{
    delete d;
}

void KexiSmallToolButton::updateAction()
{

}

void KexiSmallToolButton::init()
{
    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Preferred);
    QFont f(font());
    f.setPointSizeF(KexiUtils::smallestReadableFont().pointSizeF());
    setFont(f);
    setAutoRaise(true);
    QToolButton::setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
}

void KexiSmallToolButton::setToolButtonStyle(Qt::ToolButtonStyle style)
{
    QToolButton::setToolButtonStyle(style);
    update(text(), icon(), false);
}

void KexiSmallToolButton::update(const QString& text, const QIcon& icon, bool tipToo)
{
    if (!text.isEmpty() && toolButtonStyle() != Qt::ToolButtonIconOnly) {
        if (toolButtonStyle() != Qt::ToolButtonTextOnly)
            QToolButton::setToolButtonStyle(Qt::ToolButtonTextBesideIcon);
        QToolButton::setText(text);
        if (tipToo)
            setToolTip(text);
    }
    if (toolButtonStyle() == Qt::ToolButtonTextOnly) {
        QToolButton::setIcon(QIcon());
    } else if (!icon.isNull()) {
        QToolButton::setIcon(icon);
    }
}

QSize KexiSmallToolButton::sizeHint() const
{
    return QToolButton::sizeHint()
           - QSize((toolButtonStyle() == Qt::ToolButtonTextBesideIcon) ? 4 : 0, 0);
}

void KexiSmallToolButton::setIcon(const QIcon& icon)
{
    update(text(), icon);
}

void KexiSmallToolButton::setIcon(const QString &iconName)
{
    setIcon(QIcon::fromTheme(iconName));
}

void KexiSmallToolButton::setText(const QString& text)
{
    update(text, icon());
}

void KexiSmallToolButton::slotActionChanged()
{
    setEnabled(d->action->isEnabled());
}

void KexiSmallToolButton::slotButtonToggled(bool checked)
{
    Q_UNUSED(checked);
    if (!d->enableSlotButtonToggled)
        return;
    //QObject *view = KDbUtils::findParent<QObject*>(this, "KexiView");
    //qDebug() << QString("checked=%1 action=%2 view=%3")
    // .arg(checked).arg(d->action ? d->action->text() : QString())
    // .arg(view ? view->objectName() : QString("??"));
    d->enableSlotActionToggled = false;
    d->enableSlotActionToggled = true;
}

void KexiSmallToolButton::slotActionToggled(bool checked)
{
    if (!d->enableSlotActionToggled)
        return;
    //QObject *view = KDbUtils::findParent<QObject*>(this, "KexiView");
    //qDebug() << QString("checked=%1 action=%2 view=%3")
    // .arg(checked).arg(d->action ? d->action->text() : QString())
    // .arg(view ? view->objectName() : QString("??"));
    d->enableSlotButtonToggled = false;
    setChecked(checked);
    d->enableSlotButtonToggled = true;
}

QAction* KexiSmallToolButton::action() const
{
    return d->action;
}

//------------------------------------------

class KexiToolBarSeparator::Private
{
public:
    Private();
    ~Private();

    Qt::Orientation orientation;
};

KexiToolBarSeparator::Private::Private()
    :orientation(Qt::Horizontal)
{

}

KexiToolBarSeparator::Private::~Private()
{

}

KexiToolBarSeparator::KexiToolBarSeparator(QWidget *parent)
        : QWidget(parent)
        , d(new Private())
{
    setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Minimum);
//! @todo
    setFixedHeight(parent->height() - 6);
}

KexiToolBarSeparator::~KexiToolBarSeparator()
{
    delete d;
}

void KexiToolBarSeparator::initStyleOption(QStyleOption *o) const
{
    o->initFrom(this);
    if (orientation() == Qt::Horizontal)
        o->state |= QStyle::State_Horizontal;
}

void KexiToolBarSeparator::setOrientation(Qt::Orientation o)
{
    d->orientation = o;
    update();
}

Qt::Orientation KexiToolBarSeparator::orientation() const
{
    return d->orientation;
}

QSize KexiToolBarSeparator::sizeHint() const
{
    QStyleOption o;
    initStyleOption(&o);
    const int sepExtent = style()->pixelMetric(
                              QStyle::PM_ToolBarSeparatorExtent, &o, 0);
    return QSize(sepExtent, sepExtent);
}

void KexiToolBarSeparator::paintEvent(QPaintEvent *e)
{
    Q_UNUSED(e);
    QPainter p(this);
    QStyleOption o;
    initStyleOption(&o);
    style()->drawPrimitive(QStyle::PE_IndicatorToolBarSeparator, &o, &p, parentWidget());
}

