/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* writerperfect
 * Version: MPL 2.0 / LGPLv2.1+
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * Major Contributor(s):
 * Copyright (C) 2002-2004 William Lachance (wrlach@gmail.com)
 * Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
 *
 * For minor contributions see the git repository.
 *
 * Alternatively, the contents of this file may be used under the terms
 * of the GNU Lesser General Public License Version 2.1 or later
 * (LGPLv2.1+), in which case the provisions of the LGPLv2.1+ are
 * applicable instead of those above.
 *
 * For further information visit http://libwpd.sourceforge.net
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <libwpd/libwpd.h>
#include <libwpg/libwpg.h>
#include <librvngabw/librvngabw.hxx>

#include "ABWStringDocumentHandler.hxx"
#include "UsageHelper.hxx"

#define TOOLNAME "wpd2abw"

using namespace libwpd;

class ABWOutputFileHelper
{
public:
	ABWOutputFileHelper(const char *outFileName) : mOutFileName(outFileName) {};
	~ABWOutputFileHelper() {};

	bool convertDocument(librevenge::RVNGInputStream &input, const char *password)
	{
		writerperfectrvngabw::ABWStringDocumentHandler contentHandler;
		librvngabw::ABWTextGenerator collector(&contentHandler);
		collector.registerEmbeddedImageHandler("image/x-wpg", &handleEmbeddedWPGImage);
		if (WPD_OK != WPDocument::parse(&input, &collector, password))
			return false;
		if (mOutFileName)
		{
			FILE *fhandle = fopen(mOutFileName, "wb");
			if (!fhandle)
				return false;
			fprintf(fhandle, "%s\n", contentHandler.getData().cstr());
			fclose(fhandle);
		}
		else
			printf("%s\n", contentHandler.getData().cstr());
		return true;
	}


	bool isSupportedFormat(librevenge::RVNGInputStream &input, const char *password)
	{
		WPDConfidence confidence = WPDocument::isFileFormatSupported(&input);
		if (WPD_CONFIDENCE_EXCELLENT != confidence && WPD_CONFIDENCE_SUPPORTED_ENCRYPTION != confidence)
		{
			fprintf(stderr, "ERROR: We have no confidence that you are giving us a valid WordPerfect document.\n");
			return false;
		}
		if (WPD_CONFIDENCE_SUPPORTED_ENCRYPTION == confidence && !password)
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and you did not give us a password.\n");
			return false;
		}
		if (confidence == WPD_CONFIDENCE_SUPPORTED_ENCRYPTION && password &&
		        (WPD_PASSWORD_MATCH_OK != WPDocument::verifyPassword(&input, password)))
		{
			fprintf(stderr, "ERROR: The WordPerfect document is encrypted and we either\n");
			fprintf(stderr, "ERROR: don't know how to decrypt it or the given password is wrong.\n");
			return false;
		}

		return true;
	}

private:

	static bool handleEmbeddedWPGImage(const librevenge::RVNGBinaryData &input, librevenge::RVNGBinaryData &output)
	{
		libwpg::WPGFileFormat fileFormat = libwpg::WPG_AUTODETECT;

		if (!libwpg::WPGraphics::isSupported(input.getDataStream()))
			fileFormat = libwpg::WPG_WPG1;

		librevenge::RVNGStringVector svgOutput;
		librevenge::RVNGSVGDrawingGenerator generator(svgOutput, "");
		bool result = libwpg::WPGraphics::parse(input.getDataStream(), &generator, fileFormat);
		if (!result || svgOutput.empty() || svgOutput[0].empty())
			return false;

		output.clear();
		const char *svgHeader = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"no\"?>\n"
		                        "<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\""
		                        " \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
		output.append((unsigned char *)svgHeader, strlen(svgHeader));
		output.append((unsigned char *)svgOutput[0].cstr(), strlen(svgOutput[0].cstr()));

		return true;
	}
	//! the ouput file name
	char const *mOutFileName;
};


int printUsage(const char *name)
{
	UsageHelper usage(name, "converts WordPerfect documents to AbiWord.");
	usage.addToDescription("If OUTPUT is missing, the result is printed to standard output.\n");
	usage.addStdoutOption();
	usage.addPasswordOption();
	return usage.printUsage();
}


int main(int argc, char *argv[])
{
	if (argc < 2)
		return printUsage(TOOLNAME);

	char *szInputFile = 0;
	char *szOutFile = 0;
	bool stdOutput = false;
	char *password = 0;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmp(argv[i], "--password"))
		{
			if (i < argc - 1)
				password = argv[++i];
		}
		else if (!strncmp(argv[i], "--password=", 11))
			password = &argv[i][11];
		else if (!strcmp(argv[i], "--stdout"))
			stdOutput = true;
		else if (!strcmp(argv[i], "--help"))
			return printUsage(TOOLNAME);
		else if (!strcmp(argv[i], "--version"))
			return UsageHelper::printVersion(TOOLNAME);
		else if (!szInputFile && strncmp(argv[i], "--", 2))
			szInputFile = argv[i];
		else if (szInputFile && !szOutFile && strncmp(argv[i], "--", 2))
			szOutFile = argv[i];
		else
			return printUsage(TOOLNAME);
	}

	if (!szInputFile)
		return printUsage(TOOLNAME);

	if (szOutFile && stdOutput)
		szOutFile = 0;

	ABWOutputFileHelper helper(szOutFile);
	librevenge::RVNGFileStream input(szInputFile);
	if (!helper.isSupportedFormat(input, password))
		return 1;

	if (!helper.convertDocument(input, password))
	{
		fprintf(stderr, "ERROR : Couldn't convert the document\n");
		return 1;
	}

	return 0;
}

/* vim:set shiftwidth=4 softtabstop=4 noexpandtab: */
