#!/bin/sh -e
#
# ploticus-doc package build script
#
# Package maintainers: if the original source is not distributed as a
# (possibly compressed) tarball, set the value of ${src_orig_pkg_name},
# and redefine the unpack() helper function appropriately.
# Also, if the Makefile rule to run the test suite is not "test", change
# the definition of ${test_rule} below.

# find out where the build script is located
tdir=`echo "$0" | sed 's%[\\/][^\\/][^\\/]*$%%'`
test "x$tdir" = "x$0" && tdir=.
scriptdir=`cd $tdir; pwd`
# find src directory.
# If scriptdir ends in SPECS, then topdir is $scriptdir/..
# If scriptdir ends in CYGWIN-PATCHES, then topdir is $scriptdir/../..
# Otherwise, we assume that topdir = scriptdir
topdir1=`echo ${scriptdir} | sed 's%/SPECS$%%'`
topdir2=`echo ${scriptdir} | sed 's%/CYGWIN-PATCHES$%%'`
if [ "x$topdir1" != "x$scriptdir" ] ; then # SPECS
  topdir=`cd ${scriptdir}/..; pwd`
else
  if [ "x$topdir2" != "x$scriptdir" ] ; then # CYGWIN-PATCHES
    topdir=`cd ${scriptdir}/../..; pwd`
  else
    topdir=`cd ${scriptdir}; pwd`
  fi
fi

scriptname=`basename $0`
tscriptname=`basename $0 .sh`
export PKG=`echo $tscriptname | sed -e 's/\-[^\-]*\-[^\-]*$//'`
export VER=`echo $tscriptname | sed -e "s/${PKG}\-//" -e 's/\-[^\-]*$//'`
export REL=`echo $tscriptname | sed -e "s/${PKG}\-${VER}\-//"`
export BASEPKG=${PKG}-${VER}
export FULLPKG=${BASEPKG}-${REL}

# determine correct decompression option and tarball filename
export src_orig_pkg_name=
if [ -e "${src_orig_pkg_name}" ] ; then
  export opt_decomp=? # Make sure tar punts if unpack() is not redefined
elif [ -e ${BASEPKG}.tar.bz2 ] ; then
  export opt_decomp=j
  export src_orig_pkg_name=${BASEPKG}.tar.bz2
elif [ -e ${BASEPKG}.tar.gz ] ; then
  export opt_decomp=z
  export src_orig_pkg_name=${BASEPKG}.tar.gz
elif [ -e ${BASEPKG}.tgz ] ; then
  export opt_decomp=z
  export src_orig_pkg_name=${BASEPKG}.tgz
elif [ -e ${BASEPKG}.tar ] ; then
  export opt_decomp=
  export src_orig_pkg_name=${BASEPKG}.tar
else
  echo Cannot find original package.
  exit 1
fi

export src_orig_pkg=${topdir}/${src_orig_pkg_name}

# determine correct names for generated files
export src_pkg_name=${FULLPKG}-src.tar.bz2
export src_patch_name=${FULLPKG}.patch
export bin_pkg_name=${FULLPKG}.tar.bz2

export src_pkg=${topdir}/${src_pkg_name}
export src_patch=${topdir}/${src_patch_name}
export bin_pkg=${topdir}/${bin_pkg_name}
export srcdir=${topdir}/${BASEPKG}
export instdir=${srcdir}/.inst
export srcinstdir=${srcdir}/.sinst

prefix=/usr
sysconfdir=/etc
localstatedir=/var
if [ -z "$MY_CFLAGS" ]; then
  MY_CFLAGS="-O2"
fi
if [ -z "$MY_LDFLAGS" ]; then
  MY_LDFLAGS=
fi

export install_docs="\
	ABOUT-NLS \
	ANNOUNCE \
	AUTHORS \
	BUG-REPORTS \
	CHANGES \
	CONTRIB* \
	COPYING \
	COPYRIGHT \
	CREDITS \
	CHANGELOG \
	ChangeLog* \
	FAQ \
	FreeType2 \
	GPL* \
	HOW-TO-CONTRIBUTE \
	INSTALL* \
	KNOWNBUG \
	LEGAL \
	LICENSE \
	NEWS \
	NOTES \
	PROGLIST \
	README \
	RELEASE_NOTES \
	THANKS \
	TODO \
"
export install_docs="`for i in ${install_docs}; do echo $i; done | sort -u`"
if [ -z "$SIG" ]; then
  export SIG=0	# set to 1 to turn on signing by default
fi

unpack() {
  tar x${opt_decomp}f "$1"
}
mkdirs() {
  (cd ${topdir}
  rmdirs
  mkdir -p ${instdir} ${srcinstdir} )
}
reldir() {
  # Return a relative path from directory $1 to directory $2,
  # i.e. $1/`reldir $1 $2` = $2 .
  # Assume $1 and $2 are absolute paths.
  (from=$1 to=$2
  # strip common leading path components
  while fd=$(expr "$from" : '\(/[^/]*\)') &&
        td=$(expr "$to"   : '\(/[^/]*\)') &&
        [ "$fd" = "$td" ]
  do
    from=$(expr $from : "$fd\(.*\)") || true
    to=$(  expr $to   : "$td\(.*\)") || true
  done
  # convert $from path components to ../
  from=$(echo $from | sed -e 's:/[^/][^/]*:\.\./:g' -e 's:\(.*\)/:\1:')
  if [ "$from" ] ; then
    echo ${from}${to}
  else
    echo "."
  fi
  )
}
stow() {
  # create in $2 a symlinked copy of the file tree in $1
  (src=$1 tgt=$2
  for sd in $(find ${src} \
    \( -path ${instdir} -o -path ${srcinstdir} \) \
    -prune -o -type d -print ) ; do
    td=${tgt}${sd#${src}}
    echo "stowing ${td#${topdir}/}"
    mkdir -p ${td}
	cd ${td}
    find $(reldir ${td} ${sd}) -maxdepth 1 \! -type d \
    | xargs -r ln -fs --target-dir=${td} 2>/dev/null || true
  done )
}
rmdirs() {
  (cd ${topdir}
  rm -fr ${instdir} ${srcinstdir} )
}
prep() {
  (cd ${topdir}
  unpack ${src_orig_pkg}
  if [ -f ${src_patch} ] ; then
    patch -Z -p0 < ${src_patch}
  fi
  mkdirs )
}
conf() {
  :
}
reconf() {
  :
}
build() {
  :
}
check() {
  :
}
clean() {
  :
}
install() {
  (rm -fr ${instdir}/*
  mkdir -p \
	${instdir}${prefix}/share/doc/${BASEPKG} \
	${instdir}${prefix}/share/doc/Cygwin \
	${instdir}${prefix}/share/doc/ploticus-${VER}/html
  cp -a \
	${topdir}/${BASEPKG}/README \
	${topdir}/${BASEPKG}/doc \
	${topdir}/${BASEPKG}/gallery \
	${topdir}/${BASEPKG}/index.html \
	${instdir}${prefix}/share/doc/ploticus-${VER}/html 
  if [ -f ${srcdir}/CYGWIN-PATCHES/${PKG}.README ]; then
    /usr/bin/install -p -m 644 ${srcdir}/CYGWIN-PATCHES/${PKG}.README \
      ${instdir}${prefix}/share/doc/Cygwin/${BASEPKG}.README
  elif [ -f ${srcdir}/CYGWIN-PATCHES/README ] ; then
    /usr/bin/install -p -m 644 ${srcdir}/CYGWIN-PATCHES/README \
      ${instdir}${prefix}/share/doc/Cygwin/${BASEPKG}.README
  fi
  ln -fs ../Cygwin/${BASEPKG}.README \
         ${instdir}${prefix}/share/doc/${BASEPKG}/README.Cygwin
  if [ -f ${srcdir}/CYGWIN-PATCHES/${PKG}.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/postinstall
    /usr/bin/install -p -m 755 ${srcdir}/CYGWIN-PATCHES/${PKG}.sh \
      ${instdir}${sysconfdir}/postinstall/${PKG}.sh
  elif [ -f ${srcdir}/CYGWIN-PATCHES/postinstall.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/postinstall
    /usr/bin/install -p -m 755 ${srcdir}/CYGWIN-PATCHES/postinstall.sh \
      ${instdir}${sysconfdir}/postinstall/${PKG}.sh
  fi
  if [ -f ${srcdir}/CYGWIN-PATCHES/preremove.sh ] ; then
    mkdir -p ${instdir}${sysconfdir}/preremove
    /usr/bin/install -p -m 755 ${srcdir}/CYGWIN-PATCHES/preremove.sh \
      ${instdir}${sysconfdir}/preremove/${PKG}.sh
  fi )
}
strip() {
  :
}
list() {
  (cd ${instdir}
  find \! -type d | sed 's%^\.%  %' | sort )
}
depend() {
  :
}
pkg() {
  (cd ${instdir}
  tar cjf ${bin_pkg} * )
}
prepatch() {
  (
  # create a fresh source directory
  finish
  mv ${topdir}/${src_patch_name} ${topdir}/${src_patch_name}.hold
  prep
  mv ${topdir}/${src_patch_name}.hold ${topdir}/${src_patch_name}
  # apply patches and extras
  cd ${srcdir}
  if [ "$(ls ${topdir}/extras/* 2>/dev/null)" ] ; then
    cp -a ${topdir}/extras/* .
  fi
  if [ "$(ls ${topdir}/patches/*.patch 2>/dev/null)" ] ; then
    for ff in ${topdir}/patches/*.patch ; do
	  patch -p2 < $ff
	done
  fi
  # remake build directories
  mkdirs
  # create Cygwin patch
  mkpatch
  )
}
mkpatch() {
  (cd ${srcdir}
  find -name "autom4te.cache" | xargs -r rm -rf
  unpack ${src_orig_pkg}
  mv ${BASEPKG} ${topdir}/${BASEPKG}-orig
  cd ${topdir}
  diff -urN -x '.build' -x '.inst' -x '.sinst' \
    ${BASEPKG}-orig ${BASEPKG} > ${srcinstdir}/${src_patch_name} || true
  rm -rf ${BASEPKG}-orig
  cmp -s ${topdir}/${src_patch_name} ${srcinstdir}/${src_patch_name} \
  || cp --backup=numbered ${srcinstdir}/${src_patch_name} ${topdir} )
}
spkg() {
  (mkpatch
  if [ "${SIG}" -eq 1 ] ; then
    name=${srcinstdir}/${src_patch_name} text="PATCH" sigfile
  fi
  cp ${src_orig_pkg} ${srcinstdir}/${src_orig_pkg_name}
  if [ -e ${src_orig_pkg}.sig ] ; then
    cp ${src_orig_pkg}.sig ${srcinstdir}
  fi
  cp $0 ${srcinstdir}/`basename $0`
  name=$0 text="SCRIPT" sigfile
  if [ "${SIG}" -eq 1 ] ; then
    cp $0.sig ${srcinstdir}
  fi
  cd ${srcinstdir}
  tar cjf ${src_pkg} * )
}
finish() {
  rm -rf ${topdir}/${BASEPKG}
}
sigfile() {
  if [ \( "${SIG}" -eq 1 \) -a \( -e $name \) -a \( \( ! -e $name.sig \) -o \( $name -nt $name.sig \) \) ]; then
    if [ -x /usr/bin/gpg ]; then
      echo "$text signature need to be updated"
      rm -f $name.sig
      /usr/bin/gpg --detach-sign $name
    else
      echo "You need the gnupg package installed in order to make signatures."
    fi
  fi
}
checksig() {
  if [ -x /usr/bin/gpg ]; then
    if [ -e ${src_orig_pkg}.sig ]; then
      echo "ORIGINAL PACKAGE signature follows:"
      /usr/bin/gpg --verify ${src_orig_pkg}.sig ${src_orig_pkg}
    else
      echo "ORIGINAL PACKAGE signature missing."
    fi
    if [ -e $0.sig ]; then
      echo "SCRIPT signature follows:"
      /usr/bin/gpg --verify $0.sig $0
    else
      echo "SCRIPT signature missing."
    fi
    if [ -e ${src_patch}.sig ]; then
      echo "PATCH signature follows:"
      /usr/bin/gpg --verify ${src_patch}.sig ${src_patch}
    else
      echo "PATCH signature missing."
    fi
  else
    echo "You need the gnupg package installed in order to check signatures."
  fi
}
usage() {
  echo "Build script for the Cygwin ${PKG} package.
Adapted from the Cygwin generic-build-script;
see http://cygwin.com/setup.html#srcpackage_contents 
for detailed usage instructions.
Usage: ${scriptname} operation...
build operations:
  all          build source and binary packages (same as:
                 checksig prep conf build check install strip pkg spkg finish)
  build        run 'make' on patched source
  conf         run ./configure
  configure    run ./configure
  check        run test suite
  make         run 'make' on patched source
  prep         unpack and patch package source; run mkdirs
  strip        strip built executables
  test         run test suite
packaging operations:
  first        build source package first time, from existing patched 
                 source directory (same as: mkdirs spkg finish)
  install      install built files into installation staging subdir 
  mkpatch      create Cygwin source patch
  package      build binary package
  pkg          build binary package
  prepatch     patch original source from extras/ and patches/ dirs
  spkg         build source package
  src-package  build source package
information:
  depend       list packages on which the binary package depends
  help         display this message
  list         list files in the binary package
  usage        display this message
minor operations:
  checksig     check package, patch, and script signatures
  clean        run 'make clean'
  finish       remove source directory
  mkdirs       create package staging directories
  reconf       remove build staging dir and rerun ./configure
  rmdirs       remove package staging directories      
"
}

test -n "$1" || set "usage"
while test -n "$1" ; do
  case $1 in
    prep)			prep ;;
    mkdirs)			mkdirs ;;
	rmdirs)			rmdirs ;;
    conf|configure)	conf ;;
    reconf)			reconf ;;
    build|make)		build ;;
    check|test)		check ;;
    clean)			clean ;;
    install)		install ;;
    list)			list ;;
    depend)			depend ;;
    strip)			strip ;;
    package|pkg)	pkg ;;
	prepatch)		prepatch ;;
    mkpatch)		mkpatch ;;
    src-package|spkg)	spkg ;;
    finish)			finish ;;
    checksig)		checksig ;;
    first)			mkdirs ; spkg ; finish ;;
    all)			checksig ; prep ; conf ; build ; install ;
					strip ; pkg ; spkg ; finish ;;
    help|-help|--help|usage)	usage ;;
    *) echo "${scriptname}: error: unknown argument \"$1\"" ; 
	   echo "${scriptname}: run \"${scriptname} help\" to get a usage summary" ;
	   exit 1 ;;
  esac
  shift
done
