/**
 * @file utils.c
 *
 * @copyright Copyright  (C)  2013 Moritz Hanke <hanke@dkrz.de>
 *                                 Rene Redler <rene.redler@mpimet.mpg.de>
 *
 * @version 1.0
 * @author Moritz Hanke <hanke@dkrz.de>
 *         Rene Redler <rene.redler@mpimet.mpg.de>
 */
/*
 * Keywords:
 * Maintainer: Moritz Hanke <hanke@dkrz.de>
 *             Rene Redler <rene.redler@mpimet.mpg.de>
 * URL: https://dkrz-sw.gitlab-pages.dkrz.de/yac/
 *
 * This file is part of YAC.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are  permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of the DKRZ GmbH nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "utils.h"

static void ** pointer_lookup_table = NULL;
static unsigned pointer_lookup_table_size = 0;

unsigned yac_pointer_to_unique_id(void * pointer) {

   pointer_lookup_table = (void **)xrealloc (pointer_lookup_table,
      ++pointer_lookup_table_size * sizeof(pointer_lookup_table[0]));

   pointer_lookup_table[pointer_lookup_table_size-1] = pointer;

   return pointer_lookup_table_size - 1;
}

void * yac_unique_id_to_pointer(unsigned id) {

   if (id < pointer_lookup_table_size)
      return pointer_lookup_table[id];
   else {
      yac_internal_abort_message(
        "ERROR(yac_unique_id_to_pointer): invalid id", __FILE__, __LINE__);
      return NULL;
   }
}

void yac_free_pointer_unique_lookup() {

  free(pointer_lookup_table);
  pointer_lookup_table = NULL;
  pointer_lookup_table_size = 0;
}

void yac_internal_abort_message(
  const char * text, const char * file, int line) {

  fprintf(stderr, "%s \n", text); 
  fprintf(stderr, "Aborting in file %s, line %i ...\n", file, line );
  exit(EXIT_FAILURE);
}

void yac_abort_message ( char * text, char * file, int line ) {
  yac_internal_abort_message ( text, file, line );
}

/* ------------------------------------

   Source: http://www.cse.yorku.ca/~oz/hash.html 

unsigned long hash(const char *str) {

  unsigned long hash = 5381;
  int c;

  while ((c = *str++))
    hash = ((hash << 5) + hash) + c; 

  return hash;
}

  ------------------------------------*/

/*  Source http://snipplr.com/view/9022/string-hash-table/ */

#define NHASH 29989 //Use a prime number!
#define MULT 31

unsigned int yac_hash(const char *str) {
  unsigned int h = 0;
  for(; *str; str++)
    h = MULT * h + *str;
  return h % NHASH;
}

#define BUF_SIZE 4096

int yac_copy_file(const char *src, const char *dst) {
  char buf[BUF_SIZE];

  FILE* src_stream = fopen(src, "r");
  if (NULL == src_stream)
    return 1;

  FILE* dst_stream = fopen(dst, "w");
  if (NULL == dst_stream)
    return 1;

  while (!feof(src_stream)) {
    size_t bytes = fread(buf, 1, sizeof(buf), src_stream);
    if (bytes) {
      fwrite(buf, 1, bytes, dst_stream);
    }
  }

  fclose(src_stream);
  fclose(dst_stream);

  return 0;
}

