/***
  This file is part of avahi.

  avahi is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  avahi is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General
  Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with avahi; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>

#include <avahi-common/domain.h>
#include <avahi-common/malloc.h>
#ifdef HAVE_BONJOUR
#include <avahi-common/timeval.h>
#endif
#include <avahi-common/error.h>

#include "browse.h"
#include "log.h"

struct AvahiSServiceBrowser {
    AvahiServer *server;
    char *domain_name;
    char *service_type;

    AvahiSRecordBrowser *record_browser;

    AvahiSServiceBrowserCallback callback;
    void* userdata;

#ifdef HAVE_BONJOUR
    AvahiProtocol protocol;
    AvahiIfIndex interface;
    AvahiTimeEvent *browse_error_event;
    AvahiTimeEvent *all_for_now_event;
    AvahiLookupFlags flags;
    AvahiWatch *watch;
    DNSServiceRef client;
#endif

    AVAHI_LLIST_FIELDS(AvahiSServiceBrowser, browser);
};

#ifdef HAVE_BONJOUR
static void browse_reply(DNSServiceRef client, const DNSServiceFlags flags, uint32_t IfIndex, DNSServiceErrorType errorCode,
        const char *name, const char *type, const char *domain, void *context) {
    AvahiSServiceBrowser *b = context;
    char *op = (flags &kDNSServiceFlagsAdd) ? "Add" : "Rmv";
    AvahiBrowserEvent event;
    AvahiIfIndex interface;
    
    interface = (IfIndex == kDNSServiceInterfaceIndexAny) ? AVAHI_IF_UNSPEC : IfIndex;

    event = (flags &kDNSServiceFlagsAdd) ? AVAHI_BROWSER_NEW : AVAHI_BROWSER_REMOVE;

    b->callback(b, interface, AVAHI_PROTO_UNSPEC, event, name, type, domain, 0, b->userdata);
    
}

static void browse_socket_event(AvahiWatch *w, int fd, AvahiWatchEvent events, void *userdata) {
    AvahiSServiceBrowser *b = userdata;
    DNSServiceErrorType ret;

    assert(w);
    assert(fd >= 0);
    assert(events & AVAHI_WATCH_IN);

    assert (fd == DNSServiceRefSockFD(b->client));
    ret = DNSServiceProcessResult(b->client);
    if (ret != kDNSServiceErr_NoError) {
        if (b->watch) {
            b->server->poll_api->watch_free(b->watch);
            b->watch = NULL;
        }
        DNSServiceRefDeallocate(b->client);
        b->client = NULL;
        avahi_server_set_errno(b->server, AVAHI_ERR_DISCONNECTED);
        b->callback(b, 
            b->interface, 
            b->protocol, 
            AVAHI_BROWSER_FAILURE,
            NULL,
            b->service_type, 
            b->domain_name,
            0,
            b->userdata);
    }
}

static void all_for_now_callback(AvahiTimeEvent *e, void* userdata) {
    AvahiSServiceBrowser *b = userdata;

    assert(e);
    assert(b);

    avahi_time_event_free(b->all_for_now_event);
    b->all_for_now_event = NULL;

    b->callback(b, 
        b->interface, 
        b->protocol,
        AVAHI_BROWSER_ALL_FOR_NOW, 
        NULL,
        b->service_type,
        NULL,
        0,
        b->userdata);
}

static void browse_error_callback(AvahiTimeEvent *e, void *userdata) {
    AvahiSServiceBrowser *b = userdata;

    if (b->browse_error_event) {
        avahi_time_event_free(b->browse_error_event);
        b->browse_error_event = NULL;
    }
    avahi_server_set_errno(b->server, AVAHI_ERR_FAILURE);
    b->callback(b, 
        b->interface, 
        b->protocol, 
        AVAHI_BROWSER_FAILURE,
        NULL,
        b->service_type, 
        b->domain_name,
        0,
        b->userdata);
}

static void avahi_browse_service_start(AvahiSServiceBrowser *b) {
    DNSServiceErrorType ret;
    struct timeval tv;

    ret = DNSServiceBrowse(&b->client,
              0,
              b->interface == AVAHI_IF_UNSPEC ?
                  kDNSServiceInterfaceIndexAny :
                  b->interface,
              b->service_type,
              b->domain_name,
              browse_reply,
              b);
    if (ret != kDNSServiceErr_NoError || !b->client) {
        b->browse_error_event = avahi_time_event_new(b->server->time_event_queue,
NULL, browse_error_callback, b);
    } else {
        b->watch = b->server->poll_api->watch_new(b->server->poll_api, DNSServiceRefSockFD(b->client), AVAHI_WATCH_IN, browse_socket_event, b);

        /* Add a second */
        gettimeofday(&tv, NULL);
        avahi_timeval_add(&tv, 1000000);
        b->all_for_now_event = avahi_time_event_new(b->server->time_event_queue, &tv, all_for_now_callback, b);
    }
}
#endif

static void record_browser_callback(
    AvahiSRecordBrowser*rr,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    AvahiBrowserEvent event,
    AvahiRecord *record,
    AvahiLookupResultFlags flags,
    void* userdata) {

    AvahiSServiceBrowser *b = userdata;

    assert(rr);
    assert(b);

    /* Filter flags */
    flags &= AVAHI_LOOKUP_RESULT_CACHED | AVAHI_LOOKUP_RESULT_MULTICAST | AVAHI_LOOKUP_RESULT_WIDE_AREA;

    if (record) {
        char service[AVAHI_LABEL_MAX], type[AVAHI_DOMAIN_NAME_MAX], domain[AVAHI_DOMAIN_NAME_MAX];

        assert(record->key->type == AVAHI_DNS_TYPE_PTR);

        if (event == AVAHI_BROWSER_NEW && avahi_server_is_service_local(b->server, interface, protocol, record->data.ptr.name))
            flags |= AVAHI_LOOKUP_RESULT_LOCAL;

        if (avahi_service_name_split(record->data.ptr.name, service, sizeof(service), type, sizeof(type), domain, sizeof(domain)) < 0) {
            avahi_log_warn("Failed to split '%s'", record->key->name);
            return;
        }

        b->callback(b, interface, protocol, event, service, type, domain, flags, b->userdata);

    } else
        b->callback(b, interface, protocol, event, NULL, b->service_type, b->domain_name, flags, b->userdata);

}

AvahiSServiceBrowser *avahi_s_service_browser_new(
    AvahiServer *server,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    const char *service_type,
    const char *domain,
    AvahiLookupFlags flags,
    AvahiSServiceBrowserCallback callback,
    void* userdata) {

    AvahiSServiceBrowser *b;
    AvahiKey *k = NULL;
    char n[AVAHI_DOMAIN_NAME_MAX];
    int r;

    assert(server);
    assert(callback);
    assert(service_type);

    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_IF_VALID(interface), AVAHI_ERR_INVALID_INTERFACE);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_PROTO_VALID(protocol), AVAHI_ERR_INVALID_PROTOCOL);
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, !domain || avahi_is_valid_domain_name(domain), AVAHI_ERR_INVALID_DOMAIN_NAME);
#ifdef HAVE_BONJOUR
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_FLAGS_VALID(flags, AVAHI_LOOKUP_USE_MULTICAST), AVAHI_ERR_INVALID_FLAGS);
#else
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, AVAHI_FLAGS_VALID(flags, AVAHI_LOOKUP_USE_WIDE_AREA|AVAHI_LOOKUP_USE_MULTICAST), AVAHI_ERR_INVALID_FLAGS);
#endif
    AVAHI_CHECK_VALIDITY_RETURN_NULL(server, avahi_is_valid_service_type_generic(service_type), AVAHI_ERR_INVALID_SERVICE_TYPE);

    if (!domain)
        domain = server->domain_name;

    if ((r = avahi_service_name_join(n, sizeof(n), NULL, service_type, domain)) < 0) {
        avahi_server_set_errno(server, r);
        return NULL;
    }

    if (!(b = avahi_new(AvahiSServiceBrowser, 1))) {
        avahi_server_set_errno(server, AVAHI_ERR_NO_MEMORY);
        return NULL;
    }

    b->server = server;
    b->domain_name = b->service_type = NULL;
    b->callback = callback;
    b->userdata = userdata;
    b->record_browser = NULL;

    AVAHI_LLIST_PREPEND(AvahiSServiceBrowser, browser, server->service_browsers, b);

    if (!(b->domain_name = avahi_normalize_name_strdup(domain)) ||
        !(b->service_type = avahi_normalize_name_strdup(service_type))) {
        avahi_server_set_errno(server, AVAHI_ERR_NO_MEMORY);
        goto fail;
    }

#ifdef HAVE_BONJOUR
    b->protocol = protocol;
    b->interface = interface;
    b->flags = flags;
    b->browse_error_event = NULL;
    b->all_for_now_event = NULL;
    b->client = NULL;
    b->watch = NULL;
    avahi_browse_service_start(b);
#else
    if (!(k = avahi_key_new(n, AVAHI_DNS_CLASS_IN, AVAHI_DNS_TYPE_PTR))) {
        avahi_server_set_errno(server, AVAHI_ERR_NO_MEMORY);
        goto fail;
    }

    if (!(b->record_browser = avahi_s_record_browser_new(server, interface, protocol, k, flags, record_browser_callback, b)))
        goto fail;

    avahi_key_unref(k);
#endif

    return b;

fail:

    if (k)
        avahi_key_unref(k);

    avahi_s_service_browser_free(b);
    return NULL;
}

void avahi_s_service_browser_free(AvahiSServiceBrowser *b) {
    assert(b);

    AVAHI_LLIST_REMOVE(AvahiSServiceBrowser, browser, b->server->service_browsers, b);

    if (b->record_browser)
        avahi_s_record_browser_free(b->record_browser);

    avahi_free(b->domain_name);
    avahi_free(b->service_type);
#ifdef HAVE_BONJOUR
    if (b->browse_error_event) {
        avahi_time_event_free(b->browse_error_event);
        b->browse_error_event = NULL;
    }
    if (b->all_for_now_event) {
        avahi_time_event_free(b->all_for_now_event);
        b->all_for_now_event = NULL;
    }

    if (b->watch)
        b->server->poll_api->watch_free(b->watch);

    if (b->client)
        DNSServiceRefDeallocate (b->client);
#endif
    avahi_free(b);
}
