/* -*- Mode: C++; c-default-style: "k&r"; indent-tabs-mode: nil; tab-width: 2; c-basic-offset: 2 -*- */

/* libstaroffice
* Version: MPL 2.0 / LGPLv2+
*
* The contents of this file are subject to the Mozilla Public License Version
* 2.0 (the "License"); you may not use this file except in compliance with
* the License or as specified alternatively below. You may obtain a copy of
* the License at http://www.mozilla.org/MPL/
*
* Software distributed under the License is distributed on an "AS IS" basis,
* WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
* for the specific language governing rights and limitations under the
* License.
*
* Major Contributor(s):
* Copyright (C) 2002 William Lachance (wrlach@gmail.com)
* Copyright (C) 2002,2004 Marc Maurer (uwog@uwog.net)
* Copyright (C) 2004-2006 Fridrich Strba (fridrich.strba@bluewin.ch)
* Copyright (C) 2006, 2007 Andrew Ziem
* Copyright (C) 2011, 2012 Alonso Laurent (alonso@loria.fr)
*
*
* All Rights Reserved.
*
* For minor contributions see the git repository.
*
* Alternatively, the contents of this file may be used under the terms of
* the GNU Lesser General Public License Version 2 or later (the "LGPLv2+"),
* in which case the provisions of the LGPLv2+ are applicable
* instead of those above.
*/

#include <cstring>
#include <iomanip>
#include <iostream>
#include <limits>
#include <sstream>

#include <librevenge/librevenge.h>

#include "StarEncodingOtherKorean.hxx"

////////////////////////////////////////////////////////////
// constructor/destructor, ...
////////////////////////////////////////////////////////////
StarEncodingOtherKorean::StarEncodingOtherKorean()
{
}

StarEncodingOtherKorean::~StarEncodingOtherKorean()
{
}

bool StarEncodingOtherKorean::readKoreanBig5
(std::vector<uint8_t> const &src, size_t &pos, StarEncoding::Encoding encoding, std::vector<uint32_t> &dest)
{
  if (encoding!=StarEncoding::E_BIG5_HKSCS) {
    STOFF_DEBUG_MSG(("StarEncodingOtherKorean::readKoreanBig5: unknown encoding\n"));
    return false;
  }
  if (pos>=src.size()) return false;
  int c=(int) src[pos++], c2=0;
  if (c>=0x80&&c<=0xfe) {
    if (pos>=src.size()) return false;
    c2=(int) src[pos++];
  }
  uint32_t unicode=uint32_t(c);
  switch (c) {
  case 0x88:
    if (c2>=0x40 && c2<=0xaa) {
      static int const(val[])= {
        0xF303,0xF304,0xF305,0xF306,0xF307,0xF308,
        0xF309,0xF30A,0xF30B,0xF30C,0xF30D,0xF30E,0xF30F,0xF310,0xF311,0xF312,
        0xF313,0xF314,0xF315,0xF316,0xF317,0xF318,0x0100,0x00C1,0x01CD,0x00C0,
        0x0112,0x00C9,0x011A,0x00C8,0x014C,0x00D3,0x01D1,0x00D2,0xF325,0x1EBE,
        0xF327,0x1EC0,0x00CA,0x0101,0x00E1,0x01CE,0x00E0,0x0251,0x0113,0x00E9,
        0x011B,0x00E8,0x012B,0x00ED,0x01D0,0x00EC,0x014D,0x00F3,0x01D2,0x00F2,
        0x016B,0x00FA,0x01D4,0x00F9,0x01D6,0x01D8,0x01DA,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x01DC,0x00FC,0xF344,0x1EBF,0xF346,0x1EC1,0x00EA,0x0261,0xF34A,
        0xF34B
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x89:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xF3A0,0xF3A1,0xffff,0x650A,0xffff,0xffff,
        0x4E3D,0x6EDD,0x9D4E,0x91DF,0xffff,0xffff,0xF3AC,0x6491,0x4F1A,0x4F28,
        0x4FA8,0x5156,0x5174,0x519C,0x51E4,0x52A1,0x52A8,0x533B,0x534E,0x53D1,
        0x53D8,0x56E2,0x58F0,0x5904,0x5907,0x5932,0x5934,0x5B66,0x5B9E,0x5B9F,
        0x5C9A,0x5E86,0x603B,0x6589,0x67FE,0x6804,0x6865,0x6D4E,0x70BC,0x7535,
        0x7EA4,0x7EAC,0x7EBA,0x7EC7,0x7ECF,0x7EDF,0x7F06,0x7F37,0x827A,0x82CF,
        0x836F,0x89C6,0x8BBE,0x8BE2,0x8F66,0x8F67,0x8F6E,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x7411,0x7CFC,0x7DCD,0x6946,0x7AC9,0x5227,0xffff,0xffff,0xffff,
        0xffff,0x918C,0x78B8,0x915E,0x80BC,0xffff,0x8D0B,0x80F6,0xF3F0,0xffff,
        0xffff,0x809F,0x9EC7,0x4CCD,0x9DC9,0x9E0C,0x4C3E,0xF3F9,0xF3FA,0x9E0A,
        0xF3FC,0x35C1,0xffff,0x6E9A,0x823E,0x7519,0xffff,0x4911,0x9A6C,0x9A8F,
        0x9F99,0x7987,0xF408,0xF409,0xF40A,0xF40B,0x4E24,0x4E81,0x4E80,0x4E87,
        0x4EBF,0x4EEB,0x4F37,0x344C,0x4FBD,0x3E48,0x5003,0x5088,0x347D,0x3493,
        0x34A5,0x5186,0x5905,0x51DB,0x51FC,0x5205,0x4E89,0x5279,0x5290,0x5327,
        0x35C7,0x53A9,0x3551,0x53B0,0x3553,0x53C2,0x5423,0x356D,0x3572,0x3681,
        0x5493,0x54A3,0x54B4,0x54B9,0x54D0,0x54EF,0x5518,0x5523,0x5528,0x3598,
        0x553F,0x35A5,0x35BF,0x55D7,0x35C5,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8a:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xF43D,0x5525,0xffff,0xF440,0xF441,0xF442,
        0x5590,0xF444,0x39EC,0xF446,0x8E46,0xF448,0xF449,0x4053,0xF44B,0x777A,
        0xF44D,0x3A34,0x47D5,0xF450,0xF451,0xF452,0x64DD,0xF454,0xF455,0xF456,
        0xF457,0x648D,0x8E7E,0xF45A,0xF45B,0xF45C,0xF45D,0xF45E,0xF45F,0xffff,
        0xF461,0xF462,0xF463,0x47F4,0xF465,0xF466,0x9AB2,0x3A67,0xF469,0x3FED,
        0x3506,0xF46C,0xF46D,0xF46E,0xF46F,0x9D6E,0x9815,0xffff,0x43D9,0xF474,
        0x64B4,0x54E3,0xF477,0xF478,0xF479,0x39FB,0xF47B,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xF47C,0xF47D,0xF47E,0x64EA,0xF480,0xF481,0x8E68,0xF483,0xF484,
        0xF485,0xffff,0x480B,0xF488,0x3FFA,0x5873,0xF48B,0xffff,0xF48D,0xF48E,
        0xF48F,0xF490,0xF491,0x5579,0x40BB,0x43BA,0xffff,0x4AB4,0xF497,0xF498,
        0x81AA,0x98F5,0xF49B,0x6379,0x39FE,0xF49E,0x8DC0,0x56A1,0x647C,0x3E43,
        0xffff,0xF4A4,0xF4A5,0xF4A6,0xF4A7,0xffff,0xF4A9,0xF4AA,0x3992,0x3A06,
        0xF4AD,0x3578,0xF4AF,0xF4B0,0x5652,0xF4B2,0xF4B3,0xF4B4,0x34BC,0x6C3D,
        0xF4B7,0xffff,0xffff,0xF4BA,0xF4BB,0xF4BC,0xF4BD,0xF4BE,0xF4BF,0xF4C0,
        0xF4C1,0x7F93,0xF4C3,0xF4C4,0xF4C5,0x35FB,0xF4C7,0xF4C8,0xF4C9,0xF4CA,
        0x3F93,0xF4CC,0xF4CD,0xF4CE,0xF4CF,0xffff,0xF4D1,0xF4D2,0xF4D3,0xF4D4,
        0xF4D5,0x3FF9,0xF4D7,0x6432,0xF4D9,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8b:
    if (c2>=0x40 && c2<=0xfd) {
      static int const(val[])= {
        0xF4DA,0xF4DB,0xF4DC,0xF4DD,0xF4DE,0xF4DF,
        0xF4E0,0x3A18,0xF4E2,0xF4E3,0xF4E4,0xF4E5,0xF4E6,0xF4E7,0xF4E8,0xF4E9,
        0x95AA,0x54CC,0x82C4,0x55B9,0xffff,0xF4EF,0x9C26,0x9AB6,0xF4F2,0xF4F3,
        0x7140,0x816D,0x80EC,0x5C1C,0xF4F8,0x8134,0x3797,0x535F,0xF4FC,0x91B6,
        0xF4FE,0xF4FF,0xF500,0xF501,0x35DD,0xF503,0x3609,0xF505,0x56AF,0xF507,
        0xF508,0xF509,0xF50A,0xF50B,0xF50C,0xF50D,0xF50E,0xF50F,0xF510,0xF511,
        0x5A54,0xF513,0xF514,0xF515,0xF516,0x579C,0xF518,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xF519,0xF51A,0xF51B,0xF51C,0xF51D,0x3703,0xF51F,0xF520,0xF521,
        0xF522,0xF523,0xF524,0xF525,0xF526,0x5899,0x5268,0x361A,0xF52A,0x7BB2,
        0x5B68,0x4800,0x4B2C,0x9F27,0x49E7,0x9C1F,0x9B8D,0xF533,0xF534,0x55FB,
        0x35F2,0x5689,0x4E28,0x5902,0xF53A,0xF53B,0x9751,0xF53D,0x4E5B,0x4EBB,
        0x353E,0x5C23,0x5F51,0x5FC4,0x38FA,0x624C,0x6535,0x6B7A,0x6C35,0x6C3A,
        0x706C,0x722B,0x4E2C,0x72AD,0xF54E,0x7F52,0x793B,0x7CF9,0x7F53,0xF553,
        0x34C1,0xffff,0xF556,0x8002,0x8080,0xF559,0xF55A,0x535D,0x8864,0x89C1,
        0xF55E,0x8BA0,0x8D1D,0x9485,0x9578,0x957F,0x95E8,0xF565,0x97E6,0x9875,
        0x98CE,0x98DE,0x9963,0xF56B,0x9C7C,0x9E1F,0x9EC4,0x6B6F,0xF907,0x4E37,
        0xF572,0x961D,0x6237,0x94A2,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8c:
    if (c2>=0x40 && c2<=0xdc) {
      static int const(val[])= {
        0x503B,0x6DFE,0xF579,0xF57A,0x3DC9,0x888F,
        0xF57D,0x7077,0x5CF5,0x4B20,0xF581,0x3559,0xF583,0x6122,0xF585,0x8FA7,
        0x91F6,0x7191,0x6719,0x73BA,0xF58B,0xF58C,0x3C8B,0xF58E,0x4B10,0x78E4,
        0x7402,0x51AE,0xF593,0x4009,0x6A63,0xF596,0x4223,0x860F,0xffff,0x7A2A,
        0xF59B,0xF59C,0x9755,0x704D,0x5324,0xF5A0,0x93F4,0x76D9,0xF5A3,0xF5A4,
        0x77DD,0x4EA3,0x4FF0,0x50BC,0x4E2F,0x4F17,0xF5AB,0x5434,0x7D8B,0x5892,
        0x58D0,0xF5B0,0x5E92,0x5E99,0x5FC2,0xF5B4,0x658B,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xF5B6,0x6919,0x6A43,0xF5B9,0x6CFF,0xffff,0x7200,0xF5BD,0x738C,
        0x3EDB,0xF5C0,0x5B15,0x74B9,0x8B83,0xF5C4,0xF5C5,0x7A93,0x7BEC,0x7CC3,
        0x7E6C,0x82F8,0x8597,0xF5CC,0x8890,0xF5CE,0x8EB9,0xF5D0,0x8FCF,0x855F,
        0x99E0,0x9221,0xF5D5,0xF5D6,0xF5D7,0x4071,0x42A2,0x5A1A,0xffff,0xffff,
        0xffff,0x9868,0x676B,0x4276,0x573D,0xffff,0x85D6,0xF5E4,0x82BF,0xF5E6,
        0x4C81,0xF5E8,0x5D7B,0xF5EA,0xF5EB,0xF5EC,0xF5ED,0x5B96,0xF5EF,0xffff,
        0x7E5B,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8d:
    if (c2>=0x60 && c2<=0xfe) {
      static int const(val[])= {
        0x5D3E,0x5D48,0x5D56,0x3DFC,
        0x380F,0x5DA4,0x5DB9,0x3820,0x3838,0x5E42,0x5EBD,0x5F25,0x5F83,0x3908,
        0x3914,0x393F,0x394D,0x60D7,0x613D,0x5CE5,0x3989,0x61B7,0x61B9,0x61CF,
        0x39B8,0x622C,0x6290,0x62E5,0x6318,0x39F8,0x56B1,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x3A03,0x63E2,0x63FB,0x6407,0x645A,0x3A4B,0x64C0,0x5D15,0x5621,
        0x9F9F,0x3A97,0x6586,0x3ABD,0x65FF,0x6653,0x3AF2,0x6692,0x3B22,0x6716,
        0x3B42,0x67A4,0x6800,0x3B58,0x684A,0x6884,0x3B72,0x3B71,0x3B7B,0x6909,
        0x6943,0x725C,0x6964,0x699F,0x6985,0x3BBC,0x69D6,0x3BDD,0x6A65,0x6A74,
        0x6A71,0x6A82,0x3BEC,0x6A99,0x3BF2,0x6AAB,0x6AB5,0x6AD4,0x6AF6,0x6B81,
        0x6BC1,0x6BEA,0x6C75,0x6CAA,0x3CCB,0x6D02,0x6D06,0x6D26,0x6D81,0x3CEF,
        0x6DA4,0x6DB1,0x6E15,0x6E18,0x6E29,0x6E86,0xF694,0x6EBB,0x6EE2,0x6EDA,
        0x9F7F,0x6EE8,0x6EE9,0x6F24,0x6F34,0x3D46,0xF69E,0x6F81,0x6FBE,0x3D6A,
        0x3D75,0x71B7,0x5C99,0x3D8A,0x702C,0x3D91,0x7050,0x7054,0x706F,0x707F,
        0x7089,0xF6AD,0x43C1,0x35F1,0xF6B0,
      };
      unicode=(uint32_t) val[c2-0x60];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8e:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE311,0x57BE,0xE313,0x713E,0xE315,0x364E,
        0x69A2,0xE318,0x5B74,0x7A49,0xE31B,0xE31C,0x7A65,0x7A7D,0xE31F,0x7ABB,
        0x7AB0,0x7AC2,0x7AC3,0x71D1,0xE325,0x41CA,0x7ADA,0x7ADD,0x7AEA,0x41EF,
        0x54B2,0xE32C,0x7B0B,0x7B55,0x7B29,0xE330,0xE331,0x7BA2,0x7B6F,0x839C,
        0xE335,0xE336,0x7BD0,0x8421,0x7B92,0x7BB8,0xE33B,0x3DAD,0xE33D,0x8492,
        0x7BFA,0x7C06,0x7C35,0xE342,0x7C44,0x7C83,0xE345,0x7CA6,0x667D,0xE348,
        0x7CC9,0x7CC7,0x7CE6,0x7C74,0x7CF3,0x7CF5,0x7CCE,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x7E67,0x451D,0xE352,0x7D5D,0xE354,0x748D,0x7D89,0x7DAB,0x7135,
        0x7DB3,0x7DD2,0xE35B,0xE35C,0x7DE4,0x3D13,0x7DF5,0xE360,0x7DE5,0xE362,
        0x7E1D,0xE364,0xE365,0x7E6E,0x7E92,0x432B,0x946C,0x7E27,0x7F40,0x7F41,
        0x7F47,0x7936,0xE36F,0x99E1,0x7F97,0xE372,0x7FA3,0xE374,0xE375,0x455C,
        0xE377,0x4503,0xE379,0x7FFA,0xE37B,0x8005,0x8008,0x801D,0x8028,0x802F,
        0xE381,0xE382,0x803B,0x803C,0x8061,0xE386,0x4989,0xE388,0xE389,0xE38A,
        0x6725,0x80A7,0xE38D,0x8107,0x811A,0x58B0,0xE391,0x6C7F,0xE393,0xE394,
        0x64E7,0xE396,0x8218,0xE398,0x6A53,0xE39A,0xE39B,0x447A,0x8229,0xE39E,
        0xE39F,0xE3A0,0x4FF9,0xE3A2,0x84E2,0x8362,0xE3A5,0xE3A6,0xE3A7,0xE3A8,
        0xE3A9,0x82AA,0x691B,0xE3AC,0x41DB,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8f:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x854B,0x82D0,0x831A,0xE3B1,0xE3B2,0x36C1,
        0xE3B4,0xE3B5,0x827B,0x82E2,0x8318,0xE3B9,0xE3BA,0xE3BB,0xE3BC,0xE3BD,
        0x3DBF,0x831D,0x55EC,0x8385,0x450B,0xE3C3,0x83AC,0x83C1,0x83D3,0x347E,
        0xE3C8,0x6A57,0x855A,0x3496,0xE3CC,0xE3CD,0x8458,0xE3CF,0x8471,0x3DD3,
        0x44E4,0x6AA7,0x844A,0xE3D5,0x7958,0x84A8,0xE3D8,0xE3D9,0xE3DA,0x84DE,
        0x840F,0x8391,0x44A0,0x8493,0x84E4,0xE3E1,0x4240,0xE3E3,0x4543,0x8534,
        0x5AF2,0xE3E7,0x4527,0x8573,0x4516,0x67BF,0x8616,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE3ED,0xE3EE,0x85C1,0xE3F0,0x8602,0xE3F2,0xE3F3,0xE3F4,0x456A,
        0x8628,0x3648,0xE3F8,0x53F7,0xE3FA,0x867E,0x8771,0xE3FD,0x87EE,0xE3FF,
        0x87B1,0x87DA,0x880F,0x5661,0x866C,0x6856,0x460F,0x8845,0x8846,0xE409,
        0xE40A,0xE40B,0x885E,0x889C,0x465B,0x88B4,0x88B5,0x63C1,0x88C5,0x7777,
        0xE414,0x8987,0x898A,0x89A6,0x89A9,0x89A7,0x89BC,0xE41B,0x89E7,0xE41D,
        0xE41E,0x8A9C,0x7793,0x91FE,0x8A90,0xE423,0x7AE9,0xE425,0xE426,0x4713,
        0xE428,0x717C,0x8B0C,0x8B1F,0xE42C,0xE42D,0x8B3F,0x8B4C,0x8B4D,0x8AA9,
        0xE432,0x8B90,0x8B9B,0x8AAF,0xE436,0x4615,0x884F,0x8C9B,0xE43A,0xE43B,
        0xE43C,0x3725,0xE43E,0x8CD6,0xE440,0xE441,0x8D12,0x8D03,0xE444,0x8CDB,
        0x705C,0x8D11,0xE448,0x3ED0,0x8D77,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x90:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x8DA9,0xE44C,0xE44D,0xE44E,0x3B7C,0xE450,
        0xE451,0x7AE7,0x8EAD,0x8EB6,0x8EC3,0x92D4,0x8F19,0x8F2D,0xE459,0xE45A,
        0x8FA5,0x9303,0xE45D,0xE45E,0x8FB3,0x492A,0xE461,0xE462,0xE463,0x5EF8,
        0xE465,0x8FF9,0xE467,0xE468,0xE469,0xE46A,0x3980,0xE46C,0x9037,0xE46E,
        0xE46F,0x9061,0xE471,0xE472,0x90A8,0xE474,0x90C4,0xE476,0x90AE,0x90FD,
        0x9167,0x3AF0,0x91A9,0x91C4,0x7CAC,0xE47E,0xE47F,0x920E,0x6C9F,0x9241,
        0x9262,0xE484,0x92B9,0xE486,0xE487,0xE488,0xE489,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE48A,0x932C,0x936B,0xE48D,0xE48E,0x708F,0x5AC3,0xE491,0xE492,
        0x4965,0x9244,0xE495,0xE496,0xE497,0x9373,0x945B,0x8EBC,0x9585,0x95A6,
        0x9426,0x95A0,0x6FF6,0x42B9,0xE4A1,0xE4A2,0xE4A3,0xE4A4,0x49DF,0x6C1C,
        0x967B,0x9696,0x416C,0x96A3,0xE4AB,0x61DA,0x96B6,0x78F5,0xE4AF,0x96BD,
        0x53CC,0x49A1,0xE4B3,0xE4B4,0xE4B5,0xE4B6,0xE4B7,0xE4B8,0xE4B9,0xE4BA,
        0x9731,0x8642,0x9736,0x4A0F,0x453D,0x4585,0xE4C1,0x7075,0x5B41,0x971B,
        0x975C,0xE4C6,0x9757,0x5B4A,0xE4C9,0x975F,0x9425,0x50D0,0xE4CD,0xE4CE,
        0x9789,0x979F,0x97B1,0x97BE,0x97C0,0x97D2,0x97E0,0xE4D6,0x97EE,0x741C,
        0xE4D9,0x97FF,0x97F5,0xE4DC,0xE4DD,0x4AD1,0x9834,0x9833,0x984B,0x9866,
        0x3B0E,0xE4E4,0x3D51,0xE4E6,0xE4E7,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x91:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE4E8,0x98CA,0x98B7,0x98C8,0x98C7,0x4AFF,
        0xE4EE,0xE4EF,0x55B0,0x98E1,0x98E6,0x98EC,0x9378,0x9939,0xE4F6,0x4B72,
        0xE4F8,0xE4F9,0x99F5,0x9A0C,0x9A3B,0x9A10,0x9A58,0xE4FF,0x36C4,0xE501,
        0xE502,0x9AE0,0x9AE2,0xE505,0x9AF4,0x4C0E,0x9B14,0x9B2D,0xE50A,0x5034,
        0x9B34,0xE50D,0x38C3,0xE50F,0x9B50,0x9B40,0xE512,0x5A45,0xE514,0x9B8E,
        0xE516,0x9C02,0x9BFF,0x9C0C,0xE51A,0x9DD4,0xE51C,0xE51D,0xE51E,0xE51F,
        0xE520,0xE521,0x9D7E,0x9D83,0xE524,0x9E0E,0x6888,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x9DC4,0xE528,0xE529,0xE52A,0xE52B,0xE52C,0x9D39,0xE52E,0xE52F,
        0x9E90,0x9E95,0x9E9E,0x9EA2,0x4D34,0x9EAA,0x9EAF,0xE537,0x9EC1,0x3B60,
        0x39E5,0x3D1D,0x4F32,0x37BE,0xE53E,0x9F02,0x9F08,0x4B96,0x9424,0xE543,
        0x9F17,0x9F16,0x9F39,0x569F,0x568A,0x9F45,0x99B8,0xE54B,0x97F2,0x847F,
        0x9F62,0x9F69,0x7ADC,0x9F8E,0x7216,0x4BBE,0xE554,0xE555,0x7177,0xE557,
        0xE558,0xE559,0x739E,0xE55B,0xE55C,0x799F,0xE55E,0xE55F,0x9369,0x93F3,
        0xE562,0x92EC,0x9381,0x93CB,0xE566,0xE567,0x7217,0x3EEB,0x7772,0x7A43,
        0x70D0,0xE56D,0xE56E,0x717E,0xE570,0x70A3,0xE572,0xE573,0x3EC7,0xE575,
        0xE576,0xE577,0x3722,0xE579,0xE57A,0x36E1,0xE57C,0xE57D,0xE57E,0x3723,
        0xE580,0x575B,0xE582,0xE583,0xE584,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x92:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE585,0xE586,0x8503,0xE588,0x8503,0x8455,
        0xE58B,0xE58C,0xE58D,0xE58E,0xE58F,0xE590,0x44F4,0xE592,0xE593,0xE594,
        0x67F9,0x3733,0x3C15,0x3DE7,0x586C,0xE59A,0x6810,0x4057,0xE59D,0xE59E,
        0xE59F,0xE5A0,0xE5A1,0x54CB,0x569E,0xE5A4,0x5692,0xE5A6,0xE5A7,0xE5A8,
        0x93C6,0xE5AA,0x939C,0x4EF8,0x512B,0x3819,0xE5AF,0x4EBC,0xE5B1,0xE5B2,
        0x4F4B,0x4F8A,0xE5B5,0x5A68,0xE5B7,0xE5B8,0x3999,0xE5BA,0xE5BB,0x3435,
        0x4F29,0xE5BE,0xE5BF,0xE5C0,0x8ADA,0xE5C2,0x4E98,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x50CD,0x510D,0x4FA2,0x4F03,0xE5C8,0xE5C9,0x4F42,0x502E,0x506C,
        0x5081,0x4FCC,0x4FE5,0x5058,0x50FC,0x5159,0x515B,0x515D,0x515E,0x6E76,
        0xE5D7,0xE5D8,0xE5D9,0x6D72,0xE5DB,0xE5DC,0x51A8,0x51C3,0xE5DF,0x44DD,
        0xE5E1,0xE5E2,0xE5E3,0x8D7A,0xE5E5,0xE5E6,0x5259,0x52A4,0xE5E9,0x52E1,
        0x936E,0x467A,0x718C,0xE5EE,0xE5EF,0xE5F0,0xE5F1,0x69D1,0xE5F3,0x7479,
        0x3EDE,0x7499,0x7414,0x7456,0x7398,0x4B8E,0xE5FB,0xE5FC,0x53D0,0x3584,
        0x720F,0xE600,0x55B4,0xE602,0x54CD,0xE604,0x571D,0x925D,0x96F4,0x9366,
        0x57DD,0x578D,0x577F,0x363E,0x58CB,0x5A99,0xE60F,0xE610,0xE611,0xE612,
        0x5A2C,0x59B8,0x928F,0x5A7E,0x5ACF,0x5A12,0xE619,0xE61A,0xE61B,0xE61C,
        0x36F5,0x6D05,0x7443,0x5A21,0xE621,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x93:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x5A81,0xE623,0xE624,0x93E0,0x748C,0xE627,
        0x7105,0x4972,0x9408,0xE62B,0x93BD,0x37A0,0x5C1E,0x5C9E,0x5E5E,0x5E48,
        0xE632,0xE633,0xE634,0x5ECD,0x5B4F,0xE637,0xE638,0x3701,0xE63A,0x36DD,
        0xE63C,0x36D3,0x812A,0xE63F,0xE640,0xE641,0xE642,0x5F0C,0x5F0E,0xE645,
        0xE646,0x5A6B,0xE648,0x5B44,0x8614,0xE64B,0x8860,0x607E,0xE64E,0xE64F,
        0x5FDB,0x3EB8,0xE652,0xE653,0xE654,0xE655,0x61C0,0xE657,0xE658,0xE659,
        0x6199,0x6198,0x6075,0xE65D,0xE65E,0xE65F,0xE660,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x6471,0xE662,0xE663,0x3A29,0xE665,0xE666,0xE667,0xE668,0x6337,
        0xE66A,0x64B6,0x6331,0x63D1,0xE66E,0xE66F,0x62A4,0xE671,0x643B,0x656B,
        0x6972,0x3BF4,0xE676,0xE677,0xE678,0xE679,0x550D,0xE67B,0xE67C,0xE67D,
        0x66CE,0xE67F,0xE680,0x3AE0,0x4190,0xE683,0xE684,0xE685,0xE686,0xE687,
        0xE688,0x78EE,0xE68A,0xE68B,0xE68C,0x3464,0xE68E,0xE68F,0xE690,0x668E,
        0xE692,0x666B,0x4B93,0x6630,0xE696,0xE697,0x6663,0xE699,0xE69A,0x661E,
        0xE69C,0x38D1,0xE69E,0xE69F,0x3B99,0xE6A1,0xE6A2,0x74D0,0x3B96,0x678F,
        0xE6A6,0x68B6,0x681E,0x3BC4,0x6ABE,0x3863,0xE6AC,0xE6AD,0x6A33,0x6A52,
        0x6AC9,0x6B05,0xE6B2,0x6511,0x6898,0x6A4C,0x3BD7,0x6A7A,0x6B57,0xE6B9,
        0xE6BA,0x93A0,0x92F2,0xE6BD,0xE6BE,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x94:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x9289,0xE6C0,0xE6C1,0x9467,0x6DA5,0x6F0B,
        0xE6C5,0x6D67,0xE6C7,0x3D8F,0x6E04,0xE6CA,0x5A3D,0x6E0A,0x5847,0x6D24,
        0x7842,0x713B,0xE6D1,0xE6D2,0x70F1,0x7250,0x7287,0x7294,0xE6D7,0xE6D8,
        0x5179,0xE6DA,0xE6DB,0x747A,0xE6DD,0xE6DE,0xE6DF,0xE6E0,0xE6E1,0x3F06,
        0x3EB1,0xE6E4,0xE6E5,0xE6E6,0x60A7,0x3EF3,0x74CC,0x743C,0x9387,0x7437,
        0x449F,0xE6EE,0x4551,0x7583,0x3F63,0xE6F2,0xE6F3,0x3F58,0x7555,0x7673,
        0xE6F7,0x3B19,0x7468,0xE6FA,0xE6FB,0xE6FC,0x3AFB,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x3DCD,0xE6FF,0x3EFF,0xE701,0xE702,0x91FA,0x5732,0x9342,0xE706,
        0xE707,0x50DF,0xE709,0xE70A,0x7778,0xE70C,0x770E,0x770F,0x777B,0xE710,
        0xE711,0x3A5E,0xE713,0x7438,0x749B,0x3EBF,0xE717,0xE718,0x40C8,0xE71A,
        0xE71B,0x9307,0xE71D,0x781E,0x788D,0x7888,0x78D2,0x73D0,0x7959,0xE724,
        0xE725,0x410E,0x799B,0x8496,0x79A5,0x6A2D,0xE72B,0x7A3A,0x79F4,0x416E,
        0xE72F,0x4132,0x9235,0x79F1,0xE733,0xE734,0xE735,0xE736,0xE737,0x3597,
        0x556B,0x3570,0x36AA,0xE73C,0xE73D,0x7AE2,0x5A59,0xE740,0xE741,0xE742,
        0x5A0D,0xE744,0x78F0,0x5A2A,0xE747,0x7AFE,0x41F9,0x7C5D,0x7C6D,0x4211,
        0xE74D,0xE74E,0xE74F,0x7CCD,0xE751,0xE752,0x7C8E,0x7C7C,0x7CAE,0x6AB2,
        0x7DDC,0x7E07,0x7DD3,0x7F4E,0xE75B,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x95:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE75C,0xE75D,0x7D97,0xE75F,0x426A,0xE761,
        0xE762,0x67D6,0xE764,0xE765,0x57C4,0xE767,0xE768,0xE769,0x7FDD,0x7B27,
        0xE76C,0xE76D,0xE76E,0x7B0C,0xE770,0x99E6,0x8645,0x9A63,0x6A1C,0xE775,
        0x39E2,0xE777,0xE778,0x9A1F,0xE77A,0x8480,0xE77C,0xE77D,0x44EA,0x8137,
        0x4402,0x80C6,0x8109,0x8142,0xE784,0x98C3,0xE786,0x8262,0x8265,0xE789,
        0x8453,0xE78B,0x8610,0xE78D,0x5A86,0x417F,0xE790,0x5B2B,0xE792,0x5AE4,
        0xE794,0x86A0,0xE796,0xE797,0x882D,0xE799,0x5A02,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x886E,0x4F45,0x8887,0x88BF,0x88E6,0x8965,0x894D,0xE7A2,0x8954,
        0xE7A4,0xE7A5,0xE7A6,0xE7A7,0xE7A8,0xE7A9,0x3EAD,0x84A3,0x46F5,0x46CF,
        0x37F2,0x8A3D,0x8A1C,0xE7B1,0x5F4D,0x922B,0xE7B4,0x65D4,0x7129,0x70C4,
        0xE7B8,0x9D6D,0x8C9F,0x8CE9,0xE7BC,0x599A,0x77C3,0x59F0,0x436E,0x36D4,
        0x8E2A,0x8EA7,0xE7C4,0x8F30,0x8F4A,0x42F4,0x6C58,0x6FBB,0xE7CA,0x489B,
        0x6F79,0x6E8B,0xE7CE,0x9BE9,0x36B5,0xE7D1,0x90BB,0x9097,0x5571,0x4906,
        0x91BB,0x9404,0xE7D8,0x4062,0xE7DA,0x9427,0xE7DC,0xE7DD,0x84E5,0x8A2B,
        0x9599,0x95A7,0x9597,0x9596,0xE7E4,0x7445,0x3EC2,0xE7E7,0xE7E8,0xE7E9,
        0x3EE7,0xE7EB,0x968F,0xE7ED,0xE7EE,0xE7EF,0x3ECC,0xE7F1,0xE7F2,0xE7F3,
        0x7412,0x746B,0x3EFC,0x9741,0xE7F8,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x96:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x6847,0x4A1D,0xE7FB,0xE7FC,0x975D,0x9368,
        0xE7FF,0xE800,0xE801,0xE802,0x92BA,0x5B11,0x8B69,0x493C,0x73F9,0xE808,
        0x979B,0x9771,0x9938,0xE80C,0x5DC1,0xE80E,0xE80F,0x981F,0xE811,0x92F6,
        0xE813,0x91E5,0x44C0,0xE816,0xE817,0xE818,0x98DC,0xE81A,0x3F00,0x922A,
        0x4925,0x8414,0x993B,0x994D,0xE821,0x3DFD,0x999B,0x4B6F,0x99AA,0x9A5C,
        0xE827,0xE828,0x6A8F,0x9A21,0x5AFE,0x9A2F,0xE82D,0x4B90,0xE82F,0x99BC,
        0x4BBD,0x4B97,0x937D,0x5872,0xE835,0x5822,0xE837,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE838,0x7844,0xE83A,0xE83B,0x68C5,0x3D7D,0x9458,0x3927,0x6150,
        0xE841,0xE842,0x6107,0x9C4F,0x9C53,0x9C7B,0x9C35,0x9C10,0x9B7F,0x9BCF,
        0xE84B,0x9B9F,0xE84D,0xE84E,0x9D21,0x4CAE,0xE851,0x9E18,0x4CB0,0x9D0C,
        0xE855,0xE856,0xE857,0xE858,0x9DA5,0x84BD,0xE85B,0xE85C,0xE85D,0x85FC,
        0x4533,0xE860,0xE861,0xE862,0x8420,0x85EE,0xE865,0xE866,0xE867,0x79E2,
        0xE869,0xE86A,0x492D,0xE86C,0x3D62,0x93DB,0x92BE,0x9348,0xE871,0x78B9,
        0x9277,0x944D,0x4FE4,0x3440,0x9064,0xE878,0x783D,0x7854,0x78B6,0x784B,
        0xE87D,0xE87E,0xE87F,0x369A,0x4F72,0x6FDA,0x6FD9,0x701E,0x701E,0x5414,
        0xE887,0x57BB,0x58F3,0x578A,0x9D16,0x57D7,0x7134,0x34AF,0xE88F,0x71EB,
        0xE891,0xE892,0x5B28,0xE894,0xE895,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x97:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x610C,0x5ACE,0x5A0B,0x42BC,0xE89A,0x372C,
        0x4B7B,0xE89D,0x93BB,0x93B8,0xE8A0,0xE8A1,0x8472,0xE8A3,0xE8A4,0xE8A5,
        0xE8A6,0xE8A7,0x5994,0xE8A9,0xE8AA,0x7DA8,0xE8AC,0xE8AD,0xE8AE,0xE8AF,
        0xE8B0,0x92E5,0x73E2,0x3EE9,0x74B4,0xE8B5,0xE8B6,0x3EE1,0xE8B8,0x6AD8,
        0x73F3,0x73FB,0x3ED6,0xE8BD,0xE8BE,0xE8BF,0xE8C0,0xE8C1,0xE8C2,0xE8C3,
        0x7448,0xE8C5,0x70A5,0xE8C7,0x9284,0x73E6,0x935F,0xE8CB,0x9331,0xE8CD,
        0xE8CE,0x9386,0xE8D0,0xE8D1,0x4935,0xE8D3,0x716B,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE8D5,0xE8D6,0x56A4,0xE8D8,0xE8D9,0xE8DA,0x5502,0x79C4,0xE8DD,
        0x7DFE,0xE8DF,0xE8E0,0xE8E1,0x452E,0x9401,0x370A,0xE8E5,0xE8E6,0x59B0,
        0xE8E8,0xE8E9,0xE8EA,0x5AA1,0x36E2,0xE8ED,0x36B0,0x925F,0x5A79,0xE8F1,
        0xE8F2,0x9374,0x3CCD,0xE8F5,0x4A96,0x398A,0x50F4,0x3D69,0x3D4C,0xE8FB,
        0x7175,0x42FB,0xE8FE,0x6E0F,0xE900,0x44EB,0x6D57,0xE903,0x7067,0x6CAF,
        0x3CD6,0xE907,0xE908,0x6E02,0x6F0C,0x3D6F,0xE90C,0x7551,0x36BC,0x34C8,
        0x4680,0x3EDA,0x4871,0x59C4,0x926E,0x493E,0x8F41,0xE917,0xE918,0x5812,
        0x57C8,0x36D6,0xE91C,0x70FE,0xE91E,0xE91F,0xE920,0xE921,0xE922,0x68B9,
        0x6967,0xE925,0xE926,0xE927,0xE928,0xE929,0xE92A,0xE92B,0xE92C,0x6A1A,
        0xE92E,0xE92F,0x843E,0x44DF,0x44CE,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x98:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE933,0xE934,0xE935,0xE936,0x6F17,0xE938,
        0x833D,0xE93A,0x83ED,0xE93C,0xE93D,0xE93E,0x5989,0x5A82,0xE941,0x5A61,
        0x5A71,0xE944,0xE945,0x372D,0x59EF,0xE948,0x36C7,0x718E,0x9390,0x669A,
        0xE94D,0x5A6E,0x5A2B,0xE950,0x6A2B,0xE952,0xE953,0xE954,0xE955,0x711D,
        0xE957,0xE958,0x4FB0,0xE95A,0x5CC2,0xE95C,0xE95D,0xE95E,0x6A0C,0xE960,
        0xE961,0x70A6,0x7133,0xE964,0x3DA5,0x6CDF,0xE967,0xE968,0x7E65,0x59EB,
        0x5D2F,0x3DF3,0x5F5C,0xE96E,0xE96F,0x7DA4,0x8426,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x5485,0xE973,0xE974,0xE975,0x577E,0xE977,0xE978,0x3FE5,0xE97A,
        0xE97B,0x7003,0xE97D,0x5D70,0x738F,0x7CD3,0xE981,0xE982,0x4FC8,0x7FE7,
        0x72CD,0x7310,0xE987,0x7338,0x7339,0xE98A,0x7341,0x7348,0x3EA9,0xE98E,
        0x906C,0x71F5,0xE991,0x73E1,0x81F6,0x3ECA,0x770C,0x3ED1,0x6CA2,0x56FD,
        0x7419,0x741E,0x741F,0x3EE2,0x3EF0,0x3EF4,0x3EFA,0x74D3,0x3F0E,0x3F53,
        0x7542,0x756D,0x7572,0x758D,0x3F7C,0x75C8,0x75DC,0x3FC0,0x764D,0x3FD7,
        0x7674,0x3FDC,0x767A,0xE9B0,0x7188,0x5623,0x8980,0x5869,0x401D,0x7743,
        0x4039,0x6761,0x4045,0x35DB,0x7798,0x406A,0x406F,0x5C5E,0x77BE,0x77CB,
        0x58F2,0x7818,0x70B9,0x781C,0x40A8,0x7839,0x7847,0x7851,0x7866,0x8448,
        0xE9CB,0x7933,0x6803,0x7932,0x4103,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x99:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x4109,0x7991,0x7999,0x8FBB,0x7A06,0x8FBC,
        0x4167,0x7A91,0x41B2,0x7ABC,0x8279,0x41C4,0x7ACF,0x7ADB,0x41CF,0x4E21,
        0x7B62,0x7B6C,0x7B7B,0x7C12,0x7C1B,0x4260,0x427A,0x7C7B,0x7C9C,0x428C,
        0x7CB8,0x4294,0x7CED,0x8F93,0x70C0,0xE9EF,0x7DCF,0x7DD4,0x7DD0,0x7DFD,
        0x7FAE,0x7FB4,0x729F,0x4397,0x8020,0x8025,0x7B39,0x802E,0x8031,0x8054,
        0x3DCC,0x57B4,0x70A0,0x80B7,0x80E9,0x43ED,0x810C,0x732A,0x810E,0x8112,
        0x7560,0x8114,0x4401,0x3B39,0x8156,0x8159,0x815A,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x4413,0x583A,0x817C,0x8184,0x4425,0x8193,0x442D,0x81A5,0x57EF,
        0x81C1,0x81E4,0x8254,0x448F,0x82A6,0x8276,0x82CA,0x82D8,0x82FF,0x44B0,
        0x8357,0x9669,0x698A,0x8405,0x70F5,0x8464,0x60E3,0x8488,0x4504,0x84BE,
        0x84E1,0x84F8,0x8510,0x8538,0x8552,0x453B,0x856F,0x8570,0x85E0,0x4577,
        0x8672,0x8692,0x86B2,0x86EF,0x9645,0x878B,0x4606,0x4617,0x88AE,0x88FF,
        0x8924,0x8947,0x8991,0xEA43,0x8A29,0x8A38,0x8A94,0x8AB4,0x8C51,0x8CD4,
        0x8CF2,0x8D1C,0x4798,0x585F,0x8DC3,0x47ED,0x4EEE,0x8E3A,0x55D8,0x5754,
        0x8E71,0x55F5,0x8EB0,0x4837,0x8ECE,0x8EE2,0x8EE4,0x8EED,0x8EF2,0x8FB7,
        0x8FC1,0x8FCA,0x8FCC,0x9033,0x99C4,0x48AD,0x98E0,0x9213,0x491E,0x9228,
        0x9258,0x926B,0x92B1,0x92AE,0x92BF,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9a:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x92E3,0x92EB,0x92F3,0x92F4,0x92FD,0x9343,
        0x9384,0x93AD,0x4945,0x4951,0x9EBF,0x9417,0x5301,0x941D,0x942D,0x943E,
        0x496A,0x9454,0x9479,0x952D,0x95A2,0x49A7,0x95F4,0x9633,0x49E5,0x67A0,
        0x4A24,0x9740,0x4A35,0x97B2,0x97C2,0x5654,0x4AE4,0x60E8,0x98B9,0x4B19,
        0x98F1,0x5844,0x990E,0x9919,0x51B4,0x991C,0x9937,0x9942,0x995D,0x9962,
        0x4B70,0x99C5,0x4B9D,0x9A3C,0x9B0F,0x7A83,0x9B69,0x9B81,0x9BDD,0x9BF1,
        0x9BF4,0x4C6D,0x9C20,0x376F,0xEAA9,0x9D49,0x9C3A,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x9EFE,0x5650,0x9D93,0x9DBD,0x9DC0,0x9DFC,0x94F6,0x8FB6,0x9E7B,
        0x9EAC,0x9EB1,0x9EBD,0x9EC6,0x94DC,0x9EE2,0x9EF1,0x9EF8,0x7AC8,0x9F44,
        0xEABF,0xEAC0,0xEAC1,0x691A,0x94C3,0x59AC,0xEAC5,0x5840,0x94C1,0x37B9,
        0xEAC9,0xEACA,0xEACB,0xEACC,0x5757,0x7173,0xEACF,0xEAD0,0xEAD1,0x546A,
        0xEAD3,0xEAD4,0x549E,0xEAD6,0xEAD7,0xEAD8,0xEAD9,0xEADA,0x60E7,0xEADC,
        0x567A,0xEADE,0xEADF,0xEAE0,0xEAE1,0xEAE2,0xEAE3,0x6955,0x9C2F,0x87A5,
        0xEAE7,0xEAE8,0xEAE9,0xEAEA,0xEAEB,0xEAEC,0x5C20,0xEAEE,0x5E0B,0xEAF0,
        0xEAF1,0xEAF2,0x671E,0xEAF4,0xEAF5,0xEAF6,0x3647,0xEAF8,0xEAF9,0xEAFA,
        0xEAFB,0x5364,0x84AD,0xEAFE,0xEAFF,0xEB00,0x8B81,0xEB02,0xEB03,0xEB04,
        0xEB05,0x4E78,0x70BB,0xEB08,0xEB09,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9b:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xEB0A,0xEB0B,0xEB0C,0xEB0D,0xEB0E,0xEB0F,
        0x62C3,0xEB11,0xEB12,0x7198,0x6855,0xEB15,0x69E9,0x36C8,0xEB18,0xEB19,
        0xEB1A,0xEB1B,0xEB1C,0xEB1D,0x82FD,0xEB1F,0xEB20,0xEB21,0x89A5,0xEB23,
        0x8FA0,0xEB25,0x97B8,0xEB27,0x9847,0x9ABD,0xEB2A,0xffff,0xEB2C,0xEB2D,
        0xEB2E,0xEB2F,0xEB30,0xEB31,0xEB32,0xEB33,0xEB34,0xEB35,0xEB36,0xEB37,
        0xEB38,0xEB39,0x5FB1,0x6648,0x66BF,0xEB3D,0xEB3E,0xEB3F,0x7201,0xEB41,
        0x77D7,0xEB43,0xEB44,0x7E87,0xEB46,0x58B5,0x670E,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x6918,0xEB4A,0xEB4B,0xEB4C,0xEB4D,0xEB4E,0xEB4F,0xEB50,0x48D0,
        0x4AB8,0xEB53,0xEB54,0xEB55,0xEB56,0xEB57,0xEB58,0xEB59,0xEB5A,0xEB5B,
        0x51D2,0xEB5D,0x599F,0xEB5F,0x3BBE,0xEB61,0xEB62,0xEB63,0x5788,0xEB65,
        0x399B,0xEB67,0xEB68,0xEB69,0x3762,0xEB6B,0x8B5E,0xEB6D,0x99D6,0xEB6F,
        0xEB70,0xEB71,0x7209,0xEB73,0xEB74,0x5965,0xEB76,0xEB77,0xEB78,0x8EDA,
        0xEB7A,0x528F,0x573F,0x7171,0xEB7E,0xEB7F,0xEB80,0xEB81,0x55BC,0xEB83,
        0xEB84,0xEB85,0x91D4,0x3473,0xEB88,0xEB89,0xEB8A,0x4718,0xEB8C,0xEB8D,
        0xEB8E,0xEB8F,0xEB90,0x5066,0x34FB,0xEB93,0x60DE,0xEB95,0x477C,0xEB97,
        0xEB98,0xEB99,0xEB9A,0xEB9B,0x57A1,0x7151,0x6FB6,0xEB9F,0xEBA0,0x9056,
        0xEBA2,0xEBA3,0x8B62,0xEBA5,0xEBA6,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9c:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x5D5B,0xEBA8,0x8F36,0xEBAA,0xEBAB,0x8AEA,
        0xEBAD,0xEBAE,0xEBAF,0xEBB0,0x4BC0,0xEBB2,0xEBB3,0xEBB4,0x9465,0xEBB6,
        0x6195,0x5A27,0xEBB9,0x4FBB,0x56B9,0xEBBC,0xEBBD,0x4E6A,0xEBBF,0x9656,
        0x6D8F,0xEBC2,0x3618,0x8977,0xEBC5,0xEBC6,0xEBC7,0xEBC8,0x71DF,0xEBCA,
        0x7B42,0xEBCC,0xEBCD,0xEBCE,0x9104,0xEBD0,0x7A45,0x9DF0,0xEBD3,0x9A26,
        0xEBD5,0x365F,0xEBD7,0xEBD8,0x7983,0xEBDA,0xEBDB,0x5D2C,0xEBDD,0x83CF,
        0xEBDF,0x46D0,0xEBE1,0x753B,0x8865,0xEBE4,0x58B6,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x371C,0xEBE7,0xEBE8,0xEBE9,0x3C54,0xEBEB,0xEBEC,0x9281,0xEBEE,
        0xEBEF,0x9330,0xEBF1,0xEBF2,0x6C39,0x949F,0xEBF5,0xEBF6,0x8827,0x88F5,
        0xEBF9,0xEBFA,0xEBFB,0x6EB8,0xEBFD,0xEBFE,0x39A4,0x36B9,0x5C10,0x79E3,
        0x453F,0x66B6,0xEC05,0xEC06,0x8943,0xEC08,0xEC09,0x56D6,0x40DF,0xEC0C,
        0x39A1,0xEC0E,0xEC0F,0xEC10,0x71AD,0x8366,0xEC13,0xEC14,0x5A67,0x4CB7,
        0xEC17,0xEC18,0xEC19,0xEC1A,0xEC1B,0xEC1C,0xEC1D,0x7B43,0x797E,0xEC20,
        0x6FB5,0xEC22,0x6A03,0xEC24,0x53A2,0xEC26,0x93BF,0x6836,0x975D,0xEC2A,
        0xEC2B,0xEC2C,0xEC2D,0xEC2E,0xEC2F,0x5D85,0xEC31,0xEC32,0x5715,0x9823,
        0xEC35,0x5DAB,0xEC37,0x65BE,0x69D5,0x53D2,0xEC3B,0xEC3C,0x3C11,0x6736,
        0xEC3F,0xEC40,0xEC41,0xEC42,0xEC43,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9d:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xEC44,0xEC45,0xEC46,0xEC47,0xEC48,0xEC49,
        0x35CA,0xEC4B,0xEC4C,0x48FA,0x63E6,0xEC4F,0x7808,0x9255,0xEC52,0x43F2,
        0xEC54,0x43DF,0xEC56,0xEC57,0xEC58,0x59F8,0xEC5A,0x8F0B,0xEC5C,0xEC5D,
        0x7B51,0xEC5F,0xEC60,0x3DF7,0xEC62,0xEC63,0x8FD0,0x728F,0x568B,0xEC67,
        0xEC68,0xEC69,0xEC6A,0xEC6B,0xEC6C,0xEC6D,0xEC6E,0xEC6F,0xEC70,0xEC71,
        0xEC72,0xEC73,0x7E9F,0xEC75,0xEC76,0x4CA4,0x9547,0xEC79,0x71A2,0xEC7B,
        0x4D91,0x9012,0xEC7E,0x4D9C,0xEC80,0x8FBE,0x55C1,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x8FBA,0xEC84,0x8FB9,0xEC86,0x4509,0x7E7F,0x6F56,0x6AB1,0x4EEA,
        0x34E4,0xEC8D,0xEC8E,0x373A,0x8E80,0xEC91,0xEC92,0xEC93,0xEC94,0xEC95,
        0xEC96,0x3DEB,0xEC98,0xEC99,0xEC9A,0xEC9B,0x4E9A,0xEC9D,0xEC9E,0x56BF,
        0xECA0,0x8E0E,0x5B6D,0xECA3,0xECA4,0x63DE,0x62D0,0xECA7,0xECA8,0x6530,
        0x562D,0xECAB,0x541A,0xECAD,0x3DC6,0xECAF,0x4C7D,0x5622,0x561E,0x7F49,
        0xECB4,0x5975,0xECB6,0x8770,0x4E1C,0xECB9,0xECBA,0xECBB,0x8117,0x9D5E,
        0x8D18,0x763B,0x9C45,0x764E,0x77B9,0x9345,0x5432,0x8148,0x82F7,0x5625,
        0x8132,0x8418,0x80BD,0x55EA,0x7962,0x5643,0x5416,0xECCF,0x35CE,0x5605,
        0x55F1,0x66F1,0xECD4,0x362D,0x7534,0x55F0,0x55BA,0x5497,0x5572,0xECDB,
        0xECDC,0x5ED0,0xECDE,0xECDF,0xECE0,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9e:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xECE1,0x9EAB,0x7D5A,0x55DE,0xECE5,0x629D,
        0x976D,0x5494,0x8CCD,0x71F6,0x9176,0x63FC,0x63B9,0x63FE,0x5569,0xECF0,
        0x9C72,0xECF2,0x519A,0x34DF,0xECF5,0x51A7,0x544D,0x551E,0x5513,0x7666,
        0x8E2D,0xECFC,0x75B1,0x80B6,0x8804,0x8786,0x88C7,0x81B6,0x841C,0xED04,
        0x44EC,0x7304,0xED07,0x5B90,0x830B,0xED0A,0x567B,0xED0C,0xED0D,0xED0E,
        0xED0F,0xED10,0xED11,0x9170,0xED13,0x9208,0xED15,0xED16,0xED17,0xED18,
        0x7266,0xED1A,0x474E,0xED1C,0xED1D,0xED1E,0x40FA,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x9C5D,0x651F,0xED22,0x48F3,0xED24,0xED25,0xED26,0xED27,0x6062,
        0xED29,0xED2A,0xffff,0xED2C,0xED2D,0x71A3,0x7E8E,0x9D50,0x4E1A,0x4E04,
        0x3577,0x5B0D,0x6CB2,0x5367,0x36AC,0x39DC,0x537D,0x36A5,0xED3B,0x589A,
        0xED3D,0x822D,0x544B,0x57AA,0xED41,0xED42,0xffff,0x3A52,0xED45,0x7374,
        0xED47,0x4D09,0x9BED,0xED4A,0xED4B,0x4C5B,0xED4D,0xED4E,0xED4F,0x845C,
        0xED51,0xED52,0xED53,0xED54,0x632E,0x7D25,0xED57,0xED58,0x3A2A,0x9008,
        0x52CC,0x3E74,0x367A,0x45E9,0xED5F,0x7640,0x5AF0,0xED62,0x787A,0x47B6,
        0x58A7,0x40BF,0x567C,0x9B8B,0x5D74,0x7654,0xED6B,0x9E85,0x4CE1,0x75F9,
        0x37FB,0x6119,0xED71,0xED72,0xffff,0x565D,0xED75,0x57A7,0xED77,0xED78,
        0x5234,0xED7A,0x35AD,0x6C4A,0x9D7C,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9f:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x7C56,0x9B39,0x57DE,0xED81,0x5C53,0x64D3,
        0xED84,0xED85,0xED86,0x86AD,0xED88,0xED89,0xED8A,0xED8B,0xffff,0x51FE,
        0xED8E,0x5D8E,0x9703,0xED91,0x9E81,0x904C,0x7B1F,0x9B02,0x5CD1,0x7BA3,
        0x6268,0x6335,0x9AFF,0x7BCF,0x9B2A,0x7C7E,0x9B2E,0x7C42,0x7C86,0x9C15,
        0x7BFC,0x9B09,0x9F17,0x9C1B,0xEDA6,0x9F5A,0x5573,0x5BC3,0x4FFD,0x9E98,
        0x4FF2,0x5260,0x3E06,0x52D1,0x5767,0x5056,0x59B7,0x5E12,0x97C8,0x9DAB,
        0x8F5C,0x5469,0x97B4,0x9940,0x97BA,0x532C,0x6130,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x692C,0x53DA,0x9C0A,0x9D02,0x4C3B,0x9641,0x6980,0x50A6,0x7546,
        0xEDC6,0x99DA,0x5273,0xffff,0x9159,0x9681,0x915C,0xffff,0x9151,0xEDCF,
        0x637F,0xEDD1,0x6ACA,0x5611,0x918E,0x757A,0x6285,0xEDD7,0x734F,0x7C70,
        0xEDDA,0xEDDB,0xffff,0xEDDD,0x76D6,0x9B9D,0x4E2A,0xEDE1,0x83BE,0x8842,
        0xffff,0x5C4A,0x69C0,0x50ED,0x577A,0x521F,0x5DF5,0x4ECE,0x6C31,0xEDED,
        0x4F39,0x549C,0x54DA,0x529A,0x8D82,0x35FE,0x5F0C,0x35F3,0xffff,0x6B52,
        0x917C,0x9FA5,0x9B97,0x982E,0x98B4,0x9ABA,0x9EA8,0x9E84,0x717A,0x7B14,
        0xffff,0x6BFA,0x8818,0x7F78,0xffff,0x5620,0xEE08,0x8E77,0x9F53,0xffff,
        0x8DD4,0x8E4F,0x9E1C,0x8E01,0x6282,0xEE11,0x8E28,0x8E75,0x7AD3,0xEE15,
        0x7A3E,0x78D8,0x6CEA,0x8A67,0x7607,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa0:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xEE1B,0x9F26,0x6CCE,0x87D6,0x75C3,0xEE20,
        0x7853,0xEE22,0x8D0C,0x72E2,0x7371,0x8B2D,0x7302,0x74F1,0x8CEB,0xEE2A,
        0x862F,0x5FBA,0x88A0,0x44B7,0xffff,0xEE30,0xEE31,0xffff,0x8A7E,0xEE34,
        0xffff,0x60FD,0x7667,0x9AD7,0x9D44,0x936E,0x9B8F,0x87F5,0xffff,0x880F,
        0x8CF7,0x732C,0x9721,0x9BB0,0x35D6,0x72B2,0x4C07,0x7C51,0x994A,0xEE48,
        0x6159,0x4C04,0x9E96,0x617D,0xffff,0x575F,0x616F,0x62A6,0x6239,0x62CE,
        0x3A5C,0x61E2,0x53AA,0xEE56,0x6364,0x6802,0x35D2,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x5D57,0xEE5B,0x8FDA,0xEE5D,0xffff,0x50D9,0xEE60,0x7906,0x5332,
        0x9638,0xEE64,0x4065,0xffff,0x77FE,0xffff,0x7CC2,0xEE6A,0x7CDA,0x7A2D,
        0x8066,0x8063,0x7D4D,0x7505,0x74F2,0x8994,0x821A,0x670C,0x8062,0xEE76,
        0x805B,0x74F0,0x8103,0x7724,0x8989,0xEE7C,0x7553,0xEE7E,0x87A9,0x87CE,
        0x81C8,0x878C,0x8A49,0x8CAD,0x8B43,0x772B,0x74F8,0x84DA,0x3635,0x69B2,
        0x8DA6,0xffff,0x89A9,0x7468,0x6DB9,0x87C1,0xEE91,0x74E7,0x3DDB,0x7176,
        0x60A4,0x619C,0x3CD1,0x7162,0x6077,0xffff,0x7F71,0xEE9C,0x7250,0x60E9,
        0x4B7E,0x5220,0x3C18,0xEEA2,0xEEA3,0xEEA4,0xEEA5,0xEEA6,0xEEA7,0xEEA8,
        0xEEA9,0xEEAA,0x5CC1,0xEEAC,0xEEAD,0xEEAE,0xEEAF,0xEEB0,0xEEB1,0x4562,
        0x5B1F,0xEEB4,0x9F50,0x9EA6,0xEEB7,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc6:
    if (c2>=0xa1 && c2<=0xfe) {
      static int const(val[])= {
        0x2460,0x2461,0x2462,0x2463,0x2464,0x2465,0x2466,0x2467,0x2468,
        0x2469,0x2474,0x2475,0x2476,0x2477,0x2478,0x2479,0x247A,0x247B,0x247C,
        0x247D,0x2170,0x2171,0x2172,0x2173,0x2174,0x2175,0x2176,0x2177,0x2178,
        0x2179,0x4E36,0x4E3F,0x4E85,0x4EA0,0x5182,0x5196,0x51AB,0x52F9,0x5338,
        0x5369,0x53B6,0x590A,0x5B80,0x5DDB,0x2F33,0x5E7F,0xF6E0,0x5F50,0x5F61,
        0x6534,0xF6E4,0x7592,0xF6E6,0x8FB5,0xF6E8,0x00A8,0x02C6,0x30FD,0x30FE,
        0x309D,0x309E,0xF6EF,0xF6F0,0x3005,0x3006,0x3007,0x30FC,0xFF3B,0xFF3D,
        0x273D,0x3041,0x3042,0x3043,0x3044,0x3045,0x3046,0x3047,0x3048,0x3049,
        0x304A,0x304B,0x304C,0x304D,0x304E,0x304F,0x3050,0x3051,0x3052,0x3053,
        0x3054,0x3055,0x3056,0x3057,0x3058,
      };
      unicode=(uint32_t) val[c2-0xa1];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc7:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x3059,0x305A,0x305B,0x305C,0x305D,0x305E,
        0x305F,0x3060,0x3061,0x3062,0x3063,0x3064,0x3065,0x3066,0x3067,0x3068,
        0x3069,0x306A,0x306B,0x306C,0x306D,0x306E,0x306F,0x3070,0x3071,0x3072,
        0x3073,0x3074,0x3075,0x3076,0x3077,0x3078,0x3079,0x307A,0x307B,0x307C,
        0x307D,0x307E,0x307F,0x3080,0x3081,0x3082,0x3083,0x3084,0x3085,0x3086,
        0x3087,0x3088,0x3089,0x308A,0x308B,0x308C,0x308D,0x308E,0x308F,0x3090,
        0x3091,0x3092,0x3093,0x30A1,0x30A2,0x30A3,0x30A4,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x30A5,0x30A6,0x30A7,0x30A8,0x30A9,0x30AA,0x30AB,0x30AC,0x30AD,
        0x30AE,0x30AF,0x30B0,0x30B1,0x30B2,0x30B3,0x30B4,0x30B5,0x30B6,0x30B7,
        0x30B8,0x30B9,0x30BA,0x30BB,0x30BC,0x30BD,0x30BE,0x30BF,0x30C0,0x30C1,
        0x30C2,0x30C3,0x30C4,0x30C5,0x30C6,0x30C7,0x30C8,0x30C9,0x30CA,0x30CB,
        0x30CC,0x30CD,0x30CE,0x30CF,0x30D0,0x30D1,0x30D2,0x30D3,0x30D4,0x30D5,
        0x30D6,0x30D7,0x30D8,0x30D9,0x30DA,0x30DB,0x30DC,0x30DD,0x30DE,0x30DF,
        0x30E0,0x30E1,0x30E2,0x30E3,0x30E4,0x30E5,0x30E6,0x30E7,0x30E8,0x30E9,
        0x30EA,0x30EB,0x30EC,0x30ED,0x30EE,0x30EF,0x30F0,0x30F1,0x30F2,0x30F3,
        0x30F4,0x30F5,0x30F6,0x0410,0x0411,0x0412,0x0413,0x0414,0x0415,0x0401,
        0x0416,0x0417,0x0418,0x0419,0x041A,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc8:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x041B,0x041C,0x041D,0x041E,0x041F,0x0420,
        0x0421,0x0422,0x0423,0x0424,0x0425,0x0426,0x0427,0x0428,0x0429,0x042A,
        0x042B,0x042C,0x042D,0x042E,0x042F,0x0430,0x0431,0x0432,0x0433,0x0434,
        0x0435,0x0451,0x0436,0x0437,0x0438,0x0439,0x043A,0x043B,0x043C,0x043D,
        0x043E,0x043F,0x0440,0x0441,0x0442,0x0443,0x0444,0x0445,0x0446,0x0447,
        0x0448,0x0449,0x044A,0x044B,0x044C,0x044D,0x044E,0x044F,0x21E7,0x21B8,
        0x21B9,0xF7E5,0xF7E6,0x4E5A,0xF7E8,0x5202,0xF7EA,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xF7EB,0x5188,0xF7ED,0xF7EE,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xFFE2,0xFFE4,0xFF07,0xFF02,0x3231,
        0x2116,0x2121,0x309B,0x309C,0x2E80,0x2E84,0x2E86,0x2E87,0x2E88,0x2E8A,
        0x2E8C,0x2E8D,0x2E95,0x2E9C,0x2E9D,0x2EA5,0x2EA7,0x2EAA,0x2EAC,0x2EAE,
        0x2EB6,0x2EBC,0x2EBE,0x2EC6,0x2ECA,0x2ECC,0x2ECD,0x2ECF,0x2ED6,0x2ED7,
        0x2EDE,0x2EE3,0xffff,0xffff,0xffff,0x0283,0x0250,0x025B,0x0254,0x0275,
        0x0153,0x00F8,0x014B,0x028A,0x026A,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf9:
    if (c2>=0xd6 && c2<=0xfe) {
      static int const(val[])= {
        0x7881,0x92B9,0x88CF,0x58BB,0x6052,0x7CA7,
        0x5AFA,0x2554,0x2566,0x2557,0x2560,0x256C,0x2563,0x255A,0x2569,0x255D,
        0x2552,0x2564,0x2555,0x255E,0x256A,0x2561,0x2558,0x2567,0x255B,0x2553,
        0x2565,0x2556,0x255F,0x256B,0x2562,0x2559,0x2568,0x255C,0x2551,0x2550,
        0x256D,0x256E,0x2570,0x256F,0xFFED,
      };
      unicode=(uint32_t) val[c2-0xd6];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xfa:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE000,0x92DB,0xE002,0xE003,0x854C,0x42B5,
        0x73EF,0x51B5,0x3649,0xE009,0xE00A,0x9344,0xE00C,0x82EE,0xE00E,0x783C,
        0x6744,0x62DF,0xE012,0xE013,0xE014,0xE015,0xE016,0x4FAB,0xE018,0x5008,
        0xE01A,0xE01B,0xE01C,0xE01D,0xE01E,0x5029,0xE020,0x5FA4,0xE022,0xE023,
        0x6EDB,0xE025,0x507D,0x5101,0x347A,0x510E,0x986C,0x3743,0x8416,0xE02D,
        0xE02E,0x5160,0xE030,0x516A,0xE032,0xE033,0xE034,0xE035,0xE036,0xE037,
        0xE038,0x5B82,0x877D,0xE03B,0xE03C,0x51B2,0x51B8,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0x9D34,0x51C9,0x51CF,0x51D1,0x3CDC,0x51D3,0xE045,0x51B3,0x51E2,
        0x5342,0x51ED,0x83CD,0x693E,0xE04C,0x5F7B,0x520B,0x5226,0x523C,0x52B5,
        0x5257,0x5294,0x52B9,0x52C5,0x7C15,0x8542,0x52E0,0x860D,0xE05A,0x5305,
        0xE05C,0x5549,0x6ED9,0xE05F,0xE060,0xE061,0x5333,0x5344,0xE064,0x6CCB,
        0xE066,0x681B,0x73D5,0x604A,0x3EAA,0x38CC,0xE06C,0x71DD,0x44A2,0x536D,
        0x5374,0xE071,0x537E,0x537F,0xE074,0xE075,0x77E6,0x5393,0xE078,0x53A0,
        0x53AB,0x53AE,0x73A7,0xE07D,0x3F59,0x739C,0x53C1,0x53C5,0x6C49,0x4E49,
        0x57FE,0x53D9,0x3AAB,0xE087,0x53E0,0xE089,0xE08A,0x53F6,0xE08C,0x5413,
        0x7079,0x552B,0x6657,0x6D5B,0x546D,0xE093,0xE094,0x555D,0x548F,0x54A4,
        0x47A6,0xE099,0xE09A,0x3DB4,0xE09C,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xfb:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE09D,0xE09E,0x5547,0x4CED,0x542F,0x7417,
        0x5586,0x55A9,0x5605,0xE0A6,0xE0A7,0x4552,0xE0A9,0x66B3,0xE0AB,0x5637,
        0x66CD,0xE0AE,0x66A4,0x66AD,0x564D,0x564F,0x78F1,0x56F1,0x9787,0x53FE,
        0x5700,0x56EF,0x56ED,0xE0BA,0x3623,0xE0BC,0x5746,0xE0BE,0x6C6E,0x708B,
        0x5742,0x36B1,0xE0C3,0x57E6,0xE0C5,0x5803,0xE0C7,0xE0C8,0x5826,0xE0CA,
        0x585C,0x58AA,0x3561,0x58E0,0x58DC,0xE0D0,0x58FB,0x5BFF,0x5743,0xE0D4,
        0xE0D5,0x93D3,0x35A1,0x591F,0x68A6,0x36C3,0x6E59,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE0DC,0x5A24,0x5553,0xE0DF,0x8505,0x59C9,0xE0E2,0xE0E3,0xE0E4,
        0xE0E5,0x59D9,0xE0E7,0xE0E8,0xE0E9,0x6D71,0xE0EB,0xE0EC,0x59F9,0xE0EE,
        0x5AAB,0x5A63,0x36E6,0xE0F2,0x5A77,0x3708,0x5A96,0x7465,0x5AD3,0xE0F8,
        0xE0F9,0x3D85,0xE0FB,0x3732,0xE0FD,0x5E83,0x52D0,0x5B76,0x6588,0x5B7C,
        0xE103,0x4004,0x485D,0xE106,0x5BD5,0x6160,0xE109,0xE10A,0xE10B,0x5BF3,
        0x5B9D,0x4D10,0x5C05,0xE110,0x5C13,0x73CE,0x5C14,0xE114,0xE115,0x5C49,
        0x48DD,0x5C85,0x5CE9,0x5CEF,0x5D8B,0xE11C,0xE11D,0x5D10,0x5D18,0x5D46,
        0xE121,0x5CBA,0x5DD7,0x82FC,0x382D,0xE126,0xE127,0xE128,0x8287,0x3836,
        0x3BC2,0x5E2E,0x6A8A,0x5E75,0x5E7A,0xE130,0xE131,0x53A6,0x4EB7,0x5ED0,
        0x53A8,0xE136,0x5E09,0x5EF4,0xE139,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xfc:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x5EF9,0x5EFB,0x38A0,0x5EFC,0x683E,0x941B,
        0x5F0D,0xE141,0xE142,0x3ADE,0x48AE,0xE145,0x5F3A,0xE147,0xE148,0x5F58,
        0xE14A,0x5F63,0x97BD,0xE14D,0x5F72,0x9340,0xE150,0x5FA7,0x5DB6,0x3D5F,
        0xE154,0xE155,0xE156,0xE157,0x91D6,0xE159,0xE15A,0x6031,0x6685,0xE15D,
        0x3963,0x3DC7,0x3639,0x5790,0xE162,0x7971,0x3E40,0x609E,0x60A4,0x60B3,
        0xE168,0xE169,0xE16A,0x74A4,0x50E1,0x5AA0,0x6164,0x8424,0x6142,0xE171,
        0xE172,0x6181,0x51F4,0xE175,0x6187,0x5BAA,0xE178,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE179,0x61D3,0xE17B,0xE17C,0x61D0,0x3932,0xE17F,0xE180,0x6023,
        0x615C,0x651E,0x638B,0xE185,0x62C5,0xE187,0x62D5,0xE189,0x636C,0xE18B,
        0x3A17,0x6438,0x63F8,0xE18F,0xE190,0x6490,0x6F8A,0xE193,0x9814,0xE195,
        0xE196,0x64E1,0x64E5,0x947B,0x3A66,0x643A,0x3A57,0x654D,0x6F16,0xE19F,
        0xE1A0,0x6585,0x656D,0x655F,0xE1A4,0x65B5,0xE1A6,0x4B37,0x65D1,0x40D8,
        0xE1AA,0x65E0,0x65E3,0x5FDF,0xE1AE,0x6618,0xE1B0,0xE1B1,0x6644,0xE1B3,
        0xE1B4,0x664B,0xE1B6,0x6667,0xE1B8,0x6673,0x6674,0xE1BB,0xE1BC,0xE1BD,
        0xE1BE,0xE1BF,0x77C5,0xE1C1,0x99A4,0x6702,0xE1C4,0xE1C5,0x3B2B,0x69FA,
        0xE1C8,0x675E,0x6767,0x6762,0xE1CC,0xE1CD,0x67D7,0x44E9,0x6822,0x6E50,
        0x923C,0x6801,0xE1D4,0xE1D5,0x685D,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xfd:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0xE1D7,0x69E1,0x6A0B,0xE1DA,0x6973,0x68C3,
        0xE1DD,0x6901,0x6900,0x3D32,0x3A01,0xE1E2,0x3B80,0x67AC,0x6961,0xE1E6,
        0x42FC,0x6936,0x6998,0x3BA1,0xE1EB,0x8363,0x5090,0x69F9,0xE1EF,0xE1F0,
        0x6A45,0xE1F2,0x6A9D,0x3BF3,0x67B1,0x6AC8,0xE1F7,0x3C0D,0x6B1D,0xE1FA,
        0x60DE,0x6B35,0x6B74,0xE1FE,0x6EB5,0xE200,0xE201,0xE202,0x3740,0x5421,
        0xE205,0x6BE1,0xE207,0x6BDC,0x6C37,0xE20A,0xE20B,0xE20C,0x6C5A,0x8226,
        0x6C79,0xE210,0x44C5,0xE212,0xE213,0xE214,0xE215,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE216,0x36E5,0x3CEB,0xE219,0x9B83,0xE21B,0xE21C,0x7F8F,0x6837,
        0xE21F,0xE220,0xE221,0x6D96,0x6D5C,0x6E7C,0x6F04,0xE226,0xE227,0xE228,
        0x8533,0xE22A,0x51C7,0x6C9C,0x6E1D,0x842E,0xE22F,0x6E2F,0xE231,0x7453,
        0xE233,0x79CC,0x6E4F,0x5A91,0xE237,0x6FF8,0x370D,0x6F9D,0xE23B,0x6EFA,
        0xE23D,0xE23E,0x4555,0x93F0,0x6F44,0x6F5C,0x3D4E,0x6F74,0xE245,0x3D3B,
        0x6F9F,0xE248,0x6FD3,0xE24A,0xE24B,0xE24C,0xE24D,0xE24E,0xE24F,0x51DF,
        0xE251,0xE252,0xE253,0xE254,0x704B,0x707E,0x70A7,0x7081,0x70CC,0x70D5,
        0x70D6,0x70DF,0x4104,0x3DE8,0x71B4,0x7196,0xE261,0x712B,0x7145,0x5A88,
        0x714A,0x716E,0x5C9C,0xE268,0x714F,0x9362,0xE26B,0x712C,0xE26D,0xE26E,
        0xE26F,0x71BA,0xE271,0x70BD,0x720E,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xfe:
    if (c2>=0x40 && c2<=0xfe) {
      static int const(val[])= {
        0x9442,0x7215,0x5911,0x9443,0x7224,0x9341,
        0xE27A,0x722E,0x7240,0xE27D,0x68BD,0x7255,0x7257,0x3E55,0xE282,0x680D,
        0x6F3D,0x7282,0x732A,0x732B,0xE288,0xE289,0x48ED,0xE28B,0x7328,0x732E,
        0x73CF,0x73AA,0xE290,0xE291,0x73C9,0x7449,0xE294,0xE295,0xE296,0x6623,
        0x36C5,0xE299,0xE29A,0xE29B,0x73F7,0x7415,0x6903,0xE29F,0x7439,0xE2A1,
        0x3ED7,0x745C,0xE2A4,0x7460,0xE2A6,0x7447,0x73E4,0x7476,0x83B9,0x746C,
        0x3730,0x7474,0x93F1,0x6A2C,0x7482,0x4953,0xE2B2,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
        0xffff,0xE2B3,0xE2B4,0xE2B5,0x5B46,0xE2B7,0xE2B8,0x74C8,0xE2BA,0x750E,
        0x74E9,0x751E,0xE2BE,0xE2BF,0x5BD7,0xE2C1,0x9385,0x754D,0x754A,0x7567,
        0x756E,0xE2C7,0x3F04,0xE2C9,0x758E,0x745D,0x759E,0x75B4,0x7602,0x762C,
        0x7651,0x764F,0x766F,0x7676,0xE2D4,0x7690,0x81EF,0x37F8,0xE2D8,0xE2D9,
        0x76A1,0x76A5,0x76B7,0x76CC,0xE2DE,0x8462,0xE2E0,0xE2E1,0xE2E2,0x771E,
        0x7726,0x7740,0x64AF,0xE2E7,0x7758,0xE2E9,0x77AF,0xE2EB,0xE2EC,0xE2ED,
        0x77F4,0x7809,0xE2F0,0xE2F1,0x68CA,0x78AF,0x78C7,0x78D3,0x96A5,0x792E,
        0xE2F8,0x78D7,0x7934,0x78B1,0xE2FC,0x8FB8,0x8884,0xE2FF,0xE300,0xE301,
        0x7986,0x8900,0x6902,0x7980,0xE306,0x799D,0xE308,0x793C,0x79A9,0x6E2A,
        0xE30C,0x3EA8,0x79C6,0xE30F,0x79D4,
      };
      unicode=(uint32_t) val[c2-0x40];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  default:
    if (c>=0x80 && c<=0xfe)
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  }
  if (!unicode) {
    STOFF_DEBUG_MSG(("StarEncodingOtherKorean::readKoreanBig5: unknown caracter %x\n", (unsigned int)c));
  }
  dest.push_back(unicode);
  return true;
}

bool StarEncodingOtherKorean::readKoreanMS1361
(std::vector<uint8_t> const &src, size_t &pos, StarEncoding::Encoding encoding, std::vector<uint32_t> &dest)
{
  if (encoding!=StarEncoding::E_MS_1361) {
    STOFF_DEBUG_MSG(("StarEncodingOtherKorean::readKoreanMS1361: unknown encoding\n"));
    return false;
  }
  if (pos>=src.size()) return false;
  int c=(int) src[pos++], c2=0;
  if ((c>=0x84&&c<=0xd3)||(c>=0xd8&&c<=0xf9)) {
    if (pos>=src.size()) return false;
    c2=(int) src[pos++];
  }
  uint32_t unicode=uint32_t(c);
  switch (c) {
  case 0x84:
    if (c2>=0x45 && c2<=0xe2) {
      static int const(val[])= {
        0x3133, 	 0, 0x3135, 0x3136, /* 0x40 */
        0, 	 0, 0x313A, 0x313B, 0x313C, 0x313D, 0x313E, 0x313F, /* 0x40 */
        0x3140, 	 0, 	 0, 	 0, 0x3144, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0x314F, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 0x3150, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x3151, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 0x3152, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 0x3153													/* 0xE0 */
      };
      unicode=(uint32_t) val[c2-0x45];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x85:
    if (c2>=0x41 && c2<=0xe1) {
      static int const(val[])= {
        0x3154, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0x3155, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 0x3156, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x3157, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 0x3158, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 0x3159													/* 0xE0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x86:
    if (c2>=0x41 && c2<=0xe1) {
      static int const(val[])= {
        0x315A, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0x315B, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 0x315C, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x315D, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xA0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xB0 */
        0, 0x315E, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xD0 */
        0, 0x315F													/* 0xE0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x87:
    if (c2>=0x41 && c2<=0xa1) {
      static int const(val[])= {
        0x3160, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0x3161, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x70 */
        0, 0x3162, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x3163													/* 0xA0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x88:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3131, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xAC00, 0xAC01, 0xAC02, 0xAC03, 0xAC04, 0xAC05, 0xAC06, /* 0x60 */
        0xAC07, 0xAC08, 0xAC09, 0xAC0A, 0xAC0B, 0xAC0C, 0xAC0D, 0xAC0E, /* 0x60 */
        0xAC0F, 0xAC10, 	 0, 0xAC11, 0xAC12, 0xAC13, 0xAC14, 0xAC15, /* 0x70 */
        0xAC16, 0xAC17, 0xAC18, 0xAC19, 0xAC1A, 0xAC1B, 	 0, 	 0, /* 0x70 */
        0, 0xAC1C, 0xAC1D, 0xAC1E, 0xAC1F, 0xAC20, 0xAC21, 0xAC22, /* 0x80 */
        0xAC23, 0xAC24, 0xAC25, 0xAC26, 0xAC27, 0xAC28, 0xAC29, 0xAC2A, /* 0x80 */
        0xAC2B, 0xAC2C, 	 0, 0xAC2D, 0xAC2E, 0xAC2F, 0xAC30, 0xAC31, /* 0x90 */
        0xAC32, 0xAC33, 0xAC34, 0xAC35, 0xAC36, 0xAC37, 	 0, 	 0, /* 0x90 */
        0, 0xAC38, 0xAC39, 0xAC3A, 0xAC3B, 0xAC3C, 0xAC3D, 0xAC3E, /* 0xA0 */
        0xAC3F, 0xAC40, 0xAC41, 0xAC42, 0xAC43, 0xAC44, 0xAC45, 0xAC46, /* 0xA0 */
        0xAC47, 0xAC48, 	 0, 0xAC49, 0xAC4A, 0xAC4B, 0xAC4C, 0xAC4D, /* 0xB0 */
        0xAC4E, 0xAC4F, 0xAC50, 0xAC51, 0xAC52, 0xAC53, 	 0, 	 0, /* 0xB0 */
        0, 0xAC54, 0xAC55, 0xAC56, 0xAC57, 0xAC58, 0xAC59, 0xAC5A, /* 0xC0 */
        0xAC5B, 0xAC5C, 0xAC5D, 0xAC5E, 0xAC5F, 0xAC60, 0xAC61, 0xAC62, /* 0xC0 */
        0xAC63, 0xAC64, 	 0, 0xAC65, 0xAC66, 0xAC67, 0xAC68, 0xAC69, /* 0xD0 */
        0xAC6A, 0xAC6B, 0xAC6C, 0xAC6D, 0xAC6E, 0xAC6F, 	 0, 	 0, /* 0xD0 */
        0, 0xAC70, 0xAC71, 0xAC72, 0xAC73, 0xAC74, 0xAC75, 0xAC76, /* 0xE0 */
        0xAC77, 0xAC78, 0xAC79, 0xAC7A, 0xAC7B, 0xAC7C, 0xAC7D, 0xAC7E, /* 0xE0 */
        0xAC7F, 0xAC80, 	 0, 0xAC81, 0xAC82, 0xAC83, 0xAC84, 0xAC85, /* 0xF0 */
        0xAC86, 0xAC87, 0xAC88, 0xAC89, 0xAC8A, 0xAC8B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x89:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xAC8C, 0xAC8D, 0xAC8E, 0xAC8F, 0xAC90, 0xAC91, 0xAC92, /* 0x40 */
        0xAC93, 0xAC94, 0xAC95, 0xAC96, 0xAC97, 0xAC98, 0xAC99, 0xAC9A, /* 0x40 */
        0xAC9B, 0xAC9C, 	 0, 0xAC9D, 0xAC9E, 0xAC9F, 0xACA0, 0xACA1, /* 0x50 */
        0xACA2, 0xACA3, 0xACA4, 0xACA5, 0xACA6, 0xACA7, 	 0, 	 0, /* 0x50 */
        0, 0xACA8, 0xACA9, 0xACAA, 0xACAB, 0xACAC, 0xACAD, 0xACAE, /* 0x60 */
        0xACAF, 0xACB0, 0xACB1, 0xACB2, 0xACB3, 0xACB4, 0xACB5, 0xACB6, /* 0x60 */
        0xACB7, 0xACB8, 	 0, 0xACB9, 0xACBA, 0xACBB, 0xACBC, 0xACBD, /* 0x70 */
        0xACBE, 0xACBF, 0xACC0, 0xACC1, 0xACC2, 0xACC3, 	 0, 	 0, /* 0x70 */
        0, 0xACC4, 0xACC5, 0xACC6, 0xACC7, 0xACC8, 0xACC9, 0xACCA, /* 0x80 */
        0xACCB, 0xACCC, 0xACCD, 0xACCE, 0xACCF, 0xACD0, 0xACD1, 0xACD2, /* 0x80 */
        0xACD3, 0xACD4, 	 0, 0xACD5, 0xACD6, 0xACD7, 0xACD8, 0xACD9, /* 0x90 */
        0xACDA, 0xACDB, 0xACDC, 0xACDD, 0xACDE, 0xACDF, 	 0, 	 0, /* 0x90 */
        0, 0xACE0, 0xACE1, 0xACE2, 0xACE3, 0xACE4, 0xACE5, 0xACE6, /* 0xA0 */
        0xACE7, 0xACE8, 0xACE9, 0xACEA, 0xACEB, 0xACEC, 0xACED, 0xACEE, /* 0xA0 */
        0xACEF, 0xACF0, 	 0, 0xACF1, 0xACF2, 0xACF3, 0xACF4, 0xACF5, /* 0xB0 */
        0xACF6, 0xACF7, 0xACF8, 0xACF9, 0xACFA, 0xACFB, 	 0, 	 0, /* 0xB0 */
        0, 0xACFC, 0xACFD, 0xACFE, 0xACFF, 0xAD00, 0xAD01, 0xAD02, /* 0xC0 */
        0xAD03, 0xAD04, 0xAD05, 0xAD06, 0xAD07, 0xAD08, 0xAD09, 0xAD0A, /* 0xC0 */
        0xAD0B, 0xAD0C, 	 0, 0xAD0D, 0xAD0E, 0xAD0F, 0xAD10, 0xAD11, /* 0xD0 */
        0xAD12, 0xAD13, 0xAD14, 0xAD15, 0xAD16, 0xAD17, 	 0, 	 0, /* 0xD0 */
        0, 0xAD18, 0xAD19, 0xAD1A, 0xAD1B, 0xAD1C, 0xAD1D, 0xAD1E, /* 0xE0 */
        0xAD1F, 0xAD20, 0xAD21, 0xAD22, 0xAD23, 0xAD24, 0xAD25, 0xAD26, /* 0xE0 */
        0xAD27, 0xAD28, 	 0, 0xAD29, 0xAD2A, 0xAD2B, 0xAD2C, 0xAD2D, /* 0xF0 */
        0xAD2E, 0xAD2F, 0xAD30, 0xAD31, 0xAD32, 0xAD33					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8a:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xAD34, 0xAD35, 0xAD36, 0xAD37, 0xAD38, 0xAD39, 0xAD3A, /* 0x40 */
        0xAD3B, 0xAD3C, 0xAD3D, 0xAD3E, 0xAD3F, 0xAD40, 0xAD41, 0xAD42, /* 0x40 */
        0xAD43, 0xAD44, 	 0, 0xAD45, 0xAD46, 0xAD47, 0xAD48, 0xAD49, /* 0x50 */
        0xAD4A, 0xAD4B, 0xAD4C, 0xAD4D, 0xAD4E, 0xAD4F, 	 0, 	 0, /* 0x50 */
        0, 0xAD50, 0xAD51, 0xAD52, 0xAD53, 0xAD54, 0xAD55, 0xAD56, /* 0x60 */
        0xAD57, 0xAD58, 0xAD59, 0xAD5A, 0xAD5B, 0xAD5C, 0xAD5D, 0xAD5E, /* 0x60 */
        0xAD5F, 0xAD60, 	 0, 0xAD61, 0xAD62, 0xAD63, 0xAD64, 0xAD65, /* 0x70 */
        0xAD66, 0xAD67, 0xAD68, 0xAD69, 0xAD6A, 0xAD6B, 	 0, 	 0, /* 0x70 */
        0, 0xAD6C, 0xAD6D, 0xAD6E, 0xAD6F, 0xAD70, 0xAD71, 0xAD72, /* 0x80 */
        0xAD73, 0xAD74, 0xAD75, 0xAD76, 0xAD77, 0xAD78, 0xAD79, 0xAD7A, /* 0x80 */
        0xAD7B, 0xAD7C, 	 0, 0xAD7D, 0xAD7E, 0xAD7F, 0xAD80, 0xAD81, /* 0x90 */
        0xAD82, 0xAD83, 0xAD84, 0xAD85, 0xAD86, 0xAD87, 	 0, 	 0, /* 0x90 */
        0, 0xAD88, 0xAD89, 0xAD8A, 0xAD8B, 0xAD8C, 0xAD8D, 0xAD8E, /* 0xA0 */
        0xAD8F, 0xAD90, 0xAD91, 0xAD92, 0xAD93, 0xAD94, 0xAD95, 0xAD96, /* 0xA0 */
        0xAD97, 0xAD98, 	 0, 0xAD99, 0xAD9A, 0xAD9B, 0xAD9C, 0xAD9D, /* 0xB0 */
        0xAD9E, 0xAD9F, 0xADA0, 0xADA1, 0xADA2, 0xADA3, 	 0, 	 0, /* 0xB0 */
        0, 0xADA4, 0xADA5, 0xADA6, 0xADA7, 0xADA8, 0xADA9, 0xADAA, /* 0xC0 */
        0xADAB, 0xADAC, 0xADAD, 0xADAE, 0xADAF, 0xADB0, 0xADB1, 0xADB2, /* 0xC0 */
        0xADB3, 0xADB4, 	 0, 0xADB5, 0xADB6, 0xADB7, 0xADB8, 0xADB9, /* 0xD0 */
        0xADBA, 0xADBB, 0xADBC, 0xADBD, 0xADBE, 0xADBF, 	 0, 	 0, /* 0xD0 */
        0, 0xADC0, 0xADC1, 0xADC2, 0xADC3, 0xADC4, 0xADC5, 0xADC6, /* 0xE0 */
        0xADC7, 0xADC8, 0xADC9, 0xADCA, 0xADCB, 0xADCC, 0xADCD, 0xADCE, /* 0xE0 */
        0xADCF, 0xADD0, 	 0, 0xADD1, 0xADD2, 0xADD3, 0xADD4, 0xADD5, /* 0xF0 */
        0xADD6, 0xADD7, 0xADD8, 0xADD9, 0xADDA, 0xADDB					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8b:
    if (c2>=0x41 && c2<=0xbc) {
      static int const(val[])= {
        0xADDC, 0xADDD, 0xADDE, 0xADDF, 0xADE0, 0xADE1, 0xADE2, /* 0x40 */
        0xADE3, 0xADE4, 0xADE5, 0xADE6, 0xADE7, 0xADE8, 0xADE9, 0xADEA, /* 0x40 */
        0xADEB, 0xADEC, 	 0, 0xADED, 0xADEE, 0xADEF, 0xADF0, 0xADF1, /* 0x50 */
        0xADF2, 0xADF3, 0xADF4, 0xADF5, 0xADF6, 0xADF7, 	 0, 	 0, /* 0x50 */
        0, 0xADF8, 0xADF9, 0xADFA, 0xADFB, 0xADFC, 0xADFD, 0xADFE, /* 0x60 */
        0xADFF, 0xAE00, 0xAE01, 0xAE02, 0xAE03, 0xAE04, 0xAE05, 0xAE06, /* 0x60 */
        0xAE07, 0xAE08, 	 0, 0xAE09, 0xAE0A, 0xAE0B, 0xAE0C, 0xAE0D, /* 0x70 */
        0xAE0E, 0xAE0F, 0xAE10, 0xAE11, 0xAE12, 0xAE13, 	 0, 	 0, /* 0x70 */
        0, 0xAE14, 0xAE15, 0xAE16, 0xAE17, 0xAE18, 0xAE19, 0xAE1A, /* 0x80 */
        0xAE1B, 0xAE1C, 0xAE1D, 0xAE1E, 0xAE1F, 0xAE20, 0xAE21, 0xAE22, /* 0x80 */
        0xAE23, 0xAE24, 	 0, 0xAE25, 0xAE26, 0xAE27, 0xAE28, 0xAE29, /* 0x90 */
        0xAE2A, 0xAE2B, 0xAE2C, 0xAE2D, 0xAE2E, 0xAE2F, 	 0, 	 0, /* 0x90 */
        0, 0xAE30, 0xAE31, 0xAE32, 0xAE33, 0xAE34, 0xAE35, 0xAE36, /* 0xA0 */
        0xAE37, 0xAE38, 0xAE39, 0xAE3A, 0xAE3B, 0xAE3C, 0xAE3D, 0xAE3E, /* 0xA0 */
        0xAE3F, 0xAE40, 	 0, 0xAE41, 0xAE42, 0xAE43, 0xAE44, 0xAE45, /* 0xB0 */
        0xAE46, 0xAE47, 0xAE48, 0xAE49, 0xAE4A, 0xAE4B					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8c:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3132, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xAE4C, 0xAE4D, 0xAE4E, 0xAE4F, 0xAE50, 0xAE51, 0xAE52, /* 0x60 */
        0xAE53, 0xAE54, 0xAE55, 0xAE56, 0xAE57, 0xAE58, 0xAE59, 0xAE5A, /* 0x60 */
        0xAE5B, 0xAE5C, 	 0, 0xAE5D, 0xAE5E, 0xAE5F, 0xAE60, 0xAE61, /* 0x70 */
        0xAE62, 0xAE63, 0xAE64, 0xAE65, 0xAE66, 0xAE67, 	 0, 	 0, /* 0x70 */
        0, 0xAE68, 0xAE69, 0xAE6A, 0xAE6B, 0xAE6C, 0xAE6D, 0xAE6E, /* 0x80 */
        0xAE6F, 0xAE70, 0xAE71, 0xAE72, 0xAE73, 0xAE74, 0xAE75, 0xAE76, /* 0x80 */
        0xAE77, 0xAE78, 	 0, 0xAE79, 0xAE7A, 0xAE7B, 0xAE7C, 0xAE7D, /* 0x90 */
        0xAE7E, 0xAE7F, 0xAE80, 0xAE81, 0xAE82, 0xAE83, 	 0, 	 0, /* 0x90 */
        0, 0xAE84, 0xAE85, 0xAE86, 0xAE87, 0xAE88, 0xAE89, 0xAE8A, /* 0xA0 */
        0xAE8B, 0xAE8C, 0xAE8D, 0xAE8E, 0xAE8F, 0xAE90, 0xAE91, 0xAE92, /* 0xA0 */
        0xAE93, 0xAE94, 	 0, 0xAE95, 0xAE96, 0xAE97, 0xAE98, 0xAE99, /* 0xB0 */
        0xAE9A, 0xAE9B, 0xAE9C, 0xAE9D, 0xAE9E, 0xAE9F, 	 0, 	 0, /* 0xB0 */
        0, 0xAEA0, 0xAEA1, 0xAEA2, 0xAEA3, 0xAEA4, 0xAEA5, 0xAEA6, /* 0xC0 */
        0xAEA7, 0xAEA8, 0xAEA9, 0xAEAA, 0xAEAB, 0xAEAC, 0xAEAD, 0xAEAE, /* 0xC0 */
        0xAEAF, 0xAEB0, 	 0, 0xAEB1, 0xAEB2, 0xAEB3, 0xAEB4, 0xAEB5, /* 0xD0 */
        0xAEB6, 0xAEB7, 0xAEB8, 0xAEB9, 0xAEBA, 0xAEBB, 	 0, 	 0, /* 0xD0 */
        0, 0xAEBC, 0xAEBD, 0xAEBE, 0xAEBF, 0xAEC0, 0xAEC1, 0xAEC2, /* 0xE0 */
        0xAEC3, 0xAEC4, 0xAEC5, 0xAEC6, 0xAEC7, 0xAEC8, 0xAEC9, 0xAECA, /* 0xE0 */
        0xAECB, 0xAECC, 	 0, 0xAECD, 0xAECE, 0xAECF, 0xAED0, 0xAED1, /* 0xF0 */
        0xAED2, 0xAED3, 0xAED4, 0xAED5, 0xAED6, 0xAED7					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8d:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xAED8, 0xAED9, 0xAEDA, 0xAEDB, 0xAEDC, 0xAEDD, 0xAEDE, /* 0x40 */
        0xAEDF, 0xAEE0, 0xAEE1, 0xAEE2, 0xAEE3, 0xAEE4, 0xAEE5, 0xAEE6, /* 0x40 */
        0xAEE7, 0xAEE8, 	 0, 0xAEE9, 0xAEEA, 0xAEEB, 0xAEEC, 0xAEED, /* 0x50 */
        0xAEEE, 0xAEEF, 0xAEF0, 0xAEF1, 0xAEF2, 0xAEF3, 	 0, 	 0, /* 0x50 */
        0, 0xAEF4, 0xAEF5, 0xAEF6, 0xAEF7, 0xAEF8, 0xAEF9, 0xAEFA, /* 0x60 */
        0xAEFB, 0xAEFC, 0xAEFD, 0xAEFE, 0xAEFF, 0xAF00, 0xAF01, 0xAF02, /* 0x60 */
        0xAF03, 0xAF04, 	 0, 0xAF05, 0xAF06, 0xAF07, 0xAF08, 0xAF09, /* 0x70 */
        0xAF0A, 0xAF0B, 0xAF0C, 0xAF0D, 0xAF0E, 0xAF0F, 	 0, 	 0, /* 0x70 */
        0, 0xAF10, 0xAF11, 0xAF12, 0xAF13, 0xAF14, 0xAF15, 0xAF16, /* 0x80 */
        0xAF17, 0xAF18, 0xAF19, 0xAF1A, 0xAF1B, 0xAF1C, 0xAF1D, 0xAF1E, /* 0x80 */
        0xAF1F, 0xAF20, 	 0, 0xAF21, 0xAF22, 0xAF23, 0xAF24, 0xAF25, /* 0x90 */
        0xAF26, 0xAF27, 0xAF28, 0xAF29, 0xAF2A, 0xAF2B, 	 0, 	 0, /* 0x90 */
        0, 0xAF2C, 0xAF2D, 0xAF2E, 0xAF2F, 0xAF30, 0xAF31, 0xAF32, /* 0xA0 */
        0xAF33, 0xAF34, 0xAF35, 0xAF36, 0xAF37, 0xAF38, 0xAF39, 0xAF3A, /* 0xA0 */
        0xAF3B, 0xAF3C, 	 0, 0xAF3D, 0xAF3E, 0xAF3F, 0xAF40, 0xAF41, /* 0xB0 */
        0xAF42, 0xAF43, 0xAF44, 0xAF45, 0xAF46, 0xAF47, 	 0, 	 0, /* 0xB0 */
        0, 0xAF48, 0xAF49, 0xAF4A, 0xAF4B, 0xAF4C, 0xAF4D, 0xAF4E, /* 0xC0 */
        0xAF4F, 0xAF50, 0xAF51, 0xAF52, 0xAF53, 0xAF54, 0xAF55, 0xAF56, /* 0xC0 */
        0xAF57, 0xAF58, 	 0, 0xAF59, 0xAF5A, 0xAF5B, 0xAF5C, 0xAF5D, /* 0xD0 */
        0xAF5E, 0xAF5F, 0xAF60, 0xAF61, 0xAF62, 0xAF63, 	 0, 	 0, /* 0xD0 */
        0, 0xAF64, 0xAF65, 0xAF66, 0xAF67, 0xAF68, 0xAF69, 0xAF6A, /* 0xE0 */
        0xAF6B, 0xAF6C, 0xAF6D, 0xAF6E, 0xAF6F, 0xAF70, 0xAF71, 0xAF72, /* 0xE0 */
        0xAF73, 0xAF74, 	 0, 0xAF75, 0xAF76, 0xAF77, 0xAF78, 0xAF79, /* 0xF0 */
        0xAF7A, 0xAF7B, 0xAF7C, 0xAF7D, 0xAF7E, 0xAF7F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8e:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xAF80, 0xAF81, 0xAF82, 0xAF83, 0xAF84, 0xAF85, 0xAF86, /* 0x40 */
        0xAF87, 0xAF88, 0xAF89, 0xAF8A, 0xAF8B, 0xAF8C, 0xAF8D, 0xAF8E, /* 0x40 */
        0xAF8F, 0xAF90, 	 0, 0xAF91, 0xAF92, 0xAF93, 0xAF94, 0xAF95, /* 0x50 */
        0xAF96, 0xAF97, 0xAF98, 0xAF99, 0xAF9A, 0xAF9B, 	 0, 	 0, /* 0x50 */
        0, 0xAF9C, 0xAF9D, 0xAF9E, 0xAF9F, 0xAFA0, 0xAFA1, 0xAFA2, /* 0x60 */
        0xAFA3, 0xAFA4, 0xAFA5, 0xAFA6, 0xAFA7, 0xAFA8, 0xAFA9, 0xAFAA, /* 0x60 */
        0xAFAB, 0xAFAC, 	 0, 0xAFAD, 0xAFAE, 0xAFAF, 0xAFB0, 0xAFB1, /* 0x70 */
        0xAFB2, 0xAFB3, 0xAFB4, 0xAFB5, 0xAFB6, 0xAFB7, 	 0, 	 0, /* 0x70 */
        0, 0xAFB8, 0xAFB9, 0xAFBA, 0xAFBB, 0xAFBC, 0xAFBD, 0xAFBE, /* 0x80 */
        0xAFBF, 0xAFC0, 0xAFC1, 0xAFC2, 0xAFC3, 0xAFC4, 0xAFC5, 0xAFC6, /* 0x80 */
        0xAFC7, 0xAFC8, 	 0, 0xAFC9, 0xAFCA, 0xAFCB, 0xAFCC, 0xAFCD, /* 0x90 */
        0xAFCE, 0xAFCF, 0xAFD0, 0xAFD1, 0xAFD2, 0xAFD3, 	 0, 	 0, /* 0x90 */
        0, 0xAFD4, 0xAFD5, 0xAFD6, 0xAFD7, 0xAFD8, 0xAFD9, 0xAFDA, /* 0xA0 */
        0xAFDB, 0xAFDC, 0xAFDD, 0xAFDE, 0xAFDF, 0xAFE0, 0xAFE1, 0xAFE2, /* 0xA0 */
        0xAFE3, 0xAFE4, 	 0, 0xAFE5, 0xAFE6, 0xAFE7, 0xAFE8, 0xAFE9, /* 0xB0 */
        0xAFEA, 0xAFEB, 0xAFEC, 0xAFED, 0xAFEE, 0xAFEF, 	 0, 	 0, /* 0xB0 */
        0, 0xAFF0, 0xAFF1, 0xAFF2, 0xAFF3, 0xAFF4, 0xAFF5, 0xAFF6, /* 0xC0 */
        0xAFF7, 0xAFF8, 0xAFF9, 0xAFFA, 0xAFFB, 0xAFFC, 0xAFFD, 0xAFFE, /* 0xC0 */
        0xAFFF, 0xB000, 	 0, 0xB001, 0xB002, 0xB003, 0xB004, 0xB005, /* 0xD0 */
        0xB006, 0xB007, 0xB008, 0xB009, 0xB00A, 0xB00B, 	 0, 	 0, /* 0xD0 */
        0, 0xB00C, 0xB00D, 0xB00E, 0xB00F, 0xB010, 0xB011, 0xB012, /* 0xE0 */
        0xB013, 0xB014, 0xB015, 0xB016, 0xB017, 0xB018, 0xB019, 0xB01A, /* 0xE0 */
        0xB01B, 0xB01C, 	 0, 0xB01D, 0xB01E, 0xB01F, 0xB020, 0xB021, /* 0xF0 */
        0xB022, 0xB023, 0xB024, 0xB025, 0xB026, 0xB027					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x8f:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xB028, 0xB029, 0xB02A, 0xB02B, 0xB02C, 0xB02D, 0xB02E, /* 0x40 */
        0xB02F, 0xB030, 0xB031, 0xB032, 0xB033, 0xB034, 0xB035, 0xB036, /* 0x40 */
        0xB037, 0xB038, 	 0, 0xB039, 0xB03A, 0xB03B, 0xB03C, 0xB03D, /* 0x50 */
        0xB03E, 0xB03F, 0xB040, 0xB041, 0xB042, 0xB043, 	 0, 	 0, /* 0x50 */
        0, 0xB044, 0xB045, 0xB046, 0xB047, 0xB048, 0xB049, 0xB04A, /* 0x60 */
        0xB04B, 0xB04C, 0xB04D, 0xB04E, 0xB04F, 0xB050, 0xB051, 0xB052, /* 0x60 */
        0xB053, 0xB054, 	 0, 0xB055, 0xB056, 0xB057, 0xB058, 0xB059, /* 0x70 */
        0xB05A, 0xB05B, 0xB05C, 0xB05D, 0xB05E, 0xB05F, 	 0, 	 0, /* 0x70 */
        0, 0xB060, 0xB061, 0xB062, 0xB063, 0xB064, 0xB065, 0xB066, /* 0x80 */
        0xB067, 0xB068, 0xB069, 0xB06A, 0xB06B, 0xB06C, 0xB06D, 0xB06E, /* 0x80 */
        0xB06F, 0xB070, 	 0, 0xB071, 0xB072, 0xB073, 0xB074, 0xB075, /* 0x90 */
        0xB076, 0xB077, 0xB078, 0xB079, 0xB07A, 0xB07B, 	 0, 	 0, /* 0x90 */
        0, 0xB07C, 0xB07D, 0xB07E, 0xB07F, 0xB080, 0xB081, 0xB082, /* 0xA0 */
        0xB083, 0xB084, 0xB085, 0xB086, 0xB087, 0xB088, 0xB089, 0xB08A, /* 0xA0 */
        0xB08B, 0xB08C, 	 0, 0xB08D, 0xB08E, 0xB08F, 0xB090, 0xB091, /* 0xB0 */
        0xB092, 0xB093, 0xB094, 0xB095, 0xB096, 0xB097					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x90:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3134, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xB098, 0xB099, 0xB09A, 0xB09B, 0xB09C, 0xB09D, 0xB09E, /* 0x60 */
        0xB09F, 0xB0A0, 0xB0A1, 0xB0A2, 0xB0A3, 0xB0A4, 0xB0A5, 0xB0A6, /* 0x60 */
        0xB0A7, 0xB0A8, 	 0, 0xB0A9, 0xB0AA, 0xB0AB, 0xB0AC, 0xB0AD, /* 0x70 */
        0xB0AE, 0xB0AF, 0xB0B0, 0xB0B1, 0xB0B2, 0xB0B3, 	 0, 	 0, /* 0x70 */
        0, 0xB0B4, 0xB0B5, 0xB0B6, 0xB0B7, 0xB0B8, 0xB0B9, 0xB0BA, /* 0x80 */
        0xB0BB, 0xB0BC, 0xB0BD, 0xB0BE, 0xB0BF, 0xB0C0, 0xB0C1, 0xB0C2, /* 0x80 */
        0xB0C3, 0xB0C4, 	 0, 0xB0C5, 0xB0C6, 0xB0C7, 0xB0C8, 0xB0C9, /* 0x90 */
        0xB0CA, 0xB0CB, 0xB0CC, 0xB0CD, 0xB0CE, 0xB0CF, 	 0, 	 0, /* 0x90 */
        0, 0xB0D0, 0xB0D1, 0xB0D2, 0xB0D3, 0xB0D4, 0xB0D5, 0xB0D6, /* 0xA0 */
        0xB0D7, 0xB0D8, 0xB0D9, 0xB0DA, 0xB0DB, 0xB0DC, 0xB0DD, 0xB0DE, /* 0xA0 */
        0xB0DF, 0xB0E0, 	 0, 0xB0E1, 0xB0E2, 0xB0E3, 0xB0E4, 0xB0E5, /* 0xB0 */
        0xB0E6, 0xB0E7, 0xB0E8, 0xB0E9, 0xB0EA, 0xB0EB, 	 0, 	 0, /* 0xB0 */
        0, 0xB0EC, 0xB0ED, 0xB0EE, 0xB0EF, 0xB0F0, 0xB0F1, 0xB0F2, /* 0xC0 */
        0xB0F3, 0xB0F4, 0xB0F5, 0xB0F6, 0xB0F7, 0xB0F8, 0xB0F9, 0xB0FA, /* 0xC0 */
        0xB0FB, 0xB0FC, 	 0, 0xB0FD, 0xB0FE, 0xB0FF, 0xB100, 0xB101, /* 0xD0 */
        0xB102, 0xB103, 0xB104, 0xB105, 0xB106, 0xB107, 	 0, 	 0, /* 0xD0 */
        0, 0xB108, 0xB109, 0xB10A, 0xB10B, 0xB10C, 0xB10D, 0xB10E, /* 0xE0 */
        0xB10F, 0xB110, 0xB111, 0xB112, 0xB113, 0xB114, 0xB115, 0xB116, /* 0xE0 */
        0xB117, 0xB118, 	 0, 0xB119, 0xB11A, 0xB11B, 0xB11C, 0xB11D, /* 0xF0 */
        0xB11E, 0xB11F, 0xB120, 0xB121, 0xB122, 0xB123					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x91:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB124, 0xB125, 0xB126, 0xB127, 0xB128, 0xB129, 0xB12A, /* 0x40 */
        0xB12B, 0xB12C, 0xB12D, 0xB12E, 0xB12F, 0xB130, 0xB131, 0xB132, /* 0x40 */
        0xB133, 0xB134, 	 0, 0xB135, 0xB136, 0xB137, 0xB138, 0xB139, /* 0x50 */
        0xB13A, 0xB13B, 0xB13C, 0xB13D, 0xB13E, 0xB13F, 	 0, 	 0, /* 0x50 */
        0, 0xB140, 0xB141, 0xB142, 0xB143, 0xB144, 0xB145, 0xB146, /* 0x60 */
        0xB147, 0xB148, 0xB149, 0xB14A, 0xB14B, 0xB14C, 0xB14D, 0xB14E, /* 0x60 */
        0xB14F, 0xB150, 	 0, 0xB151, 0xB152, 0xB153, 0xB154, 0xB155, /* 0x70 */
        0xB156, 0xB157, 0xB158, 0xB159, 0xB15A, 0xB15B, 	 0, 	 0, /* 0x70 */
        0, 0xB15C, 0xB15D, 0xB15E, 0xB15F, 0xB160, 0xB161, 0xB162, /* 0x80 */
        0xB163, 0xB164, 0xB165, 0xB166, 0xB167, 0xB168, 0xB169, 0xB16A, /* 0x80 */
        0xB16B, 0xB16C, 	 0, 0xB16D, 0xB16E, 0xB16F, 0xB170, 0xB171, /* 0x90 */
        0xB172, 0xB173, 0xB174, 0xB175, 0xB176, 0xB177, 	 0, 	 0, /* 0x90 */
        0, 0xB178, 0xB179, 0xB17A, 0xB17B, 0xB17C, 0xB17D, 0xB17E, /* 0xA0 */
        0xB17F, 0xB180, 0xB181, 0xB182, 0xB183, 0xB184, 0xB185, 0xB186, /* 0xA0 */
        0xB187, 0xB188, 	 0, 0xB189, 0xB18A, 0xB18B, 0xB18C, 0xB18D, /* 0xB0 */
        0xB18E, 0xB18F, 0xB190, 0xB191, 0xB192, 0xB193, 	 0, 	 0, /* 0xB0 */
        0, 0xB194, 0xB195, 0xB196, 0xB197, 0xB198, 0xB199, 0xB19A, /* 0xC0 */
        0xB19B, 0xB19C, 0xB19D, 0xB19E, 0xB19F, 0xB1A0, 0xB1A1, 0xB1A2, /* 0xC0 */
        0xB1A3, 0xB1A4, 	 0, 0xB1A5, 0xB1A6, 0xB1A7, 0xB1A8, 0xB1A9, /* 0xD0 */
        0xB1AA, 0xB1AB, 0xB1AC, 0xB1AD, 0xB1AE, 0xB1AF, 	 0, 	 0, /* 0xD0 */
        0, 0xB1B0, 0xB1B1, 0xB1B2, 0xB1B3, 0xB1B4, 0xB1B5, 0xB1B6, /* 0xE0 */
        0xB1B7, 0xB1B8, 0xB1B9, 0xB1BA, 0xB1BB, 0xB1BC, 0xB1BD, 0xB1BE, /* 0xE0 */
        0xB1BF, 0xB1C0, 	 0, 0xB1C1, 0xB1C2, 0xB1C3, 0xB1C4, 0xB1C5, /* 0xF0 */
        0xB1C6, 0xB1C7, 0xB1C8, 0xB1C9, 0xB1CA, 0xB1CB					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x92:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB1CC, 0xB1CD, 0xB1CE, 0xB1CF, 0xB1D0, 0xB1D1, 0xB1D2, /* 0x40 */
        0xB1D3, 0xB1D4, 0xB1D5, 0xB1D6, 0xB1D7, 0xB1D8, 0xB1D9, 0xB1DA, /* 0x40 */
        0xB1DB, 0xB1DC, 	 0, 0xB1DD, 0xB1DE, 0xB1DF, 0xB1E0, 0xB1E1, /* 0x50 */
        0xB1E2, 0xB1E3, 0xB1E4, 0xB1E5, 0xB1E6, 0xB1E7, 	 0, 	 0, /* 0x50 */
        0, 0xB1E8, 0xB1E9, 0xB1EA, 0xB1EB, 0xB1EC, 0xB1ED, 0xB1EE, /* 0x60 */
        0xB1EF, 0xB1F0, 0xB1F1, 0xB1F2, 0xB1F3, 0xB1F4, 0xB1F5, 0xB1F6, /* 0x60 */
        0xB1F7, 0xB1F8, 	 0, 0xB1F9, 0xB1FA, 0xB1FB, 0xB1FC, 0xB1FD, /* 0x70 */
        0xB1FE, 0xB1FF, 0xB200, 0xB201, 0xB202, 0xB203, 	 0, 	 0, /* 0x70 */
        0, 0xB204, 0xB205, 0xB206, 0xB207, 0xB208, 0xB209, 0xB20A, /* 0x80 */
        0xB20B, 0xB20C, 0xB20D, 0xB20E, 0xB20F, 0xB210, 0xB211, 0xB212, /* 0x80 */
        0xB213, 0xB214, 	 0, 0xB215, 0xB216, 0xB217, 0xB218, 0xB219, /* 0x90 */
        0xB21A, 0xB21B, 0xB21C, 0xB21D, 0xB21E, 0xB21F, 	 0, 	 0, /* 0x90 */
        0, 0xB220, 0xB221, 0xB222, 0xB223, 0xB224, 0xB225, 0xB226, /* 0xA0 */
        0xB227, 0xB228, 0xB229, 0xB22A, 0xB22B, 0xB22C, 0xB22D, 0xB22E, /* 0xA0 */
        0xB22F, 0xB230, 	 0, 0xB231, 0xB232, 0xB233, 0xB234, 0xB235, /* 0xB0 */
        0xB236, 0xB237, 0xB238, 0xB239, 0xB23A, 0xB23B, 	 0, 	 0, /* 0xB0 */
        0, 0xB23C, 0xB23D, 0xB23E, 0xB23F, 0xB240, 0xB241, 0xB242, /* 0xC0 */
        0xB243, 0xB244, 0xB245, 0xB246, 0xB247, 0xB248, 0xB249, 0xB24A, /* 0xC0 */
        0xB24B, 0xB24C, 	 0, 0xB24D, 0xB24E, 0xB24F, 0xB250, 0xB251, /* 0xD0 */
        0xB252, 0xB253, 0xB254, 0xB255, 0xB256, 0xB257, 	 0, 	 0, /* 0xD0 */
        0, 0xB258, 0xB259, 0xB25A, 0xB25B, 0xB25C, 0xB25D, 0xB25E, /* 0xE0 */
        0xB25F, 0xB260, 0xB261, 0xB262, 0xB263, 0xB264, 0xB265, 0xB266, /* 0xE0 */
        0xB267, 0xB268, 	 0, 0xB269, 0xB26A, 0xB26B, 0xB26C, 0xB26D, /* 0xF0 */
        0xB26E, 0xB26F, 0xB270, 0xB271, 0xB272, 0xB273					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x93:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xB274, 0xB275, 0xB276, 0xB277, 0xB278, 0xB279, 0xB27A, /* 0x40 */
        0xB27B, 0xB27C, 0xB27D, 0xB27E, 0xB27F, 0xB280, 0xB281, 0xB282, /* 0x40 */
        0xB283, 0xB284, 	 0, 0xB285, 0xB286, 0xB287, 0xB288, 0xB289, /* 0x50 */
        0xB28A, 0xB28B, 0xB28C, 0xB28D, 0xB28E, 0xB28F, 	 0, 	 0, /* 0x50 */
        0, 0xB290, 0xB291, 0xB292, 0xB293, 0xB294, 0xB295, 0xB296, /* 0x60 */
        0xB297, 0xB298, 0xB299, 0xB29A, 0xB29B, 0xB29C, 0xB29D, 0xB29E, /* 0x60 */
        0xB29F, 0xB2A0, 	 0, 0xB2A1, 0xB2A2, 0xB2A3, 0xB2A4, 0xB2A5, /* 0x70 */
        0xB2A6, 0xB2A7, 0xB2A8, 0xB2A9, 0xB2AA, 0xB2AB, 	 0, 	 0, /* 0x70 */
        0, 0xB2AC, 0xB2AD, 0xB2AE, 0xB2AF, 0xB2B0, 0xB2B1, 0xB2B2, /* 0x80 */
        0xB2B3, 0xB2B4, 0xB2B5, 0xB2B6, 0xB2B7, 0xB2B8, 0xB2B9, 0xB2BA, /* 0x80 */
        0xB2BB, 0xB2BC, 	 0, 0xB2BD, 0xB2BE, 0xB2BF, 0xB2C0, 0xB2C1, /* 0x90 */
        0xB2C2, 0xB2C3, 0xB2C4, 0xB2C5, 0xB2C6, 0xB2C7, 	 0, 	 0, /* 0x90 */
        0, 0xB2C8, 0xB2C9, 0xB2CA, 0xB2CB, 0xB2CC, 0xB2CD, 0xB2CE, /* 0xA0 */
        0xB2CF, 0xB2D0, 0xB2D1, 0xB2D2, 0xB2D3, 0xB2D4, 0xB2D5, 0xB2D6, /* 0xA0 */
        0xB2D7, 0xB2D8, 	 0, 0xB2D9, 0xB2DA, 0xB2DB, 0xB2DC, 0xB2DD, /* 0xB0 */
        0xB2DE, 0xB2DF, 0xB2E0, 0xB2E1, 0xB2E2, 0xB2E3					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x94:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3137, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xB2E4, 0xB2E5, 0xB2E6, 0xB2E7, 0xB2E8, 0xB2E9, 0xB2EA, /* 0x60 */
        0xB2EB, 0xB2EC, 0xB2ED, 0xB2EE, 0xB2EF, 0xB2F0, 0xB2F1, 0xB2F2, /* 0x60 */
        0xB2F3, 0xB2F4, 	 0, 0xB2F5, 0xB2F6, 0xB2F7, 0xB2F8, 0xB2F9, /* 0x70 */
        0xB2FA, 0xB2FB, 0xB2FC, 0xB2FD, 0xB2FE, 0xB2FF, 	 0, 	 0, /* 0x70 */
        0, 0xB300, 0xB301, 0xB302, 0xB303, 0xB304, 0xB305, 0xB306, /* 0x80 */
        0xB307, 0xB308, 0xB309, 0xB30A, 0xB30B, 0xB30C, 0xB30D, 0xB30E, /* 0x80 */
        0xB30F, 0xB310, 	 0, 0xB311, 0xB312, 0xB313, 0xB314, 0xB315, /* 0x90 */
        0xB316, 0xB317, 0xB318, 0xB319, 0xB31A, 0xB31B, 	 0, 	 0, /* 0x90 */
        0, 0xB31C, 0xB31D, 0xB31E, 0xB31F, 0xB320, 0xB321, 0xB322, /* 0xA0 */
        0xB323, 0xB324, 0xB325, 0xB326, 0xB327, 0xB328, 0xB329, 0xB32A, /* 0xA0 */
        0xB32B, 0xB32C, 	 0, 0xB32D, 0xB32E, 0xB32F, 0xB330, 0xB331, /* 0xB0 */
        0xB332, 0xB333, 0xB334, 0xB335, 0xB336, 0xB337, 	 0, 	 0, /* 0xB0 */
        0, 0xB338, 0xB339, 0xB33A, 0xB33B, 0xB33C, 0xB33D, 0xB33E, /* 0xC0 */
        0xB33F, 0xB340, 0xB341, 0xB342, 0xB343, 0xB344, 0xB345, 0xB346, /* 0xC0 */
        0xB347, 0xB348, 	 0, 0xB349, 0xB34A, 0xB34B, 0xB34C, 0xB34D, /* 0xD0 */
        0xB34E, 0xB34F, 0xB350, 0xB351, 0xB352, 0xB353, 	 0, 	 0, /* 0xD0 */
        0, 0xB354, 0xB355, 0xB356, 0xB357, 0xB358, 0xB359, 0xB35A, /* 0xE0 */
        0xB35B, 0xB35C, 0xB35D, 0xB35E, 0xB35F, 0xB360, 0xB361, 0xB362, /* 0xE0 */
        0xB363, 0xB364, 	 0, 0xB365, 0xB366, 0xB367, 0xB368, 0xB369, /* 0xF0 */
        0xB36A, 0xB36B, 0xB36C, 0xB36D, 0xB36E, 0xB36F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x95:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB370, 0xB371, 0xB372, 0xB373, 0xB374, 0xB375, 0xB376, /* 0x40 */
        0xB377, 0xB378, 0xB379, 0xB37A, 0xB37B, 0xB37C, 0xB37D, 0xB37E, /* 0x40 */
        0xB37F, 0xB380, 	 0, 0xB381, 0xB382, 0xB383, 0xB384, 0xB385, /* 0x50 */
        0xB386, 0xB387, 0xB388, 0xB389, 0xB38A, 0xB38B, 	 0, 	 0, /* 0x50 */
        0, 0xB38C, 0xB38D, 0xB38E, 0xB38F, 0xB390, 0xB391, 0xB392, /* 0x60 */
        0xB393, 0xB394, 0xB395, 0xB396, 0xB397, 0xB398, 0xB399, 0xB39A, /* 0x60 */
        0xB39B, 0xB39C, 	 0, 0xB39D, 0xB39E, 0xB39F, 0xB3A0, 0xB3A1, /* 0x70 */
        0xB3A2, 0xB3A3, 0xB3A4, 0xB3A5, 0xB3A6, 0xB3A7, 	 0, 	 0, /* 0x70 */
        0, 0xB3A8, 0xB3A9, 0xB3AA, 0xB3AB, 0xB3AC, 0xB3AD, 0xB3AE, /* 0x80 */
        0xB3AF, 0xB3B0, 0xB3B1, 0xB3B2, 0xB3B3, 0xB3B4, 0xB3B5, 0xB3B6, /* 0x80 */
        0xB3B7, 0xB3B8, 	 0, 0xB3B9, 0xB3BA, 0xB3BB, 0xB3BC, 0xB3BD, /* 0x90 */
        0xB3BE, 0xB3BF, 0xB3C0, 0xB3C1, 0xB3C2, 0xB3C3, 	 0, 	 0, /* 0x90 */
        0, 0xB3C4, 0xB3C5, 0xB3C6, 0xB3C7, 0xB3C8, 0xB3C9, 0xB3CA, /* 0xA0 */
        0xB3CB, 0xB3CC, 0xB3CD, 0xB3CE, 0xB3CF, 0xB3D0, 0xB3D1, 0xB3D2, /* 0xA0 */
        0xB3D3, 0xB3D4, 	 0, 0xB3D5, 0xB3D6, 0xB3D7, 0xB3D8, 0xB3D9, /* 0xB0 */
        0xB3DA, 0xB3DB, 0xB3DC, 0xB3DD, 0xB3DE, 0xB3DF, 	 0, 	 0, /* 0xB0 */
        0, 0xB3E0, 0xB3E1, 0xB3E2, 0xB3E3, 0xB3E4, 0xB3E5, 0xB3E6, /* 0xC0 */
        0xB3E7, 0xB3E8, 0xB3E9, 0xB3EA, 0xB3EB, 0xB3EC, 0xB3ED, 0xB3EE, /* 0xC0 */
        0xB3EF, 0xB3F0, 	 0, 0xB3F1, 0xB3F2, 0xB3F3, 0xB3F4, 0xB3F5, /* 0xD0 */
        0xB3F6, 0xB3F7, 0xB3F8, 0xB3F9, 0xB3FA, 0xB3FB, 	 0, 	 0, /* 0xD0 */
        0, 0xB3FC, 0xB3FD, 0xB3FE, 0xB3FF, 0xB400, 0xB401, 0xB402, /* 0xE0 */
        0xB403, 0xB404, 0xB405, 0xB406, 0xB407, 0xB408, 0xB409, 0xB40A, /* 0xE0 */
        0xB40B, 0xB40C, 	 0, 0xB40D, 0xB40E, 0xB40F, 0xB410, 0xB411, /* 0xF0 */
        0xB412, 0xB413, 0xB414, 0xB415, 0xB416, 0xB417					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x96:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB418, 0xB419, 0xB41A, 0xB41B, 0xB41C, 0xB41D, 0xB41E, /* 0x40 */
        0xB41F, 0xB420, 0xB421, 0xB422, 0xB423, 0xB424, 0xB425, 0xB426, /* 0x40 */
        0xB427, 0xB428, 	 0, 0xB429, 0xB42A, 0xB42B, 0xB42C, 0xB42D, /* 0x50 */
        0xB42E, 0xB42F, 0xB430, 0xB431, 0xB432, 0xB433, 	 0, 	 0, /* 0x50 */
        0, 0xB434, 0xB435, 0xB436, 0xB437, 0xB438, 0xB439, 0xB43A, /* 0x60 */
        0xB43B, 0xB43C, 0xB43D, 0xB43E, 0xB43F, 0xB440, 0xB441, 0xB442, /* 0x60 */
        0xB443, 0xB444, 	 0, 0xB445, 0xB446, 0xB447, 0xB448, 0xB449, /* 0x70 */
        0xB44A, 0xB44B, 0xB44C, 0xB44D, 0xB44E, 0xB44F, 	 0, 	 0, /* 0x70 */
        0, 0xB450, 0xB451, 0xB452, 0xB453, 0xB454, 0xB455, 0xB456, /* 0x80 */
        0xB457, 0xB458, 0xB459, 0xB45A, 0xB45B, 0xB45C, 0xB45D, 0xB45E, /* 0x80 */
        0xB45F, 0xB460, 	 0, 0xB461, 0xB462, 0xB463, 0xB464, 0xB465, /* 0x90 */
        0xB466, 0xB467, 0xB468, 0xB469, 0xB46A, 0xB46B, 	 0, 	 0, /* 0x90 */
        0, 0xB46C, 0xB46D, 0xB46E, 0xB46F, 0xB470, 0xB471, 0xB472, /* 0xA0 */
        0xB473, 0xB474, 0xB475, 0xB476, 0xB477, 0xB478, 0xB479, 0xB47A, /* 0xA0 */
        0xB47B, 0xB47C, 	 0, 0xB47D, 0xB47E, 0xB47F, 0xB480, 0xB481, /* 0xB0 */
        0xB482, 0xB483, 0xB484, 0xB485, 0xB486, 0xB487, 	 0, 	 0, /* 0xB0 */
        0, 0xB488, 0xB489, 0xB48A, 0xB48B, 0xB48C, 0xB48D, 0xB48E, /* 0xC0 */
        0xB48F, 0xB490, 0xB491, 0xB492, 0xB493, 0xB494, 0xB495, 0xB496, /* 0xC0 */
        0xB497, 0xB498, 	 0, 0xB499, 0xB49A, 0xB49B, 0xB49C, 0xB49D, /* 0xD0 */
        0xB49E, 0xB49F, 0xB4A0, 0xB4A1, 0xB4A2, 0xB4A3, 	 0, 	 0, /* 0xD0 */
        0, 0xB4A4, 0xB4A5, 0xB4A6, 0xB4A7, 0xB4A8, 0xB4A9, 0xB4AA, /* 0xE0 */
        0xB4AB, 0xB4AC, 0xB4AD, 0xB4AE, 0xB4AF, 0xB4B0, 0xB4B1, 0xB4B2, /* 0xE0 */
        0xB4B3, 0xB4B4, 	 0, 0xB4B5, 0xB4B6, 0xB4B7, 0xB4B8, 0xB4B9, /* 0xF0 */
        0xB4BA, 0xB4BB, 0xB4BC, 0xB4BD, 0xB4BE, 0xB4BF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x97:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xB4C0, 0xB4C1, 0xB4C2, 0xB4C3, 0xB4C4, 0xB4C5, 0xB4C6, /* 0x40 */
        0xB4C7, 0xB4C8, 0xB4C9, 0xB4CA, 0xB4CB, 0xB4CC, 0xB4CD, 0xB4CE, /* 0x40 */
        0xB4CF, 0xB4D0, 	 0, 0xB4D1, 0xB4D2, 0xB4D3, 0xB4D4, 0xB4D5, /* 0x50 */
        0xB4D6, 0xB4D7, 0xB4D8, 0xB4D9, 0xB4DA, 0xB4DB, 	 0, 	 0, /* 0x50 */
        0, 0xB4DC, 0xB4DD, 0xB4DE, 0xB4DF, 0xB4E0, 0xB4E1, 0xB4E2, /* 0x60 */
        0xB4E3, 0xB4E4, 0xB4E5, 0xB4E6, 0xB4E7, 0xB4E8, 0xB4E9, 0xB4EA, /* 0x60 */
        0xB4EB, 0xB4EC, 	 0, 0xB4ED, 0xB4EE, 0xB4EF, 0xB4F0, 0xB4F1, /* 0x70 */
        0xB4F2, 0xB4F3, 0xB4F4, 0xB4F5, 0xB4F6, 0xB4F7, 	 0, 	 0, /* 0x70 */
        0, 0xB4F8, 0xB4F9, 0xB4FA, 0xB4FB, 0xB4FC, 0xB4FD, 0xB4FE, /* 0x80 */
        0xB4FF, 0xB500, 0xB501, 0xB502, 0xB503, 0xB504, 0xB505, 0xB506, /* 0x80 */
        0xB507, 0xB508, 	 0, 0xB509, 0xB50A, 0xB50B, 0xB50C, 0xB50D, /* 0x90 */
        0xB50E, 0xB50F, 0xB510, 0xB511, 0xB512, 0xB513, 	 0, 	 0, /* 0x90 */
        0, 0xB514, 0xB515, 0xB516, 0xB517, 0xB518, 0xB519, 0xB51A, /* 0xA0 */
        0xB51B, 0xB51C, 0xB51D, 0xB51E, 0xB51F, 0xB520, 0xB521, 0xB522, /* 0xA0 */
        0xB523, 0xB524, 	 0, 0xB525, 0xB526, 0xB527, 0xB528, 0xB529, /* 0xB0 */
        0xB52A, 0xB52B, 0xB52C, 0xB52D, 0xB52E, 0xB52F					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x98:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3138, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xB530, 0xB531, 0xB532, 0xB533, 0xB534, 0xB535, 0xB536, /* 0x60 */
        0xB537, 0xB538, 0xB539, 0xB53A, 0xB53B, 0xB53C, 0xB53D, 0xB53E, /* 0x60 */
        0xB53F, 0xB540, 	 0, 0xB541, 0xB542, 0xB543, 0xB544, 0xB545, /* 0x70 */
        0xB546, 0xB547, 0xB548, 0xB549, 0xB54A, 0xB54B, 	 0, 	 0, /* 0x70 */
        0, 0xB54C, 0xB54D, 0xB54E, 0xB54F, 0xB550, 0xB551, 0xB552, /* 0x80 */
        0xB553, 0xB554, 0xB555, 0xB556, 0xB557, 0xB558, 0xB559, 0xB55A, /* 0x80 */
        0xB55B, 0xB55C, 	 0, 0xB55D, 0xB55E, 0xB55F, 0xB560, 0xB561, /* 0x90 */
        0xB562, 0xB563, 0xB564, 0xB565, 0xB566, 0xB567, 	 0, 	 0, /* 0x90 */
        0, 0xB568, 0xB569, 0xB56A, 0xB56B, 0xB56C, 0xB56D, 0xB56E, /* 0xA0 */
        0xB56F, 0xB570, 0xB571, 0xB572, 0xB573, 0xB574, 0xB575, 0xB576, /* 0xA0 */
        0xB577, 0xB578, 	 0, 0xB579, 0xB57A, 0xB57B, 0xB57C, 0xB57D, /* 0xB0 */
        0xB57E, 0xB57F, 0xB580, 0xB581, 0xB582, 0xB583, 	 0, 	 0, /* 0xB0 */
        0, 0xB584, 0xB585, 0xB586, 0xB587, 0xB588, 0xB589, 0xB58A, /* 0xC0 */
        0xB58B, 0xB58C, 0xB58D, 0xB58E, 0xB58F, 0xB590, 0xB591, 0xB592, /* 0xC0 */
        0xB593, 0xB594, 	 0, 0xB595, 0xB596, 0xB597, 0xB598, 0xB599, /* 0xD0 */
        0xB59A, 0xB59B, 0xB59C, 0xB59D, 0xB59E, 0xB59F, 	 0, 	 0, /* 0xD0 */
        0, 0xB5A0, 0xB5A1, 0xB5A2, 0xB5A3, 0xB5A4, 0xB5A5, 0xB5A6, /* 0xE0 */
        0xB5A7, 0xB5A8, 0xB5A9, 0xB5AA, 0xB5AB, 0xB5AC, 0xB5AD, 0xB5AE, /* 0xE0 */
        0xB5AF, 0xB5B0, 	 0, 0xB5B1, 0xB5B2, 0xB5B3, 0xB5B4, 0xB5B5, /* 0xF0 */
        0xB5B6, 0xB5B7, 0xB5B8, 0xB5B9, 0xB5BA, 0xB5BB					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x99:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB5BC, 0xB5BD, 0xB5BE, 0xB5BF, 0xB5C0, 0xB5C1, 0xB5C2, /* 0x40 */
        0xB5C3, 0xB5C4, 0xB5C5, 0xB5C6, 0xB5C7, 0xB5C8, 0xB5C9, 0xB5CA, /* 0x40 */
        0xB5CB, 0xB5CC, 	 0, 0xB5CD, 0xB5CE, 0xB5CF, 0xB5D0, 0xB5D1, /* 0x50 */
        0xB5D2, 0xB5D3, 0xB5D4, 0xB5D5, 0xB5D6, 0xB5D7, 	 0, 	 0, /* 0x50 */
        0, 0xB5D8, 0xB5D9, 0xB5DA, 0xB5DB, 0xB5DC, 0xB5DD, 0xB5DE, /* 0x60 */
        0xB5DF, 0xB5E0, 0xB5E1, 0xB5E2, 0xB5E3, 0xB5E4, 0xB5E5, 0xB5E6, /* 0x60 */
        0xB5E7, 0xB5E8, 	 0, 0xB5E9, 0xB5EA, 0xB5EB, 0xB5EC, 0xB5ED, /* 0x70 */
        0xB5EE, 0xB5EF, 0xB5F0, 0xB5F1, 0xB5F2, 0xB5F3, 	 0, 	 0, /* 0x70 */
        0, 0xB5F4, 0xB5F5, 0xB5F6, 0xB5F7, 0xB5F8, 0xB5F9, 0xB5FA, /* 0x80 */
        0xB5FB, 0xB5FC, 0xB5FD, 0xB5FE, 0xB5FF, 0xB600, 0xB601, 0xB602, /* 0x80 */
        0xB603, 0xB604, 	 0, 0xB605, 0xB606, 0xB607, 0xB608, 0xB609, /* 0x90 */
        0xB60A, 0xB60B, 0xB60C, 0xB60D, 0xB60E, 0xB60F, 	 0, 	 0, /* 0x90 */
        0, 0xB610, 0xB611, 0xB612, 0xB613, 0xB614, 0xB615, 0xB616, /* 0xA0 */
        0xB617, 0xB618, 0xB619, 0xB61A, 0xB61B, 0xB61C, 0xB61D, 0xB61E, /* 0xA0 */
        0xB61F, 0xB620, 	 0, 0xB621, 0xB622, 0xB623, 0xB624, 0xB625, /* 0xB0 */
        0xB626, 0xB627, 0xB628, 0xB629, 0xB62A, 0xB62B, 	 0, 	 0, /* 0xB0 */
        0, 0xB62C, 0xB62D, 0xB62E, 0xB62F, 0xB630, 0xB631, 0xB632, /* 0xC0 */
        0xB633, 0xB634, 0xB635, 0xB636, 0xB637, 0xB638, 0xB639, 0xB63A, /* 0xC0 */
        0xB63B, 0xB63C, 	 0, 0xB63D, 0xB63E, 0xB63F, 0xB640, 0xB641, /* 0xD0 */
        0xB642, 0xB643, 0xB644, 0xB645, 0xB646, 0xB647, 	 0, 	 0, /* 0xD0 */
        0, 0xB648, 0xB649, 0xB64A, 0xB64B, 0xB64C, 0xB64D, 0xB64E, /* 0xE0 */
        0xB64F, 0xB650, 0xB651, 0xB652, 0xB653, 0xB654, 0xB655, 0xB656, /* 0xE0 */
        0xB657, 0xB658, 	 0, 0xB659, 0xB65A, 0xB65B, 0xB65C, 0xB65D, /* 0xF0 */
        0xB65E, 0xB65F, 0xB660, 0xB661, 0xB662, 0xB663					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9a:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB664, 0xB665, 0xB666, 0xB667, 0xB668, 0xB669, 0xB66A, /* 0x40 */
        0xB66B, 0xB66C, 0xB66D, 0xB66E, 0xB66F, 0xB670, 0xB671, 0xB672, /* 0x40 */
        0xB673, 0xB674, 	 0, 0xB675, 0xB676, 0xB677, 0xB678, 0xB679, /* 0x50 */
        0xB67A, 0xB67B, 0xB67C, 0xB67D, 0xB67E, 0xB67F, 	 0, 	 0, /* 0x50 */
        0, 0xB680, 0xB681, 0xB682, 0xB683, 0xB684, 0xB685, 0xB686, /* 0x60 */
        0xB687, 0xB688, 0xB689, 0xB68A, 0xB68B, 0xB68C, 0xB68D, 0xB68E, /* 0x60 */
        0xB68F, 0xB690, 	 0, 0xB691, 0xB692, 0xB693, 0xB694, 0xB695, /* 0x70 */
        0xB696, 0xB697, 0xB698, 0xB699, 0xB69A, 0xB69B, 	 0, 	 0, /* 0x70 */
        0, 0xB69C, 0xB69D, 0xB69E, 0xB69F, 0xB6A0, 0xB6A1, 0xB6A2, /* 0x80 */
        0xB6A3, 0xB6A4, 0xB6A5, 0xB6A6, 0xB6A7, 0xB6A8, 0xB6A9, 0xB6AA, /* 0x80 */
        0xB6AB, 0xB6AC, 	 0, 0xB6AD, 0xB6AE, 0xB6AF, 0xB6B0, 0xB6B1, /* 0x90 */
        0xB6B2, 0xB6B3, 0xB6B4, 0xB6B5, 0xB6B6, 0xB6B7, 	 0, 	 0, /* 0x90 */
        0, 0xB6B8, 0xB6B9, 0xB6BA, 0xB6BB, 0xB6BC, 0xB6BD, 0xB6BE, /* 0xA0 */
        0xB6BF, 0xB6C0, 0xB6C1, 0xB6C2, 0xB6C3, 0xB6C4, 0xB6C5, 0xB6C6, /* 0xA0 */
        0xB6C7, 0xB6C8, 	 0, 0xB6C9, 0xB6CA, 0xB6CB, 0xB6CC, 0xB6CD, /* 0xB0 */
        0xB6CE, 0xB6CF, 0xB6D0, 0xB6D1, 0xB6D2, 0xB6D3, 	 0, 	 0, /* 0xB0 */
        0, 0xB6D4, 0xB6D5, 0xB6D6, 0xB6D7, 0xB6D8, 0xB6D9, 0xB6DA, /* 0xC0 */
        0xB6DB, 0xB6DC, 0xB6DD, 0xB6DE, 0xB6DF, 0xB6E0, 0xB6E1, 0xB6E2, /* 0xC0 */
        0xB6E3, 0xB6E4, 	 0, 0xB6E5, 0xB6E6, 0xB6E7, 0xB6E8, 0xB6E9, /* 0xD0 */
        0xB6EA, 0xB6EB, 0xB6EC, 0xB6ED, 0xB6EE, 0xB6EF, 	 0, 	 0, /* 0xD0 */
        0, 0xB6F0, 0xB6F1, 0xB6F2, 0xB6F3, 0xB6F4, 0xB6F5, 0xB6F6, /* 0xE0 */
        0xB6F7, 0xB6F8, 0xB6F9, 0xB6FA, 0xB6FB, 0xB6FC, 0xB6FD, 0xB6FE, /* 0xE0 */
        0xB6FF, 0xB700, 	 0, 0xB701, 0xB702, 0xB703, 0xB704, 0xB705, /* 0xF0 */
        0xB706, 0xB707, 0xB708, 0xB709, 0xB70A, 0xB70B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9b:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xB70C, 0xB70D, 0xB70E, 0xB70F, 0xB710, 0xB711, 0xB712, /* 0x40 */
        0xB713, 0xB714, 0xB715, 0xB716, 0xB717, 0xB718, 0xB719, 0xB71A, /* 0x40 */
        0xB71B, 0xB71C, 	 0, 0xB71D, 0xB71E, 0xB71F, 0xB720, 0xB721, /* 0x50 */
        0xB722, 0xB723, 0xB724, 0xB725, 0xB726, 0xB727, 	 0, 	 0, /* 0x50 */
        0, 0xB728, 0xB729, 0xB72A, 0xB72B, 0xB72C, 0xB72D, 0xB72E, /* 0x60 */
        0xB72F, 0xB730, 0xB731, 0xB732, 0xB733, 0xB734, 0xB735, 0xB736, /* 0x60 */
        0xB737, 0xB738, 	 0, 0xB739, 0xB73A, 0xB73B, 0xB73C, 0xB73D, /* 0x70 */
        0xB73E, 0xB73F, 0xB740, 0xB741, 0xB742, 0xB743, 	 0, 	 0, /* 0x70 */
        0, 0xB744, 0xB745, 0xB746, 0xB747, 0xB748, 0xB749, 0xB74A, /* 0x80 */
        0xB74B, 0xB74C, 0xB74D, 0xB74E, 0xB74F, 0xB750, 0xB751, 0xB752, /* 0x80 */
        0xB753, 0xB754, 	 0, 0xB755, 0xB756, 0xB757, 0xB758, 0xB759, /* 0x90 */
        0xB75A, 0xB75B, 0xB75C, 0xB75D, 0xB75E, 0xB75F, 	 0, 	 0, /* 0x90 */
        0, 0xB760, 0xB761, 0xB762, 0xB763, 0xB764, 0xB765, 0xB766, /* 0xA0 */
        0xB767, 0xB768, 0xB769, 0xB76A, 0xB76B, 0xB76C, 0xB76D, 0xB76E, /* 0xA0 */
        0xB76F, 0xB770, 	 0, 0xB771, 0xB772, 0xB773, 0xB774, 0xB775, /* 0xB0 */
        0xB776, 0xB777, 0xB778, 0xB779, 0xB77A, 0xB77B					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9c:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3139, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xB77C, 0xB77D, 0xB77E, 0xB77F, 0xB780, 0xB781, 0xB782, /* 0x60 */
        0xB783, 0xB784, 0xB785, 0xB786, 0xB787, 0xB788, 0xB789, 0xB78A, /* 0x60 */
        0xB78B, 0xB78C, 	 0, 0xB78D, 0xB78E, 0xB78F, 0xB790, 0xB791, /* 0x70 */
        0xB792, 0xB793, 0xB794, 0xB795, 0xB796, 0xB797, 	 0, 	 0, /* 0x70 */
        0, 0xB798, 0xB799, 0xB79A, 0xB79B, 0xB79C, 0xB79D, 0xB79E, /* 0x80 */
        0xB79F, 0xB7A0, 0xB7A1, 0xB7A2, 0xB7A3, 0xB7A4, 0xB7A5, 0xB7A6, /* 0x80 */
        0xB7A7, 0xB7A8, 	 0, 0xB7A9, 0xB7AA, 0xB7AB, 0xB7AC, 0xB7AD, /* 0x90 */
        0xB7AE, 0xB7AF, 0xB7B0, 0xB7B1, 0xB7B2, 0xB7B3, 	 0, 	 0, /* 0x90 */
        0, 0xB7B4, 0xB7B5, 0xB7B6, 0xB7B7, 0xB7B8, 0xB7B9, 0xB7BA, /* 0xA0 */
        0xB7BB, 0xB7BC, 0xB7BD, 0xB7BE, 0xB7BF, 0xB7C0, 0xB7C1, 0xB7C2, /* 0xA0 */
        0xB7C3, 0xB7C4, 	 0, 0xB7C5, 0xB7C6, 0xB7C7, 0xB7C8, 0xB7C9, /* 0xB0 */
        0xB7CA, 0xB7CB, 0xB7CC, 0xB7CD, 0xB7CE, 0xB7CF, 	 0, 	 0, /* 0xB0 */
        0, 0xB7D0, 0xB7D1, 0xB7D2, 0xB7D3, 0xB7D4, 0xB7D5, 0xB7D6, /* 0xC0 */
        0xB7D7, 0xB7D8, 0xB7D9, 0xB7DA, 0xB7DB, 0xB7DC, 0xB7DD, 0xB7DE, /* 0xC0 */
        0xB7DF, 0xB7E0, 	 0, 0xB7E1, 0xB7E2, 0xB7E3, 0xB7E4, 0xB7E5, /* 0xD0 */
        0xB7E6, 0xB7E7, 0xB7E8, 0xB7E9, 0xB7EA, 0xB7EB, 	 0, 	 0, /* 0xD0 */
        0, 0xB7EC, 0xB7ED, 0xB7EE, 0xB7EF, 0xB7F0, 0xB7F1, 0xB7F2, /* 0xE0 */
        0xB7F3, 0xB7F4, 0xB7F5, 0xB7F6, 0xB7F7, 0xB7F8, 0xB7F9, 0xB7FA, /* 0xE0 */
        0xB7FB, 0xB7FC, 	 0, 0xB7FD, 0xB7FE, 0xB7FF, 0xB800, 0xB801, /* 0xF0 */
        0xB802, 0xB803, 0xB804, 0xB805, 0xB806, 0xB807					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9d:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB808, 0xB809, 0xB80A, 0xB80B, 0xB80C, 0xB80D, 0xB80E, /* 0x40 */
        0xB80F, 0xB810, 0xB811, 0xB812, 0xB813, 0xB814, 0xB815, 0xB816, /* 0x40 */
        0xB817, 0xB818, 	 0, 0xB819, 0xB81A, 0xB81B, 0xB81C, 0xB81D, /* 0x50 */
        0xB81E, 0xB81F, 0xB820, 0xB821, 0xB822, 0xB823, 	 0, 	 0, /* 0x50 */
        0, 0xB824, 0xB825, 0xB826, 0xB827, 0xB828, 0xB829, 0xB82A, /* 0x60 */
        0xB82B, 0xB82C, 0xB82D, 0xB82E, 0xB82F, 0xB830, 0xB831, 0xB832, /* 0x60 */
        0xB833, 0xB834, 	 0, 0xB835, 0xB836, 0xB837, 0xB838, 0xB839, /* 0x70 */
        0xB83A, 0xB83B, 0xB83C, 0xB83D, 0xB83E, 0xB83F, 	 0, 	 0, /* 0x70 */
        0, 0xB840, 0xB841, 0xB842, 0xB843, 0xB844, 0xB845, 0xB846, /* 0x80 */
        0xB847, 0xB848, 0xB849, 0xB84A, 0xB84B, 0xB84C, 0xB84D, 0xB84E, /* 0x80 */
        0xB84F, 0xB850, 	 0, 0xB851, 0xB852, 0xB853, 0xB854, 0xB855, /* 0x90 */
        0xB856, 0xB857, 0xB858, 0xB859, 0xB85A, 0xB85B, 	 0, 	 0, /* 0x90 */
        0, 0xB85C, 0xB85D, 0xB85E, 0xB85F, 0xB860, 0xB861, 0xB862, /* 0xA0 */
        0xB863, 0xB864, 0xB865, 0xB866, 0xB867, 0xB868, 0xB869, 0xB86A, /* 0xA0 */
        0xB86B, 0xB86C, 	 0, 0xB86D, 0xB86E, 0xB86F, 0xB870, 0xB871, /* 0xB0 */
        0xB872, 0xB873, 0xB874, 0xB875, 0xB876, 0xB877, 	 0, 	 0, /* 0xB0 */
        0, 0xB878, 0xB879, 0xB87A, 0xB87B, 0xB87C, 0xB87D, 0xB87E, /* 0xC0 */
        0xB87F, 0xB880, 0xB881, 0xB882, 0xB883, 0xB884, 0xB885, 0xB886, /* 0xC0 */
        0xB887, 0xB888, 	 0, 0xB889, 0xB88A, 0xB88B, 0xB88C, 0xB88D, /* 0xD0 */
        0xB88E, 0xB88F, 0xB890, 0xB891, 0xB892, 0xB893, 	 0, 	 0, /* 0xD0 */
        0, 0xB894, 0xB895, 0xB896, 0xB897, 0xB898, 0xB899, 0xB89A, /* 0xE0 */
        0xB89B, 0xB89C, 0xB89D, 0xB89E, 0xB89F, 0xB8A0, 0xB8A1, 0xB8A2, /* 0xE0 */
        0xB8A3, 0xB8A4, 	 0, 0xB8A5, 0xB8A6, 0xB8A7, 0xB8A8, 0xB8A9, /* 0xF0 */
        0xB8AA, 0xB8AB, 0xB8AC, 0xB8AD, 0xB8AE, 0xB8AF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9e:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xB8B0, 0xB8B1, 0xB8B2, 0xB8B3, 0xB8B4, 0xB8B5, 0xB8B6, /* 0x40 */
        0xB8B7, 0xB8B8, 0xB8B9, 0xB8BA, 0xB8BB, 0xB8BC, 0xB8BD, 0xB8BE, /* 0x40 */
        0xB8BF, 0xB8C0, 	 0, 0xB8C1, 0xB8C2, 0xB8C3, 0xB8C4, 0xB8C5, /* 0x50 */
        0xB8C6, 0xB8C7, 0xB8C8, 0xB8C9, 0xB8CA, 0xB8CB, 	 0, 	 0, /* 0x50 */
        0, 0xB8CC, 0xB8CD, 0xB8CE, 0xB8CF, 0xB8D0, 0xB8D1, 0xB8D2, /* 0x60 */
        0xB8D3, 0xB8D4, 0xB8D5, 0xB8D6, 0xB8D7, 0xB8D8, 0xB8D9, 0xB8DA, /* 0x60 */
        0xB8DB, 0xB8DC, 	 0, 0xB8DD, 0xB8DE, 0xB8DF, 0xB8E0, 0xB8E1, /* 0x70 */
        0xB8E2, 0xB8E3, 0xB8E4, 0xB8E5, 0xB8E6, 0xB8E7, 	 0, 	 0, /* 0x70 */
        0, 0xB8E8, 0xB8E9, 0xB8EA, 0xB8EB, 0xB8EC, 0xB8ED, 0xB8EE, /* 0x80 */
        0xB8EF, 0xB8F0, 0xB8F1, 0xB8F2, 0xB8F3, 0xB8F4, 0xB8F5, 0xB8F6, /* 0x80 */
        0xB8F7, 0xB8F8, 	 0, 0xB8F9, 0xB8FA, 0xB8FB, 0xB8FC, 0xB8FD, /* 0x90 */
        0xB8FE, 0xB8FF, 0xB900, 0xB901, 0xB902, 0xB903, 	 0, 	 0, /* 0x90 */
        0, 0xB904, 0xB905, 0xB906, 0xB907, 0xB908, 0xB909, 0xB90A, /* 0xA0 */
        0xB90B, 0xB90C, 0xB90D, 0xB90E, 0xB90F, 0xB910, 0xB911, 0xB912, /* 0xA0 */
        0xB913, 0xB914, 	 0, 0xB915, 0xB916, 0xB917, 0xB918, 0xB919, /* 0xB0 */
        0xB91A, 0xB91B, 0xB91C, 0xB91D, 0xB91E, 0xB91F, 	 0, 	 0, /* 0xB0 */
        0, 0xB920, 0xB921, 0xB922, 0xB923, 0xB924, 0xB925, 0xB926, /* 0xC0 */
        0xB927, 0xB928, 0xB929, 0xB92A, 0xB92B, 0xB92C, 0xB92D, 0xB92E, /* 0xC0 */
        0xB92F, 0xB930, 	 0, 0xB931, 0xB932, 0xB933, 0xB934, 0xB935, /* 0xD0 */
        0xB936, 0xB937, 0xB938, 0xB939, 0xB93A, 0xB93B, 	 0, 	 0, /* 0xD0 */
        0, 0xB93C, 0xB93D, 0xB93E, 0xB93F, 0xB940, 0xB941, 0xB942, /* 0xE0 */
        0xB943, 0xB944, 0xB945, 0xB946, 0xB947, 0xB948, 0xB949, 0xB94A, /* 0xE0 */
        0xB94B, 0xB94C, 	 0, 0xB94D, 0xB94E, 0xB94F, 0xB950, 0xB951, /* 0xF0 */
        0xB952, 0xB953, 0xB954, 0xB955, 0xB956, 0xB957					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0x9f:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xB958, 0xB959, 0xB95A, 0xB95B, 0xB95C, 0xB95D, 0xB95E, /* 0x40 */
        0xB95F, 0xB960, 0xB961, 0xB962, 0xB963, 0xB964, 0xB965, 0xB966, /* 0x40 */
        0xB967, 0xB968, 	 0, 0xB969, 0xB96A, 0xB96B, 0xB96C, 0xB96D, /* 0x50 */
        0xB96E, 0xB96F, 0xB970, 0xB971, 0xB972, 0xB973, 	 0, 	 0, /* 0x50 */
        0, 0xB974, 0xB975, 0xB976, 0xB977, 0xB978, 0xB979, 0xB97A, /* 0x60 */
        0xB97B, 0xB97C, 0xB97D, 0xB97E, 0xB97F, 0xB980, 0xB981, 0xB982, /* 0x60 */
        0xB983, 0xB984, 	 0, 0xB985, 0xB986, 0xB987, 0xB988, 0xB989, /* 0x70 */
        0xB98A, 0xB98B, 0xB98C, 0xB98D, 0xB98E, 0xB98F, 	 0, 	 0, /* 0x70 */
        0, 0xB990, 0xB991, 0xB992, 0xB993, 0xB994, 0xB995, 0xB996, /* 0x80 */
        0xB997, 0xB998, 0xB999, 0xB99A, 0xB99B, 0xB99C, 0xB99D, 0xB99E, /* 0x80 */
        0xB99F, 0xB9A0, 	 0, 0xB9A1, 0xB9A2, 0xB9A3, 0xB9A4, 0xB9A5, /* 0x90 */
        0xB9A6, 0xB9A7, 0xB9A8, 0xB9A9, 0xB9AA, 0xB9AB, 	 0, 	 0, /* 0x90 */
        0, 0xB9AC, 0xB9AD, 0xB9AE, 0xB9AF, 0xB9B0, 0xB9B1, 0xB9B2, /* 0xA0 */
        0xB9B3, 0xB9B4, 0xB9B5, 0xB9B6, 0xB9B7, 0xB9B8, 0xB9B9, 0xB9BA, /* 0xA0 */
        0xB9BB, 0xB9BC, 	 0, 0xB9BD, 0xB9BE, 0xB9BF, 0xB9C0, 0xB9C1, /* 0xB0 */
        0xB9C2, 0xB9C3, 0xB9C4, 0xB9C5, 0xB9C6, 0xB9C7					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa0:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3141, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xB9C8, 0xB9C9, 0xB9CA, 0xB9CB, 0xB9CC, 0xB9CD, 0xB9CE, /* 0x60 */
        0xB9CF, 0xB9D0, 0xB9D1, 0xB9D2, 0xB9D3, 0xB9D4, 0xB9D5, 0xB9D6, /* 0x60 */
        0xB9D7, 0xB9D8, 	 0, 0xB9D9, 0xB9DA, 0xB9DB, 0xB9DC, 0xB9DD, /* 0x70 */
        0xB9DE, 0xB9DF, 0xB9E0, 0xB9E1, 0xB9E2, 0xB9E3, 	 0, 	 0, /* 0x70 */
        0, 0xB9E4, 0xB9E5, 0xB9E6, 0xB9E7, 0xB9E8, 0xB9E9, 0xB9EA, /* 0x80 */
        0xB9EB, 0xB9EC, 0xB9ED, 0xB9EE, 0xB9EF, 0xB9F0, 0xB9F1, 0xB9F2, /* 0x80 */
        0xB9F3, 0xB9F4, 	 0, 0xB9F5, 0xB9F6, 0xB9F7, 0xB9F8, 0xB9F9, /* 0x90 */
        0xB9FA, 0xB9FB, 0xB9FC, 0xB9FD, 0xB9FE, 0xB9FF, 	 0, 	 0, /* 0x90 */
        0, 0xBA00, 0xBA01, 0xBA02, 0xBA03, 0xBA04, 0xBA05, 0xBA06, /* 0xA0 */
        0xBA07, 0xBA08, 0xBA09, 0xBA0A, 0xBA0B, 0xBA0C, 0xBA0D, 0xBA0E, /* 0xA0 */
        0xBA0F, 0xBA10, 	 0, 0xBA11, 0xBA12, 0xBA13, 0xBA14, 0xBA15, /* 0xB0 */
        0xBA16, 0xBA17, 0xBA18, 0xBA19, 0xBA1A, 0xBA1B, 	 0, 	 0, /* 0xB0 */
        0, 0xBA1C, 0xBA1D, 0xBA1E, 0xBA1F, 0xBA20, 0xBA21, 0xBA22, /* 0xC0 */
        0xBA23, 0xBA24, 0xBA25, 0xBA26, 0xBA27, 0xBA28, 0xBA29, 0xBA2A, /* 0xC0 */
        0xBA2B, 0xBA2C, 	 0, 0xBA2D, 0xBA2E, 0xBA2F, 0xBA30, 0xBA31, /* 0xD0 */
        0xBA32, 0xBA33, 0xBA34, 0xBA35, 0xBA36, 0xBA37, 	 0, 	 0, /* 0xD0 */
        0, 0xBA38, 0xBA39, 0xBA3A, 0xBA3B, 0xBA3C, 0xBA3D, 0xBA3E, /* 0xE0 */
        0xBA3F, 0xBA40, 0xBA41, 0xBA42, 0xBA43, 0xBA44, 0xBA45, 0xBA46, /* 0xE0 */
        0xBA47, 0xBA48, 	 0, 0xBA49, 0xBA4A, 0xBA4B, 0xBA4C, 0xBA4D, /* 0xF0 */
        0xBA4E, 0xBA4F, 0xBA50, 0xBA51, 0xBA52, 0xBA53					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa1:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBA54, 0xBA55, 0xBA56, 0xBA57, 0xBA58, 0xBA59, 0xBA5A, /* 0x40 */
        0xBA5B, 0xBA5C, 0xBA5D, 0xBA5E, 0xBA5F, 0xBA60, 0xBA61, 0xBA62, /* 0x40 */
        0xBA63, 0xBA64, 	 0, 0xBA65, 0xBA66, 0xBA67, 0xBA68, 0xBA69, /* 0x50 */
        0xBA6A, 0xBA6B, 0xBA6C, 0xBA6D, 0xBA6E, 0xBA6F, 	 0, 	 0, /* 0x50 */
        0, 0xBA70, 0xBA71, 0xBA72, 0xBA73, 0xBA74, 0xBA75, 0xBA76, /* 0x60 */
        0xBA77, 0xBA78, 0xBA79, 0xBA7A, 0xBA7B, 0xBA7C, 0xBA7D, 0xBA7E, /* 0x60 */
        0xBA7F, 0xBA80, 	 0, 0xBA81, 0xBA82, 0xBA83, 0xBA84, 0xBA85, /* 0x70 */
        0xBA86, 0xBA87, 0xBA88, 0xBA89, 0xBA8A, 0xBA8B, 	 0, 	 0, /* 0x70 */
        0, 0xBA8C, 0xBA8D, 0xBA8E, 0xBA8F, 0xBA90, 0xBA91, 0xBA92, /* 0x80 */
        0xBA93, 0xBA94, 0xBA95, 0xBA96, 0xBA97, 0xBA98, 0xBA99, 0xBA9A, /* 0x80 */
        0xBA9B, 0xBA9C, 	 0, 0xBA9D, 0xBA9E, 0xBA9F, 0xBAA0, 0xBAA1, /* 0x90 */
        0xBAA2, 0xBAA3, 0xBAA4, 0xBAA5, 0xBAA6, 0xBAA7, 	 0, 	 0, /* 0x90 */
        0, 0xBAA8, 0xBAA9, 0xBAAA, 0xBAAB, 0xBAAC, 0xBAAD, 0xBAAE, /* 0xA0 */
        0xBAAF, 0xBAB0, 0xBAB1, 0xBAB2, 0xBAB3, 0xBAB4, 0xBAB5, 0xBAB6, /* 0xA0 */
        0xBAB7, 0xBAB8, 	 0, 0xBAB9, 0xBABA, 0xBABB, 0xBABC, 0xBABD, /* 0xB0 */
        0xBABE, 0xBABF, 0xBAC0, 0xBAC1, 0xBAC2, 0xBAC3, 	 0, 	 0, /* 0xB0 */
        0, 0xBAC4, 0xBAC5, 0xBAC6, 0xBAC7, 0xBAC8, 0xBAC9, 0xBACA, /* 0xC0 */
        0xBACB, 0xBACC, 0xBACD, 0xBACE, 0xBACF, 0xBAD0, 0xBAD1, 0xBAD2, /* 0xC0 */
        0xBAD3, 0xBAD4, 	 0, 0xBAD5, 0xBAD6, 0xBAD7, 0xBAD8, 0xBAD9, /* 0xD0 */
        0xBADA, 0xBADB, 0xBADC, 0xBADD, 0xBADE, 0xBADF, 	 0, 	 0, /* 0xD0 */
        0, 0xBAE0, 0xBAE1, 0xBAE2, 0xBAE3, 0xBAE4, 0xBAE5, 0xBAE6, /* 0xE0 */
        0xBAE7, 0xBAE8, 0xBAE9, 0xBAEA, 0xBAEB, 0xBAEC, 0xBAED, 0xBAEE, /* 0xE0 */
        0xBAEF, 0xBAF0, 	 0, 0xBAF1, 0xBAF2, 0xBAF3, 0xBAF4, 0xBAF5, /* 0xF0 */
        0xBAF6, 0xBAF7, 0xBAF8, 0xBAF9, 0xBAFA, 0xBAFB					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa2:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBAFC, 0xBAFD, 0xBAFE, 0xBAFF, 0xBB00, 0xBB01, 0xBB02, /* 0x40 */
        0xBB03, 0xBB04, 0xBB05, 0xBB06, 0xBB07, 0xBB08, 0xBB09, 0xBB0A, /* 0x40 */
        0xBB0B, 0xBB0C, 	 0, 0xBB0D, 0xBB0E, 0xBB0F, 0xBB10, 0xBB11, /* 0x50 */
        0xBB12, 0xBB13, 0xBB14, 0xBB15, 0xBB16, 0xBB17, 	 0, 	 0, /* 0x50 */
        0, 0xBB18, 0xBB19, 0xBB1A, 0xBB1B, 0xBB1C, 0xBB1D, 0xBB1E, /* 0x60 */
        0xBB1F, 0xBB20, 0xBB21, 0xBB22, 0xBB23, 0xBB24, 0xBB25, 0xBB26, /* 0x60 */
        0xBB27, 0xBB28, 	 0, 0xBB29, 0xBB2A, 0xBB2B, 0xBB2C, 0xBB2D, /* 0x70 */
        0xBB2E, 0xBB2F, 0xBB30, 0xBB31, 0xBB32, 0xBB33, 	 0, 	 0, /* 0x70 */
        0, 0xBB34, 0xBB35, 0xBB36, 0xBB37, 0xBB38, 0xBB39, 0xBB3A, /* 0x80 */
        0xBB3B, 0xBB3C, 0xBB3D, 0xBB3E, 0xBB3F, 0xBB40, 0xBB41, 0xBB42, /* 0x80 */
        0xBB43, 0xBB44, 	 0, 0xBB45, 0xBB46, 0xBB47, 0xBB48, 0xBB49, /* 0x90 */
        0xBB4A, 0xBB4B, 0xBB4C, 0xBB4D, 0xBB4E, 0xBB4F, 	 0, 	 0, /* 0x90 */
        0, 0xBB50, 0xBB51, 0xBB52, 0xBB53, 0xBB54, 0xBB55, 0xBB56, /* 0xA0 */
        0xBB57, 0xBB58, 0xBB59, 0xBB5A, 0xBB5B, 0xBB5C, 0xBB5D, 0xBB5E, /* 0xA0 */
        0xBB5F, 0xBB60, 	 0, 0xBB61, 0xBB62, 0xBB63, 0xBB64, 0xBB65, /* 0xB0 */
        0xBB66, 0xBB67, 0xBB68, 0xBB69, 0xBB6A, 0xBB6B, 	 0, 	 0, /* 0xB0 */
        0, 0xBB6C, 0xBB6D, 0xBB6E, 0xBB6F, 0xBB70, 0xBB71, 0xBB72, /* 0xC0 */
        0xBB73, 0xBB74, 0xBB75, 0xBB76, 0xBB77, 0xBB78, 0xBB79, 0xBB7A, /* 0xC0 */
        0xBB7B, 0xBB7C, 	 0, 0xBB7D, 0xBB7E, 0xBB7F, 0xBB80, 0xBB81, /* 0xD0 */
        0xBB82, 0xBB83, 0xBB84, 0xBB85, 0xBB86, 0xBB87, 	 0, 	 0, /* 0xD0 */
        0, 0xBB88, 0xBB89, 0xBB8A, 0xBB8B, 0xBB8C, 0xBB8D, 0xBB8E, /* 0xE0 */
        0xBB8F, 0xBB90, 0xBB91, 0xBB92, 0xBB93, 0xBB94, 0xBB95, 0xBB96, /* 0xE0 */
        0xBB97, 0xBB98, 	 0, 0xBB99, 0xBB9A, 0xBB9B, 0xBB9C, 0xBB9D, /* 0xF0 */
        0xBB9E, 0xBB9F, 0xBBA0, 0xBBA1, 0xBBA2, 0xBBA3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa3:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xBBA4, 0xBBA5, 0xBBA6, 0xBBA7, 0xBBA8, 0xBBA9, 0xBBAA, /* 0x40 */
        0xBBAB, 0xBBAC, 0xBBAD, 0xBBAE, 0xBBAF, 0xBBB0, 0xBBB1, 0xBBB2, /* 0x40 */
        0xBBB3, 0xBBB4, 	 0, 0xBBB5, 0xBBB6, 0xBBB7, 0xBBB8, 0xBBB9, /* 0x50 */
        0xBBBA, 0xBBBB, 0xBBBC, 0xBBBD, 0xBBBE, 0xBBBF, 	 0, 	 0, /* 0x50 */
        0, 0xBBC0, 0xBBC1, 0xBBC2, 0xBBC3, 0xBBC4, 0xBBC5, 0xBBC6, /* 0x60 */
        0xBBC7, 0xBBC8, 0xBBC9, 0xBBCA, 0xBBCB, 0xBBCC, 0xBBCD, 0xBBCE, /* 0x60 */
        0xBBCF, 0xBBD0, 	 0, 0xBBD1, 0xBBD2, 0xBBD3, 0xBBD4, 0xBBD5, /* 0x70 */
        0xBBD6, 0xBBD7, 0xBBD8, 0xBBD9, 0xBBDA, 0xBBDB, 	 0, 	 0, /* 0x70 */
        0, 0xBBDC, 0xBBDD, 0xBBDE, 0xBBDF, 0xBBE0, 0xBBE1, 0xBBE2, /* 0x80 */
        0xBBE3, 0xBBE4, 0xBBE5, 0xBBE6, 0xBBE7, 0xBBE8, 0xBBE9, 0xBBEA, /* 0x80 */
        0xBBEB, 0xBBEC, 	 0, 0xBBED, 0xBBEE, 0xBBEF, 0xBBF0, 0xBBF1, /* 0x90 */
        0xBBF2, 0xBBF3, 0xBBF4, 0xBBF5, 0xBBF6, 0xBBF7, 	 0, 	 0, /* 0x90 */
        0, 0xBBF8, 0xBBF9, 0xBBFA, 0xBBFB, 0xBBFC, 0xBBFD, 0xBBFE, /* 0xA0 */
        0xBBFF, 0xBC00, 0xBC01, 0xBC02, 0xBC03, 0xBC04, 0xBC05, 0xBC06, /* 0xA0 */
        0xBC07, 0xBC08, 	 0, 0xBC09, 0xBC0A, 0xBC0B, 0xBC0C, 0xBC0D, /* 0xB0 */
        0xBC0E, 0xBC0F, 0xBC10, 0xBC11, 0xBC12, 0xBC13					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa4:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3142, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xBC14, 0xBC15, 0xBC16, 0xBC17, 0xBC18, 0xBC19, 0xBC1A, /* 0x60 */
        0xBC1B, 0xBC1C, 0xBC1D, 0xBC1E, 0xBC1F, 0xBC20, 0xBC21, 0xBC22, /* 0x60 */
        0xBC23, 0xBC24, 	 0, 0xBC25, 0xBC26, 0xBC27, 0xBC28, 0xBC29, /* 0x70 */
        0xBC2A, 0xBC2B, 0xBC2C, 0xBC2D, 0xBC2E, 0xBC2F, 	 0, 	 0, /* 0x70 */
        0, 0xBC30, 0xBC31, 0xBC32, 0xBC33, 0xBC34, 0xBC35, 0xBC36, /* 0x80 */
        0xBC37, 0xBC38, 0xBC39, 0xBC3A, 0xBC3B, 0xBC3C, 0xBC3D, 0xBC3E, /* 0x80 */
        0xBC3F, 0xBC40, 	 0, 0xBC41, 0xBC42, 0xBC43, 0xBC44, 0xBC45, /* 0x90 */
        0xBC46, 0xBC47, 0xBC48, 0xBC49, 0xBC4A, 0xBC4B, 	 0, 	 0, /* 0x90 */
        0, 0xBC4C, 0xBC4D, 0xBC4E, 0xBC4F, 0xBC50, 0xBC51, 0xBC52, /* 0xA0 */
        0xBC53, 0xBC54, 0xBC55, 0xBC56, 0xBC57, 0xBC58, 0xBC59, 0xBC5A, /* 0xA0 */
        0xBC5B, 0xBC5C, 	 0, 0xBC5D, 0xBC5E, 0xBC5F, 0xBC60, 0xBC61, /* 0xB0 */
        0xBC62, 0xBC63, 0xBC64, 0xBC65, 0xBC66, 0xBC67, 	 0, 	 0, /* 0xB0 */
        0, 0xBC68, 0xBC69, 0xBC6A, 0xBC6B, 0xBC6C, 0xBC6D, 0xBC6E, /* 0xC0 */
        0xBC6F, 0xBC70, 0xBC71, 0xBC72, 0xBC73, 0xBC74, 0xBC75, 0xBC76, /* 0xC0 */
        0xBC77, 0xBC78, 	 0, 0xBC79, 0xBC7A, 0xBC7B, 0xBC7C, 0xBC7D, /* 0xD0 */
        0xBC7E, 0xBC7F, 0xBC80, 0xBC81, 0xBC82, 0xBC83, 	 0, 	 0, /* 0xD0 */
        0, 0xBC84, 0xBC85, 0xBC86, 0xBC87, 0xBC88, 0xBC89, 0xBC8A, /* 0xE0 */
        0xBC8B, 0xBC8C, 0xBC8D, 0xBC8E, 0xBC8F, 0xBC90, 0xBC91, 0xBC92, /* 0xE0 */
        0xBC93, 0xBC94, 	 0, 0xBC95, 0xBC96, 0xBC97, 0xBC98, 0xBC99, /* 0xF0 */
        0xBC9A, 0xBC9B, 0xBC9C, 0xBC9D, 0xBC9E, 0xBC9F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa5:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBCA0, 0xBCA1, 0xBCA2, 0xBCA3, 0xBCA4, 0xBCA5, 0xBCA6, /* 0x40 */
        0xBCA7, 0xBCA8, 0xBCA9, 0xBCAA, 0xBCAB, 0xBCAC, 0xBCAD, 0xBCAE, /* 0x40 */
        0xBCAF, 0xBCB0, 	 0, 0xBCB1, 0xBCB2, 0xBCB3, 0xBCB4, 0xBCB5, /* 0x50 */
        0xBCB6, 0xBCB7, 0xBCB8, 0xBCB9, 0xBCBA, 0xBCBB, 	 0, 	 0, /* 0x50 */
        0, 0xBCBC, 0xBCBD, 0xBCBE, 0xBCBF, 0xBCC0, 0xBCC1, 0xBCC2, /* 0x60 */
        0xBCC3, 0xBCC4, 0xBCC5, 0xBCC6, 0xBCC7, 0xBCC8, 0xBCC9, 0xBCCA, /* 0x60 */
        0xBCCB, 0xBCCC, 	 0, 0xBCCD, 0xBCCE, 0xBCCF, 0xBCD0, 0xBCD1, /* 0x70 */
        0xBCD2, 0xBCD3, 0xBCD4, 0xBCD5, 0xBCD6, 0xBCD7, 	 0, 	 0, /* 0x70 */
        0, 0xBCD8, 0xBCD9, 0xBCDA, 0xBCDB, 0xBCDC, 0xBCDD, 0xBCDE, /* 0x80 */
        0xBCDF, 0xBCE0, 0xBCE1, 0xBCE2, 0xBCE3, 0xBCE4, 0xBCE5, 0xBCE6, /* 0x80 */
        0xBCE7, 0xBCE8, 	 0, 0xBCE9, 0xBCEA, 0xBCEB, 0xBCEC, 0xBCED, /* 0x90 */
        0xBCEE, 0xBCEF, 0xBCF0, 0xBCF1, 0xBCF2, 0xBCF3, 	 0, 	 0, /* 0x90 */
        0, 0xBCF4, 0xBCF5, 0xBCF6, 0xBCF7, 0xBCF8, 0xBCF9, 0xBCFA, /* 0xA0 */
        0xBCFB, 0xBCFC, 0xBCFD, 0xBCFE, 0xBCFF, 0xBD00, 0xBD01, 0xBD02, /* 0xA0 */
        0xBD03, 0xBD04, 	 0, 0xBD05, 0xBD06, 0xBD07, 0xBD08, 0xBD09, /* 0xB0 */
        0xBD0A, 0xBD0B, 0xBD0C, 0xBD0D, 0xBD0E, 0xBD0F, 	 0, 	 0, /* 0xB0 */
        0, 0xBD10, 0xBD11, 0xBD12, 0xBD13, 0xBD14, 0xBD15, 0xBD16, /* 0xC0 */
        0xBD17, 0xBD18, 0xBD19, 0xBD1A, 0xBD1B, 0xBD1C, 0xBD1D, 0xBD1E, /* 0xC0 */
        0xBD1F, 0xBD20, 	 0, 0xBD21, 0xBD22, 0xBD23, 0xBD24, 0xBD25, /* 0xD0 */
        0xBD26, 0xBD27, 0xBD28, 0xBD29, 0xBD2A, 0xBD2B, 	 0, 	 0, /* 0xD0 */
        0, 0xBD2C, 0xBD2D, 0xBD2E, 0xBD2F, 0xBD30, 0xBD31, 0xBD32, /* 0xE0 */
        0xBD33, 0xBD34, 0xBD35, 0xBD36, 0xBD37, 0xBD38, 0xBD39, 0xBD3A, /* 0xE0 */
        0xBD3B, 0xBD3C, 	 0, 0xBD3D, 0xBD3E, 0xBD3F, 0xBD40, 0xBD41, /* 0xF0 */
        0xBD42, 0xBD43, 0xBD44, 0xBD45, 0xBD46, 0xBD47					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa6:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBD48, 0xBD49, 0xBD4A, 0xBD4B, 0xBD4C, 0xBD4D, 0xBD4E, /* 0x40 */
        0xBD4F, 0xBD50, 0xBD51, 0xBD52, 0xBD53, 0xBD54, 0xBD55, 0xBD56, /* 0x40 */
        0xBD57, 0xBD58, 	 0, 0xBD59, 0xBD5A, 0xBD5B, 0xBD5C, 0xBD5D, /* 0x50 */
        0xBD5E, 0xBD5F, 0xBD60, 0xBD61, 0xBD62, 0xBD63, 	 0, 	 0, /* 0x50 */
        0, 0xBD64, 0xBD65, 0xBD66, 0xBD67, 0xBD68, 0xBD69, 0xBD6A, /* 0x60 */
        0xBD6B, 0xBD6C, 0xBD6D, 0xBD6E, 0xBD6F, 0xBD70, 0xBD71, 0xBD72, /* 0x60 */
        0xBD73, 0xBD74, 	 0, 0xBD75, 0xBD76, 0xBD77, 0xBD78, 0xBD79, /* 0x70 */
        0xBD7A, 0xBD7B, 0xBD7C, 0xBD7D, 0xBD7E, 0xBD7F, 	 0, 	 0, /* 0x70 */
        0, 0xBD80, 0xBD81, 0xBD82, 0xBD83, 0xBD84, 0xBD85, 0xBD86, /* 0x80 */
        0xBD87, 0xBD88, 0xBD89, 0xBD8A, 0xBD8B, 0xBD8C, 0xBD8D, 0xBD8E, /* 0x80 */
        0xBD8F, 0xBD90, 	 0, 0xBD91, 0xBD92, 0xBD93, 0xBD94, 0xBD95, /* 0x90 */
        0xBD96, 0xBD97, 0xBD98, 0xBD99, 0xBD9A, 0xBD9B, 	 0, 	 0, /* 0x90 */
        0, 0xBD9C, 0xBD9D, 0xBD9E, 0xBD9F, 0xBDA0, 0xBDA1, 0xBDA2, /* 0xA0 */
        0xBDA3, 0xBDA4, 0xBDA5, 0xBDA6, 0xBDA7, 0xBDA8, 0xBDA9, 0xBDAA, /* 0xA0 */
        0xBDAB, 0xBDAC, 	 0, 0xBDAD, 0xBDAE, 0xBDAF, 0xBDB0, 0xBDB1, /* 0xB0 */
        0xBDB2, 0xBDB3, 0xBDB4, 0xBDB5, 0xBDB6, 0xBDB7, 	 0, 	 0, /* 0xB0 */
        0, 0xBDB8, 0xBDB9, 0xBDBA, 0xBDBB, 0xBDBC, 0xBDBD, 0xBDBE, /* 0xC0 */
        0xBDBF, 0xBDC0, 0xBDC1, 0xBDC2, 0xBDC3, 0xBDC4, 0xBDC5, 0xBDC6, /* 0xC0 */
        0xBDC7, 0xBDC8, 	 0, 0xBDC9, 0xBDCA, 0xBDCB, 0xBDCC, 0xBDCD, /* 0xD0 */
        0xBDCE, 0xBDCF, 0xBDD0, 0xBDD1, 0xBDD2, 0xBDD3, 	 0, 	 0, /* 0xD0 */
        0, 0xBDD4, 0xBDD5, 0xBDD6, 0xBDD7, 0xBDD8, 0xBDD9, 0xBDDA, /* 0xE0 */
        0xBDDB, 0xBDDC, 0xBDDD, 0xBDDE, 0xBDDF, 0xBDE0, 0xBDE1, 0xBDE2, /* 0xE0 */
        0xBDE3, 0xBDE4, 	 0, 0xBDE5, 0xBDE6, 0xBDE7, 0xBDE8, 0xBDE9, /* 0xF0 */
        0xBDEA, 0xBDEB, 0xBDEC, 0xBDED, 0xBDEE, 0xBDEF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa7:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xBDF0, 0xBDF1, 0xBDF2, 0xBDF3, 0xBDF4, 0xBDF5, 0xBDF6, /* 0x40 */
        0xBDF7, 0xBDF8, 0xBDF9, 0xBDFA, 0xBDFB, 0xBDFC, 0xBDFD, 0xBDFE, /* 0x40 */
        0xBDFF, 0xBE00, 	 0, 0xBE01, 0xBE02, 0xBE03, 0xBE04, 0xBE05, /* 0x50 */
        0xBE06, 0xBE07, 0xBE08, 0xBE09, 0xBE0A, 0xBE0B, 	 0, 	 0, /* 0x50 */
        0, 0xBE0C, 0xBE0D, 0xBE0E, 0xBE0F, 0xBE10, 0xBE11, 0xBE12, /* 0x60 */
        0xBE13, 0xBE14, 0xBE15, 0xBE16, 0xBE17, 0xBE18, 0xBE19, 0xBE1A, /* 0x60 */
        0xBE1B, 0xBE1C, 	 0, 0xBE1D, 0xBE1E, 0xBE1F, 0xBE20, 0xBE21, /* 0x70 */
        0xBE22, 0xBE23, 0xBE24, 0xBE25, 0xBE26, 0xBE27, 	 0, 	 0, /* 0x70 */
        0, 0xBE28, 0xBE29, 0xBE2A, 0xBE2B, 0xBE2C, 0xBE2D, 0xBE2E, /* 0x80 */
        0xBE2F, 0xBE30, 0xBE31, 0xBE32, 0xBE33, 0xBE34, 0xBE35, 0xBE36, /* 0x80 */
        0xBE37, 0xBE38, 	 0, 0xBE39, 0xBE3A, 0xBE3B, 0xBE3C, 0xBE3D, /* 0x90 */
        0xBE3E, 0xBE3F, 0xBE40, 0xBE41, 0xBE42, 0xBE43, 	 0, 	 0, /* 0x90 */
        0, 0xBE44, 0xBE45, 0xBE46, 0xBE47, 0xBE48, 0xBE49, 0xBE4A, /* 0xA0 */
        0xBE4B, 0xBE4C, 0xBE4D, 0xBE4E, 0xBE4F, 0xBE50, 0xBE51, 0xBE52, /* 0xA0 */
        0xBE53, 0xBE54, 	 0, 0xBE55, 0xBE56, 0xBE57, 0xBE58, 0xBE59, /* 0xB0 */
        0xBE5A, 0xBE5B, 0xBE5C, 0xBE5D, 0xBE5E, 0xBE5F					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa8:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3143, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xBE60, 0xBE61, 0xBE62, 0xBE63, 0xBE64, 0xBE65, 0xBE66, /* 0x60 */
        0xBE67, 0xBE68, 0xBE69, 0xBE6A, 0xBE6B, 0xBE6C, 0xBE6D, 0xBE6E, /* 0x60 */
        0xBE6F, 0xBE70, 	 0, 0xBE71, 0xBE72, 0xBE73, 0xBE74, 0xBE75, /* 0x70 */
        0xBE76, 0xBE77, 0xBE78, 0xBE79, 0xBE7A, 0xBE7B, 	 0, 	 0, /* 0x70 */
        0, 0xBE7C, 0xBE7D, 0xBE7E, 0xBE7F, 0xBE80, 0xBE81, 0xBE82, /* 0x80 */
        0xBE83, 0xBE84, 0xBE85, 0xBE86, 0xBE87, 0xBE88, 0xBE89, 0xBE8A, /* 0x80 */
        0xBE8B, 0xBE8C, 	 0, 0xBE8D, 0xBE8E, 0xBE8F, 0xBE90, 0xBE91, /* 0x90 */
        0xBE92, 0xBE93, 0xBE94, 0xBE95, 0xBE96, 0xBE97, 	 0, 	 0, /* 0x90 */
        0, 0xBE98, 0xBE99, 0xBE9A, 0xBE9B, 0xBE9C, 0xBE9D, 0xBE9E, /* 0xA0 */
        0xBE9F, 0xBEA0, 0xBEA1, 0xBEA2, 0xBEA3, 0xBEA4, 0xBEA5, 0xBEA6, /* 0xA0 */
        0xBEA7, 0xBEA8, 	 0, 0xBEA9, 0xBEAA, 0xBEAB, 0xBEAC, 0xBEAD, /* 0xB0 */
        0xBEAE, 0xBEAF, 0xBEB0, 0xBEB1, 0xBEB2, 0xBEB3, 	 0, 	 0, /* 0xB0 */
        0, 0xBEB4, 0xBEB5, 0xBEB6, 0xBEB7, 0xBEB8, 0xBEB9, 0xBEBA, /* 0xC0 */
        0xBEBB, 0xBEBC, 0xBEBD, 0xBEBE, 0xBEBF, 0xBEC0, 0xBEC1, 0xBEC2, /* 0xC0 */
        0xBEC3, 0xBEC4, 	 0, 0xBEC5, 0xBEC6, 0xBEC7, 0xBEC8, 0xBEC9, /* 0xD0 */
        0xBECA, 0xBECB, 0xBECC, 0xBECD, 0xBECE, 0xBECF, 	 0, 	 0, /* 0xD0 */
        0, 0xBED0, 0xBED1, 0xBED2, 0xBED3, 0xBED4, 0xBED5, 0xBED6, /* 0xE0 */
        0xBED7, 0xBED8, 0xBED9, 0xBEDA, 0xBEDB, 0xBEDC, 0xBEDD, 0xBEDE, /* 0xE0 */
        0xBEDF, 0xBEE0, 	 0, 0xBEE1, 0xBEE2, 0xBEE3, 0xBEE4, 0xBEE5, /* 0xF0 */
        0xBEE6, 0xBEE7, 0xBEE8, 0xBEE9, 0xBEEA, 0xBEEB					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xa9:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBEEC, 0xBEED, 0xBEEE, 0xBEEF, 0xBEF0, 0xBEF1, 0xBEF2, /* 0x40 */
        0xBEF3, 0xBEF4, 0xBEF5, 0xBEF6, 0xBEF7, 0xBEF8, 0xBEF9, 0xBEFA, /* 0x40 */
        0xBEFB, 0xBEFC, 	 0, 0xBEFD, 0xBEFE, 0xBEFF, 0xBF00, 0xBF01, /* 0x50 */
        0xBF02, 0xBF03, 0xBF04, 0xBF05, 0xBF06, 0xBF07, 	 0, 	 0, /* 0x50 */
        0, 0xBF08, 0xBF09, 0xBF0A, 0xBF0B, 0xBF0C, 0xBF0D, 0xBF0E, /* 0x60 */
        0xBF0F, 0xBF10, 0xBF11, 0xBF12, 0xBF13, 0xBF14, 0xBF15, 0xBF16, /* 0x60 */
        0xBF17, 0xBF18, 	 0, 0xBF19, 0xBF1A, 0xBF1B, 0xBF1C, 0xBF1D, /* 0x70 */
        0xBF1E, 0xBF1F, 0xBF20, 0xBF21, 0xBF22, 0xBF23, 	 0, 	 0, /* 0x70 */
        0, 0xBF24, 0xBF25, 0xBF26, 0xBF27, 0xBF28, 0xBF29, 0xBF2A, /* 0x80 */
        0xBF2B, 0xBF2C, 0xBF2D, 0xBF2E, 0xBF2F, 0xBF30, 0xBF31, 0xBF32, /* 0x80 */
        0xBF33, 0xBF34, 	 0, 0xBF35, 0xBF36, 0xBF37, 0xBF38, 0xBF39, /* 0x90 */
        0xBF3A, 0xBF3B, 0xBF3C, 0xBF3D, 0xBF3E, 0xBF3F, 	 0, 	 0, /* 0x90 */
        0, 0xBF40, 0xBF41, 0xBF42, 0xBF43, 0xBF44, 0xBF45, 0xBF46, /* 0xA0 */
        0xBF47, 0xBF48, 0xBF49, 0xBF4A, 0xBF4B, 0xBF4C, 0xBF4D, 0xBF4E, /* 0xA0 */
        0xBF4F, 0xBF50, 	 0, 0xBF51, 0xBF52, 0xBF53, 0xBF54, 0xBF55, /* 0xB0 */
        0xBF56, 0xBF57, 0xBF58, 0xBF59, 0xBF5A, 0xBF5B, 	 0, 	 0, /* 0xB0 */
        0, 0xBF5C, 0xBF5D, 0xBF5E, 0xBF5F, 0xBF60, 0xBF61, 0xBF62, /* 0xC0 */
        0xBF63, 0xBF64, 0xBF65, 0xBF66, 0xBF67, 0xBF68, 0xBF69, 0xBF6A, /* 0xC0 */
        0xBF6B, 0xBF6C, 	 0, 0xBF6D, 0xBF6E, 0xBF6F, 0xBF70, 0xBF71, /* 0xD0 */
        0xBF72, 0xBF73, 0xBF74, 0xBF75, 0xBF76, 0xBF77, 	 0, 	 0, /* 0xD0 */
        0, 0xBF78, 0xBF79, 0xBF7A, 0xBF7B, 0xBF7C, 0xBF7D, 0xBF7E, /* 0xE0 */
        0xBF7F, 0xBF80, 0xBF81, 0xBF82, 0xBF83, 0xBF84, 0xBF85, 0xBF86, /* 0xE0 */
        0xBF87, 0xBF88, 	 0, 0xBF89, 0xBF8A, 0xBF8B, 0xBF8C, 0xBF8D, /* 0xF0 */
        0xBF8E, 0xBF8F, 0xBF90, 0xBF91, 0xBF92, 0xBF93					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xaa:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xBF94, 0xBF95, 0xBF96, 0xBF97, 0xBF98, 0xBF99, 0xBF9A, /* 0x40 */
        0xBF9B, 0xBF9C, 0xBF9D, 0xBF9E, 0xBF9F, 0xBFA0, 0xBFA1, 0xBFA2, /* 0x40 */
        0xBFA3, 0xBFA4, 	 0, 0xBFA5, 0xBFA6, 0xBFA7, 0xBFA8, 0xBFA9, /* 0x50 */
        0xBFAA, 0xBFAB, 0xBFAC, 0xBFAD, 0xBFAE, 0xBFAF, 	 0, 	 0, /* 0x50 */
        0, 0xBFB0, 0xBFB1, 0xBFB2, 0xBFB3, 0xBFB4, 0xBFB5, 0xBFB6, /* 0x60 */
        0xBFB7, 0xBFB8, 0xBFB9, 0xBFBA, 0xBFBB, 0xBFBC, 0xBFBD, 0xBFBE, /* 0x60 */
        0xBFBF, 0xBFC0, 	 0, 0xBFC1, 0xBFC2, 0xBFC3, 0xBFC4, 0xBFC5, /* 0x70 */
        0xBFC6, 0xBFC7, 0xBFC8, 0xBFC9, 0xBFCA, 0xBFCB, 	 0, 	 0, /* 0x70 */
        0, 0xBFCC, 0xBFCD, 0xBFCE, 0xBFCF, 0xBFD0, 0xBFD1, 0xBFD2, /* 0x80 */
        0xBFD3, 0xBFD4, 0xBFD5, 0xBFD6, 0xBFD7, 0xBFD8, 0xBFD9, 0xBFDA, /* 0x80 */
        0xBFDB, 0xBFDC, 	 0, 0xBFDD, 0xBFDE, 0xBFDF, 0xBFE0, 0xBFE1, /* 0x90 */
        0xBFE2, 0xBFE3, 0xBFE4, 0xBFE5, 0xBFE6, 0xBFE7, 	 0, 	 0, /* 0x90 */
        0, 0xBFE8, 0xBFE9, 0xBFEA, 0xBFEB, 0xBFEC, 0xBFED, 0xBFEE, /* 0xA0 */
        0xBFEF, 0xBFF0, 0xBFF1, 0xBFF2, 0xBFF3, 0xBFF4, 0xBFF5, 0xBFF6, /* 0xA0 */
        0xBFF7, 0xBFF8, 	 0, 0xBFF9, 0xBFFA, 0xBFFB, 0xBFFC, 0xBFFD, /* 0xB0 */
        0xBFFE, 0xBFFF, 0xC000, 0xC001, 0xC002, 0xC003, 	 0, 	 0, /* 0xB0 */
        0, 0xC004, 0xC005, 0xC006, 0xC007, 0xC008, 0xC009, 0xC00A, /* 0xC0 */
        0xC00B, 0xC00C, 0xC00D, 0xC00E, 0xC00F, 0xC010, 0xC011, 0xC012, /* 0xC0 */
        0xC013, 0xC014, 	 0, 0xC015, 0xC016, 0xC017, 0xC018, 0xC019, /* 0xD0 */
        0xC01A, 0xC01B, 0xC01C, 0xC01D, 0xC01E, 0xC01F, 	 0, 	 0, /* 0xD0 */
        0, 0xC020, 0xC021, 0xC022, 0xC023, 0xC024, 0xC025, 0xC026, /* 0xE0 */
        0xC027, 0xC028, 0xC029, 0xC02A, 0xC02B, 0xC02C, 0xC02D, 0xC02E, /* 0xE0 */
        0xC02F, 0xC030, 	 0, 0xC031, 0xC032, 0xC033, 0xC034, 0xC035, /* 0xF0 */
        0xC036, 0xC037, 0xC038, 0xC039, 0xC03A, 0xC03B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xab:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xC03C, 0xC03D, 0xC03E, 0xC03F, 0xC040, 0xC041, 0xC042, /* 0x40 */
        0xC043, 0xC044, 0xC045, 0xC046, 0xC047, 0xC048, 0xC049, 0xC04A, /* 0x40 */
        0xC04B, 0xC04C, 	 0, 0xC04D, 0xC04E, 0xC04F, 0xC050, 0xC051, /* 0x50 */
        0xC052, 0xC053, 0xC054, 0xC055, 0xC056, 0xC057, 	 0, 	 0, /* 0x50 */
        0, 0xC058, 0xC059, 0xC05A, 0xC05B, 0xC05C, 0xC05D, 0xC05E, /* 0x60 */
        0xC05F, 0xC060, 0xC061, 0xC062, 0xC063, 0xC064, 0xC065, 0xC066, /* 0x60 */
        0xC067, 0xC068, 	 0, 0xC069, 0xC06A, 0xC06B, 0xC06C, 0xC06D, /* 0x70 */
        0xC06E, 0xC06F, 0xC070, 0xC071, 0xC072, 0xC073, 	 0, 	 0, /* 0x70 */
        0, 0xC074, 0xC075, 0xC076, 0xC077, 0xC078, 0xC079, 0xC07A, /* 0x80 */
        0xC07B, 0xC07C, 0xC07D, 0xC07E, 0xC07F, 0xC080, 0xC081, 0xC082, /* 0x80 */
        0xC083, 0xC084, 	 0, 0xC085, 0xC086, 0xC087, 0xC088, 0xC089, /* 0x90 */
        0xC08A, 0xC08B, 0xC08C, 0xC08D, 0xC08E, 0xC08F, 	 0, 	 0, /* 0x90 */
        0, 0xC090, 0xC091, 0xC092, 0xC093, 0xC094, 0xC095, 0xC096, /* 0xA0 */
        0xC097, 0xC098, 0xC099, 0xC09A, 0xC09B, 0xC09C, 0xC09D, 0xC09E, /* 0xA0 */
        0xC09F, 0xC0A0, 	 0, 0xC0A1, 0xC0A2, 0xC0A3, 0xC0A4, 0xC0A5, /* 0xB0 */
        0xC0A6, 0xC0A7, 0xC0A8, 0xC0A9, 0xC0AA, 0xC0AB					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xac:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3145, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xC0AC, 0xC0AD, 0xC0AE, 0xC0AF, 0xC0B0, 0xC0B1, 0xC0B2, /* 0x60 */
        0xC0B3, 0xC0B4, 0xC0B5, 0xC0B6, 0xC0B7, 0xC0B8, 0xC0B9, 0xC0BA, /* 0x60 */
        0xC0BB, 0xC0BC, 	 0, 0xC0BD, 0xC0BE, 0xC0BF, 0xC0C0, 0xC0C1, /* 0x70 */
        0xC0C2, 0xC0C3, 0xC0C4, 0xC0C5, 0xC0C6, 0xC0C7, 	 0, 	 0, /* 0x70 */
        0, 0xC0C8, 0xC0C9, 0xC0CA, 0xC0CB, 0xC0CC, 0xC0CD, 0xC0CE, /* 0x80 */
        0xC0CF, 0xC0D0, 0xC0D1, 0xC0D2, 0xC0D3, 0xC0D4, 0xC0D5, 0xC0D6, /* 0x80 */
        0xC0D7, 0xC0D8, 	 0, 0xC0D9, 0xC0DA, 0xC0DB, 0xC0DC, 0xC0DD, /* 0x90 */
        0xC0DE, 0xC0DF, 0xC0E0, 0xC0E1, 0xC0E2, 0xC0E3, 	 0, 	 0, /* 0x90 */
        0, 0xC0E4, 0xC0E5, 0xC0E6, 0xC0E7, 0xC0E8, 0xC0E9, 0xC0EA, /* 0xA0 */
        0xC0EB, 0xC0EC, 0xC0ED, 0xC0EE, 0xC0EF, 0xC0F0, 0xC0F1, 0xC0F2, /* 0xA0 */
        0xC0F3, 0xC0F4, 	 0, 0xC0F5, 0xC0F6, 0xC0F7, 0xC0F8, 0xC0F9, /* 0xB0 */
        0xC0FA, 0xC0FB, 0xC0FC, 0xC0FD, 0xC0FE, 0xC0FF, 	 0, 	 0, /* 0xB0 */
        0, 0xC100, 0xC101, 0xC102, 0xC103, 0xC104, 0xC105, 0xC106, /* 0xC0 */
        0xC107, 0xC108, 0xC109, 0xC10A, 0xC10B, 0xC10C, 0xC10D, 0xC10E, /* 0xC0 */
        0xC10F, 0xC110, 	 0, 0xC111, 0xC112, 0xC113, 0xC114, 0xC115, /* 0xD0 */
        0xC116, 0xC117, 0xC118, 0xC119, 0xC11A, 0xC11B, 	 0, 	 0, /* 0xD0 */
        0, 0xC11C, 0xC11D, 0xC11E, 0xC11F, 0xC120, 0xC121, 0xC122, /* 0xE0 */
        0xC123, 0xC124, 0xC125, 0xC126, 0xC127, 0xC128, 0xC129, 0xC12A, /* 0xE0 */
        0xC12B, 0xC12C, 	 0, 0xC12D, 0xC12E, 0xC12F, 0xC130, 0xC131, /* 0xF0 */
        0xC132, 0xC133, 0xC134, 0xC135, 0xC136, 0xC137					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xad:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC138, 0xC139, 0xC13A, 0xC13B, 0xC13C, 0xC13D, 0xC13E, /* 0x40 */
        0xC13F, 0xC140, 0xC141, 0xC142, 0xC143, 0xC144, 0xC145, 0xC146, /* 0x40 */
        0xC147, 0xC148, 	 0, 0xC149, 0xC14A, 0xC14B, 0xC14C, 0xC14D, /* 0x50 */
        0xC14E, 0xC14F, 0xC150, 0xC151, 0xC152, 0xC153, 	 0, 	 0, /* 0x50 */
        0, 0xC154, 0xC155, 0xC156, 0xC157, 0xC158, 0xC159, 0xC15A, /* 0x60 */
        0xC15B, 0xC15C, 0xC15D, 0xC15E, 0xC15F, 0xC160, 0xC161, 0xC162, /* 0x60 */
        0xC163, 0xC164, 	 0, 0xC165, 0xC166, 0xC167, 0xC168, 0xC169, /* 0x70 */
        0xC16A, 0xC16B, 0xC16C, 0xC16D, 0xC16E, 0xC16F, 	 0, 	 0, /* 0x70 */
        0, 0xC170, 0xC171, 0xC172, 0xC173, 0xC174, 0xC175, 0xC176, /* 0x80 */
        0xC177, 0xC178, 0xC179, 0xC17A, 0xC17B, 0xC17C, 0xC17D, 0xC17E, /* 0x80 */
        0xC17F, 0xC180, 	 0, 0xC181, 0xC182, 0xC183, 0xC184, 0xC185, /* 0x90 */
        0xC186, 0xC187, 0xC188, 0xC189, 0xC18A, 0xC18B, 	 0, 	 0, /* 0x90 */
        0, 0xC18C, 0xC18D, 0xC18E, 0xC18F, 0xC190, 0xC191, 0xC192, /* 0xA0 */
        0xC193, 0xC194, 0xC195, 0xC196, 0xC197, 0xC198, 0xC199, 0xC19A, /* 0xA0 */
        0xC19B, 0xC19C, 	 0, 0xC19D, 0xC19E, 0xC19F, 0xC1A0, 0xC1A1, /* 0xB0 */
        0xC1A2, 0xC1A3, 0xC1A4, 0xC1A5, 0xC1A6, 0xC1A7, 	 0, 	 0, /* 0xB0 */
        0, 0xC1A8, 0xC1A9, 0xC1AA, 0xC1AB, 0xC1AC, 0xC1AD, 0xC1AE, /* 0xC0 */
        0xC1AF, 0xC1B0, 0xC1B1, 0xC1B2, 0xC1B3, 0xC1B4, 0xC1B5, 0xC1B6, /* 0xC0 */
        0xC1B7, 0xC1B8, 	 0, 0xC1B9, 0xC1BA, 0xC1BB, 0xC1BC, 0xC1BD, /* 0xD0 */
        0xC1BE, 0xC1BF, 0xC1C0, 0xC1C1, 0xC1C2, 0xC1C3, 	 0, 	 0, /* 0xD0 */
        0, 0xC1C4, 0xC1C5, 0xC1C6, 0xC1C7, 0xC1C8, 0xC1C9, 0xC1CA, /* 0xE0 */
        0xC1CB, 0xC1CC, 0xC1CD, 0xC1CE, 0xC1CF, 0xC1D0, 0xC1D1, 0xC1D2, /* 0xE0 */
        0xC1D3, 0xC1D4, 	 0, 0xC1D5, 0xC1D6, 0xC1D7, 0xC1D8, 0xC1D9, /* 0xF0 */
        0xC1DA, 0xC1DB, 0xC1DC, 0xC1DD, 0xC1DE, 0xC1DF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xae:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC1E0, 0xC1E1, 0xC1E2, 0xC1E3, 0xC1E4, 0xC1E5, 0xC1E6, /* 0x40 */
        0xC1E7, 0xC1E8, 0xC1E9, 0xC1EA, 0xC1EB, 0xC1EC, 0xC1ED, 0xC1EE, /* 0x40 */
        0xC1EF, 0xC1F0, 	 0, 0xC1F1, 0xC1F2, 0xC1F3, 0xC1F4, 0xC1F5, /* 0x50 */
        0xC1F6, 0xC1F7, 0xC1F8, 0xC1F9, 0xC1FA, 0xC1FB, 	 0, 	 0, /* 0x50 */
        0, 0xC1FC, 0xC1FD, 0xC1FE, 0xC1FF, 0xC200, 0xC201, 0xC202, /* 0x60 */
        0xC203, 0xC204, 0xC205, 0xC206, 0xC207, 0xC208, 0xC209, 0xC20A, /* 0x60 */
        0xC20B, 0xC20C, 	 0, 0xC20D, 0xC20E, 0xC20F, 0xC210, 0xC211, /* 0x70 */
        0xC212, 0xC213, 0xC214, 0xC215, 0xC216, 0xC217, 	 0, 	 0, /* 0x70 */
        0, 0xC218, 0xC219, 0xC21A, 0xC21B, 0xC21C, 0xC21D, 0xC21E, /* 0x80 */
        0xC21F, 0xC220, 0xC221, 0xC222, 0xC223, 0xC224, 0xC225, 0xC226, /* 0x80 */
        0xC227, 0xC228, 	 0, 0xC229, 0xC22A, 0xC22B, 0xC22C, 0xC22D, /* 0x90 */
        0xC22E, 0xC22F, 0xC230, 0xC231, 0xC232, 0xC233, 	 0, 	 0, /* 0x90 */
        0, 0xC234, 0xC235, 0xC236, 0xC237, 0xC238, 0xC239, 0xC23A, /* 0xA0 */
        0xC23B, 0xC23C, 0xC23D, 0xC23E, 0xC23F, 0xC240, 0xC241, 0xC242, /* 0xA0 */
        0xC243, 0xC244, 	 0, 0xC245, 0xC246, 0xC247, 0xC248, 0xC249, /* 0xB0 */
        0xC24A, 0xC24B, 0xC24C, 0xC24D, 0xC24E, 0xC24F, 	 0, 	 0, /* 0xB0 */
        0, 0xC250, 0xC251, 0xC252, 0xC253, 0xC254, 0xC255, 0xC256, /* 0xC0 */
        0xC257, 0xC258, 0xC259, 0xC25A, 0xC25B, 0xC25C, 0xC25D, 0xC25E, /* 0xC0 */
        0xC25F, 0xC260, 	 0, 0xC261, 0xC262, 0xC263, 0xC264, 0xC265, /* 0xD0 */
        0xC266, 0xC267, 0xC268, 0xC269, 0xC26A, 0xC26B, 	 0, 	 0, /* 0xD0 */
        0, 0xC26C, 0xC26D, 0xC26E, 0xC26F, 0xC270, 0xC271, 0xC272, /* 0xE0 */
        0xC273, 0xC274, 0xC275, 0xC276, 0xC277, 0xC278, 0xC279, 0xC27A, /* 0xE0 */
        0xC27B, 0xC27C, 	 0, 0xC27D, 0xC27E, 0xC27F, 0xC280, 0xC281, /* 0xF0 */
        0xC282, 0xC283, 0xC284, 0xC285, 0xC286, 0xC287					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xaf:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xC288, 0xC289, 0xC28A, 0xC28B, 0xC28C, 0xC28D, 0xC28E, /* 0x40 */
        0xC28F, 0xC290, 0xC291, 0xC292, 0xC293, 0xC294, 0xC295, 0xC296, /* 0x40 */
        0xC297, 0xC298, 	 0, 0xC299, 0xC29A, 0xC29B, 0xC29C, 0xC29D, /* 0x50 */
        0xC29E, 0xC29F, 0xC2A0, 0xC2A1, 0xC2A2, 0xC2A3, 	 0, 	 0, /* 0x50 */
        0, 0xC2A4, 0xC2A5, 0xC2A6, 0xC2A7, 0xC2A8, 0xC2A9, 0xC2AA, /* 0x60 */
        0xC2AB, 0xC2AC, 0xC2AD, 0xC2AE, 0xC2AF, 0xC2B0, 0xC2B1, 0xC2B2, /* 0x60 */
        0xC2B3, 0xC2B4, 	 0, 0xC2B5, 0xC2B6, 0xC2B7, 0xC2B8, 0xC2B9, /* 0x70 */
        0xC2BA, 0xC2BB, 0xC2BC, 0xC2BD, 0xC2BE, 0xC2BF, 	 0, 	 0, /* 0x70 */
        0, 0xC2C0, 0xC2C1, 0xC2C2, 0xC2C3, 0xC2C4, 0xC2C5, 0xC2C6, /* 0x80 */
        0xC2C7, 0xC2C8, 0xC2C9, 0xC2CA, 0xC2CB, 0xC2CC, 0xC2CD, 0xC2CE, /* 0x80 */
        0xC2CF, 0xC2D0, 	 0, 0xC2D1, 0xC2D2, 0xC2D3, 0xC2D4, 0xC2D5, /* 0x90 */
        0xC2D6, 0xC2D7, 0xC2D8, 0xC2D9, 0xC2DA, 0xC2DB, 	 0, 	 0, /* 0x90 */
        0, 0xC2DC, 0xC2DD, 0xC2DE, 0xC2DF, 0xC2E0, 0xC2E1, 0xC2E2, /* 0xA0 */
        0xC2E3, 0xC2E4, 0xC2E5, 0xC2E6, 0xC2E7, 0xC2E8, 0xC2E9, 0xC2EA, /* 0xA0 */
        0xC2EB, 0xC2EC, 	 0, 0xC2ED, 0xC2EE, 0xC2EF, 0xC2F0, 0xC2F1, /* 0xB0 */
        0xC2F2, 0xC2F3, 0xC2F4, 0xC2F5, 0xC2F6, 0xC2F7					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb0:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3146, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xC2F8, 0xC2F9, 0xC2FA, 0xC2FB, 0xC2FC, 0xC2FD, 0xC2FE, /* 0x60 */
        0xC2FF, 0xC300, 0xC301, 0xC302, 0xC303, 0xC304, 0xC305, 0xC306, /* 0x60 */
        0xC307, 0xC308, 	 0, 0xC309, 0xC30A, 0xC30B, 0xC30C, 0xC30D, /* 0x70 */
        0xC30E, 0xC30F, 0xC310, 0xC311, 0xC312, 0xC313, 	 0, 	 0, /* 0x70 */
        0, 0xC314, 0xC315, 0xC316, 0xC317, 0xC318, 0xC319, 0xC31A, /* 0x80 */
        0xC31B, 0xC31C, 0xC31D, 0xC31E, 0xC31F, 0xC320, 0xC321, 0xC322, /* 0x80 */
        0xC323, 0xC324, 	 0, 0xC325, 0xC326, 0xC327, 0xC328, 0xC329, /* 0x90 */
        0xC32A, 0xC32B, 0xC32C, 0xC32D, 0xC32E, 0xC32F, 	 0, 	 0, /* 0x90 */
        0, 0xC330, 0xC331, 0xC332, 0xC333, 0xC334, 0xC335, 0xC336, /* 0xA0 */
        0xC337, 0xC338, 0xC339, 0xC33A, 0xC33B, 0xC33C, 0xC33D, 0xC33E, /* 0xA0 */
        0xC33F, 0xC340, 	 0, 0xC341, 0xC342, 0xC343, 0xC344, 0xC345, /* 0xB0 */
        0xC346, 0xC347, 0xC348, 0xC349, 0xC34A, 0xC34B, 	 0, 	 0, /* 0xB0 */
        0, 0xC34C, 0xC34D, 0xC34E, 0xC34F, 0xC350, 0xC351, 0xC352, /* 0xC0 */
        0xC353, 0xC354, 0xC355, 0xC356, 0xC357, 0xC358, 0xC359, 0xC35A, /* 0xC0 */
        0xC35B, 0xC35C, 	 0, 0xC35D, 0xC35E, 0xC35F, 0xC360, 0xC361, /* 0xD0 */
        0xC362, 0xC363, 0xC364, 0xC365, 0xC366, 0xC367, 	 0, 	 0, /* 0xD0 */
        0, 0xC368, 0xC369, 0xC36A, 0xC36B, 0xC36C, 0xC36D, 0xC36E, /* 0xE0 */
        0xC36F, 0xC370, 0xC371, 0xC372, 0xC373, 0xC374, 0xC375, 0xC376, /* 0xE0 */
        0xC377, 0xC378, 	 0, 0xC379, 0xC37A, 0xC37B, 0xC37C, 0xC37D, /* 0xF0 */
        0xC37E, 0xC37F, 0xC380, 0xC381, 0xC382, 0xC383					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb1:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC384, 0xC385, 0xC386, 0xC387, 0xC388, 0xC389, 0xC38A, /* 0x40 */
        0xC38B, 0xC38C, 0xC38D, 0xC38E, 0xC38F, 0xC390, 0xC391, 0xC392, /* 0x40 */
        0xC393, 0xC394, 	 0, 0xC395, 0xC396, 0xC397, 0xC398, 0xC399, /* 0x50 */
        0xC39A, 0xC39B, 0xC39C, 0xC39D, 0xC39E, 0xC39F, 	 0, 	 0, /* 0x50 */
        0, 0xC3A0, 0xC3A1, 0xC3A2, 0xC3A3, 0xC3A4, 0xC3A5, 0xC3A6, /* 0x60 */
        0xC3A7, 0xC3A8, 0xC3A9, 0xC3AA, 0xC3AB, 0xC3AC, 0xC3AD, 0xC3AE, /* 0x60 */
        0xC3AF, 0xC3B0, 	 0, 0xC3B1, 0xC3B2, 0xC3B3, 0xC3B4, 0xC3B5, /* 0x70 */
        0xC3B6, 0xC3B7, 0xC3B8, 0xC3B9, 0xC3BA, 0xC3BB, 	 0, 	 0, /* 0x70 */
        0, 0xC3BC, 0xC3BD, 0xC3BE, 0xC3BF, 0xC3C0, 0xC3C1, 0xC3C2, /* 0x80 */
        0xC3C3, 0xC3C4, 0xC3C5, 0xC3C6, 0xC3C7, 0xC3C8, 0xC3C9, 0xC3CA, /* 0x80 */
        0xC3CB, 0xC3CC, 	 0, 0xC3CD, 0xC3CE, 0xC3CF, 0xC3D0, 0xC3D1, /* 0x90 */
        0xC3D2, 0xC3D3, 0xC3D4, 0xC3D5, 0xC3D6, 0xC3D7, 	 0, 	 0, /* 0x90 */
        0, 0xC3D8, 0xC3D9, 0xC3DA, 0xC3DB, 0xC3DC, 0xC3DD, 0xC3DE, /* 0xA0 */
        0xC3DF, 0xC3E0, 0xC3E1, 0xC3E2, 0xC3E3, 0xC3E4, 0xC3E5, 0xC3E6, /* 0xA0 */
        0xC3E7, 0xC3E8, 	 0, 0xC3E9, 0xC3EA, 0xC3EB, 0xC3EC, 0xC3ED, /* 0xB0 */
        0xC3EE, 0xC3EF, 0xC3F0, 0xC3F1, 0xC3F2, 0xC3F3, 	 0, 	 0, /* 0xB0 */
        0, 0xC3F4, 0xC3F5, 0xC3F6, 0xC3F7, 0xC3F8, 0xC3F9, 0xC3FA, /* 0xC0 */
        0xC3FB, 0xC3FC, 0xC3FD, 0xC3FE, 0xC3FF, 0xC400, 0xC401, 0xC402, /* 0xC0 */
        0xC403, 0xC404, 	 0, 0xC405, 0xC406, 0xC407, 0xC408, 0xC409, /* 0xD0 */
        0xC40A, 0xC40B, 0xC40C, 0xC40D, 0xC40E, 0xC40F, 	 0, 	 0, /* 0xD0 */
        0, 0xC410, 0xC411, 0xC412, 0xC413, 0xC414, 0xC415, 0xC416, /* 0xE0 */
        0xC417, 0xC418, 0xC419, 0xC41A, 0xC41B, 0xC41C, 0xC41D, 0xC41E, /* 0xE0 */
        0xC41F, 0xC420, 	 0, 0xC421, 0xC422, 0xC423, 0xC424, 0xC425, /* 0xF0 */
        0xC426, 0xC427, 0xC428, 0xC429, 0xC42A, 0xC42B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb2:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC42C, 0xC42D, 0xC42E, 0xC42F, 0xC430, 0xC431, 0xC432, /* 0x40 */
        0xC433, 0xC434, 0xC435, 0xC436, 0xC437, 0xC438, 0xC439, 0xC43A, /* 0x40 */
        0xC43B, 0xC43C, 	 0, 0xC43D, 0xC43E, 0xC43F, 0xC440, 0xC441, /* 0x50 */
        0xC442, 0xC443, 0xC444, 0xC445, 0xC446, 0xC447, 	 0, 	 0, /* 0x50 */
        0, 0xC448, 0xC449, 0xC44A, 0xC44B, 0xC44C, 0xC44D, 0xC44E, /* 0x60 */
        0xC44F, 0xC450, 0xC451, 0xC452, 0xC453, 0xC454, 0xC455, 0xC456, /* 0x60 */
        0xC457, 0xC458, 	 0, 0xC459, 0xC45A, 0xC45B, 0xC45C, 0xC45D, /* 0x70 */
        0xC45E, 0xC45F, 0xC460, 0xC461, 0xC462, 0xC463, 	 0, 	 0, /* 0x70 */
        0, 0xC464, 0xC465, 0xC466, 0xC467, 0xC468, 0xC469, 0xC46A, /* 0x80 */
        0xC46B, 0xC46C, 0xC46D, 0xC46E, 0xC46F, 0xC470, 0xC471, 0xC472, /* 0x80 */
        0xC473, 0xC474, 	 0, 0xC475, 0xC476, 0xC477, 0xC478, 0xC479, /* 0x90 */
        0xC47A, 0xC47B, 0xC47C, 0xC47D, 0xC47E, 0xC47F, 	 0, 	 0, /* 0x90 */
        0, 0xC480, 0xC481, 0xC482, 0xC483, 0xC484, 0xC485, 0xC486, /* 0xA0 */
        0xC487, 0xC488, 0xC489, 0xC48A, 0xC48B, 0xC48C, 0xC48D, 0xC48E, /* 0xA0 */
        0xC48F, 0xC490, 	 0, 0xC491, 0xC492, 0xC493, 0xC494, 0xC495, /* 0xB0 */
        0xC496, 0xC497, 0xC498, 0xC499, 0xC49A, 0xC49B, 	 0, 	 0, /* 0xB0 */
        0, 0xC49C, 0xC49D, 0xC49E, 0xC49F, 0xC4A0, 0xC4A1, 0xC4A2, /* 0xC0 */
        0xC4A3, 0xC4A4, 0xC4A5, 0xC4A6, 0xC4A7, 0xC4A8, 0xC4A9, 0xC4AA, /* 0xC0 */
        0xC4AB, 0xC4AC, 	 0, 0xC4AD, 0xC4AE, 0xC4AF, 0xC4B0, 0xC4B1, /* 0xD0 */
        0xC4B2, 0xC4B3, 0xC4B4, 0xC4B5, 0xC4B6, 0xC4B7, 	 0, 	 0, /* 0xD0 */
        0, 0xC4B8, 0xC4B9, 0xC4BA, 0xC4BB, 0xC4BC, 0xC4BD, 0xC4BE, /* 0xE0 */
        0xC4BF, 0xC4C0, 0xC4C1, 0xC4C2, 0xC4C3, 0xC4C4, 0xC4C5, 0xC4C6, /* 0xE0 */
        0xC4C7, 0xC4C8, 	 0, 0xC4C9, 0xC4CA, 0xC4CB, 0xC4CC, 0xC4CD, /* 0xF0 */
        0xC4CE, 0xC4CF, 0xC4D0, 0xC4D1, 0xC4D2, 0xC4D3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb3:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xC4D4, 0xC4D5, 0xC4D6, 0xC4D7, 0xC4D8, 0xC4D9, 0xC4DA, /* 0x40 */
        0xC4DB, 0xC4DC, 0xC4DD, 0xC4DE, 0xC4DF, 0xC4E0, 0xC4E1, 0xC4E2, /* 0x40 */
        0xC4E3, 0xC4E4, 	 0, 0xC4E5, 0xC4E6, 0xC4E7, 0xC4E8, 0xC4E9, /* 0x50 */
        0xC4EA, 0xC4EB, 0xC4EC, 0xC4ED, 0xC4EE, 0xC4EF, 	 0, 	 0, /* 0x50 */
        0, 0xC4F0, 0xC4F1, 0xC4F2, 0xC4F3, 0xC4F4, 0xC4F5, 0xC4F6, /* 0x60 */
        0xC4F7, 0xC4F8, 0xC4F9, 0xC4FA, 0xC4FB, 0xC4FC, 0xC4FD, 0xC4FE, /* 0x60 */
        0xC4FF, 0xC500, 	 0, 0xC501, 0xC502, 0xC503, 0xC504, 0xC505, /* 0x70 */
        0xC506, 0xC507, 0xC508, 0xC509, 0xC50A, 0xC50B, 	 0, 	 0, /* 0x70 */
        0, 0xC50C, 0xC50D, 0xC50E, 0xC50F, 0xC510, 0xC511, 0xC512, /* 0x80 */
        0xC513, 0xC514, 0xC515, 0xC516, 0xC517, 0xC518, 0xC519, 0xC51A, /* 0x80 */
        0xC51B, 0xC51C, 	 0, 0xC51D, 0xC51E, 0xC51F, 0xC520, 0xC521, /* 0x90 */
        0xC522, 0xC523, 0xC524, 0xC525, 0xC526, 0xC527, 	 0, 	 0, /* 0x90 */
        0, 0xC528, 0xC529, 0xC52A, 0xC52B, 0xC52C, 0xC52D, 0xC52E, /* 0xA0 */
        0xC52F, 0xC530, 0xC531, 0xC532, 0xC533, 0xC534, 0xC535, 0xC536, /* 0xA0 */
        0xC537, 0xC538, 	 0, 0xC539, 0xC53A, 0xC53B, 0xC53C, 0xC53D, /* 0xB0 */
        0xC53E, 0xC53F, 0xC540, 0xC541, 0xC542, 0xC543					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb4:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3147, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xC544, 0xC545, 0xC546, 0xC547, 0xC548, 0xC549, 0xC54A, /* 0x60 */
        0xC54B, 0xC54C, 0xC54D, 0xC54E, 0xC54F, 0xC550, 0xC551, 0xC552, /* 0x60 */
        0xC553, 0xC554, 	 0, 0xC555, 0xC556, 0xC557, 0xC558, 0xC559, /* 0x70 */
        0xC55A, 0xC55B, 0xC55C, 0xC55D, 0xC55E, 0xC55F, 	 0, 	 0, /* 0x70 */
        0, 0xC560, 0xC561, 0xC562, 0xC563, 0xC564, 0xC565, 0xC566, /* 0x80 */
        0xC567, 0xC568, 0xC569, 0xC56A, 0xC56B, 0xC56C, 0xC56D, 0xC56E, /* 0x80 */
        0xC56F, 0xC570, 	 0, 0xC571, 0xC572, 0xC573, 0xC574, 0xC575, /* 0x90 */
        0xC576, 0xC577, 0xC578, 0xC579, 0xC57A, 0xC57B, 	 0, 	 0, /* 0x90 */
        0, 0xC57C, 0xC57D, 0xC57E, 0xC57F, 0xC580, 0xC581, 0xC582, /* 0xA0 */
        0xC583, 0xC584, 0xC585, 0xC586, 0xC587, 0xC588, 0xC589, 0xC58A, /* 0xA0 */
        0xC58B, 0xC58C, 	 0, 0xC58D, 0xC58E, 0xC58F, 0xC590, 0xC591, /* 0xB0 */
        0xC592, 0xC593, 0xC594, 0xC595, 0xC596, 0xC597, 	 0, 	 0, /* 0xB0 */
        0, 0xC598, 0xC599, 0xC59A, 0xC59B, 0xC59C, 0xC59D, 0xC59E, /* 0xC0 */
        0xC59F, 0xC5A0, 0xC5A1, 0xC5A2, 0xC5A3, 0xC5A4, 0xC5A5, 0xC5A6, /* 0xC0 */
        0xC5A7, 0xC5A8, 	 0, 0xC5A9, 0xC5AA, 0xC5AB, 0xC5AC, 0xC5AD, /* 0xD0 */
        0xC5AE, 0xC5AF, 0xC5B0, 0xC5B1, 0xC5B2, 0xC5B3, 	 0, 	 0, /* 0xD0 */
        0, 0xC5B4, 0xC5B5, 0xC5B6, 0xC5B7, 0xC5B8, 0xC5B9, 0xC5BA, /* 0xE0 */
        0xC5BB, 0xC5BC, 0xC5BD, 0xC5BE, 0xC5BF, 0xC5C0, 0xC5C1, 0xC5C2, /* 0xE0 */
        0xC5C3, 0xC5C4, 	 0, 0xC5C5, 0xC5C6, 0xC5C7, 0xC5C8, 0xC5C9, /* 0xF0 */
        0xC5CA, 0xC5CB, 0xC5CC, 0xC5CD, 0xC5CE, 0xC5CF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb5:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC5D0, 0xC5D1, 0xC5D2, 0xC5D3, 0xC5D4, 0xC5D5, 0xC5D6, /* 0x40 */
        0xC5D7, 0xC5D8, 0xC5D9, 0xC5DA, 0xC5DB, 0xC5DC, 0xC5DD, 0xC5DE, /* 0x40 */
        0xC5DF, 0xC5E0, 	 0, 0xC5E1, 0xC5E2, 0xC5E3, 0xC5E4, 0xC5E5, /* 0x50 */
        0xC5E6, 0xC5E7, 0xC5E8, 0xC5E9, 0xC5EA, 0xC5EB, 	 0, 	 0, /* 0x50 */
        0, 0xC5EC, 0xC5ED, 0xC5EE, 0xC5EF, 0xC5F0, 0xC5F1, 0xC5F2, /* 0x60 */
        0xC5F3, 0xC5F4, 0xC5F5, 0xC5F6, 0xC5F7, 0xC5F8, 0xC5F9, 0xC5FA, /* 0x60 */
        0xC5FB, 0xC5FC, 	 0, 0xC5FD, 0xC5FE, 0xC5FF, 0xC600, 0xC601, /* 0x70 */
        0xC602, 0xC603, 0xC604, 0xC605, 0xC606, 0xC607, 	 0, 	 0, /* 0x70 */
        0, 0xC608, 0xC609, 0xC60A, 0xC60B, 0xC60C, 0xC60D, 0xC60E, /* 0x80 */
        0xC60F, 0xC610, 0xC611, 0xC612, 0xC613, 0xC614, 0xC615, 0xC616, /* 0x80 */
        0xC617, 0xC618, 	 0, 0xC619, 0xC61A, 0xC61B, 0xC61C, 0xC61D, /* 0x90 */
        0xC61E, 0xC61F, 0xC620, 0xC621, 0xC622, 0xC623, 	 0, 	 0, /* 0x90 */
        0, 0xC624, 0xC625, 0xC626, 0xC627, 0xC628, 0xC629, 0xC62A, /* 0xA0 */
        0xC62B, 0xC62C, 0xC62D, 0xC62E, 0xC62F, 0xC630, 0xC631, 0xC632, /* 0xA0 */
        0xC633, 0xC634, 	 0, 0xC635, 0xC636, 0xC637, 0xC638, 0xC639, /* 0xB0 */
        0xC63A, 0xC63B, 0xC63C, 0xC63D, 0xC63E, 0xC63F, 	 0, 	 0, /* 0xB0 */
        0, 0xC640, 0xC641, 0xC642, 0xC643, 0xC644, 0xC645, 0xC646, /* 0xC0 */
        0xC647, 0xC648, 0xC649, 0xC64A, 0xC64B, 0xC64C, 0xC64D, 0xC64E, /* 0xC0 */
        0xC64F, 0xC650, 	 0, 0xC651, 0xC652, 0xC653, 0xC654, 0xC655, /* 0xD0 */
        0xC656, 0xC657, 0xC658, 0xC659, 0xC65A, 0xC65B, 	 0, 	 0, /* 0xD0 */
        0, 0xC65C, 0xC65D, 0xC65E, 0xC65F, 0xC660, 0xC661, 0xC662, /* 0xE0 */
        0xC663, 0xC664, 0xC665, 0xC666, 0xC667, 0xC668, 0xC669, 0xC66A, /* 0xE0 */
        0xC66B, 0xC66C, 	 0, 0xC66D, 0xC66E, 0xC66F, 0xC670, 0xC671, /* 0xF0 */
        0xC672, 0xC673, 0xC674, 0xC675, 0xC676, 0xC677					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb6:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC678, 0xC679, 0xC67A, 0xC67B, 0xC67C, 0xC67D, 0xC67E, /* 0x40 */
        0xC67F, 0xC680, 0xC681, 0xC682, 0xC683, 0xC684, 0xC685, 0xC686, /* 0x40 */
        0xC687, 0xC688, 	 0, 0xC689, 0xC68A, 0xC68B, 0xC68C, 0xC68D, /* 0x50 */
        0xC68E, 0xC68F, 0xC690, 0xC691, 0xC692, 0xC693, 	 0, 	 0, /* 0x50 */
        0, 0xC694, 0xC695, 0xC696, 0xC697, 0xC698, 0xC699, 0xC69A, /* 0x60 */
        0xC69B, 0xC69C, 0xC69D, 0xC69E, 0xC69F, 0xC6A0, 0xC6A1, 0xC6A2, /* 0x60 */
        0xC6A3, 0xC6A4, 	 0, 0xC6A5, 0xC6A6, 0xC6A7, 0xC6A8, 0xC6A9, /* 0x70 */
        0xC6AA, 0xC6AB, 0xC6AC, 0xC6AD, 0xC6AE, 0xC6AF, 	 0, 	 0, /* 0x70 */
        0, 0xC6B0, 0xC6B1, 0xC6B2, 0xC6B3, 0xC6B4, 0xC6B5, 0xC6B6, /* 0x80 */
        0xC6B7, 0xC6B8, 0xC6B9, 0xC6BA, 0xC6BB, 0xC6BC, 0xC6BD, 0xC6BE, /* 0x80 */
        0xC6BF, 0xC6C0, 	 0, 0xC6C1, 0xC6C2, 0xC6C3, 0xC6C4, 0xC6C5, /* 0x90 */
        0xC6C6, 0xC6C7, 0xC6C8, 0xC6C9, 0xC6CA, 0xC6CB, 	 0, 	 0, /* 0x90 */
        0, 0xC6CC, 0xC6CD, 0xC6CE, 0xC6CF, 0xC6D0, 0xC6D1, 0xC6D2, /* 0xA0 */
        0xC6D3, 0xC6D4, 0xC6D5, 0xC6D6, 0xC6D7, 0xC6D8, 0xC6D9, 0xC6DA, /* 0xA0 */
        0xC6DB, 0xC6DC, 	 0, 0xC6DD, 0xC6DE, 0xC6DF, 0xC6E0, 0xC6E1, /* 0xB0 */
        0xC6E2, 0xC6E3, 0xC6E4, 0xC6E5, 0xC6E6, 0xC6E7, 	 0, 	 0, /* 0xB0 */
        0, 0xC6E8, 0xC6E9, 0xC6EA, 0xC6EB, 0xC6EC, 0xC6ED, 0xC6EE, /* 0xC0 */
        0xC6EF, 0xC6F0, 0xC6F1, 0xC6F2, 0xC6F3, 0xC6F4, 0xC6F5, 0xC6F6, /* 0xC0 */
        0xC6F7, 0xC6F8, 	 0, 0xC6F9, 0xC6FA, 0xC6FB, 0xC6FC, 0xC6FD, /* 0xD0 */
        0xC6FE, 0xC6FF, 0xC700, 0xC701, 0xC702, 0xC703, 	 0, 	 0, /* 0xD0 */
        0, 0xC704, 0xC705, 0xC706, 0xC707, 0xC708, 0xC709, 0xC70A, /* 0xE0 */
        0xC70B, 0xC70C, 0xC70D, 0xC70E, 0xC70F, 0xC710, 0xC711, 0xC712, /* 0xE0 */
        0xC713, 0xC714, 	 0, 0xC715, 0xC716, 0xC717, 0xC718, 0xC719, /* 0xF0 */
        0xC71A, 0xC71B, 0xC71C, 0xC71D, 0xC71E, 0xC71F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb7:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xC720, 0xC721, 0xC722, 0xC723, 0xC724, 0xC725, 0xC726, /* 0x40 */
        0xC727, 0xC728, 0xC729, 0xC72A, 0xC72B, 0xC72C, 0xC72D, 0xC72E, /* 0x40 */
        0xC72F, 0xC730, 	 0, 0xC731, 0xC732, 0xC733, 0xC734, 0xC735, /* 0x50 */
        0xC736, 0xC737, 0xC738, 0xC739, 0xC73A, 0xC73B, 	 0, 	 0, /* 0x50 */
        0, 0xC73C, 0xC73D, 0xC73E, 0xC73F, 0xC740, 0xC741, 0xC742, /* 0x60 */
        0xC743, 0xC744, 0xC745, 0xC746, 0xC747, 0xC748, 0xC749, 0xC74A, /* 0x60 */
        0xC74B, 0xC74C, 	 0, 0xC74D, 0xC74E, 0xC74F, 0xC750, 0xC751, /* 0x70 */
        0xC752, 0xC753, 0xC754, 0xC755, 0xC756, 0xC757, 	 0, 	 0, /* 0x70 */
        0, 0xC758, 0xC759, 0xC75A, 0xC75B, 0xC75C, 0xC75D, 0xC75E, /* 0x80 */
        0xC75F, 0xC760, 0xC761, 0xC762, 0xC763, 0xC764, 0xC765, 0xC766, /* 0x80 */
        0xC767, 0xC768, 	 0, 0xC769, 0xC76A, 0xC76B, 0xC76C, 0xC76D, /* 0x90 */
        0xC76E, 0xC76F, 0xC770, 0xC771, 0xC772, 0xC773, 	 0, 	 0, /* 0x90 */
        0, 0xC774, 0xC775, 0xC776, 0xC777, 0xC778, 0xC779, 0xC77A, /* 0xA0 */
        0xC77B, 0xC77C, 0xC77D, 0xC77E, 0xC77F, 0xC780, 0xC781, 0xC782, /* 0xA0 */
        0xC783, 0xC784, 	 0, 0xC785, 0xC786, 0xC787, 0xC788, 0xC789, /* 0xB0 */
        0xC78A, 0xC78B, 0xC78C, 0xC78D, 0xC78E, 0xC78F					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb8:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3148, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xC790, 0xC791, 0xC792, 0xC793, 0xC794, 0xC795, 0xC796, /* 0x60 */
        0xC797, 0xC798, 0xC799, 0xC79A, 0xC79B, 0xC79C, 0xC79D, 0xC79E, /* 0x60 */
        0xC79F, 0xC7A0, 	 0, 0xC7A1, 0xC7A2, 0xC7A3, 0xC7A4, 0xC7A5, /* 0x70 */
        0xC7A6, 0xC7A7, 0xC7A8, 0xC7A9, 0xC7AA, 0xC7AB, 	 0, 	 0, /* 0x70 */
        0, 0xC7AC, 0xC7AD, 0xC7AE, 0xC7AF, 0xC7B0, 0xC7B1, 0xC7B2, /* 0x80 */
        0xC7B3, 0xC7B4, 0xC7B5, 0xC7B6, 0xC7B7, 0xC7B8, 0xC7B9, 0xC7BA, /* 0x80 */
        0xC7BB, 0xC7BC, 	 0, 0xC7BD, 0xC7BE, 0xC7BF, 0xC7C0, 0xC7C1, /* 0x90 */
        0xC7C2, 0xC7C3, 0xC7C4, 0xC7C5, 0xC7C6, 0xC7C7, 	 0, 	 0, /* 0x90 */
        0, 0xC7C8, 0xC7C9, 0xC7CA, 0xC7CB, 0xC7CC, 0xC7CD, 0xC7CE, /* 0xA0 */
        0xC7CF, 0xC7D0, 0xC7D1, 0xC7D2, 0xC7D3, 0xC7D4, 0xC7D5, 0xC7D6, /* 0xA0 */
        0xC7D7, 0xC7D8, 	 0, 0xC7D9, 0xC7DA, 0xC7DB, 0xC7DC, 0xC7DD, /* 0xB0 */
        0xC7DE, 0xC7DF, 0xC7E0, 0xC7E1, 0xC7E2, 0xC7E3, 	 0, 	 0, /* 0xB0 */
        0, 0xC7E4, 0xC7E5, 0xC7E6, 0xC7E7, 0xC7E8, 0xC7E9, 0xC7EA, /* 0xC0 */
        0xC7EB, 0xC7EC, 0xC7ED, 0xC7EE, 0xC7EF, 0xC7F0, 0xC7F1, 0xC7F2, /* 0xC0 */
        0xC7F3, 0xC7F4, 	 0, 0xC7F5, 0xC7F6, 0xC7F7, 0xC7F8, 0xC7F9, /* 0xD0 */
        0xC7FA, 0xC7FB, 0xC7FC, 0xC7FD, 0xC7FE, 0xC7FF, 	 0, 	 0, /* 0xD0 */
        0, 0xC800, 0xC801, 0xC802, 0xC803, 0xC804, 0xC805, 0xC806, /* 0xE0 */
        0xC807, 0xC808, 0xC809, 0xC80A, 0xC80B, 0xC80C, 0xC80D, 0xC80E, /* 0xE0 */
        0xC80F, 0xC810, 	 0, 0xC811, 0xC812, 0xC813, 0xC814, 0xC815, /* 0xF0 */
        0xC816, 0xC817, 0xC818, 0xC819, 0xC81A, 0xC81B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xb9:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC81C, 0xC81D, 0xC81E, 0xC81F, 0xC820, 0xC821, 0xC822, /* 0x40 */
        0xC823, 0xC824, 0xC825, 0xC826, 0xC827, 0xC828, 0xC829, 0xC82A, /* 0x40 */
        0xC82B, 0xC82C, 	 0, 0xC82D, 0xC82E, 0xC82F, 0xC830, 0xC831, /* 0x50 */
        0xC832, 0xC833, 0xC834, 0xC835, 0xC836, 0xC837, 	 0, 	 0, /* 0x50 */
        0, 0xC838, 0xC839, 0xC83A, 0xC83B, 0xC83C, 0xC83D, 0xC83E, /* 0x60 */
        0xC83F, 0xC840, 0xC841, 0xC842, 0xC843, 0xC844, 0xC845, 0xC846, /* 0x60 */
        0xC847, 0xC848, 	 0, 0xC849, 0xC84A, 0xC84B, 0xC84C, 0xC84D, /* 0x70 */
        0xC84E, 0xC84F, 0xC850, 0xC851, 0xC852, 0xC853, 	 0, 	 0, /* 0x70 */
        0, 0xC854, 0xC855, 0xC856, 0xC857, 0xC858, 0xC859, 0xC85A, /* 0x80 */
        0xC85B, 0xC85C, 0xC85D, 0xC85E, 0xC85F, 0xC860, 0xC861, 0xC862, /* 0x80 */
        0xC863, 0xC864, 	 0, 0xC865, 0xC866, 0xC867, 0xC868, 0xC869, /* 0x90 */
        0xC86A, 0xC86B, 0xC86C, 0xC86D, 0xC86E, 0xC86F, 	 0, 	 0, /* 0x90 */
        0, 0xC870, 0xC871, 0xC872, 0xC873, 0xC874, 0xC875, 0xC876, /* 0xA0 */
        0xC877, 0xC878, 0xC879, 0xC87A, 0xC87B, 0xC87C, 0xC87D, 0xC87E, /* 0xA0 */
        0xC87F, 0xC880, 	 0, 0xC881, 0xC882, 0xC883, 0xC884, 0xC885, /* 0xB0 */
        0xC886, 0xC887, 0xC888, 0xC889, 0xC88A, 0xC88B, 	 0, 	 0, /* 0xB0 */
        0, 0xC88C, 0xC88D, 0xC88E, 0xC88F, 0xC890, 0xC891, 0xC892, /* 0xC0 */
        0xC893, 0xC894, 0xC895, 0xC896, 0xC897, 0xC898, 0xC899, 0xC89A, /* 0xC0 */
        0xC89B, 0xC89C, 	 0, 0xC89D, 0xC89E, 0xC89F, 0xC8A0, 0xC8A1, /* 0xD0 */
        0xC8A2, 0xC8A3, 0xC8A4, 0xC8A5, 0xC8A6, 0xC8A7, 	 0, 	 0, /* 0xD0 */
        0, 0xC8A8, 0xC8A9, 0xC8AA, 0xC8AB, 0xC8AC, 0xC8AD, 0xC8AE, /* 0xE0 */
        0xC8AF, 0xC8B0, 0xC8B1, 0xC8B2, 0xC8B3, 0xC8B4, 0xC8B5, 0xC8B6, /* 0xE0 */
        0xC8B7, 0xC8B8, 	 0, 0xC8B9, 0xC8BA, 0xC8BB, 0xC8BC, 0xC8BD, /* 0xF0 */
        0xC8BE, 0xC8BF, 0xC8C0, 0xC8C1, 0xC8C2, 0xC8C3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xba:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xC8C4, 0xC8C5, 0xC8C6, 0xC8C7, 0xC8C8, 0xC8C9, 0xC8CA, /* 0x40 */
        0xC8CB, 0xC8CC, 0xC8CD, 0xC8CE, 0xC8CF, 0xC8D0, 0xC8D1, 0xC8D2, /* 0x40 */
        0xC8D3, 0xC8D4, 	 0, 0xC8D5, 0xC8D6, 0xC8D7, 0xC8D8, 0xC8D9, /* 0x50 */
        0xC8DA, 0xC8DB, 0xC8DC, 0xC8DD, 0xC8DE, 0xC8DF, 	 0, 	 0, /* 0x50 */
        0, 0xC8E0, 0xC8E1, 0xC8E2, 0xC8E3, 0xC8E4, 0xC8E5, 0xC8E6, /* 0x60 */
        0xC8E7, 0xC8E8, 0xC8E9, 0xC8EA, 0xC8EB, 0xC8EC, 0xC8ED, 0xC8EE, /* 0x60 */
        0xC8EF, 0xC8F0, 	 0, 0xC8F1, 0xC8F2, 0xC8F3, 0xC8F4, 0xC8F5, /* 0x70 */
        0xC8F6, 0xC8F7, 0xC8F8, 0xC8F9, 0xC8FA, 0xC8FB, 	 0, 	 0, /* 0x70 */
        0, 0xC8FC, 0xC8FD, 0xC8FE, 0xC8FF, 0xC900, 0xC901, 0xC902, /* 0x80 */
        0xC903, 0xC904, 0xC905, 0xC906, 0xC907, 0xC908, 0xC909, 0xC90A, /* 0x80 */
        0xC90B, 0xC90C, 	 0, 0xC90D, 0xC90E, 0xC90F, 0xC910, 0xC911, /* 0x90 */
        0xC912, 0xC913, 0xC914, 0xC915, 0xC916, 0xC917, 	 0, 	 0, /* 0x90 */
        0, 0xC918, 0xC919, 0xC91A, 0xC91B, 0xC91C, 0xC91D, 0xC91E, /* 0xA0 */
        0xC91F, 0xC920, 0xC921, 0xC922, 0xC923, 0xC924, 0xC925, 0xC926, /* 0xA0 */
        0xC927, 0xC928, 	 0, 0xC929, 0xC92A, 0xC92B, 0xC92C, 0xC92D, /* 0xB0 */
        0xC92E, 0xC92F, 0xC930, 0xC931, 0xC932, 0xC933, 	 0, 	 0, /* 0xB0 */
        0, 0xC934, 0xC935, 0xC936, 0xC937, 0xC938, 0xC939, 0xC93A, /* 0xC0 */
        0xC93B, 0xC93C, 0xC93D, 0xC93E, 0xC93F, 0xC940, 0xC941, 0xC942, /* 0xC0 */
        0xC943, 0xC944, 	 0, 0xC945, 0xC946, 0xC947, 0xC948, 0xC949, /* 0xD0 */
        0xC94A, 0xC94B, 0xC94C, 0xC94D, 0xC94E, 0xC94F, 	 0, 	 0, /* 0xD0 */
        0, 0xC950, 0xC951, 0xC952, 0xC953, 0xC954, 0xC955, 0xC956, /* 0xE0 */
        0xC957, 0xC958, 0xC959, 0xC95A, 0xC95B, 0xC95C, 0xC95D, 0xC95E, /* 0xE0 */
        0xC95F, 0xC960, 	 0, 0xC961, 0xC962, 0xC963, 0xC964, 0xC965, /* 0xF0 */
        0xC966, 0xC967, 0xC968, 0xC969, 0xC96A, 0xC96B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xbb:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xC96C, 0xC96D, 0xC96E, 0xC96F, 0xC970, 0xC971, 0xC972, /* 0x40 */
        0xC973, 0xC974, 0xC975, 0xC976, 0xC977, 0xC978, 0xC979, 0xC97A, /* 0x40 */
        0xC97B, 0xC97C, 	 0, 0xC97D, 0xC97E, 0xC97F, 0xC980, 0xC981, /* 0x50 */
        0xC982, 0xC983, 0xC984, 0xC985, 0xC986, 0xC987, 	 0, 	 0, /* 0x50 */
        0, 0xC988, 0xC989, 0xC98A, 0xC98B, 0xC98C, 0xC98D, 0xC98E, /* 0x60 */
        0xC98F, 0xC990, 0xC991, 0xC992, 0xC993, 0xC994, 0xC995, 0xC996, /* 0x60 */
        0xC997, 0xC998, 	 0, 0xC999, 0xC99A, 0xC99B, 0xC99C, 0xC99D, /* 0x70 */
        0xC99E, 0xC99F, 0xC9A0, 0xC9A1, 0xC9A2, 0xC9A3, 	 0, 	 0, /* 0x70 */
        0, 0xC9A4, 0xC9A5, 0xC9A6, 0xC9A7, 0xC9A8, 0xC9A9, 0xC9AA, /* 0x80 */
        0xC9AB, 0xC9AC, 0xC9AD, 0xC9AE, 0xC9AF, 0xC9B0, 0xC9B1, 0xC9B2, /* 0x80 */
        0xC9B3, 0xC9B4, 	 0, 0xC9B5, 0xC9B6, 0xC9B7, 0xC9B8, 0xC9B9, /* 0x90 */
        0xC9BA, 0xC9BB, 0xC9BC, 0xC9BD, 0xC9BE, 0xC9BF, 	 0, 	 0, /* 0x90 */
        0, 0xC9C0, 0xC9C1, 0xC9C2, 0xC9C3, 0xC9C4, 0xC9C5, 0xC9C6, /* 0xA0 */
        0xC9C7, 0xC9C8, 0xC9C9, 0xC9CA, 0xC9CB, 0xC9CC, 0xC9CD, 0xC9CE, /* 0xA0 */
        0xC9CF, 0xC9D0, 	 0, 0xC9D1, 0xC9D2, 0xC9D3, 0xC9D4, 0xC9D5, /* 0xB0 */
        0xC9D6, 0xC9D7, 0xC9D8, 0xC9D9, 0xC9DA, 0xC9DB					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xbc:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x3149, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xC9DC, 0xC9DD, 0xC9DE, 0xC9DF, 0xC9E0, 0xC9E1, 0xC9E2, /* 0x60 */
        0xC9E3, 0xC9E4, 0xC9E5, 0xC9E6, 0xC9E7, 0xC9E8, 0xC9E9, 0xC9EA, /* 0x60 */
        0xC9EB, 0xC9EC, 	 0, 0xC9ED, 0xC9EE, 0xC9EF, 0xC9F0, 0xC9F1, /* 0x70 */
        0xC9F2, 0xC9F3, 0xC9F4, 0xC9F5, 0xC9F6, 0xC9F7, 	 0, 	 0, /* 0x70 */
        0, 0xC9F8, 0xC9F9, 0xC9FA, 0xC9FB, 0xC9FC, 0xC9FD, 0xC9FE, /* 0x80 */
        0xC9FF, 0xCA00, 0xCA01, 0xCA02, 0xCA03, 0xCA04, 0xCA05, 0xCA06, /* 0x80 */
        0xCA07, 0xCA08, 	 0, 0xCA09, 0xCA0A, 0xCA0B, 0xCA0C, 0xCA0D, /* 0x90 */
        0xCA0E, 0xCA0F, 0xCA10, 0xCA11, 0xCA12, 0xCA13, 	 0, 	 0, /* 0x90 */
        0, 0xCA14, 0xCA15, 0xCA16, 0xCA17, 0xCA18, 0xCA19, 0xCA1A, /* 0xA0 */
        0xCA1B, 0xCA1C, 0xCA1D, 0xCA1E, 0xCA1F, 0xCA20, 0xCA21, 0xCA22, /* 0xA0 */
        0xCA23, 0xCA24, 	 0, 0xCA25, 0xCA26, 0xCA27, 0xCA28, 0xCA29, /* 0xB0 */
        0xCA2A, 0xCA2B, 0xCA2C, 0xCA2D, 0xCA2E, 0xCA2F, 	 0, 	 0, /* 0xB0 */
        0, 0xCA30, 0xCA31, 0xCA32, 0xCA33, 0xCA34, 0xCA35, 0xCA36, /* 0xC0 */
        0xCA37, 0xCA38, 0xCA39, 0xCA3A, 0xCA3B, 0xCA3C, 0xCA3D, 0xCA3E, /* 0xC0 */
        0xCA3F, 0xCA40, 	 0, 0xCA41, 0xCA42, 0xCA43, 0xCA44, 0xCA45, /* 0xD0 */
        0xCA46, 0xCA47, 0xCA48, 0xCA49, 0xCA4A, 0xCA4B, 	 0, 	 0, /* 0xD0 */
        0, 0xCA4C, 0xCA4D, 0xCA4E, 0xCA4F, 0xCA50, 0xCA51, 0xCA52, /* 0xE0 */
        0xCA53, 0xCA54, 0xCA55, 0xCA56, 0xCA57, 0xCA58, 0xCA59, 0xCA5A, /* 0xE0 */
        0xCA5B, 0xCA5C, 	 0, 0xCA5D, 0xCA5E, 0xCA5F, 0xCA60, 0xCA61, /* 0xF0 */
        0xCA62, 0xCA63, 0xCA64, 0xCA65, 0xCA66, 0xCA67					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xbd:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCA68, 0xCA69, 0xCA6A, 0xCA6B, 0xCA6C, 0xCA6D, 0xCA6E, /* 0x40 */
        0xCA6F, 0xCA70, 0xCA71, 0xCA72, 0xCA73, 0xCA74, 0xCA75, 0xCA76, /* 0x40 */
        0xCA77, 0xCA78, 	 0, 0xCA79, 0xCA7A, 0xCA7B, 0xCA7C, 0xCA7D, /* 0x50 */
        0xCA7E, 0xCA7F, 0xCA80, 0xCA81, 0xCA82, 0xCA83, 	 0, 	 0, /* 0x50 */
        0, 0xCA84, 0xCA85, 0xCA86, 0xCA87, 0xCA88, 0xCA89, 0xCA8A, /* 0x60 */
        0xCA8B, 0xCA8C, 0xCA8D, 0xCA8E, 0xCA8F, 0xCA90, 0xCA91, 0xCA92, /* 0x60 */
        0xCA93, 0xCA94, 	 0, 0xCA95, 0xCA96, 0xCA97, 0xCA98, 0xCA99, /* 0x70 */
        0xCA9A, 0xCA9B, 0xCA9C, 0xCA9D, 0xCA9E, 0xCA9F, 	 0, 	 0, /* 0x70 */
        0, 0xCAA0, 0xCAA1, 0xCAA2, 0xCAA3, 0xCAA4, 0xCAA5, 0xCAA6, /* 0x80 */
        0xCAA7, 0xCAA8, 0xCAA9, 0xCAAA, 0xCAAB, 0xCAAC, 0xCAAD, 0xCAAE, /* 0x80 */
        0xCAAF, 0xCAB0, 	 0, 0xCAB1, 0xCAB2, 0xCAB3, 0xCAB4, 0xCAB5, /* 0x90 */
        0xCAB6, 0xCAB7, 0xCAB8, 0xCAB9, 0xCABA, 0xCABB, 	 0, 	 0, /* 0x90 */
        0, 0xCABC, 0xCABD, 0xCABE, 0xCABF, 0xCAC0, 0xCAC1, 0xCAC2, /* 0xA0 */
        0xCAC3, 0xCAC4, 0xCAC5, 0xCAC6, 0xCAC7, 0xCAC8, 0xCAC9, 0xCACA, /* 0xA0 */
        0xCACB, 0xCACC, 	 0, 0xCACD, 0xCACE, 0xCACF, 0xCAD0, 0xCAD1, /* 0xB0 */
        0xCAD2, 0xCAD3, 0xCAD4, 0xCAD5, 0xCAD6, 0xCAD7, 	 0, 	 0, /* 0xB0 */
        0, 0xCAD8, 0xCAD9, 0xCADA, 0xCADB, 0xCADC, 0xCADD, 0xCADE, /* 0xC0 */
        0xCADF, 0xCAE0, 0xCAE1, 0xCAE2, 0xCAE3, 0xCAE4, 0xCAE5, 0xCAE6, /* 0xC0 */
        0xCAE7, 0xCAE8, 	 0, 0xCAE9, 0xCAEA, 0xCAEB, 0xCAEC, 0xCAED, /* 0xD0 */
        0xCAEE, 0xCAEF, 0xCAF0, 0xCAF1, 0xCAF2, 0xCAF3, 	 0, 	 0, /* 0xD0 */
        0, 0xCAF4, 0xCAF5, 0xCAF6, 0xCAF7, 0xCAF8, 0xCAF9, 0xCAFA, /* 0xE0 */
        0xCAFB, 0xCAFC, 0xCAFD, 0xCAFE, 0xCAFF, 0xCB00, 0xCB01, 0xCB02, /* 0xE0 */
        0xCB03, 0xCB04, 	 0, 0xCB05, 0xCB06, 0xCB07, 0xCB08, 0xCB09, /* 0xF0 */
        0xCB0A, 0xCB0B, 0xCB0C, 0xCB0D, 0xCB0E, 0xCB0F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xbe:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCB10, 0xCB11, 0xCB12, 0xCB13, 0xCB14, 0xCB15, 0xCB16, /* 0x40 */
        0xCB17, 0xCB18, 0xCB19, 0xCB1A, 0xCB1B, 0xCB1C, 0xCB1D, 0xCB1E, /* 0x40 */
        0xCB1F, 0xCB20, 	 0, 0xCB21, 0xCB22, 0xCB23, 0xCB24, 0xCB25, /* 0x50 */
        0xCB26, 0xCB27, 0xCB28, 0xCB29, 0xCB2A, 0xCB2B, 	 0, 	 0, /* 0x50 */
        0, 0xCB2C, 0xCB2D, 0xCB2E, 0xCB2F, 0xCB30, 0xCB31, 0xCB32, /* 0x60 */
        0xCB33, 0xCB34, 0xCB35, 0xCB36, 0xCB37, 0xCB38, 0xCB39, 0xCB3A, /* 0x60 */
        0xCB3B, 0xCB3C, 	 0, 0xCB3D, 0xCB3E, 0xCB3F, 0xCB40, 0xCB41, /* 0x70 */
        0xCB42, 0xCB43, 0xCB44, 0xCB45, 0xCB46, 0xCB47, 	 0, 	 0, /* 0x70 */
        0, 0xCB48, 0xCB49, 0xCB4A, 0xCB4B, 0xCB4C, 0xCB4D, 0xCB4E, /* 0x80 */
        0xCB4F, 0xCB50, 0xCB51, 0xCB52, 0xCB53, 0xCB54, 0xCB55, 0xCB56, /* 0x80 */
        0xCB57, 0xCB58, 	 0, 0xCB59, 0xCB5A, 0xCB5B, 0xCB5C, 0xCB5D, /* 0x90 */
        0xCB5E, 0xCB5F, 0xCB60, 0xCB61, 0xCB62, 0xCB63, 	 0, 	 0, /* 0x90 */
        0, 0xCB64, 0xCB65, 0xCB66, 0xCB67, 0xCB68, 0xCB69, 0xCB6A, /* 0xA0 */
        0xCB6B, 0xCB6C, 0xCB6D, 0xCB6E, 0xCB6F, 0xCB70, 0xCB71, 0xCB72, /* 0xA0 */
        0xCB73, 0xCB74, 	 0, 0xCB75, 0xCB76, 0xCB77, 0xCB78, 0xCB79, /* 0xB0 */
        0xCB7A, 0xCB7B, 0xCB7C, 0xCB7D, 0xCB7E, 0xCB7F, 	 0, 	 0, /* 0xB0 */
        0, 0xCB80, 0xCB81, 0xCB82, 0xCB83, 0xCB84, 0xCB85, 0xCB86, /* 0xC0 */
        0xCB87, 0xCB88, 0xCB89, 0xCB8A, 0xCB8B, 0xCB8C, 0xCB8D, 0xCB8E, /* 0xC0 */
        0xCB8F, 0xCB90, 	 0, 0xCB91, 0xCB92, 0xCB93, 0xCB94, 0xCB95, /* 0xD0 */
        0xCB96, 0xCB97, 0xCB98, 0xCB99, 0xCB9A, 0xCB9B, 	 0, 	 0, /* 0xD0 */
        0, 0xCB9C, 0xCB9D, 0xCB9E, 0xCB9F, 0xCBA0, 0xCBA1, 0xCBA2, /* 0xE0 */
        0xCBA3, 0xCBA4, 0xCBA5, 0xCBA6, 0xCBA7, 0xCBA8, 0xCBA9, 0xCBAA, /* 0xE0 */
        0xCBAB, 0xCBAC, 	 0, 0xCBAD, 0xCBAE, 0xCBAF, 0xCBB0, 0xCBB1, /* 0xF0 */
        0xCBB2, 0xCBB3, 0xCBB4, 0xCBB5, 0xCBB6, 0xCBB7					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xbf:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xCBB8, 0xCBB9, 0xCBBA, 0xCBBB, 0xCBBC, 0xCBBD, 0xCBBE, /* 0x40 */
        0xCBBF, 0xCBC0, 0xCBC1, 0xCBC2, 0xCBC3, 0xCBC4, 0xCBC5, 0xCBC6, /* 0x40 */
        0xCBC7, 0xCBC8, 	 0, 0xCBC9, 0xCBCA, 0xCBCB, 0xCBCC, 0xCBCD, /* 0x50 */
        0xCBCE, 0xCBCF, 0xCBD0, 0xCBD1, 0xCBD2, 0xCBD3, 	 0, 	 0, /* 0x50 */
        0, 0xCBD4, 0xCBD5, 0xCBD6, 0xCBD7, 0xCBD8, 0xCBD9, 0xCBDA, /* 0x60 */
        0xCBDB, 0xCBDC, 0xCBDD, 0xCBDE, 0xCBDF, 0xCBE0, 0xCBE1, 0xCBE2, /* 0x60 */
        0xCBE3, 0xCBE4, 	 0, 0xCBE5, 0xCBE6, 0xCBE7, 0xCBE8, 0xCBE9, /* 0x70 */
        0xCBEA, 0xCBEB, 0xCBEC, 0xCBED, 0xCBEE, 0xCBEF, 	 0, 	 0, /* 0x70 */
        0, 0xCBF0, 0xCBF1, 0xCBF2, 0xCBF3, 0xCBF4, 0xCBF5, 0xCBF6, /* 0x80 */
        0xCBF7, 0xCBF8, 0xCBF9, 0xCBFA, 0xCBFB, 0xCBFC, 0xCBFD, 0xCBFE, /* 0x80 */
        0xCBFF, 0xCC00, 	 0, 0xCC01, 0xCC02, 0xCC03, 0xCC04, 0xCC05, /* 0x90 */
        0xCC06, 0xCC07, 0xCC08, 0xCC09, 0xCC0A, 0xCC0B, 	 0, 	 0, /* 0x90 */
        0, 0xCC0C, 0xCC0D, 0xCC0E, 0xCC0F, 0xCC10, 0xCC11, 0xCC12, /* 0xA0 */
        0xCC13, 0xCC14, 0xCC15, 0xCC16, 0xCC17, 0xCC18, 0xCC19, 0xCC1A, /* 0xA0 */
        0xCC1B, 0xCC1C, 	 0, 0xCC1D, 0xCC1E, 0xCC1F, 0xCC20, 0xCC21, /* 0xB0 */
        0xCC22, 0xCC23, 0xCC24, 0xCC25, 0xCC26, 0xCC27					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc0:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x314A, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xCC28, 0xCC29, 0xCC2A, 0xCC2B, 0xCC2C, 0xCC2D, 0xCC2E, /* 0x60 */
        0xCC2F, 0xCC30, 0xCC31, 0xCC32, 0xCC33, 0xCC34, 0xCC35, 0xCC36, /* 0x60 */
        0xCC37, 0xCC38, 	 0, 0xCC39, 0xCC3A, 0xCC3B, 0xCC3C, 0xCC3D, /* 0x70 */
        0xCC3E, 0xCC3F, 0xCC40, 0xCC41, 0xCC42, 0xCC43, 	 0, 	 0, /* 0x70 */
        0, 0xCC44, 0xCC45, 0xCC46, 0xCC47, 0xCC48, 0xCC49, 0xCC4A, /* 0x80 */
        0xCC4B, 0xCC4C, 0xCC4D, 0xCC4E, 0xCC4F, 0xCC50, 0xCC51, 0xCC52, /* 0x80 */
        0xCC53, 0xCC54, 	 0, 0xCC55, 0xCC56, 0xCC57, 0xCC58, 0xCC59, /* 0x90 */
        0xCC5A, 0xCC5B, 0xCC5C, 0xCC5D, 0xCC5E, 0xCC5F, 	 0, 	 0, /* 0x90 */
        0, 0xCC60, 0xCC61, 0xCC62, 0xCC63, 0xCC64, 0xCC65, 0xCC66, /* 0xA0 */
        0xCC67, 0xCC68, 0xCC69, 0xCC6A, 0xCC6B, 0xCC6C, 0xCC6D, 0xCC6E, /* 0xA0 */
        0xCC6F, 0xCC70, 	 0, 0xCC71, 0xCC72, 0xCC73, 0xCC74, 0xCC75, /* 0xB0 */
        0xCC76, 0xCC77, 0xCC78, 0xCC79, 0xCC7A, 0xCC7B, 	 0, 	 0, /* 0xB0 */
        0, 0xCC7C, 0xCC7D, 0xCC7E, 0xCC7F, 0xCC80, 0xCC81, 0xCC82, /* 0xC0 */
        0xCC83, 0xCC84, 0xCC85, 0xCC86, 0xCC87, 0xCC88, 0xCC89, 0xCC8A, /* 0xC0 */
        0xCC8B, 0xCC8C, 	 0, 0xCC8D, 0xCC8E, 0xCC8F, 0xCC90, 0xCC91, /* 0xD0 */
        0xCC92, 0xCC93, 0xCC94, 0xCC95, 0xCC96, 0xCC97, 	 0, 	 0, /* 0xD0 */
        0, 0xCC98, 0xCC99, 0xCC9A, 0xCC9B, 0xCC9C, 0xCC9D, 0xCC9E, /* 0xE0 */
        0xCC9F, 0xCCA0, 0xCCA1, 0xCCA2, 0xCCA3, 0xCCA4, 0xCCA5, 0xCCA6, /* 0xE0 */
        0xCCA7, 0xCCA8, 	 0, 0xCCA9, 0xCCAA, 0xCCAB, 0xCCAC, 0xCCAD, /* 0xF0 */
        0xCCAE, 0xCCAF, 0xCCB0, 0xCCB1, 0xCCB2, 0xCCB3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc1:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCCB4, 0xCCB5, 0xCCB6, 0xCCB7, 0xCCB8, 0xCCB9, 0xCCBA, /* 0x40 */
        0xCCBB, 0xCCBC, 0xCCBD, 0xCCBE, 0xCCBF, 0xCCC0, 0xCCC1, 0xCCC2, /* 0x40 */
        0xCCC3, 0xCCC4, 	 0, 0xCCC5, 0xCCC6, 0xCCC7, 0xCCC8, 0xCCC9, /* 0x50 */
        0xCCCA, 0xCCCB, 0xCCCC, 0xCCCD, 0xCCCE, 0xCCCF, 	 0, 	 0, /* 0x50 */
        0, 0xCCD0, 0xCCD1, 0xCCD2, 0xCCD3, 0xCCD4, 0xCCD5, 0xCCD6, /* 0x60 */
        0xCCD7, 0xCCD8, 0xCCD9, 0xCCDA, 0xCCDB, 0xCCDC, 0xCCDD, 0xCCDE, /* 0x60 */
        0xCCDF, 0xCCE0, 	 0, 0xCCE1, 0xCCE2, 0xCCE3, 0xCCE4, 0xCCE5, /* 0x70 */
        0xCCE6, 0xCCE7, 0xCCE8, 0xCCE9, 0xCCEA, 0xCCEB, 	 0, 	 0, /* 0x70 */
        0, 0xCCEC, 0xCCED, 0xCCEE, 0xCCEF, 0xCCF0, 0xCCF1, 0xCCF2, /* 0x80 */
        0xCCF3, 0xCCF4, 0xCCF5, 0xCCF6, 0xCCF7, 0xCCF8, 0xCCF9, 0xCCFA, /* 0x80 */
        0xCCFB, 0xCCFC, 	 0, 0xCCFD, 0xCCFE, 0xCCFF, 0xCD00, 0xCD01, /* 0x90 */
        0xCD02, 0xCD03, 0xCD04, 0xCD05, 0xCD06, 0xCD07, 	 0, 	 0, /* 0x90 */
        0, 0xCD08, 0xCD09, 0xCD0A, 0xCD0B, 0xCD0C, 0xCD0D, 0xCD0E, /* 0xA0 */
        0xCD0F, 0xCD10, 0xCD11, 0xCD12, 0xCD13, 0xCD14, 0xCD15, 0xCD16, /* 0xA0 */
        0xCD17, 0xCD18, 	 0, 0xCD19, 0xCD1A, 0xCD1B, 0xCD1C, 0xCD1D, /* 0xB0 */
        0xCD1E, 0xCD1F, 0xCD20, 0xCD21, 0xCD22, 0xCD23, 	 0, 	 0, /* 0xB0 */
        0, 0xCD24, 0xCD25, 0xCD26, 0xCD27, 0xCD28, 0xCD29, 0xCD2A, /* 0xC0 */
        0xCD2B, 0xCD2C, 0xCD2D, 0xCD2E, 0xCD2F, 0xCD30, 0xCD31, 0xCD32, /* 0xC0 */
        0xCD33, 0xCD34, 	 0, 0xCD35, 0xCD36, 0xCD37, 0xCD38, 0xCD39, /* 0xD0 */
        0xCD3A, 0xCD3B, 0xCD3C, 0xCD3D, 0xCD3E, 0xCD3F, 	 0, 	 0, /* 0xD0 */
        0, 0xCD40, 0xCD41, 0xCD42, 0xCD43, 0xCD44, 0xCD45, 0xCD46, /* 0xE0 */
        0xCD47, 0xCD48, 0xCD49, 0xCD4A, 0xCD4B, 0xCD4C, 0xCD4D, 0xCD4E, /* 0xE0 */
        0xCD4F, 0xCD50, 	 0, 0xCD51, 0xCD52, 0xCD53, 0xCD54, 0xCD55, /* 0xF0 */
        0xCD56, 0xCD57, 0xCD58, 0xCD59, 0xCD5A, 0xCD5B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc2:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCD5C, 0xCD5D, 0xCD5E, 0xCD5F, 0xCD60, 0xCD61, 0xCD62, /* 0x40 */
        0xCD63, 0xCD64, 0xCD65, 0xCD66, 0xCD67, 0xCD68, 0xCD69, 0xCD6A, /* 0x40 */
        0xCD6B, 0xCD6C, 	 0, 0xCD6D, 0xCD6E, 0xCD6F, 0xCD70, 0xCD71, /* 0x50 */
        0xCD72, 0xCD73, 0xCD74, 0xCD75, 0xCD76, 0xCD77, 	 0, 	 0, /* 0x50 */
        0, 0xCD78, 0xCD79, 0xCD7A, 0xCD7B, 0xCD7C, 0xCD7D, 0xCD7E, /* 0x60 */
        0xCD7F, 0xCD80, 0xCD81, 0xCD82, 0xCD83, 0xCD84, 0xCD85, 0xCD86, /* 0x60 */
        0xCD87, 0xCD88, 	 0, 0xCD89, 0xCD8A, 0xCD8B, 0xCD8C, 0xCD8D, /* 0x70 */
        0xCD8E, 0xCD8F, 0xCD90, 0xCD91, 0xCD92, 0xCD93, 	 0, 	 0, /* 0x70 */
        0, 0xCD94, 0xCD95, 0xCD96, 0xCD97, 0xCD98, 0xCD99, 0xCD9A, /* 0x80 */
        0xCD9B, 0xCD9C, 0xCD9D, 0xCD9E, 0xCD9F, 0xCDA0, 0xCDA1, 0xCDA2, /* 0x80 */
        0xCDA3, 0xCDA4, 	 0, 0xCDA5, 0xCDA6, 0xCDA7, 0xCDA8, 0xCDA9, /* 0x90 */
        0xCDAA, 0xCDAB, 0xCDAC, 0xCDAD, 0xCDAE, 0xCDAF, 	 0, 	 0, /* 0x90 */
        0, 0xCDB0, 0xCDB1, 0xCDB2, 0xCDB3, 0xCDB4, 0xCDB5, 0xCDB6, /* 0xA0 */
        0xCDB7, 0xCDB8, 0xCDB9, 0xCDBA, 0xCDBB, 0xCDBC, 0xCDBD, 0xCDBE, /* 0xA0 */
        0xCDBF, 0xCDC0, 	 0, 0xCDC1, 0xCDC2, 0xCDC3, 0xCDC4, 0xCDC5, /* 0xB0 */
        0xCDC6, 0xCDC7, 0xCDC8, 0xCDC9, 0xCDCA, 0xCDCB, 	 0, 	 0, /* 0xB0 */
        0, 0xCDCC, 0xCDCD, 0xCDCE, 0xCDCF, 0xCDD0, 0xCDD1, 0xCDD2, /* 0xC0 */
        0xCDD3, 0xCDD4, 0xCDD5, 0xCDD6, 0xCDD7, 0xCDD8, 0xCDD9, 0xCDDA, /* 0xC0 */
        0xCDDB, 0xCDDC, 	 0, 0xCDDD, 0xCDDE, 0xCDDF, 0xCDE0, 0xCDE1, /* 0xD0 */
        0xCDE2, 0xCDE3, 0xCDE4, 0xCDE5, 0xCDE6, 0xCDE7, 	 0, 	 0, /* 0xD0 */
        0, 0xCDE8, 0xCDE9, 0xCDEA, 0xCDEB, 0xCDEC, 0xCDED, 0xCDEE, /* 0xE0 */
        0xCDEF, 0xCDF0, 0xCDF1, 0xCDF2, 0xCDF3, 0xCDF4, 0xCDF5, 0xCDF6, /* 0xE0 */
        0xCDF7, 0xCDF8, 	 0, 0xCDF9, 0xCDFA, 0xCDFB, 0xCDFC, 0xCDFD, /* 0xF0 */
        0xCDFE, 0xCDFF, 0xCE00, 0xCE01, 0xCE02, 0xCE03					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc3:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xCE04, 0xCE05, 0xCE06, 0xCE07, 0xCE08, 0xCE09, 0xCE0A, /* 0x40 */
        0xCE0B, 0xCE0C, 0xCE0D, 0xCE0E, 0xCE0F, 0xCE10, 0xCE11, 0xCE12, /* 0x40 */
        0xCE13, 0xCE14, 	 0, 0xCE15, 0xCE16, 0xCE17, 0xCE18, 0xCE19, /* 0x50 */
        0xCE1A, 0xCE1B, 0xCE1C, 0xCE1D, 0xCE1E, 0xCE1F, 	 0, 	 0, /* 0x50 */
        0, 0xCE20, 0xCE21, 0xCE22, 0xCE23, 0xCE24, 0xCE25, 0xCE26, /* 0x60 */
        0xCE27, 0xCE28, 0xCE29, 0xCE2A, 0xCE2B, 0xCE2C, 0xCE2D, 0xCE2E, /* 0x60 */
        0xCE2F, 0xCE30, 	 0, 0xCE31, 0xCE32, 0xCE33, 0xCE34, 0xCE35, /* 0x70 */
        0xCE36, 0xCE37, 0xCE38, 0xCE39, 0xCE3A, 0xCE3B, 	 0, 	 0, /* 0x70 */
        0, 0xCE3C, 0xCE3D, 0xCE3E, 0xCE3F, 0xCE40, 0xCE41, 0xCE42, /* 0x80 */
        0xCE43, 0xCE44, 0xCE45, 0xCE46, 0xCE47, 0xCE48, 0xCE49, 0xCE4A, /* 0x80 */
        0xCE4B, 0xCE4C, 	 0, 0xCE4D, 0xCE4E, 0xCE4F, 0xCE50, 0xCE51, /* 0x90 */
        0xCE52, 0xCE53, 0xCE54, 0xCE55, 0xCE56, 0xCE57, 	 0, 	 0, /* 0x90 */
        0, 0xCE58, 0xCE59, 0xCE5A, 0xCE5B, 0xCE5C, 0xCE5D, 0xCE5E, /* 0xA0 */
        0xCE5F, 0xCE60, 0xCE61, 0xCE62, 0xCE63, 0xCE64, 0xCE65, 0xCE66, /* 0xA0 */
        0xCE67, 0xCE68, 	 0, 0xCE69, 0xCE6A, 0xCE6B, 0xCE6C, 0xCE6D, /* 0xB0 */
        0xCE6E, 0xCE6F, 0xCE70, 0xCE71, 0xCE72, 0xCE73					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc4:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x314B, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xCE74, 0xCE75, 0xCE76, 0xCE77, 0xCE78, 0xCE79, 0xCE7A, /* 0x60 */
        0xCE7B, 0xCE7C, 0xCE7D, 0xCE7E, 0xCE7F, 0xCE80, 0xCE81, 0xCE82, /* 0x60 */
        0xCE83, 0xCE84, 	 0, 0xCE85, 0xCE86, 0xCE87, 0xCE88, 0xCE89, /* 0x70 */
        0xCE8A, 0xCE8B, 0xCE8C, 0xCE8D, 0xCE8E, 0xCE8F, 	 0, 	 0, /* 0x70 */
        0, 0xCE90, 0xCE91, 0xCE92, 0xCE93, 0xCE94, 0xCE95, 0xCE96, /* 0x80 */
        0xCE97, 0xCE98, 0xCE99, 0xCE9A, 0xCE9B, 0xCE9C, 0xCE9D, 0xCE9E, /* 0x80 */
        0xCE9F, 0xCEA0, 	 0, 0xCEA1, 0xCEA2, 0xCEA3, 0xCEA4, 0xCEA5, /* 0x90 */
        0xCEA6, 0xCEA7, 0xCEA8, 0xCEA9, 0xCEAA, 0xCEAB, 	 0, 	 0, /* 0x90 */
        0, 0xCEAC, 0xCEAD, 0xCEAE, 0xCEAF, 0xCEB0, 0xCEB1, 0xCEB2, /* 0xA0 */
        0xCEB3, 0xCEB4, 0xCEB5, 0xCEB6, 0xCEB7, 0xCEB8, 0xCEB9, 0xCEBA, /* 0xA0 */
        0xCEBB, 0xCEBC, 	 0, 0xCEBD, 0xCEBE, 0xCEBF, 0xCEC0, 0xCEC1, /* 0xB0 */
        0xCEC2, 0xCEC3, 0xCEC4, 0xCEC5, 0xCEC6, 0xCEC7, 	 0, 	 0, /* 0xB0 */
        0, 0xCEC8, 0xCEC9, 0xCECA, 0xCECB, 0xCECC, 0xCECD, 0xCECE, /* 0xC0 */
        0xCECF, 0xCED0, 0xCED1, 0xCED2, 0xCED3, 0xCED4, 0xCED5, 0xCED6, /* 0xC0 */
        0xCED7, 0xCED8, 	 0, 0xCED9, 0xCEDA, 0xCEDB, 0xCEDC, 0xCEDD, /* 0xD0 */
        0xCEDE, 0xCEDF, 0xCEE0, 0xCEE1, 0xCEE2, 0xCEE3, 	 0, 	 0, /* 0xD0 */
        0, 0xCEE4, 0xCEE5, 0xCEE6, 0xCEE7, 0xCEE8, 0xCEE9, 0xCEEA, /* 0xE0 */
        0xCEEB, 0xCEEC, 0xCEED, 0xCEEE, 0xCEEF, 0xCEF0, 0xCEF1, 0xCEF2, /* 0xE0 */
        0xCEF3, 0xCEF4, 	 0, 0xCEF5, 0xCEF6, 0xCEF7, 0xCEF8, 0xCEF9, /* 0xF0 */
        0xCEFA, 0xCEFB, 0xCEFC, 0xCEFD, 0xCEFE, 0xCEFF					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc5:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCF00, 0xCF01, 0xCF02, 0xCF03, 0xCF04, 0xCF05, 0xCF06, /* 0x40 */
        0xCF07, 0xCF08, 0xCF09, 0xCF0A, 0xCF0B, 0xCF0C, 0xCF0D, 0xCF0E, /* 0x40 */
        0xCF0F, 0xCF10, 	 0, 0xCF11, 0xCF12, 0xCF13, 0xCF14, 0xCF15, /* 0x50 */
        0xCF16, 0xCF17, 0xCF18, 0xCF19, 0xCF1A, 0xCF1B, 	 0, 	 0, /* 0x50 */
        0, 0xCF1C, 0xCF1D, 0xCF1E, 0xCF1F, 0xCF20, 0xCF21, 0xCF22, /* 0x60 */
        0xCF23, 0xCF24, 0xCF25, 0xCF26, 0xCF27, 0xCF28, 0xCF29, 0xCF2A, /* 0x60 */
        0xCF2B, 0xCF2C, 	 0, 0xCF2D, 0xCF2E, 0xCF2F, 0xCF30, 0xCF31, /* 0x70 */
        0xCF32, 0xCF33, 0xCF34, 0xCF35, 0xCF36, 0xCF37, 	 0, 	 0, /* 0x70 */
        0, 0xCF38, 0xCF39, 0xCF3A, 0xCF3B, 0xCF3C, 0xCF3D, 0xCF3E, /* 0x80 */
        0xCF3F, 0xCF40, 0xCF41, 0xCF42, 0xCF43, 0xCF44, 0xCF45, 0xCF46, /* 0x80 */
        0xCF47, 0xCF48, 	 0, 0xCF49, 0xCF4A, 0xCF4B, 0xCF4C, 0xCF4D, /* 0x90 */
        0xCF4E, 0xCF4F, 0xCF50, 0xCF51, 0xCF52, 0xCF53, 	 0, 	 0, /* 0x90 */
        0, 0xCF54, 0xCF55, 0xCF56, 0xCF57, 0xCF58, 0xCF59, 0xCF5A, /* 0xA0 */
        0xCF5B, 0xCF5C, 0xCF5D, 0xCF5E, 0xCF5F, 0xCF60, 0xCF61, 0xCF62, /* 0xA0 */
        0xCF63, 0xCF64, 	 0, 0xCF65, 0xCF66, 0xCF67, 0xCF68, 0xCF69, /* 0xB0 */
        0xCF6A, 0xCF6B, 0xCF6C, 0xCF6D, 0xCF6E, 0xCF6F, 	 0, 	 0, /* 0xB0 */
        0, 0xCF70, 0xCF71, 0xCF72, 0xCF73, 0xCF74, 0xCF75, 0xCF76, /* 0xC0 */
        0xCF77, 0xCF78, 0xCF79, 0xCF7A, 0xCF7B, 0xCF7C, 0xCF7D, 0xCF7E, /* 0xC0 */
        0xCF7F, 0xCF80, 	 0, 0xCF81, 0xCF82, 0xCF83, 0xCF84, 0xCF85, /* 0xD0 */
        0xCF86, 0xCF87, 0xCF88, 0xCF89, 0xCF8A, 0xCF8B, 	 0, 	 0, /* 0xD0 */
        0, 0xCF8C, 0xCF8D, 0xCF8E, 0xCF8F, 0xCF90, 0xCF91, 0xCF92, /* 0xE0 */
        0xCF93, 0xCF94, 0xCF95, 0xCF96, 0xCF97, 0xCF98, 0xCF99, 0xCF9A, /* 0xE0 */
        0xCF9B, 0xCF9C, 	 0, 0xCF9D, 0xCF9E, 0xCF9F, 0xCFA0, 0xCFA1, /* 0xF0 */
        0xCFA2, 0xCFA3, 0xCFA4, 0xCFA5, 0xCFA6, 0xCFA7					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc6:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xCFA8, 0xCFA9, 0xCFAA, 0xCFAB, 0xCFAC, 0xCFAD, 0xCFAE, /* 0x40 */
        0xCFAF, 0xCFB0, 0xCFB1, 0xCFB2, 0xCFB3, 0xCFB4, 0xCFB5, 0xCFB6, /* 0x40 */
        0xCFB7, 0xCFB8, 	 0, 0xCFB9, 0xCFBA, 0xCFBB, 0xCFBC, 0xCFBD, /* 0x50 */
        0xCFBE, 0xCFBF, 0xCFC0, 0xCFC1, 0xCFC2, 0xCFC3, 	 0, 	 0, /* 0x50 */
        0, 0xCFC4, 0xCFC5, 0xCFC6, 0xCFC7, 0xCFC8, 0xCFC9, 0xCFCA, /* 0x60 */
        0xCFCB, 0xCFCC, 0xCFCD, 0xCFCE, 0xCFCF, 0xCFD0, 0xCFD1, 0xCFD2, /* 0x60 */
        0xCFD3, 0xCFD4, 	 0, 0xCFD5, 0xCFD6, 0xCFD7, 0xCFD8, 0xCFD9, /* 0x70 */
        0xCFDA, 0xCFDB, 0xCFDC, 0xCFDD, 0xCFDE, 0xCFDF, 	 0, 	 0, /* 0x70 */
        0, 0xCFE0, 0xCFE1, 0xCFE2, 0xCFE3, 0xCFE4, 0xCFE5, 0xCFE6, /* 0x80 */
        0xCFE7, 0xCFE8, 0xCFE9, 0xCFEA, 0xCFEB, 0xCFEC, 0xCFED, 0xCFEE, /* 0x80 */
        0xCFEF, 0xCFF0, 	 0, 0xCFF1, 0xCFF2, 0xCFF3, 0xCFF4, 0xCFF5, /* 0x90 */
        0xCFF6, 0xCFF7, 0xCFF8, 0xCFF9, 0xCFFA, 0xCFFB, 	 0, 	 0, /* 0x90 */
        0, 0xCFFC, 0xCFFD, 0xCFFE, 0xCFFF, 0xD000, 0xD001, 0xD002, /* 0xA0 */
        0xD003, 0xD004, 0xD005, 0xD006, 0xD007, 0xD008, 0xD009, 0xD00A, /* 0xA0 */
        0xD00B, 0xD00C, 	 0, 0xD00D, 0xD00E, 0xD00F, 0xD010, 0xD011, /* 0xB0 */
        0xD012, 0xD013, 0xD014, 0xD015, 0xD016, 0xD017, 	 0, 	 0, /* 0xB0 */
        0, 0xD018, 0xD019, 0xD01A, 0xD01B, 0xD01C, 0xD01D, 0xD01E, /* 0xC0 */
        0xD01F, 0xD020, 0xD021, 0xD022, 0xD023, 0xD024, 0xD025, 0xD026, /* 0xC0 */
        0xD027, 0xD028, 	 0, 0xD029, 0xD02A, 0xD02B, 0xD02C, 0xD02D, /* 0xD0 */
        0xD02E, 0xD02F, 0xD030, 0xD031, 0xD032, 0xD033, 	 0, 	 0, /* 0xD0 */
        0, 0xD034, 0xD035, 0xD036, 0xD037, 0xD038, 0xD039, 0xD03A, /* 0xE0 */
        0xD03B, 0xD03C, 0xD03D, 0xD03E, 0xD03F, 0xD040, 0xD041, 0xD042, /* 0xE0 */
        0xD043, 0xD044, 	 0, 0xD045, 0xD046, 0xD047, 0xD048, 0xD049, /* 0xF0 */
        0xD04A, 0xD04B, 0xD04C, 0xD04D, 0xD04E, 0xD04F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc7:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xD050, 0xD051, 0xD052, 0xD053, 0xD054, 0xD055, 0xD056, /* 0x40 */
        0xD057, 0xD058, 0xD059, 0xD05A, 0xD05B, 0xD05C, 0xD05D, 0xD05E, /* 0x40 */
        0xD05F, 0xD060, 	 0, 0xD061, 0xD062, 0xD063, 0xD064, 0xD065, /* 0x50 */
        0xD066, 0xD067, 0xD068, 0xD069, 0xD06A, 0xD06B, 	 0, 	 0, /* 0x50 */
        0, 0xD06C, 0xD06D, 0xD06E, 0xD06F, 0xD070, 0xD071, 0xD072, /* 0x60 */
        0xD073, 0xD074, 0xD075, 0xD076, 0xD077, 0xD078, 0xD079, 0xD07A, /* 0x60 */
        0xD07B, 0xD07C, 	 0, 0xD07D, 0xD07E, 0xD07F, 0xD080, 0xD081, /* 0x70 */
        0xD082, 0xD083, 0xD084, 0xD085, 0xD086, 0xD087, 	 0, 	 0, /* 0x70 */
        0, 0xD088, 0xD089, 0xD08A, 0xD08B, 0xD08C, 0xD08D, 0xD08E, /* 0x80 */
        0xD08F, 0xD090, 0xD091, 0xD092, 0xD093, 0xD094, 0xD095, 0xD096, /* 0x80 */
        0xD097, 0xD098, 	 0, 0xD099, 0xD09A, 0xD09B, 0xD09C, 0xD09D, /* 0x90 */
        0xD09E, 0xD09F, 0xD0A0, 0xD0A1, 0xD0A2, 0xD0A3, 	 0, 	 0, /* 0x90 */
        0, 0xD0A4, 0xD0A5, 0xD0A6, 0xD0A7, 0xD0A8, 0xD0A9, 0xD0AA, /* 0xA0 */
        0xD0AB, 0xD0AC, 0xD0AD, 0xD0AE, 0xD0AF, 0xD0B0, 0xD0B1, 0xD0B2, /* 0xA0 */
        0xD0B3, 0xD0B4, 	 0, 0xD0B5, 0xD0B6, 0xD0B7, 0xD0B8, 0xD0B9, /* 0xB0 */
        0xD0BA, 0xD0BB, 0xD0BC, 0xD0BD, 0xD0BE, 0xD0BF					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc8:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x314C, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xD0C0, 0xD0C1, 0xD0C2, 0xD0C3, 0xD0C4, 0xD0C5, 0xD0C6, /* 0x60 */
        0xD0C7, 0xD0C8, 0xD0C9, 0xD0CA, 0xD0CB, 0xD0CC, 0xD0CD, 0xD0CE, /* 0x60 */
        0xD0CF, 0xD0D0, 	 0, 0xD0D1, 0xD0D2, 0xD0D3, 0xD0D4, 0xD0D5, /* 0x70 */
        0xD0D6, 0xD0D7, 0xD0D8, 0xD0D9, 0xD0DA, 0xD0DB, 	 0, 	 0, /* 0x70 */
        0, 0xD0DC, 0xD0DD, 0xD0DE, 0xD0DF, 0xD0E0, 0xD0E1, 0xD0E2, /* 0x80 */
        0xD0E3, 0xD0E4, 0xD0E5, 0xD0E6, 0xD0E7, 0xD0E8, 0xD0E9, 0xD0EA, /* 0x80 */
        0xD0EB, 0xD0EC, 	 0, 0xD0ED, 0xD0EE, 0xD0EF, 0xD0F0, 0xD0F1, /* 0x90 */
        0xD0F2, 0xD0F3, 0xD0F4, 0xD0F5, 0xD0F6, 0xD0F7, 	 0, 	 0, /* 0x90 */
        0, 0xD0F8, 0xD0F9, 0xD0FA, 0xD0FB, 0xD0FC, 0xD0FD, 0xD0FE, /* 0xA0 */
        0xD0FF, 0xD100, 0xD101, 0xD102, 0xD103, 0xD104, 0xD105, 0xD106, /* 0xA0 */
        0xD107, 0xD108, 	 0, 0xD109, 0xD10A, 0xD10B, 0xD10C, 0xD10D, /* 0xB0 */
        0xD10E, 0xD10F, 0xD110, 0xD111, 0xD112, 0xD113, 	 0, 	 0, /* 0xB0 */
        0, 0xD114, 0xD115, 0xD116, 0xD117, 0xD118, 0xD119, 0xD11A, /* 0xC0 */
        0xD11B, 0xD11C, 0xD11D, 0xD11E, 0xD11F, 0xD120, 0xD121, 0xD122, /* 0xC0 */
        0xD123, 0xD124, 	 0, 0xD125, 0xD126, 0xD127, 0xD128, 0xD129, /* 0xD0 */
        0xD12A, 0xD12B, 0xD12C, 0xD12D, 0xD12E, 0xD12F, 	 0, 	 0, /* 0xD0 */
        0, 0xD130, 0xD131, 0xD132, 0xD133, 0xD134, 0xD135, 0xD136, /* 0xE0 */
        0xD137, 0xD138, 0xD139, 0xD13A, 0xD13B, 0xD13C, 0xD13D, 0xD13E, /* 0xE0 */
        0xD13F, 0xD140, 	 0, 0xD141, 0xD142, 0xD143, 0xD144, 0xD145, /* 0xF0 */
        0xD146, 0xD147, 0xD148, 0xD149, 0xD14A, 0xD14B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xc9:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD14C, 0xD14D, 0xD14E, 0xD14F, 0xD150, 0xD151, 0xD152, /* 0x40 */
        0xD153, 0xD154, 0xD155, 0xD156, 0xD157, 0xD158, 0xD159, 0xD15A, /* 0x40 */
        0xD15B, 0xD15C, 	 0, 0xD15D, 0xD15E, 0xD15F, 0xD160, 0xD161, /* 0x50 */
        0xD162, 0xD163, 0xD164, 0xD165, 0xD166, 0xD167, 	 0, 	 0, /* 0x50 */
        0, 0xD168, 0xD169, 0xD16A, 0xD16B, 0xD16C, 0xD16D, 0xD16E, /* 0x60 */
        0xD16F, 0xD170, 0xD171, 0xD172, 0xD173, 0xD174, 0xD175, 0xD176, /* 0x60 */
        0xD177, 0xD178, 	 0, 0xD179, 0xD17A, 0xD17B, 0xD17C, 0xD17D, /* 0x70 */
        0xD17E, 0xD17F, 0xD180, 0xD181, 0xD182, 0xD183, 	 0, 	 0, /* 0x70 */
        0, 0xD184, 0xD185, 0xD186, 0xD187, 0xD188, 0xD189, 0xD18A, /* 0x80 */
        0xD18B, 0xD18C, 0xD18D, 0xD18E, 0xD18F, 0xD190, 0xD191, 0xD192, /* 0x80 */
        0xD193, 0xD194, 	 0, 0xD195, 0xD196, 0xD197, 0xD198, 0xD199, /* 0x90 */
        0xD19A, 0xD19B, 0xD19C, 0xD19D, 0xD19E, 0xD19F, 	 0, 	 0, /* 0x90 */
        0, 0xD1A0, 0xD1A1, 0xD1A2, 0xD1A3, 0xD1A4, 0xD1A5, 0xD1A6, /* 0xA0 */
        0xD1A7, 0xD1A8, 0xD1A9, 0xD1AA, 0xD1AB, 0xD1AC, 0xD1AD, 0xD1AE, /* 0xA0 */
        0xD1AF, 0xD1B0, 	 0, 0xD1B1, 0xD1B2, 0xD1B3, 0xD1B4, 0xD1B5, /* 0xB0 */
        0xD1B6, 0xD1B7, 0xD1B8, 0xD1B9, 0xD1BA, 0xD1BB, 	 0, 	 0, /* 0xB0 */
        0, 0xD1BC, 0xD1BD, 0xD1BE, 0xD1BF, 0xD1C0, 0xD1C1, 0xD1C2, /* 0xC0 */
        0xD1C3, 0xD1C4, 0xD1C5, 0xD1C6, 0xD1C7, 0xD1C8, 0xD1C9, 0xD1CA, /* 0xC0 */
        0xD1CB, 0xD1CC, 	 0, 0xD1CD, 0xD1CE, 0xD1CF, 0xD1D0, 0xD1D1, /* 0xD0 */
        0xD1D2, 0xD1D3, 0xD1D4, 0xD1D5, 0xD1D6, 0xD1D7, 	 0, 	 0, /* 0xD0 */
        0, 0xD1D8, 0xD1D9, 0xD1DA, 0xD1DB, 0xD1DC, 0xD1DD, 0xD1DE, /* 0xE0 */
        0xD1DF, 0xD1E0, 0xD1E1, 0xD1E2, 0xD1E3, 0xD1E4, 0xD1E5, 0xD1E6, /* 0xE0 */
        0xD1E7, 0xD1E8, 	 0, 0xD1E9, 0xD1EA, 0xD1EB, 0xD1EC, 0xD1ED, /* 0xF0 */
        0xD1EE, 0xD1EF, 0xD1F0, 0xD1F1, 0xD1F2, 0xD1F3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xca:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD1F4, 0xD1F5, 0xD1F6, 0xD1F7, 0xD1F8, 0xD1F9, 0xD1FA, /* 0x40 */
        0xD1FB, 0xD1FC, 0xD1FD, 0xD1FE, 0xD1FF, 0xD200, 0xD201, 0xD202, /* 0x40 */
        0xD203, 0xD204, 	 0, 0xD205, 0xD206, 0xD207, 0xD208, 0xD209, /* 0x50 */
        0xD20A, 0xD20B, 0xD20C, 0xD20D, 0xD20E, 0xD20F, 	 0, 	 0, /* 0x50 */
        0, 0xD210, 0xD211, 0xD212, 0xD213, 0xD214, 0xD215, 0xD216, /* 0x60 */
        0xD217, 0xD218, 0xD219, 0xD21A, 0xD21B, 0xD21C, 0xD21D, 0xD21E, /* 0x60 */
        0xD21F, 0xD220, 	 0, 0xD221, 0xD222, 0xD223, 0xD224, 0xD225, /* 0x70 */
        0xD226, 0xD227, 0xD228, 0xD229, 0xD22A, 0xD22B, 	 0, 	 0, /* 0x70 */
        0, 0xD22C, 0xD22D, 0xD22E, 0xD22F, 0xD230, 0xD231, 0xD232, /* 0x80 */
        0xD233, 0xD234, 0xD235, 0xD236, 0xD237, 0xD238, 0xD239, 0xD23A, /* 0x80 */
        0xD23B, 0xD23C, 	 0, 0xD23D, 0xD23E, 0xD23F, 0xD240, 0xD241, /* 0x90 */
        0xD242, 0xD243, 0xD244, 0xD245, 0xD246, 0xD247, 	 0, 	 0, /* 0x90 */
        0, 0xD248, 0xD249, 0xD24A, 0xD24B, 0xD24C, 0xD24D, 0xD24E, /* 0xA0 */
        0xD24F, 0xD250, 0xD251, 0xD252, 0xD253, 0xD254, 0xD255, 0xD256, /* 0xA0 */
        0xD257, 0xD258, 	 0, 0xD259, 0xD25A, 0xD25B, 0xD25C, 0xD25D, /* 0xB0 */
        0xD25E, 0xD25F, 0xD260, 0xD261, 0xD262, 0xD263, 	 0, 	 0, /* 0xB0 */
        0, 0xD264, 0xD265, 0xD266, 0xD267, 0xD268, 0xD269, 0xD26A, /* 0xC0 */
        0xD26B, 0xD26C, 0xD26D, 0xD26E, 0xD26F, 0xD270, 0xD271, 0xD272, /* 0xC0 */
        0xD273, 0xD274, 	 0, 0xD275, 0xD276, 0xD277, 0xD278, 0xD279, /* 0xD0 */
        0xD27A, 0xD27B, 0xD27C, 0xD27D, 0xD27E, 0xD27F, 	 0, 	 0, /* 0xD0 */
        0, 0xD280, 0xD281, 0xD282, 0xD283, 0xD284, 0xD285, 0xD286, /* 0xE0 */
        0xD287, 0xD288, 0xD289, 0xD28A, 0xD28B, 0xD28C, 0xD28D, 0xD28E, /* 0xE0 */
        0xD28F, 0xD290, 	 0, 0xD291, 0xD292, 0xD293, 0xD294, 0xD295, /* 0xF0 */
        0xD296, 0xD297, 0xD298, 0xD299, 0xD29A, 0xD29B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xcb:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xD29C, 0xD29D, 0xD29E, 0xD29F, 0xD2A0, 0xD2A1, 0xD2A2, /* 0x40 */
        0xD2A3, 0xD2A4, 0xD2A5, 0xD2A6, 0xD2A7, 0xD2A8, 0xD2A9, 0xD2AA, /* 0x40 */
        0xD2AB, 0xD2AC, 	 0, 0xD2AD, 0xD2AE, 0xD2AF, 0xD2B0, 0xD2B1, /* 0x50 */
        0xD2B2, 0xD2B3, 0xD2B4, 0xD2B5, 0xD2B6, 0xD2B7, 	 0, 	 0, /* 0x50 */
        0, 0xD2B8, 0xD2B9, 0xD2BA, 0xD2BB, 0xD2BC, 0xD2BD, 0xD2BE, /* 0x60 */
        0xD2BF, 0xD2C0, 0xD2C1, 0xD2C2, 0xD2C3, 0xD2C4, 0xD2C5, 0xD2C6, /* 0x60 */
        0xD2C7, 0xD2C8, 	 0, 0xD2C9, 0xD2CA, 0xD2CB, 0xD2CC, 0xD2CD, /* 0x70 */
        0xD2CE, 0xD2CF, 0xD2D0, 0xD2D1, 0xD2D2, 0xD2D3, 	 0, 	 0, /* 0x70 */
        0, 0xD2D4, 0xD2D5, 0xD2D6, 0xD2D7, 0xD2D8, 0xD2D9, 0xD2DA, /* 0x80 */
        0xD2DB, 0xD2DC, 0xD2DD, 0xD2DE, 0xD2DF, 0xD2E0, 0xD2E1, 0xD2E2, /* 0x80 */
        0xD2E3, 0xD2E4, 	 0, 0xD2E5, 0xD2E6, 0xD2E7, 0xD2E8, 0xD2E9, /* 0x90 */
        0xD2EA, 0xD2EB, 0xD2EC, 0xD2ED, 0xD2EE, 0xD2EF, 	 0, 	 0, /* 0x90 */
        0, 0xD2F0, 0xD2F1, 0xD2F2, 0xD2F3, 0xD2F4, 0xD2F5, 0xD2F6, /* 0xA0 */
        0xD2F7, 0xD2F8, 0xD2F9, 0xD2FA, 0xD2FB, 0xD2FC, 0xD2FD, 0xD2FE, /* 0xA0 */
        0xD2FF, 0xD300, 	 0, 0xD301, 0xD302, 0xD303, 0xD304, 0xD305, /* 0xB0 */
        0xD306, 0xD307, 0xD308, 0xD309, 0xD30A, 0xD30B					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xcc:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x314D, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xD30C, 0xD30D, 0xD30E, 0xD30F, 0xD310, 0xD311, 0xD312, /* 0x60 */
        0xD313, 0xD314, 0xD315, 0xD316, 0xD317, 0xD318, 0xD319, 0xD31A, /* 0x60 */
        0xD31B, 0xD31C, 	 0, 0xD31D, 0xD31E, 0xD31F, 0xD320, 0xD321, /* 0x70 */
        0xD322, 0xD323, 0xD324, 0xD325, 0xD326, 0xD327, 	 0, 	 0, /* 0x70 */
        0, 0xD328, 0xD329, 0xD32A, 0xD32B, 0xD32C, 0xD32D, 0xD32E, /* 0x80 */
        0xD32F, 0xD330, 0xD331, 0xD332, 0xD333, 0xD334, 0xD335, 0xD336, /* 0x80 */
        0xD337, 0xD338, 	 0, 0xD339, 0xD33A, 0xD33B, 0xD33C, 0xD33D, /* 0x90 */
        0xD33E, 0xD33F, 0xD340, 0xD341, 0xD342, 0xD343, 	 0, 	 0, /* 0x90 */
        0, 0xD344, 0xD345, 0xD346, 0xD347, 0xD348, 0xD349, 0xD34A, /* 0xA0 */
        0xD34B, 0xD34C, 0xD34D, 0xD34E, 0xD34F, 0xD350, 0xD351, 0xD352, /* 0xA0 */
        0xD353, 0xD354, 	 0, 0xD355, 0xD356, 0xD357, 0xD358, 0xD359, /* 0xB0 */
        0xD35A, 0xD35B, 0xD35C, 0xD35D, 0xD35E, 0xD35F, 	 0, 	 0, /* 0xB0 */
        0, 0xD360, 0xD361, 0xD362, 0xD363, 0xD364, 0xD365, 0xD366, /* 0xC0 */
        0xD367, 0xD368, 0xD369, 0xD36A, 0xD36B, 0xD36C, 0xD36D, 0xD36E, /* 0xC0 */
        0xD36F, 0xD370, 	 0, 0xD371, 0xD372, 0xD373, 0xD374, 0xD375, /* 0xD0 */
        0xD376, 0xD377, 0xD378, 0xD379, 0xD37A, 0xD37B, 	 0, 	 0, /* 0xD0 */
        0, 0xD37C, 0xD37D, 0xD37E, 0xD37F, 0xD380, 0xD381, 0xD382, /* 0xE0 */
        0xD383, 0xD384, 0xD385, 0xD386, 0xD387, 0xD388, 0xD389, 0xD38A, /* 0xE0 */
        0xD38B, 0xD38C, 	 0, 0xD38D, 0xD38E, 0xD38F, 0xD390, 0xD391, /* 0xF0 */
        0xD392, 0xD393, 0xD394, 0xD395, 0xD396, 0xD397					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xcd:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD398, 0xD399, 0xD39A, 0xD39B, 0xD39C, 0xD39D, 0xD39E, /* 0x40 */
        0xD39F, 0xD3A0, 0xD3A1, 0xD3A2, 0xD3A3, 0xD3A4, 0xD3A5, 0xD3A6, /* 0x40 */
        0xD3A7, 0xD3A8, 	 0, 0xD3A9, 0xD3AA, 0xD3AB, 0xD3AC, 0xD3AD, /* 0x50 */
        0xD3AE, 0xD3AF, 0xD3B0, 0xD3B1, 0xD3B2, 0xD3B3, 	 0, 	 0, /* 0x50 */
        0, 0xD3B4, 0xD3B5, 0xD3B6, 0xD3B7, 0xD3B8, 0xD3B9, 0xD3BA, /* 0x60 */
        0xD3BB, 0xD3BC, 0xD3BD, 0xD3BE, 0xD3BF, 0xD3C0, 0xD3C1, 0xD3C2, /* 0x60 */
        0xD3C3, 0xD3C4, 	 0, 0xD3C5, 0xD3C6, 0xD3C7, 0xD3C8, 0xD3C9, /* 0x70 */
        0xD3CA, 0xD3CB, 0xD3CC, 0xD3CD, 0xD3CE, 0xD3CF, 	 0, 	 0, /* 0x70 */
        0, 0xD3D0, 0xD3D1, 0xD3D2, 0xD3D3, 0xD3D4, 0xD3D5, 0xD3D6, /* 0x80 */
        0xD3D7, 0xD3D8, 0xD3D9, 0xD3DA, 0xD3DB, 0xD3DC, 0xD3DD, 0xD3DE, /* 0x80 */
        0xD3DF, 0xD3E0, 	 0, 0xD3E1, 0xD3E2, 0xD3E3, 0xD3E4, 0xD3E5, /* 0x90 */
        0xD3E6, 0xD3E7, 0xD3E8, 0xD3E9, 0xD3EA, 0xD3EB, 	 0, 	 0, /* 0x90 */
        0, 0xD3EC, 0xD3ED, 0xD3EE, 0xD3EF, 0xD3F0, 0xD3F1, 0xD3F2, /* 0xA0 */
        0xD3F3, 0xD3F4, 0xD3F5, 0xD3F6, 0xD3F7, 0xD3F8, 0xD3F9, 0xD3FA, /* 0xA0 */
        0xD3FB, 0xD3FC, 	 0, 0xD3FD, 0xD3FE, 0xD3FF, 0xD400, 0xD401, /* 0xB0 */
        0xD402, 0xD403, 0xD404, 0xD405, 0xD406, 0xD407, 	 0, 	 0, /* 0xB0 */
        0, 0xD408, 0xD409, 0xD40A, 0xD40B, 0xD40C, 0xD40D, 0xD40E, /* 0xC0 */
        0xD40F, 0xD410, 0xD411, 0xD412, 0xD413, 0xD414, 0xD415, 0xD416, /* 0xC0 */
        0xD417, 0xD418, 	 0, 0xD419, 0xD41A, 0xD41B, 0xD41C, 0xD41D, /* 0xD0 */
        0xD41E, 0xD41F, 0xD420, 0xD421, 0xD422, 0xD423, 	 0, 	 0, /* 0xD0 */
        0, 0xD424, 0xD425, 0xD426, 0xD427, 0xD428, 0xD429, 0xD42A, /* 0xE0 */
        0xD42B, 0xD42C, 0xD42D, 0xD42E, 0xD42F, 0xD430, 0xD431, 0xD432, /* 0xE0 */
        0xD433, 0xD434, 	 0, 0xD435, 0xD436, 0xD437, 0xD438, 0xD439, /* 0xF0 */
        0xD43A, 0xD43B, 0xD43C, 0xD43D, 0xD43E, 0xD43F					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xce:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD440, 0xD441, 0xD442, 0xD443, 0xD444, 0xD445, 0xD446, /* 0x40 */
        0xD447, 0xD448, 0xD449, 0xD44A, 0xD44B, 0xD44C, 0xD44D, 0xD44E, /* 0x40 */
        0xD44F, 0xD450, 	 0, 0xD451, 0xD452, 0xD453, 0xD454, 0xD455, /* 0x50 */
        0xD456, 0xD457, 0xD458, 0xD459, 0xD45A, 0xD45B, 	 0, 	 0, /* 0x50 */
        0, 0xD45C, 0xD45D, 0xD45E, 0xD45F, 0xD460, 0xD461, 0xD462, /* 0x60 */
        0xD463, 0xD464, 0xD465, 0xD466, 0xD467, 0xD468, 0xD469, 0xD46A, /* 0x60 */
        0xD46B, 0xD46C, 	 0, 0xD46D, 0xD46E, 0xD46F, 0xD470, 0xD471, /* 0x70 */
        0xD472, 0xD473, 0xD474, 0xD475, 0xD476, 0xD477, 	 0, 	 0, /* 0x70 */
        0, 0xD478, 0xD479, 0xD47A, 0xD47B, 0xD47C, 0xD47D, 0xD47E, /* 0x80 */
        0xD47F, 0xD480, 0xD481, 0xD482, 0xD483, 0xD484, 0xD485, 0xD486, /* 0x80 */
        0xD487, 0xD488, 	 0, 0xD489, 0xD48A, 0xD48B, 0xD48C, 0xD48D, /* 0x90 */
        0xD48E, 0xD48F, 0xD490, 0xD491, 0xD492, 0xD493, 	 0, 	 0, /* 0x90 */
        0, 0xD494, 0xD495, 0xD496, 0xD497, 0xD498, 0xD499, 0xD49A, /* 0xA0 */
        0xD49B, 0xD49C, 0xD49D, 0xD49E, 0xD49F, 0xD4A0, 0xD4A1, 0xD4A2, /* 0xA0 */
        0xD4A3, 0xD4A4, 	 0, 0xD4A5, 0xD4A6, 0xD4A7, 0xD4A8, 0xD4A9, /* 0xB0 */
        0xD4AA, 0xD4AB, 0xD4AC, 0xD4AD, 0xD4AE, 0xD4AF, 	 0, 	 0, /* 0xB0 */
        0, 0xD4B0, 0xD4B1, 0xD4B2, 0xD4B3, 0xD4B4, 0xD4B5, 0xD4B6, /* 0xC0 */
        0xD4B7, 0xD4B8, 0xD4B9, 0xD4BA, 0xD4BB, 0xD4BC, 0xD4BD, 0xD4BE, /* 0xC0 */
        0xD4BF, 0xD4C0, 	 0, 0xD4C1, 0xD4C2, 0xD4C3, 0xD4C4, 0xD4C5, /* 0xD0 */
        0xD4C6, 0xD4C7, 0xD4C8, 0xD4C9, 0xD4CA, 0xD4CB, 	 0, 	 0, /* 0xD0 */
        0, 0xD4CC, 0xD4CD, 0xD4CE, 0xD4CF, 0xD4D0, 0xD4D1, 0xD4D2, /* 0xE0 */
        0xD4D3, 0xD4D4, 0xD4D5, 0xD4D6, 0xD4D7, 0xD4D8, 0xD4D9, 0xD4DA, /* 0xE0 */
        0xD4DB, 0xD4DC, 	 0, 0xD4DD, 0xD4DE, 0xD4DF, 0xD4E0, 0xD4E1, /* 0xF0 */
        0xD4E2, 0xD4E3, 0xD4E4, 0xD4E5, 0xD4E6, 0xD4E7					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xcf:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xD4E8, 0xD4E9, 0xD4EA, 0xD4EB, 0xD4EC, 0xD4ED, 0xD4EE, /* 0x40 */
        0xD4EF, 0xD4F0, 0xD4F1, 0xD4F2, 0xD4F3, 0xD4F4, 0xD4F5, 0xD4F6, /* 0x40 */
        0xD4F7, 0xD4F8, 	 0, 0xD4F9, 0xD4FA, 0xD4FB, 0xD4FC, 0xD4FD, /* 0x50 */
        0xD4FE, 0xD4FF, 0xD500, 0xD501, 0xD502, 0xD503, 	 0, 	 0, /* 0x50 */
        0, 0xD504, 0xD505, 0xD506, 0xD507, 0xD508, 0xD509, 0xD50A, /* 0x60 */
        0xD50B, 0xD50C, 0xD50D, 0xD50E, 0xD50F, 0xD510, 0xD511, 0xD512, /* 0x60 */
        0xD513, 0xD514, 	 0, 0xD515, 0xD516, 0xD517, 0xD518, 0xD519, /* 0x70 */
        0xD51A, 0xD51B, 0xD51C, 0xD51D, 0xD51E, 0xD51F, 	 0, 	 0, /* 0x70 */
        0, 0xD520, 0xD521, 0xD522, 0xD523, 0xD524, 0xD525, 0xD526, /* 0x80 */
        0xD527, 0xD528, 0xD529, 0xD52A, 0xD52B, 0xD52C, 0xD52D, 0xD52E, /* 0x80 */
        0xD52F, 0xD530, 	 0, 0xD531, 0xD532, 0xD533, 0xD534, 0xD535, /* 0x90 */
        0xD536, 0xD537, 0xD538, 0xD539, 0xD53A, 0xD53B, 	 0, 	 0, /* 0x90 */
        0, 0xD53C, 0xD53D, 0xD53E, 0xD53F, 0xD540, 0xD541, 0xD542, /* 0xA0 */
        0xD543, 0xD544, 0xD545, 0xD546, 0xD547, 0xD548, 0xD549, 0xD54A, /* 0xA0 */
        0xD54B, 0xD54C, 	 0, 0xD54D, 0xD54E, 0xD54F, 0xD550, 0xD551, /* 0xB0 */
        0xD552, 0xD553, 0xD554, 0xD555, 0xD556, 0xD557					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xd0:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0x314E, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x50 */
        0, 0xD558, 0xD559, 0xD55A, 0xD55B, 0xD55C, 0xD55D, 0xD55E, /* 0x60 */
        0xD55F, 0xD560, 0xD561, 0xD562, 0xD563, 0xD564, 0xD565, 0xD566, /* 0x60 */
        0xD567, 0xD568, 	 0, 0xD569, 0xD56A, 0xD56B, 0xD56C, 0xD56D, /* 0x70 */
        0xD56E, 0xD56F, 0xD570, 0xD571, 0xD572, 0xD573, 	 0, 	 0, /* 0x70 */
        0, 0xD574, 0xD575, 0xD576, 0xD577, 0xD578, 0xD579, 0xD57A, /* 0x80 */
        0xD57B, 0xD57C, 0xD57D, 0xD57E, 0xD57F, 0xD580, 0xD581, 0xD582, /* 0x80 */
        0xD583, 0xD584, 	 0, 0xD585, 0xD586, 0xD587, 0xD588, 0xD589, /* 0x90 */
        0xD58A, 0xD58B, 0xD58C, 0xD58D, 0xD58E, 0xD58F, 	 0, 	 0, /* 0x90 */
        0, 0xD590, 0xD591, 0xD592, 0xD593, 0xD594, 0xD595, 0xD596, /* 0xA0 */
        0xD597, 0xD598, 0xD599, 0xD59A, 0xD59B, 0xD59C, 0xD59D, 0xD59E, /* 0xA0 */
        0xD59F, 0xD5A0, 	 0, 0xD5A1, 0xD5A2, 0xD5A3, 0xD5A4, 0xD5A5, /* 0xB0 */
        0xD5A6, 0xD5A7, 0xD5A8, 0xD5A9, 0xD5AA, 0xD5AB, 	 0, 	 0, /* 0xB0 */
        0, 0xD5AC, 0xD5AD, 0xD5AE, 0xD5AF, 0xD5B0, 0xD5B1, 0xD5B2, /* 0xC0 */
        0xD5B3, 0xD5B4, 0xD5B5, 0xD5B6, 0xD5B7, 0xD5B8, 0xD5B9, 0xD5BA, /* 0xC0 */
        0xD5BB, 0xD5BC, 	 0, 0xD5BD, 0xD5BE, 0xD5BF, 0xD5C0, 0xD5C1, /* 0xD0 */
        0xD5C2, 0xD5C3, 0xD5C4, 0xD5C5, 0xD5C6, 0xD5C7, 	 0, 	 0, /* 0xD0 */
        0, 0xD5C8, 0xD5C9, 0xD5CA, 0xD5CB, 0xD5CC, 0xD5CD, 0xD5CE, /* 0xE0 */
        0xD5CF, 0xD5D0, 0xD5D1, 0xD5D2, 0xD5D3, 0xD5D4, 0xD5D5, 0xD5D6, /* 0xE0 */
        0xD5D7, 0xD5D8, 	 0, 0xD5D9, 0xD5DA, 0xD5DB, 0xD5DC, 0xD5DD, /* 0xF0 */
        0xD5DE, 0xD5DF, 0xD5E0, 0xD5E1, 0xD5E2, 0xD5E3					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xd1:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD5E4, 0xD5E5, 0xD5E6, 0xD5E7, 0xD5E8, 0xD5E9, 0xD5EA, /* 0x40 */
        0xD5EB, 0xD5EC, 0xD5ED, 0xD5EE, 0xD5EF, 0xD5F0, 0xD5F1, 0xD5F2, /* 0x40 */
        0xD5F3, 0xD5F4, 	 0, 0xD5F5, 0xD5F6, 0xD5F7, 0xD5F8, 0xD5F9, /* 0x50 */
        0xD5FA, 0xD5FB, 0xD5FC, 0xD5FD, 0xD5FE, 0xD5FF, 	 0, 	 0, /* 0x50 */
        0, 0xD600, 0xD601, 0xD602, 0xD603, 0xD604, 0xD605, 0xD606, /* 0x60 */
        0xD607, 0xD608, 0xD609, 0xD60A, 0xD60B, 0xD60C, 0xD60D, 0xD60E, /* 0x60 */
        0xD60F, 0xD610, 	 0, 0xD611, 0xD612, 0xD613, 0xD614, 0xD615, /* 0x70 */
        0xD616, 0xD617, 0xD618, 0xD619, 0xD61A, 0xD61B, 	 0, 	 0, /* 0x70 */
        0, 0xD61C, 0xD61D, 0xD61E, 0xD61F, 0xD620, 0xD621, 0xD622, /* 0x80 */
        0xD623, 0xD624, 0xD625, 0xD626, 0xD627, 0xD628, 0xD629, 0xD62A, /* 0x80 */
        0xD62B, 0xD62C, 	 0, 0xD62D, 0xD62E, 0xD62F, 0xD630, 0xD631, /* 0x90 */
        0xD632, 0xD633, 0xD634, 0xD635, 0xD636, 0xD637, 	 0, 	 0, /* 0x90 */
        0, 0xD638, 0xD639, 0xD63A, 0xD63B, 0xD63C, 0xD63D, 0xD63E, /* 0xA0 */
        0xD63F, 0xD640, 0xD641, 0xD642, 0xD643, 0xD644, 0xD645, 0xD646, /* 0xA0 */
        0xD647, 0xD648, 	 0, 0xD649, 0xD64A, 0xD64B, 0xD64C, 0xD64D, /* 0xB0 */
        0xD64E, 0xD64F, 0xD650, 0xD651, 0xD652, 0xD653, 	 0, 	 0, /* 0xB0 */
        0, 0xD654, 0xD655, 0xD656, 0xD657, 0xD658, 0xD659, 0xD65A, /* 0xC0 */
        0xD65B, 0xD65C, 0xD65D, 0xD65E, 0xD65F, 0xD660, 0xD661, 0xD662, /* 0xC0 */
        0xD663, 0xD664, 	 0, 0xD665, 0xD666, 0xD667, 0xD668, 0xD669, /* 0xD0 */
        0xD66A, 0xD66B, 0xD66C, 0xD66D, 0xD66E, 0xD66F, 	 0, 	 0, /* 0xD0 */
        0, 0xD670, 0xD671, 0xD672, 0xD673, 0xD674, 0xD675, 0xD676, /* 0xE0 */
        0xD677, 0xD678, 0xD679, 0xD67A, 0xD67B, 0xD67C, 0xD67D, 0xD67E, /* 0xE0 */
        0xD67F, 0xD680, 	 0, 0xD681, 0xD682, 0xD683, 0xD684, 0xD685, /* 0xF0 */
        0xD686, 0xD687, 0xD688, 0xD689, 0xD68A, 0xD68B					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xd2:
    if (c2>=0x41 && c2<=0xfd) {
      static int const(val[])= {
        0xD68C, 0xD68D, 0xD68E, 0xD68F, 0xD690, 0xD691, 0xD692, /* 0x40 */
        0xD693, 0xD694, 0xD695, 0xD696, 0xD697, 0xD698, 0xD699, 0xD69A, /* 0x40 */
        0xD69B, 0xD69C, 	 0, 0xD69D, 0xD69E, 0xD69F, 0xD6A0, 0xD6A1, /* 0x50 */
        0xD6A2, 0xD6A3, 0xD6A4, 0xD6A5, 0xD6A6, 0xD6A7, 	 0, 	 0, /* 0x50 */
        0, 0xD6A8, 0xD6A9, 0xD6AA, 0xD6AB, 0xD6AC, 0xD6AD, 0xD6AE, /* 0x60 */
        0xD6AF, 0xD6B0, 0xD6B1, 0xD6B2, 0xD6B3, 0xD6B4, 0xD6B5, 0xD6B6, /* 0x60 */
        0xD6B7, 0xD6B8, 	 0, 0xD6B9, 0xD6BA, 0xD6BB, 0xD6BC, 0xD6BD, /* 0x70 */
        0xD6BE, 0xD6BF, 0xD6C0, 0xD6C1, 0xD6C2, 0xD6C3, 	 0, 	 0, /* 0x70 */
        0, 0xD6C4, 0xD6C5, 0xD6C6, 0xD6C7, 0xD6C8, 0xD6C9, 0xD6CA, /* 0x80 */
        0xD6CB, 0xD6CC, 0xD6CD, 0xD6CE, 0xD6CF, 0xD6D0, 0xD6D1, 0xD6D2, /* 0x80 */
        0xD6D3, 0xD6D4, 	 0, 0xD6D5, 0xD6D6, 0xD6D7, 0xD6D8, 0xD6D9, /* 0x90 */
        0xD6DA, 0xD6DB, 0xD6DC, 0xD6DD, 0xD6DE, 0xD6DF, 	 0, 	 0, /* 0x90 */
        0, 0xD6E0, 0xD6E1, 0xD6E2, 0xD6E3, 0xD6E4, 0xD6E5, 0xD6E6, /* 0xA0 */
        0xD6E7, 0xD6E8, 0xD6E9, 0xD6EA, 0xD6EB, 0xD6EC, 0xD6ED, 0xD6EE, /* 0xA0 */
        0xD6EF, 0xD6F0, 	 0, 0xD6F1, 0xD6F2, 0xD6F3, 0xD6F4, 0xD6F5, /* 0xB0 */
        0xD6F6, 0xD6F7, 0xD6F8, 0xD6F9, 0xD6FA, 0xD6FB, 	 0, 	 0, /* 0xB0 */
        0, 0xD6FC, 0xD6FD, 0xD6FE, 0xD6FF, 0xD700, 0xD701, 0xD702, /* 0xC0 */
        0xD703, 0xD704, 0xD705, 0xD706, 0xD707, 0xD708, 0xD709, 0xD70A, /* 0xC0 */
        0xD70B, 0xD70C, 	 0, 0xD70D, 0xD70E, 0xD70F, 0xD710, 0xD711, /* 0xD0 */
        0xD712, 0xD713, 0xD714, 0xD715, 0xD716, 0xD717, 	 0, 	 0, /* 0xD0 */
        0, 0xD718, 0xD719, 0xD71A, 0xD71B, 0xD71C, 0xD71D, 0xD71E, /* 0xE0 */
        0xD71F, 0xD720, 0xD721, 0xD722, 0xD723, 0xD724, 0xD725, 0xD726, /* 0xE0 */
        0xD727, 0xD728, 	 0, 0xD729, 0xD72A, 0xD72B, 0xD72C, 0xD72D, /* 0xF0 */
        0xD72E, 0xD72F, 0xD730, 0xD731, 0xD732, 0xD733					/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xd3:
    if (c2>=0x41 && c2<=0xbd) {
      static int const(val[])= {
        0xD734, 0xD735, 0xD736, 0xD737, 0xD738, 0xD739, 0xD73A, /* 0x40 */
        0xD73B, 0xD73C, 0xD73D, 0xD73E, 0xD73F, 0xD740, 0xD741, 0xD742, /* 0x40 */
        0xD743, 0xD744, 	 0, 0xD745, 0xD746, 0xD747, 0xD748, 0xD749, /* 0x50 */
        0xD74A, 0xD74B, 0xD74C, 0xD74D, 0xD74E, 0xD74F, 	 0, 	 0, /* 0x50 */
        0, 0xD750, 0xD751, 0xD752, 0xD753, 0xD754, 0xD755, 0xD756, /* 0x60 */
        0xD757, 0xD758, 0xD759, 0xD75A, 0xD75B, 0xD75C, 0xD75D, 0xD75E, /* 0x60 */
        0xD75F, 0xD760, 	 0, 0xD761, 0xD762, 0xD763, 0xD764, 0xD765, /* 0x70 */
        0xD766, 0xD767, 0xD768, 0xD769, 0xD76A, 0xD76B, 	 0, 	 0, /* 0x70 */
        0, 0xD76C, 0xD76D, 0xD76E, 0xD76F, 0xD770, 0xD771, 0xD772, /* 0x80 */
        0xD773, 0xD774, 0xD775, 0xD776, 0xD777, 0xD778, 0xD779, 0xD77A, /* 0x80 */
        0xD77B, 0xD77C, 	 0, 0xD77D, 0xD77E, 0xD77F, 0xD780, 0xD781, /* 0x90 */
        0xD782, 0xD783, 0xD784, 0xD785, 0xD786, 0xD787, 	 0, 	 0, /* 0x90 */
        0, 0xD788, 0xD789, 0xD78A, 0xD78B, 0xD78C, 0xD78D, 0xD78E, /* 0xA0 */
        0xD78F, 0xD790, 0xD791, 0xD792, 0xD793, 0xD794, 0xD795, 0xD796, /* 0xA0 */
        0xD797, 0xD798, 	 0, 0xD799, 0xD79A, 0xD79B, 0xD79C, 0xD79D, /* 0xB0 */
        0xD79E, 0xD79F, 0xD7A0, 0xD7A1, 0xD7A2, 0xD7A3					/* 0xB0 */
      };
      unicode=(uint32_t) val[c2-0x41];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xd9:
    if (c2>=0x31 && c2<=0xe5) {
      static int const(val[])= {
        0x3000, 0x3001, 0x3002, 0x00B7, 0x2025, 0x2026, 0x00A8, /* 0x30 */
        0x3003, 0x00AD, 0x2015, 0x2225, 0xFF3C, 0x223C, 0x2018, 0x2019, /* 0x30 */
        0x201C, 0x201D, 0x3014, 0x3015, 0x3008, 0x3009, 0x300A, 0x300B, /* 0x40 */
        0x300C, 0x300D, 0x300E, 0x300F, 0x3010, 0x3011, 0x00B1, 0x00D7, /* 0x40 */
        0x00F7, 0x2260, 0x2264, 0x2265, 0x221E, 0x2234, 0x00B0, 0x2032, /* 0x50 */
        0x2033, 0x2103, 0x212B, 0xFFE0, 0xFFE1, 0xFFE5, 0x2642, 0x2640, /* 0x50 */
        0x2220, 0x22A5, 0x2312, 0x2202, 0x2207, 0x2261, 0x2252, 0x00A7, /* 0x60 */
        0x203B, 0x2606, 0x2605, 0x25CB, 0x25CF, 0x25CE, 0x25C7, 0x25C6, /* 0x60 */
        0x25A1, 0x25A0, 0x25B3, 0x25B2, 0x25BD, 0x25BC, 0x2192, 0x2190, /* 0x70 */
        0x2191, 0x2193, 0x2194, 0x3013, 0x226A, 0x226B, 0x221A, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x223D, 0x221D, 0x2235, 0x222B, 0x222C, 0x2208, 0x220B, /* 0x90 */
        0x2286, 0x2287, 0x2282, 0x2283, 0x222A, 0x2229, 0x2227, 0x2228, /* 0x90 */
        0xFFE2, 0x21D2, 0x21D4, 0x2200, 0x2203, 0x00B4, 0xFF5E, 0x02C7, /* 0xA0 */
        0x02D8, 0x02DD, 0x02DA, 0x02D9, 0x00B8, 0x02DB, 0x00A1, 0x00BF, /* 0xA0 */
        0x02D0, 0x222E, 0x2211, 0x220F, 0x00A4, 0x2109, 0x2030, 0x25C1, /* 0xB0 */
        0x25C0, 0x25B7, 0x25B6, 0x2664, 0x2660, 0x2661, 0x2665, 0x2667, /* 0xB0 */
        0x2663, 0x2299, 0x25C8, 0x25A3, 0x25D0, 0x25D1, 0x2592, 0x25A4, /* 0xC0 */
        0x25A5, 0x25A8, 0x25A7, 0x25A6, 0x25A9, 0x2668, 0x260F, 0x260E, /* 0xC0 */
        0x261C, 0x261E, 0x00B6, 0x2020, 0x2021, 0x2195, 0x2197, 0x2199, /* 0xD0 */
        0x2196, 0x2198, 0x266D, 0x2669, 0x266A, 0x266C, 0x327F, 0x321C, /* 0xD0 */
        0x2116, 0x33C7, 0x2122, 0x33C2, 0x33D8, 0x2121					/* 0xE0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xda:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0xFF01, 0xFF02, 0xFF03, 0xFF04, 0xFF05, 0xFF06, 0xFF07, /* 0x30 */
        0xFF08, 0xFF09, 0xFF0A, 0xFF0B, 0xFF0C, 0xFF0D, 0xFF0E, 0xFF0F, /* 0x30 */
        0xFF10, 0xFF11, 0xFF12, 0xFF13, 0xFF14, 0xFF15, 0xFF16, 0xFF17, /* 0x40 */
        0xFF18, 0xFF19, 0xFF1A, 0xFF1B, 0xFF1C, 0xFF1D, 0xFF1E, 0xFF1F, /* 0x40 */
        0xFF20, 0xFF21, 0xFF22, 0xFF23, 0xFF24, 0xFF25, 0xFF26, 0xFF27, /* 0x50 */
        0xFF28, 0xFF29, 0xFF2A, 0xFF2B, 0xFF2C, 0xFF2D, 0xFF2E, 0xFF2F, /* 0x50 */
        0xFF30, 0xFF31, 0xFF32, 0xFF33, 0xFF34, 0xFF35, 0xFF36, 0xFF37, /* 0x60 */
        0xFF38, 0xFF39, 0xFF3A, 0xFF3B, 0xFFE6, 0xFF3D, 0xFF3E, 0xFF3F, /* 0x60 */
        0xFF40, 0xFF41, 0xFF42, 0xFF43, 0xFF44, 0xFF45, 0xFF46, 0xFF47, /* 0x70 */
        0xFF48, 0xFF49, 0xFF4A, 0xFF4B, 0xFF4C, 0xFF4D, 0xFF4E, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0xFF4F, 0xFF50, 0xFF51, 0xFF52, 0xFF53, 0xFF54, 0xFF55, /* 0x90 */
        0xFF56, 0xFF57, 0xFF58, 0xFF59, 0xFF5A, 0xFF5B, 0xFF5C, 0xFF5D, /* 0x90 */
        0xFFE3, 0x3131, 0x3132, 0x3133, 0x3134, 0x3135, 0x3136, 0x3137, /* 0xA0 */
        0x3138, 0x3139, 0x313A, 0x313B, 0x313C, 0x313D, 0x313E, 0x313F, /* 0xA0 */
        0x3140, 0x3141, 0x3142, 0x3143, 0x3144, 0x3145, 0x3146, 0x3147, /* 0xB0 */
        0x3148, 0x3149, 0x314A, 0x314B, 0x314C, 0x314D, 0x314E, 0x314F, /* 0xB0 */
        0x3150, 0x3151, 0x3152, 0x3153, 0x3154, 0x3155, 0x3156, 0x3157, /* 0xC0 */
        0x3158, 0x3159, 0x315A, 0x315B, 0x315C, 0x315D, 0x315E, 0x315F, /* 0xC0 */
        0x3160, 0x3161, 0x3162, 0x3163, 0x3164, 0x3165, 0x3166, 0x3167, /* 0xD0 */
        0x3168, 0x3169, 0x316A, 0x316B, 0x316C, 0x316D, 0x316E, 0x316F, /* 0xD0 */
        0x3170, 0x3171, 0x3172, 0x3173, 0x3174, 0x3175, 0x3176, 0x3177, /* 0xE0 */
        0x3178, 0x3179, 0x317A, 0x317B, 0x317C, 0x317D, 0x317E, 0x317F, /* 0xE0 */
        0x3180, 0x3181, 0x3182, 0x3183, 0x3184, 0x3185, 0x3186, 0x3187, /* 0xF0 */
        0x3188, 0x3189, 0x318A, 0x318B, 0x318C, 0x318D, 0x318E			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xdb:
    if (c2>=0x31 && c2<=0xe4) {
      static int const(val[])= {
        0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, /* 0x30 */
        0x2177, 0x2178, 0x2179, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x30 */
        0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165, 0x2166, 0x2167, /* 0x40 */
        0x2168, 0x2169, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x40 */
        0, 0x0391, 0x0392, 0x0393, 0x0394, 0x0395, 0x0396, 0x0397, /* 0x50 */
        0x0398, 0x0399, 0x039A, 0x039B, 0x039C, 0x039D, 0x039E, 0x039F, /* 0x50 */
        0x03A0, 0x03A1, 0x03A3, 0x03A4, 0x03A5, 0x03A6, 0x03A7, 0x03A8, /* 0x60 */
        0x03A9, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x60 */
        0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7, /* 0x70 */
        0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x03BF, 0x03C0, 0x03C1, 0x03C3, 0x03C4, 0x03C5, 0x03C6, /* 0x90 */
        0x03C7, 0x03C8, 0x03C9, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x2500, 0x2502, 0x250C, 0x2510, 0x2518, 0x2514, 0x251C, /* 0xA0 */
        0x252C, 0x2524, 0x2534, 0x253C, 0x2501, 0x2503, 0x250F, 0x2513, /* 0xA0 */
        0x251B, 0x2517, 0x2523, 0x2533, 0x252B, 0x253B, 0x254B, 0x2520, /* 0xB0 */
        0x252F, 0x2528, 0x2537, 0x253F, 0x251D, 0x2530, 0x2525, 0x2538, /* 0xB0 */
        0x2542, 0x2512, 0x2511, 0x251A, 0x2519, 0x2516, 0x2515, 0x250E, /* 0xC0 */
        0x250D, 0x251E, 0x251F, 0x2521, 0x2522, 0x2526, 0x2527, 0x2529, /* 0xC0 */
        0x252A, 0x252D, 0x252E, 0x2531, 0x2532, 0x2535, 0x2536, 0x2539, /* 0xD0 */
        0x253A, 0x253D, 0x253E, 0x2540, 0x2541, 0x2543, 0x2544, 0x2545, /* 0xD0 */
        0x2546, 0x2547, 0x2548, 0x2549, 0x254A							/* 0xE0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xdc:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x3395, 0x3396, 0x3397, 0x2113, 0x3398, 0x33C4, 0x33A3, /* 0x30 */
        0x33A4, 0x33A5, 0x33A6, 0x3399, 0x339A, 0x339B, 0x339C, 0x339D, /* 0x30 */
        0x339E, 0x339F, 0x33A0, 0x33A1, 0x33A2, 0x33CA, 0x338D, 0x338E, /* 0x40 */
        0x338F, 0x33CF, 0x3388, 0x3389, 0x33C8, 0x33A7, 0x33A8, 0x33B0, /* 0x40 */
        0x33B1, 0x33B2, 0x33B3, 0x33B4, 0x33B5, 0x33B6, 0x33B7, 0x33B8, /* 0x50 */
        0x33B9, 0x3380, 0x3381, 0x3382, 0x3383, 0x3384, 0x33BA, 0x33BB, /* 0x50 */
        0x33BC, 0x33BD, 0x33BE, 0x33BF, 0x3390, 0x3391, 0x3392, 0x3393, /* 0x60 */
        0x3394, 0x2126, 0x33C0, 0x33C1, 0x338A, 0x338B, 0x338C, 0x33D6, /* 0x60 */
        0x33C5, 0x33AD, 0x33AE, 0x33AF, 0x33DB, 0x33A9, 0x33AA, 0x33AB, /* 0x70 */
        0x33AC, 0x33DD, 0x33D0, 0x33D3, 0x33C3, 0x33C9, 0x33DC, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x33C6, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x00C6, 0x00D0, 0x00AA, 0x0126, 	 0, 0x0132, 	 0, /* 0xA0 */
        0x013F, 0x0141, 0x00D8, 0x0152, 0x00BA, 0x00DE, 0x0166, 0x014A, /* 0xA0 */
        0, 0x3260, 0x3261, 0x3262, 0x3263, 0x3264, 0x3265, 0x3266, /* 0xB0 */
        0x3267, 0x3268, 0x3269, 0x326A, 0x326B, 0x326C, 0x326D, 0x326E, /* 0xB0 */
        0x326F, 0x3270, 0x3271, 0x3272, 0x3273, 0x3274, 0x3275, 0x3276, /* 0xC0 */
        0x3277, 0x3278, 0x3279, 0x327A, 0x327B, 0x24D0, 0x24D1, 0x24D2, /* 0xC0 */
        0x24D3, 0x24D4, 0x24D5, 0x24D6, 0x24D7, 0x24D8, 0x24D9, 0x24DA, /* 0xD0 */
        0x24DB, 0x24DC, 0x24DD, 0x24DE, 0x24DF, 0x24E0, 0x24E1, 0x24E2, /* 0xD0 */
        0x24E3, 0x24E4, 0x24E5, 0x24E6, 0x24E7, 0x24E8, 0x24E9, 0x2460, /* 0xE0 */
        0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467, 0x2468, /* 0xE0 */
        0x2469, 0x246A, 0x246B, 0x246C, 0x246D, 0x246E, 0x00BD, 0x2153, /* 0xF0 */
        0x2154, 0x00BC, 0x00BE, 0x215B, 0x215C, 0x215D, 0x215E			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xdd:
    if (c2>=0x31 && c2<=0xf3) {
      static int const(val[])= {
        0x00E6, 0x0111, 0x00F0, 0x0127, 0x0131, 0x0133, 0x0138, /* 0x30 */
        0x0140, 0x0142, 0x00F8, 0x0153, 0x00DF, 0x00FE, 0x0167, 0x014B, /* 0x30 */
        0x0149, 0x3200, 0x3201, 0x3202, 0x3203, 0x3204, 0x3205, 0x3206, /* 0x40 */
        0x3207, 0x3208, 0x3209, 0x320A, 0x320B, 0x320C, 0x320D, 0x320E, /* 0x40 */
        0x320F, 0x3210, 0x3211, 0x3212, 0x3213, 0x3214, 0x3215, 0x3216, /* 0x50 */
        0x3217, 0x3218, 0x3219, 0x321A, 0x321B, 0x249C, 0x249D, 0x249E, /* 0x50 */
        0x249F, 0x24A0, 0x24A1, 0x24A2, 0x24A3, 0x24A4, 0x24A5, 0x24A6, /* 0x60 */
        0x24A7, 0x24A8, 0x24A9, 0x24AA, 0x24AB, 0x24AC, 0x24AD, 0x24AE, /* 0x60 */
        0x24AF, 0x24B0, 0x24B1, 0x24B2, 0x24B3, 0x24B4, 0x24B5, 0x2474, /* 0x70 */
        0x2475, 0x2476, 0x2477, 0x2478, 0x2479, 0x247A, 0x247B, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x247C, 0x247D, 0x247E, 0x247F, 0x2480, 0x2481, 0x2482, /* 0x90 */
        0x00B9, 0x00B2, 0x00B3, 0x2074, 0x207F, 0x2081, 0x2082, 0x2083, /* 0x90 */
        0x2084, 0x3041, 0x3042, 0x3043, 0x3044, 0x3045, 0x3046, 0x3047, /* 0xA0 */
        0x3048, 0x3049, 0x304A, 0x304B, 0x304C, 0x304D, 0x304E, 0x304F, /* 0xA0 */
        0x3050, 0x3051, 0x3052, 0x3053, 0x3054, 0x3055, 0x3056, 0x3057, /* 0xB0 */
        0x3058, 0x3059, 0x305A, 0x305B, 0x305C, 0x305D, 0x305E, 0x305F, /* 0xB0 */
        0x3060, 0x3061, 0x3062, 0x3063, 0x3064, 0x3065, 0x3066, 0x3067, /* 0xC0 */
        0x3068, 0x3069, 0x306A, 0x306B, 0x306C, 0x306D, 0x306E, 0x306F, /* 0xC0 */
        0x3070, 0x3071, 0x3072, 0x3073, 0x3074, 0x3075, 0x3076, 0x3077, /* 0xD0 */
        0x3078, 0x3079, 0x307A, 0x307B, 0x307C, 0x307D, 0x307E, 0x307F, /* 0xD0 */
        0x3080, 0x3081, 0x3082, 0x3083, 0x3084, 0x3085, 0x3086, 0x3087, /* 0xE0 */
        0x3088, 0x3089, 0x308A, 0x308B, 0x308C, 0x308D, 0x308E, 0x308F, /* 0xE0 */
        0x3090, 0x3091, 0x3092, 0x3093									/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xde:
    if (c2>=0x31 && c2<=0xf1) {
      static int const(val[])= {
        0x30A1, 0x30A2, 0x30A3, 0x30A4, 0x30A5, 0x30A6, 0x30A7, /* 0x30 */
        0x30A8, 0x30A9, 0x30AA, 0x30AB, 0x30AC, 0x30AD, 0x30AE, 0x30AF, /* 0x30 */
        0x30B0, 0x30B1, 0x30B2, 0x30B3, 0x30B4, 0x30B5, 0x30B6, 0x30B7, /* 0x40 */
        0x30B8, 0x30B9, 0x30BA, 0x30BB, 0x30BC, 0x30BD, 0x30BE, 0x30BF, /* 0x40 */
        0x30C0, 0x30C1, 0x30C2, 0x30C3, 0x30C4, 0x30C5, 0x30C6, 0x30C7, /* 0x50 */
        0x30C8, 0x30C9, 0x30CA, 0x30CB, 0x30CC, 0x30CD, 0x30CE, 0x30CF, /* 0x50 */
        0x30D0, 0x30D1, 0x30D2, 0x30D3, 0x30D4, 0x30D5, 0x30D6, 0x30D7, /* 0x60 */
        0x30D8, 0x30D9, 0x30DA, 0x30DB, 0x30DC, 0x30DD, 0x30DE, 0x30DF, /* 0x60 */
        0x30E0, 0x30E1, 0x30E2, 0x30E3, 0x30E4, 0x30E5, 0x30E6, 0x30E7, /* 0x70 */
        0x30E8, 0x30E9, 0x30EA, 0x30EB, 0x30EC, 0x30ED, 0x30EE, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x30EF, 0x30F0, 0x30F1, 0x30F2, 0x30F3, 0x30F4, 0x30F5, /* 0x90 */
        0x30F6, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x90 */
        0, 0x0410, 0x0411, 0x0412, 0x0413, 0x0414, 0x0415, 0x0401, /* 0xA0 */
        0x0416, 0x0417, 0x0418, 0x0419, 0x041A, 0x041B, 0x041C, 0x041D, /* 0xA0 */
        0x041E, 0x041F, 0x0420, 0x0421, 0x0422, 0x0423, 0x0424, 0x0425, /* 0xB0 */
        0x0426, 0x0427, 0x0428, 0x0429, 0x042A, 0x042B, 0x042C, 0x042D, /* 0xB0 */
        0x042E, 0x042F, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0xC0 */
        0, 0x0430, 0x0431, 0x0432, 0x0433, 0x0434, 0x0435, 0x0451, /* 0xD0 */
        0x0436, 0x0437, 0x0438, 0x0439, 0x043A, 0x043B, 0x043C, 0x043D, /* 0xD0 */
        0x043E, 0x043F, 0x0440, 0x0441, 0x0442, 0x0443, 0x0444, 0x0445, /* 0xE0 */
        0x0446, 0x0447, 0x0448, 0x0449, 0x044A, 0x044B, 0x044C, 0x044D, /* 0xE0 */
        0x044E, 0x044F													/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe0:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x4F3D, 0x4F73, 0x5047, 0x50F9, 0x52A0, 0x53EF, 0x5475, /* 0x30 */
        0x54E5, 0x5609, 0x5AC1, 0x5BB6, 0x6687, 0x67B6, 0x67B7, 0x67EF, /* 0x30 */
        0x6B4C, 0x73C2, 0x75C2, 0x7A3C, 0x82DB, 0x8304, 0x8857, 0x8888, /* 0x40 */
        0x8A36, 0x8CC8, 0x8DCF, 0x8EFB, 0x8FE6, 0x99D5, 0x523B, 0x5374, /* 0x40 */
        0x5404, 0x606A, 0x6164, 0x6BBC, 0x73CF, 0x811A, 0x89BA, 0x89D2, /* 0x50 */
        0x95A3, 0x4F83, 0x520A, 0x58BE, 0x5978, 0x59E6, 0x5E72, 0x5E79, /* 0x50 */
        0x61C7, 0x63C0, 0x6746, 0x67EC, 0x687F, 0x6F97, 0x764E, 0x770B, /* 0x60 */
        0x78F5, 0x7A08, 0x7AFF, 0x7C21, 0x809D, 0x826E, 0x8271, 0x8AEB, /* 0x60 */
        0x9593, 0x4E6B, 0x559D, 0x66F7, 0x6E34, 0x78A3, 0x7AED, 0x845B, /* 0x70 */
        0x8910, 0x874E, 0x97A8, 0x52D8, 0x574E, 0x582A, 0x5D4C, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x611F, 0x61BE, 0x6221, 0x6562, 0x67D1, 0x6A44, 0x6E1B, /* 0x90 */
        0x7518, 0x75B3, 0x76E3, 0x77B0, 0x7D3A, 0x90AF, 0x9451, 0x9452, /* 0x90 */
        0x9F95, 0x5323, 0x5CAC, 0x7532, 0x80DB, 0x9240, 0x9598, 0x525B, /* 0xA0 */
        0x5808, 0x59DC, 0x5CA1, 0x5D17, 0x5EB7, 0x5F3A, 0x5F4A, 0x6177, /* 0xA0 */
        0x6C5F, 0x757A, 0x7586, 0x7CE0, 0x7D73, 0x7DB1, 0x7F8C, 0x8154, /* 0xB0 */
        0x8221, 0x8591, 0x8941, 0x8B1B, 0x92FC, 0x964D, 0x9C47, 0x4ECB, /* 0xB0 */
        0x4EF7, 0x500B, 0x51F1, 0x584F, 0x6137, 0x613E, 0x6168, 0x6539, /* 0xC0 */
        0x69EA, 0x6F11, 0x75A5, 0x7686, 0x76D6, 0x7B87, 0x82A5, 0x84CB, /* 0xC0 */
        0xF900, 0x93A7, 0x958B, 0x5580, 0x5BA2, 0x5751, 0xF901, 0x7CB3, /* 0xD0 */
        0x7FB9, 0x91B5, 0x5028, 0x53BB, 0x5C45, 0x5DE8, 0x62D2, 0x636E, /* 0xD0 */
        0x64DA, 0x64E7, 0x6E20, 0x70AC, 0x795B, 0x8DDD, 0x8E1E, 0xF902, /* 0xE0 */
        0x907D, 0x9245, 0x92F8, 0x4E7E, 0x4EF6, 0x5065, 0x5DFE, 0x5EFA, /* 0xE0 */
        0x6106, 0x6957, 0x8171, 0x8654, 0x8E47, 0x9375, 0x9A2B, 0x4E5E, /* 0xF0 */
        0x5091, 0x6770, 0x6840, 0x5109, 0x528D, 0x5292, 0x6AA2			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe1:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x77BC, 0x9210, 0x9ED4, 0x52AB, 0x602F, 0x8FF2, 0x5048, /* 0x30 */
        0x61A9, 0x63ED, 0x64CA, 0x683C, 0x6A84, 0x6FC0, 0x8188, 0x89A1, /* 0x30 */
        0x9694, 0x5805, 0x727D, 0x72AC, 0x7504, 0x7D79, 0x7E6D, 0x80A9, /* 0x40 */
        0x898B, 0x8B74, 0x9063, 0x9D51, 0x6289, 0x6C7A, 0x6F54, 0x7D50, /* 0x40 */
        0x7F3A, 0x8A23, 0x517C, 0x614A, 0x7B9D, 0x8B19, 0x9257, 0x938C, /* 0x50 */
        0x4EAC, 0x4FD3, 0x501E, 0x50BE, 0x5106, 0x52C1, 0x52CD, 0x537F, /* 0x50 */
        0x5770, 0x5883, 0x5E9A, 0x5F91, 0x6176, 0x61AC, 0x64CE, 0x656C, /* 0x60 */
        0x666F, 0x66BB, 0x66F4, 0x6897, 0x6D87, 0x7085, 0x70F1, 0x749F, /* 0x60 */
        0x74A5, 0x74CA, 0x75D9, 0x786C, 0x78EC, 0x7ADF, 0x7AF6, 0x7D45, /* 0x70 */
        0x7D93, 0x8015, 0x803F, 0x811B, 0x8396, 0x8B66, 0x8F15, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x9015, 0x93E1, 0x9803, 0x9838, 0x9A5A, 0x9BE8, 0x4FC2, /* 0x90 */
        0x5553, 0x583A, 0x5951, 0x5B63, 0x5C46, 0x60B8, 0x6212, 0x6842, /* 0x90 */
        0x68B0, 0x68E8, 0x6EAA, 0x754C, 0x7678, 0x78CE, 0x7A3D, 0x7CFB, /* 0xA0 */
        0x7E6B, 0x7E7C, 0x8A08, 0x8AA1, 0x8C3F, 0x968E, 0x9DC4, 0x53E4, /* 0xA0 */
        0x53E9, 0x544A, 0x5471, 0x56FA, 0x59D1, 0x5B64, 0x5C3B, 0x5EAB, /* 0xB0 */
        0x62F7, 0x6537, 0x6545, 0x6572, 0x66A0, 0x67AF, 0x69C1, 0x6CBD, /* 0xB0 */
        0x75FC, 0x7690, 0x777E, 0x7A3F, 0x7F94, 0x8003, 0x80A1, 0x818F, /* 0xC0 */
        0x82E6, 0x82FD, 0x83F0, 0x85C1, 0x8831, 0x88B4, 0x8AA5, 0xF903, /* 0xC0 */
        0x8F9C, 0x932E, 0x96C7, 0x9867, 0x9AD8, 0x9F13, 0x54ED, 0x659B, /* 0xD0 */
        0x66F2, 0x688F, 0x7A40, 0x8C37, 0x9D60, 0x56F0, 0x5764, 0x5D11, /* 0xD0 */
        0x6606, 0x68B1, 0x68CD, 0x6EFE, 0x7428, 0x889E, 0x9BE4, 0x6C68, /* 0xE0 */
        0xF904, 0x9AA8, 0x4F9B, 0x516C, 0x5171, 0x529F, 0x5B54, 0x5DE5, /* 0xE0 */
        0x6050, 0x606D, 0x62F1, 0x63A7, 0x653B, 0x73D9, 0x7A7A, 0x86A3, /* 0xF0 */
        0x8CA2, 0x978F, 0x4E32, 0x5BE1, 0x6208, 0x679C, 0x74DC			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe2:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x79D1, 0x83D3, 0x8A87, 0x8AB2, 0x8DE8, 0x904E, 0x934B, /* 0x30 */
        0x9846, 0x5ED3, 0x69E8, 0x85FF, 0x90ED, 0xF905, 0x51A0, 0x5B98, /* 0x30 */
        0x5BEC, 0x6163, 0x68FA, 0x6B3E, 0x704C, 0x742F, 0x74D8, 0x7BA1, /* 0x40 */
        0x7F50, 0x83C5, 0x89C0, 0x8CAB, 0x95DC, 0x9928, 0x522E, 0x605D, /* 0x40 */
        0x62EC, 0x9002, 0x4F8A, 0x5149, 0x5321, 0x58D9, 0x5EE3, 0x66E0, /* 0x50 */
        0x6D38, 0x709A, 0x72C2, 0x73D6, 0x7B50, 0x80F1, 0x945B, 0x5366, /* 0x50 */
        0x639B, 0x7F6B, 0x4E56, 0x5080, 0x584A, 0x58DE, 0x602A, 0x6127, /* 0x60 */
        0x62D0, 0x69D0, 0x9B41, 0x5B8F, 0x7D18, 0x80B1, 0x8F5F, 0x4EA4, /* 0x60 */
        0x50D1, 0x54AC, 0x55AC, 0x5B0C, 0x5DA0, 0x5DE7, 0x652A, 0x654E, /* 0x70 */
        0x6821, 0x6A4B, 0x72E1, 0x768E, 0x77EF, 0x7D5E, 0x7FF9, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x81A0, 0x854E, 0x86DF, 0x8F03, 0x8F4E, 0x90CA, 0x9903, /* 0x90 */
        0x9A55, 0x9BAB, 0x4E18, 0x4E45, 0x4E5D, 0x4EC7, 0x4FF1, 0x5177, /* 0x90 */
        0x52FE, 0x5340, 0x53E3, 0x53E5, 0x548E, 0x5614, 0x5775, 0x57A2, /* 0xA0 */
        0x5BC7, 0x5D87, 0x5ED0, 0x61FC, 0x62D8, 0x6551, 0x67B8, 0x67E9, /* 0xA0 */
        0x69CB, 0x6B50, 0x6BC6, 0x6BEC, 0x6C42, 0x6E9D, 0x7078, 0x72D7, /* 0xB0 */
        0x7396, 0x7403, 0x77BF, 0x77E9, 0x7A76, 0x7D7F, 0x8009, 0x81FC, /* 0xB0 */
        0x8205, 0x820A, 0x82DF, 0x8862, 0x8B33, 0x8CFC, 0x8EC0, 0x9011, /* 0xC0 */
        0x90B1, 0x9264, 0x92B6, 0x99D2, 0x9A45, 0x9CE9, 0x9DD7, 0x9F9C, /* 0xC0 */
        0x570B, 0x5C40, 0x83CA, 0x97A0, 0x97AB, 0x9EB4, 0x541B, 0x7A98, /* 0xD0 */
        0x7FA4, 0x88D9, 0x8ECD, 0x90E1, 0x5800, 0x5C48, 0x6398, 0x7A9F, /* 0xD0 */
        0x5BAE, 0x5F13, 0x7A79, 0x7AAE, 0x828E, 0x8EAC, 0x5026, 0x5238, /* 0xE0 */
        0x52F8, 0x5377, 0x5708, 0x62F3, 0x6372, 0x6B0A, 0x6DC3, 0x7737, /* 0xE0 */
        0x53A5, 0x7357, 0x8568, 0x8E76, 0x95D5, 0x673A, 0x6AC3, 0x6F70, /* 0xF0 */
        0x8A6D, 0x8ECC, 0x994B, 0xF906, 0x6677, 0x6B78, 0x8CB4			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe3:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x9B3C, 0xF907, 0x53EB, 0x572D, 0x594E, 0x63C6, 0x69FB, /* 0x30 */
        0x73EA, 0x7845, 0x7ABA, 0x7AC5, 0x7CFE, 0x8475, 0x898F, 0x8D73, /* 0x30 */
        0x9035, 0x95A8, 0x52FB, 0x5747, 0x7547, 0x7B60, 0x83CC, 0x921E, /* 0x40 */
        0xF908, 0x6A58, 0x514B, 0x524B, 0x5287, 0x621F, 0x68D8, 0x6975, /* 0x40 */
        0x9699, 0x50C5, 0x52A4, 0x52E4, 0x61C3, 0x65A4, 0x6839, 0x69FF, /* 0x50 */
        0x747E, 0x7B4B, 0x82B9, 0x83EB, 0x89B2, 0x8B39, 0x8FD1, 0x9949, /* 0x50 */
        0xF909, 0x4ECA, 0x5997, 0x64D2, 0x6611, 0x6A8E, 0x7434, 0x7981, /* 0x60 */
        0x79BD, 0x82A9, 0x887E, 0x887F, 0x895F, 0xF90A, 0x9326, 0x4F0B, /* 0x60 */
        0x53CA, 0x6025, 0x6271, 0x6C72, 0x7D1A, 0x7D66, 0x4E98, 0x5162, /* 0x70 */
        0x77DC, 0x80AF, 0x4F01, 0x4F0E, 0x5176, 0x5180, 0x55DC, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x5668, 0x573B, 0x57FA, 0x57FC, 0x5914, 0x5947, 0x5993, /* 0x90 */
        0x5BC4, 0x5C90, 0x5D0E, 0x5DF1, 0x5E7E, 0x5FCC, 0x6280, 0x65D7, /* 0x90 */
        0x65E3, 0x671E, 0x671F, 0x675E, 0x68CB, 0x68C4, 0x6A5F, 0x6B3A, /* 0xA0 */
        0x6C23, 0x6C7D, 0x6C82, 0x6DC7, 0x7398, 0x7426, 0x742A, 0x7482, /* 0xA0 */
        0x74A3, 0x7578, 0x757F, 0x7881, 0x78EF, 0x7941, 0x7947, 0x7948, /* 0xB0 */
        0x797A, 0x7B95, 0x7D00, 0x7DBA, 0x7F88, 0x8006, 0x802D, 0x808C, /* 0xB0 */
        0x8A18, 0x8B4F, 0x8C48, 0x8D77, 0x9321, 0x9324, 0x98E2, 0x9951, /* 0xC0 */
        0x9A0E, 0x9A0F, 0x9A65, 0x9E92, 0x7DCA, 0x4F76, 0x5409, 0x62EE, /* 0xC0 */
        0x6854, 0x91D1, 0x55AB, 0x513A, 0xF90B, 0xF90C, 0x5A1C, 0x61E6, /* 0xD0 */
        0xF90D, 0x62CF, 0x62FF, 0xF90E, 0xF90F, 0xF910, 0xF911, 0xF912, /* 0xD0 */
        0xF913, 0x90A3, 0xF914, 0xF915, 0xF916, 0xF917, 0xF918, 0x8AFE, /* 0xE0 */
        0xF919, 0xF91A, 0xF91B, 0xF91C, 0x6696, 0xF91D, 0x7156, 0xF91E, /* 0xE0 */
        0xF91F, 0x96E3, 0xF920, 0x634F, 0x637A, 0x5357, 0xF921, 0x678F, /* 0xF0 */
        0x6960, 0x6E73, 0xF922, 0x7537, 0xF923, 0xF924, 0xF925			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe4:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x7D0D, 0xF926, 0xF927, 0x8872, 0x56CA, 0x5A18, 0xF928, /* 0x30 */
        0xF929, 0xF92A, 0xF92B, 0xF92C, 0x4E43, 0xF92D, 0x5167, 0x5948, /* 0x30 */
        0x67F0, 0x8010, 0xF92E, 0x5973, 0x5E74, 0x649A, 0x79CA, 0x5FF5, /* 0x40 */
        0x606C, 0x62C8, 0x637B, 0x5BE7, 0x5BD7, 0x52AA, 0xF92F, 0x5974, /* 0x40 */
        0x5F29, 0x6012, 0xF930, 0xF931, 0xF932, 0x7459, 0xF933, 0xF934, /* 0x50 */
        0xF935, 0xF936, 0xF937, 0xF938, 0x99D1, 0xF939, 0xF93A, 0xF93B, /* 0x50 */
        0xF93C, 0xF93D, 0xF93E, 0xF93F, 0xF940, 0xF941, 0xF942, 0xF943, /* 0x60 */
        0x6FC3, 0xF944, 0xF945, 0x81BF, 0x8FB2, 0x60F1, 0xF946, 0xF947, /* 0x60 */
        0x8166, 0xF948, 0xF949, 0x5C3F, 0xF94A, 0xF94B, 0xF94C, 0xF94D, /* 0x70 */
        0xF94E, 0xF94F, 0xF950, 0xF951, 0x5AE9, 0x8A25, 0x677B, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x7D10, 0xF952, 0xF953, 0xF954, 0xF955, 0xF956, 0xF957, /* 0x90 */
        0x80FD, 0xF958, 0xF959, 0x5C3C, 0x6CE5, 0x533F, 0x6EBA, 0x591A, /* 0x90 */
        0x8336, 0x4E39, 0x4EB6, 0x4F46, 0x55AE, 0x5718, 0x58C7, 0x5F56, /* 0xA0 */
        0x65B7, 0x65E6, 0x6A80, 0x6BB5, 0x6E4D, 0x77ED, 0x7AEF, 0x7C1E, /* 0xA0 */
        0x7DDE, 0x86CB, 0x8892, 0x9132, 0x935B, 0x64BB, 0x6FBE, 0x737A, /* 0xB0 */
        0x75B8, 0x9054, 0x5556, 0x574D, 0x61BA, 0x64D4, 0x66C7, 0x6DE1, /* 0xB0 */
        0x6E5B, 0x6F6D, 0x6FB9, 0x75F0, 0x8043, 0x81BD, 0x8541, 0x8983, /* 0xC0 */
        0x8AC7, 0x8B5A, 0x931F, 0x6C93, 0x7553, 0x7B54, 0x8E0F, 0x905D, /* 0xC0 */
        0x5510, 0x5802, 0x5858, 0x5E62, 0x6207, 0x649E, 0x68E0, 0x7576, /* 0xD0 */
        0x7CD6, 0x87B3, 0x9EE8, 0x4EE3, 0x5788, 0x576E, 0x5927, 0x5C0D, /* 0xD0 */
        0x5CB1, 0x5E36, 0x5F85, 0x6234, 0x64E1, 0x73B3, 0x81FA, 0x888B, /* 0xE0 */
        0x8CB8, 0x968A, 0x9EDB, 0x5B85, 0x5FB7, 0x60B3, 0x5012, 0x5200, /* 0xE0 */
        0x5230, 0x5716, 0x5835, 0x5857, 0x5C0E, 0x5C60, 0x5CF6, 0x5D8B, /* 0xF0 */
        0x5EA6, 0x5F92, 0x60BC, 0x6311, 0x6389, 0x6417, 0x6843			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe5:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x68F9, 0x6AC2, 0x6DD8, 0x6E21, 0x6ED4, 0x6FE4, 0x71FE, /* 0x30 */
        0x76DC, 0x7779, 0x79B1, 0x7A3B, 0x8404, 0x89A9, 0x8CED, 0x8DF3, /* 0x30 */
        0x8E48, 0x9003, 0x9014, 0x9053, 0x90FD, 0x934D, 0x9676, 0x97DC, /* 0x40 */
        0x6BD2, 0x7006, 0x7258, 0x72A2, 0x7368, 0x7763, 0x79BF, 0x7BE4, /* 0x40 */
        0x7E9B, 0x8B80, 0x58A9, 0x60C7, 0x6566, 0x65FD, 0x66BE, 0x6C8C, /* 0x50 */
        0x711E, 0x71C9, 0x8C5A, 0x9813, 0x4E6D, 0x7A81, 0x4EDD, 0x51AC, /* 0x50 */
        0x51CD, 0x52D5, 0x540C, 0x61A7, 0x6771, 0x6850, 0x68DF, 0x6D1E, /* 0x60 */
        0x6F7C, 0x75BC, 0x77B3, 0x7AE5, 0x80F4, 0x8463, 0x9285, 0x515C, /* 0x60 */
        0x6597, 0x675C, 0x6793, 0x75D8, 0x7AC7, 0x8373, 0xF95A, 0x8C46, /* 0x70 */
        0x9017, 0x982D, 0x5C6F, 0x81C0, 0x829A, 0x9041, 0x906F, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x920D, 0x5F97, 0x5D9D, 0x6A59, 0x71C8, 0x767B, 0x7B49, /* 0x90 */
        0x85E4, 0x8B04, 0x9127, 0x9A30, 0x5587, 0x61F6, 0xF95B, 0x7669, /* 0x90 */
        0x7F85, 0x863F, 0x87BA, 0x88F8, 0x908F, 0xF95C, 0x6D1B, 0x70D9, /* 0xA0 */
        0x73DE, 0x7D61, 0x843D, 0xF95D, 0x916A, 0x99F1, 0xF95E, 0x4E82, /* 0xA0 */
        0x5375, 0x6B04, 0x6B12, 0x703E, 0x721B, 0x862D, 0x9E1E, 0x524C, /* 0xB0 */
        0x8FA3, 0x5D50, 0x64E5, 0x652C, 0x6B16, 0x6FEB, 0x7C43, 0x7E9C, /* 0xB0 */
        0x85CD, 0x8964, 0x89BD, 0x62C9, 0x81D8, 0x881F, 0x5ECA, 0x6717, /* 0xC0 */
        0x6D6A, 0x72FC, 0x7405, 0x746F, 0x8782, 0x90DE, 0x4F86, 0x5D0D, /* 0xC0 */
        0x5FA0, 0x840A, 0x51B7, 0x63A0, 0x7565, 0x4EAE, 0x5006, 0x5169, /* 0xD0 */
        0x51C9, 0x6881, 0x6A11, 0x7CAE, 0x7CB1, 0x7CE7, 0x826F, 0x8AD2, /* 0xD0 */
        0x8F1B, 0x91CF, 0x4FB6, 0x5137, 0x52F5, 0x5442, 0x5EEC, 0x616E, /* 0xE0 */
        0x623E, 0x65C5, 0x6ADA, 0x6FFE, 0x792A, 0x85DC, 0x8823, 0x95AD, /* 0xE0 */
        0x9A62, 0x9A6A, 0x9E97, 0x9ECE, 0x529B, 0x66C6, 0x6B77, 0x701D, /* 0xF0 */
        0x792B, 0x8F62, 0x9742, 0x6190, 0x6200, 0x6523, 0x6F23			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe6:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x7149, 0x7489, 0x7DF4, 0x806F, 0x84EE, 0x8F26, 0x9023, /* 0x30 */
        0x934A, 0x51BD, 0x5217, 0x52A3, 0x6D0C, 0x70C8, 0x88C2, 0x5EC9, /* 0x30 */
        0x6582, 0x6BAE, 0x6FC2, 0x7C3E, 0x7375, 0x4EE4, 0x4F36, 0x56F9, /* 0x40 */
        0xF95F, 0x5CBA, 0x5DBA, 0x601C, 0x73B2, 0x7B2D, 0x7F9A, 0x7FCE, /* 0x40 */
        0x8046, 0x901E, 0x9234, 0x96F6, 0x9748, 0x9818, 0x9F61, 0x4F8B, /* 0x50 */
        0x6FA7, 0x79AE, 0x91B4, 0x96B7, 0x52DE, 0xF960, 0x6488, 0x64C4, /* 0x50 */
        0x6AD3, 0x6F5E, 0x7018, 0x7210, 0x76E7, 0x8001, 0x8606, 0x865C, /* 0x60 */
        0x8DEF, 0x8F05, 0x9732, 0x9B6F, 0x9DFA, 0x9E75, 0x788C, 0x797F, /* 0x60 */
        0x7DA0, 0x83C9, 0x9304, 0x9E7F, 0x9E93, 0x8AD6, 0x58DF, 0x5F04, /* 0x70 */
        0x6727, 0x7027, 0x74CF, 0x7C60, 0x807E, 0x5121, 0x7028, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x7262, 0x78CA, 0x8CC2, 0x8CDA, 0x8CF4, 0x96F7, 0x4E86, /* 0x90 */
        0x50DA, 0x5BEE, 0x5ED6, 0x6599, 0x71CE, 0x7642, 0x77AD, 0x804A, /* 0x90 */
        0x84FC, 0x907C, 0x9B27, 0x9F8D, 0x58D8, 0x5A41, 0x5C62, 0x6A13, /* 0xA0 */
        0x6DDA, 0x6F0F, 0x763B, 0x7D2F, 0x7E37, 0x851E, 0x8938, 0x93E4, /* 0xA0 */
        0x964B, 0x5289, 0x65D2, 0x67F3, 0x69B4, 0x6D41, 0x6E9C, 0x700F, /* 0xB0 */
        0x7409, 0x7460, 0x7559, 0x7624, 0x786B, 0x8B2C, 0x985E, 0x516D, /* 0xB0 */
        0x622E, 0x9678, 0x4F96, 0x502B, 0x5D19, 0x6DEA, 0x7DB8, 0x8F2A, /* 0xC0 */
        0x5F8B, 0x6144, 0x6817, 0xF961, 0x9686, 0x52D2, 0x808B, 0x51DC, /* 0xC0 */
        0x51CC, 0x695E, 0x7A1C, 0x7DBE, 0x83F1, 0x9675, 0x4FDA, 0x5229, /* 0xD0 */
        0x5398, 0x540F, 0x550E, 0x5C65, 0x60A7, 0x674E, 0x68A8, 0x6D6C, /* 0xD0 */
        0x7281, 0x72F8, 0x7406, 0x7483, 0xF962, 0x75E2, 0x7C6C, 0x7F79, /* 0xE0 */
        0x7FB8, 0x8389, 0x88CF, 0x88E1, 0x91CC, 0x91D0, 0x96E2, 0x9BC9, /* 0xE0 */
        0x541D, 0x6F7E, 0x71D0, 0x7498, 0x85FA, 0x8EAA, 0x96A3, 0x9C57, /* 0xF0 */
        0x9E9F, 0x6797, 0x6DCB, 0x7433, 0x81E8, 0x9716, 0x782C			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe7:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x7ACB, 0x7B20, 0x7C92, 0x6469, 0x746A, 0x75F2, 0x78BC, /* 0x30 */
        0x78E8, 0x99AC, 0x9B54, 0x9EBB, 0x5BDE, 0x5E55, 0x6F20, 0x819C, /* 0x30 */
        0x83AB, 0x9088, 0x4E07, 0x534D, 0x5A29, 0x5DD2, 0x5F4E, 0x6162, /* 0x40 */
        0x633D, 0x6669, 0x66FC, 0x6EFF, 0x6F2B, 0x7063, 0x779E, 0x842C, /* 0x40 */
        0x8513, 0x883B, 0x8F13, 0x9945, 0x9C3B, 0x551C, 0x62B9, 0x672B, /* 0x50 */
        0x6CAB, 0x8309, 0x896A, 0x977A, 0x4EA1, 0x5984, 0x5FD8, 0x5FD9, /* 0x50 */
        0x671B, 0x7DB2, 0x7F54, 0x8292, 0x832B, 0x83BD, 0x8F1E, 0x9099, /* 0x60 */
        0x57CB, 0x59B9, 0x5A92, 0x5BD0, 0x6627, 0x679A, 0x6885, 0x6BCF, /* 0x60 */
        0x7164, 0x7F75, 0x8CB7, 0x8CE3, 0x9081, 0x9B45, 0x8108, 0x8C8A, /* 0x70 */
        0x964C, 0x9A40, 0x9EA5, 0x5B5F, 0x6C13, 0x731B, 0x76F2, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x76DF, 0x840C, 0x51AA, 0x8993, 0x514D, 0x5195, 0x52C9, /* 0x90 */
        0x68C9, 0x6C94, 0x7704, 0x7720, 0x7DBF, 0x7DEC, 0x9762, 0x9EB5, /* 0x90 */
        0x6EC5, 0x8511, 0x51A5, 0x540D, 0x547D, 0x660E, 0x669D, 0x6927, /* 0xA0 */
        0x6E9F, 0x76BF, 0x7791, 0x8317, 0x84C2, 0x879F, 0x9169, 0x9298, /* 0xA0 */
        0x9CF4, 0x8882, 0x4FAE, 0x5192, 0x52DF, 0x59C6, 0x5E3D, 0x6155, /* 0xB0 */
        0x6478, 0x6479, 0x66AE, 0x67D0, 0x6A21, 0x6BCD, 0x6BDB, 0x725F, /* 0xB0 */
        0x7261, 0x7441, 0x7738, 0x77DB, 0x8017, 0x82BC, 0x8305, 0x8B00, /* 0xC0 */
        0x8B28, 0x8C8C, 0x6728, 0x6C90, 0x7267, 0x76EE, 0x7766, 0x7A46, /* 0xC0 */
        0x9DA9, 0x6B7F, 0x6C92, 0x5922, 0x6726, 0x8499, 0x536F, 0x5893, /* 0xD0 */
        0x5999, 0x5EDF, 0x63CF, 0x6634, 0x6773, 0x6E3A, 0x732B, 0x7AD7, /* 0xD0 */
        0x82D7, 0x9328, 0x52D9, 0x5DEB, 0x61AE, 0x61CB, 0x620A, 0x62C7, /* 0xE0 */
        0x64AB, 0x65E0, 0x6959, 0x6B66, 0x6BCB, 0x7121, 0x73F7, 0x755D, /* 0xE0 */
        0x7E46, 0x821E, 0x8302, 0x856A, 0x8AA3, 0x8CBF, 0x9727, 0x9D61, /* 0xF0 */
        0x58A8, 0x9ED8, 0x5011, 0x520E, 0x543B, 0x554F, 0x6587			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe8:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x6C76, 0x7D0A, 0x7D0B, 0x805E, 0x868A, 0x9580, 0x96EF, /* 0x30 */
        0x52FF, 0x6C95, 0x7269, 0x5473, 0x5A9A, 0x5C3E, 0x5D4B, 0x5F4C, /* 0x30 */
        0x5FAE, 0x672A, 0x68B6, 0x6963, 0x6E3C, 0x6E44, 0x7709, 0x7C73, /* 0x40 */
        0x7F8E, 0x8587, 0x8B0E, 0x8FF7, 0x9761, 0x9EF4, 0x5CB7, 0x60B6, /* 0x40 */
        0x610D, 0x61AB, 0x654F, 0x65FB, 0x65FC, 0x6C11, 0x6CEF, 0x739F, /* 0x50 */
        0x73C9, 0x7DE1, 0x9594, 0x5BC6, 0x871C, 0x8B10, 0x525D, 0x535A, /* 0x50 */
        0x62CD, 0x640F, 0x64B2, 0x6734, 0x6A38, 0x6CCA, 0x73C0, 0x749E, /* 0x60 */
        0x7B94, 0x7C95, 0x7E1B, 0x818A, 0x8236, 0x8584, 0x8FEB, 0x96F9, /* 0x60 */
        0x99C1, 0x4F34, 0x534A, 0x53CD, 0x53DB, 0x62CC, 0x642C, 0x6500, /* 0x70 */
        0x6591, 0x69C3, 0x6CEE, 0x6F58, 0x73ED, 0x7554, 0x7622, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x76E4, 0x76FC, 0x78D0, 0x78FB, 0x792C, 0x7D46, 0x822C, /* 0x90 */
        0x87E0, 0x8FD4, 0x9812, 0x98EF, 0x52C3, 0x62D4, 0x64A5, 0x6E24, /* 0x90 */
        0x6F51, 0x767C, 0x8DCB, 0x91B1, 0x9262, 0x9AEE, 0x9B43, 0x5023, /* 0xA0 */
        0x508D, 0x574A, 0x59A8, 0x5C28, 0x5E47, 0x5F77, 0x623F, 0x653E, /* 0xA0 */
        0x65B9, 0x65C1, 0x6609, 0x678B, 0x699C, 0x6EC2, 0x78C5, 0x7D21, /* 0xB0 */
        0x80AA, 0x8180, 0x822B, 0x82B3, 0x84A1, 0x868C, 0x8A2A, 0x8B17, /* 0xB0 */
        0x90A6, 0x9632, 0x9F90, 0x500D, 0x4FF3, 0xF963, 0x57F9, 0x5F98, /* 0xC0 */
        0x62DC, 0x6392, 0x676F, 0x6E43, 0x7119, 0x76C3, 0x80CC, 0x80DA, /* 0xC0 */
        0x88F4, 0x88F5, 0x8919, 0x8CE0, 0x8F29, 0x914D, 0x966A, 0x4F2F, /* 0xD0 */
        0x4F70, 0x5E1B, 0x67CF, 0x6822, 0x767D, 0x767E, 0x9B44, 0x5E61, /* 0xD0 */
        0x6A0A, 0x7169, 0x71D4, 0x756A, 0xF964, 0x7E41, 0x8543, 0x85E9, /* 0xE0 */
        0x98DC, 0x4F10, 0x7B4F, 0x7F70, 0x95A5, 0x51E1, 0x5E06, 0x68B5, /* 0xE0 */
        0x6C3E, 0x6C4E, 0x6CDB, 0x72AF, 0x7BC4, 0x8303, 0x6CD5, 0x743A, /* 0xF0 */
        0x50FB, 0x5288, 0x58C1, 0x64D8, 0x6A97, 0x74A7, 0x7656			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xe9:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x78A7, 0x8617, 0x95E2, 0x9739, 0xF965, 0x535E, 0x5F01, /* 0x30 */
        0x8B8A, 0x8FA8, 0x8FAF, 0x908A, 0x5225, 0x77A5, 0x9C49, 0x9F08, /* 0x30 */
        0x4E19, 0x5002, 0x5175, 0x5C5B, 0x5E77, 0x661E, 0x663A, 0x67C4, /* 0x40 */
        0x68C5, 0x70B3, 0x7501, 0x75C5, 0x79C9, 0x7ADD, 0x8F27, 0x9920, /* 0x40 */
        0x9A08, 0x4FDD, 0x5821, 0x5831, 0x5BF6, 0x666E, 0x6B65, 0x6D11, /* 0x50 */
        0x6E7A, 0x6F7D, 0x73E4, 0x752B, 0x83E9, 0x88DC, 0x8913, 0x8B5C, /* 0x50 */
        0x8F14, 0x4F0F, 0x50D5, 0x5310, 0x535C, 0x5B93, 0x5FA9, 0x670D, /* 0x60 */
        0x798F, 0x8179, 0x832F, 0x8514, 0x8907, 0x8986, 0x8F39, 0x8F3B, /* 0x60 */
        0x99A5, 0x9C12, 0x672C, 0x4E76, 0x4FF8, 0x5949, 0x5C01, 0x5CEF, /* 0x70 */
        0x5CF0, 0x6367, 0x68D2, 0x70FD, 0x71A2, 0x742B, 0x7E2B, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x84EC, 0x8702, 0x9022, 0x92D2, 0x9CF3, 0x4E0D, 0x4ED8, /* 0x90 */
        0x4FEF, 0x5085, 0x5256, 0x526F, 0x5426, 0x5490, 0x57E0, 0x592B, /* 0x90 */
        0x5A66, 0x5B5A, 0x5B75, 0x5BCC, 0x5E9C, 0xF966, 0x6276, 0x6577, /* 0xA0 */
        0x65A7, 0x6D6E, 0x6EA5, 0x7236, 0x7B26, 0x7C3F, 0x7F36, 0x8150, /* 0xA0 */
        0x8151, 0x819A, 0x8240, 0x8299, 0x83A9, 0x8A03, 0x8CA0, 0x8CE6, /* 0xB0 */
        0x8CFB, 0x8D74, 0x8DBA, 0x90E8, 0x91DC, 0x961C, 0x9644, 0x99D9, /* 0xB0 */
        0x9CE7, 0x5317, 0x5206, 0x5429, 0x5674, 0x58B3, 0x5954, 0x596E, /* 0xC0 */
        0x5FFF, 0x61A4, 0x626E, 0x6610, 0x6C7E, 0x711A, 0x76C6, 0x7C89, /* 0xC0 */
        0x7CDE, 0x7D1B, 0x82AC, 0x8CC1, 0x96F0, 0xF967, 0x4F5B, 0x5F17, /* 0xD0 */
        0x5F7F, 0x62C2, 0x5D29, 0x670B, 0x68DA, 0x787C, 0x7E43, 0x9D6C, /* 0xD0 */
        0x4E15, 0x5099, 0x5315, 0x532A, 0x5351, 0x5983, 0x5A62, 0x5E87, /* 0xE0 */
        0x60B2, 0x618A, 0x6249, 0x6279, 0x6590, 0x6787, 0x69A7, 0x6BD4, /* 0xE0 */
        0x6BD6, 0x6BD7, 0x6BD8, 0x6CB8, 0xF968, 0x7435, 0x75FA, 0x7812, /* 0xF0 */
        0x7891, 0x79D5, 0x79D8, 0x7C83, 0x7DCB, 0x7FE1, 0x80A5			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xea:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x813E, 0x81C2, 0x83F2, 0x871A, 0x88E8, 0x8AB9, 0x8B6C, /* 0x30 */
        0x8CBB, 0x9119, 0x975E, 0x98DB, 0x9F3B, 0x56AC, 0x5B2A, 0x5F6C, /* 0x30 */
        0x658C, 0x6AB3, 0x6BAF, 0x6D5C, 0x6FF1, 0x7015, 0x725D, 0x73AD, /* 0x40 */
        0x8CA7, 0x8CD3, 0x983B, 0x6191, 0x6C37, 0x8058, 0x9A01, 0x4E4D, /* 0x40 */
        0x4E8B, 0x4E9B, 0x4ED5, 0x4F3A, 0x4F3C, 0x4F7F, 0x4FDF, 0x50FF, /* 0x50 */
        0x53F2, 0x53F8, 0x5506, 0x55E3, 0x56DB, 0x58EB, 0x5962, 0x5A11, /* 0x50 */
        0x5BEB, 0x5BFA, 0x5C04, 0x5DF3, 0x5E2B, 0x5F99, 0x601D, 0x6368, /* 0x60 */
        0x659C, 0x65AF, 0x67F6, 0x67FB, 0x68AD, 0x6B7B, 0x6C99, 0x6CD7, /* 0x60 */
        0x6E23, 0x7009, 0x7345, 0x7802, 0x793E, 0x7940, 0x7960, 0x79C1, /* 0x70 */
        0x7BE9, 0x7D17, 0x7D72, 0x8086, 0x820D, 0x838E, 0x84D1, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x86C7, 0x88DF, 0x8A50, 0x8A5E, 0x8B1D, 0x8CDC, 0x8D66, /* 0x90 */
        0x8FAD, 0x90AA, 0x98FC, 0x99DF, 0x9E9D, 0x524A, 0xF969, 0x6714, /* 0x90 */
        0xF96A, 0x5098, 0x522A, 0x5C71, 0x6563, 0x6C55, 0x73CA, 0x7523, /* 0xA0 */
        0x759D, 0x7B97, 0x849C, 0x9178, 0x9730, 0x4E77, 0x6492, 0x6BBA, /* 0xA0 */
        0x715E, 0x85A9, 0x4E09, 0xF96B, 0x6749, 0x68EE, 0x6E17, 0x829F, /* 0xB0 */
        0x8518, 0x886B, 0x63F7, 0x6F81, 0x9212, 0x98AF, 0x4E0A, 0x50B7, /* 0xB0 */
        0x50CF, 0x511F, 0x5546, 0x55AA, 0x5617, 0x5B40, 0x5C19, 0x5CE0, /* 0xC0 */
        0x5E38, 0x5E8A, 0x5EA0, 0x5EC2, 0x60F3, 0x6851, 0x6A61, 0x6E58, /* 0xC0 */
        0x723D, 0x7240, 0x72C0, 0x76F8, 0x7965, 0x7BB1, 0x7FD4, 0x88F3, /* 0xD0 */
        0x89F4, 0x8A73, 0x8C61, 0x8CDE, 0x971C, 0x585E, 0x74BD, 0x8CFD, /* 0xD0 */
        0x55C7, 0xF96C, 0x7A61, 0x7D22, 0x8272, 0x7272, 0x751F, 0x7525, /* 0xE0 */
        0xF96D, 0x7B19, 0x5885, 0x58FB, 0x5DBC, 0x5E8F, 0x5EB6, 0x5F90, /* 0xE0 */
        0x6055, 0x6292, 0x637F, 0x654D, 0x6691, 0x66D9, 0x66F8, 0x6816, /* 0xF0 */
        0x68F2, 0x7280, 0x745E, 0x7B6E, 0x7D6E, 0x7DD6, 0x7F72			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xeb:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x80E5, 0x8212, 0x85AF, 0x897F, 0x8A93, 0x901D, 0x92E4, /* 0x30 */
        0x9ECD, 0x9F20, 0x5915, 0x596D, 0x5E2D, 0x60DC, 0x6614, 0x6673, /* 0x30 */
        0x6790, 0x6C50, 0x6DC5, 0x6F5F, 0x77F3, 0x78A9, 0x84C6, 0x91CB, /* 0x40 */
        0x932B, 0x4ED9, 0x50CA, 0x5148, 0x5584, 0x5B0B, 0x5BA3, 0x6247, /* 0x40 */
        0x657E, 0x65CB, 0x6E32, 0x717D, 0x7401, 0x7444, 0x7487, 0x74BF, /* 0x50 */
        0x766C, 0x79AA, 0x7DDA, 0x7E55, 0x7FA8, 0x817A, 0x81B3, 0x8239, /* 0x50 */
        0x861A, 0x87EC, 0x8A75, 0x8DE3, 0x9078, 0x9291, 0x9425, 0x994D, /* 0x60 */
        0x9BAE, 0x5368, 0x5C51, 0x6954, 0x6CC4, 0x6D29, 0x6E2B, 0x820C, /* 0x60 */
        0x859B, 0x893B, 0x8A2D, 0x8AAA, 0x96EA, 0x9F67, 0x5261, 0x66B9, /* 0x70 */
        0x6BB2, 0x7E96, 0x87FE, 0x8D0D, 0x9583, 0x965D, 0x651D, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x6D89, 0x71EE, 0xF96E, 0x57CE, 0x59D3, 0x5BAC, 0x6027, /* 0x90 */
        0x60FA, 0x6210, 0x661F, 0x665F, 0x7329, 0x73F9, 0x76DB, 0x7701, /* 0x90 */
        0x7B6C, 0x8056, 0x8072, 0x8165, 0x8AA0, 0x9192, 0x4E16, 0x52E2, /* 0xA0 */
        0x6B72, 0x6D17, 0x7A05, 0x7B39, 0x7D30, 0xF96F, 0x8CB0, 0x53EC, /* 0xA0 */
        0x562F, 0x5851, 0x5BB5, 0x5C0F, 0x5C11, 0x5DE2, 0x6240, 0x6383, /* 0xB0 */
        0x6414, 0x662D, 0x68B3, 0x6CBC, 0x6D88, 0x6EAF, 0x701F, 0x70A4, /* 0xB0 */
        0x71D2, 0x7526, 0x758F, 0x758E, 0x7619, 0x7B11, 0x7BE0, 0x7C2B, /* 0xC0 */
        0x7D20, 0x7D39, 0x852C, 0x856D, 0x8607, 0x8A34, 0x900D, 0x9061, /* 0xC0 */
        0x90B5, 0x92B7, 0x97F6, 0x9A37, 0x4FD7, 0x5C6C, 0x675F, 0x6D91, /* 0xD0 */
        0x7C9F, 0x7E8C, 0x8B16, 0x8D16, 0x901F, 0x5B6B, 0x5DFD, 0x640D, /* 0xD0 */
        0x84C0, 0x905C, 0x98E1, 0x7387, 0x5B8B, 0x609A, 0x677E, 0x6DDE, /* 0xE0 */
        0x8A1F, 0x8AA6, 0x9001, 0x980C, 0x5237, 0xF970, 0x7051, 0x788E, /* 0xE0 */
        0x9396, 0x8870, 0x91D7, 0x4FEE, 0x53D7, 0x55FD, 0x56DA, 0x5782, /* 0xF0 */
        0x58FD, 0x5AC2, 0x5B88, 0x5CAB, 0x5CC0, 0x5E25, 0x6101			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xec:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x620D, 0x624B, 0x6388, 0x641C, 0x6536, 0x6578, 0x6A39, /* 0x30 */
        0x6B8A, 0x6C34, 0x6D19, 0x6F31, 0x71E7, 0x72E9, 0x7378, 0x7407, /* 0x30 */
        0x74B2, 0x7626, 0x7761, 0x79C0, 0x7A57, 0x7AEA, 0x7CB9, 0x7D8F, /* 0x40 */
        0x7DAC, 0x7E61, 0x7F9E, 0x8129, 0x8331, 0x8490, 0x84DA, 0x85EA, /* 0x40 */
        0x8896, 0x8AB0, 0x8B90, 0x8F38, 0x9042, 0x9083, 0x916C, 0x9296, /* 0x50 */
        0x92B9, 0x968B, 0x96A7, 0x96A8, 0x96D6, 0x9700, 0x9808, 0x9996, /* 0x50 */
        0x9AD3, 0x9B1A, 0x53D4, 0x587E, 0x5919, 0x5B70, 0x5BBF, 0x6DD1, /* 0x60 */
        0x6F5A, 0x719F, 0x7421, 0x74B9, 0x8085, 0x83FD, 0x5DE1, 0x5F87, /* 0x60 */
        0x5FAA, 0x6042, 0x65EC, 0x6812, 0x696F, 0x6A53, 0x6B89, 0x6D35, /* 0x70 */
        0x6DF3, 0x73E3, 0x76FE, 0x77AC, 0x7B4D, 0x7D14, 0x8123, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x821C, 0x8340, 0x84F4, 0x8563, 0x8A62, 0x8AC4, 0x9187, /* 0x90 */
        0x931E, 0x9806, 0x99B4, 0x620C, 0x8853, 0x8FF0, 0x9265, 0x5D07, /* 0x90 */
        0x5D27, 0x5D69, 0x745F, 0x819D, 0x8768, 0x6FD5, 0x62FE, 0x7FD2, /* 0xA0 */
        0x8936, 0x8972, 0x4E1E, 0x4E58, 0x50E7, 0x52DD, 0x5347, 0x627F, /* 0xA0 */
        0x6607, 0x7E69, 0x8805, 0x965E, 0x4F8D, 0x5319, 0x5636, 0x59CB, /* 0xB0 */
        0x5AA4, 0x5C38, 0x5C4E, 0x5C4D, 0x5E02, 0x5F11, 0x6043, 0x65BD, /* 0xB0 */
        0x662F, 0x6642, 0x67BE, 0x67F4, 0x731C, 0x77E2, 0x793A, 0x7FC5, /* 0xC0 */
        0x8494, 0x84CD, 0x8996, 0x8A66, 0x8A69, 0x8AE1, 0x8C55, 0x8C7A, /* 0xC0 */
        0x57F4, 0x5BD4, 0x5F0F, 0x606F, 0x62ED, 0x690D, 0x6B96, 0x6E5C, /* 0xD0 */
        0x7184, 0x7BD2, 0x8755, 0x8B58, 0x8EFE, 0x98DF, 0x98FE, 0x4F38, /* 0xD0 */
        0x4F81, 0x4FE1, 0x547B, 0x5A20, 0x5BB8, 0x613C, 0x65B0, 0x6668, /* 0xE0 */
        0x71FC, 0x7533, 0x795E, 0x7D33, 0x814E, 0x81E3, 0x8398, 0x85AA, /* 0xE0 */
        0x85CE, 0x8703, 0x8A0A, 0x8EAB, 0x8F9B, 0xF971, 0x8FC5, 0x5931, /* 0xF0 */
        0x5BA4, 0x5BE6, 0x6089, 0x5BE9, 0x5C0B, 0x5FC3, 0x6C81			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xed:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0xF972, 0x6DF1, 0x700B, 0x751A, 0x82AF, 0x8AF6, 0x4EC0, /* 0x30 */
        0x5341, 0xF973, 0x96D9, 0x6C0F, 0x4E9E, 0x4FC4, 0x5152, 0x555E, /* 0x30 */
        0x5A25, 0x5CE8, 0x6211, 0x7259, 0x82BD, 0x83AA, 0x86FE, 0x8859, /* 0x40 */
        0x8A1D, 0x963F, 0x96C5, 0x9913, 0x9D09, 0x9D5D, 0x580A, 0x5CB3, /* 0x40 */
        0x5DBD, 0x5E44, 0x60E1, 0x6115, 0x63E1, 0x6A02, 0x6E25, 0x9102, /* 0x50 */
        0x9354, 0x984E, 0x9C10, 0x9F77, 0x5B89, 0x5CB8, 0x6309, 0x664F, /* 0x50 */
        0x6848, 0x773C, 0x96C1, 0x978D, 0x9854, 0x9B9F, 0x65A1, 0x8B01, /* 0x60 */
        0x8ECB, 0x95BC, 0x5535, 0x5CA9, 0x5DD6, 0x5EB5, 0x6697, 0x764C, /* 0x60 */
        0x83F4, 0x95C7, 0x58D3, 0x62BC, 0x72CE, 0x9D28, 0x4EF0, 0x592E, /* 0x70 */
        0x600F, 0x663B, 0x6B83, 0x79E7, 0x9D26, 0x5393, 0x54C0, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x57C3, 0x5D16, 0x611B, 0x66D6, 0x6DAF, 0x788D, 0x827E, /* 0x90 */
        0x9698, 0x9744, 0x5384, 0x627C, 0x6396, 0x6DB2, 0x7E0A, 0x814B, /* 0x90 */
        0x984D, 0x6AFB, 0x7F4C, 0x9DAF, 0x9E1A, 0x4E5F, 0x503B, 0x51B6, /* 0xA0 */
        0x591C, 0x60F9, 0x63F6, 0x6930, 0x723A, 0x8036, 0xF974, 0x91CE, /* 0xA0 */
        0x5F31, 0xF975, 0xF976, 0x7D04, 0x82E5, 0x846F, 0x84BB, 0x85E5, /* 0xB0 */
        0x8E8D, 0xF977, 0x4F6F, 0xF978, 0xF979, 0x58E4, 0x5B43, 0x6059, /* 0xB0 */
        0x63DA, 0x6518, 0x656D, 0x6698, 0xF97A, 0x694A, 0x6A23, 0x6D0B, /* 0xC0 */
        0x7001, 0x716C, 0x75D2, 0x760D, 0x79B3, 0x7A70, 0xF97B, 0x7F8A, /* 0xC0 */
        0xF97C, 0x8944, 0xF97D, 0x8B93, 0x91C0, 0x967D, 0xF97E, 0x990A, /* 0xD0 */
        0x5704, 0x5FA1, 0x65BC, 0x6F01, 0x7600, 0x79A6, 0x8A9E, 0x99AD, /* 0xD0 */
        0x9B5A, 0x9F6C, 0x5104, 0x61B6, 0x6291, 0x6A8D, 0x81C6, 0x5043, /* 0xE0 */
        0x5830, 0x5F66, 0x7109, 0x8A00, 0x8AFA, 0x5B7C, 0x8616, 0x4FFA, /* 0xE0 */
        0x513C, 0x56B4, 0x5944, 0x63A9, 0x6DF9, 0x5DAA, 0x696D, 0x5186, /* 0xF0 */
        0x4E88, 0x4F59, 0xF97F, 0xF980, 0xF981, 0x5982, 0xF982			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xee:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0xF983, 0x6B5F, 0x6C5D, 0xF984, 0x74B5, 0x7916, 0xF985, /* 0x30 */
        0x8207, 0x8245, 0x8339, 0x8F3F, 0x8F5D, 0xF986, 0x9918, 0xF987, /* 0x30 */
        0xF988, 0xF989, 0x4EA6, 0xF98A, 0x57DF, 0x5F79, 0x6613, 0xF98B, /* 0x40 */
        0xF98C, 0x75AB, 0x7E79, 0x8B6F, 0xF98D, 0x9006, 0x9A5B, 0x56A5, /* 0x40 */
        0x5827, 0x59F8, 0x5A1F, 0x5BB4, 0xF98E, 0x5EF6, 0xF98F, 0xF990, /* 0x50 */
        0x6350, 0x633B, 0xF991, 0x693D, 0x6C87, 0x6CBF, 0x6D8E, 0x6D93, /* 0x50 */
        0x6DF5, 0x6F14, 0xF992, 0x70DF, 0x7136, 0x7159, 0xF993, 0x71C3, /* 0x60 */
        0x71D5, 0xF994, 0x784F, 0x786F, 0xF995, 0x7B75, 0x7DE3, 0xF996, /* 0x60 */
        0x7E2F, 0xF997, 0x884D, 0x8EDF, 0xF998, 0xF999, 0xF99A, 0x925B, /* 0x70 */
        0xF99B, 0x9CF6, 0xF99C, 0xF99D, 0xF99E, 0x6085, 0x6D85, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0xF99F, 0x71B1, 0xF9A0, 0xF9A1, 0x95B1, 0x53AD, 0xF9A2, /* 0x90 */
        0xF9A3, 0xF9A4, 0x67D3, 0xF9A5, 0x708E, 0x7130, 0x7430, 0x8276, /* 0x90 */
        0x82D2, 0xF9A6, 0x95BB, 0x9AE5, 0x9E7D, 0x66C4, 0xF9A7, 0x71C1, /* 0xA0 */
        0x8449, 0xF9A8, 0xF9A9, 0x584B, 0xF9AA, 0xF9AB, 0x5DB8, 0x5F71, /* 0xA0 */
        0xF9AC, 0x6620, 0x668E, 0x6979, 0x69AE, 0x6C38, 0x6CF3, 0x6E36, /* 0xB0 */
        0x6F41, 0x6FDA, 0x701B, 0x702F, 0x7150, 0x71DF, 0x7370, 0xF9AD, /* 0xB0 */
        0x745B, 0xF9AE, 0x74D4, 0x76C8, 0x7A4E, 0x7E93, 0xF9AF, 0xF9B0, /* 0xC0 */
        0x82F1, 0x8A60, 0x8FCE, 0xF9B1, 0x9348, 0xF9B2, 0x9719, 0xF9B3, /* 0xC0 */
        0xF9B4, 0x4E42, 0x502A, 0xF9B5, 0x5208, 0x53E1, 0x66F3, 0x6C6D, /* 0xD0 */
        0x6FCA, 0x730A, 0x777F, 0x7A62, 0x82AE, 0x85DD, 0x8602, 0xF9B6, /* 0xD0 */
        0x88D4, 0x8A63, 0x8B7D, 0x8C6B, 0xF9B7, 0x92B3, 0xF9B8, 0x9713, /* 0xE0 */
        0x9810, 0x4E94, 0x4F0D, 0x4FC9, 0x50B2, 0x5348, 0x543E, 0x5433, /* 0xE0 */
        0x55DA, 0x5862, 0x58BA, 0x5967, 0x5A1B, 0x5BE4, 0x609F, 0xF9B9, /* 0xF0 */
        0x61CA, 0x6556, 0x65FF, 0x6664, 0x68A7, 0x6C5A, 0x6FB3			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xef:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x70CF, 0x71AC, 0x7352, 0x7B7D, 0x8708, 0x8AA4, 0x9C32, /* 0x30 */
        0x9F07, 0x5C4B, 0x6C83, 0x7344, 0x7389, 0x923A, 0x6EAB, 0x7465, /* 0x30 */
        0x761F, 0x7A69, 0x7E15, 0x860A, 0x5140, 0x58C5, 0x64C1, 0x74EE, /* 0x40 */
        0x7515, 0x7670, 0x7FC1, 0x9095, 0x96CD, 0x9954, 0x6E26, 0x74E6, /* 0x40 */
        0x7AA9, 0x7AAA, 0x81E5, 0x86D9, 0x8778, 0x8A1B, 0x5A49, 0x5B8C, /* 0x50 */
        0x5B9B, 0x68A1, 0x6900, 0x6D63, 0x73A9, 0x7413, 0x742C, 0x7897, /* 0x50 */
        0x7DE9, 0x7FEB, 0x8118, 0x8155, 0x839E, 0x8C4C, 0x962E, 0x9811, /* 0x60 */
        0x66F0, 0x5F80, 0x65FA, 0x6789, 0x6C6A, 0x738B, 0x502D, 0x5A03, /* 0x60 */
        0x6B6A, 0x77EE, 0x5916, 0x5D6C, 0x5DCD, 0x7325, 0x754F, 0xF9BA, /* 0x70 */
        0xF9BB, 0x50E5, 0x51F9, 0x582F, 0x592D, 0x5996, 0x59DA, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x5BE5, 0xF9BC, 0xF9BD, 0x5DA2, 0x62D7, 0x6416, 0x6493, /* 0x90 */
        0x64FE, 0xF9BE, 0x66DC, 0xF9BF, 0x6A48, 0xF9C0, 0x71FF, 0x7464, /* 0x90 */
        0xF9C1, 0x7A88, 0x7AAF, 0x7E47, 0x7E5E, 0x8000, 0x8170, 0xF9C2, /* 0xA0 */
        0x87EF, 0x8981, 0x8B20, 0x9059, 0xF9C3, 0x9080, 0x9952, 0x617E, /* 0xA0 */
        0x6B32, 0x6D74, 0x7E1F, 0x8925, 0x8FB1, 0x4FD1, 0x50AD, 0x5197, /* 0xB0 */
        0x52C7, 0x57C7, 0x5889, 0x5BB9, 0x5EB8, 0x6142, 0x6995, 0x6D8C, /* 0xB0 */
        0x6E67, 0x6EB6, 0x7194, 0x7462, 0x7528, 0x752C, 0x8073, 0x8338, /* 0xC0 */
        0x84C9, 0x8E0A, 0x9394, 0x93DE, 0xF9C4, 0x4E8E, 0x4F51, 0x5076, /* 0xC0 */
        0x512A, 0x53C8, 0x53CB, 0x53F3, 0x5B87, 0x5BD3, 0x5C24, 0x611A, /* 0xD0 */
        0x6182, 0x65F4, 0x725B, 0x7397, 0x7440, 0x76C2, 0x7950, 0x7991, /* 0xD0 */
        0x79B9, 0x7D06, 0x7FBD, 0x828B, 0x85D5, 0x865E, 0x8FC2, 0x9047, /* 0xE0 */
        0x90F5, 0x91EA, 0x9685, 0x96E8, 0x96E9, 0x52D6, 0x5F67, 0x65ED, /* 0xE0 */
        0x6631, 0x682F, 0x715C, 0x7A36, 0x90C1, 0x980A, 0x4E91, 0xF9C5, /* 0xF0 */
        0x6A52, 0x6B9E, 0x6F90, 0x7189, 0x8018, 0x82B8, 0x8553			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf0:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x904B, 0x9695, 0x96F2, 0x97FB, 0x851A, 0x9B31, 0x4E90, /* 0x30 */
        0x718A, 0x96C4, 0x5143, 0x539F, 0x54E1, 0x5713, 0x5712, 0x57A3, /* 0x30 */
        0x5A9B, 0x5AC4, 0x5BC3, 0x6028, 0x613F, 0x63F4, 0x6C85, 0x6D39, /* 0x40 */
        0x6E72, 0x6E90, 0x7230, 0x733F, 0x7457, 0x82D1, 0x8881, 0x8F45, /* 0x40 */
        0x9060, 0xF9C6, 0x9662, 0x9858, 0x9D1B, 0x6708, 0x8D8A, 0x925E, /* 0x50 */
        0x4F4D, 0x5049, 0x50DE, 0x5371, 0x570D, 0x59D4, 0x5A01, 0x5C09, /* 0x50 */
        0x6170, 0x6690, 0x6E2D, 0x7232, 0x744B, 0x7DEF, 0x80C3, 0x840E, /* 0x60 */
        0x8466, 0x853F, 0x875F, 0x885B, 0x8918, 0x8B02, 0x9055, 0x97CB, /* 0x60 */
        0x9B4F, 0x4E73, 0x4F91, 0x5112, 0x516A, 0xF9C7, 0x552F, 0x55A9, /* 0x70 */
        0x5B7A, 0x5BA5, 0x5E7C, 0x5E7D, 0x5EBE, 0x60A0, 0x60DF, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x6108, 0x6109, 0x63C4, 0x6538, 0x6709, 0xF9C8, 0x67D4, /* 0x90 */
        0x67DA, 0xF9C9, 0x6961, 0x6962, 0x6CB9, 0x6D27, 0xF9CA, 0x6E38, /* 0x90 */
        0xF9CB, 0x6FE1, 0x7336, 0x7337, 0xF9CC, 0x745C, 0x7531, 0xF9CD, /* 0xA0 */
        0x7652, 0xF9CE, 0xF9CF, 0x7DAD, 0x81FE, 0x8438, 0x88D5, 0x8A98, /* 0xA0 */
        0x8ADB, 0x8AED, 0x8E30, 0x8E42, 0x904A, 0x903E, 0x907A, 0x9149, /* 0xB0 */
        0x91C9, 0x936E, 0xF9D0, 0xF9D1, 0x5809, 0xF9D2, 0x6BD3, 0x8089, /* 0xB0 */
        0x80B2, 0xF9D3, 0xF9D4, 0x5141, 0x596B, 0x5C39, 0xF9D5, 0xF9D6, /* 0xC0 */
        0x6F64, 0x73A7, 0x80E4, 0x8D07, 0xF9D7, 0x9217, 0x958F, 0xF9D8, /* 0xC0 */
        0xF9D9, 0xF9DA, 0xF9DB, 0x807F, 0x620E, 0x701C, 0x7D68, 0x878D, /* 0xD0 */
        0xF9DC, 0x57A0, 0x6069, 0x6147, 0x6BB7, 0x8ABE, 0x9280, 0x96B1, /* 0xD0 */
        0x4E59, 0x541F, 0x6DEB, 0x852D, 0x9670, 0x97F3, 0x98EE, 0x63D6, /* 0xE0 */
        0x6CE3, 0x9091, 0x51DD, 0x61C9, 0x81BA, 0x9DF9, 0x4F9D, 0x501A, /* 0xE0 */
        0x5100, 0x5B9C, 0x610F, 0x61FF, 0x64EC, 0x6905, 0x6BC5, 0x7591, /* 0xF0 */
        0x77E3, 0x7FA9, 0x8264, 0x858F, 0x87FB, 0x8863, 0x8ABC			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf1:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x8B70, 0x91AB, 0x4E8C, 0x4EE5, 0x4F0A, 0xF9DD, 0xF9DE, /* 0x30 */
        0x5937, 0x59E8, 0xF9DF, 0x5DF2, 0x5F1B, 0x5F5B, 0x6021, 0xF9E0, /* 0x30 */
        0xF9E1, 0xF9E2, 0xF9E3, 0x723E, 0x73E5, 0xF9E4, 0x7570, 0x75CD, /* 0x40 */
        0xF9E5, 0x79FB, 0xF9E6, 0x800C, 0x8033, 0x8084, 0x82E1, 0x8351, /* 0x40 */
        0xF9E7, 0xF9E8, 0x8CBD, 0x8CB3, 0x9087, 0xF9E9, 0xF9EA, 0x98F4, /* 0x50 */
        0x990C, 0xF9EB, 0xF9EC, 0x7037, 0x76CA, 0x7FCA, 0x7FCC, 0x7FFC, /* 0x50 */
        0x8B1A, 0x4EBA, 0x4EC1, 0x5203, 0x5370, 0xF9ED, 0x54BD, 0x56E0, /* 0x60 */
        0x59FB, 0x5BC5, 0x5F15, 0x5FCD, 0x6E6E, 0xF9EE, 0xF9EF, 0x7D6A, /* 0x60 */
        0x8335, 0xF9F0, 0x8693, 0x8A8D, 0xF9F1, 0x976D, 0x9777, 0xF9F2, /* 0x70 */
        0xF9F3, 0x4E00, 0x4F5A, 0x4F7E, 0x58F9, 0x65E5, 0x6EA2, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x9038, 0x93B0, 0x99B9, 0x4EFB, 0x58EC, 0x598A, 0x59D9, /* 0x90 */
        0x6041, 0xF9F4, 0xF9F5, 0x7A14, 0xF9F6, 0x834F, 0x8CC3, 0x5165, /* 0x90 */
        0x5344, 0xF9F7, 0xF9F8, 0xF9F9, 0x4ECD, 0x5269, 0x5B55, 0x82BF, /* 0xA0 */
        0x4ED4, 0x523A, 0x54A8, 0x59C9, 0x59FF, 0x5B50, 0x5B57, 0x5B5C, /* 0xA0 */
        0x6063, 0x6148, 0x6ECB, 0x7099, 0x716E, 0x7386, 0x74F7, 0x75B5, /* 0xB0 */
        0x78C1, 0x7D2B, 0x8005, 0x81EA, 0x8328, 0x8517, 0x85C9, 0x8AEE, /* 0xB0 */
        0x8CC7, 0x96CC, 0x4F5C, 0x52FA, 0x56BC, 0x65AB, 0x6628, 0x707C, /* 0xC0 */
        0x70B8, 0x7235, 0x7DBD, 0x828D, 0x914C, 0x96C0, 0x9D72, 0x5B71, /* 0xC0 */
        0x68E7, 0x6B98, 0x6F7A, 0x76DE, 0x5C91, 0x66AB, 0x6F5B, 0x7BB4, /* 0xD0 */
        0x7C2A, 0x8836, 0x96DC, 0x4E08, 0x4ED7, 0x5320, 0x5834, 0x58BB, /* 0xD0 */
        0x58EF, 0x596C, 0x5C07, 0x5E33, 0x5E84, 0x5F35, 0x638C, 0x66B2, /* 0xE0 */
        0x6756, 0x6A1F, 0x6AA3, 0x6B0C, 0x6F3F, 0x7246, 0xF9FA, 0x7350, /* 0xE0 */
        0x748B, 0x7AE0, 0x7CA7, 0x8178, 0x81DF, 0x81E7, 0x838A, 0x846C, /* 0xF0 */
        0x8523, 0x8594, 0x85CF, 0x88DD, 0x8D13, 0x91AC, 0x9577			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf2:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x969C, 0x518D, 0x54C9, 0x5728, 0x5BB0, 0x624D, 0x6750, /* 0x30 */
        0x683D, 0x6893, 0x6E3D, 0x6ED3, 0x707D, 0x7E21, 0x88C1, 0x8CA1, /* 0x30 */
        0x8F09, 0x9F4B, 0x9F4E, 0x722D, 0x7B8F, 0x8ACD, 0x931A, 0x4F47, /* 0x40 */
        0x4F4E, 0x5132, 0x5480, 0x59D0, 0x5E95, 0x62B5, 0x6775, 0x696E, /* 0x40 */
        0x6A17, 0x6CAE, 0x6E1A, 0x72D9, 0x732A, 0x75BD, 0x7BB8, 0x7D35, /* 0x50 */
        0x82E7, 0x83F9, 0x8457, 0x85F7, 0x8A5B, 0x8CAF, 0x8E87, 0x9019, /* 0x50 */
        0x90B8, 0x96CE, 0x9F5F, 0x52E3, 0x540A, 0x5AE1, 0x5BC2, 0x6458, /* 0x60 */
        0x6575, 0x6EF4, 0x72C4, 0xF9FB, 0x7684, 0x7A4D, 0x7B1B, 0x7C4D, /* 0x60 */
        0x7E3E, 0x7FDF, 0x837B, 0x8B2B, 0x8CCA, 0x8D64, 0x8DE1, 0x8E5F, /* 0x70 */
        0x8FEA, 0x8FF9, 0x9069, 0x93D1, 0x4F43, 0x4F7A, 0x50B3, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x5168, 0x5178, 0x524D, 0x526A, 0x5861, 0x587C, 0x5960, /* 0x90 */
        0x5C08, 0x5C55, 0x5EDB, 0x609B, 0x6230, 0x6813, 0x6BBF, 0x6C08, /* 0x90 */
        0x6FB1, 0x714E, 0x7420, 0x7530, 0x7538, 0x7551, 0x7672, 0x7B4C, /* 0xA0 */
        0x7B8B, 0x7BAD, 0x7BC6, 0x7E8F, 0x8A6E, 0x8F3E, 0x8F49, 0x923F, /* 0xA0 */
        0x9293, 0x9322, 0x942B, 0x96FB, 0x985A, 0x986B, 0x991E, 0x5207, /* 0xB0 */
        0x622A, 0x6298, 0x6D59, 0x7664, 0x7ACA, 0x7BC0, 0x7D76, 0x5360, /* 0xB0 */
        0x5CBE, 0x5E97, 0x6F38, 0x70B9, 0x7C98, 0x9711, 0x9B8E, 0x9EDE, /* 0xC0 */
        0x63A5, 0x647A, 0x8776, 0x4E01, 0x4E95, 0x4EAD, 0x505C, 0x5075, /* 0xC0 */
        0x5448, 0x59C3, 0x5B9A, 0x5E40, 0x5EAD, 0x5EF7, 0x5F81, 0x60C5, /* 0xD0 */
        0x633A, 0x653F, 0x6574, 0x65CC, 0x6676, 0x6678, 0x67FE, 0x6968, /* 0xD0 */
        0x6A89, 0x6B63, 0x6C40, 0x6DC0, 0x6DE8, 0x6E1F, 0x6E5E, 0x701E, /* 0xE0 */
        0x70A1, 0x738E, 0x73FD, 0x753A, 0x775B, 0x7887, 0x798E, 0x7A0B, /* 0xE0 */
        0x7A7D, 0x7CBE, 0x7D8E, 0x8247, 0x8A02, 0x8AEA, 0x8C9E, 0x912D, /* 0xF0 */
        0x914A, 0x91D8, 0x9266, 0x92CC, 0x9320, 0x9706, 0x9756			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf3:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x975C, 0x9802, 0x9F0E, 0x5236, 0x5291, 0x557C, 0x5824, /* 0x30 */
        0x5E1D, 0x5F1F, 0x608C, 0x63D0, 0x68AF, 0x6FDF, 0x796D, 0x7B2C, /* 0x30 */
        0x81CD, 0x85BA, 0x88FD, 0x8AF8, 0x8E44, 0x918D, 0x9664, 0x969B, /* 0x40 */
        0x973D, 0x984C, 0x9F4A, 0x4FCE, 0x5146, 0x51CB, 0x52A9, 0x5632, /* 0x40 */
        0x5F14, 0x5F6B, 0x63AA, 0x64CD, 0x65E9, 0x6641, 0x66FA, 0x66F9, /* 0x50 */
        0x671D, 0x689D, 0x68D7, 0x69FD, 0x6F15, 0x6F6E, 0x7167, 0x71E5, /* 0x50 */
        0x722A, 0x74AA, 0x773A, 0x7956, 0x795A, 0x79DF, 0x7A20, 0x7A95, /* 0x60 */
        0x7C97, 0x7CDF, 0x7D44, 0x7E70, 0x8087, 0x85FB, 0x86A4, 0x8A54, /* 0x60 */
        0x8ABF, 0x8D99, 0x8E81, 0x9020, 0x906D, 0x91E3, 0x963B, 0x96D5, /* 0x70 */
        0x9CE5, 0x65CF, 0x7C07, 0x8DB3, 0x93C3, 0x5B58, 0x5C0A, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x5352, 0x62D9, 0x731D, 0x5027, 0x5B97, 0x5F9E, 0x60B0, /* 0x90 */
        0x616B, 0x68D5, 0x6DD9, 0x742E, 0x7A2E, 0x7D42, 0x7D9C, 0x7E31, /* 0x90 */
        0x816B, 0x8E2A, 0x8E35, 0x937E, 0x9418, 0x4F50, 0x5750, 0x5DE6, /* 0xA0 */
        0x5EA7, 0x632B, 0x7F6A, 0x4E3B, 0x4F4F, 0x4F8F, 0x505A, 0x59DD, /* 0xA0 */
        0x80C4, 0x546A, 0x5468, 0x55FE, 0x594F, 0x5B99, 0x5DDE, 0x5EDA, /* 0xB0 */
        0x665D, 0x6731, 0x67F1, 0x682A, 0x6CE8, 0x6D32, 0x6E4A, 0x6F8D, /* 0xB0 */
        0x70B7, 0x73E0, 0x7587, 0x7C4C, 0x7D02, 0x7D2C, 0x7DA2, 0x821F, /* 0xC0 */
        0x86DB, 0x8A3B, 0x8A85, 0x8D70, 0x8E8A, 0x8F33, 0x9031, 0x914E, /* 0xC0 */
        0x9152, 0x9444, 0x99D0, 0x7AF9, 0x7CA5, 0x4FCA, 0x5101, 0x51C6, /* 0xD0 */
        0x57C8, 0x5BEF, 0x5CFB, 0x6659, 0x6A3D, 0x6D5A, 0x6E96, 0x6FEC, /* 0xD0 */
        0x710C, 0x756F, 0x7AE3, 0x8822, 0x9021, 0x9075, 0x96CB, 0x99FF, /* 0xE0 */
        0x8301, 0x4E2D, 0x4EF2, 0x8846, 0x91CD, 0x537D, 0x6ADB, 0x696B, /* 0xE0 */
        0x6C41, 0x847A, 0x589E, 0x618E, 0x66FE, 0x62EF, 0x70DD, 0x7511, /* 0xF0 */
        0x75C7, 0x7E52, 0x84B8, 0x8B49, 0x8D08, 0x4E4B, 0x53EA			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf4:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x54AB, 0x5730, 0x5740, 0x5FD7, 0x6301, 0x6307, 0x646F, /* 0x30 */
        0x652F, 0x65E8, 0x667A, 0x679D, 0x67B3, 0x6B62, 0x6C60, 0x6C9A, /* 0x30 */
        0x6F2C, 0x77E5, 0x7825, 0x7949, 0x7957, 0x7D19, 0x80A2, 0x8102, /* 0x40 */
        0x81F3, 0x829D, 0x82B7, 0x8718, 0x8A8C, 0xF9FC, 0x8D04, 0x8DBE, /* 0x40 */
        0x9072, 0x76F4, 0x7A19, 0x7A37, 0x7E54, 0x8077, 0x5507, 0x55D4, /* 0x50 */
        0x5875, 0x632F, 0x6422, 0x6649, 0x664B, 0x686D, 0x699B, 0x6B84, /* 0x50 */
        0x6D25, 0x6EB1, 0x73CD, 0x7468, 0x74A1, 0x755B, 0x75B9, 0x76E1, /* 0x60 */
        0x771E, 0x778B, 0x79E6, 0x7E09, 0x7E1D, 0x81FB, 0x852F, 0x8897, /* 0x60 */
        0x8A3A, 0x8CD1, 0x8EEB, 0x8FB0, 0x9032, 0x93AD, 0x9663, 0x9673, /* 0x70 */
        0x9707, 0x4F84, 0x53F1, 0x59EA, 0x5AC9, 0x5E19, 0x684E, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x74C6, 0x75BE, 0x79E9, 0x7A92, 0x81A3, 0x86ED, 0x8CEA, /* 0x90 */
        0x8DCC, 0x8FED, 0x659F, 0x6715, 0xF9FD, 0x57F7, 0x6F57, 0x7DDD, /* 0x90 */
        0x8F2F, 0x93F6, 0x96C6, 0x5FB5, 0x61F2, 0x6F84, 0x4E14, 0x4F98, /* 0xA0 */
        0x501F, 0x53C9, 0x55DF, 0x5D6F, 0x5DEE, 0x6B21, 0x6B64, 0x78CB, /* 0xA0 */
        0x7B9A, 0xF9FE, 0x8E49, 0x8ECA, 0x906E, 0x6349, 0x643E, 0x7740, /* 0xB0 */
        0x7A84, 0x932F, 0x947F, 0x9F6A, 0x64B0, 0x6FAF, 0x71E6, 0x74A8, /* 0xB0 */
        0x74DA, 0x7AC4, 0x7C12, 0x7E82, 0x7CB2, 0x7E98, 0x8B9A, 0x8D0A, /* 0xC0 */
        0x947D, 0x9910, 0x994C, 0x5239, 0x5BDF, 0x64E6, 0x672D, 0x7D2E, /* 0xC0 */
        0x50ED, 0x53C3, 0x5879, 0x6158, 0x6159, 0x61FA, 0x65AC, 0x7AD9, /* 0xD0 */
        0x8B92, 0x8B96, 0x5009, 0x5021, 0x5275, 0x5531, 0x5A3C, 0x5EE0, /* 0xD0 */
        0x5F70, 0x6134, 0x655E, 0x660C, 0x6636, 0x66A2, 0x69CD, 0x6EC4, /* 0xE0 */
        0x6F32, 0x7316, 0x7621, 0x7A93, 0x8139, 0x8259, 0x83D6, 0x84BC, /* 0xE0 */
        0x50B5, 0x57F0, 0x5BC0, 0x5BE8, 0x5F69, 0x63A1, 0x7826, 0x7DB5, /* 0xF0 */
        0x83DC, 0x8521, 0x91C7, 0x91F5, 0x518A, 0x67F5, 0x7B56			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf5:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x8CAC, 0x51C4, 0x59BB, 0x60BD, 0x8655, 0x501C, 0xF9FF, /* 0x30 */
        0x5254, 0x5C3A, 0x617D, 0x621A, 0x62D3, 0x64F2, 0x65A5, 0x6ECC, /* 0x30 */
        0x7620, 0x810A, 0x8E60, 0x965F, 0x96BB, 0x4EDF, 0x5343, 0x5598, /* 0x40 */
        0x5929, 0x5DDD, 0x64C5, 0x6CC9, 0x6DFA, 0x7394, 0x7A7F, 0x821B, /* 0x40 */
        0x85A6, 0x8CE4, 0x8E10, 0x9077, 0x91E7, 0x95E1, 0x9621, 0x97C6, /* 0x50 */
        0x51F8, 0x54F2, 0x5586, 0x5FB9, 0x64A4, 0x6F88, 0x7DB4, 0x8F1F, /* 0x50 */
        0x8F4D, 0x9435, 0x50C9, 0x5C16, 0x6CBE, 0x6DFB, 0x751B, 0x77BB, /* 0x60 */
        0x7C3D, 0x7C64, 0x8A79, 0x8AC2, 0x581E, 0x59BE, 0x5E16, 0x6377, /* 0x60 */
        0x7252, 0x758A, 0x776B, 0x8ADC, 0x8CBC, 0x8F12, 0x5EF3, 0x6674, /* 0x70 */
        0x6DF8, 0x807D, 0x83C1, 0x8ACB, 0x9751, 0x9BD6, 0xFA00, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x5243, 0x66FF, 0x6D95, 0x6EEF, 0x7DE0, 0x8AE6, 0x902E, /* 0x90 */
        0x905E, 0x9AD4, 0x521D, 0x527F, 0x54E8, 0x6194, 0x6284, 0x62DB, /* 0x90 */
        0x68A2, 0x6912, 0x695A, 0x6A35, 0x7092, 0x7126, 0x785D, 0x7901, /* 0xA0 */
        0x790E, 0x79D2, 0x7A0D, 0x8096, 0x8278, 0x82D5, 0x8349, 0x8549, /* 0xA0 */
        0x8C82, 0x8D85, 0x9162, 0x918B, 0x91AE, 0x4FC3, 0x56D1, 0x71ED, /* 0xB0 */
        0x77D7, 0x8700, 0x89F8, 0x5BF8, 0x5FD6, 0x6751, 0x90A8, 0x53E2, /* 0xB0 */
        0x585A, 0x5BF5, 0x60A4, 0x6181, 0x6460, 0x7E3D, 0x8070, 0x8525, /* 0xC0 */
        0x9283, 0x64AE, 0x50AC, 0x5D14, 0x6700, 0x589C, 0x62BD, 0x63A8, /* 0xC0 */
        0x690E, 0x6978, 0x6A1E, 0x6E6B, 0x76BA, 0x79CB, 0x82BB, 0x8429, /* 0xD0 */
        0x8ACF, 0x8DA8, 0x8FFD, 0x9112, 0x914B, 0x919C, 0x9310, 0x9318, /* 0xD0 */
        0x939A, 0x96DB, 0x9A36, 0x9C0D, 0x4E11, 0x755C, 0x795D, 0x7AFA, /* 0xE0 */
        0x7B51, 0x7BC9, 0x7E2E, 0x84C4, 0x8E59, 0x8E74, 0x8EF8, 0x9010, /* 0xE0 */
        0x6625, 0x693F, 0x7443, 0x51FA, 0x672E, 0x9EDC, 0x5145, 0x5FE0, /* 0xF0 */
        0x6C96, 0x87F2, 0x885D, 0x8877, 0x60B4, 0x81B5, 0x8403			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf6:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x8D05, 0x53D6, 0x5439, 0x5634, 0x5A36, 0x5C31, 0x708A, /* 0x30 */
        0x7FE0, 0x805A, 0x8106, 0x81ED, 0x8DA3, 0x9189, 0x9A5F, 0x9DF2, /* 0x30 */
        0x5074, 0x4EC4, 0x53A0, 0x60FB, 0x6E2C, 0x5C64, 0x4F88, 0x5024, /* 0x40 */
        0x55E4, 0x5CD9, 0x5E5F, 0x6065, 0x6894, 0x6CBB, 0x6DC4, 0x71BE, /* 0x40 */
        0x75D4, 0x75F4, 0x7661, 0x7A1A, 0x7A49, 0x7DC7, 0x7DFB, 0x7F6E, /* 0x50 */
        0x81F4, 0x86A9, 0x8F1C, 0x96C9, 0x99B3, 0x9F52, 0x5247, 0x52C5, /* 0x50 */
        0x98ED, 0x89AA, 0x4E03, 0x67D2, 0x6F06, 0x4FB5, 0x5BE2, 0x6795, /* 0x60 */
        0x6C88, 0x6D78, 0x741B, 0x7827, 0x91DD, 0x937C, 0x87C4, 0x79E4, /* 0x60 */
        0x7A31, 0x5FEB, 0x4ED6, 0x54A4, 0x553E, 0x58AE, 0x59A5, 0x60F0, /* 0x70 */
        0x6253, 0x62D6, 0x6736, 0x6955, 0x8235, 0x9640, 0x99B1, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x99DD, 0x502C, 0x5353, 0x5544, 0x577C, 0xFA01, 0x6258, /* 0x90 */
        0xFA02, 0x64E2, 0x666B, 0x67DD, 0x6FC1, 0x6FEF, 0x7422, 0x7438, /* 0x90 */
        0x8A17, 0x9438, 0x5451, 0x5606, 0x5766, 0x5F48, 0x619A, 0x6B4E, /* 0xA0 */
        0x7058, 0x70AD, 0x7DBB, 0x8A95, 0x596A, 0x812B, 0x63A2, 0x7708, /* 0xA0 */
        0x803D, 0x8CAA, 0x5854, 0x642D, 0x69BB, 0x5B95, 0x5E11, 0x6E6F, /* 0xB0 */
        0xFA03, 0x8569, 0x514C, 0x53F0, 0x592A, 0x6020, 0x614B, 0x6B86, /* 0xB0 */
        0x6C70, 0x6CF0, 0x7B1E, 0x80CE, 0x82D4, 0x8DC6, 0x90B0, 0x98B1, /* 0xC0 */
        0xFA04, 0x64C7, 0x6FA4, 0x6491, 0x6504, 0x514E, 0x5410, 0x571F, /* 0xC0 */
        0x8A0E, 0x615F, 0x6876, 0xFA05, 0x75DB, 0x7B52, 0x7D71, 0x901A, /* 0xD0 */
        0x5806, 0x69CC, 0x817F, 0x892A, 0x9000, 0x9839, 0x5078, 0x5957, /* 0xD0 */
        0x59AC, 0x6295, 0x900F, 0x9B2A, 0x615D, 0x7279, 0x95D6, 0x5761, /* 0xE0 */
        0x5A46, 0x5DF4, 0x628A, 0x64AD, 0x64FA, 0x6777, 0x6CE2, 0x6D3E, /* 0xE0 */
        0x722C, 0x7436, 0x7834, 0x7F77, 0x82AD, 0x8DDB, 0x9817, 0x5224, /* 0xF0 */
        0x5742, 0x677F, 0x7248, 0x74E3, 0x8CA9, 0x8FA6, 0x9211			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf7:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x962A, 0x516B, 0x53ED, 0x634C, 0x4F69, 0x5504, 0x6096, /* 0x30 */
        0x6557, 0x6C9B, 0x6D7F, 0x724C, 0x72FD, 0x7A17, 0x8987, 0x8C9D, /* 0x30 */
        0x5F6D, 0x6F8E, 0x70F9, 0x81A8, 0x610E, 0x4FBF, 0x504F, 0x6241, /* 0x40 */
        0x7247, 0x7BC7, 0x7DE8, 0x7FE9, 0x904D, 0x97AD, 0x9A19, 0x8CB6, /* 0x40 */
        0x576A, 0x5E73, 0x67B0, 0x840D, 0x8A55, 0x5420, 0x5B16, 0x5E63, /* 0x50 */
        0x5EE2, 0x5F0A, 0x6583, 0x80BA, 0x853D, 0x9589, 0x965B, 0x4F48, /* 0x50 */
        0x5305, 0x530D, 0x530F, 0x5486, 0x54FA, 0x5703, 0x5E03, 0x6016, /* 0x60 */
        0x629B, 0x62B1, 0x6355, 0xFA06, 0x6CE1, 0x6D66, 0x75B1, 0x7832, /* 0x60 */
        0x80DE, 0x812F, 0x82DE, 0x8461, 0x84B2, 0x888D, 0x8912, 0x900B, /* 0x70 */
        0x92EA, 0x98FD, 0x9B91, 0x5E45, 0x66B4, 0x66DD, 0x7011, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x7206, 0xFA07, 0x4FF5, 0x527D, 0x5F6A, 0x6153, 0x6753, /* 0x90 */
        0x6A19, 0x6F02, 0x74E2, 0x7968, 0x8868, 0x8C79, 0x98C7, 0x98C4, /* 0x90 */
        0x9A43, 0x54C1, 0x7A1F, 0x6953, 0x8AF7, 0x8C4A, 0x98A8, 0x99AE, /* 0xA0 */
        0x5F7C, 0x62AB, 0x75B2, 0x76AE, 0x88AB, 0x907F, 0x9642, 0x5339, /* 0xA0 */
        0x5F3C, 0x5FC5, 0x6CCC, 0x73CC, 0x7562, 0x758B, 0x7B46, 0x82FE, /* 0xB0 */
        0x999D, 0x4E4F, 0x903C, 0x4E0B, 0x4F55, 0x53A6, 0x590F, 0x5EC8, /* 0xB0 */
        0x6630, 0x6CB3, 0x7455, 0x8377, 0x8766, 0x8CC0, 0x9050, 0x971E, /* 0xC0 */
        0x9C15, 0x58D1, 0x5B78, 0x8650, 0x8B14, 0x9DB4, 0x5BD2, 0x6068, /* 0xC0 */
        0x608D, 0x65F1, 0x6C57, 0x6F22, 0x6FA3, 0x701A, 0x7F55, 0x7FF0, /* 0xD0 */
        0x9591, 0x9592, 0x9650, 0x97D3, 0x5272, 0x8F44, 0x51FD, 0x542B, /* 0xD0 */
        0x54B8, 0x5563, 0x558A, 0x6ABB, 0x6DB5, 0x7DD8, 0x8266, 0x929C, /* 0xE0 */
        0x9677, 0x9E79, 0x5408, 0x54C8, 0x76D2, 0x86E4, 0x95A4, 0x95D4, /* 0xE0 */
        0x965C, 0x4EA2, 0x4F09, 0x59EE, 0x5AE6, 0x5DF7, 0x6052, 0x6297, /* 0xF0 */
        0x676D, 0x6841, 0x6C86, 0x6E2F, 0x7F38, 0x809B, 0x822A			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf8:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0xFA08, 0xFA09, 0x9805, 0x4EA5, 0x5055, 0x54B3, 0x5793, /* 0x30 */
        0x595A, 0x5B69, 0x5BB3, 0x61C8, 0x6977, 0x6D77, 0x7023, 0x87F9, /* 0x30 */
        0x89E3, 0x8A72, 0x8AE7, 0x9082, 0x99ED, 0x9AB8, 0x52BE, 0x6838, /* 0x40 */
        0x5016, 0x5E78, 0x674F, 0x8347, 0x884C, 0x4EAB, 0x5411, 0x56AE, /* 0x40 */
        0x73E6, 0x9115, 0x97FF, 0x9909, 0x9957, 0x9999, 0x5653, 0x589F, /* 0x50 */
        0x865B, 0x8A31, 0x61B2, 0x6AF6, 0x737B, 0x8ED2, 0x6B47, 0x96AA, /* 0x50 */
        0x9A57, 0x5955, 0x7200, 0x8D6B, 0x9769, 0x4FD4, 0x5CF4, 0x5F26, /* 0x60 */
        0x61F8, 0x665B, 0x6CEB, 0x70AB, 0x7384, 0x73B9, 0x73FE, 0x7729, /* 0x60 */
        0x774D, 0x7D43, 0x7D62, 0x7E23, 0x8237, 0x8852, 0xFA0A, 0x8CE2, /* 0x70 */
        0x9249, 0x986F, 0x5B51, 0x7A74, 0x8840, 0x9801, 0x5ACC, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x4FE0, 0x5354, 0x593E, 0x5CFD, 0x633E, 0x6D79, 0x72F9, /* 0x90 */
        0x8105, 0x8107, 0x83A2, 0x92CF, 0x9830, 0x4EA8, 0x5144, 0x5211, /* 0x90 */
        0x578B, 0x5F62, 0x6CC2, 0x6ECE, 0x7005, 0x7050, 0x70AF, 0x7192, /* 0xA0 */
        0x73E9, 0x7469, 0x834A, 0x87A2, 0x8861, 0x9008, 0x90A2, 0x93A3, /* 0xA0 */
        0x99A8, 0x516E, 0x5F57, 0x60E0, 0x6167, 0x66B3, 0x8559, 0x8E4A, /* 0xB0 */
        0x91AF, 0x978B, 0x4E4E, 0x4E92, 0x547C, 0x58D5, 0x58FA, 0x597D, /* 0xB0 */
        0x5CB5, 0x5F27, 0x6236, 0x6248, 0x660A, 0x6667, 0x6BEB, 0x6D69, /* 0xC0 */
        0x6DCF, 0x6E56, 0x6EF8, 0x6F94, 0x6FE0, 0x6FE9, 0x705D, 0x72D0, /* 0xC0 */
        0x7425, 0x745A, 0x74E0, 0x7693, 0x795C, 0x7CCA, 0x7E1E, 0x80E1, /* 0xD0 */
        0x82A6, 0x846B, 0x84BF, 0x864E, 0x865F, 0x8774, 0x8B77, 0x8C6A, /* 0xD0 */
        0x93AC, 0x9800, 0x9865, 0x60D1, 0x6216, 0x9177, 0x5A5A, 0x660F, /* 0xE0 */
        0x6DF7, 0x6E3E, 0x743F, 0x9B42, 0x5FFD, 0x60DA, 0x7B0F, 0x54C4, /* 0xE0 */
        0x5F18, 0x6C5E, 0x6CD3, 0x6D2A, 0x70D8, 0x7D05, 0x8679, 0x8A0C, /* 0xF0 */
        0x9D3B, 0x5316, 0x548C, 0x5B05, 0x6A3A, 0x706B, 0x7575			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  case 0xf9:
    if (c2>=0x31 && c2<=0xfe) {
      static int const(val[])= {
        0x798D, 0x79BE, 0x82B1, 0x83EF, 0x8A71, 0x8B41, 0x8CA8, /* 0x30 */
        0x9774, 0xFA0B, 0x64F4, 0x652B, 0x78BA, 0x78BB, 0x7A6B, 0x4E38, /* 0x30 */
        0x559A, 0x5950, 0x5BA6, 0x5E7B, 0x60A3, 0x63DB, 0x6B61, 0x6665, /* 0x40 */
        0x6853, 0x6E19, 0x7165, 0x74B0, 0x7D08, 0x9084, 0x9A69, 0x9C25, /* 0x40 */
        0x6D3B, 0x6ED1, 0x733E, 0x8C41, 0x95CA, 0x51F0, 0x5E4C, 0x5FA8, /* 0x50 */
        0x604D, 0x60F6, 0x6130, 0x614C, 0x6643, 0x6644, 0x69A5, 0x6CC1, /* 0x50 */
        0x6E5F, 0x6EC9, 0x6F62, 0x714C, 0x749C, 0x7687, 0x7BC1, 0x7C27, /* 0x60 */
        0x8352, 0x8757, 0x9051, 0x968D, 0x9EC3, 0x532F, 0x56DE, 0x5EFB, /* 0x60 */
        0x5F8A, 0x6062, 0x6094, 0x61F7, 0x6666, 0x6703, 0x6A9C, 0x6DEE, /* 0x70 */
        0x6FAE, 0x7070, 0x736A, 0x7E6A, 0x81BE, 0x8334, 0x86D4, 	 0, /* 0x70 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, 	 0, /* 0x80 */
        0, 0x8AA8, 0x8CC4, 0x5283, 0x7372, 0x5B96, 0x6A6B, 0x9404, /* 0x90 */
        0x54EE, 0x5686, 0x5B5D, 0x6548, 0x6585, 0x66C9, 0x689F, 0x6D8D, /* 0x90 */
        0x6DC6, 0x723B, 0x80B4, 0x9175, 0x9A4D, 0x4FAF, 0x5019, 0x539A, /* 0xA0 */
        0x540E, 0x543C, 0x5589, 0x55C5, 0x5E3F, 0x5F8C, 0x673D, 0x7166, /* 0xA0 */
        0x73DD, 0x9005, 0x52DB, 0x52F3, 0x5864, 0x58CE, 0x7104, 0x718F, /* 0xB0 */
        0x71FB, 0x85B0, 0x8A13, 0x6688, 0x85A8, 0x55A7, 0x6684, 0x714A, /* 0xB0 */
        0x8431, 0x5349, 0x5599, 0x6BC1, 0x5F59, 0x5FBD, 0x63EE, 0x6689, /* 0xC0 */
        0x7147, 0x8AF1, 0x8F1D, 0x9EBE, 0x4F11, 0x643A, 0x70CB, 0x7566, /* 0xC0 */
        0x8667, 0x6064, 0x8B4E, 0x9DF8, 0x5147, 0x51F6, 0x5308, 0x6D36, /* 0xD0 */
        0x80F8, 0x9ED1, 0x6615, 0x6B23, 0x7098, 0x75D5, 0x5403, 0x5C79, /* 0xD0 */
        0x7D07, 0x8A16, 0x6B20, 0x6B3D, 0x6B46, 0x5438, 0x6070, 0x6D3D, /* 0xE0 */
        0x7FD5, 0x8208, 0x50D6, 0x51DE, 0x559C, 0x566B, 0x56CD, 0x59EC, /* 0xE0 */
        0x5B09, 0x5E0C, 0x6199, 0x6198, 0x6231, 0x665E, 0x66E6, 0x7199, /* 0xF0 */
        0x71B9, 0x71BA, 0x72A7, 0x79A7, 0x7A00, 0x7FB2, 0x8A70			/* 0xF0 */
      };
      unicode=(uint32_t) val[c2-0x31];
    }
    else
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
    break;
  default:
    if (c==0xd8 || c==0xdf) {
      unicode=uint32_t((unicode<<8)+(uint32_t) c2);
      break;
    }
    break;
  }
  if (!unicode) {
    STOFF_DEBUG_MSG(("StarEncodingOtherKorean::readKoreanMS1361: unknown caracter %x\n", (unsigned int)c));
  }
  dest.push_back(unicode);
  return true;
}
// vim: set filetype=cpp tabstop=2 shiftwidth=2 cindent autoindent smartindent noexpandtab:
