/*
  Copyright Red Hat, Inc. 2002-2003

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2, or (at your option) any
  later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; see the file COPYING.  If not, write to the
  Free Software Foundation, Inc.,  675 Mass Ave, Cambridge, 
  MA 02139, USA.
*/
/** @file
 * The New And Improved Cluster Service Admin Utility.
 * TODO Clean up the code.
 */
#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <libgen.h>
#include <resgroup.h>
#include <platform.h>
#include <magma.h>
#include <magmamsg.h>
#include <resgroup.h>
#include <msgsimple.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


void
build_message(SmMessageSt *msgp, int action, char *svcName, uint64_t target,
	      uint32_t arg1, uint32_t arg2)
{
	msgp->sm_hdr.gh_magic = GENERIC_HDR_MAGIC;
	msgp->sm_hdr.gh_command = RG_ACTION_REQUEST;
	msgp->sm_hdr.gh_length = sizeof(*msgp);
	msgp->sm_hdr.gh_arg1 = arg1;
	msgp->sm_hdr.gh_arg2 = arg2;
	msgp->sm_data.d_action = action;
	strncpy(msgp->sm_data.d_svcName, svcName,
		sizeof(msgp->sm_data.d_svcName));
	msgp->sm_data.d_svcOwner = target;
	msgp->sm_data.d_ret = 0;

	swab_SmMessageSt(msgp);
}


int
do_lock_req(int req)
{
	int cfd = -1;
	int fd = -1;
	int ret = RG_FAIL;
	cluster_member_list_t *membership = NULL;
	uint64_t me;
	generic_msg_hdr hdr;

	fd = clu_connect(RG_SERVICE_GROUP, 0);
	if (fd < 0) {
		printf("Could not connect to cluster service\n");
		goto out;
	}

	membership = clu_member_list(RG_SERVICE_GROUP);
	msg_update(membership);
	clu_local_nodeid(RG_SERVICE_GROUP, &me);

	fd = msg_open(me, RG_PORT, 0, 5);
	if (fd < 0) {
		printf("Could not connect to resource group manager\n");
		goto out;
	}

	if (msg_send_simple(fd, req, 0, 0) < 0) {
		printf("Communication failed\n");
		goto out;
	}

	if (msg_receive_timeout(fd, &hdr, sizeof(hdr), 5) < sizeof(hdr)) {
		printf("Receive failed\n");
		goto out;
	}

	swab_generic_msg_hdr(&hdr);
	ret = hdr.gh_command;

out:
	if (membership)
		cml_free(membership);

	if (fd >= 0)
		msg_close(fd);

	if (cfd >= 0)
		clu_disconnect(cfd);

	return ret;
}


int
do_lock(void)
{
	if (do_lock_req(RG_LOCK) != RG_SUCCESS) {
		printf("Lock operation failed\n");
		return 1;
	}
	printf("Resource groups locked\n");
	return 0;
}


int
do_unlock(void)
{
	if (do_lock_req(RG_UNLOCK) != RG_SUCCESS) {
		printf("Unlock operation failed\n");
		return 1;
	}
	printf("Resource groups unlocked\n");
	return 0;
}


int
do_query_lock(void)
{
	switch(do_lock_req(RG_QUERY_LOCK)) {
	case RG_LOCK:
		printf("Resource groups locked\n");
		break;
	case RG_UNLOCK:
		printf("Resource groups unlocked\n");
		break;
	default:
		printf("Query operation failed\n");
		return 1;
	}
	return 0;
}


int
do_msg_receive(uint64_t msgtarget, int fd, void *buf, size_t len)
{
	int ret;
	cluster_member_list_t *m = NULL;
	
	if ((int64_t)msgtarget < (int64_t)0)
		return msg_receive(fd, buf, len);
	
	/* Make sure a node hasn't died while processing our request. */
	do {
		ret = msg_receive_timeout(fd, buf, len, 20);
		if (ret < (int)len) {
			if (ret < 0 && errno == ETIMEDOUT) {
				m = clu_member_list(RG_SERVICE_GROUP);
				if (!memb_online(m, msgtarget)) {
					ret = RG_ENODEDEATH;
					break;
				}
				cml_free(m);
				m = NULL;
				continue;
			}
			
			/* Make sure we don't overwrite ENODEDEATH */
			if (ret < 0)
				ret = -1;
		}
		break;
	} while(1);
	
	if (m)
		cml_free(m);
	return ret;
}


void
usage(char *name)
{
printf("Resource Group Control Commands:\n");
printf("       %s -v                     Display version and exit\n",name);
printf("       %s -d <group>             Disable <group>\n", name);
printf("       %s -e <group>             Enable <group> on the local node\n",
       name);
printf("       %s -e <group> -F          Enable <group> according to failover\n"
       "                                 domain rules\n", name);
printf("       %s -e <group> -m <member> Enable <group>"
       " on <member>\n", name);
printf("       %s -r <group> -m <member> Relocate <group> [to <member>]\n",
	       name);
printf("       %s -q                     Quiet operation\n", name);
printf("       %s -R <group>             Restart a group in place.\n",
       name);
printf("       %s -s <group>             Stop <group>\n", name);
printf("\n");
printf("Resource Group Locking (for cluster Shutdown / Debugging):\n");
printf("       %s -l                     Lock local resource group manager.\n"
       "                                 This prevents resource groups from\n"
       "                                 starting on the local node.\n",
       name);
printf("       %s -S                     Show lock state\n", name);
printf("       %s -u                     Unlock local resource group manager.\n"
       "                                 This allows resource groups to start\n"
       "                                 on the local node.\n", name);
}


int
main(int argc, char **argv)
{
	extern char *optarg;
	char *svcname=NULL, nodename[256];
	int opt;
	int msgfd = -1, fd, fod = 0;
	SmMessageSt msg;
	int action = RG_STATUS;
	int node_specified = 0;
       	uint64_t msgtarget, me, svctarget = NODE_ID_NONE;
	char *actionstr = NULL;
	cluster_member_list_t *membership;

	if (geteuid() != (uid_t) 0) {
		fprintf(stderr, "%s must be run as the root user.\n", argv[0]);
		return 1;
	}

	while ((opt = getopt(argc, argv, "lSue:d:r:Fn:m:vR:s:qh?")) != EOF) {
		switch (opt) {
		case 'l':
			return do_lock();

		case 'S':
			return do_query_lock();

		case 'u':
			return do_unlock();

		case 'e':
			/* ENABLE */
			actionstr = "trying to enable";
			action = RG_ENABLE;
			svcname = optarg;
			break;
		case 'F':
			if (node_specified) {
				fprintf(stderr,
					"Cannot use '-F' with '-n' or '-m'\n");
				return 1;
			}
			fod = 1;
			break;
		case 'd':
			/* DISABLE */
			actionstr = "disabling";
			action = RG_DISABLE;
			svcname = optarg;
			break;
		case 'r':
			/* RELOCATE */
			actionstr = "trying to relocate";
			action = RG_RELOCATE;
			svcname = optarg;
			break;
		case 's':
			/* stop */
			actionstr = "stopping";
			action = RG_STOP_USER;
			svcname = optarg;
			break;
		case 'R':
			actionstr = "trying to restart";
			action = RG_RESTART;
			svcname = optarg;
			break;
		case 'm': /* member ... */
		case 'n': /* node .. same thing */
			if (fod) {
				fprintf(stderr,
					"Cannot use '-F' with '-n' or '-m'\n");
				return 1;
			}
			strncpy(nodename,optarg,sizeof(nodename));
			node_specified = 1;
			break;
		case 'v':
			printf("%s\n",PACKAGE_VERSION);
			return 0;
		case 'q':
			close(STDOUT_FILENO);
			break;
		case 'h':
		case '?':
		default:
			usage(basename(argv[0]));
			return 1;
		}
	}

	if (!svcname) {
		usage(basename(argv[0]));
		return 1;
	}

	/* No login */
	fd = clu_connect(RG_SERVICE_GROUP, 0);
	if (fd < 0) {
		printf("Could not connect to cluster service\n");
		return 1;
	}

	membership = clu_member_list(RG_SERVICE_GROUP);
	msg_update(membership);
	clu_local_nodeid(RG_SERVICE_GROUP, &me);

	if (node_specified) {
		msgtarget = memb_name_to_id(membership, nodename);
		if (msgtarget == NODE_ID_NONE) {
			fprintf(stderr, "Member %s not in membership list\n",
				nodename);
			return 1;
		}
		svctarget = msgtarget;
	} else {
		clu_local_nodeid(RG_SERVICE_GROUP, &msgtarget);
		clu_local_nodename(RG_SERVICE_GROUP, nodename,
				   sizeof(nodename));
	}
	
	build_message(&msg, action, svcname, svctarget, fod, 0);

	if (action != RG_RELOCATE) {
		printf("Member %s %s %s", nodename, actionstr, svcname);
		printf("...");
		fflush(stdout);
		msgfd = msg_open(msgtarget, RG_PORT, 0, 5);
	} else {
		if (node_specified)
			printf("Trying to relocate %s to %s", svcname, nodename);
		else
			printf("Trying to relocate %s", svcname);
		printf("...");
		fflush(stdout);
		msgfd = msg_open(me, RG_PORT, 0, 5);
		/* just do a normal receive from the local node */
		msgtarget = (uint64_t)-1;
	}

	if (msgfd < 0) {
		fprintf(stderr,
			"Could not connect to resource group manager!\n");
		return 1;
	}

	if (msg_send(msgfd, &msg, sizeof(msg)) != sizeof(msg)) {
		perror("msg_send");
		fprintf(stderr, "Could not send entire message!\n");
		return 1;
	}

	/* reusing opt */
	opt = do_msg_receive(msgtarget, msgfd, &msg,
			     sizeof(msg));
	if (opt < (int)sizeof(msg)) {
		if (opt != RG_ENODEDEATH) {
			perror("msg_receive");
			fprintf(stderr, "Error receiving reply!\n");
			return 1;
		}
		
		/*
		 * XXX hack to enable node death processing along side
		 * all the rest of the possible responses.  If an end-node 
		 * died while processing, this will have been set by the 
		 * rgmanager and a response with RG_ENODEDEATH as the d_ret
		 * would have been received.
		 */
		msg.sm_data.d_ret = RG_ENODEDEATH;
		swab_SmMessageSt(&msg);
	}
	
	/* Decode */
	swab_SmMessageSt(&msg);
	switch (msg.sm_data.d_ret) {
	case SUCCESS:
		printf("success\n");

		/* Non-start/relo request: done */
		if (action != RG_RELOCATE && action != RG_ENABLE)
			break;

	    	if (svctarget != NODE_ID_NONE &&
		    msg.sm_data.d_svcOwner != svctarget) {
			/* Service running somewhere besides where requested */
	    		printf("Warning: Service %s is running on %s "
	    			"instead of %s\n", svcname,
	    			memb_id_to_name(membership,
				       msg.sm_data.d_svcOwner),
				memb_id_to_name(membership, svctarget));
			break;
	    	}

		/* No node specified or service running where requested */
	    	printf("Service %s is now running on %s\n", svcname, 
			memb_id_to_name(membership, msg.sm_data.d_svcOwner));
		break;
	case RG_EFAIL:
		printf("failed\n");
		break;
	case RG_ENODEDEATH:
		printf("node processing request died\n");
		printf("(Status unknown)\n");
		break;
	case RG_EABORT:
		printf("cancelled by resource manager\n");
		break;
	case RG_ENOSERVICE:
		printf("failed: Service does not exist\n");
		break;
	case RG_EDEADLCK:
		printf("failed: Operation would deadlock\n");
		break;
	case RG_EAGAIN:
		printf("failed: Try again (resource groups locked)\n");
		break;
	case RG_ERUN:
		printf("failed: Service is already running\n");
		return 0;
		break;
	default:
		printf("failed: unknown reason %d\n", msg.sm_data.d_ret);
		break;
	}

	return msg.sm_data.d_ret;
}
