(in-package :sb-bsd-sockets)

#|| <h2>INET-domain sockets</h2>

<p>The TCP and UDP sockets that you know and love.  Some representation issues:
<ul>
<li>These functions do not accept hostnames directly: see <a href="#name-service">name resolution</a>
<li>Internet <b>addresses</b> are represented by vectors of <tt>(unsigned-byte 8)</tt> - viz. <tt>#(127 0 0 1)</tt>.  <b>Ports</b> are just integers: <tt>6010</tt>.  No conversion between network- and host-order data is needed from the user of this package.
<li><b><i>socket addresses</i></b> are represented by the two values for <b>address</b> and <b>port</b>, so for example, <tt>(<a href="#SOCKET-CONNECT">socket-connect</a> s #(192 168 1 1) 80)</tt>
</ul>

|#

;;; Our class and constructor

(eval-when (:compile-toplevel :load-toplevel :execute)
  (defclass inet-socket (socket)
    ((family :initform sockint::AF-INET))))

;;; XXX should we *...* this?
(defparameter inet-address-any (vector 0 0 0 0))

;;; binding a socket to an address and port.  Doubt that anyone's
;;; actually using this much, to be honest.

(defun make-inet-address (dotted-quads)
  "Return a vector of octets given a string DOTTED-QUADS in the format
\"127.0.0.1\""
  (coerce
   (mapcar #'parse-integer
           (split dotted-quads nil '(#\.)))
   'vector))

;;; getprotobyname only works in the internet domain, which is why this
;;; is here
(defun get-protocol-by-name (name)      ;exported
  "Returns the network protocol number associated with the string NAME,
using getprotobyname(2) which typically looks in NIS or /etc/protocols"
  ;; for extra brownie points, could return canonical protocol name
  ;; and aliases as extra values
  (let ((ent (sockint::foreign-vector (sockint::getprotobyname name) 1
				      sockint::size-of-protoent)))
    (sockint::protoent-proto ent)))


;;; sockaddr protocol
;;; (1) sockaddrs are represented as the semi-foreign array-of-octets
;;; thing
;;; (2) a protocol provides make-sockaddr-for, size-of-sockaddr,
;;; bits-of-sockaddr

(defmethod make-sockaddr-for ((socket inet-socket) &optional sockaddr &rest address &aux (host (first address)) (port (second address)))
  (let ((sockaddr (or sockaddr (sockint::allocate-sockaddr-in))))
    (when (and host port)
      ;; port and host are represented in C as "network-endian" unsigned
      ;; integers of various lengths.  This is stupid.  The value of the
      ;; integer doesn't matter (and will change depending on your
      ;; machine's endianness); what the bind(2) call is interested in
      ;; is the pattern of bytes within that integer.
      
      ;; We have no truck with such dreadful type punning.  Octets to
      ;; octets, dust to dust.
      
      (setf (sockint::sockaddr-in-family sockaddr) sockint::af-inet)
      (setf (sockint::sockaddr-in-port sockaddr 0) (ldb (byte 8 8) port))
      (setf (sockint::sockaddr-in-port sockaddr 1) (ldb (byte 8 0) port))
      
      (setf (sockint::sockaddr-in-addr sockaddr 0) (elt host 0))
      (setf (sockint::sockaddr-in-addr sockaddr 1) (elt host 1))
      (setf (sockint::sockaddr-in-addr sockaddr 2) (elt host 2))
      (setf (sockint::sockaddr-in-addr sockaddr 3) (elt host 3)))
    sockaddr))

(defmethod size-of-sockaddr ((socket inet-socket))
  sockint::size-of-sockaddr-in)

(defmethod bits-of-sockaddr ((socket inet-socket) sockaddr)
  "Returns address and port of SOCKADDR as multiple values"
  (values
   (vector
    (sockint::sockaddr-in-addr sockaddr 0) 
    (sockint::sockaddr-in-addr sockaddr 1) 
    (sockint::sockaddr-in-addr sockaddr 2) 
    (sockint::sockaddr-in-addr sockaddr 3))
   (+ (* 256 (sockint::sockaddr-in-port sockaddr 0))
      (sockint::sockaddr-in-port sockaddr 1))))  
   
(defun make-inet-socket (type protocol)
  "Make an INET socket.  Deprecated in favour of make-instance"
  (make-instance 'inet-socket :type type :protocol protocol))
