/* -*- Mode: C; tab-width: 4 -*- */
/* life --- Conway's game of Life */

#if !defined( lint ) && !defined( SABER )
static const char sccsid[] = "@(#)life.c	5.24 2007/01/18 xlockmore";

#endif

/*-
 * Copyright (c) 1991 by Patrick J. Naughton.
 * Copyright (c) 1997 by David Bagley.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 * event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * Revision History:
 * 18-Jan-2007: Added vertical option.
 * 01-Mar-2003: Added shooters for triangular life.
 * 26-Feb-2003: Added LWSS, MWSS, HWSS in shooter.
 * 25-Feb-2003: Randomly rotate trilife
 * 25-Jan-2003: Spawned a life.h
 * 23-Jan-2003: Added life from Stephen Silver's Life Lexicon
 *		http://www.argentum.freeserve.co.uk/lex_home.htm
 * 23-Jan-2003: Tri life from Carter Bays.  Added B45/S34 which I use as
 *              the true trilife, also available B456/S45 & B45/S23.
 *              Right now the glider for trilife is not integrated into
 *              the "shooter" part of the program.
 *              Other neat ones are B4/S456, B46/S24, B4/S35, B456/S12,
 *              B3/S23, & B4/S46.  See:
 *              http://www.cse.sc.edu/~bays/trilife3/home.html
 * 15-Jan-2003: Moves on if screen blank or static
 * 15-Jan-2003: Alternate life rules from Nathan Thompson's "Day and Night"
 *              B3678/S34678 and David I. Bell's "HighLife" B36/S23.
 *              See http://www.tip.net.au/~dbell/
 *              Other rules that may be neat:
 *              http://entropymine.com/jason/life/alt/
 *              "Diamoeba" B35678/S5678, "Move" B36(8)/S245
 * 01-Nov-2000: Allocation checks
 * 03-Oct-2000: Added more randomness in pattern 90 degree orientation and
 *              mirror image.
 * 08-Dec-1997: Paul Callahan's B2a/S2b34 rule added.
 *              Described on the news site for cellular-automata.
 *              <ppc997@aber.ac.uk>
 *   http://www.cs.jhu.edu/~callahan/lifepage.html
 *   http://www.cs.jhu.edu/~callahan/hexrule.txt
 *   B2a/S2b34: Birth of x if 2a,
 *     Survival of x if 2b, 3, or 4 neighbors
 *     Assume symmetry.
 *     (2a, 2b, 2c: 2o, 2m, 2p original notation)
 *           O O            O .            O .
 *     2a:  . x .     2b:  . x O     2c:  . x .
 *           . .            . .            . O
 *     Also Bob Andreen's rule (my own notation for consistency)
 *     B2a3a4b/S2a2b4a (original notation: 234'B/22'4S)
 *     <andreen@msmc.edu>
 *           O O            O O            O .
 *     3a:  . x O     3b:  . x .     3c:  . x O
 *           . .            . O            O .
 *
 *           O O            O O            O O
 *     4a:  . x O     4b:  . x O     4c:  . x .
 *           . O            O .            O O
 *     Some other rules
 *     B2a3b3c5/S12b2c3a4b4c6
 *     B23a3c4b4c6/S12b2c3c4a56
 *     B2a2c6/S13b
 * 27-Oct-1997: xpm and ras capability added.
 * 04-Jun-1997: Removed old algorithm, now use wator's.  I could not
 *              understand it and had trouble adding more features.
 *              New algorithm is more efficient iff there lots of blank
 *              areas (ptr loop rather than a double array loop)
 * 10-May-1997: Compatible with xscreensaver
 * 07-May-1997: life neighbor option.  Still have to fix -neighbors 3
 * 07-Jan-1995: life now has a random soup pattern.
 * 07-Dec-1994: life now has new organisms.  They are now better centered.
 *              Some of the nonperiodic forms were removed. New life
 *              forms were taken from xlife (an AMAZING collection of life
 *              forms).  life's gliders now come from the edge of the screen
 *              except when generated by a life form.
 * 23-Nov-1994: Bug fix for different iconified window sizes
 * 21-Jul-1994: Took out bzero & bcopy since memset & memcpy is more portable
 * 10-Jun-1994: Changed name of function 'kill', which is a libc function on
 *              many systems from Victor Langeveld <vic@mbfys.kun.nl>
 * Changes in original xlock
 * 24-May-1991: Added wraparound code from johnson@bugs.comm.mot.com.
 *              Made old cells stay blue.
 *              Made batchcount control the number of generations until restart.
 * 29-Jul-1990: support for multiple screens.
 * 07-Feb-1990: remove bogus semi-colon after #include line.
 * 15-Dec-1989: Fix for proper skipping of {White,Black}Pixel() in colors.
 * 08-Oct-1989: Moved seconds() to an extern.
 * 20-Sep-1989: Written, life algorithm courtesy of Jim Graham <flar@sun.com>
 *
 *
 * References:
 * Bays, Carter, "Cellular Automata in the Trianglular Tessellation",
 *   1994, Complex Systems 8 127-150
 * Bays, Carter, "A Note on the Game of Life in Hexagonal and Pentagonal
 *   Tesselations", 2005, Complex Systems 15 245-252
 */

/*-
  Grid     Number of Neighbors
  ----     ------------------
  Square   4 or 8
  Hexagon  6
  Triangle 3, 9, or 12
  Pentagon 5 or 7

  Conway's Life: -neighbors 8 -rule S23/B3 LIFE, CONWAY
  Other things to try:
    -neighbors 8 -rule B36/S23 <HIGHLIFE, BELL>
    -neighbors 8 -rule B3678/S34678 <DAY_NIGHT, THOMPSON>
    -neighbors 4 -rule B2/S234
    -neighbors 6 -rule B3/S23
    -neighbors 3 -rule B23/S12
    -neighbors 6 -rule B2a/S2b34     <CALLAHAN>
    -neighbors 6 -rule B2b3a4b/S2a2b4a  <ANDREEN>
    -neighbors 6 -rule B2/S35    <BAYS>
    -neighbors 12 -rule B45/S34  <TRILIFE, BAYS>
    -neighbors 12 -rule B456/S45  <TRILIFE1, BAYS>
    -neighbors 12 -rule B45/S23  <TRILIFE2, BAYS>
    -neighbors 5 -rule B2S24
    -neighbors 7 -rule B346/S23
    -neighbors 9 -rule B34/S3
*/

#ifdef STANDALONE
#define MODE_life
#define PROGCLASS "Life"
#define HACK_INIT init_life
#define HACK_DRAW draw_life
#define life_opts xlockmore_opts
#define DEFAULTS "*delay: 750000 \n" \
 "*count: 40 \n" \
 "*cycles: 140 \n" \
 "*size: 0 \n" \
 "*ncolors: 200 \n" \
 "*bitmap: \n" \
 "*neighbors: 0 \n" \
 "*verbose: False \n"
#define UNIFORM_COLORS
#include "xlockmore.h"		/* in xscreensaver distribution */
#else /* STANDALONE */
#include "xlock.h"		/* in xlockmore distribution */
#include "color.h"
#endif /* STANDALONE */
#include "iostuff.h"
#include "automata.h"

#ifdef MODE_life
#define LIFE_NAMES 1
#include "life.h"

#ifdef LIFE_NAMES
#define DEF_LABEL "True"
#define FONT_HEIGHT 19
#define FONT_WIDTH 15
#endif
#define DEF_DRAW "True"
#define DEF_NEIGHBORS  "0"	/* choose best value (8) */
#define DEF_SERIAL "False"
#define DEF_REPEAT "0"		/* Frequently 2, 12 */

#if 1
#define DEF_RULE  "G"   /* All rules with known gliders */
#else
#define DEF_RULE  "P"   /* All rules with known patterns, currently G==P */
#define DEF_RULE  "B3/S23"      /* LIFE */
#define DEF_RULE  "B36/S23"     /* HIGHLIFE */
#define DEF_RULE  "B3678/S34678"       /* "B3678/S34678" DAY_NIGHT*/
#define DEF_RULE  "B2a/S2b34"   /* CALLAHAN */
#define DEF_RULE  "B2b3a4b/S2a2b4a"    /* ANDREEN */
#define DEF_RULE  "B2/S35"      /* BAYS */
#define DEF_RULE  "B45/S34"     /* TRILIFE */
#define DEF_RULE  "B456/S45"    /* TRILIFE1 */
#define DEF_RULE  "B45/S23"     /* TRILIFE2 */
#endif

#define DEF_VERTICAL  "False"
#define DEF_GLIDERSEARCH  "False"
#define DEF_PATTERNSEARCH  "False"
#define DEF_NONTOT  "False"
#define DEF_CONWAY  "False"
#define DEF_HIGHLIFE  "False"
#define DEF_DAY_NIGHT  "False"
#define DEF_CALLAHAN  "False"
#define DEF_ANDREEN  "False"
#define DEF_BAYS  "False"
#define DEF_TRILIFE  "False"
#define DEF_TRILIFE1  "False"
#define DEF_TRILIFE2  "False"
#define DEF_PENT  "False" /* 5 neighbors */
#define DEF_PENT2  "False" /* 7 neighbors */
  /* calling Bays' rule pent2 as its easier to remember, but found first */

#ifdef LIFE_NAMES
static Bool label;
#endif
static Bool draw;
static int  neighbors;
static int  repeat;
static char *rule;
static char *lifefile;
static Bool serial;
static Bool vertical;
static Bool glidersearch;
static Bool patternsearch;
static Bool nontot;
static Bool conway;
static Bool highlife;
static Bool daynight;
static Bool callahan;
static Bool andreen;
static Bool bays;
static Bool trilife;
static Bool trilife1;
static Bool trilife2;
static Bool pent;
static Bool pent2;

static XrmOptionDescRec opts[] =
{
#ifdef LIFE_NAMES
	{(char *) "-label", (char *) ".life.label", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+label", (char *) ".life.label", XrmoptionNoArg, (caddr_t) "off"},
#endif
	{(char *) "-draw", (char *) ".life.draw", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+draw", (char *) ".life.draw", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-neighbors", (char *) ".life.neighbors", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-repeat", (char *) ".life.repeat", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-rule", (char *) ".life.rule", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-lifefile", (char *) ".life.lifefile", XrmoptionSepArg, (caddr_t) NULL},
	{(char *) "-serial", (char *) ".life.serial", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+serial", (char *) ".life.serial", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-vertical", (char *) ".life.vertical", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+vertical", (char *) ".life.vertical", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-glidersearch", (char *) ".life.glidersearch", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+glidersearch", (char *) ".life.glidersearch", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-patternsearch", (char *) ".life.patternsearch", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+patternsearch", (char *) ".life.patternsearch", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-nontot", (char *) ".life.nontot", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+nontot", (char *) ".life.nontot", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-conway", (char *) ".life.conway", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+conway", (char *) ".life.conway", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-highlife", (char *) ".life.highlife", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+highlife", (char *) ".life.highlife", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-daynight", (char *) ".life.daynight", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+daynight", (char *) ".life.daynight", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-callahan", (char *) ".life.callahan", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+callahan", (char *) ".life.callahan", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-andreen", (char *) ".life.andreen", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+andreen", (char *) ".life.andreen", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-bays", (char *) ".life.bays", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+bays", (char *) ".life.bays", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-trilife", (char *) ".life.trilife", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+trilife", (char *) ".life.trilife", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-trilife1", (char *) ".life.trilife1", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+trilife1", (char *) ".life.trilife1", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-trilife2", (char *) ".life.trilife2", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+trilife2", (char *) ".life.trilife2", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-pent", (char *) ".life.pent", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+pent", (char *) ".life.pent", XrmoptionNoArg, (caddr_t) "off"},
	{(char *) "-pent2", (char *) ".life.pent2", XrmoptionNoArg, (caddr_t) "on"},
	{(char *) "+pent2", (char *) ".life.pent2", XrmoptionNoArg, (caddr_t) "off"},
};
static argtype vars[] =
{
#ifdef LIFE_NAMES
	{(void *) & label, (char *) "label", (char *) "Label", (char *) DEF_LABEL, t_Bool},
#endif
	{(void *) & draw, (char *) "draw", (char *) "Draw", (char *) DEF_DRAW, t_Bool},
	{(void *) & neighbors, (char *) "neighbors", (char *) "Neighbors", (char *) DEF_NEIGHBORS, t_Int},
	{(void *) & repeat, (char *) "repeat", (char *) "Repeat", (char *) DEF_REPEAT, t_Int},
	{(void *) & rule, (char *) "rule", (char *) "Rule", (char *) DEF_RULE, t_String},
	{(void *) & lifefile, (char *) "lifefile", (char *) "LifeFile", (char *) "", t_String},
	{(void *) & serial, (char *) "serial", (char *) "Serial", (char *) DEF_SERIAL, t_Bool},
	{(void *) & vertical, (char *) "vertical", (char *) "Vertical", (char *) DEF_VERTICAL, t_Bool},
	{(void *) & glidersearch, (char *) "glidersearch", (char *) "GliderSearch", (char *) DEF_GLIDERSEARCH, t_Bool},
	{(void *) & patternsearch, (char *) "patternsearch", (char *) "PatternSearch", (char *) DEF_PATTERNSEARCH, t_Bool},
	{(void *) & nontot, (char *) "nontot", (char *) "Nontot", (char *) DEF_NONTOT, t_Bool},
	{(void *) & conway, (char *) "conway", (char *) "Conway", (char *) DEF_CONWAY, t_Bool},
	{(void *) & highlife, (char *) "highlife", (char *) "HighLife", (char *) DEF_HIGHLIFE, t_Bool},
	{(void *) & daynight, (char *) "daynight", (char *) "DayNight", (char *) DEF_DAY_NIGHT, t_Bool},
	{(void *) & callahan, (char *) "callahan", (char *) "Callahan", (char *) DEF_CALLAHAN, t_Bool},
	{(void *) & andreen, (char *) "andreen", (char *) "Andreen", (char *) DEF_ANDREEN, t_Bool},
	{(void *) & bays, (char *) "bays", (char *) "Bays", (char *) DEF_BAYS, t_Bool},
	{(void *) & trilife, (char *) "trilife", (char *) "TriLife", (char *) DEF_TRILIFE, t_Bool},
	{(void *) & trilife1, (char *) "trilife1", (char *) "TriLife1", (char *) DEF_TRILIFE1, t_Bool},
	{(void *) & trilife2, (char *) "trilife2", (char *) "TriLife2", (char *) DEF_TRILIFE2, t_Bool},
	{(void *) & pent, (char *) "pent", (char *) "Pent", (char *) DEF_PENT, t_Bool},
	{(void *) & pent2, (char *) "pent2", (char *) "Pent2", (char *) DEF_PENT2, t_Bool},
};
static OptionStruct desc[] =
{
#ifdef LIFE_NAMES
	{(char *) "-/+label", (char *) "turn on/off name labeling"},
#endif
	{(char *) "-/+draw", (char *) "turn on/off drawing to speed search"},
	{(char *) "-neighbors num", (char *) "squares 4 or 8, hexagons 6, triangles 3, 9 or 12, pentagons 5 or 7"},
	{(char *) "-repeat num", (char *) "repeat for period to exclude in search"}, /* this is not as good as Bays' signature idea */
	{(char *) "-rule string", (char *) "B<birth_neighborhood/S<survival_neighborhood> parameters"},
	{(char *) "-lifefile file", (char *) "life file"},
	{(char *) "-/+serial", (char *) "turn on/off picking of sequential patterns"},
	{(char *) "-/+vertical", (char *) "change orientation for hexagons and triangles"},
	{(char *) "-/+glidersearch", (char *) "search for gliders"},
	{(char *) "-/+patternsearch", (char *) "search for patterns"},
	{(char *) "-/+nontot", (char *) "turn on/off nontotalistic rule B3/S2a2b2c2d3 (known as Rule 3/2ab3)"},
	{(char *) "-/+conway", (char *) "turn on/off Conway's original Life rule B3/S23"},
	{(char *) "-/+highlife", (char *) "turn on/off Bell's HighLife rule B36/S23"},
	{(char *) "-/+daynight", (char *) "turn on/off Thompson's Day and Night rule B3678/S34678"},
	{(char *) "-/+callahan", (char *) "turn on/off Callahan's hex rule B2a/S2b34"},
	{(char *) "-/+andreen", (char *) "turn on/off Andreen's hex rule B2a3a4b/S2a2b4a"},
	{(char *) "-/+bays", (char *) "turn on/off Bays' hex rule B2/S35"},
	{(char *) "-/+trilife", (char *) "turn on/off Bays' tri rule B45/S34"},
	{(char *) "-/+trilife1", (char *) "turn on/off Bays' tri rule B456/S45"},
	{(char *) "-/+trilife2", (char *) "turn on/off Bays' tri rule B45/S23"},
	{(char *) "-/+pent", (char *) "turn on/off Bagley's pentagon rule B2/S24"},
	{(char *) "-/+pent2", (char *) "turn on/off Bays' pentagon rule B346/S23"},
};

ModeSpecOpt life_opts =
{sizeof opts / sizeof opts[0], opts, sizeof vars / sizeof vars[0], vars, desc};

#ifdef USE_MODULES
ModStruct   life_description =
{"life", "init_life", "draw_life", "release_life",
 "refresh_life", "change_life", (char *) NULL, &life_opts,
 750000, 40, 140, 0, 64, 1.0, "",
 "Shows Conway's game of Life", 0, NULL};

#endif

/* aliases for vars defined in the bitmap file */
#define CELL_WIDTH   image_width
#define CELL_HEIGHT    image_height
#define CELL_BITS    image_bits

#include "life.xbm"
#ifdef XBM_GRELB
#include "life2.xbm"
#define CELL2_WIDTH   image2_width
#define CELL2_HEIGHT  image2_height
#define CELL2_BITS    image2_bits
static XImage bimage =
{
  0, 0, 0, XYBitmap, 0, LSBFirst, 8, LSBFirst, 8, 1
};
#endif

#ifdef HAVE_XPM
#define CELL_NAME image_name
#if 1
static char *image_name[] =
{(char *) ""};
#else
/* Kind of boring... */
#include "life.xpm"
#endif
#define DEFAULT_XPM 0
#endif

#define REDRAWSTEP 2000		/* How many cells to draw per cycle */
#define MINGRIDSIZE 20
#define MINSIZE 4
#define DEAD 0
#define LIVE 1
#define STATES 2
#define SOUPPERCENT 30
#define SOUPSIZE(s)  MAX(s/4,10)
#define SetListFromGet(c,r) if (!setcell(mi,c,r,LIVE)) return
/* Triangles are stored left/right not up/down */
#define SetList(c,r) if (lp->polygon == 3) { \
		if (!setcell(mi,r,c,LIVE)) return; \
	} else \
		if (!setcell(mi,c,r,LIVE)) return

typedef struct {
	long position;
	unsigned short age;
	unsigned char state;
	unsigned char toggle;
} cellstruct;

/* Doubly linked list */
typedef struct _CellList {
	cellstruct  info;
	struct _CellList *previous, *next;
} CellList;

typedef struct {
	Bool        painted, vertical;
	paramstruct param;
	int         pattern, patterned_rule;
	int         pixelmode;
	int         generation;
	int         xs, ys, xb, yb;	/* cell size, grid border */
	int         nrows, ncols, npositions;
	int         width, height;
	int         state;
	int         noChangeCount;
	int         redrawing, redrawpos;
	int         ncells[STATES];
	CellList   *last[STATES], *first[STATES];
	CellList  **arr;
	union {
		XPoint      hexagon[6];
		XPoint      triangle[2][3];
		XPoint      pentagon[4][5];
	} shape;
	XImage     *logo;
#ifdef XBM_GRELB
	XImage     *logo2;
#endif
	Colormap    cmap;
	unsigned long black;
	int         graphics_format;
	GC          backGC;
	int         neighbors, polygon;
	int         nontot, conway, highlife, daynight, callahan, andreen, bays;
	int         trilife, trilife1, trilife2, pent, pent2;
	int         allPatterns, allGliders;
	paramstruct input_param;
	int         labelOffsetX, labelOffsetY;
	Bool        foundBirth, foundSurvival;
	char        ruleString[80], nameString[80];
} lifestruct;

static lifestruct *lifes = (lifestruct *) NULL;

static char *filePattern = (char *) NULL;

static int
invplot(int local_neighbors)
{
	switch (local_neighbors) {
	case 3:
		return 0;
	case 4:
		return 1;
	case 5:
		return 2;
	case 6:
		return 3;
	case 7:
		return 4;
	case 8:
		return 5;
	case 9:
		return 6;
	case 12:
		return 7;
	default:
		(void) fprintf(stderr, "no neighborhood like %d known\n",
				local_neighbors);
			return 3;
	}
}

static int
codeToPatternedRule(int local_neighbors, paramstruct param)
{
	unsigned int i;
	int g, neighborKind;

	neighborKind = invplot(local_neighbors);
	switch (local_neighbors) {
	case 5:
		for (i = 0; i < LIFE_5RULES; i++)
			if (param_5rules[i].birth == param.birth &&
			    param_5rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_5rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_5rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_5RULES;
	case 6:
		for (i = 0; i < LIFE_6RULES; i++)
			if (param_6rules[i].birth == param.birth &&
			    param_6rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_6rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_6rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_6RULES;
	case 7:
		for (i = 0; i < LIFE_7RULES; i++)
			if (param_7rules[i].birth == param.birth &&
			    param_7rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_7rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_7rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_7RULES;
	case 8:
		for (i = 0; i < LIFE_8RULES; i++)
			if (param_8rules[i].birth == param.birth &&
			    param_8rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_8rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_8rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_8RULES;
	case 9:
		for (i = 0; i < LIFE_9RULES; i++)
			if (param_9rules[i].birth == param.birth &&
			    param_9rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_9rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_9rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_9RULES;
	case 12:
		for (i = 0; i < LIFE_12RULES; i++)
			if (param_12rules[i].birth == param.birth &&
			    param_12rules[i].survival == param.survival) {
				for (g = 0; g < maxgroups[neighborKind]; g++) {
					if (param_12rules[i].birthGroup[g] !=
					    param.birthGroup[g] ||
					    param_12rules[i].survivalGroup[g] !=
					    param.survivalGroup[g]) {
						break;
					}
				}
				if (g == maxgroups[neighborKind])
					return i;
			}
		return LIFE_12RULES;
	}
	return 0;
}

static void
copyFromPatternedRule(int local_neighbors, paramstruct * param,
		int patterned_rule)
{
	int i, neighborKind;

	neighborKind = invplot(local_neighbors);
	switch (local_neighbors) {
	case 5:
		param->birth = param_5rules[patterned_rule].birth;
		param->survival = param_5rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_5rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_5rules[patterned_rule].survivalGroup[i];
		}
		break;
	case 6:
		param->birth = param_6rules[patterned_rule].birth;
		param->survival = param_6rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_6rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_6rules[patterned_rule].survivalGroup[i];
		}
		break;
	case 7:
		param->birth = param_7rules[patterned_rule].birth;
		param->survival = param_7rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_7rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_7rules[patterned_rule].survivalGroup[i];
		}
		break;
	case 8:
		param->birth = param_8rules[patterned_rule].birth;
		param->survival = param_8rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_8rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_8rules[patterned_rule].survivalGroup[i];
		}
		break;
	case 9:
		param->birth = param_9rules[patterned_rule].birth;
		param->survival = param_9rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_9rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_9rules[patterned_rule].survivalGroup[i];
		}
		break;
	case 12:
		param->birth = param_12rules[patterned_rule].birth;
		param->survival = param_12rules[patterned_rule].survival;
		for (i = 0; i < maxgroups[neighborKind]; i++) {
			param->birthGroup[i] =
				param_12rules[patterned_rule].birthGroup[i];
			param->survivalGroup[i] =
				param_12rules[patterned_rule].survivalGroup[i];
		}
		break;
	}
}

static void
printRule(int local_neighbors, char * string, paramstruct param, Bool verbose)
{
	int i = 1, l, g, neighborKind;
	Bool found;

	string[0] = 'B';
	if (verbose)
		(void) fprintf(stdout,
			"rule (Birth/Survival %d neighborhood): ",
			local_neighbors);
	neighborKind = invplot(local_neighbors);
	for (l = 0; l <= local_neighbors && l < 10; l++) {
		if ((param.birth & (1 << l)) != 0) {
			(void) sprintf(&(string[i]), "%d", l);
			i++;
		} else if (l >= FIRSTGROUP && l < FIRSTGROUP + maxgroups[neighborKind]) {
			for (g = 0; g < groupnumber[neighborKind][l - FIRSTGROUP]; g++) {
				if (param.birthGroup[l - FIRSTGROUP] & (1 << g)) {
					(void) sprintf(&(string[i]), "%d%c",
						l, 'a' + g);
					i += 2;
				}
			}
		}
	}
	(void) sprintf(&(string[i]), "/S");
	i += 2;
	for (l = 0; l <= local_neighbors && l < 10; l++) {
		if ((param.survival & (1 << l)) != 0) {
			(void) sprintf(&(string[i]), "%d", l);
			i++;
		} else if (l >= FIRSTGROUP && l < FIRSTGROUP + maxgroups[neighborKind]) {
			for (g = 0; g < groupnumber[neighborKind][l - FIRSTGROUP]; g++) {
				if (param.survivalGroup[l - FIRSTGROUP] & (1 << g)) {
					(void) sprintf(&(string[i]), "%d%c",
						l, 'a' + g);
					i += 2;
				}
			}
		}
	}
	string[i] = '\0';
	found = False;
	for (l = 0; l < maxgroups[neighborKind]; l++) {
		if (param.birthGroup[l] || param.survivalGroup[l]) {
			found = True;
			break;
		}
	}
	if (verbose) {
		(void) fprintf(stdout, "%s\nbinary rule%s: Birth 0x%X, Survival 0x%X\n",
			string, ((found) ? " (nongroup)" : ""),
			param.birth, param.survival);
		if (found) {
			for (l = 0; l < maxgroups[neighborKind]; l++) {
				(void) fprintf(stdout,
				       "groups in neighborhood %d: Birth 0x%X, Survival 0x%X\n",
				       l + FIRSTGROUP, param.birthGroup[l], param.survivalGroup[l]);
			}
		}
	}
}

static int
positionOfNeighbor(lifestruct * lp, int n, int col, int row)
{
	int dir = n * (360 / lp->neighbors);

	if (lp->polygon == 4 || lp->polygon == 6) {
		switch (dir) {
		case 0:
			col = (col + 1 == lp->ncols) ? 0 : col + 1;
			break;
		case 45:
			col = (col + 1 == lp->ncols) ? 0 : col + 1;
			row = (!row) ? lp->nrows - 1 : row - 1;
			break;
		case 60:
			if (!(row & 1))
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
			row = (!row) ? lp->nrows - 1 : row - 1;
			break;
		case 90:
			row = (!row) ? lp->nrows - 1 : row - 1;
			break;
		case 120:
			if (row & 1)
				col = (!col) ? lp->ncols - 1 : col - 1;
			row = (!row) ? lp->nrows - 1 : row - 1;
			break;
		case 135:
			col = (!col) ? lp->ncols - 1 : col - 1;
			row = (!row) ? lp->nrows - 1 : row - 1;
			break;
		case 180:
			col = (!col) ? lp->ncols - 1 : col - 1;
			break;
		case 225:
			col = (!col) ? lp->ncols - 1 : col - 1;
			row = (row + 1 == lp->nrows) ? 0 : row + 1;
			break;
		case 240:
			if (row & 1)
				col = (!col) ? lp->ncols - 1 : col - 1;
			row = (row + 1 == lp->nrows) ? 0 : row + 1;
			break;
		case 270:
			row = (row + 1 == lp->nrows) ? 0 : row + 1;
			break;
		case 300:
			if (!(row & 1))
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
			row = (row + 1 == lp->nrows) ? 0 : row + 1;
			break;
		case 315:
			col = (col + 1 == lp->ncols) ? 0 : col + 1;
			row = (row + 1 == lp->nrows) ? 0 : row + 1;
			break;
		default:
			(void) fprintf(stderr, "wrong direction %d\n", dir);
		}
	} else if (lp->polygon == 3) {
		if ((col + row) & 1) {	/* right */
			switch (dir) {
			case 0:
				col = (!col) ? lp->ncols - 1 : col - 1;
				break;
			case 30:
			case 40:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 60:
				col = (!col) ? lp->ncols - 1 : col - 1;
				if (row + 1 == lp->nrows)
					row = 1;
				else if (row + 2 == lp->nrows)
					row = 0;
				else
					row = row + 2;
				break;
			case 80:
			case 90:
				if (row + 1 == lp->nrows)
					row = 1;
				else if (row + 2 == lp->nrows)
					row = 0;
				else
					row = row + 2;
				break;
			case 120:
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 150:
			case 160:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 180:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				break;
			case 200:
			case 210:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 240:
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 270:
			case 280:
				if (!row)
					row = lp->nrows - 2;
				else if (!(row - 1))
					row = lp->nrows - 1;
				else
					row = row - 2;
				break;
			case 300:
				col = (!col) ? lp->ncols - 1 : col - 1;
				if (!row)
					row = lp->nrows - 2;
				else if (!(row - 1))
					row = lp->nrows - 1;
				else
					row = row - 2;
				break;
			case 320:
			case 330:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
		} else {	/* left */
			switch (dir) {
			case 0:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				break;
			case 30:
			case 40:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 60:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				if (!row)
					row = lp->nrows - 2;
				else if (row == 1)
					row = lp->nrows - 1;
				else
					row = row - 2;
				break;
			case 80:
			case 90:
				if (!row)
					row = lp->nrows - 2;
				else if (row == 1)
					row = lp->nrows - 1;
				else
					row = row - 2;
				break;
			case 120:
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 150:
			case 160:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (!row) ? lp->nrows - 1 : row - 1;
				break;
			case 180:
				col = (!col) ? lp->ncols - 1 : col - 1;
				break;
			case 200:
			case 210:
				col = (!col) ? lp->ncols - 1 : col - 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 240:
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 270:
			case 280:
				if (row + 1 == lp->nrows)
					row = 1;
				else if (row + 2 == lp->nrows)
					row = 0;
				else
					row = row + 2;
				break;
			case 300:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				if (row + 1 == lp->nrows)
					row = 1;
				else if (row + 2 == lp->nrows)
					row = 0;
				else
					row = row + 2;
				break;
			case 320:
			case 330:
				col = (col + 1 == lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
		}
	} else {
		int orient = ((row & 1) * 2 + col) % 4;
		switch (orient) { /* up, down, left, right */
		case 0:
			switch (dir) {
			case 0:
				col++;
				break;
			case 51: /* 5 */
			case 72: /* 7 */
				col = (col + 2 >= lp->ncols) ? 0 : col + 2;
				break;
			case 102: /* 7 corner */
				col = (col + 3 >= lp->ncols) ? 1 : col + 3;
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 144: /* 5 */
			case 153: /* 7 */
				col++;
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 204: /* 5 */
			case 216: /* 7 */
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 255: /* 7 */
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 288: /* 5 */
			case 306: /* 7 */
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
			break;
		case 1:
			switch (dir) {
			case 0:
				col--;
				break;
			case 51: /* 7 */
			case 72: /* 5 */
				col = (col == 1) ? lp->ncols - 1 : col - 2;
				break;
			case 102: /* 7 */
				col = (col == 1) ? lp->ncols - 2 : col - 3;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 144: /* 7 */
			case 153: /* 5 */
				col--;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 204: /* 7 */
			case 216: /* 5 */
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 255: /* 7 */
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 288: /* 7 */
			case 306: /* 5 */
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
			break;
		case 2:
			switch (dir) {
			case 0:
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				break;
			case 51: /* 7 */
			case 72: /* 5 */
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				col++;
				break;
			case 102: /* 7 */
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 144: /* 5 */
			case 153: /* 7 */
				col = (col == 0) ? lp->ncols - 2 : col - 2;
				break;
			case 204: /* 7 */
			case 216: /* 5 */
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				break;
			case 255: /* 7 */
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				break;
			case 288: /* 5 */
			case 306: /* 7 */
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
			break;
		case 3:
			switch (dir) {
			case 0:
				col--;
				break;
			case 51: /* 7 */
			case 72: /* 5 */
				col = (col == 0) ? lp->ncols - 1 : col - 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 102: /* 7 */
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				row = (row + 1 == lp->nrows) ? 0 : row + 1;
				break;
			case 144: /* 5 */
			case 153: /* 7 */
				col = (col + 2 >= lp->ncols) ? 1 : col + 2;
				break;
			case 204: /* 7 */
			case 216: /* 5 */
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				break;
			case 255: /* 7 */
				col = (col + 1 >= lp->ncols) ? 0 : col + 1;
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			case 288: /* 5 */
			case 306: /* 7 */
				row = (row == 0) ? lp->nrows - 1 : row - 1;
				break;
			default:
				(void) fprintf(stderr, "wrong direction %d\n",
					dir);
			}
			break;
		default:
			(void) fprintf(stderr, "wrong orient %d\n",
				orient);
		}
	}
	return (row * lp->ncols + col);
}

static void
parseRule(ModeInfo * mi, char * string)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int n, g = 0, l = 0, neighborKind, lifeRule = 0;
	char serving = 0;

	if (lp->foundBirth && lp->foundSurvival)
		return;
	lp->foundBirth = lp->foundSurvival = False;
	lp->input_param.birth = lp->input_param.survival = 0;
	for (n = 0; n < MAXGROUPS; n++) {
		lp->input_param.birthGroup[n] =
			lp->input_param.survivalGroup[n] = 0;
	}
	/* 8 is first so it will default to this */
	if (lp->nontot || lp->conway || lp->highlife || lp->daynight) {
		lp->neighbors = 8;
		lp->polygon = 4;
		neighborKind = invplot(lp->neighbors);
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		if (lp->nontot)
			lifeRule = LIFE_8B3S2a2b2c2d3;
		else if (lp->conway)
			lifeRule = LIFE_8B3S23;
		else if (lp->highlife)
			lifeRule = LIFE_8B36S23;
		else if (lp->daynight)
			lifeRule = LIFE_8B3678S34678;
		lp->input_param.birth = param_8rules[lifeRule].birth;
		lp->input_param.survival = param_8rules[lifeRule].survival;
		for (n = 0; n < maxgroups[neighborKind]; n++) {
			lp->input_param.birthGroup[n] =
				param_8rules[lifeRule].birthGroup[n];
			lp->input_param.survivalGroup[n] =
				param_8rules[lifeRule].survivalGroup[n];
		}
		return;
	} else if (lp->callahan || lp->andreen || lp->bays) {
		lp->neighbors = 6;
		lp->polygon = 6;
		neighborKind = invplot(lp->neighbors);
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		if (lp->callahan)
			lifeRule = LIFE_6B2aS2b34;
		else if (lp->andreen)
			lifeRule = LIFE_6B2a3a4bS2a2b4a;
		else if (lp->bays)
			lifeRule = LIFE_6B2S35;
		lp->input_param.birth = param_6rules[lifeRule].birth;
		lp->input_param.survival = param_6rules[lifeRule].survival;
		for (n = 0; n < maxgroups[neighborKind]; n++) {
			lp->input_param.birthGroup[n] =
				param_6rules[lifeRule].birthGroup[n];
			lp->input_param.survivalGroup[n] =
				param_6rules[lifeRule].survivalGroup[n];
		}
		return;
	} else if (lp->pent) {
		lp->neighbors = 5;
		lp->polygon = 5;
		neighborKind = invplot(lp->neighbors);
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		lifeRule = LIFE_5B2S24;
		lp->input_param.birth = param_5rules[lifeRule].birth;
		lp->input_param.survival = param_5rules[lifeRule].survival;
		for (n = 0; n < maxgroups[neighborKind]; n++) {
			lp->input_param.birthGroup[n] =
				param_5rules[lifeRule].birthGroup[n];
			lp->input_param.survivalGroup[n] =
				param_5rules[lifeRule].survivalGroup[n];
		}
		return;
	} else if (lp->pent2) {
		lp->neighbors = 7;
		lp->polygon = 5;
		neighborKind = invplot(lp->neighbors);
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		lifeRule = LIFE_7B346S23;
		lp->input_param.birth = param_7rules[lifeRule].birth;
		lp->input_param.survival = param_7rules[lifeRule].survival;
		for (n = 0; n < maxgroups[neighborKind]; n++) {
			lp->input_param.birthGroup[n] =
				param_7rules[lifeRule].birthGroup[n];
			lp->input_param.survivalGroup[n] =
				param_7rules[lifeRule].survivalGroup[n];
		}
		return;
	} else if (lp->trilife || lp->trilife1 || lp->trilife2) {
		lp->neighbors = 12;
		lp->polygon = 3;
		neighborKind = invplot(lp->neighbors);
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		if (lp->trilife)
			lifeRule = LIFE_12B45S34;
		else if (lp->trilife1)
			lifeRule = LIFE_12B456S45;
		else if (lp->trilife2)
			lifeRule = LIFE_12B45S23;
		lp->input_param.birth = param_12rules[lifeRule].birth;
		lp->input_param.survival = param_12rules[lifeRule].survival;
		for (n = 0; n < maxgroups[neighborKind]; n++) {
			lp->input_param.birthGroup[n] =
				param_12rules[lifeRule].birthGroup[n];
			lp->input_param.survivalGroup[n] =
				param_12rules[lifeRule].survivalGroup[n];
		}
		return;
	}
	if (!lp->neighbors) {
		for (n = 0; n < NEIGHBORKINDS; n++) {
			if (neighbors == plots[n]) {
				lp->neighbors = neighbors;
				break;
			}
			if (n == NEIGHBORKINDS - 1) {
#if 0
				lp->neighbors = plots[NRAND(NEIGHBORKINDS)];
				lp->neighbors = (LRAND() & 1) ? 4 : 8;
				lp->polygon = 4;
#else
				lp->neighbors = 8;
				lp->polygon = 4;
#endif
				break;
			}
		}
	}
	if (lp->neighbors == 6) {
		lp->polygon = 6;
	} else if (lp->neighbors % 3 == 0) { /* TRI */
		lp->polygon = 3;
	} else if (lp->neighbors == 5 || lp->neighbors == 7) {
		lp->polygon = 5;
	} else /* if (lp->neighbors == 4 || lp->neighbors == 8) */ {
		lp->polygon = 4;
	}
	neighborKind = invplot(lp->neighbors);
	if (rule) {
		n = 0;
		while (rule[n]) {
			if (rule[n] == 'P' || rule[n] == 'p') {
				lp->allPatterns = True;
				lp->foundBirth = lp->foundSurvival = True;
				if (MI_IS_VERBOSE(mi))
					(void) fprintf(stdout,
						"rule: All rules with known patterns\n");
				return;
			} else if (rule[n] == 'G' || rule[n] == 'g') {
				lp->allGliders = True;
				lp->foundBirth = lp->foundSurvival = True;
				if (MI_IS_VERBOSE(mi))
					(void) fprintf(stdout,
						"rule: All rules with known gliders\n");
				return;
			} else if (rule[n] == 'B' || rule[n] == 'F' || rule[n] == 'K' ||
					rule[n] == 'f' || rule[n] == 'k') {
				/* 'b'  is used in groups */
				serving = 'B';
			} else if (rule[n] == 'S' || rule[n] == 'E' || rule[n] == 'L' ||
					rule[n] == 's' || rule[n] == 'e' || rule[n] == 'l') {
				serving = 'S';
			} else {
				l = rule[n] - '0';
				if (l >= 0 && l <= 9 && l <= lp->neighbors) { /* no 10, 11, 12 */
					g = rule[n + 1] - 'a';
					if (l >= FIRSTGROUP && l < FIRSTGROUP + maxgroups[neighborKind] &&
					    g >= 0 && g < groupnumber[neighborKind][l - FIRSTGROUP]) { /* Groupings */
						if (serving == 'B') {
							lp->foundBirth = True;
							lp->input_param.birthGroup[l - FIRSTGROUP] |= (1 << g);
						} else if (serving == 'S') {
							lp->foundSurvival = True;
							lp->input_param.survivalGroup[l - FIRSTGROUP] |= (1 << g);
						}
					} else {
						if (serving == 'B') {
							lp->foundBirth = True;
							lp->input_param.birth |= (1 << l);
						} else if (serving == 'S') {
							lp->foundSurvival = True;
							lp->input_param.survival |= (1 << l);
						}
					}
				}
			}
			n++;
		}
	}
	if (!lp->foundBirth || !lp->foundSurvival) {
		/* Default to known interesting rules if rule does not make sense */
		lp->allGliders = True;
		lp->foundBirth = lp->foundSurvival = !MI_IS_FULLRANDOM(mi);
		if (MI_IS_VERBOSE(mi))
			(void) fprintf(stdout,
				"rule: Defaulting to all rules with known gliders\n");
		return;
	}
}

static void
parseFile(ModeInfo *mi)
{
	FILE *file;
	static Bool done = False;
	int firstx, x = 0, y = 0, i = 0;
	int l;
	int found = 0;
	int c, size;
	char line[256];
#define LIVE_CHARS 8
	char liveChar[LIVE_CHARS] = {'*', '0', 'O', 'o', 'A', 'V', 'J', 'D'};

	if (done)
		return;
	done = True;
	if (MI_IS_FULLRANDOM(mi) || !lifefile || !*lifefile)
		return;
	if ((file = my_fopenSize(lifefile, "r", &size)) == NULL) {
		(void) fprintf(stderr, "could not read file \"%s\"\n", lifefile);
		return;
	}
	for (;;) {
		if (!fgets(line, 256, file)) {
			(void) fprintf(stderr, "could not read header of file \"%s\"\n",
				lifefile);
			(void) fclose(file);
			return;
		}
		if (strncmp(line, "#P", (size_t) 2) == 0 &&
		    sscanf(line, "#P %d %d", &x, &y) == 2)
			break;
	}
	c = getc(file);
	while (c != EOF && !(c == '.')) {
		found = 0;
		for (l = 0; l < LIVE_CHARS; l++) {
			if (c == liveChar[l]) {
				found = 1;
				break;
			}
		}
		if (found)
			break;
		c = getc(file);
	}
	if (c == EOF || x <= -127 || y <= -127 || x >= 127 || y >= 127) {
		(void) fprintf(stderr, "corrupt file \"%s\" or file to large\n",
			lifefile);
		(void) fclose(file);
		return;
	}
	firstx = x;
	if ((filePattern = (char *) malloc((2 * size) *
			sizeof (char))) == NULL) {
		(void) fprintf(stderr, "not enough memory\n");
		(void) fclose(file);
		return;
	}

	while (c != EOF && x < 127 && y < 127 && i < 2 * size) {
		found = 0;
		for (l = 0; l < LIVE_CHARS; l++) {
			if (c == liveChar[l]) {
				filePattern[i++] = x++;
				filePattern[i++] = y;
				found = 1;
				break;
			}
		}
		if (!found) {
			if (c == '.') {
				x++;
			} else if (c == '\n') {
				x = firstx;
				y++;
			}
		}
		c = getc(file);
	}
	(void) fclose(file);
	filePattern[i] = 127;
}

static Bool
initList(lifestruct * lp, int state)
{
	/* Waste some space at the beginning and end of list
	   so we do not have to complicated checks against falling off the ends. */
	if ((lp->last[state] = (CellList *) malloc(sizeof (CellList))) == NULL) {
		return False;
	}
	if ((lp->first[state] = (CellList *) malloc(sizeof (CellList))) == NULL) {
		free(lp->last[state]);
		lp->last[state] = (CellList *) NULL;
		return False;
	}
	lp->first[state]->previous = lp->last[state]->next =
		 (struct _CellList *) NULL;
	lp->first[state]->next = lp->last[state]->previous =
		 (struct _CellList *) NULL;
	lp->first[state]->next = lp->last[state];
	lp->last[state]->previous = lp->first[state];
	return True;
}

static Bool
addToList(lifestruct * lp, int state, cellstruct info)
{
	CellList *curr;

	if ((curr = (CellList *) malloc(sizeof (CellList))) == NULL)
		return False;
	lp->last[state]->previous->next = curr;
	curr->previous = lp->last[state]->previous;
	curr->next = lp->last[state];
	lp->last[state]->previous = curr;
	curr->info = info;
	if (info.position >= 0) {
		lp->arr[info.position] = curr;
		lp->ncells[state]++;
	}
	return True;
}

static void
removeFromList(lifestruct * lp, int state, CellList * curr)
{
	curr->previous->next = curr->next;
	curr->next->previous = curr->previous;
	if (curr->info.position >= 0) {
		lp->arr[curr->info.position] = (CellList *) NULL;
		lp->ncells[state]--;
	}
	free(curr);
}

#ifdef DEBUG
static void
printState(ModeInfo * mi, int state)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	CellList *curr;
	int i = 0;

	curr = lp->first[state]->next;
	(void) printf("state %d\n", state);
	while (curr != lp->last[state]) {
		(void) printf("%d: position %ld,	age %d, state %d, toggle %d\n",
			i, curr->info.position, curr->info.age,
			curr->info.state, curr->info.toggle);
		curr = curr->next;
		i++;
	}
}

#endif

static void
printList(lifestruct * lp, int i)
{
	/* did not make this hack multiscreen ready */
	static int tcol[2], trow[2], tcol2[2], trow2[2], tcount[2];

	tcol[i] = 0;
	trow[i] = 0;
	tcol2[i] = 0;
	trow2[i] = 0;
	tcount[i] = lp->ncells[LIVE];
	if ((glidersearch || patternsearch) && tcount[0] < 72) {
		CellList *curr = lp->last[LIVE]->previous;

		while (curr != lp->first[LIVE]) {
			int col, row;

			if (lp->polygon == 4) {

				col = (curr->info.position % lp->ncols) - (lp->ncols / 2);
				row = (curr->info.position / lp->ncols) - (lp->nrows / 2);
			} else if (lp->polygon == 6) {
				col = (curr->info.position % lp->ncols) - 2 * (lp->ncols / 4);
				row = (curr->info.position / lp->ncols) - 2 * (lp->nrows / 4) - 1;
				if (row < 0)
					col += ((lp->nrows / 2) & 1) ? row / 2 :
						(row - 1) / 2;
				else
					col += ((lp->nrows / 2) & 1) ? (row + 1) / 2 :
						row / 2;
			} else {
				col = (curr->info.position % lp->ncols) - 2 * (lp->ncols / 4) - 1;
				row = (curr->info.position / lp->ncols) - 2 * (lp->nrows / 4);
			}
			(void) printf("%d, %d,\n", col, row);
			/* fudging a signature */
			tcol[i] += col;
			trow[i] += row;
			tcol2[i] += col * col;
			trow2[i] += row * row;
			curr = curr->previous;
		}
		(void) printf("%d#%d, pos_sum %d %d %d %d\n",
			i, tcount[i], tcol[i], trow[i], tcol2[i], trow2[i]);
		if (i == 1 && (tcount[0] <= NUMPTS || tcount[1] <= NUMPTS) &&
				(tcount[0] != tcount[1] ||
				tcol[0] != tcol[1] || trow[0] != trow[1] ||
				tcol2[0] != tcol2[1] || trow2[0] != trow2[1]))
			(void) printf("found\n");
	}
}

static void
flushList(lifestruct * lp, int state)
{
	while (lp->last[state]->previous != lp->first[state]) {
		CellList *curr = lp->last[state]->previous;

		curr->previous->next = lp->last[state];
		lp->last[state]->previous = curr->previous;
		free(curr);
	}
	lp->ncells[state] = 0;
}

/*static void
drawPolygon(Display *display, Window window, XPoint points, int size) {
	int i = 0;

	XSetForeground(display, gc, MI_BLACK_PIXEL(mi));
	for (i = 0; i < size - 1; i++) {
		XDrawLine(points[i].x, points[i].y,
			points[i + 1].x, points[i + 1].y);
	XDrawLine(points[size - 1].x, points[size - 1].y,
		points[0].x, points[0].y);
}*/

static void
drawCell(ModeInfo * mi, cellstruct info)
{
	Display *display = MI_DISPLAY(mi);
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	GC gc = lp->backGC;
	int col, row;

	if (!draw)
		return;
	col = (int) (info.position % lp->ncols);
	row = (int) (info.position / lp->ncols);
	if (info.state == LIVE) {
		if (MI_NPIXELS(mi) > 2)
			XSetForeground(display, gc, MI_PIXEL(mi, info.age));
		else
			XSetForeground(display, gc, MI_WHITE_PIXEL(mi));
	} else
		XSetForeground(display, gc, lp->black);

	if (lp->polygon == 3) {
		int orient = (col + row) & 1; /* O left 1 right */
#ifdef WIN32
		int offset = 2;
#else
		int offset = 1;
#endif

		if (lp->vertical) {
			lp->shape.triangle[orient][0].x = lp->xb + col * lp->xs;
			lp->shape.triangle[orient][0].y = lp->yb + row * lp->ys;
		} else {
			lp->shape.triangle[orient][0].y = lp->xb + col * lp->xs;
			lp->shape.triangle[orient][0].x = lp->yb + row * lp->ys;
		}
		if (lp->xs <= 3 || lp->ys <= 3)
			XDrawPoint(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
			((orient) ? -1 : 1) + lp->shape.triangle[orient][0].x,
				lp->shape.triangle[orient][0].y);
		else {
			if (lp->vertical) {
				if (orient)
					lp->shape.triangle[orient][0].x += (lp->xs / 2 - offset);
				else
					lp->shape.triangle[orient][0].x -= (lp->xs / 2 - offset);

			} else {
				if (orient)
					lp->shape.triangle[orient][0].y += (lp->xs / 2 - offset);
				else
					lp->shape.triangle[orient][0].y -= (lp->xs / 2 - offset);
			}
			XFillPolygon(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
				lp->shape.triangle[orient], 3, Convex, CoordModePrevious);
			/*drawPolygon(MI_DISPLAY(mi), MI_WINDOW(mi),
				lp->shape.triangle[orient], 3);*/
		}
	} else if (lp->polygon == 4) {
		if (lp->pixelmode || info.state == DEAD)
			XFillRectangle(display, MI_WINDOW(mi), gc,
				lp->xb + lp->xs * col, lp->yb + lp->ys * row,
				lp->xs - (lp->xs > 3 && lp->pixelmode),
				lp->ys - (lp->ys > 3 && lp->pixelmode));
		else {
/*-
 * PURIFY 4.0.1 on SunOS4 and on Solaris 2 reports a 132 byte memory leak on
 * the next line */
#ifdef XBM_GRELB
		    if (lp->logo2) {
			(void) XPutImage(display, MI_WINDOW(mi), gc,
				(LRAND() & 1) ? lp->logo : lp->logo2,
			  0, 0, lp->xb + lp->xs * col, lp->yb + lp->ys * row,
					 lp->logo->width, lp->logo->height);
		    } else
#endif
		    {
			(void) XPutImage(display, MI_WINDOW(mi), gc, lp->logo,
			  0, 0, lp->xb + lp->xs * col, lp->yb + lp->ys * row,
					 lp->logo->width, lp->logo->height);
		    }
		}
	} else if (lp->polygon == 5) {
		int map[4] = {2, 0, 1, 3};
		int orient = ((row & 1) * 2 + col) % 4;
		int offsetX = 0, offsetY = 0;
		switch (orient) {
		case 0: /* up */
			offsetX = lp->xs;
			break;
		case 1: /* down */
			offsetY = 3 * lp->ys / 2 - 1;
			break;
		case 2: /* left */
			offsetX = -lp->xs / 2;
			offsetY = 3 * lp->ys / 4;
			break;
		case 3: /* right */
			offsetX = 3 * lp->xs / 2 - 1;
			offsetY = 3 * lp->ys / 4;
			break;
		default:
			(void) printf("wrong orient %d\n", orient);
		}
		orient = map[orient];
		lp->shape.pentagon[orient][0].x =  lp->xb +
			col * lp->xs + offsetX;
		lp->shape.pentagon[orient][0].y = lp->yb +
			row * lp->ys + offsetY;
		XFillPolygon(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
			lp->shape.pentagon[orient], 5, Convex, CoordModePrevious);
		/*drawPolygon(MI_DISPLAY(mi), MI_WINDOW(mi),
			lp->shape.pentagon[orient], 5);*/
	} else if (lp->polygon == 6) {
		int ccol = 2 * col + !(row & 1), crow = 2 * row;

		if (lp->vertical) {
			lp->shape.hexagon[0].x = lp->xb + ccol * lp->xs;
			lp->shape.hexagon[0].y = lp->yb + crow * lp->ys;
		} else {
			lp->shape.hexagon[0].y = lp->xb + ccol * lp->xs;
			lp->shape.hexagon[0].x = lp->yb + crow * lp->ys;
		}
		if (lp->xs == 1 && lp->ys == 1)
			XDrawPoint(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
				lp->shape.hexagon[0].x, lp->shape.hexagon[0].y);
		else {
			XFillPolygon(MI_DISPLAY(mi), MI_WINDOW(mi), lp->backGC,
				lp->shape.hexagon, 6, Convex, CoordModePrevious);
			/*drawPolygon(MI_DISPLAY(mi), MI_WINDOW(mi),
				lp->shape.hexagon, 6);*/
		}
	}
}

static void
setcelltoggles(ModeInfo * mi, int col, int row)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int position;
	CellList *curr;

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	if (!curr) {
		(void) fprintf(stderr, "state toggling but not on list\n");
		return;
	}
	curr->info.toggle = True;
}

static void
free_cells(lifestruct * lp)
{
	if (lp->arr != NULL) {
		free(lp->arr);
		lp->arr = (CellList **) NULL;
	}
}

static void
free_stuff(Display * display, lifestruct * lp)
{
	if (lp->cmap != None) {
		XFreeColormap(display, lp->cmap);
		if (lp->backGC != None) {
			XFreeGC(display, lp->backGC);
			lp->backGC = None;
		}
		lp->cmap = None;
	} else
		lp->backGC = None;
}

static void
free_life(Display * display, lifestruct * lp)
{
	int state;

	for (state = 0; state < STATES; state++) {
		if (lp->first[state])
			flushList(lp, state);
		if (lp->last[state])
			free(lp->last[state]);
		lp->last[state] = (CellList *) NULL;
		if (lp->first[state])
			free(lp->first[state]);
		lp->first[state] = (CellList *) NULL;
	}
	free_cells(lp);
	free_stuff(display, lp);
	if (lp->logo != None) {
		destroyImage(&lp->logo, &lp->graphics_format);
		lp->logo = None;
	}
}

static Bool
setcellfromtoggle(ModeInfo * mi, int col, int row)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int neighbor, n, position;
	cellstruct info;
	CellList *curr, *currn;

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	if (!curr)
		return True;
	if ((curr->info.state == DEAD && curr->info.toggle) ||
	    (curr->info.state == LIVE && !curr->info.toggle)) {
		for (n = 0; n < lp->neighbors; n++) {
			neighbor = positionOfNeighbor(lp, n, col, row);
			currn = lp->arr[neighbor];
			if (!currn) {
				info.position = neighbor;
				info.age = 0;
				info.state = DEAD;
				info.toggle = False;
				if (!addToList(lp, DEAD, info)) {
					free_life(MI_DISPLAY(mi), lp);
					return False;
				}
			}
		}
	}
	if (curr->info.state == DEAD && curr->info.toggle) {
		removeFromList(lp, DEAD, curr);
		if (!patternsearch ||  /* throws away stuff over border */
				(position % lp->ncols != 0 &&
				position / lp->ncols != 0)) {
			info.age = 0;
			info.position = position;
			info.toggle = False;
			info.state = LIVE;
			if (!addToList(lp, LIVE, info)) {
				free_life(MI_DISPLAY(mi), lp);
				return False;
			}
			drawCell(mi, info);
		}
	} else if (curr->info.state == LIVE && !curr->info.toggle) {
		info = curr->info;
		/* if we aren't up to blue yet, then keep aging the cell. */
		if ((MI_NPIXELS(mi) > 2) &&
		    (info.age < (unsigned short) (MI_NPIXELS(mi) * 0.7))) {
			++(info.age);
			/* cc: error 1405: "/opt/ansic/lbin/ccom"
			   terminated abnormally with signal 11.
			   *** Error exit code 9 */
			/* Next 2 line trips up HP cc -g -O, remove a flag */
			curr->info.age = info.age;
			drawCell(mi, info);
		}
		else if (glidersearch) {
			/* Aging out of static blocks, not good in general, use wisely. */
			if (info.age < 64) {
				++(info.age);
				curr->info.age = info.age;
			} else
				curr->info.state = DEAD;
		}
	}
	return True;
}

static Bool
setcell(ModeInfo * mi, int col, int row, int state)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int neighbor, n, position;
	cellstruct info;
	CellList *curr, *currn;

	if (col < 0 || row < 0 || col >= lp->ncols || row >= lp->nrows) {
#ifdef DEBUG
		(void) printf("col %d, row %d outside grid\n", col, row);
#endif
		return True; /* Actually its a 3rd case */
	}

	position = row * lp->ncols + col;
	curr = lp->arr[position];
	/* cc: error 1405: "/opt/ansic/lbin/ccom"
	   terminated abnormally with signal 11.
	   *** Error exit code 9 */
	/* Following lines trip up HP cc -g -O, remove a flag */
	if (state == LIVE) {
		if (curr && curr->info.state == DEAD) {
			removeFromList(lp, DEAD, curr);
			curr = (CellList *) NULL;
		}
		if (!curr) {
			for (n = 0; n < lp->neighbors; n++) {
				neighbor = positionOfNeighbor(lp, n, col, row);
				currn = lp->arr[neighbor];
				if (!currn) {
					info.age = 0;
					info.position = neighbor;
					info.state = DEAD;
					info.toggle = False;
					if (!addToList(lp, DEAD, info)) {
						free_life(MI_DISPLAY(mi), lp);
						return False;
					}
				}
			}
			info.age = 0;
			info.position = position;
			info.state = LIVE;
			info.toggle = False;
			if (!addToList(lp, LIVE, info)) {
				free_life(MI_DISPLAY(mi), lp);
				return False;
			}
			drawCell(mi, info);
		} else {
			info = curr->info;
			info.age = 0;
			drawCell(mi, info);
		}
	} else if (curr && curr->info.state == LIVE) {
		info.age = 0;
		info.position = position;
		info.state = DEAD;
		info.toggle = False;
		removeFromList(lp, LIVE, curr);	/* Just in case... */
		if (!addToList(lp, DEAD, info)) {
			free_life(MI_DISPLAY(mi), lp);
			return False;
		}
		drawCell(mi, info);
	}
	return True;
}

#if 0
static int
n_neighbors(lifestruct * lp, CellList * curr)
{
	int col, row, n, p, count = 0;

	col = curr->info.position % lp->ncols;
	row = curr->info.position / lp->ncols;
	for (n = 0; n < lp->neighbors; n++) {
		p = positionOfNeighbor(lp, n, col, row);
		if (lp->arr[p] && lp->arr[p]->info.state == LIVE) {
			count++;
		}
	}
	return count;
}
#endif

/* make work with negative x */
#define MOD(x,y) ((x)<0)?((y)-((-x)%(y)))%(y):(x)%(y)
#define DIV(x,y) ((x)<0)?((x)+1)/(y)-1:(x)/(y)
#define CEIL(z) ((z)+(((z)>=0)?1:0))/2

static int
ng_neighbors(lifestruct * lp, CellList * curr, int *group)
{
	int col, row, n, p, count = 0, gcount = 0;

	col = (int) (curr->info.position % lp->ncols);
	row = (int) (curr->info.position / lp->ncols);
	for (n = 0; n < lp->neighbors; n++) {
		p = positionOfNeighbor(lp, n, col, row);
		gcount <<= 1;
		if (lp->arr[p] && lp->arr[p]->info.state == LIVE) {
			count++;
			gcount++;
		}
	}
	*group = gcount;
	return count;
}

static void
RandomSoup(ModeInfo * mi, int symmetryIndex, int percent, int fieldX, int fieldY)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int col, row;
	int halfcols = lp->ncols / 2;
	int halfrows = lp->nrows / 2;
	int vx = fieldX / 2 + 1;
	int vy = fieldY / 2 + 1;
	int even = NRAND(2);
	int yEven = NRAND(2);

	if (lp->polygon == 3) {
		/* Triangles are stored left/right not up/down */
		halfcols = lp->nrows / 2;
		halfrows = lp->ncols / 2;
		vx = fieldY / 2;
		vy = fieldX / 2;
	}
	if (vx < 1)
		vx = 1;
	if (vy < 1)
		vy = 1;
	switch (symmetryIndex) {
	case 0: /* Full Random */
		for (row = halfrows - vy; row < halfrows + vy; ++row) {
			for (col = halfcols - vx; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					SetList(col, row);
				}
			}
		}
		(void) strcpy(lp->nameString, "Full Random");
		break;
	case 1: /* Sym X Random */
		for (row = halfrows - vy; row < halfrows + vy; ++row) {
			for (col = halfcols; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					if (lp->polygon != 5) {
						SetList(col, row);
					}
					if (lp->polygon == 3) {
						SetList(2 * halfcols - col, row);
					} else if (lp->polygon == 4) {
						SetList(2 * halfcols - col - even, row);
					} else if (lp->polygon == 5) {
						int tcol = 2 * halfcols - col;
						int mod = (col + 2 * row) % 4;

						if (mod == 2 && col == halfcols)
							continue;
						SetList(col, row);
						if (mod == 1)
							tcol += 2;
						else if (mod == 2 || mod == 3)
							tcol++;
						SetList(tcol, row);
					} else if (lp->polygon == 6) {
						SetList(2 * halfcols - col - (((row & 1) == 1) ? 0 : 1), row);
					}
				}
			}
		}
		(void) strcpy(lp->nameString, "Sym X Random");
		break;
	case 2: /* Sym Y Random */
		for (row = halfrows; row < halfrows + vy; ++row) {
			for (col = halfcols - vx; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					if (lp->polygon != 5) {
						SetList(col, row);
					}
					if (lp->polygon == 3) {
						SetList(col, 2 * halfrows - row - 1);
					} else if (lp->polygon == 4) {
						SetList(col, 2 * halfrows - row - even);
					} else if (lp->polygon == 5) {
						int mod = (col + 2 * row) % 4;
						int tcol = col;
						int trow = 2 * halfrows - row;

						if (mod == 1 && row == halfrows)
							continue;
						SetList(col, row);
						if (mod == 0)
							tcol++;
						else if (mod == 1)
							tcol--;
						SetList(tcol, trow);
					} else if (lp->polygon == 6) {
						SetList(col, 2 * halfrows - row);
					}
				}
			}
		}
		(void) strcpy(lp->nameString, "Sym Y Random");
		break;
	case 3: /* Sym XY Random */
		for (row = halfrows; row < halfrows + vy; ++row) {
			for (col = halfcols; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					if (lp->polygon != 5) {
						SetList(col, row);
					}
					if (lp->polygon == 3) {
						SetList(col, 2 * halfrows - row - 1);
						SetList(2 * halfcols - col, row);
						SetList(2 * halfcols - col, 2 * halfrows - row - 1);
					} else if (lp->polygon == 4) {
						SetList(col, 2 * halfrows - row - yEven);
						SetList(2 * halfcols - col - even, row);
						SetList(2 * halfcols - col - even, 2 * halfrows - row - yEven);
					} else if (lp->polygon == 5) {
						int tcol = 2 * halfcols - col;
						int trow = 2 * halfrows - row;
						int mod = (col + 2 * row) % 4;

						if ((mod == 2 && col == halfcols) ||
						    (mod == 1 && row == halfrows))
							continue;
						SetList(col, row);
						if (mod == 1)
							tcol += 2;
						else if (mod == 2 || mod == 3)
							tcol++;
						SetList(tcol, row);
						tcol = col;
						if (mod == 0)
							tcol++;
						else if (mod == 1)
							tcol--;
						SetList(tcol, trow);
						tcol = 2 * halfcols - col + 1;
						SetList(tcol, trow);
					} else if (lp->polygon == 6) {
						SetList(col, 2 * halfrows - row);
						SetList(2 * halfcols - col - (((row & 1) == 1) ? 0 : 1), row);
						SetList(2 * halfcols - col - (((row & 1) == 1) ? 0 : 1), 2 * halfrows - row);
					}
				}
			}
		}
		(void) strcpy(lp->nameString, "Sym XY Random");
		break;
	case 4: /* Antisym X Random */
		for (row = halfrows - vy; row < halfrows + vy; ++row) {
			for (col = halfcols; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					if (lp->polygon != 5) {
						SetList(col, row);
					}
					if (lp->polygon == 3) {
						SetList(2 * halfcols - col, 2 * halfrows - row - 1);
					} else if (lp->polygon == 4) {
						SetList(2 * halfcols - col, 2 * halfrows - row - even);
					} else if (lp->polygon == 5) {
						int tcol = 2 * halfcols - col - even;
						int trow = 2 * halfrows - row - even;
						int mod = (col + 2 * row) % 4;

						if (even == 0 && mod == 2 && col == halfcols)
							continue;
						SetList(col, row);
						if (even == 1) {
							SetList(tcol, trow);
						} else {
							SetList(tcol + 1, trow);
						}
					} else if (lp->polygon == 6) {
						SetList(2 * halfcols - col, 2 * halfrows - row - 1);
					}
				}
			}
		}
		(void) strcpy(lp->nameString, "Antisym X Random");
		break;
	case 5: /* Antisym Y Random */
		for (row = halfrows; row < halfrows + vy; ++row) {
			for (col = halfcols - vx; col < halfcols + vx; ++col) {
				if (NRAND(100) < percent) {
					if (lp->polygon != 5) {
						SetList(col, row);
					}
					if (lp->polygon == 3) {
						SetList(2 * halfcols - col - 1, 2 * halfrows - row);
					} else if (lp->polygon == 4) {
						SetList(2 * halfcols - col - 1, 2 * halfrows - row - even);
					} else if (lp->polygon == 5) {
						int tcol = 2 * halfcols - col - even;
						int trow = 2 * halfrows - row - even;
						int mod = (col + 2 * row) % 4;

						if (even == 0 && mod == 0 && row == halfrows)
							continue;
						SetList(col, row);
						if (even == 1) {
							SetList(tcol, trow);
						} else {
							SetList(tcol + 1, trow);
						}
					} else if (lp->polygon == 6) {
						SetList(2 * halfcols - col - 1, 2 * halfrows - row - 1);
					}
				}
			}
		}
		(void) strcpy(lp->nameString, "Antisym Y Random");
		break;
	case 6: /* Half Antisym Random */
		// Diagonal, Triangular (on face, on corner)
		if (lp->polygon == 3) {
			int odd = (halfcols + halfrows) % 2;

			if (even == 1) {
				vx = vy = (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col <  2 * row + 1; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + col - row, halfrows - row - 1 + odd);
							SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows + col / 2 + odd);
							SetList(halfcols - col / 2 - row - 1, halfrows + row - (col + 1) / 2 + odd);
						}
						if (NRAND(100) < percent) {
							SetList(halfcols - col + row, halfrows + row + odd);
							SetList(halfcols + (col + 1) / 2 - 2 * row - 1, halfrows - col / 2 - 1 + odd);
							SetList(halfcols + col / 2 + row + 1, halfrows - row + (col + 1) / 2 - 1 + odd);
						}
					}
				}
			} else {
				vx = vy =  (vx + vy) / 4;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < 6 * row + 5; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + col / 2, halfrows + (col + 1) / 2 - 2 * row - 2 + odd);
							SetList(halfcols + (col + 1) / 2 - 3 * row - 3, halfrows + row + 2 - col / 2 - 1 + odd);
							SetList(halfcols - col + 3 * row + 3, halfrows + row + 1 + odd);
						}
					}
				}
				if (NRAND(100) < percent) {
					/* center triangle */
					SetList(halfcols, halfrows + odd);
				}
			}
		} else if (lp->polygon == 4) {
			vx = vy = (vx + vy) / 2;
			for (row = -vy; row < vy; ++row) {
				for (col = row; col < vx; ++col) {
					if (NRAND(100) < percent) {
						if (even == 1) {
							SetList(col + halfcols, row + halfrows);
							SetList(row + halfcols, col + halfrows);
						} else {
							SetList(-col + halfcols, row + halfrows);
							SetList(-row + halfcols, col + halfrows);
						}
					}
				}
			}
		} else if (lp->polygon == 5) {
			for (row = halfrows; row < halfrows + vy; ++row) {
				for (col = halfcols; col < halfcols + 2 * vy; ++col) {
					if (NRAND(100) < percent) {
						int tcol = 2 * halfcols - col - 1;
						int trow = 2 * halfrows - row - 1;
						int mod = (col + 2 * row) % 4;

						SetList(col, row);
						SetList(tcol, trow);
						tcol = (row - halfrows) * 2 + halfcols;
						trow = (col - halfcols) / 2 + halfrows;
						if (mod == 1 || mod == 2)
							tcol++;
						SetList(2 * halfcols - tcol - 1, trow);
						SetList(tcol, 2 * halfrows - trow - 1);
					}
				}
			}
		} else if (lp->polygon == 6) {
			if (even == 1) {
				vx = vy =  2 * (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < row; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2, halfrows + row);
							SetList(halfcols + (col + row + halfrows % 2) / 2 - halfrows % 2, halfrows - row + col + 1);
							SetList(halfcols - row + (col + halfrows % 2) / 2 + (halfrows + 1) % 2, halfrows - col + 1);
						}
					}
				}
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < row - 1; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + row - (col + (halfrows + 1) % 2) / 2 - halfrows % 2, halfrows + col + 1);
							SetList(halfcols + col - (row + (halfrows + 1) % 2) / 2 + (halfrows + 1) % 2, halfrows - row + 1);
							SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows + row - col);
							}
					}
				}
			} else {
				vx = vy = 2 * (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < row; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2, halfrows + row);
							SetList(halfcols + (col + row + (halfrows + 1) % 2) / 2, halfrows - row + col);
							SetList(halfcols - row + (col + (halfrows + 1) % 2) / 2, halfrows - col);
						}
						if (NRAND(100) < percent) {
							SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows + col);
							SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows - row);
							SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows + row - col);
						}
					}
				}
				if (NRAND(100) < percent) {
					SetList(halfcols, halfrows);
				}
			}
		}
		(void) strcpy(lp->nameString, "Half Antisym Random");
		break;
	case 7: /* Half Sym Random */
		// Diagonal, Triangular (on face, on corner)
		if (lp->polygon == 3) {
			int odd = (halfcols + halfrows) % 2;

			if (even == 1) {
				vx = vy = (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col <  2 * row + 1; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + col - row, halfrows - row - 1 + odd);
							SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows - col / 2 - 1 + odd);
							SetList(halfcols - col / 2 - row - 1, halfrows + row - (col + 1) / 2 + odd);
							SetList(halfcols + col - row, halfrows + row + odd);
							SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows + col / 2 + odd);
							SetList(halfcols - col / 2 - row - 1, halfrows - row + (col + 1) / 2 - 1 + odd);
						}
					}
				}
			} else {
				vx = vy = (vx + vy) / 4;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < 3 * row + 4; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + col / 2, halfrows + (col + 1) / 2 - 2 * row - 2 + odd);
							SetList(halfcols - col / 2, halfrows + (col + 1) / 2 - 2 * row - 2 + odd);
							SetList(halfcols + (col + 1) / 2 - 3 * row - 3, halfrows + row + 2 - col / 2 - 1 + odd);
							SetList(halfcols - (col + 1) / 2 + 3 * row + 3, halfrows + row + 2 - col / 2 - 1 + odd);
							SetList(halfcols - col + 3 * row + 3, halfrows + row + 1 + odd);
							SetList(halfcols + col - 3 * row - 3, halfrows + row + 1 + odd);
						}
					}
				}
				if (NRAND(100) < percent) {
					/* center triangle */
					SetList(halfcols, halfrows + odd);
				}
			}
		} else if (lp->polygon == 4) {
			vx = vy = (vx + vy) / 2;
			for (row = -vy; row < 0; ++row) {
				for (col = row; col < vx; ++col) {
					if (NRAND(100) < percent) {
						SetList(col + halfcols, row + halfrows);
						SetList(row + halfcols, col + halfrows);
						SetList(-col + halfcols, -row + halfrows);
						SetList(-row + halfcols, -col + halfrows);
					}
				}
			}
		} else if (lp->polygon == 6) {
			if (even == 1) {
				vx = vy =  2 * (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < row / 2 + 1; ++col) { // 2 6 10
						if (NRAND(100) < percent) {
							SetList(halfcols + (col + row + (halfrows + 1) % 2) / 2 - 1 + halfrows % 2, halfrows - row + col + 2);
							SetList(halfcols + row - (col + (halfrows + 1) % 2) / 2 - 1, halfrows - col + 1);
							SetList(halfcols - col + (row + halfrows % 2) / 2, halfrows + row - 1);
							SetList(halfcols + col - (row + (halfrows + 1) % 2) / 2 + 1, halfrows + row - 1);
							SetList(halfcols - row + (col + halfrows % 2) / 2 + 2, halfrows - col + 1);
							SetList(halfcols - (col + row + halfrows % 2) / 2 + 1 + halfrows % 2, halfrows - row + col + 2);
						}
					}
				}
				for (row = 0; row < vy; ++row) {
					for (col = 0; col < row / 2; ++col) {
						if (NRAND(100) < percent) { // 0 4 8
							SetList(halfcols + col - (row + halfrows % 2) / 2 + 1 + halfrows % 2, halfrows - row + 2);
							SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2 + halfrows % 2 - 1, halfrows - row + 2);
							SetList(halfcols + row - (col + (halfrows + 1) % 2) / 2 - 1, halfrows + col + 1);
							SetList(halfcols + (col + row + halfrows % 2) / 2, halfrows + row - col - 1);
							SetList(halfcols - (col + row + (halfrows + 1) % 2) / 2 + 1, halfrows + row - col - 1);
							SetList(halfcols - row + (col + halfrows % 2) / 2 + 2, halfrows + col + 1);
						}
					}
				}
			} else {
				vx = vy =  2 * (vx + vy) / 3;
				for (row = 0; row < vy; ++row) {
					for (col = 0; col <= row; ++col) {
						if (NRAND(100) < percent) {
							SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows + row);
							SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows - col);
							SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows - row + col);
							SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows + col);
							SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows - row);
							SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows + row - col);
						}
					}
				}
				if (NRAND(100) < percent) {
					SetList(halfcols, halfrows);
				}
			}
		}
		(void) strcpy(lp->nameString, "Half Sym Random");
		break;
	case 8: /* Full Antisym Random */
		// Square, Hexagonal (on face hex, on corner tri)
		if (lp->polygon == 3) {
			int odd = (halfcols + halfrows) % 2;

			vx = vy = (vx + vy) / 3;
			for (row = 0; row < vy; ++row) {
				for (col = 0; col <  2 * row + 1; ++col) {
					if (NRAND(100) < percent) {
						SetList(halfcols + col - row, halfrows - row - 1 + odd);
						SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows + col / 2 + odd);
						SetList(halfcols - col / 2 - row - 1, halfrows + row - (col + 1) / 2 + odd);
						SetList(halfcols - col + row, halfrows + row + odd);
						SetList(halfcols + (col + 1) / 2 - 2 * row - 1, halfrows - col / 2 - 1 + odd);
						SetList(halfcols + col / 2 + row + 1, halfrows - row + (col + 1) / 2 - 1 + odd);
					}
				}
			}
		} else if (lp->polygon == 4) {
			vx = vy = (vx + vy) / 2;
			for (row = -vy; row < 0; ++row) {
				for (col = row; col < vx; ++col) {
					if (NRAND(100) < percent) {
						SetList(col + halfcols + even, row + halfrows + even);
						SetList(row + halfcols + even, -col + halfrows);
						SetList(-col + halfcols, -row + halfrows);
						SetList(-row + halfcols, col + halfrows + even);
					}
				}
			}
			if (even == 0 && NRAND(100) < percent) {
				SetList(halfcols, halfrows);
			}
		} else if (lp->polygon == 6) {
			vx = vy = 2 * (vx + vy) / 3;
			for (row = 0; row < vy; ++row) {
				for (col = 0; col < row; ++col) {
					if (NRAND(100) < percent) {
						SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2, halfrows + row);
						SetList(halfcols + (col + row + (halfrows + 1) % 2) / 2, halfrows - row + col);
						SetList(halfcols - row + (col + (halfrows + 1) % 2) / 2, halfrows - col);
						SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows + col);
						SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows - row);
						SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows + row - col);
					}
				}
			}
			if (NRAND(100) < percent) {
				SetList(halfcols, halfrows);
			}
		}
		(void) strcpy(lp->nameString, "Full Antisym Random");
		break;
	case 9: /* Full Sym Random */
		// Square, Hexagonal (on face hex, on corner tri)
		if (lp->polygon == 3) {
			int odd = (halfcols + halfrows) % 2;

			vx = vy = (vx + vy) / 3;
			for (row = 0; row < vy; ++row) {
				for (col = 0; col <  row + 1; ++col) {
					if (NRAND(100) < percent) {
						SetList(halfcols - col + row, halfrows - row - 1 + odd); /* 0 */
						SetList(halfcols + col / 2 + row + 1, halfrows - row + (col + 1) / 2 - 1 + odd); /* 1 */
						SetList(halfcols + (col + 1) / 2 - 2 * row - 1, halfrows + col / 2 + odd); /* 2 */
						SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows + col / 2 + odd); /* 3 */
						SetList(halfcols - col / 2 - row - 1, halfrows - row + (col + 1) / 2 - 1 + odd); /* 4 */
						SetList(halfcols - col + row, halfrows + row + odd); /* 5 */
						SetList(halfcols + col - row, halfrows + row + odd); /* 6 */
						SetList(halfcols - col / 2 - row - 1, halfrows + row - (col + 1) / 2 + odd); /* 7 */
						SetList(halfcols - (col + 1) / 2 + 2 * row + 1, halfrows - col / 2 - 1 + odd); /* 8 */
						SetList(halfcols + (col + 1) / 2 - 2 * row - 1, halfrows - col / 2 - 1 + odd); /* 9 */
						SetList(halfcols + col / 2 + row + 1, halfrows + row - (col + 1) / 2 + odd); /* 10 */
						SetList(halfcols + col - row, halfrows - row - 1 + odd); /* 11 */
					}
				}
			}
		} else if (lp->polygon == 4) {
			vx = vy = (vx + vy) / 2;
			for (row = -vy; row < 0; ++row) {
				for (col = row; col <= 0; ++col) {
					if (NRAND(100) < percent) {
						SetList(col + halfcols + even, row + halfrows + even);
						SetList(row + halfcols + even, -col + halfrows);
						SetList(-col + halfcols, -row + halfrows);
						SetList(-row + halfcols, col + halfrows + even);
						SetList(-col + halfcols, row + halfrows + even);
						SetList(-row + halfcols, -col + halfrows);
						SetList(col + halfcols + even, -row + halfrows);
						SetList(row + halfcols + even, col + halfrows + even);
					}
				}
			}
			if (even == 0 && NRAND(100) < percent) {
				SetList(halfcols, halfrows);
			}
		} else if (lp->polygon == 6) {
			vx = vy = 2 * (vx + vy) / 3;
			for (row = 0; row < vy; ++row) {
				for (col = 0; col <= row / 2; ++col) {
					if (NRAND(100) < percent) {
						SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2, halfrows + row);
						SetList(halfcols + (col + row + (halfrows + 1) % 2) / 2, halfrows - row + col);
						SetList(halfcols - row + (col + (halfrows + 1) % 2) / 2, halfrows - col);
						SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows + col);
						SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows - row);
						SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows + row - col);
						SetList(halfcols + col - (row + halfrows % 2) / 2, halfrows + row);
						SetList(halfcols + row - (col + halfrows % 2) / 2, halfrows - col);
						SetList(halfcols - (col + row + halfrows % 2) / 2, halfrows - row + col);
						SetList(halfcols - row + (col + (halfrows + 1) % 2) / 2, halfrows + col);
						SetList(halfcols + (col + row + (halfrows + 1) % 2) / 2, halfrows + row - col);
						SetList(halfcols - col + (row + (halfrows + 1) % 2) / 2, halfrows - row);
					}
				}
			}
			if (NRAND(100) < percent) {
				SetList(halfcols, halfrows);
			}
		}
		(void) strcpy(lp->nameString, "Full Sym Random");
		break;
	default:
		(void) fprintf(stderr, "wrong symmetry %d\n", symmetryIndex);
	}
	if (MI_IS_VERBOSE(mi)) {
		(void) fprintf(stdout, "%s\n", lp->nameString);
	}
}

static void
GetPattern(ModeInfo * mi, int pattern_rule, int pattern)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int row, col, orient = 0, temp, startX, startY;
	char *patptr = (char *) NULL;
#ifdef LIFE_NAMES
	int pat = 2 * pattern + 1;
	char * patstrg = (char *) "";
#else
	int pat = pattern;
#endif

	if (filePattern) {
		patptr = &filePattern[0];
#ifdef LIFE_NAMES
		(void) strcpy(lp->nameString, patstrg);
#endif
	} else {
		switch (lp->neighbors) {
		case 5:
			switch (pattern_rule) {
			case LIFE_5B2S24:
				patptr = &patterns_5B2S24[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_5B2S24[2 * pattern][0];
#endif
			break;
			case LIFE_5B24S2:
				patptr = &patterns_5B24S2[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_5B24S2[2 * pattern][0];
#endif
			}
			break;
		case 6:
			switch (pattern_rule) {
			case LIFE_6B2aS2b34:
				patptr = &patterns_6B2aS2b34[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_6B2aS2b34[2 * pattern][0];
#endif
				break;
			case LIFE_6B2a3a4bS2a2b4a:
				patptr = &patterns_6B2a3a4bS2a2b4a[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_6B2a3a4bS2a2b4a[2 * pattern][0];
#endif
				break;
			case LIFE_6B2S35:
				patptr = &patterns_6B2S35[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_6B2S35[2 * pattern][0];
#endif
				break;
			case LIFE_6B245S3:
				patptr = &patterns_6B245S3[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_6B245S3[2 * pattern][0];
#endif
				break;
			case LIFE_6B2S34:
				patptr = &patterns_6B2S34[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_6B2S34[2 * pattern][0];
#endif
				break;
			}
			break;
		case 7:
			/*switch (pattern_rule) {
			case LIFE_7B346S23:*/
				patptr = &patterns_7B346S23[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_7B346S23[2 * pattern][0];
#endif
				/*break;*/
			break;
		case 8:
			switch (pattern_rule) {
			case LIFE_8B3S2a2b2c2d3:
				patptr = &patterns_8B3S2a2b2c2d3[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_8B3S2a2b2c2d3[2 * pattern][0];
#endif
				break;
			case LIFE_8B3S23:
				if (pattern < (int) common_8size) {
				  patptr = &patterns_8B3_6S23[pat][0];
#ifdef LIFE_NAMES
				  patstrg = &patterns_8B3_6S23[2 * pattern][0];
#endif
				} else {
				  patptr = &patterns_8B3S23[pat - PAT_SPACE * common_8size][0];
#ifdef LIFE_NAMES
				  patstrg = &patterns_8B3S23[2 * pattern - 2 * common_8size][0];
#endif
				}
				break;
			case LIFE_8B36S23:
				if (pattern < (int) common_8size) {
				  patptr = &patterns_8B3_6S23[pat][0];
#ifdef LIFE_NAMES
				  patstrg = &patterns_8B3_6S23[2 * pattern][0];
#endif
				} else {
				  patptr = &patterns_8B36S23[pat - PAT_SPACE * common_8size][0];
#ifdef LIFE_NAMES
				  patstrg = &patterns_8B36S23[2 * pattern - 2 * common_8size][0];
#endif
				}
				break;
			case LIFE_8B3678S34678:
				patptr = &patterns_8B3678S34678[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_8B3678S34678[2 * pattern][0];
#endif
				break;
			case LIFE_8B3S245:
				patptr = &patterns_8B3S245[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_8B3S245[2 * pattern][0];
#endif
				break;
			case LIFE_8B36S245:
				patptr = &patterns_8B36S245[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_8B36S245[2 * pattern][0];
#endif
				break;
			}
			break;
		case 9:
			switch (pattern_rule) {
			case LIFE_9B34S3:
				patptr = &patterns_9B34S3[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_9B34S3[2 * pattern][0];
#endif
				break;
			case LIFE_9B3S34: /* has replicator */
				patptr = &patterns_9B3S34[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_9B3S34[2 * pattern][0];
#endif
				break;
			}
			break;
		case 12:
			switch (pattern_rule) {
			case LIFE_12B45S34:
				patptr = &patterns_12B45S34[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B45S34[2 * pattern][0];
#endif
				break;
			case LIFE_12B456S45:
				patptr = &patterns_12B456S45[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B456S45[2 * pattern][0];
#endif
				break;
			case LIFE_12B45S23:
				patptr = &patterns_12B45S23[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B45S23[2 * pattern][0];
#endif
				break;
			case LIFE_12B456S23:
				patptr = &patterns_12B456S23[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B456S23[2 * pattern][0];
#endif
				break;
			case LIFE_12B456S12:
				patptr = &patterns_12B456S12[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B456S12[2 * pattern][0];
#endif
				break;
			case LIFE_12B3S2:
				patptr = &patterns_12B3S2[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B3S2[2 * pattern][0];
#endif
				break;
			case LIFE_12B3S27:
				patptr = &patterns_12B3S27[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B3S27[2 * pattern][0];
#endif
				break;
			case LIFE_12B3S278:
				patptr = &patterns_12B3S278[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B3S278[2 * pattern][0];
#endif
				break;
			case LIFE_12B46S24:
				patptr = &patterns_12B46S24[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B46S24[2 * pattern][0];
#endif
				break;
			case LIFE_12B46S246:
				patptr = &patterns_12B46S246[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B46S246[2 * pattern][0];
#endif
				break;
			case LIFE_12B456S34:
				patptr = &patterns_12B456S34[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B456S34[2 * pattern][0];
#endif
				break;
			case LIFE_12B4S35:
				patptr = &patterns_12B4S35[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B4S35[2 * pattern][0];
#endif
				break;
			case LIFE_12B46S35:
				patptr = &patterns_12B46S35[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B46S35[2 * pattern][0];
#endif
				break;
			case LIFE_12B46S356:
				patptr = &patterns_12B46S356[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B46S356[2 * pattern][0];
#endif
				break;
			case LIFE_12B4S456:
				patptr = &patterns_12B4S456[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B4S456[2 * pattern][0];
#endif
				break;
			case LIFE_12B3S23:
				patptr = &patterns_12B3S23[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B3S23[2 * pattern][0];
#endif
				break;
			case LIFE_12B4S345:
				patptr = &patterns_12B4S345[pat][0];
#ifdef LIFE_NAMES
				patstrg = &patterns_12B4S345[2 * pattern][0];
#endif
				break;
			}
			break;
		}
#ifdef LIFE_NAMES
		(void) strcpy(lp->nameString, patstrg);
#endif
	}
#ifndef DEBUG
	if (lp->polygon == 3 || lp->polygon == 6) {
		orient = NRAND(12);
	} else if (lp->polygon == 4 || lp->polygon == 5) {
		orient = NRAND(8);
	}
#endif
	if (MI_IS_VERBOSE(mi) && !filePattern) {
#ifdef LIFE_NAMES
		(void) fprintf(stdout, "%s, ", patstrg);
#endif
		(void) fprintf(stdout, "table number %d\n", pattern);
	}
	while ((col = *patptr++) != 127) {
		row = *patptr++;
		if (lp->polygon == 3) {
			temp = col;
			col = row;
			row = temp;
			if (orient >= 6) {
				row = -row;
			}
			/* (a,b) => (b, b-a) */
			switch (orient % 6) {
			case 0:
				break;
		       case 1:
				temp = col;
				col = CEIL(temp + row);
				row = CEIL(row - temp + 1) - temp;
				break;
			case 2:
				temp = col;
				col = CEIL(temp - row);
				row = temp - CEIL(1 - row - temp);
				break;
			case 3:
				col = -col + 1;
				row = -row;
				break;
			case 4:
				temp = col;
				col = 1 - CEIL(temp + row);
				row = temp - CEIL(row - temp + 1);
				break;
			case 5:
				temp = col;
				col = 1 - CEIL(temp - row);
				row = CEIL(1 - row - temp) - temp;
				break;
			}
		} else if (lp->polygon == 4) {
			if (orient >= 4) {
				temp = col;
				col = row;
				row = temp;
			}
			/* Could have made it symmetrical with hexagons where
			 (a,b) => (-b, a), this should be equivalent */
			if (orient % 4 >= 2) {
				row = -row;
			}
			if (orient & 1) {
				col = -col;
			}
		} else if (lp->polygon == 5) {
			int mod = MOD(col + 2 * row, 4);
			int tcol = row * 2;
			int trow = DIV(col, 2);

			if (mod == 1 || mod == 2)
				tcol++;
			if (orient % 4 == 1) {
				col = -tcol - 1;
				row = trow;
			} else if (orient % 4 == 2) {
				col = -col - 1;
				row = -row - 1;
			} else if (orient % 4 == 3) {
				col = tcol;
				row = -trow - 1;
			}
			if (orient >= 4) {
				col = -col;
				col += 1;
				mod = MOD(col + 2 * row, 4);
				if (mod == 0)
					col++;
				if (mod == 1)
					col--;
			}
		} else if (lp->polygon == 6) {
			if (orient >= 6) {
				temp = col;
				col = row;
				row = temp;
			}
			/* (a,b) => (b, b-a) */
			switch (orient % 6) {
			case 0:
				break;
			case 1:
				temp = row;
				row = temp - col;
				col = temp;
				break;
			case 2:
				temp = -col;
				col = temp + row;
				row = temp;
				break;
			case 3:
				col = -col;
				row = -row;
				break;
			case 4:
				temp = -row;
				row = temp + col;
				col = temp;
				break;
			case 5:
				temp = col;
				col = temp - row;
				row = temp;
				break;
			}
		}
		startX = lp->ncols / 2;
		startY = lp->nrows / 2;
		if (lp->polygon == 5) {
			startX -= (startX + startY * 2) % 4;
		}
		col += startX;
		if (lp->polygon == 6) {
			if (row < 0)
				col += (startX & 1) ? -row / 2 :
					-(row - 1) / 2;
			else
				col += (startX & 1) ? -(row + 1) / 2 :
					-row / 2;
		}
		row += startY;
		if (lp->polygon == 3 &&
				(((startX + startY + 1) & 1) == 1)) {
			row++;
		}
		SetListFromGet(col, row);
	}
}

static void
shooter(ModeInfo * mi)
{
	lifestruct *lp = &lifes[MI_SCREEN(mi)];
	int hsp, vsp, hoff = 1, voff = 1, temp;

	/* Generate the glider at the edge of the screen */
	if (lp->neighbors == 6 && (lp->patterned_rule == LIFE_6B2aS2b34 ||
	    lp->patterned_rule == LIFE_6B2a3a4bS2a2b4a ||
	    lp->patterned_rule == LIFE_6B2S35 ||
	    lp->patterned_rule == LIFE_6B245S3)) {
		int hhex = 0, diagonal, quad;

		diagonal = NRAND(3);
		if (diagonal) {
			temp = MIN((lp->nrows + lp->ncols) / 3, 18);
			temp = NRAND(temp) - temp / 2;
			/* Take into account it is a 60 degree angle not 45 */
			if ((lp->ncols + temp) * 1.35 > lp->nrows) {

				hsp = (int) ((lp->ncols + temp) * 1.35 -
					lp->nrows) / 2;
				vsp = 0;
#ifdef HEXDEBUG
				(void) printf("a: ");
#endif
			} else {
				hsp = 0;
				vsp = (int) (lp->nrows -
					(lp->ncols + temp)) / 2;
#ifdef HEXDEBUG
				(void) printf("b: ");
#endif
			}
			quad = NRAND(4);
#ifdef HEXDEBUG
			(void) printf("quad=%d, ", quad);
#endif
			switch (quad) {
			case 0:  /* Upper left */
				vsp = (vsp & 1) ? vsp : vsp + 1;
				hhex = 1;
				break;
			case 1:  /* Lower left */
				hoff = -1;
				hsp = lp->ncols - 1 - hsp;
				vsp = (vsp & 1) ? vsp + 1 : vsp;
				hhex = hoff;
				break;
			case 2:  /* Upper right */
				voff = -1;
				vsp = lp->nrows - 1 - vsp;
				vsp = (vsp & 1) ? vsp : vsp - 1;
				hhex = hoff;
				break;
			case 3:  /* Lower right */
				voff = -1;
				hoff = -1;
				hsp = lp->ncols - 1 - hsp;
				vsp = lp->nrows - 1 - vsp;
				vsp = (vsp & 1) ? vsp - 1 : vsp;
				hhex = hoff;
				break;
			}
		} else {
			temp = MIN(lp->nrows / 3, 18);
			vsp = lp->nrows / 2 + NRAND(temp) - temp / 2;
			if (LRAND() & 1) {
				hsp = lp->ncols - 1;
				hoff = -1;
				hhex = (vsp & 1) ? 0 : hoff;
			} else {
				hsp = 0;
				hhex = (vsp & 1) ? hoff : 0;
			}
			voff = (LRAND() & 1) ? 1 : -1; /* Mirror image */
		}
#ifdef HEXDEBUG
		(void) printf("hhex=%d, hsp=%d, hoff=%d, vsp=%d, voff=%d\n",
			hhex, hsp, hoff, vsp, voff);
#endif
		if (lp->patterned_rule == LIFE_6B2aS2b34) {
			if (diagonal) {
				SetList(hsp + hhex, vsp);
				if (LRAND() & 1) {
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 1 * hoff + hhex, vsp + 4 * voff);
					SetList(hsp + hhex, vsp + 4 * voff);
				} else { /* Mirror image */
					SetList(hsp + 3 * hoff + hhex, vsp + 2 * voff);
					SetList(hsp + 0 * hoff, vsp + 3 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
				}
			} else {
				SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff + hhex, vsp + 0 * voff);
				SetList(hsp + 0 * hoff + hhex, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp - 1 * voff);
				SetList(hsp + 3 * hoff + hhex, vsp - 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_6B2a3a4bS2a2b4a) {
			if (diagonal) {
				switch (NRAND(3)) { /* 3 different gliders */
				case 0:
					/* No mirror image */
					SetList(hsp + 2 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 3 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 1 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
					SetList(hsp + 0 * hoff + hhex, vsp + 2 * voff);
					break;
				case 1:
					if (LRAND() & 1) {
						SetList(hsp + 0 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 1 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 3 * hoff, vsp + 1 * voff);
						SetList(hsp + 4 * hoff, vsp + 1 * voff);
						SetList(hsp + 0 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 1 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
					} else {
						SetList(hsp + 1 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 1 * hoff, vsp + 1 * voff);
						SetList(hsp + 3 * hoff, vsp + 1 * voff);
						SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 1 * hoff, vsp + 3 * voff);
						SetList(hsp + 2 * hoff, vsp + 3 * voff);
						SetList(hsp + 0 * hoff + hhex, vsp + 4 * voff);
					}
					break;
				case 2:
					if (LRAND() & 1) {
						SetList(hsp + 1 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 3 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 0 * hoff, vsp + 3 * voff);
						SetList(hsp + 1 * hoff, vsp + 3 * voff);
						SetList(hsp + 2 * hoff, vsp + 3 * voff);
					} else {
						SetList(hsp + 0 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 3 * hoff + hhex, vsp + 0 * voff);
						SetList(hsp + 3 * hoff, vsp + 1 * voff);
						SetList(hsp + 1 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 2 * hoff + hhex, vsp + 2 * voff);
						SetList(hsp + 1 * hoff, vsp + 3 * voff);
					}
				}
			} else {
				switch (NRAND(3)) { /* 3 different gliders */
				case 0:
					SetList(hsp + 0 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 0 * hoff + hhex, vsp - 2 * voff);
					SetList(hsp + 0 * hoff + hhex, vsp + 2 * voff);
					SetList(hsp + 0 * hoff, vsp - 1 * voff);
					SetList(hsp + 0 * hoff, vsp + 1 * voff);
					break;
				case 1:
					SetList(hsp + 0 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 0 * hoff, vsp + 1 * voff);
					SetList(hsp + 1 * hoff + hhex, vsp + 2 * voff);
					SetList(hsp + 2 * hoff + hhex, vsp - 2 * voff);
					SetList(hsp + 2 * hoff, vsp - 1 * voff);
					SetList(hsp + 2 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
					break;
				case 2:
					SetList(hsp + 0 * hoff, vsp - 1 * voff);
					SetList(hsp + 1 * hoff + hhex, vsp + 2 * voff);
					SetList(hsp + 2 * hoff + hhex, vsp - 2 * voff);
					SetList(hsp + 2 * hoff, vsp - 1 * voff);
					SetList(hsp + 2 * hoff + hhex, vsp + 0 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
				}
			}
		} else if (lp->patterned_rule == LIFE_6B2S35) {
			if (diagonal) {
				/* No mirror image */
				SetList(hsp + 0 * hoff + hhex, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 4 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp + 5 * voff);
				SetList(hsp + 2 * hoff + hhex, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 6 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 3 * voff);
				SetList(hsp + 5 * hoff, vsp + 5 * voff);
			} else {
				SetList(hsp + 5 * hoff, vsp - 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff + hhex, vsp - 2 * voff);
				SetList(hsp + 3 * hoff + hhex, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp - 3 * voff);
				SetList(hsp + 2 * hoff, vsp - 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 0 * hoff, vsp - 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
			}
		} else if (lp->patterned_rule == LIFE_6B245S3) {
			if (diagonal) {
				/* No mirror image */
				SetList(hsp + 3 * hoff + hhex, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff + hhex, vsp + 2 * voff);
				SetList(hsp + 3 * hoff + hhex, vsp + 2 * voff);
				SetList(hsp + 5 * hoff + hhex, vsp + 2 * voff);

				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 4 * hoff + hhex, vsp + 4 * voff);
				SetList(hsp + 1 * hoff, vsp + 5 * voff);
				SetList(hsp + 3 * hoff, vsp + 5 * voff);
			} else {
				SetList(hsp + 4 * hoff, vsp - 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp - 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp - 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff + hhex, vsp - 2 * voff);
				SetList(hsp + 0 * hoff + hhex, vsp - 0 * voff);
				SetList(hsp + 0 * hoff + hhex, vsp + 2 * voff);
			}
		}
	} else if ((lp->neighbors == 5 && lp->patterned_rule == LIFE_5B2S24) ||
			(lp->neighbors == 7 && lp->patterned_rule == LIFE_7B346S23)) {
		int reverse = 0; /* up/down pentagons need special handling */

		if (LRAND() & 1) {
			hsp = (LRAND() & 1) ? 0 : lp->ncols - 3;
			vsp = 2 * (NRAND(lp->nrows / 2));
			if (vsp > lp->nrows / 2)
				voff = -1;
			if (hsp > lp->ncols / 2)
				hoff = -1;
		} else {
			vsp = (LRAND() & 1) ? 0 : lp->nrows - 2;
			hsp = 4 * (NRAND(lp->ncols / 4));
			if (vsp > lp->nrows / 2) {
				voff = -1;
			}
			if (hsp > lp->ncols / 2) {
				hoff = -1;
				hsp -= 3;
			}
		}
		if ((hoff == -1 && voff != -1) ||
				(hoff != -1 && voff == -1)) {
			reverse = 1;
		}
		/* for reversing, up/down pentagons have to be swapped */
		if (lp->neighbors == 5 && lp->patterned_rule == LIFE_5B2S24) {
			if (LRAND() & 1) {
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
			}
			SetList(hsp + ((reverse) ? 3 : 2) * hoff, vsp + 1 * voff);
			SetList(hsp + ((reverse) ? 0 : 1) * hoff, vsp + 2 * voff);
			SetList(hsp + 2 * hoff, vsp + 2 * voff);
			SetList(hsp + 3 * hoff, vsp + 2 * voff);
			SetList(hsp + 4 * hoff, vsp + 2 * voff);
			SetList(hsp + 5 * hoff, vsp + 2 * voff);
			SetList(hsp + ((reverse) ? 3 : 2) * hoff, vsp + 3 * voff);
#if 0
			SetList(hsp + ((reverse) ? 0 : 1) * hoff, vsp + 0 * voff);
			SetList(hsp + ((reverse) ? 4 : 5) * hoff, vsp + 0 * voff);
			SetList(hsp + ((reverse) ? 2 : 3) * hoff, vsp + 1 * voff);
			SetList(hsp + 4 * hoff, vsp + 1 * voff);
			SetList(hsp + ((reverse) ? 1 : 0) * hoff, vsp + 2 * voff);
			SetList(hsp + 2 * hoff, vsp + 2 * voff);
			SetList(hsp + 3 * hoff, vsp + 2 * voff);
			SetList(hsp + ((reverse) ? 5 : 4) * hoff, vsp + 2 * voff);
			SetList(hsp + ((reverse) ? 3 : 2) * hoff, vsp + 3 * voff);
#endif
		} else if (lp->neighbors == 7 && lp->patterned_rule == LIFE_7B346S23) {
			SetList(hsp + 5 * hoff, vsp + 1 * voff);
			SetList(hsp + ((reverse) ? 6 : 7) * hoff, vsp + 1 * voff);
			SetList(hsp + ((reverse) ? 0 : 1) * hoff, vsp + 2 * voff);
			SetList(hsp + ((reverse) ? 4 : 5) * hoff, vsp + 2 * voff);
			SetList(hsp + 6 * hoff, vsp + 2 * voff);
			SetList(hsp + 1 * hoff, vsp + 3 * voff);
			SetList(hsp + 5 * hoff, vsp + 3 * voff);
			SetList(hsp + 2 * hoff, vsp + 4 * voff);
			SetList(hsp + 3 * hoff, vsp + 4 * voff);
		}
	} else if (lp->neighbors == 8 && (lp->patterned_rule == LIFE_8B3S2a2b2c2d3)) {
		/* Generate a space ship: c/5 p5, c/3 p3 */
		/* where p5 more common */
		int coord;
		int SS = NRAND(3);

		SS = (SS < 2) ? 0 : 1;
		if (LRAND() & 1) {
			hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
			vsp = NRAND(lp->nrows / 2) + lp->nrows / 4;
			coord = 1;
		} else {
			vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
			hsp = NRAND(lp->ncols / 2) + lp->ncols / 4;
			coord = 0;
		}
		if (vsp > lp->nrows / 2)
			voff = -1;
		if (hsp > lp->ncols / 2)
			hoff = -1;
		if (coord == 1) {
			if (SS) {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 3 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 4 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
			}
		} else {
			if (SS) {
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
			}
		}
	} else if (lp->neighbors == 8 && (lp->patterned_rule == LIFE_8B3S23 ||
		   lp->patterned_rule == LIFE_8B36S23)) {
		if (NRAND(3) != 0) {
			/* Generate a glider */
			if (LRAND() & 1) {
				hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
				vsp = NRAND(lp->nrows);
			} else {
				vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
				hsp = NRAND(lp->ncols);
			}
			if (vsp > lp->nrows / 2)
				voff = -1;
			if (hsp > lp->ncols / 2)
				hoff = -1;
			SetList(hsp + 2 * hoff, vsp + 0 * voff);
			SetList(hsp + 2 * hoff, vsp + 1 * voff);
			SetList(hsp + 2 * hoff, vsp + 2 * voff);
			SetList(hsp + 1 * hoff, vsp + 2 * voff);
			SetList(hsp + 0 * hoff, vsp + 1 * voff);
		} else {
			/* Generate a space ship: LWSS, MWSS, HWSS */
			/* where Light Weight ships are more common */
			int coord;
			int SS = NRAND(6);

			SS = (SS < 3) ? 0 : ((SS >= 5) ? 2 : 1);
			if (LRAND() & 1) {
				hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
				vsp = NRAND(lp->nrows / 2) + lp->nrows / 4;
				coord = 1;
			} else {
				vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
				hsp = NRAND(lp->ncols / 2) + lp->ncols / 4;
				coord = 0;
			}
			if (vsp > lp->nrows / 2)
				voff = -1;
			if (hsp > lp->ncols / 2)
				hoff = -1;
			if (coord == 1) {
				SetList(hsp + (SS + 4) * hoff, vsp + 0 * voff);
				SetList(hsp + (SS + 3) * hoff, vsp + 0 * voff);
				SetList(hsp + (SS + 2) * hoff, vsp + 0 * voff);
				SetList(hsp + (SS + 1) * hoff, vsp + 0 * voff);
				if (SS == 2) {
					SetList(hsp + 2 * hoff, vsp + 0 * voff);
				}
				if (SS != 0) {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
				}
				SetList(hsp + (SS + 4) * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + (SS + 4) * hoff, vsp + 2 * voff);
				SetList(hsp + (SS + 3) * hoff, vsp + 3 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + (SS + 4) * voff);
				SetList(hsp + 0 * hoff, vsp + (SS + 3) * voff);
				SetList(hsp + 0 * hoff, vsp + (SS + 2) * voff);
				SetList(hsp + 0 * hoff, vsp + (SS + 1) * voff);
				if (SS == 2) {
					SetList(hsp + 0 * hoff, vsp + 2 * voff);
				}
				if (SS != 0) {
					SetList(hsp + 0 * hoff, vsp + 1 * voff);
				}
				SetList(hsp + 1 * hoff, vsp + (SS + 4) * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + (SS + 4) * voff);
				SetList(hsp + 3 * hoff, vsp + (SS + 3) * voff);
			}
		}
	} else if (lp->neighbors == 8 && (lp->patterned_rule == LIFE_8B3S245 ||
			lp->patterned_rule == LIFE_8B36S245)) {
		if (LRAND() & 1) {
			hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
			vsp = NRAND(lp->nrows);
		} else {
			vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
			hsp = NRAND(lp->ncols);
		}
		if (vsp > lp->nrows / 2)
			voff = -1;
		if (hsp > lp->ncols / 2)
			hoff = -1;
		SetList(hsp + 2 * hoff, vsp + 0 * voff);
		SetList(hsp + 1 * hoff, vsp + 0 * voff);
		SetList(hsp + 1 * hoff, vsp + 1 * voff);
		SetList(hsp + 0 * hoff, vsp + 0 * voff);
		SetList(hsp + 0 * hoff, vsp + 1 * voff);
		SetList(hsp + 0 * hoff, vsp + 2 * voff);
	} else if (lp->neighbors == 8 &&
			lp->patterned_rule == LIFE_8B3678S34678) {
		/* Generate a butterfly */
		if (LRAND() & 1) {
			hsp = (LRAND() & 1) ? 0 : lp->ncols - 1;
			vsp = NRAND(lp->nrows);
		} else {
			vsp = (LRAND() & 1) ? 0 : lp->nrows - 1;
			hsp = NRAND(lp->ncols);
		}
		if (vsp > lp->nrows / 2)
			voff = -1;
		if (hsp > lp->ncols / 2)
			hoff = -1;
		SetList(hsp + 4 * hoff, vsp + 3 * voff);
		SetList(hsp + 4 * hoff, vsp + 2 * voff);
		SetList(hsp + 4 * hoff, vsp + 1 * voff);
		SetList(hsp + 3 * hoff, vsp + 4 * voff);
		SetList(hsp + 3 * hoff, vsp + 2 * voff);
		SetList(hsp + 3 * hoff, vsp + 1 * voff);
		SetList(hsp + 3 * hoff, vsp + 0 * voff);
		SetList(hsp + 2 * hoff, vsp + 4 * voff);
		SetList(hsp + 2 * hoff, vsp + 3 * voff);
		SetList(hsp + 1 * hoff, vsp + 4 * voff);
		SetList(hsp + 1 * hoff, vsp + 3 * voff);
		SetList(hsp + 0 * hoff, vsp + 3 * voff);
	} else if ((lp->neighbors == 12 &&
			(lp->patterned_rule == LIFE_12B456S12 ||
			lp->patterned_rule == LIFE_12B45S23 ||
			lp->patterned_rule == LIFE_12B456S23 ||
			lp->patterned_rule == LIFE_12B46S246 ||
			lp->patterned_rule == LIFE_12B3S27 ||
			lp->patterned_rule == LIFE_12B3S278 ||
			lp->patterned_rule == LIFE_12B45S34 ||
			lp->patterned_rule == LIFE_12B456S34 ||
			lp->patterned_rule == LIFE_12B4S35 ||
			lp->patterned_rule == LIFE_12B46S35 ||
			lp->patterned_rule == LIFE_12B46S356 ||
			lp->patterned_rule == LIFE_12B456S45 ||
			lp->patterned_rule == LIFE_12B4S456)) ||
			(lp->neighbors == 9 &&
			lp->patterned_rule == LIFE_9B34S3)) {
		Bool sides = LRAND() & 1; /* off by 1 if true */
		Bool pick = 0;

		if (sides) { /* lower right */
			hsp = NRAND(lp->nrows);
			if (LRAND() & 1) {
				vsp = 0;
				if ((vsp + hsp) & 1)
					vsp++;
			} else {
				vsp = lp->ncols - 1;
				if (!((vsp + hsp) & 1))
					vsp--;
			}
			if (hsp > lp->nrows / 2)
				hoff = -1;
			if (vsp > lp->ncols / 2)
				voff = -1;
		} else { /* down */
			vsp = NRAND(lp->ncols / 2) + lp->ncols / 4;
			if (LRAND() & 1) {
				hsp = 0;
			} else {
				hsp = lp->nrows - 1;
			}
			if (hsp > lp->nrows / 2)
				hoff = -1;
			if (vsp > lp->ncols / 2) {
				voff = -1;
				if ((vsp + hsp) & 1)
					vsp--;
			} else {
				if (!((vsp + hsp) & 1))
					vsp++;
			}
		}
		if (lp->neighbors == 12 && lp->patterned_rule == LIFE_12B3S278) {
			pick = LRAND() & 1;
		}
		if (lp->neighbors == 9 /* need here only */ &&
				lp->patterned_rule == LIFE_9B34S3) {
			/* glider (3/20, p20) */
			int SS = NRAND(10);

			if (SS == 0) {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 6 * voff);
					SetList(hsp + 2 * hoff, vsp + 7 * voff);
					SetList(hsp + 3 * hoff, vsp + 7 * voff);
					SetList(hsp + 4 * hoff, vsp + 7 * voff);
					SetList(hsp + 5 * hoff, vsp + 7 * voff);
					SetList(hsp + 6 * hoff, vsp + 7 * voff);
					SetList(hsp + 7 * hoff, vsp + 6 * voff);
					SetList(hsp + 8 * hoff, vsp + 6 * voff);
					SetList(hsp + 9 * hoff, vsp + 6 * voff);
					SetList(hsp + 11 * hoff, vsp + 5 * voff);
					SetList(hsp + 12 * hoff, vsp + 5 * voff);
					SetList(hsp + 14 * hoff, vsp + 4 * voff);
					SetList(hsp + 15 * hoff, vsp + 4 * voff);
					SetList(hsp + 15 * hoff, vsp + 3 * voff);
					SetList(hsp + 17 * hoff, vsp + 3 * voff);
					SetList(hsp + 18 * hoff, vsp + 3 * voff);
					SetList(hsp + 18 * hoff, vsp + 2 * voff);
					SetList(hsp + 19 * hoff, vsp + 2 * voff);
					SetList(hsp + 19 * hoff, vsp + 1 * voff);
					SetList(hsp + 19 * hoff, vsp + 0 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 13 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 12 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 4 * voff);
					SetList(hsp + 4 * hoff, vsp + 9 * voff);
					SetList(hsp + 4 * hoff, vsp + 11 * voff);
					SetList(hsp + 4 * hoff, vsp + 12 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
					SetList(hsp + 5 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 4 * voff);
					SetList(hsp + 5 * hoff, vsp + 5 * voff);
					SetList(hsp + 5 * hoff, vsp + 6 * voff);
					SetList(hsp + 5 * hoff, vsp + 7 * voff);
					SetList(hsp + 5 * hoff, vsp + 8 * voff);
					SetList(hsp + 5 * hoff, vsp + 9 * voff);
					SetList(hsp + 5 * hoff, vsp + 10 * voff);
					SetList(hsp + 5 * hoff, vsp + 11 * voff);
				}
			} else if (SS == 1) {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 4 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 4 * voff);
					SetList(hsp + 2 * hoff, vsp + 5 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 4 * voff);
					SetList(hsp + 4 * hoff, vsp + 5 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
					SetList(hsp + 5 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 4 * voff);
					SetList(hsp + 6 * hoff, vsp + 4 * voff);
					SetList(hsp + 7 * hoff, vsp + 3 * voff);
					SetList(hsp + 7 * hoff, vsp + 4 * voff);
					SetList(hsp + 9 * hoff, vsp + 3 * voff);
					SetList(hsp + 10 * hoff, vsp + 2 * voff);
					SetList(hsp + 10 * hoff, vsp + 3 * voff);
					SetList(hsp + 11 * hoff, vsp + 0 * voff);
					SetList(hsp + 11 * hoff, vsp + 1 * voff);
					SetList(hsp + 11 * hoff, vsp + 2 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 4 * voff);
					SetList(hsp + 1 * hoff, vsp + 7 * voff);
					SetList(hsp + 2 * hoff, vsp + 5 * voff);
					SetList(hsp + 2 * hoff, vsp + 6 * voff);
					SetList(hsp + 2 * hoff, vsp + 8 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 5 * voff);
					SetList(hsp + 3 * hoff, vsp + 7 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 4 * voff);
					SetList(hsp + 4 * hoff, vsp + 5 * voff);
					SetList(hsp + 4 * hoff, vsp + 6 * voff);
					SetList(hsp + 4 * hoff, vsp + 7 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
					SetList(hsp + 5 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 4 * voff);
					SetList(hsp + 5 * hoff, vsp + 5 * voff);
					SetList(hsp + 5 * hoff, vsp + 6 * voff);
				}
			} else if (SS <= 4) {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 6 * voff);
					SetList(hsp + 2 * hoff, vsp + 7 * voff);
					SetList(hsp + 3 * hoff, vsp + 7 * voff);
					SetList(hsp + 4 * hoff, vsp + 7 * voff);
					SetList(hsp + 5 * hoff, vsp + 7 * voff);
					SetList(hsp + 6 * hoff, vsp + 7 * voff);
					SetList(hsp + 7 * hoff, vsp + 6 * voff);
					SetList(hsp + 8 * hoff, vsp + 6 * voff);
					SetList(hsp + 9 * hoff, vsp + 6 * voff);
					SetList(hsp + 11 * hoff, vsp + 5 * voff);
					SetList(hsp + 12 * hoff, vsp + 5 * voff);
					SetList(hsp + 12 * hoff, vsp + 4 * voff);
					SetList(hsp + 14 * hoff, vsp + 4 * voff);
					SetList(hsp + 15 * hoff, vsp + 4 * voff);
					SetList(hsp + 15 * hoff, vsp + 3 * voff);
					SetList(hsp + 16 * hoff, vsp + 3 * voff);
					SetList(hsp + 16 * hoff, vsp + 2 * voff);
					SetList(hsp + 16 * hoff, vsp + 1 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 11 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 10 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 4 * voff);
					SetList(hsp + 4 * hoff, vsp + 7 * voff);
					SetList(hsp + 4 * hoff, vsp + 9 * voff);
					SetList(hsp + 4 * hoff, vsp + 10 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
					SetList(hsp + 5 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 4 * voff);
					SetList(hsp + 5 * hoff, vsp + 5 * voff);
					SetList(hsp + 5 * hoff, vsp + 6 * voff);
					SetList(hsp + 5 * hoff, vsp + 7 * voff);
					SetList(hsp + 5 * hoff, vsp + 8 * voff);
					SetList(hsp + 5 * hoff, vsp + 9 * voff);
				}
			} else {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 2 * voff);
					SetList(hsp + 1 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 3 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 1 * voff);
					SetList(hsp + 1 * hoff, vsp + 2 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
				}
			}
		} else if (lp->patterned_rule == LIFE_12B456S12) {
			/* glider (c/6, p12) */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 3 * voff);
				SetList(hsp + 4 * hoff, vsp + 3 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 3 * voff);
				SetList(hsp + 7 * hoff, vsp + 1 * voff);
				SetList(hsp + 7 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp + 4 * voff);
				SetList(hsp + 3 * hoff, vsp + 3 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 3 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 3 * voff);
				SetList(hsp + 6 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B45S23 ||
				lp->patterned_rule == LIFE_12B456S23) {
			/* glider (c/4, p8) flutters */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 3 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 3 * voff);
				SetList(hsp + 5 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 3 * voff);
				SetList(hsp + 6 * hoff, vsp + 1 * voff);
				SetList(hsp + 6 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B46S246) {
			/* glider (c/10, p10) */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B3S27 || pick == 1) {
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B3S278) {
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B45S34) {
			/* glider (c/7, p7) */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B456S34) {
			/* glider (c/6, p12) */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 3 * voff);
				SetList(hsp + 1 * hoff, vsp + 4 * voff);
				SetList(hsp + 2 * hoff, vsp + 5 * voff);
				SetList(hsp + 3 * hoff, vsp + 5 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 5 * voff);
				SetList(hsp + 6 * hoff, vsp + 2 * voff);
				SetList(hsp + 6 * hoff, vsp + 5 * voff);
				SetList(hsp + 7 * hoff, vsp + 3 * voff);
				SetList(hsp + 7 * hoff, vsp + 4 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 3 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 3 * voff);
				SetList(hsp + 6 * hoff, vsp + 0 * voff);
				SetList(hsp + 6 * hoff, vsp + 3 * voff);
				SetList(hsp + 8 * hoff, vsp + 0 * voff);
				SetList(hsp + 8 * hoff, vsp + 3 * voff);
				SetList(hsp + 9 * hoff, vsp + 0 * voff);
				SetList(hsp + 9 * hoff, vsp + 3 * voff);
				SetList(hsp + 10 * hoff, vsp + 1 * voff);
				SetList(hsp + 10 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B4S35) {
			/* glider (c/3, p3) flutters */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B46S35) {
			/* glider (c/3, p3) flutters */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 3 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 3 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B46S356) {
			/* glider (c/2, p2) flutters */
			int SS = NRAND(4);

			if (SS < 2) {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 2 * voff);
					SetList(hsp + 0 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 5 * hoff, vsp + 1 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 0 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
				}
			} else if (SS == 2) {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 3 * voff);
					SetList(hsp + 1 * hoff, vsp + 2 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 1 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 0 * voff);
					SetList(hsp + 1 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 0 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 0 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
				}
			} else {
				if (sides) {
					SetList(hsp + 0 * hoff, vsp + 2 * voff);
					SetList(hsp + 0 * hoff, vsp + 3 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
					SetList(hsp + 4 * hoff, vsp + 3 * voff);
					SetList(hsp + 5 * hoff, vsp + 1 * voff);
					SetList(hsp + 5 * hoff, vsp + 2 * voff);
				} else {
					SetList(hsp + 1 * hoff, vsp + 1 * voff);
					SetList(hsp + 1 * hoff, vsp + 3 * voff);
					SetList(hsp + 2 * hoff, vsp + 0 * voff);
					SetList(hsp + 2 * hoff, vsp + 1 * voff);
					SetList(hsp + 2 * hoff, vsp + 2 * voff);
					SetList(hsp + 2 * hoff, vsp + 3 * voff);
					SetList(hsp + 3 * hoff, vsp + 0 * voff);
					SetList(hsp + 3 * hoff, vsp + 1 * voff);
					SetList(hsp + 3 * hoff, vsp + 2 * voff);
					SetList(hsp + 3 * hoff, vsp + 3 * voff);
					SetList(hsp + 4 * hoff, vsp + 1 * voff);
					SetList(hsp + 4 * hoff, vsp + 2 * voff);
				}
			}
		} else if (lp->patterned_rule == LIFE_12B456S45) {
			/* glider (c/4, p8) flutters */
			if (sides) {
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 0 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
			} else {
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 4 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 2 * voff);
				SetList(hsp + 5 * hoff, vsp + 1 * voff);
				SetList(hsp + 5 * hoff, vsp + 2 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B4S456) {
			/* glider (c/4, p8) flutters */
			if (sides) {
				/*SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);*/
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 2 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 2 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 0 * voff);
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 0 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
			}
		}
	} else if (lp->neighbors == 12 &&
			(lp->patterned_rule == LIFE_12B3S2 ||
			lp->patterned_rule == LIFE_12B46S24)) {
		Bool sides = LRAND() & 1; // beware, sides offset by 1

		if (sides) { /* lower right */
			if (LRAND() & 1) {
				hsp = NRAND(lp->nrows / 4);
				hsp += (LRAND() & 1) ? 3 * lp->nrows / 4 : 0;
				if (LRAND() & 1) {
					vsp = 0;
					if ((vsp + hsp) & 1)
						vsp++;
				} else {
					vsp = lp->ncols - 1;
					if (!((vsp + hsp) & 1))
						vsp--;
				}
				if (hsp > lp->nrows / 2)
					hoff = -1;
				if (vsp > lp->ncols / 2)
					voff = -1;
			} else {
				vsp = NRAND(lp->ncols / 4);
				vsp += (LRAND() & 1) ? 3 * lp->ncols / 4 : 0;
				if (LRAND() & 1) {
					hsp = 0;
					if ((vsp + hsp) & 1)
						hsp++;
				} else {
					hsp = lp->nrows - 1;
					if (!((vsp + hsp) & 1))
						hsp--;
				}
				if (hsp > lp->nrows / 2) {
					hsp--;
					hoff = -1;
				}
				if (vsp > lp->ncols / 2) {
					vsp--;
					voff = -1;
				}
			}
		} else { /* down */
			hsp = NRAND(lp->nrows / 2) + lp->nrows / 4;
			if (LRAND() & 1) {
				vsp = 0;
				if ((vsp + hsp) & 1)
					vsp++;
			} else {
				vsp = lp->ncols - 1;
				if (!((vsp + hsp) & 1))
					vsp--;
			}
			if (hsp > lp->nrows / 2)
				hoff = -1;
			if (vsp > lp->ncols / 2)
				voff = -1;
		}
		if (lp->patterned_rule == LIFE_12B3S2) {
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 6 * voff);
				SetList(hsp + 1 * hoff, vsp + 7 * voff);
				SetList(hsp + 2 * hoff, vsp + 7 * voff);
				SetList(hsp + 3 * hoff, vsp + 7 * voff);
				SetList(hsp + 6 * hoff, vsp + 0 * voff);
				SetList(hsp + 8 * hoff, vsp + 0 * voff);
				SetList(hsp + 8 * hoff, vsp + 1 * voff);
				SetList(hsp + 9 * hoff, vsp + 1 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 13 * hoff, vsp + 0 * voff);
				SetList(hsp + 13 * hoff, vsp + 1 * voff);
				SetList(hsp + 13 * hoff, vsp + 2 * voff);
				SetList(hsp + 14 * hoff, vsp + 1 * voff);
			}
		} else if (lp->patterned_rule == LIFE_12B46S24) {
			/* glider (c/4, p8) */
			if (sides) {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 0 * hoff, vsp + 2 * voff);
				SetList(hsp + 1 * hoff, vsp + 0 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 2 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
			} else {
				SetList(hsp + 0 * hoff, vsp + 1 * voff);
				SetList(hsp + 1 * hoff, vsp + 1 * voff);
				SetList(hsp + 2 * hoff, vsp + 0 * voff);
				SetList(hsp + 2 * hoff, vsp + 1 * voff);
				SetList(hsp + 3 * hoff, vsp + 1 * voff);
				SetList(hsp + 4 * hoff, vsp + 0 * voff);
			}
		}
	}
}

static int
randomSymmetry(lifestruct *lp, Bool gliderSearch) {
	if (lp->polygon == 4) {
		if (gliderSearch) {
			int symmetry = NRAND(4);
			if (symmetry == 3)
				symmetry = 6;
			return symmetry;
		} else {
			return NRAND(10);
		}
	} else if (lp->polygon == 3 || lp->polygon == 6) {
		if (gliderSearch) {
			return NRAND(3);
		} else {
			return NRAND(10);
		}
	} else {
		if (gliderSearch) {
			return NRAND(3);
		} else {
			return NRAND(7);
		}
	}
}

static Bool
init_stuff(ModeInfo * mi)
{
	Display *display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	lifestruct *lp = &lifes[MI_SCREEN(mi)];

	if (lp->logo == NULL) {
		getImage(mi, &lp->logo, CELL_WIDTH, CELL_HEIGHT, CELL_BITS,
#ifdef HAVE_XPM
			 DEFAULT_XPM, CELL_NAME,
#endif
			 &lp->graphics_format, &lp->cmap, &lp->black);
		if (lp->logo == NULL) {
			free_life(display, lp);
			return False;
		}
#ifdef XBM_GRELB
		if (lp->cmap == None && lp->graphics_format == IS_XBM) {
		/* probably do not need the first but I am cautious... */
			if (!bimage.data) { /* Only need to do this once */
				bimage.data = (char *) CELL2_BITS;
				bimage.width = CELL2_WIDTH;
				bimage.height = CELL2_HEIGHT;
				bimage.bytes_per_line = (CELL2_WIDTH + 7) / 8;
			}
			lp->logo2 = &bimage;
		}
#endif
	}
#ifndef STANDALONE
	if (lp->cmap != None) {
		setColormap(display, window, lp->cmap, MI_IS_INWINDOW(mi));
		if (lp->backGC == None) {
			XGCValues xgcv;

			xgcv.background = lp->black;
			xgcv.foreground = lp->black;
			if ((lp->backGC = XCreateGC(display, window,
					GCForeground | GCBackground,
					&xgcv)) == None) {
				free_life(display, lp);
				return False;
			}
		}
	} else
#endif /* STANDALONE */
	{
		lp->black = MI_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
	}
	return True;
}

void
init_life(ModeInfo * mi)
{
	Display *display = MI_DISPLAY(mi);
	int size = MI_SIZE(mi), npats, i;
	lifestruct *lp;

	if (lifes == NULL) {
		if ((lifes = (lifestruct *) calloc(MI_NUM_SCREENS(mi),
				sizeof (lifestruct))) == NULL)
			return;
	}
	lp = &lifes[MI_SCREEN(mi)];

	lp->generation = 0;
	lp->redrawing = 0;
	lp->foundBirth = lp->foundSurvival = 0;
	if (MI_IS_FULLRANDOM(mi)) {
		int r5n1 = patterns_5rules[0];
		int r6n1 = patterns_6rules[0] + patterns_6rules[1];
		int r6n2 = r6n1 + patterns_6rules[2];
		int r6n3 = r6n2 + patterns_6rules[3];
		int r7n1 = patterns_7rules[0];
		int r8n1 = patterns_8rules[0] + patterns_8rules[1];
		int r8n2 = r8n1 + patterns_8rules[2];
		int r8n3 = r8n2 + patterns_8rules[3];
		int r8n4 = r8n3 + patterns_8rules[4];
		int r9n1 = patterns_9rules[0];
		int r12n1 = patterns_12rules[0] + patterns_12rules[1];
		int r12n2 = r12n1 + patterns_12rules[2];
		int r12n12 = r12n2 + patterns_12rules[3] + patterns_12rules[4] +
			patterns_12rules[5] + patterns_12rules[6] +
			patterns_12rules[7] + patterns_12rules[8] +
			patterns_12rules[9]  + patterns_12rules[10] +
			patterns_12rules[11] + patterns_12rules[12];
#if 1
		lp->neighbors = (NRAND(r8n4 + r6n3 + r9n1 + r12n12 + r5n1 + r7n1) < r8n4 + r6n3)
			? ((NRAND(r8n4 + r6n3) < r8n4) ? 8 : 6)
			: (NRAND(r9n1 + r12n12 + r5n1 + r7n1) < r5n1 + r7n1)
			? ((NRAND(r5n1 + r7n1) < r5n1) ? 5 : 7)
			: ((NRAND(r9n1 + r12n12) < r9n1) ? 9 : 12);
#else
		lp->neighbors = 8;
#endif
		if (lp->neighbors == 6) {
			lp->polygon = 6;
		} else if (lp->neighbors % 3 == 0) { /* TRI */
			lp->polygon = 3;
		} else if (lp->neighbors == 5 || lp->neighbors == 7) {
			lp->polygon = 5;
		} else /* if (lp->neighbors == 4 || lp->neighbors == 8) */ {
			lp->polygon = 4;
		}
		if (lp->polygon == 3 || lp->polygon == 6) {
			lp->vertical = LRAND() & 1;
		}
#if 1
		lp->allGliders = True;
#else
		if (lp->neighbors == 8) {
			int n = NRAND(r8n4);

			lp->nontot = (n < patterns_8rules[0]);
			lp->conway = (n >= patterns_8rules[0] && n < r8n1);
			lp->highlife = (n >= patterns_8rules[1] && n < r8n2);
			lp->daynight = (n >= r8n2);
		} else {
			lp->nontot = lp->conway = lp->highlife = lp->daynight = False;
		}
		if (lp->neighbors == 6) {
			int n = NRAND(r6n2);

			lp->callahan = (n < patterns_6rules[0]);
			lp->andreen = (n >= patterns_6rules[0] && n < r6n1);
			lp->bays = (n >= r6n1 );
		} else {
			lp->andreen = lp->callahan = lp->bays = False;
		}
		if (lp->neighbors == 5) {
			/*r5n1 = patterns_5rules[0];*/
			lp->pent = True;
		} else {
			lp->pent = False;
		}
		if (lp->neighbors == 7) {
			/*r7n1 = patterns_7rules[0];*/
			lp->pent2 = True;
		} else {
			lp->pent2 = False;
		}
		if (lp->neighbors == 12) {
			int n = NRAND(r12n2);

			lp->trilife = (n < patterns_12rules[0]);
			lp->trilife1 = (n >= patterns_12rules[0] && n < r12n1);
			lp->trilife2 = (n >= r12n1);
		} else {
			lp->trilife = lp->trilife1 = lp->trilife2 = False;
		}
#endif
	} else {
		lp->nontot = nontot;
		lp->conway = conway;
		lp->highlife = highlife;
		lp->daynight = daynight;
		lp->callahan = callahan;
		lp->andreen = andreen;
		lp->bays = bays;
		lp->trilife = trilife;
		lp->trilife1 = trilife1;
		lp->trilife2 = trilife2;
		lp->pent = pent;
		lp->pent2 = pent2;
		lp->vertical = vertical;
	}
	parseRule(mi, lp->ruleString);
	lp->labelOffsetX = NRAND(8);
	lp->labelOffsetY = NRAND(8);
	parseFile(mi);
	if (lp->allPatterns) {
		switch (lp->neighbors) {
		case 5:
			lp->patterned_rule = NRAND(LIFE_5RULES);
			break;
		case 6:
			lp->patterned_rule = NRAND(LIFE_6RULES);
			break;
		case 7:
			lp->patterned_rule = NRAND(LIFE_7RULES);
			break;
		case 8:
			lp->patterned_rule = NRAND(LIFE_8RULES);
			break;
		case 9:
			lp->patterned_rule = NRAND(LIFE_9RULES);
			break;
		case 12:
			lp->patterned_rule = NRAND(LIFE_12RULES);
			break;
		}
		copyFromPatternedRule(lp->neighbors, &lp->param, lp->patterned_rule);
		printRule(lp->neighbors, lp->ruleString, lp->param,
			MI_IS_VERBOSE(mi));
	} else if (lp->allGliders) {
		switch (lp->neighbors) {
		case 5:
			lp->patterned_rule = NRAND(LIFE_5GLIDERS);
			break;
		case 6:
			lp->patterned_rule = NRAND(LIFE_6GLIDERS);
			break;
		case 7:
			lp->patterned_rule = NRAND(LIFE_7GLIDERS);
			break;
		case 8:
			lp->patterned_rule = NRAND(LIFE_8GLIDERS);
			break;
		case 9:
			lp->patterned_rule = NRAND(LIFE_9GLIDERS);
			break;
		case 12:
			lp->patterned_rule = NRAND(LIFE_12GLIDERS);
			break;
		}
		copyFromPatternedRule(lp->neighbors, &lp->param, lp->patterned_rule);
		printRule(lp->neighbors, lp->ruleString, lp->param,
			MI_IS_VERBOSE(mi));
	} else {
		lp->param.birth = lp->input_param.birth;
		lp->param.survival = lp->input_param.survival;
		for (i = 0; i < maxgroups[invplot(lp->neighbors)]; i++) {
			lp->param.birthGroup[i] = lp->input_param.birthGroup[i];
			lp->param.survivalGroup[i] = lp->input_param.survivalGroup[i];
		}
		lp->patterned_rule = codeToPatternedRule(lp->neighbors, lp->param);
		printRule(lp->neighbors, lp->ruleString, lp->param,
			MI_IS_VERBOSE(mi));
	}
	lp->width = MI_WIDTH(mi);
	lp->height = MI_HEIGHT(mi);

	if (lp->first[0]) {
		for (i = 0; i < STATES; i++)
			flushList(lp, i);
	} else {
		for (i = 0; i < STATES; i++)
			if (!initList(lp, i)) {
				free_life(display, lp);
				return;
			}
	}
	free_cells(lp);
	free_stuff(display, lp);

	if (lp->polygon == 3) {
		int orient, side;
#ifdef WIN32
		int offset = 4;
#else
		int offset = 2;
#endif

		if (!lp->vertical) {
			lp->height = MI_WIDTH(mi);
			lp->width = MI_HEIGHT(mi);
		}
		lp->black = MI_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 2)
			lp->height = 2;
		if (size < -MINSIZE) {
			lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
			/* do again to trend smaller */
			lp->ys = NRAND(MIN(lp->ys, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		} else if (size < MINSIZE) {
			if (!size) {
				int min = MIN(lp->width, lp->height) / (4 * MINGRIDSIZE);
				int max = MIN(lp->width, lp->height) / (MINGRIDSIZE);

				lp->ys = MAX(MINSIZE, min + NRAND(max - min + 1));
				max = min + NRAND(lp->ys - min + 1);
				/* do again to trend smaller */
				lp->ys = MAX(MINSIZE, max);
			} else
				lp->ys = MINSIZE;
		} else
			lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE));
		lp->xs = (int) (1.52 * lp->ys);
		lp->ncols = (MAX(lp->width / lp->xs - 1, 2) / 2) * 2;
		lp->nrows = (MAX(lp->height / lp->ys - 1, 2) / 2) * 2;
		lp->xb = (lp->width - lp->xs * lp->ncols) / 2 + lp->xs / 2;
		lp->yb = (lp->height - lp->ys * lp->nrows) / 2 + lp->ys / 2;
	
		for (orient = 0; orient < 2; orient++) {
			for (side = 0; side < 3; side++) {
				if (lp->vertical) {
					lp->shape.triangle[orient][side].x =
						(lp->xs - offset) * triangleUnit[orient][side].x;
					lp->shape.triangle[orient][side].y =
						(lp->ys - offset) * triangleUnit[orient][side].y;
				} else {
					lp->shape.triangle[orient][side].y =
						(lp->xs - offset) * triangleUnit[orient][side].x;
					lp->shape.triangle[orient][side].x =
						(lp->ys - offset) * triangleUnit[orient][side].y;
				}
			}
		}
	} else if (lp->polygon == 4) {
		if (!init_stuff(mi))
			return;
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 2)
			lp->height = 2;
#if 0
		if (size == 0 && !MI_IS_ICONIC(mi)) {
			lp->pixelmode = False;
			lp->xs = lp->logo->width;
			lp->ys = lp->logo->height;
		}
#else
		if (size == 0 ||
		    MINGRIDSIZE * size > lp->width || MINGRIDSIZE * size > lp->height) {
			if (lp->width > MINGRIDSIZE * lp->logo->width &&
			    lp->height > MINGRIDSIZE * lp->logo->height) {
				lp->pixelmode = False;
				lp->xs = lp->logo->width;
				lp->ys = lp->logo->height;
			} else
			{
				int min = MIN(lp->width, lp->height) / (8 * MINGRIDSIZE);
				int max = MIN(lp->width, lp->height) / (2 * MINGRIDSIZE);

				lp->ys = MAX(MINSIZE, min + NRAND(max - min + 1));
				/* do again to trend smaller */
				max = min + NRAND(lp->ys - min + 1);
				lp->xs = lp->ys = MAX(MINSIZE, max);
				lp->pixelmode = True;
			}
		}
		else
#endif
		{
			lp->pixelmode = True;
			if (size < -MINSIZE) {
				lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
					MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
				/* do again to trend smaller */
				lp->ys = NRAND(MIN(lp->ys, MAX(MINSIZE, MIN(lp->width, lp->height) /
					MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
			} else if (size < MINSIZE)
				lp->ys = MINSIZE;
			else
				lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
					MINGRIDSIZE));
			lp->xs = lp->ys;
		}
		lp->ncols = MAX(lp->width / lp->xs, 4);
		lp->nrows = MAX(lp->height / lp->ys, 4);
		lp->xb = (lp->width - lp->xs * lp->ncols) / 2;
		lp->yb = (lp->height - lp->ys * lp->nrows) / 2;
	} else if (lp->polygon == 5) {
		int orient, side;
		lp->black = MI_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 2)
			lp->height = 2;
		if (size < -MINSIZE) {
			lp->xs = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
			/* do again to trend smaller */
			lp->xs = NRAND(MIN(lp->xs, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		} else if (size < MINSIZE) {
			if (!size) {
				int min = MIN(lp->width, lp->height) / (4 * MINGRIDSIZE);
				int max = MIN(lp->width, lp->height) / (MINGRIDSIZE);

				lp->xs = MAX(MINSIZE, min + NRAND(max - min + 1));
				/* do again to trend smaller */
				max = min + NRAND(lp->xs - min + 1);
				lp->xs = MAX(MINSIZE, max);
			} else
				lp->xs = MINSIZE;
		} else
			lp->xs = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE));
		lp->ys = lp->xs * 2;
		lp->ncols = (MAX((lp->width - 4) / lp->xs, 8) / 4) * 4;
		lp->nrows = (MAX((lp->height - lp->ys / 2) / lp->ys, 8)) / 2 * 2;
		lp->xb = (lp->width - lp->xs * lp->ncols) / 2;
		lp->yb = (lp->height - lp->ys * lp->nrows) / 2 - 2;
		for (orient = 0; orient < 4; orient++) {
			for (side = 0; side < 5; side++) {
				lp->shape.pentagon[orient][side].x =
					2 * (lp->xs - 2) * pentagonUnit[orient][side].x / 4;
				lp->shape.pentagon[orient][side].y =
					(lp->ys - 2) * pentagonUnit[orient][side].y / 4;
			}
		}
	} else if (lp->polygon == 6) {
		int nccols, ncrows, side;

		if (!lp->vertical) {
			lp->height = MI_WIDTH(mi);
			lp->width = MI_HEIGHT(mi);
		}
		lp->black = MI_BLACK_PIXEL(mi);
		lp->backGC = MI_GC(mi);
		if (lp->width < 2)
			lp->width = 2;
		if (lp->height < 4)
			lp->height = 4;
		if (size < -MINSIZE) {
			lp->ys = NRAND(MIN(-size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
			/* do again to trend smaller */
			lp->ys = NRAND(MIN(lp->ys, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE)) - MINSIZE + 1) + MINSIZE;
		} else if (size < MINSIZE) {
			if (!size) {
				int min = MIN(lp->width, lp->height) / (4 * MINGRIDSIZE);
				int max = MIN(lp->width, lp->height) / (2 * MINGRIDSIZE);

				lp->ys = MAX(MINSIZE, min + NRAND(max - min + 1));
				/* do again to trend smaller */
				max = min + NRAND(lp->ys - min + 1);
				lp->ys = MAX(MINSIZE, max);
			} else
				lp->ys = MINSIZE;
		} else
			lp->ys = MIN(size, MAX(MINSIZE, MIN(lp->width, lp->height) /
				MINGRIDSIZE));
		lp->xs = lp->ys;
		nccols = MAX(lp->width / lp->xs - 2, 2);
		ncrows = MAX(lp->height / lp->ys - 1, 2);
		lp->ncols = nccols / 2;
		lp->nrows = 2 * (ncrows / 4);
		if (lp->ncols % 4 < 2 && lp->nrows % 4 != 0)
			lp->nrows -= 2;
		else if (lp->ncols % 4 >= 2 && lp->nrows % 4 == 0)
			lp->nrows -= 2;
		lp->xb = (lp->width - lp->xs * nccols) / 2 + lp->xs / 2;
		lp->yb = (lp->height - lp->ys * (ncrows / 2) * 2) / 2 +
			lp->ys - 2;
		for (side = 0; side < 6; side++) {
			if (lp->vertical) {
				lp->shape.hexagon[side].x =
					(lp->xs - 1) * hexagonUnit[side].x;
				lp->shape.hexagon[side].y =
					((lp->ys - 1) * hexagonUnit[side].y /
					2) * 4 / 3;
			} else {
				lp->shape.hexagon[side].y =
					(lp->xs - 1) * hexagonUnit[side].x;
				lp->shape.hexagon[side].x =
					((lp->ys - 1) * hexagonUnit[side].y /
					2) * 4 / 3;
			}
		}
	}
	lp->npositions = lp->nrows * lp->ncols;

	MI_CLEARWINDOWCOLORMAP(mi, lp->backGC, lp->black);
	lp->painted = False;
	if ((lp->arr = (CellList **) calloc(lp->npositions,
			sizeof (CellList *))) == NULL) {
		free_life(display, lp);
		return;
	}

	lp->patterned_rule = codeToPatternedRule(lp->neighbors, lp->param);
	npats = 0;
	switch (lp->neighbors) {
	case 5:
		if ((unsigned) lp->patterned_rule < LIFE_5RULES)
			npats = patterns_5rules[lp->patterned_rule];
		break;
	case 6:
		if ((unsigned) lp->patterned_rule < LIFE_6RULES)
			npats = patterns_6rules[lp->patterned_rule];
		break;
	case 7:
		if ((unsigned) lp->patterned_rule < LIFE_7RULES)
			npats = patterns_7rules[lp->patterned_rule];
		break;
	case 8:
		if ((unsigned) lp->patterned_rule < LIFE_8RULES)
			npats = patterns_8rules[lp->patterned_rule];
		break;
	case 9:
		if ((unsigned) lp->patterned_rule < LIFE_9RULES)
			npats = patterns_9rules[lp->patterned_rule];
		break;
	case 12:
		if ((unsigned) lp->patterned_rule < LIFE_12RULES)
			npats = patterns_12rules[lp->patterned_rule];
		break;
	}
	if (glidersearch || patternsearch) {
		/* trying to find new patterns */
		RandomSoup(mi, randomSymmetry(lp, glidersearch), SOUPPERCENT,
			SOUPSIZE(lp->ncols), SOUPSIZE(lp->nrows));
	} else {
		if (!filePattern)
			lp->pattern = NRAND(npats + 3);
		if (lp->pattern >= npats && !filePattern)
			RandomSoup(mi, randomSymmetry(lp, False), SOUPPERCENT,
				SOUPSIZE(lp->ncols), SOUPSIZE(lp->nrows));
		else
			GetPattern(mi, lp->patterned_rule, lp->pattern);
	}
}

void
draw_life(ModeInfo * mi)
{
	Display *display = MI_DISPLAY(mi);
	Window window = MI_WINDOW(mi);
	GC gc = MI_GC(mi);
	CellList *middle[STATES];	/* To distinguish between old and new stuff */
	CellList *curr;
	cellstruct info;
	int i, count, gcount, neighborKind;
	Bool visible = False;
	lifestruct *lp;

	if (lifes == NULL)
		return;
	lp = &lifes[MI_SCREEN(mi)];

/*-
 * LIVE list are the on cells
 * DEAD list are the cells that may go on in the next iteration.
 * Init plan:
     Create live list and dead list which border all live cells
       (no good for rules like B0/S0 :) )
 * Big loop plan:
     Setup toggles, toggle state next iteration?
     Remove all from dead list except toggled and remove all from live list
       that are dead (but in this case draw background square)
     Toggle toggled states, age existing ones, create a new dead list, draw
 */

	/* Go through dead list to see if anything spawns (generate new lists),
	   then delete the used dead list */

	MI_IS_DRAWN(mi) = True;

	/* Setup toggles */
	curr = lp->first[DEAD]->next;
	while (curr != lp->last[DEAD]) {
		count = ng_neighbors(lp, curr, &gcount);
		if ((lp->param.birth & (1 << count)) || (count >= FIRSTGROUP &&
			     count < FIRSTGROUP + maxgroups[invplot(lp->neighbors)] &&
				((lp->neighbors == 6 &&
				(lp->param.birthGroup[count - FIRSTGROUP] &
				(1 << nontotalistic6[gcount])) != 0) ||
				(lp->neighbors == 8 &&
				(lp->param.birthGroup[count - FIRSTGROUP] &
				(1 << nontotalistic8[gcount])) != 0)))) {
			setcelltoggles(mi, (int) (curr->info.position % lp->ncols),
				(int) (curr->info.position / lp->ncols));
			visible = True;
		}
		curr = curr->next;
	}
	curr = lp->first[LIVE]->next;
	neighborKind = invplot(lp->neighbors);
	while (curr != lp->last[LIVE]) {
		count = ng_neighbors(lp, curr, &gcount);
		if (!((lp->param.survival & (1 << count)) || (count >= FIRSTGROUP &&
			     count < FIRSTGROUP + maxgroups[neighborKind] &&
				((lp->neighbors == 6 &&
				(lp->param.survivalGroup[count - FIRSTGROUP] &
				(1 << nontotalistic6[gcount])) != 0) ||
				(lp->neighbors == 8 &&
				(lp->param.survivalGroup[count - FIRSTGROUP] &
				(1 << nontotalistic8[gcount])) != 0))))) {
			setcelltoggles(mi, (int) (curr->info.position % lp->ncols),
				(int) (curr->info.position / lp->ncols));
			visible = True;
		}
		curr = curr->next;
	}

	/* Bring out your dead! */
	curr = lp->first[DEAD]->next;
	while (curr != lp->last[DEAD]) {
		curr = curr->next;
		if (!curr->previous->info.toggle)
			removeFromList(lp, DEAD, curr->previous);
	}
	curr = lp->first[LIVE]->next;
	while (curr != lp->last[LIVE]) {
		if (curr->info.toggle) {
			curr->info.state = DEAD;
			drawCell(mi, curr->info);
			curr = curr->next;
			removeFromList(lp, LIVE, curr->previous);
		} else
			curr = curr->next;
	}

	/* Fence off the babies */
	info.position = -1;	/* dummy value */
	info.age = 0;		/* dummy value */
	info.state = 0;		/* dummy value */
	info.toggle = 0;	/* dummy value */
	if (!addToList(lp, DEAD, info)) {
		free_life(MI_DISPLAY(mi), lp);
		return;
	}
	if (!addToList(lp, LIVE, info)) {
		free_life(MI_DISPLAY(mi), lp);
		return;
	}
	middle[DEAD] = lp->last[DEAD]->previous;
	middle[LIVE] = lp->last[LIVE]->previous;

	/* Toggle toggled states, age existing ones, create a new dead list */
	while (lp->first[DEAD]->next != middle[DEAD]) {
		curr = lp->first[DEAD]->next;
		if (!setcellfromtoggle(mi, (int) (curr->info.position % lp->ncols),
				(int) (curr->info.position / lp->ncols)))
			return;
	}
	curr = lp->first[LIVE]->next;
	while (curr != middle[LIVE]) {
		if (!setcellfromtoggle(mi, (int) (curr->info.position % lp->ncols),
				(int) (curr->info.position / lp->ncols)))
			return;
		curr = curr->next;
	}
	removeFromList(lp, DEAD, middle[DEAD]);
	removeFromList(lp, LIVE, middle[LIVE]);

	if (draw && lp->redrawing) {
		for (i = 0; i < REDRAWSTEP; i++) {
			CellList *redraw_curr = lp->arr[lp->redrawpos];

			/* TODO: More efficient to use list rather than array. */
			if (redraw_curr && redraw_curr->info.state == LIVE) {
				drawCell(mi, redraw_curr->info);
			}
			if (++(lp->redrawpos) >= lp->npositions) {
				lp->redrawing = 0;
				break;
			}
		}
	}
	if (visible)
		lp->noChangeCount = 0;
	else
		lp->noChangeCount++;
	if (lp->noChangeCount >= 8)
		init_life(mi);
	else if (++lp->generation > MI_CYCLES(mi)) {
		if (patternsearch || glidersearch)
			printList(lp, 1);
		init_life(mi);
	} else {
		if (MI_IS_VERBOSE(mi))
			(void) fprintf(stdout, "%s (%d cells)\n",
				lp->nameString, lp->ncells[LIVE]);
		if (patternsearch || glidersearch) {
			if (lp->generation == MI_CYCLES(mi) - repeat + 1)
				printList(lp, 0);
		}
		lp->painted = True;
	}
	/*
	 * generate a randomized shooter aimed roughly toward the center of the
	 * screen after batchcount.
	 */

	if (MI_COUNT(mi)) {
		if (lp->generation && lp->generation %
				((MI_COUNT(mi) < 0) ? 1 : MI_COUNT(mi)) == 0)
			shooter(mi);
	}
	if (draw && label) {
		int size = MAX(MIN(MI_WIDTH(mi), MI_HEIGHT(mi)) - 1, 1);

		if (size >= 10 * FONT_WIDTH) {
			char ruleString[120];

			/* hard code these to corners */
			XSetForeground(display, gc, MI_WHITE_PIXEL(mi));
			(void) sprintf(ruleString, "N%d:%s",
				lp->neighbors, lp->ruleString);
			XDrawString(display, window, gc,
				16 + lp->labelOffsetX,
				16 + lp->labelOffsetY + FONT_HEIGHT,
				ruleString, strlen(ruleString));
			XDrawString(display, window, gc,
				16 + lp->labelOffsetX, MI_HEIGHT(mi) - 16 -
				lp->labelOffsetY - FONT_HEIGHT / 2,
				lp->nameString, strlen(lp->nameString));
		}
	}
}

void
release_life(ModeInfo * mi)
{
	if (lifes != NULL) {
		int screen;

		for (screen = 0; screen < MI_NUM_SCREENS(mi); screen++)
			free_life(MI_DISPLAY(mi), &lifes[screen]);
		free(lifes);
		lifes = (lifestruct *) NULL;
	}
}

void
refresh_life(ModeInfo * mi)
{
	lifestruct *lp;

	if (lifes == NULL)
		return;
	lp = &lifes[MI_SCREEN(mi)];

#ifdef HAVE_XPM
	if (lp->graphics_format >= IS_XPM) {
		/* This is needed when another program changes the colormap. */
		free_life(MI_DISPLAY(mi), lp);
		init_life(mi);
		return;
	}
#endif
	if (lp->painted) {
		MI_CLEARWINDOWCOLORMAP(mi, lp->backGC, lp->black);
		lp->redrawing = 1;
		lp->redrawpos = 0;
		lp->painted = False;
	}
}

void
change_life(ModeInfo * mi)
{
	int npats, i;
	lifestruct *lp;

	if (lifes == NULL)
		return;
	lp = &lifes[MI_SCREEN(mi)];

	lp->generation = 0;
	if (lp->first[0]) {
		for (i = 0; i < STATES; i++)
			flushList(lp, i);
	} else {
		for (i = 0; i < STATES; i++)
			if (!initList(lp, i)) {
				free_life(MI_DISPLAY(mi), lp);
				return;
			}
	}
	free_cells(lp);
	if ((lp->arr = (CellList **) calloc(lp->npositions,
			sizeof (CellList *))) == NULL) {
		free_life(MI_DISPLAY(mi), lp);
		return;
	}

	MI_CLEARWINDOWCOLORMAP(mi, lp->backGC, lp->black);

	if (!filePattern)
		lp->pattern++;
	lp->patterned_rule = codeToPatternedRule(lp->neighbors, lp->param);
	npats = 0;
	switch (lp->neighbors) {
	case 5:
		if ((unsigned) lp->patterned_rule < LIFE_5RULES)
			npats = patterns_5rules[lp->patterned_rule];
		break;
	case 6:
		if ((unsigned) lp->patterned_rule < LIFE_6RULES)
			npats = patterns_6rules[lp->patterned_rule];
		break;
	case 7:
		if ((unsigned) lp->patterned_rule < LIFE_7RULES)
			npats = patterns_7rules[lp->patterned_rule];
		break;
	case 8:
		if ((unsigned) lp->patterned_rule < LIFE_8RULES)
			npats = patterns_8rules[lp->patterned_rule];
		break;
	case 9:
		if ((unsigned) lp->patterned_rule < LIFE_9RULES)
			npats = patterns_9rules[lp->patterned_rule];
		break;
	case 12:
		if ((unsigned) lp->patterned_rule < LIFE_12RULES)
			npats = patterns_12rules[lp->patterned_rule];
		break;
	}
	if (lp->pattern >= npats + 2) {
		lp->pattern = 0;
		if (lp->allPatterns) {
			lp->patterned_rule++;
			switch (lp->neighbors) {
			case 5:
				if ((unsigned) lp->patterned_rule >= LIFE_5RULES)
					lp->patterned_rule = 0;
				break;
			case 6:
				if ((unsigned) lp->patterned_rule >= LIFE_6RULES)
					lp->patterned_rule = 0;
				break;
			case 7:
				if ((unsigned) lp->patterned_rule >= LIFE_7RULES)
					lp->patterned_rule = 0;
				break;
			case 8:
				if ((unsigned) lp->patterned_rule >= LIFE_8RULES)
					lp->patterned_rule = 0;
				break;
			case 9:
				if ((unsigned) lp->patterned_rule >= LIFE_9RULES)
					lp->patterned_rule = 0;
				break;
			case 12:
				if ((unsigned) lp->patterned_rule >= LIFE_12RULES)
					lp->patterned_rule = 0;
				break;
			}
			copyFromPatternedRule(lp->neighbors, &lp->param,
				lp->patterned_rule);
			printRule(lp->neighbors, lp->ruleString, lp->param,
				MI_IS_VERBOSE(mi));
		} else if (lp->allGliders) {
			lp->patterned_rule++;
			switch (lp->neighbors) {
			case 5:
				if ((unsigned) lp->patterned_rule >= LIFE_5GLIDERS)
					lp->patterned_rule = 0;
				break;
			case 6:
				if ((unsigned) lp->patterned_rule >= LIFE_6GLIDERS)
					lp->patterned_rule = 0;
				break;
			case 7:
				if ((unsigned) lp->patterned_rule >= LIFE_7GLIDERS)
					lp->patterned_rule = 0;
				break;
			case 8:
				if ((unsigned) lp->patterned_rule >= LIFE_8GLIDERS)
					lp->patterned_rule = 0;
				break;
			case 9:
				if ((unsigned) lp->patterned_rule >= LIFE_9GLIDERS)
					lp->patterned_rule = 0;
				break;
			case 12:
				if ((unsigned) lp->patterned_rule >= LIFE_12GLIDERS)
					lp->patterned_rule = 0;
				break;
			}
			copyFromPatternedRule(lp->neighbors, &lp->param,
				lp->patterned_rule);
			printRule(lp->neighbors, lp->ruleString, lp->param,
				MI_IS_VERBOSE(mi));
		}
	}
	if (glidersearch || patternsearch) {
		/* trying to find new patterns */
		RandomSoup(mi, randomSymmetry(lp, glidersearch), SOUPPERCENT,
			SOUPSIZE(lp->ncols), SOUPSIZE(lp->nrows));
	} else {
		if (!serial && !filePattern)
			lp->pattern = NRAND(npats + 3);
		if (lp->pattern >= npats)
			RandomSoup(mi, randomSymmetry(lp, False), SOUPPERCENT,
				SOUPSIZE(lp->ncols), SOUPSIZE(lp->nrows));
		else
			GetPattern(mi, lp->patterned_rule, lp->pattern);
	}
}

#endif /* MODE_life */
